/*************************************************************************
*
*
*	Name:  format.c
*
*	Description:  Formated scan and print routines.
*
*
*	History:
*	Date		By		Comments
*
*	03/03/84	waf
*
*
*
*  This document contains confidential/proprietary information.
*
*  Copyright (c) 1983, 1984 by Digital Communication Assoc..
*
*************************************************************************
*  SForm routines module.  */




/*  Notes -

	  These fn's are based upon scanf() and printf(), but are specialized
	for the sform routines. They implement i/o both to/from the terminal and
	to/form a string.

	  NOTE that there are many restrictions upon these dedicated fn's as
	compared to the general purpose scanf() and printf().


	To Do -
	> Allow multiple fmt descriptors.
	> Longs in sfprintf.

*/

#include	"/sform/src/sfint.h"
#include	<ctype.h>


/* Global vars */

/* fld desc components */
static	int		ljust ;
static	int		cjust ;
static	int		zfill ;
static	int		fldmin ;
static	int		fldmax ;
static	int		longflag ;
static	char	dtype ;

static	long	shftl() ;
static	char	skipsp() ;
static	unsigned getdec() ;



/*************************************
**		Formated Scan Funtion.		**
**************************************


The syntax of the currently supported scan format string is:
fmt			:=  % <space>? <fmt_desc> <space>? <end_of_str>
<space>		:=  ( ' ' )
<fmt_desc>	:=  <fld_wid>? <conv_char>
<fld_wid>	:=  0? <dec_num>
<dec_num>	:=  ( a decimal number )
<conv_char>	:=  l? <conv_let>
<conv_let>	:=  d | u | o | x
<end_of_str> := ( null )

The following restrictions apply to the scanning funtion:
> When scanning numbers, an error code is returned if the number is too big,
  and all input must be legal digits.
> An error code is returned if the input number is too big.
> Embedded spaces in the input string are NOT allowed. Leading and trailing
  spaces are allowed, but 'space' means a 'blank' only, not 'white space'.
> Only format specs are allowed in the format string.
> Assignment suppression ('*') is not supported.
> After all fmt specs are satisfied, the end of the input string must
  have been reached.
> Only ONE format spec is currently allowed in the format string.

Notes -
> The sfscanf fn is analgous to the sscanf() standard function.

*/

sfscanf ( srcstr, fmtstr, retvar )

char	*srcstr ;	/* source (input) data */
char	*fmtstr ;	/* format string */
char	*retvar[] ;	/* return variable address(es)
					   (currently, only one supported) */

/*
  Synopsis -
	Special puropose sscanf() function.

  Description -
	See previous page.

  Return -
	return val	= Number of successful conversions (>0)
					or
				  Error indicator (<=0).

  Notes -
  > Currently, only one conversion specification is allowed in the format.
  > Currently, only numbers can be scanned.
  > The field width specification is allowed, but not used, by sfscanf().
	(The same fmt str is used by sfprintf()).
*/

{
	register char	c ;
	int		fmtptr ;
	int		strix ;
	long	numval ;
	int		rvix ;		/* ptr into ret var addr list */


	rvix = 0 ;		/* point to first address in ret var list */

	/* Get fmt desc */
	fmtptr = 0 ;		/* 1st char of fmt */
	skipsp(fmtstr, &fmtptr) ;	/* skip leading spaces */
	fmtptr = fmtdesc(fmtstr, fmtptr) ;
	if ( fmtptr < 0 )
		return(SF_FFMT) ;	/* return error */
	if ( dtype == 'c' || dtype == 's' )
		sfnotimp() ;	/* not supported */

	/* Chk for end of fmt str */
	c = skipsp(fmtstr, &fmtptr) ;	/* skip trailing spaces */
	if ( c != '\0' )
		return(SF_FFMT) ;	/* format parse error */


	/* Get number */
	strix = getnum(srcstr, &numval) ;
	if ( strix < 0 )
		return(strix) ;		/* error */

	/* Return value */
	if ( longflag )
		*(long *)retvar = numval ;
	else
		*(int *)retvar = (int) numval ;


	/* Inc element count */
	rvix++ ;

	/* Chk for legal end of string.
		   Only blanks allowed to end of string. */
	c = srcstr[strix] ;		/* next char after number */
	if ( c == ' ' ) {		/* if not yet at end of str */
		c = skipsp(srcstr, &strix) ;
	}
	if ( c != '\0' )		/* must be end of string */
		return(SF_FSTR) ;	/* input string scan error */

	/* Return # of elements processed */
	return(rvix) ;
}

static	long	guardbit ;
static	int		ovflag ;



static	getnum ( srcstr, rvptr )

char	srcstr[] ;
long	*rvptr ;		/* return value ptr */

/* Get a numeric value from the string data in 'srcstr'.
*/

{
	register char	c ;
	int		strix ;
	int		sign ;
	long	numval ;
	long	guardval ;
	char	guardc ;


	/* Set the 'guard values'.
		   The guardbit is used to chk for overflow. It flags the bit that
		   must not be set *before* a left shift.
		   The guardval & guardc char are used for decimal numbers. If the
		   value is > guardval, then multiplying by 10 will create an ovflow.
		   If the value is == guardval, then the next digit to be added (after
		   the multiply) must be checked, and can not be > (negative number)
		   or >= (positive number) guardc. */
	if ( dtype == 'd' ) {
		/* decimal */
		if ( longflag ) {
			guardval = 214748364 ;
			guardc = '8' ;
		}
		else {
			guardval = 3276L ;
			guardc = '8' ;
		}
	}
	else if ( dtype == 'u' ) {
		/* unsigned decimal */
		if ( longflag ) {
			guardval = 429496729 ;
			guardc = '6' ;
		}
		else {
			guardval = 6553L ;
			guardc = '6' ;
		}
	}
	else {
		/* hex or octal */
		guardbit = (long) 0x8000 ;		/* unsigned */
		if ( longflag )
			guardbit <<= 16 ;
	}


	/** Do conversion **/
	numval = 0 ;
	strix = 0 ;	/* input (src) string ptr */
	c = skipsp(srcstr, &strix) ;	/* skip leading spaces */
	strix++ ;		/* point to next char */

	/* Chk for '-' */
	sign = 0 ;		/* assume unsigned */
	if ( dtype == 'd' && c == '-' ) {
		sign = -1 ;
		c = srcstr[strix++] ;
	}

	/* Get the number */
	ovflag = 0 ;	/* overflag flag */
	do {

		/* Get next char */
		switch ( dtype ) {
			case 'u' :
			case 'd' :
				if ( isdigit(c) == 0 )
					return(SF_FSTR) ;	/* input string scan error */
				if ( numval > guardval )
					ovflag = 1 ;
				else if ( numval == guardval ) {
					if ( sign ) {
						if ( c > guardc )
							ovflag = 1 ;
					}
					else {
						if ( c >= guardc )
							ovflag = 1 ;
					}
				}
				numval *= 10 ;
				break ;
			case 'o' :
				if ( isdigit(c) == 0 || c == '8' || c == '9' )
					return(SF_FSTR) ;	/* input string scan error */
				numval = shftl(numval, 3) ;
				break ;
			case 'x' :
				if ( isxdigit(c) == 0 )
					return(SF_FSTR) ;	/* input string scan error */
				numval = shftl(numval, 4) ;
				if ( c > '9' ) {
					/* adjust hex digit */
					if ( c >= 'a' )
						c = c - 'a' ;
					else
						c = c - 'A' ;
					c += 10 + '0' ;
				}
				break ;
		}
		numval += (long)(c - '0') ;
		if ( ovflag )
			return(SF_FNOVF) ;		/* overflow occured */
		c = srcstr[strix++] ;
	}
	while ( c != '\0' && c != ' ' ) ;		/* until end of string */

	/* Chk sign flag */
	if ( sign )
		numval = -numval ;

	*rvptr = numval ;	/* return number */
	return(strix) ;
}



static	long	shftl ( lval, cnt )

long	lval ;
int		cnt ;

/* Shift the long value to the left 'cnt' times.
   The global var 'guardbit' is used to chk for 'overflow'.
*/
{
	register int	i ;


	for ( i=cnt ; i-- > 0 ; ) {
		if ( lval & guardbit )
			ovflag = -1 ;
		lval <<= 1 ;
	}

	return(lval) ;
}

static	fmtdesc ( fmtstr, fmtptr )

char	fmtstr[] ;		/* fmt string */
int		fmtptr ;		/* ptr to current pos in fmt string */

/* Get generic fmt descriptor.
   A 'generic' fmt desc may contain any chars which would be found in a
   scanf or printf fmt desc. Unsupported features are trapped.
   A '+' can be used instead of a '-' to signify center justification.

   Results returned in global vars -
		ljust		- set if left justification requested.
		cjust		- set if center justification requested.
		fldmin		- min fld width (-1 if not specified).
		fldmax		- max fld width (-1 if not specified).
		zfill		- set if zero fill requested.
		dtype		- type of data.
		longflag	- set if long data requested.

   Returns error code if error, else returns updated ptr into fmt string.
*/

{
	register char	c ;
	register int	fptr ;
	int		i ;


	fptr = fmtptr ;		/* use reg var */
	c = fmtstr[fptr++] ;

	/* Chk for '%' */
	if ( c != '%' )
		return(SF_FFMT) ;	/* format parse error */
	c = fmtstr[fptr++] ;

	/* Chk for justification */
	ljust = cjust = 0 ;
	if ( c == '-' || c == '+' ) {
		if ( c == '-' )
			ljust = -1 ;	/* left justify */
		else
			cjust = -1 ;	/* center justify */
		c = fmtstr[fptr++] ;
	}

	/* Chk for field width specification */
	zfill = 0 ;		/* assume no zero fill */
	fldmin = fldmax = -1 ;	/* assume no fld width spec */
	if ( isdigit(c) ) {
		/* get min fld wid spec */
		if ( c == '0' ) {
			/* zero fill */
			zfill = -1 ;
		}
		/* get min wid */
		i = --fptr ;		/* put ix in non-reg var */
		fldmin = getdec(fmtstr, &i) ;
		fptr = i ;		/* put ix back in reg var */
		c = fmtstr[fptr++] ;
	}
	/* chk for max fld wid spec */
	if ( c == '.' ) {
		/* get max fld wid */
		i = fptr ;
		fldmax = getdec(fmtstr, &i) ;
		fptr = i ;
		c = fmtstr[fptr++] ;
	}

	/* Chk for 'long' prefix */
	longflag = 0 ;
	if ( c == 'l' ) {
		longflag = 1 ;
		c = fmtstr[fptr++] ;
	}

	/* Get conversion char */
	switch ( c ) {

		case 'u' :
		case 'd' :
		case 'o' :
		case 'x' :
		case 's' :
		case 'c' :
			/* save type */
			dtype = c ;
			break ;

			/*	case 'e' :
					case 'f' :
					case 'g' :
					case 'h' :
					case '*' :	*/
		default :
			/* illegal/unsupported type */
			return(SF_FFMT) ;	/* format parse error */
	}

	return(fptr) ;
}

static	char	skipsp ( str, ptr )

char	*str ;
int		*ptr ;

/* Return 1st non-space char in string, starting at pos '*ptr'.
   Upon return, 'ptr' points to the first non-space char.
   Note - 'space' means ONLY ' ', not 'white space'.
*/

{
	register char	c ;
	register int	ix ;


	ix = *ptr ;
	while ( (c = str[ix]) == ' ' )
		ix++ ;
	*ptr = ix ;
	return(c) ;
}



static	unsigned	getdec ( str, ptr )

char	str[] ;
int		*ptr ;

/* Extract an unsigned, one word, decimal number from the input string,
   starting at 'ptr'.
   The ptr is updated to point to the next char after the numeric string.
   Returns an *unsigned integer* value. Return value is '0' if the string
   does not contain a number.

   Note - leading sign *not supported*.
*/

{
	register unsigned val ;
	register char	c ;
	int		ix ;


	ix = *ptr ;		/* string index */
	val = 0 ;
	while ( isdigit(c=str[ix++]) )
		val = val * 10 + (int)(c & 0xf) ;
	*ptr = --ix ;	/* next char */
	return(val) ;
}



/*************************************
**		Formated Print Funtion.		**
**************************************


The syntax of the currently supported print format string is:
fmt			:=  % <char>* <fmt_desc> <char>* <end_of_str>
<char>		:=  ( any char except a single '%' )
<fmt_desc>	:=  <fld_wid>? <conv_char>
<fld_wid>	:=  just? ('0')? <dec_num> ('.' <dec_num>)?
<just>		:=  '-' | '+'
<dec_num>	:=  ( a decimal number )
<conv_char>	:=  ('l')? <conv_let>
<conv_let>	:=  d | u | o | x
<end_of_str> := ( null )

The following restrictions apply to the print funtion:
> Only ONE format spec is currently allowed in the format string.

Notes -
> The sfprintf fn is analgous to the printf() and sprintf() standard functions.
The value of the 1st arg determines whether output is to a string or to the
screen.
> Zero fill and left justification is supported. Default justification is right
justify.
> Support for center justification has been added. Center justification is 
specified by a '+' instead of the '-' used for left justification.

*/

char	*outptr ;			/* Output ptr ('0' if output to scr) */



sfprintf ( dststr, fmtstr, datalist )

char	*dststr ;	/* dest string addr or '0' for screen output */
char	*fmtstr ;	/* format string */
int 	datalist ;	/* binary data or data addr list
					   **> (currently, only one int or addr allowed) */

/*
  Synopsis -
	Dedicated formated print.

  Description -
	See above.

  Return -
	return val	= Number of items printed (>0)
					or
				  Error code (<=0).

  Notes -
  ***> Can only handle one word args.
  **> Currently, this fn ignores justification and zero fill. This functionality
  is currently implemented in the sfflddisp() fn.
  *> If the dest string address is '0', output is to the screen. Otherwise,
  output is to the string whose address is 'dststr'.
  *> This code assumes that no output will be greater than MAXOUT chars.
  > This code assumes that ptrs are the size of int's.
*/

{
	register int fmtptr ;
	register char	c ;
	int		nelems ;


	/* dataix = 0 ;	/* index into data list */
	nelems = 0 ;	/* # elements printed */
	fmtptr = 0 ;	/* ptr into fmt str */
	outptr = dststr ;	/* dest ptr */

	/* Get to '%', printing all chars found */
	fmtptr = shwfc(fmtstr, 0) ;
	c = fmtstr[fmtptr] ;
	if ( c == '\0' )
		goto xit ;		/* no fmt desc */

	/* Get fmt desc */
	fmtptr = fmtdesc(fmtstr, fmtptr) ;
	if ( fmtptr < 0 )
		return(fmtptr) ;	/* error */
	if ( longflag )
		sfnotimp() ;		/* <<< longs not supported */


	/* Output data */
	if ( dtype == 'c' ) {

		/* Char data */
		outfld(&datalist) ;
	}

	else if ( dtype == 's' ) {

		/* String data */
		outfld(datalist) ;
	}

	else {

		/* Numeric data */
		outnum(&datalist) ;
	}


	/* Output the rest of the format string */
	fmtptr = shwfc(fmtstr, fmtptr) ;
	if ( fmtstr[fmtptr] != '\0' )
		return(SF_FFMT) ;	/* format string error */
	/* Inc data element count */
	nelems++ ;


xit:
	/* Null term the string */
	if ( outptr )		/* if output to str */
		*outptr = '\0' ;

	return(nelems) ;		/* return # items printed */
}

static	outnum ( dataptr )

char	*dataptr ;

/* Create a numeric string, using the binary data at dataptr.
   The number is of type 'dtype'.
   The number is 'built backwards' in the buffer.
*/

{
#define	NSSIZE	16
	char	numstr[NSSIZE] ;	/* build number string here */
	register int nsix ;			/* ptr into numstr */
	register int n ;
	int		shftcnt, mask ;
	long	lvalue ;
	int		sign ;


	nsix = NSSIZE - 1 ;		/* numstr index */
	numstr[nsix--] = '\0' ;	/* term the string */

	/* Get data */
	if ( longflag )
		lvalue = *(long *)dataptr ;		/* get long value */
	else {
		lvalue = (long)(*(int *)dataptr) ;	/* get int value */
		if ( dtype == 'o' || dtype == 'x' || dtype == 'u' )
			lvalue &= 0xffffL ;		/* abrogate sign extension */
	}

	/* Chk for < 0  */
	sign = 0 ;
	if ( lvalue < 0L ) {
		sign = -1 ;
		lvalue = -lvalue ;
	}

	/* Chk for 0 */
	if ( lvalue == 0 ) {
		numstr[nsix--] = '0' ;
	}

	else if ( dtype == 'o' || dtype == 'x' ) {

		/* Octal/Hex number */
		if ( dtype == 'o' ) {
			mask = 7 ;
			shftcnt = 3 ;
		}
		else {
			mask = 0xf ;
			shftcnt = 4 ;
		}
		while ( lvalue != 0L ) {
			n = (int)(lvalue & (long)mask) ;	/* lsd */
			lvalue >>= shftcnt ;
			if ( n > 9 )
				n += 87 ;	/* 'a' - 'f' */
			else
				n += 48 ;	/* '0' - '9' */
			numstr[nsix--] = n ;
		}
	}

	else {

		/* decimal */
		while ( lvalue != 0L ) {
			n = lvalue % 10 ;
			lvalue /= 10 ;
			numstr[nsix--] = n + 0x30 ;
		}
	}

	/* Chk sign */
	if ( sign ) 
		numstr[nsix--] = '-' ;

	/* Output the formated number string */
	nsix++ ;
	outfld(numstr + nsix) ;

}

static	outfld ( str )

char	*str ;

/* Output the string, using the fmt desc information.
*/

{
	register int l,r ;	/* # blanks on left & right side */
	char	*strptr ;
	int		flddif ;
	int		strsiz ;
	char	fillchar ;


	/* Get string size & string ptr */
	strsiz = strlen(str) ;
	if ( fldmax >= 0 && strsiz > fldmax )
		strsiz = fldmax ;	/* truncate */
	strptr = str ;

	/* Get # of leading/trailing spaces in field */
	flddif = fldmin - strsiz ;
	if ( flddif < 0 )
		flddif = 0 ;
	l = r = flddif ;
	if ( ljust )
		/* left justify */
		l = 0 ;
	else if ( cjust ) {
		/* center justify */
		l >>= 1 ;
		r = flddif - l ;
	}
	else {
		/* right justify */
		r = 0 ;
	}

	/* Get fill char */
	if ( zfill ) {
		fillchar = '0' ;
		if ( dtype == 'd' ) {
			/* chk for neg number */
			if ( *strptr == '-' ) {
				outcat(strptr, 1) ;		/* show leading '-' */
				strptr++ ;		/* skip '-' */
				strsiz-- ;
			}
		}
	}
	else
		fillchar = ' ' ;

	/* Output leading spaces/zeroes */
	outncat(&fillchar, l) ;

	/* Output the string */
	outcat(strptr, strsiz) ;

	/* Output trailing spaces */
	outncat(" ", r) ;
}

static	outcat ( data, cnt )

char	*data ;
int		cnt ;		/* # chars */

/* Catenate 'cnt' chars of data onto output string or screen.
*/

{


	while ( cnt-- > 0 )
		if ( outptr )
			/* catenate to string */
			*outptr++ = *data++ ;
		else
			/* output to screen */
			stputc(*data++) ;
}



static	outncat ( cptr, cnt )

char	*cptr ;
int		cnt ;

/* Call outcat() 'cnt' times with the char at cptr.
*/

{
	register int	i ;


	i = cnt ;		/* use reg var */
	while ( i-- > 0 )
		outcat(cptr, 1) ;
}

static	shwfc ( fmtstr, ix )

char	fmtstr[] ;
int		ix ;

/* Scan fmt string and display all chars found up to a fmt desc or the
   end of the fmt string.
   Escaped chars are supported.
   Returns the index of the next char after scan (either end of line or
   start of a fmt desc).
*/

{
	register char	c ;
	register int	n ;
	char	nrc ;


	for (;;) {
		c = fmtstr[ix++] ;
		if ( c == '\0') {
			/* end of fmt str */
			ix-- ;
			break ;
		}
		else if ( c == '%' ) {
			if ( fmtstr[ix] != '%' ) {	/* chk for '%%' */
				/* start of fmt desc */
				ix-- ;
				break ;		/* found start of fmt desc */
			}
			else {
				/* '%%' */
				ix++ ;		/* skip '%' & show '%' */
			}
		}
		else if ( c == '\\' ) {
			/* show escaped char */
			c = fmtstr[ix++] ;
			if ( isdigit(c) ) {
				/* get numeric val */
				n = getdec(fmtstr, &ix) ;
			}
			else {
				/* get escaped char */
				switch ( c ) {
					case 'n' :
						n = '\n' ;
						break ;
					case 't' :
						n = '\t' ;
						break ;
					case 'r' :
						n = '\r' ;
						break ;
					case 'f' :
						n = '\f' ;
						break ;
					case '\\' :
						n = '\\' ;
						break ;
					case '\'' :
						n = '\'' ;
						break ;
					default :
						n = c ;		/* ignore '\' */
				}
			}
			c = (char) n ;		/* escaped char value */
		}
		/* show char */
		nrc = c ;		/* non-register var */
		outcat(&nrc, 1) ;
	}	/* loop */

	return(ix) ;
}
