/*************************************************************************
*
*
*	Name:  enmio.c
*
*	Description:  Enumerator field i/o.
*					sfenmio()	- Enumerator i/o.
*
*
*	History:
*	Date		By		Comments
*
*	03/28/84	waf
*	06/11/84	waf		**  Rev. 2.0  **
*
*
*
*  Copyright (c) 1983, 1984 by Digital Communication Assoc..
*  This document contains confidential/proprietary information.
*
*************************************************************************
*  SForm routines module.  */




/*  Notes -

	Note that enumerator data is of type 'signed integer'.
	Note that enumerator comparisons are type insensitive if the SF_UCASE
	bit flag is set.

	Note that, currently, the enumerator string is parsed and the info is
	placed in the enmary structure. This allows faster operation, but uses
	much space and creates a limit on the size & number of enumerators.

	The enumerator strings supplied to these fn's are assumed to have
	the same syntax as in the 'c' language, including initializers.
	At present, the code for scanning the enumerator string is rather
	simple minded -
	  Unquoted enumerators may consist of only letters and numbers.
	  Quoted enumerators allow all chars up to the closing quote.
	  The only legal seperator is ','.

*/

#include	"/sform/src/sfint.h"
#include	<ctype.h>


#define		ENMSTR		16		/* Max # chars in an enumerator */
#define		ENMMAX		16		/* Max # of enumerators allowed */


/* Enumerator array.
   The input enum string is parsed and the info is put in this structure */
static	struct	{
	int		value ;				/* (int) value of enumerator */
	char	name[ENMSTR+1] ;	/* enumerator name */
} enmary[ENMMAX] ;

static	char	*esptr ;		/* enum string ptr */
static	int		leix ;			/* ix of last entry in enm ary */

sfenmio ( fld_desc, mode, data_ptr, tmp_data )
struct SF_FIELD *fld_desc ;
int		mode ;
int		*data_ptr ;
char	*tmp_data ;
/*
  Synopsis -
	Enumerator field i/o.

  Description -

  Return -
	return val	=  SForm completion code (see sform.h).

  Notes -
*/
{
	register int	sfc ;		/* completion code */
	int				eval ;		/* enum value */
	int				eix ;		/* enum ary index */
	int				ce ;
	int				icflg ;
	struct SF_EFDATA *iodata ;
	char			*val2nam() ;
	unsigned		mskwd ;
	int				flags ;
	char			*enmstr ;
	long			tl ;
	long			sfmask() ;


	/** Initialize **/
	iodata = (struct SF_EFDATA *)fld_desc->sf_iodata ;
	flags = iodata->sf_eflags ;		/* bit flags */
	enmstr = iodata->sf_enmstr ;	/* enum string */
	mskwd = iodata->sf_emask ;		/* mask word */
	/* Get the UCASE bit flag.
	   (Ignore enum name case if UCASE set). */
	icflg = flags & SF_UCASE ;


	/** Build the enumerator array **/
	esptr = enmstr ;
	sfc = esparse() ;				/* parse the enum string */
	if (sfc < 0)
		return(sfc) ;				/* enum string parse error - can't cont. */
	leix = sfc ;					/* ix of last enum */
	

	/** Do i/o on str **/
	/* Get the current value & eix of current value */
	esptr = val2nam(*data_ptr, &eix) ;
	strcpy(tmp_data, esptr) ;

again:
	sfc = sfgenio(fld_desc, mode, tmp_data) ;
	if (sfc < 0)
		return(sfc) ;			/* return error code */

	/* Chk comp code */
	if (sfc < SF_CB2) {
		/* Process special codes */
		switch (sfc) {
			case  SF_FLDFMT :
				/* Show fld format */
				enmfmt(fld_desc->sf_width, enmstr, flags, mskwd) ;
				goto again ;
			case  SF_NXTENM :
			case  SF_PRVENM :
				/* Select next/prev enum */
				if (sfc == SF_NXTENM) {
					eix++ ;						/* inc array index */
					if (eix > leix)
						eix = 0 ;				/* wrap to first enum */
				} else {
					eix-- ;						/* dec array index */
					if (eix < 0)
						eix = leix ;			/* wrap to last enum */
				}
				strcpy(tmp_data, enmary[eix].name) ;	/* new enm name */
				st_mdt = 1 ;					/* flag 'fld modified' */
				goto again ;
			default :
				/* Let caller handle it */
				return(sfc) ;
		}
	}


	/** Chk for fld modified **/
	if (st_mdt == 0)
		return(sfc) ;			/* no change */
	

	/** Fld was modified **/
	/* Get the enum val, using the updated enum name */
	ce = nam2val(tmp_data, &eval, icflg) ;
	if (ce) {
		/* Enumerator name error - report error & retry.
		   (This should only happen if user has entered a new name) */
		(*sf_usrerr)(ce) ;
		goto again ;
	}

	/* Found it - update binary data & return */
	tl = sfmask((long) *data_ptr, (long) eval, mskwd) ;
	*data_ptr = (int) tl ;
	return(sfc) ;
}

static	char	*skipsp ()
/*
  'esptr' points to next char.
  Skip any spaces, and return with global esptr updated. 
*/
{

	while (isspace(*esptr))
		esptr++ ;
}



static	esparse ()
/*
  Parse the enumerator string.
  The 'enmary' structure is built, using the enum string.
  The global var 'esptr' is ptr to the enum string.

  Return val	= Parse error code if error, else ix of last enum.
*/
{
	int		eix ;			/* array index */
	int		eval ;			/* enum value */
	char	c ;
	int		ce ;

	eix = 0 ;						/* enum ary index */
	eval = 0 ;						/* default enum values start at zero */

	/* Scan the enum string */
	skipsp() ;						/* skip leading spaces */
	c = *esptr ;
	while (c != '\0') {

		if (eix > ENMMAX)
			goto lperr ;			/* too many enumerators */

		/* Get next enum */
		ce = getname(&esptr, enmary[eix].name) ;
		if (ce < 0)
			return(ce) ;			/* enum name parse error */

		/* Get enum value.
		   Use initializer if it exists, else use default */
		skipsp() ;					/* skip spaces after name */
		c = *esptr ;
		if (c == '=') {
			/* get initializer value */
			esptr++ ;
			if (sfaptoi(&esptr, &eval) < 0)
				goto lperr ;
		}
		/* Enter val in enum ary */
		enmary[eix].value = eval ;
		eix++ ;						/* inc ary index */

		/* Inc enum default value */
		eval++ ;

		/* Chk for ',' or end of string */
		skipsp() ;
		c = *esptr ;
		if (c != '\0') {			/* if not end of str */
			if (c != ',')			/*   must be a separator */
				goto lperr ;
			esptr++ ;
			skipsp() ;
			c = *esptr ;
		}
	}

	/* Return ix of last entry */
	if (eix == 0)
		goto lperr ;				/* empty enum string */
	return(eix - 1) ;


lperr:
	/* Enumerator string parse error */
	return(SF_EENMS) ;
}



static	getname ( strptr, outstr )
char	**strptr ;
char	outstr[] ;
/*
  Get the next enumerator name from the enumerator string.
  Name is put in 'outstr'. Max # chars in name is defined by ENMSTR.
  Returns error code is parse error.
  The enum string ptr is updated.
  Enum name rules -
	> Quoted names are supported. At this time only single quotes (') 
	  are allowed. ALL chars until next quote are included in name.
	> If name is not quoted, only alphanumeric chars are allowed in name.
*/
{
	register char	*cp ;
	register char	c ;
	int				quotflg ;
	int				ix ;

	cp = *strptr ;					/* enum str ptr */

	/* Chk for quoted enum */
	quotflg = 0 ;				/* assume not */
	if (*cp == '\'') {
		/* Quoted enum */
		quotflg-- ;				/* set flag */
		cp++ ;
	}

	/* Get name */
	ix = 0 ;
	for (;;) {
		c = *cp ;
		if (quotflg) {
			if (c == '\'') {
				cp++ ;
				break ;
			}
		} else {
			if (!isalnum(c))
				break ;
		}
		outstr[ix++] = c ;
		if (ix == ENMSTR)
			return(SF_EENMS) ;		/* name too big */
		cp++ ;
	}

	/* Null term outstr */
	outstr[ix] = '\0' ;

	/* Update enum str ptr */
	*strptr = cp ;

	return(0) ;
}



static	char	*val2nam ( val, eix )
int		val ;
int		*eix ;
/*
  Given a value, find the enumerator with that value.
  Return the enmary index of the entry in 'eix'.
  Returns ptr to enum name, or "?" if undefined value.
  (eix return val = 0 if undefined val).
*/
{
	register char	*cp ;
	register int	i ;
	int		n ;

	cp = "?" ;							/* display this if not found */
	n = leix + 1 ;

	for (i=0 ; i < n ; i++)
		if (val == enmary[i].value) {
			*eix = i ;					/* enmary index */
			return(enmary[i].name) ;	/* ptr to name */
		}
	
	/* Not found */
	*eix = 0 ;
	return(cp) ;
}



static	nam2val ( name, valptr, icflg )
char	*name ;
int		*valptr ;
int		icflg ;				/* 'ignore case' flag */
/*
  Given an enum name, return the value for that enumerator.
  Value is returned in valptr.
  If icflg is set, name comparison is case insensitive.
  Note - Minimally unique abbreviations supported. Null string matches only
  another null string.

  Return val	= Error code if search error, else 0.
*/
{
	register int	eix ;
	register int	cnt ;
	int				rc ;
	int				n ;
	char			buf1[ENMSTR], buf2[ENMSTR] ;

	strcpy(buf1, name) ;			/* target name */
	if (icflg)
		sfmakupr(buf1) ;			/* ignore case */

	cnt = 0 ;						/* # of matches */
	n = leix + 1 ;
	for (eix = 0 ; eix < n ; eix++ ) {
		strcpy(buf2, enmary[eix].name) ;
		if (icflg)
			sfmakupr(buf2) ;
		if ((rc = sfascmp(buf1, buf2)) <= 0) {
			/* Found match */
			*valptr = enmary[eix].value ;
			if (rc == 0) {
				/* Exact match - use this one */
				cnt = 1 ;
				break ;
			}
			cnt++ ;					/* inc match cntr */
		}
	}

	/* Get return val */
	if (cnt == 0)
		/* No match */
		eix = SF_EINVV ;
	else if (--cnt == 0)
		/* Name found */
		eix = 0 ;
	else
		/* Name was not unique */
		eix = SF_ENUNQ ;
	return(eix) ;
}



static	sfascmp ( str1, str2 )
char	*str1, *str2 ;
/*
  String compare, with abbreviations.
  Note - If str1 is null, str2 must be null for match.
  Note - Character values of \377 in str1 are not allowed.

  Return val	=  0 if str1 is exactly equal to str2,
  				  -1 if str1 is a substring of str2,
				  else return <0
*/
{
	register char	*cp1, *cp2 ;
	char			c ;
	int				rv ;

	cp1 = str1 ;
	cp2 = str2 ;
	if (*cp1 == '\0')
		/* Null target str */
		return(*cp2) ;

	while (c = *cp1++) {
		if (c != *cp2)
			break ;
		cp2++ ;
	}

	/* Get return val */
	if (c == 0) {
		if (*cp2 != '\0')
			rv = -1 ;			/* flag as substring match */
		else
			rv = 0 ;			/* flag as exact match */
	} else
		rv = (int) c & 255 ;	/* flag as no match */

	return(rv) ;				/* return flag */
}

static	enmfmt ( fwid, enmstr, flags, mask )
int		fwid ;
char	*enmstr ;
int		flags ;
unsigned mask ;
/*
  Show fld format.
  If the enumerator string is too big to fit on the status line, as
  much as possible is shown, and the substring is terminated with '>'.
*/
{
	register int	n, slen ;
	char			msg[SF_MAXCOL+1] ;
	char			c ;

	/* Standard info */
	sffldfmt(msg, fwid, "Enum", flags) ;

	/* Chk mask */
	if (mask)
		strcat(msg, "Masked  ") ;
	
	/* Enum string */
	strcat(msg, "Enums=\"") ;
	slen = strlen(msg) ;
	n = SF_MAXCOL - slen ;
	sfblkmov((msg + slen), enmstr, n) ;
	msg[SF_MAXCOL-1] = '\0' ;		/* in case of overflow */
	slen = strlen(msg) ;
	c = '"' ;						/* assume enum str fits on line */
	if (slen > (SF_MAXCOL - 2))
		c = '>' ;					/* enmu str didn't fit */
	msg[slen] = c ;
	msg[++slen] = '\0' ;

	/* Show fmt */
	sfmsg(msg) ;
}
