/*************************************************************************
*
*
*	Name:  dgtip.c
*
*	Description:  TIP routines for DG terminal (on Altos system).
*					tinit()		- Initialize for TIP i/o.
*					tputc()		- Put a (int) char / TIP code.
*					tputa()		- Put a list of chars / TIP codes.
*					tputs()		- Put a list of chars.
*					tgetc()		- Get a (char) char / TIP code.
*					tgets()		- Get a string of chars / TIP codes.
*					tterm()		- Terminate TIP i/o.
*
*
*	History:
*	Date		By		Comments
*
*	04/27/84	waf
*
*
*
*  This document contains confidential/proprietary information.
*
*  Copyright (c) 1983, 1984 by Digital Communication Assoc..
*
*************************************************************************
*  SForm routines module.  */




/*  Notes -

	NOTE that tputc() has * integer * args and tgetc() returns a * character *
	value. See notes in tputc().

	** NOTE **
	The 'delete' and 'cursor left' codes are defined in dgtip.h. Note that
	the delete (erase) key defined by Xenix is NOT recognized.
	On the DG terminal, the 'left arrow' key returns a TI_MCL (move cursor left)
	code, and the 'delete' key returns a TI_DEL (delete) code.


	To Do -
	*> Line wrapping - implement TO_SWRAP & TO_EWRAP.

*/

/************************************************************************/
/*																		*/
/*	Screen Handling subroutines for the DG6053.
/*																		*/
/*	These routines are an adaptation and enhancement of the				*/
/*	IBMTIP routines, hence the similarity in name.						*/
/*																		*/
/*	These rotuines are NOT interchangeable with the others and			*/
/*	they contain features that the Lattice IBMTIP routines do not.		*/
/*																		*/
/************************************************************************/



#include        "dgtip.h"
#include		<sgtty.h>



static	char	vab[4]	= {0,0,80,25};	/* video area block */
static	struct	sgttyb	oldmode, newmode ;
static	int		erachar, killchar ;


char	*cntrl[] ;	/* control string array (defined below) */
char	tpgetc() ;

tinit ()

/*
  Synopsis -
	Set up to use TIP routines (initialize).

  Description -
	The screen size is defined.
	The current io characteristics are saved, and new characteristics
	set (i.e. cbreak, etc.).

  Return -
	return val	= <undefined>

  Notes -
*/

{


	/* Set screen size */
	vab[0] = vab[1] = 0;		/* set screen size parameters */
	vab[2] = 79 ;				/* 80 cols */
	vab[3] = 23;				/* 24 rows */

	/* Get current char's and set new char's */
	ioctl(0, TIOCGETP, &oldmode) ;	/* get current mode */
	newmode = oldmode ;
	newmode.sg_flags |= CBREAK ;	/* cbreak */
	newmode.sg_flags &= ~ECHO ;		/* -echo */
	ioctl(0, TIOCSETP, &newmode) ; /* Set -echo and cbreak mode */

	/* Get erase (DEL) & kill (CAN) chars */
	erachar = oldmode.sg_erase & 255 ;
	killchar = oldmode.sg_kill & 255 ;

	}

tterm ()

/*
  Synopsis -
	Reset terminal characteristics.

  Description -
	$

  Return -
	return val	= <undefined>

  Notes -
*/

{


	/* Reset char's */
	ioctl(0, TIOCSETP, &oldmode) ;

	}

/************************************************************************/
/*																		*/
/*	tputc - put a character to the screen.								*/
/*																		*/
/* synopsis																*/
/*																		*/
/*	n = tputc(c,d,e)													*/
/*	int	n;	returns an integer = number of chars used					*/
/*	char	c;	character to be output									*/
/*	char	d,e;	additional characters, used when c is a command		*/
/*																		*/
/* description															*/
/*																		*/
/*	This function will output the character c to the screen with		*/
/*	the attribute wich is currently selected.  If c is one of the		*/
/*	command characters defined in screen.c(c > 127) then tputc will		*/
/*	perform the desired screen function, in lieu of writing a 			*/
/*	character to the screen.											*/
/*
/* Notes -	
	**>> NOTE
		The input args are defined as INTEGER in this function, but
		the calling code may be sending a char value.
		This code takes advantage of the fact that 'char's are converted
		to 'int's before being used as a fn arg.
		Note that all 'sign extended' char input args can be differentiated 
		from TIP codes.
/*																		*/
/*																		*/
/************************************************************************/


tputc(c,d,e)
int	 c,d,e;			/* <<** NOTE */
{
	int	 row,col,n ;
	register unsigned ic ;
	char	*vattr ;	/* video attribute string ptr */


	ic = (c < 0)? (0 - c) : c ;		/* chk for sign extended chars */
	n = 1;			/* return value of 1 */
	if ( ic >= TO_START) switch (ic) {

		/* Process TIP output code */
		case TO_INIT:
			break ;			/* nothing to do */
		case TO_CS:			/* clear the screen */
			tpputc(*cntrl[30-20]) ;
		case TO_MCH:			/* move the cursor home */
			row =0; col = 0;
			goto move;

		case TO_MCL:			/* move cursor left one */
			tputs(cntrl[23-20]) ;
			break ;

#if	(SMALLTIP == 0)
		case TO_MCU:			/* move cursor up one line */
			tputs(cntrl[24-20]) ;
			break ;
		case TO_MCD:			/* move cursor down one line */
			tputs(cntrl[22-20]) ;
			break ;
		case TO_MCR:			/* move cursor right one */
			tputs(cntrl[21-20]) ;
			break ;
		case TO_MCV:			/* move cursor vertically */
			row = d;
			col = 0;
			n = 2;
			goto move;
#endif
		case TO_MC:			/* move cursor */
			row = d;
			col = e;
			n = 3;			/* uses three characters */
move:		/* do the movement */
			poscur(row, col) ;
			break;

		case TO_ERL:			/* erase to end of line */
			tpputc(*cntrl[32-20]) ;
			break;

#if	(SMALLTIP == 0)
		case TO_ERS:			/* erase to end of screen */
			break;
		case TO_INSLN:			/* insert a line */
			break;
		case TO_DELLN:			/* delete a line */
			break;
		case TO_INSP:			/* insert a  space */
			break;
		case TO_DEL:			/* delete a character */
			break;
		case TO_FRA:			/* the the rows attributes */
			break;
#endif

		/* DG style video attribute control */
		case TO_SDIM :		/* Start dim field */
			vattr = cntrl[38-20] ;
			goto attr ;
		case  TO_EDIM :		/* End dim field */
			vattr = cntrl[39-20] ;
			goto attr ;
		case  TO_SUNDL :	/* Start Underline */
			vattr = cntrl[52-20] ;
			goto attr ;
		case  TO_EUNDL :	/* End Underline */
			vattr = cntrl[53-20] ;
			goto attr ;
		case  TO_SBLNK :	/* Start Blinking */
			vattr = cntrl[40-20] ;
			goto attr ;
		case  TO_EBLNK :	/* End Blinking */
			vattr = cntrl[41-20] ;
			goto attr ;
		case  TO_SREV :		/* Start Reverse Video */
			vattr = cntrl[57-20] ;
			goto attr ;
		case  TO_EREV :		/* End Reverse Video */
			vattr = cntrl[58-20] ;
			goto attr ;
		case  TO_SSCRL :	/* Start (enable) scrolling */
			tputs(cntrl[59-20]) ;
			break ;
		case  TO_ESCRL :	/* End (disable) scrolling */
			tputs(cntrl[60-20]) ;
			break ;
		case  TO_VNORM :		/* Set to 'normal' attrs */
			vattr = cntrl[56-20] ;

attr:
			tputs(vattr) ;		/* Set attribute(s) */
			break;	

		case  TO_BELL :			/* Sound bell */
			tpputc(TP_BELL) ;
			break ;


		default:			/* not a command */
			/* tpputc(0) ;		/* ??? */
			n = 0;
		}


	else {

		/* an ordinary character */
		tpputc(c) ;
		}

	return(n);
	}

tputa ( ary, cnt )

int		ary[] ;		/* list of chars / codes */
int		cnt ;		/* # of chars / codes */

/*
  Synopsis -
	tputc() a list of characters / TIP codes.

  Description -
	This fn takes the place of 'tputs()'. It can handle both 'characters'
	and TIP codes.

  Return -
	return val	= <undefined>

  Notes -
  **>> NOTE - The input values are * integer *.
  > Note that this fn is no more efficient than calling tputc() for each
  value.
*/

{
	register int	ix ;
	register int	i ;


	ix = 0 ;
	while ( ix < cnt ) {
		if ( (i = tputc(ary[ix], ary[ix+1], ary[ix+2])) )
			ix += i ;
		else
			ix++ ;		/* ignore undefined code */
		}
	}



tputs ( str )

char	*str ;

/*
  Synopsis -
	tputc() a list of CHARACTER values.
	This fn will NOT handle TIP codes.

  Description -
	The string must be null terminated.

  Return -
	return val	= $

  Notes -
  > Note that this is a much simpler version of the original tputs().
	This version CAN NOT HANDLE CONTROL CODES.
*/

{
	register char	*cp ;
	register char	c ;


	cp = str ;
	while ( (c = *cp++) )
		tputc(c) ;
	}


/*****************************
**							**
**		Input routines		**
**							**
*****************************/


static	char	pushchar = 0 ;		/* 'Push stack' */



/* ASCII control code to TIP code conversion table */
static	char	asc2tip[][2] = {
/*	ASCII	TIP			*/
	TP_CR,	TI_UNP,		/* Unpend */
	TP_NL,	TI_UNP,		/* Unpend */
	TP_DEL,	TI_DEL,		/* Delete (del key) */
	TP_BS,	TI_DEL,		/* Delete (backspace) */
	TP_MCL,	TI_MCL,		/* Move Cursor Left */
	TP_MCR,	TI_MCR,		/* Move Cursor Right */
	TP_MCU,	TI_MCU,		/* Move Cursor Up */
	TP_MCD,	TI_MCD,		/* Move Cursor Down */
	TP_ESC,	TI_ESC,		/* Escape key */
	TP_QUIT, TI_QUIT,	/* Quit (^C) */
	TP_LCAN, TI_CAN,	/* Line cancel (^U) */
	0,		TI_UNDCC
	} ;

char	tgetc ()

/*
  Synopsis -
	Get next input character.
	Control chars are translated to TIP codes.

  Description -
	$

  Return -
	return val	= Character (short integer) value representing
					an ASCII char or code ( 0 - 127 )
				  or
					TIP code ( 128 - 255 )

  Notes -
*/

{
	register unsigned  ascii ;
	char	tpop() ;



	/* Chk 'stack' */
	if ( pushchar ) {
		/* Return stk value */
		return(tpop()) ;
		}

	/* Get next char */
	ascii = tpgetc() & 255 ;
	if ( ascii == TP_FNLD ) {	 /* fn key lead in */

		/* Fn key */
		ascii = tpgetc() & 255 ;		/* get fn key code */
		if ( ascii >= TP_F1 && ascii <= TP_FHI )
			/* Normal Fn key */
			ascii = TI_F1 + (ascii - TP_F1) ;
		else if ( ascii >= TP_SF1 && ascii <= TP_SFHI )
			/* Shifted Fn key */
			ascii = TI_SF1 + (ascii - TP_SF1) ;
		else if ( ascii >= TP_CF1 && ascii <= TP_CFHI )
			/* Control Fn key */
			ascii = TI_CF1 + (ascii - TP_CF1) ;
		else
			/* Undefined Fn key */
			ascii = TI_UNDEF ;		/* TI_UNDFN ??? */
		}

#if	1		/** over-ride Xenix definition **/
	else if ( ascii == TP_DEL ) {
#else
	else if ( ascii == erachar ) {
#endif

		/* erase (DEL) char */
		ascii = TI_DEL ;
		}
	
	else if ( ascii == killchar ) {

		/* kill (CAN) char */
		ascii = TI_CAN ;
		}

	else if ( ascii < 32 ) {		/* ??? */

		/* ASCII control code */
		ascii = tipcode((char)ascii, asc2tip) ;	/* Convert to TIP code */
		}

	/* Return char / TIP code */
	return((char) ascii) ;
	}

tgets ( strptr )

char	*strptr ;

/* Get a string, which is a sequence of 0 or more characters terminated
   with the TI_UNP char.
   The return string is null terminated (and does not contain the unpend
   character).
*/

{
	register char	*cp ;


	cp = strptr ;
	while ( (*cp = tgetc()) != TI_UNP )
		cp++ ;
	*cp = '\0' ;		/* null term return str */
	}

static	tipcode ( code, codetbl )

char	code ;			/* value to convert to TIP code */
char	codetbl[][2] ;	/* code conversion table */

/* Convert the input value to a TIP code, using the input conversion table.
   The table is terminated by a code value of '0' and a TIP code of 'TI_UNDCC'.
   
   Return value = TIP (int) code (from table),
				  or TI_UNDEF if value not in table.
*/

{
	register int	ix ;
	register char	x ;
	int		rv ;


	for ( ix = 0 ; (x = codetbl[ix][0]) ; ix++ ) {
		if ( x == code )
			break ;
		}
	
	rv = ((int) codetbl[ix][1]) & 255 ;		/* TIP code */
	return(rv) ;
	}

tpush ( c )

int		c ;

/* Push a char.

   Notes -
   > '0' flag is currently ignored.
   > 'Stack' overflow is NOT checked.
*/

{


	if ( c != 0 )
		pushchar = c ;		/* overflow not checked */
	}



static	char	tpop ()

/* 'Pop' the top value on the input char stack.
*/

{
	register int	rv ;


	rv = pushchar ;
	pushchar = 0 ;		/* 'Clear' the stack */
	return(rv) ;
	}

/***
**
**	Lowest level i/o routines.
**
***/


tpputc( c )

char	c ;

/* Output a character */

{


	write(1, &c, 1) ;
	}



char	tpgetc ()

/* Input a character */

{
	register char	c ;


	read(1, &c, 1) ;
	/* Chk for error ??? */

	return(c) ;
	}

/* The following table was taken from the /bb/run/src/crts.c module,
   and is used for controlling the 6053 terminal on an Altos system.
*/

/* terminal control strings */
static	char *cntrl[] = {		/* terminal type 6 */
	"\010",			/* -20 home cursor */
	"\030",			/* -21 cursor right */
	"\032",			/* -22 cursor down */
	"\031",			/* -23 cursor left */
	"\027",			/* -24 cursor up */
	"\007",			/* -25 bell */
	"\011",			/* -26 tab */
	"\015",			/* -27 return (no line-feed) */
	"\012",			/* -28 new-line */
	"\377",			/* -29 back tab */
	"\014",			/* -30 clear all to spaces */
	"\377",			/* -31 clear unprotected to spaces */
	"\013",			/* -32 clear to end of line */
	"\377",			/* -33 clear to end of page */
	"\377",			/* -34 keyboard lock */
	"\377",			/* -35 keyboard unlock */
	"\377",			/* -36 insert line */
	"\377",			/* -37 delete line */
	"\034",			/* -38 start protect (low intensity) */
	"\035",			/* -39 start unprotect (high intensity) */
	"\016",			/* -40 start blink */
	"\017",			/* -41 end blink */
	"\023",			/* -42 set format mode */
	"\022",			/* -43 end format mode */
	"\377",			/* -44 set program mode */
	"\377",			/* -45 end program mode */
	"\377",			/* -46 set block mode */
	"\377",			/* -47 set conversation mode */
	"\377",			/* -48 set flag 1 */
	"\377",			/* -49 clear flag 1 */
	"\377",			/* -50 send line (field) unprotected */
	"\377",			/* -51 send line (field) all */
	"\024",			/* -52 start underscore */
	"\025",			/* -53 end under score */
	"\035\024",		/* -54 start field (bright & underscore ) */
	"\025\034",		/* -55 end field (dim & normal ) */
	"\025\035\017\022",	/* -56 return to normal */

	/* waf - added these, for internal use */
	/* Fake 'reverse video' */
	"\024",			/* 57 (Pseudo) reverse video */
	"\025",			/* 58 (Pseudo) end reverse video */
	/* Scroll control */
	"\022",			/* 59 Enable scrolling */
	"\023",			/* 60 Disable scrolling */
	} ;

/* This function was taken from the BB src file 'crts.c'.
   It is used to position the cursor.
*/

/*							 */
/*   terminal type 6 - Data General 6053, D/100, D/200   */
/*							 */

#define	bufst	0
#define C_WR6   '\020'
#define C_LEFT6 '\031'
#define C_UP6   '\027'
#define MAXROW6 24
#define MAXCOL6 80

static	poscur(row, col)		/* cursor position function */

int row, col;

/* waf - Modified to output the control char sequence
		and to use 0 based values. 
*/
{
	register unsigned	arg1,arg2;
	int	colflag;
	int	rowflag;
	char	buf[16];


	/* waf - convert to 1 based args */
	arg1 = row ; arg2 = col ;	/* use reg vars */
	arg1++ ; arg2++ ;

	colflag = rowflag = 0 ;

	buf[bufst+0] = C_WR6;		/* cursor-write lead in */

	/* adjust for screen size */
	arg1 = ((arg1 > MAXROW6)? MAXROW6 : arg1) - 1 ;
	arg2 = ((arg2 > MAXCOL6)? MAXCOL6 : arg2) - 1 ;

	/* chk for strangeness at certain pixels.
	   Xenix will not accept certain binary strings, so we have to
	   fool it by going past the desired position, and then
	   backing up.
	   Note - assumes arg1 & arg2 have 0 hi bytes. */
	if ( arg2 == 9 || arg2 == 4 || arg2 == 10 || arg2 == 13 ) {
		arg2++;
		colflag++;	/* set col flag */
		}
	buf[bufst+1] = arg2 | 0200;
	if ( arg1 == 9 || arg1 == 4 || arg1 == 10 || arg1 == 13 ) {
		arg1++;
		rowflag++;	/* set row flag */
		}
	buf[bufst+2] = arg1 | 0200;
	/* backup, if necessary */
	arg1 = bufst + 3 ;		/* buf len before correction chars added */
	if ( colflag )
		buf[arg1++] = C_LEFT6;
	if ( rowflag )
		buf[arg1++] = C_UP6;


	/* waf - Output control string */
	buf[arg1] = '\0' ;		/* term string */
	tputs(buf) ;

	}
