/*************************************************************************
*
*
*	Name:  locks.c
*
*	Description:  Internal locking routines called by other functions
*
*
*	History:
*	Date		By	Comments
*
*	03/11/83	mas
*	04/26/83	mas	took error checks out of unlks()
*	04/27/83	mas	fixed lockf to seek back to starting
*						position
*	05/09/83	mas	changed to set the error variable to -1
*						to signify success
*	05/16/83	mas	removed unused automatics and used registers
*						where possible
*	05/24/83	mas	added lockd routine to do lock given filename
*	07/11/83	mas	added check for lock beyond the end of the
*						logical file
*	07/19/83	waf	BB_PC conditional code added to ignore lock requests.
*	07/25/83	mas	fixed check to be log recno+1.
*	08/16/83	waf		Add code to handle ikey while waiting for lock request.
*	08/24/83	waf		Rewrite code to handle ikey while waiting for
*					lock request.
*	11/16/83	waf		Use LCKTBLSIZ parameter.
*
*
*
*  This document contains confidential/proprietary information.
*
*  Copyright 1983, 1984 by Digital Communications Assoc.
*
*************************************************************************
* BB/Xenix Runtime Module */




/*  Notes -

waf	7/19/83
	The PC version of BB IGNORES all lock requests (as it is a single user
	system).

waf	8/16/83
	If the locking() function is invoked, and the area is already locked,
	then the RLOCK call will wait. If an ikey occurs while it is waiting,
	the locking() code returns an error code of -86. If ikeys are enabled,
	then the ikey should be handled 'normally'. If ikeys are disabled
	(STMA 6,5), then the locking() request is RESUBMITTED (e.g. the ikey
	is 'ignored'.


.SH*/

#include "/bb/include/ptype.h"
#include "/bb/include/pextern.h"
#include "/bb/include/bberms.h"
#include <errno.h>

#define RLOCK 3
#define UNLOCK 0

extern	int	errno;
extern	long	lseek();


lockf (lockid,lfno,recno,errvar,flagmask)

/*
	synopsis -
		file lock request.

	action -

	return -
		return val	= <null return>
		errvar is set to -1 if no error, else set to <??>.
*/

NUMDES	*errvar; 	/* error variable */
int	lockid,lfno;	/* lock id and logical file number */
unsigned flagmask;	/* flag mask for flag in LFTABL$ */
long	recno;		/* record to lock */
	{
register int	fd;
	long	start,m,n;
register LFTENT	*lfptr;
	LFTENT	lfdup,*flftbl();

	putvj(errvar,-1);		/* set errvar to no error */

#if	BB_X86		/* X86 processes locks */
	fd = lftofd(lfno,&start);	/* get start of file + fd */

	n = lseek(fd,0L,1);		/* save current position */

	lfptr = flftbl(lfno,&lfdup);	/* point to LFTENT */
	if (lfptr == (LFTENT *)0)
		errchk(errvar,BELSE);	/* LFTABL$ string err */

	if (recno > lfptr->lastrecno)
		errchk(errvar,BEIRN);	/* beyond end of file */

	if ((lfptr->lflags & flagmask) != 0)
		return;			/* return if no lock flag on */

	/* compute where to start lock */
	m = start + (recno * lfptr->lreclen);

	/* do lock */
	locks(lockid,fd,m,(long)lfptr->lreclen,errvar);

	lseek(fd,n,0);			/* seek back to where we were */
#endif

	return;
	}
/* 
.SH */

lockd (lockid,lfname,lfstart,lflen,errvar,flagmask)

/*
	synopsis -
		<??>

	action -
		<??>

	return -
		return val	= <null return>
*/

int	lockid;		/* lock id */
STRDES	lfname;		/* logical file name */
long	lfstart;	/* logical start of lock */
long	lflen;		/* log file lock area length */
NUMDES	*errvar; 	/* error variable */
unsigned flagmask;	/* flag mask for flag in LFTABL$ */
	{
register int	fd;
	int	lfno;
	long	start,m,n,x;
register LFTENT	*lfptr;
	LFTENT	lfdup,*flftbl();

	putvj(errvar,-1);		/* set errvar to no error */

#if	BB_X86
	fd = lfntofd(&lfname,&lfno,&start); /* get start of file, fd  + log #*/

	n = lseek(fd,0L,1);		/* save current position */

	lfptr = flftbl(lfno,&lfdup);	/* point to LFTENT */
	if (lfptr == (LFTENT *)0)
		errchk(errvar,BELSE);	/* LFTABL$ string err */

	if ((lfptr->lflags & flagmask) != 0)
		return;			/* return if no lock flag on */
	
	x = (lfptr->lastrecno + 1) * lfptr->lreclen;
	if (lfstart + lflen > x)
		errchk(errvar,BEIRN);

	/* compute where to start lock */
	m = start + lfstart;

	/* do lock */
	locks(lockid,fd,m,lflen,errvar);

	lseek(fd,n,0);			/* seek back to where we were */
#endif

	return;
	}

/* 
.SH */

locks (lockid, fd, start, length, errvar)

/*
	synopsis -
		Process a lock request.

	description -
		<??>

	return -
		return val	= <null return>
		If errvar exists, is used to return error flag, else code causes 
			error trap.
*/

int	lockid,fd;
long	start,length;
NUMDES	*errvar;
	{
	register int	i;

#if	BB_X86
	/* check for out of range lockid */
	if (lockid < -2 || lockid == 0) {
		errchk(errvar,BEDLK);
		return;
		}

	if (lseek(fd,start,0) < 0) {	/* seek to the spot to lock */
		errxechk( errvar );
		return;
		}

	for (i=0; i<LCKTBLSIZ; ++i) {	/* check for duplicate lockids */
		if (ust.ltab[i].lfd != IDLE && ust.ltab[i].lno == lockid) {
			errchk(errvar,BEDLK);
			return;
			}
		}

	for (i=0; i<LCKTBLSIZ; ++i) {	/* update lock table */
		if (ust.ltab[i].lfd == IDLE) {
			lkslp:
			if ( locking(fd, RLOCK, length) < 0) {
				/* error */
				if ( errno == EINTR && ust.noikey == TRUE )
					/* ikey error, ikey disabled */
					goto lkslp;
				else
					/* some other error */
					errxechk( errvar );
				return;
				}

			ust.ltab[i].lno = lockid;
			ust.ltab[i].lfd = fd;
			ust.ltab[i].lstart = start;
			ust.ltab[i].lsiz = length;
			return;
			}
		}
	errchk(errvar,BELTF);	/* lock table full */
#endif

	return;
	}
/* 
.SH */

unlks (lockid)

/*
	synopsis -
		Process an Unlock request.

	action -

	return -
		return val	= <null return>
*/

int	lockid;
	{
	register int	i;

#if	BB_X86
	for (i=0; i<LCKTBLSIZ; ++i) {
		if (ust.ltab[i].lfd != IDLE) {
			if (lockid != 0 && ust.ltab[i].lno != lockid)
				continue;
			lseek(ust.ltab[i].lfd,ust.ltab[i].lstart,0);
			locking(ust.ltab[i].lfd,UNLOCK,ust.ltab[i].lsiz);
			ust.ltab[i].lfd = IDLE;
			ust.ltab[i].lno = 0;
			}
		}
#endif

	return;
	}

