/*
 * Sparse file archiving program.
 *
 *      Modification history
 *
 * HCR: SP-1: Steve Pozgaj, February, 1981
 *	-add multivolume capability to 'rc' options
 *
 * arw	 1 Jun 1981	M000
 *	-Allowed minimum size for multivolue to be 500 for 5.25" floppies.
 *	-Added some register declarations.
 *
 * arw	11 Aug 1981	M001
 *	-Changed default tape name to avoid inadvertent destruction of
 *       mounted mag tapes.  Conditional on MSLOCAL.
 *
 * arw	 1 Sep 1981	M002
 *	-Corrected flushtape to properly handle blocked, multi-volume tapes.
 *	-Added check that blocklim is a multiple of nblock, corrected
 *	 computation of actual blocks to be used.
 *	-Allowed interruption between volumes.
 *
 * jgl  8 Sep 1981      M003
 *      - fixed bug preventing tar from handling 512 byte records via
 *        raw mag tape
 *
 * MJN	6/19/82		M004
 *	files too big to fit on one floppy are split.  Extents are
 *	supported transparently, only with 's' (mulvol) option
 *	New option 'n' meaning "not a magtape" causes seek instead
 *	of read when skipping over files.
 *
 * MJN	6/28/82		M005
 *	changed 's' option to match 'k' option in dump:  now takes
 *	a kilobyte size.  All sizes are printed in K instead of blocks.
 *	Corrected previous correction to flushtape().
 *	Changed all printf's of sizes to use %lu instead of %ld or %D.
 *
 *	***	BASIC ASSUMPTIONS
 *	Tar assumes that the system block size (currently either
 *	512 or 1024 bytes--see BSIZE in /usr/include/sys/param.h) is
 *		(1) a multiple of TBLOCK (512),
 *		(2) equal to TBLOCK, or
 *		(3) a divisor of TBLOCK.
 *	If it is a multiple, for example on a 1K block filesystem,
 *	extra code is compiled in to ensure that read/write/seek on
 *	non-magtape devices is performed only on 1K boundaries.
 *	Otherwise, no check is necessary.
 *
 * JGL	07/28/82	M006
 *	- fixed bug which left open channels when special files 
 *	  encountered.  More tha 16 special files and tar couldnt
 *	  do any more opens.
 *
 * MJN	8/7/82		M007
 *	Changed "End of file reread failure" to be a warning only.
 *
 * MK	8/10/82		M008
 *	added p switch which restores files to their original modes,
 *	ignoring present umask value.  Setuid and sticky information
 *	will be also restored to the super-user.
 *
 * jkr	4/15/83
 *	- made it act like old altos tar's if mulvol option not given
 *	  (i.e. write to end of volume and continue on the next as if
 *	  nothing happened)
 *	- made default device /dev/tar
 *
 * jkr	6/1/83
 *	-- modified tar to become spar, sparse file archiver
 *	- all files are assumed to be sparse and are dissected
 *	  so that only allocated blocks and indirect blocks
 *	  are archived
 *	-- reverted extent stuff
 *
 * jkr 8/23/83
 *	- fixed waiting on pwd process, to prevent hanging on interrupted proc
 *	- changed checksum, to differentiate spar and tar archives
 *	- added stuff to handle interrupted archives
 *
 * jml 2/6/84
 *      - since the tape drivers rewind the tape at EOT ,the close and open
 *        between multiple cartridges was no longer necessary. The close and
 *        open statements were commented out.
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/ino.h>
#include <sys/inode.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>

extern int	errno;

/* -DDEBUG	ONLY for debugging */
#ifdef	DEBUG
#undef	DEBUG
#define	DEBUG(a,b,c)	fprintf(stderr,"DEBUG - "),fprintf(stderr, a, b, c)
#else
#define	DEBUG(a,b,c)
#endif

#define	TBLOCK	512	/* tape block size--should be universal */

#ifdef	BSIZE
#define	SYS_BLOCK BSIZE	/* from sys/param.h:  secondary block size */
#else	BSIZE
#define	SYS_BLOCK 512	/* default if no BSIZE in param.h */
#endif	BSIZE

#define NBLOCK	20
#define NAMSIZ	100
#define	MODEMASK 07777	/* file creation mode mask */
#define	MAXEXT	100	/* M004 reasonable max # extents for a file */
#define	EXTMIN	50	/* M004 min blks left on floppy to split a file */

#define EQUAL	0	/* SP-1: for `strcmp' return */
#define UPDATE	2	/* SP-1: for `open' call */
#define	TBLOCKS(bytes)	(((bytes) + TBLOCK - 1)/TBLOCK)	/* useful roundup */
#define	K(tblocks)	((tblocks+1)/2)	/* tblocks to Kbytes for printing */

union hblock {
	char dummy[TBLOCK];
	struct header {
		char name[NAMSIZ];
		char mode[8];
		char uid[8];
		char gid[8];
		char size[12];	/* M004 size of this extent if file split */
		char mtime[12];
		char chksum[8];
		char linkflag;
		char linkname[NAMSIZ];
		char extno[4];		/* M004 extent #, null if not split */
		char extotal[4];	/* M004 total extents */
		char efsize[12];	/* M004 size of entire file */
	} dbuf;
} dblock, tbuf[NBLOCK];

struct linkbuf {
	ino_t	inum;
	dev_t	devnum;
	int     count;
	char	pathname[NAMSIZ];
	struct	linkbuf *nextp;
} *ihead;

struct stat stbuf;

int	rflag, xflag, vflag, tflag, mt, cflag, mflag, pflag;  /* M008 */
int	term, chksum, wflag, recno, first, linkerrok;
int	freemem = 1;
int	nblock = 1;
int	dbg = 0;

daddr_t	low;
daddr_t	high;

FILE	*tfile;
char	tname[] = "/tmp/tarXXXXXX";
char	*usefile;

#ifdef NOTALTOS
#ifdef  MSLOCAL
char	magtape[]	= "/dev/tar/default";	/* M001 - no default file */
#else
char	magtape[]	= "/dev/mt1";
#endif
#else	/* ALTOS */
char	magtape[]	= "/dev/tar";
#endif

char	Rebuf[2 * NAMSIZ];
char	*malloc();
char	*sprintf();
daddr_t	bsrch();

int	mulvol;		/* SP-1: multi-volume option selected */
long	blocklim;	/* SP-1: number of blocks to accept per volume */
long	tapepos;	/* SP-1: current block number to be written */
long	atol();		/* SP-1: to get blocklim */
int	NotTape;		/* M004 true if tape is a disk */
int	dumping;	/* M004 true if writing a tape or other archive */
int	extno;		/* M004 number of extent:  starts at 1 */
int	extotal;	/* M004 total extents in this file */
long	efsize;		/* M004 size of entire file */
long	bytes;		/* number of bytes in file */
short	extra;		/* flag to indicate whether there were more bytes than
				expected */

main(argc, argv)
int	argc;
char	*argv[];
{
	char *cp;
	int onintr(), onquit(), onhup() /* , onterm() */;

	if (argc < 2)
		usage();

	tfile = NULL;
	usefile =  magtape;
	argv[argc] = 0;
	argv++;
	for (cp = *argv++; *cp; cp++) 
		switch(*cp) {
		case 'f':
			usefile = *argv++;
			if (nblock == 1)
				nblock = 0;
			break;
		case 'c':
			cflag++;
			rflag++;
			break;
		case 'u':
			mktemp(tname);
			if ((tfile = fopen(tname, "w")) == NULL) {
				fprintf(stderr, "spar: cannot create temporary file (%s)\n", tname);
				done(1);
			}
			fprintf(tfile, "!!!!!/!/!/!/!/!/!/! 000\n");
			/* FALL THROUGH */
		case 'r':
			rflag++;
			if (nblock != 1 && cflag == 0) {
noupdate:
				fprintf(stderr, "spar: Blocked tapes cannot be updated (yet)\n");
				done(1);
			}
			break;
		case 'v':
			vflag++;
			break;
		case 'w':
			wflag++;
			break;
		case 'x':
			xflag++;
			break;
		case 't':
			tflag++;
			break;
		case 'm':
			mflag++;
			break;
		case 'p':	/* M008 */
			pflag++;
			break;
		case '-':
			break;
		case '0':
		case '1':
			magtape[7] = *cp;
			usefile = magtape;
			break;
		case 'b':
			nblock = atoi(*argv++);
			if (nblock > NBLOCK || nblock <= 0) {
				fprintf(stderr, "spar: Invalid blocksize. (Max %d)\n", NBLOCK);
				done(1);
			}
			if (rflag && !cflag)
				goto noupdate;
			break;
		case 's':
			fprintf(stderr, "spar does not support the 's' option");
			fprintf(stderr, "\n -- ignoring size parameter\n");
			argv++;
			break;
/*		case 'k':	/* M005 size of volume in KBytes */
/*			blocklim = atol(*argv++);
/*			if (blocklim < 250L) {		/* M000 */
/*				fprintf(stderr, "\nspar: sizes below 250K not supported (%lu).\n", blocklim);
/*				done(1);
/*			}
/*			blocklim *= 1024L/TBLOCK; /* M005 convert to TBLOCKs */
/*			mulvol++;
*/
			/* FALL THROUGH */
		case 'n':	/* M004 not a magtape (instead of 'k') */
			NotTape++;	/* assume non-magtape */
			break;
		case 'l':
			linkerrok++;
			break;
		default:
			fprintf(stderr, "spar: %c: unknown option\n", *cp);
			usage();
		}

#if SYS_BLOCK > TBLOCK
	/* M005 if user gave blocksize for non-tape device check integrity */
	if (NotTape && nblock > 1 && (nblock % (SYS_BLOCK / TBLOCK)) != 0) {
		fprintf(stderr, "spar: blocksize must be multiple of %d\n",
			SYS_BLOCK/TBLOCK);
		done(1);
	}
#endif

	if (rflag) {
		if (cflag && tfile != NULL) {
			usage();
			done(1);
		}
		if (signal(SIGINT, SIG_IGN) != SIG_IGN)
			signal(SIGINT, onintr);
		if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
			signal(SIGHUP, onhup);
		if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
			signal(SIGQUIT, onquit);
/*
		if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
			signal(SIGTERM, onterm);
*/
		if (strcmp(usefile, "-") == 0) {
			if (cflag == 0) {
				fprintf(stderr, "spar: can only create standard output archives\n");
				done(1);
			}
			mt = dup(1);
			nblock = 1;
		}
		else if ((mt = open(usefile, 2)) < 0) {
			if (cflag == 0 || (mt =  creat(usefile, 0666)) < 0) {
openerr:
				fprintf(stderr, "spar: cannot open %s\n", usefile);
				done(1);
			}
		}
		if (cflag == 0 && nblock == 0)
			nblock = 1;
		dorep(argv);
	}
	else if (xflag)  {
		if (strcmp(usefile, "-") == 0) {
			mt = dup(0);
			nblock = 1;
		}
		else if ((mt = open(usefile, 0)) < 0)
			goto openerr;
		doxtract(argv);
	}
	else if (tflag) {
		if (strcmp(usefile, "-") == 0) {
			mt = dup(0);
			nblock = 1;
		}
		else if ((mt = open(usefile, 0)) < 0)
			goto openerr;
		dotable();
	}
	else
		usage();
	done(0);
}

usage()
{
	fprintf(stderr, "spar usage: spar [-]{ctxru}[vfblm] [tapefile] [blocksize] [tapesize] files ...\n");	/* SP-1 */
	done(1);
}

dorep(argv)
char	*argv[];
{
	register char *cp, *cp2;
	char wdir[60];

	if (!cflag) {
		getdir();
		do {
			passtape();
			if (term)
				done(0);
			getdir();
		} while (!endtape());
		if (tfile != NULL) {
			char buf[200];

			sprintf(buf, "sort +0 -1 +1nr %s -o %s; awk '$1 != prev {print; prev=$1}' %s >%sX;mv %sX %s",
				tname, tname, tname, tname, tname, tname);
			fflush(tfile);
			system(buf);
			freopen(tname, "r", tfile);
			fstat(fileno(tfile), &stbuf);
			high = stbuf.st_size;
		}
	}

	dumping = 1;	/* M004 */
	getwdir(wdir);
	if (mulvol) {	/* SP-1 */
		if (nblock && (blocklim%nblock) != 0) {		/* M002 */
			fprintf(stderr, "spar: Volume size not a multiple of block size\n");
			done(1);
		}
		blocklim -= 2;			/* M002 - for trailer records */
		if (vflag)
			printf("Volume ends at %luK, blocking factor = %dK\n",
				K(blocklim - 1), K(nblock));
	}
	while (*argv && ! term) {
		cp2 = *argv;
		for (cp = *argv; *cp; cp++)
			if (*cp == '/')
				cp2 = cp;
		if (cp2 != *argv) {
			*cp2 = '\0';
			chdir(*argv);
			*cp2 = '/';
			cp2++;
		}
		if (*cp2 == '/') {
			cp2++;
			chdir("/");
		}
		putfile(*argv++, cp2);
		chdir(wdir);
	}
	closevol();	/* SP-1 */
	if (linkerrok == 1)
		for (; ihead != NULL; ihead = ihead->nextp)
			if (ihead->count != 0)
				fprintf(stderr, "spar: Missing links to %s\n", ihead->pathname);
}

endtape()
{
	if (dblock.dbuf.name[0] == '\0') {	/* null header = EOT */
		backtape();
		return(1);
	}
	else
		return(0);
}

getdir()
{
	register struct stat *sp;
	int i;

	readtape( (char *) &dblock);
	if (dblock.dbuf.name[0] == '\0')
		return;
	sp = &stbuf;
	sscanf(dblock.dbuf.mode, "%o", &i);
	sp->st_mode = i;
	sscanf(dblock.dbuf.uid, "%o", &i);
	sp->st_uid = i;
	sscanf(dblock.dbuf.gid, "%o", &i);
	sp->st_gid = i;
	sscanf(dblock.dbuf.size, "%lo", &sp->st_size);
	sscanf(dblock.dbuf.mtime, "%lo", &sp->st_mtime);
	sscanf(dblock.dbuf.chksum, "%o", &chksum);
	if (dblock.dbuf.extno[0] != '\0') {	/* M004 split file? */
		sscanf(dblock.dbuf.extno, "%o", &extno);
		sscanf(dblock.dbuf.extotal, "%o", &extotal);
		sscanf(dblock.dbuf.efsize, "%lo", &efsize);
	} else
		extno = 0;	/* M004 tell others file not split */
	if (chksum != (i = checksum())) {
		fprintf(stderr, "spar: directory checksum error\n");
		if (chksum + 5 == i) /* tar archive */
			done(12);
		done(2);
	}
	if (tfile != NULL)
		fprintf(tfile, "%s %s\n", dblock.dbuf.name, dblock.dbuf.mtime);
}

passtape()
{
	if (dblock.dbuf.linkflag == '1')
		return;
	bytes = stbuf.st_size;
	extra = 0;

	xspfile(-1);
}

putfile(longname, shortname)
char *longname;
char *shortname;
{
	int infile;
	long blocks;
	char buf[TBLOCK];
	register char *cp, *cp2;
	struct direct dbuf;
	int i, j;

	Rebuf[0] = '\0';
	infile = open(shortname, 0);
	if (infile < 0) {
		fprintf(stderr, "spar: %s: cannot open file\n", longname);
		return;
	}

	fstat(infile, &stbuf);

	if (tfile != NULL && checkupdate(longname) == 0) {
		close(infile);
		return;
	}
	if (checkw('r', longname) == 0) {
		close(infile);
		return;
	}

	if ((stbuf.st_mode & S_IFMT) == S_IFDIR) {
		for (i = 0, cp = buf; *cp++ = longname[i++];);
		*--cp = '/';
		cp++;
		i = 0;
		chdir(shortname);
		while (read(infile, (char *)&dbuf, sizeof(dbuf)) > 0 && !term) {
			if (dbuf.d_ino == 0) {
				i++;
				continue;
			}
			if (strcmp(".", dbuf.d_name) == 0 || strcmp("..", dbuf.d_name) == 0) {
				i++;
				continue;
			}
			cp2 = cp;
			for (j=0; j < DIRSIZ; j++)
				*cp2++ = dbuf.d_name[j];
			*cp2 = '\0';
			close(infile);
			putfile(buf, cp);
			infile = open(".", 0);
			i++;
			lseek(infile, (long) (sizeof(dbuf) * i), 0);
		}
		close(infile);
		chdir("..");
		return;
	}
	if ((stbuf.st_mode & S_IFMT) != S_IFREG) {
		fprintf(stderr, "spar: %s is not a file. Not dumped\n", longname);
		close(infile);				/* M006 */
		return;
	}

	tomodes(&stbuf);

	cp2 = longname;
	for (cp = dblock.dbuf.name, i=0; (*cp++ = *cp2++) && i < NAMSIZ; i++);
	if (i >= NAMSIZ) {
		fprintf(stderr, "spar: %s: file name too long\n", longname);
		close(infile);
		return;
	}

	if (stbuf.st_nlink > 1) {
		struct linkbuf *lp;
		int found = 0;

		for (lp = ihead; lp != NULL; lp = lp->nextp) {
			if (lp->inum == stbuf.st_ino && lp->devnum == stbuf.st_dev) {
				found++;
				break;
			}
		}
		if (found) {
			strcpy(dblock.dbuf.linkname, lp->pathname);
			dblock.dbuf.linkflag = '1';
			sprintf(dblock.dbuf.chksum, "%6o", checksum());
			if (mulvol && tapepos + 1 >= blocklim)	/* M004 */
				newvol();
			writetape( (char *) &dblock);
			if (vflag) {
				if (mulvol)	/* SP-1 M005 */
					printf("seek = %luK\t", K(tapepos));
				printf("a %s ", longname);
				printf("link to %s\n", lp->pathname);
				fflush(stdout);
			}
			lp->count--;
			close(infile);
			return;
		}
		else {
			lp = (struct linkbuf *) malloc(sizeof(*lp));
			if (lp == NULL) {
				if (freemem) {
					fprintf(stderr, "spar: Out of memory. Link information lost\n");
					freemem = 0;
				}
			}
			else {
				lp->nextp = ihead;
				ihead = lp;
				lp->inum = stbuf.st_ino;
				lp->devnum = stbuf.st_dev;
				lp->count = stbuf.st_nlink - 1;
				strcpy(lp->pathname, longname);
			}
		}
	}

	blocks = TBLOCKS(stbuf.st_size);
	bytes = stbuf.st_size;
	extra = 0;
	DEBUG("putfile: %s wants %lu blocks\n", longname, blocks);

	/* handle end of volume			M004 */
/*	if (mulvol && tapepos + blocks + 1 > blocklim) { /* file won't fit */
/*	    	if (blocklim - tapepos >= EXTMIN	/* & floppy has room */
/*	    	    && blocks + 1 >= blocklim / 10) {	/* & pretty big file */
/*			splitfile(longname, infile);	/* M004 */
/*			return;
/*		}
/*		newvol();	/* not worth it--just get new volume */
/*	}
*/

	if (vflag) {
		if (mulvol)	/* SP-1 */
			printf("seek = %luK\t", K(tapepos)); /* SP-1 M005 */
		sprintf(Rebuf, "a %s", longname);
		printf("a %s", longname);
		fflush(stdout);
	}
	sprintf(dblock.dbuf.chksum, "%6o", checksum());
	writetape( (char *) &dblock);

	aspfile(longname, shortname, infile, buf);
	close(infile);
	if ((bytes != (long) 0) || (extra != 0)) {
		printf("%s: file changed size\n");
		fflush(stdout);
	}
}

doxtract(argv)
char	*argv[];
{
	long blocks;
	char **cp;
	int ofile;

	dumping = 0;	/* M004 for newvol(), et al:  we are not writing */
	for (;;) {
		getdir();
		if (endtape())
			break;

		Rebuf[0] = '\0';
		if (*argv == 0)
			goto gotit;

		for (cp = argv; *cp; cp++)
			if (prefix(*cp, dblock.dbuf.name))
				goto gotit;
		passtape();
		continue;

gotit:
		if (checkw('x', dblock.dbuf.name) == 0) {
			passtape();
			continue;
		}

		checkdir(dblock.dbuf.name);

		if (dblock.dbuf.linkflag == '1') {
			unlink(dblock.dbuf.name);
			if (link(dblock.dbuf.linkname, dblock.dbuf.name) < 0) {
				fprintf(stderr, "spar: %s: cannot link\n", dblock.dbuf.name);
				continue;
			}
			if (vflag)
				printf("%s linked to %s\n", dblock.dbuf.name, dblock.dbuf.linkname);
			fflush(stdout);
			continue;
		}
		if ((ofile = creat(dblock.dbuf.name, stbuf.st_mode & MODEMASK)) < 0) {
			fprintf(stderr, "\nspar: %s - cannot create\n", dblock.dbuf.name);
			passtape();
			continue;
		}

		chown(dblock.dbuf.name, stbuf.st_uid, stbuf.st_gid);

/*		if (extno != 0)	{	/* file is in pieces M004 */
/*			if (extotal < 1 || extotal > MAXEXT)
/*				fprintf(stderr, "spar: ignoring bad extent info for %s\n", dblock.dbuf.name);
/*			else {
/*				xsfile(ofile);	/* M004 extract it */
/*				goto filedone;
/*			}
/*		} 
*/
		extno = 0;	/* let everyone know file is not split */
		blocks = TBLOCKS(bytes = stbuf.st_size);
		extra = 0;
		if (vflag) {
			sprintf(Rebuf, "x %s, %lu bytes", dblock.dbuf.name, bytes);
			printf("x %s, %lu bytes", dblock.dbuf.name, bytes);
			/*
			if (mulvol)
				printf("%luK\n", K(blocks));
			else
				printf("%lu tape blocks\n", blocks);
			*/
			fflush(stdout);
		}
		/*
		xblocks(bytes, ofile);
		*/
		xspfile(ofile);
		close(ofile);
filedone:
		if (mflag == 0) {
			time_t timep[2];

			timep[0] = time(NULL);
			timep[1] = stbuf.st_mtime;
			utime(dblock.dbuf.name, timep);
		}
		if (pflag)		/* M008 */
			chmod(dblock.dbuf.name, stbuf.st_mode&MODEMASK);
	}
}



dotable()
{
	dumping = 0;	/* M004 */

	/* M005 if not on magtape, maximize seek speed */
	if (NotTape) {
#if SYS_BLOCK > TBLOCK
		nblock = SYS_BLOCK / TBLOCK;
#else
		nblock = 1;
#endif
	}

	for (;;) {
		getdir();
		if (endtape())
			break;
		if (vflag)
			longt(&stbuf);
		Rebuf[0] = '\0';
		printf("%s", dblock.dbuf.name);
		if (extno != 0) {	/* M004 */
			if (vflag)
				printf("\n [extent #%d of %d] %lu bytes total",
					extno, extotal, efsize);
			else
				printf(" [extent #%d of %d]", extno, extotal);
		}
		if (dblock.dbuf.linkflag == '1')
			printf(" linked to %s\n", dblock.dbuf.linkname);
		else
		{
			sprintf(Rebuf, "%s", dblock.dbuf.name);
			fflush(stdout);
			xspfile(-1);
		}
		fflush(stdout);
	}
}

putempty()
{
	char buf[TBLOCK];
	char *cp;

	for (cp = buf; cp < &buf[TBLOCK]; )
		*cp++ = '\0';
	writetape(buf);
}

longt(st)
register struct stat *st;
{
	register char *cp;
	char *ctime();

	pmode(st);
	printf("%3d/%-3d", st->st_uid, st->st_gid);
	printf("%7lu", st->st_size);
	cp = ctime(&st->st_mtime);
	printf(" %-12.12s %-4.4s ", cp+4, cp+20);
}

#define	SUID	04000
#define	SGID	02000
#define	ROWN	0400
#define	WOWN	0200
#define	XOWN	0100
#define	RGRP	040
#define	WGRP	020
#define	XGRP	010
#define	ROTH	04
#define	WOTH	02
#define	XOTH	01
#define	STXT	01000
int	m1[] = { 1, ROWN, 'r', '-' };
int	m2[] = { 1, WOWN, 'w', '-' };
int	m3[] = { 2, SUID, 's', XOWN, 'x', '-' };
int	m4[] = { 1, RGRP, 'r', '-' };
int	m5[] = { 1, WGRP, 'w', '-' };
int	m6[] = { 2, SGID, 's', XGRP, 'x', '-' };
int	m7[] = { 1, ROTH, 'r', '-' };
int	m8[] = { 1, WOTH, 'w', '-' };
int	m9[] = { 2, STXT, 't', XOTH, 'x', '-' };

int	*m[] = { m1, m2, m3, m4, m5, m6, m7, m8, m9};

pmode(st)
register struct stat *st;
{
	register int **mp;

	for (mp = &m[0]; mp < &m[9];)
		select(*mp++, st);
}

select(pairp, st)
int *pairp;
struct stat *st;
{
	register int n, *ap;

	ap = pairp;
	n = *ap++;
	while (--n>=0 && (st->st_mode&*ap++)==0)
		ap++;
	printf("%c", *ap);
}

checkdir(name)
register char *name;
{
	register char *cp;
	int i;
	for (cp = name; *cp; cp++) {
		if (*cp == '/') {
			*cp = '\0';
			if (access(name, 01) < 0) {
				if (fork() == 0) {
					execl("/bin/mkdir", "mkdir", name, 0);
					execl("/usr/bin/mkdir", "mkdir", name, 0);
					fprintf(stderr, "\nspar: cannot find mkdir!\n");
					done(0);
				}
				while (wait(&i) >= 0);
				chown(name, stbuf.st_uid, stbuf.st_gid);
			}
			*cp = '/';
		}
	}
}

onintr()
{
	signal(SIGINT, SIG_IGN);
	term++;
}

onquit()
{
	signal(SIGQUIT, SIG_IGN);
	term++;
}

onhup()
{
	signal(SIGHUP, SIG_IGN);
	term++;
}

/*	uncomment if you need it
onterm()
{
	signal(SIGTERM, SIG_IGN);
	term++;
}
*/

tomodes(sp)
register struct stat *sp;
{
	register char *cp;

	for (cp = dblock.dummy; cp < &dblock.dummy[TBLOCK]; cp++)
		*cp = '\0';
	sprintf(dblock.dbuf.mode, "%6o ", sp->st_mode & MODEMASK);
	sprintf(dblock.dbuf.uid, "%6o ", sp->st_uid);
	sprintf(dblock.dbuf.gid, "%6o ", sp->st_gid);
	sprintf(dblock.dbuf.size, "%11lo ", sp->st_size);
	sprintf(dblock.dbuf.mtime, "%11lo ", sp->st_mtime);
}

checksum()
{
	register i;
	register char *cp;

	for (cp = dblock.dbuf.chksum; cp < &dblock.dbuf.chksum[sizeof(dblock.dbuf.chksum)]; cp++)
		*cp = ' ';
	i = 0;
	for (cp = dblock.dummy; cp < &dblock.dummy[TBLOCK]; cp++)
		i += *cp;
	return(i + 5);	/* + 5 to differentiate spar from tar */
}

checkw(c, name)
char *name;
{
	if (wflag) {
		printf("%c ", c);
		if (vflag)
			longt(&stbuf);
		printf("%s: ", name);
		if (response() == 'y'){
			return(1);
		}
		return(0);
	}
	return(1);
}

response()
{
	register int c;

	c = getchar();
	if (c != '\n')
		while (getchar() != '\n');
	else c = 'n';
	return((c >= 'A' && c <= 'Z') ? c + ('a'-'A') : c);
}

checkupdate(arg)
char	*arg;
{
	char name[100];
	long	mtime;
	daddr_t seekp;
	daddr_t	lookup();

	rewind(tfile);
	for (;;) {
		if ((seekp = lookup(arg)) < 0)
			return(1);
		fseek(tfile, seekp, 0);
		fscanf(tfile, "%s %lo", name, &mtime);
		if (stbuf.st_mtime > mtime)
			return(1);
		else
			return(0);
	}
}

/***	newvol	get new floppy (or tape) volume			M004
 *
 *	newvol();		resets tapepos and first to zero, prompts for
 *				for new volume, and waits.
 *	if dumping, end-of-file is written onto the tape.
 */

newvol()
{
	if (dumping) {
		DEBUG("newvol called with 'dumping' set\n", 0, 0);
		closevol();
		sync();
		tapepos = 0;
	} else
		first = 0;
	chgvol();
}

chgvol()
{
	register int c;
	int	enough = 0;
       
       sync(); sync(); sync();
	/*  close(mt);  */
	fprintf(stderr, "\nspar: insert next tape/disk, then type 'y <return>' ");
	fseek(stdin, 0L, 2);	/* scan over read-ahead */
	for(;;) {
		while ((c = getchar()) != '\n' && ! term)
			if (c == EOF)
				done(0);
			else
				enough++;
		if (term || enough)
			break;
		/* else */
		fprintf(stderr, "Please type something more.\n");
	}
	if (term)
		done(0);
  /*	mt = strcmp(usefile, "-") == EQUAL  ?  dup(1) : open(usefile, dumping ? UPDATE : 0);
	if (mt < 0) {
		fprintf(stderr, "spar: cannot reopen %s (%s)\n", dumping ? "output" : "input", usefile);
		done(2);
	}  */
       
	lseek (mt, 0L, 0);
	
	/* rewrite line for CDC folks */
	fprintf(stderr, "%s", Rebuf);
}

/*
 * SP-1: Write a trailer portion to close out the current output volume.
 */

closevol()
{
	putempty();
	putempty();
	flushtape();
}

done(n)
{
	unlink(tname);
	exit(n);
}

prefix(s1, s2)
register char *s1, *s2;
{
	while (*s1)
		if (*s1++ != *s2++)
			return(0);
	if (*s2)
		return(*s2 == '/');
	return(1);
}

getwdir(s)
char *s;
{
	int i, x;
	int	pipdes[2];

	pipe(pipdes);
	if ((i = fork()) == 0) {
		close(1);
		dup(pipdes[1]);
		execl("/bin/pwd", "pwd", 0);
		execl("/usr/bin/pwd", "pwd", 0);
		fprintf(stderr, "\nspar: pwd failed!\n");
		printf("/\n");
		exit(1);
	}
	if (i == -1) {
		fprintf(stderr, "\nspar: No process to get directory name!\n");
		done(9);
	}
	while ((x = wait(&i)) != -1) {
		if (term) {
			s = "say what";
			return;
		}
	}
	if (errno == EINTR) {
		s = "say who";
		return;
	}
	if (read(pipdes[0], s, 50) < 0) {
		fprintf(stderr, "\nspar: read from pwd process failed\n");
		done(9);
	}
	while(*s != '\n')
		s++;
	*s = '\0';
	close(pipdes[0]);
	close(pipdes[1]);
}

#define	N	200
int	njab;
daddr_t
lookup(s)
char *s;
{
	register i;
	daddr_t a;

	for(i=0; s[i]; i++)
		if(s[i] == ' ')
			break;
	a = bsrch(s, i, low, high);
	return(a);
}

daddr_t
bsrch(s, n, l, h)
daddr_t l, h;
char *s;
{
	register i, j;
	char b[N];
	daddr_t m, m1;

	njab = 0;

loop:
	if(l >= h)
		return(-1L);
	m = l + (h-l)/2 - N/2;
	if(m < l)
		m = l;
	fseek(tfile, m, 0);
	fread(b, 1, N, tfile);
	njab++;
	for(i=0; i<N; i++) {
		if(b[i] == '\n')
			break;
		m++;
	}
	if(m >= h)
		return(-1L);
	m1 = m;
	j = i;
	for(i++; i<N; i++) {
		m1++;
		if(b[i] == '\n')
			break;
	}
	i = cmp(b+j, s, n);
	if(i < 0) {
		h = m;
		goto loop;
	}
	if(i > 0) {
		l = m1;
		goto loop;
	}
	return(m);
}

cmp(b, s, n)
  register char *b, *s;		/* M000 */
{
	register i;

	if(b[0] != '\n')
		exit(2);
	for(i=0; i<n; i++) {
		if(b[i+1] > s[i])
			return(-1);
		if(b[i+1] < s[i])
			return(1);
	}
	return(b[i+1] == ' '? 0 : -1);
}

/***	seekdisk	seek to next file on archive		M004
 *
 *	called by passtape() only
 *
 *	WARNING: expects "nblock" to be set, that is, readtape() to have
 *		already been called.  Since passtape() is only called
 *		after a file header block has been read (why else would
 *		we skip to next file?), this is currently safe.
 *
 *	M005 changed to guarantee SYS_BLOCK boundary
 */
seekdisk(blocks)
long blocks;
{
	long seekval;
#if SYS_BLOCK > TBLOCK
	/* handle non-multiple of SYS_BLOCK */
	register nxb;	/* # extra blocks */
#endif
	DEBUG("seekdisk(%lu) called\n", blocks, 0);
	if (recno + blocks <= nblock) {
		recno += blocks;
		return;
	}
	if (recno > nblock)
		recno = nblock;
	seekval = blocks - (nblock - recno);
	recno = nblock;	/* so readtape() reads next time through */
#if SYS_BLOCK > TBLOCK
	nxb = (int) (seekval % (long)(SYS_BLOCK / TBLOCK));
	DEBUG("xtrablks=%d seekval=%lu blks\n", nxb, seekval);
	if (nxb && nxb > seekval) /* don't seek--we'll read */
		goto noseek;
	seekval -= (long) nxb;	/* don't seek quite so far */
#endif
	if (lseek(mt, (long) TBLOCK * seekval, 1) == -1L) {
		fprintf(stderr, "\nspar: device seek error\n");
		done(3);
	}
#if SYS_BLOCK > TBLOCK
	/* read those extra blocks */
noseek:
	if (nxb) {
		DEBUG("reading extra blocks\n",0,0);
		if (read(mt, tbuf, TBLOCK*nblock) < 0) {
			fprintf(stderr, "\nspar: read error while skipping file\n");
			done(8);
		}
		recno = nxb;	/* so we don't read in next readtape() */
	}
#endif
}

readtape(buffer)
char *buffer;
{
	register int i, j;		/* M000 */

	if (recno >= nblock || first == 0) {
		if (first == 0 && nblock == 0)
			j = NBLOCK;
		else
			j = nblock;
	readit:
		if ((i = read(mt, tbuf, TBLOCK*j)) < 0) {
			if (errno != ENXIO) {
				fprintf(stderr, "\nspar: tape read error\n");
				done(3);
			} else {
				chgvol();
				goto readit;
			}
		}
		if (first == 0) {
			if ((i % TBLOCK) != 0) {
				fprintf(stderr, "\nspar: tape blocksize error i %d j %d\n", i, j);
				done(3);
			}
			i /= TBLOCK;
			if (rflag && i != 1) {
				fprintf(stderr, "\nspar: Cannot update blocked tapes (yet)\n");
				done(4);
			}
			if (i != nblock && i != 1) {
				if (NotTape)
					fprintf(stderr, "spar: buffer size = %dK\n", K(i));
				else
					fprintf(stderr, "spar: blocksize = %d\n", i);
			}
			nblock = i;                     /* M003 */
		}
		recno = 0;
	}

	first = 1;
	copy(buffer, &tbuf[recno++]);
	return(TBLOCK);
}

writetape(buffer)
char *buffer;
{
	tapepos++;	/* M004 output block count */

	first = 1;
	if (nblock == 0) {
		nblock = 1;
#if SYS_BLOCK > TBLOCK
		if (NotTape)	/* for raw device	M005 */
			nblock = SYS_BLOCK / TBLOCK;
#endif
	}
	if (recno >= nblock)
		flushtape();
	copy(&tbuf[recno++], buffer);
	if (recno >= nblock)
		flushtape();
}

/*** backtape	reposition tape after reading soft "EOF" record
 *
 *	M005	if "NotTape" is set, nothing is done.
 *		it is only necessary to reread "eof" if first
 *		eof was at the end of a tape block.
 */
backtape()
{
	DEBUG("backtape() called, recno=%d nblock=%d\n", recno, nblock);
	if (NotTape || recno < nblock) {
		/*
		 * 1. backing up only sensible for tape
		 * 2. not necessary to reread unless "eof" block came
		 * 	at end of a buffered tape block
		 */
		return;
	}
	/*
	 * The first seek positions the tape to before the eof;
	 * this currently fails on raw devices.
	 * Thus, we ignore the return value from lseek().
	 * The second seek does the same.
	 */
	lseek(mt, (long) -TBLOCK, 1);	/* back one small tape block */
	recno = nblock - 1;		/* last small blk in buffered blk */
	if (read(mt, tbuf, TBLOCK*nblock) < 0) {
		fprintf(stderr, "\nspar: End of file reread failed (warning only)\n");
		/* done(4);	M007 */
	}
	lseek(mt, (long) -TBLOCK, 1);	/* back small block again */
}

/*** flushtape	write buffered block(s) onto tape	M005
 *
 * recno points to next free block in tbuf.  If nonzero, a write is done.
 * Care is taken to write in multiples of SYS_BLOCK when device is
 * non-magtape in case raw i/o is used.
 *
 * NOTE:	this is called by writetape() to do the actual writing
 */
flushtape()
{
	register i;

	DEBUG("flushtape() called, recno=%d\n", recno, 0);
	if (recno > 0) {	/* anything buffered? */
		if (NotTape) {
#if SYS_BLOCK > TBLOCK
			/* an odd-block write can only happen when
			 * we are at the end of a volume that is not a tape.
			 * Here we round recno up to an even SYS_BLOCK
			 * boundary.
			 */
			if ((i = recno % (SYS_BLOCK / TBLOCK)) != 0) {
				DEBUG("flushtape() %d rounding blocks\n", i, 0);
				recno += i;	/* round up to even SYS_BLOCK */
			}
#endif
			if (recno > nblock)
				recno = nblock;
		}
	writeit:
		if(dbg = write(mt, tbuf,(NotTape ? recno : nblock) * TBLOCK) < 0) {
			DEBUG("dbg=%d, nblock=%d\n", dbg, nblock);
			if (errno != ENXIO) {
				fprintf(stderr, "\nspar: tape write error\n");
				done(2);
			} else {
				chgvol();
				goto writeit;
			}
		}
		recno = 0;
	}
}

copy(to, from)
register char *to, *from;
{
	register i;

	i = TBLOCK;
	do {
		*to++ = *from++;
	} while (--i);
}

/*****************************************************************************
**
**	SPARSE FILE ROUTINES
**
**	xspfile(), aspfile(), iarch(), tloop(), ario(), iexpand()
*/

#define	FLOPTOHARD	(char) 1
#define	HARDTOFLOP	(char) 0

char	Spbuf[BSIZE];
long	Datablocks;
long	Indblocks;

/*
**	xspfile - extract sparse file
*/
xspfile(ofile)
{
	char	dibuf[BSIZE];
	char	ibuf[BSIZE];
	register char	*p;
	int	dummy = 1;

	Indblocks = Datablocks = (long) 0;
	/* get inode block */
	readtape(dibuf);

	for (p = dibuf; p < &dibuf[BSIZE]; p++)
		if (*p) {
			dummy = 0;
			break;
		}
	
	if (dummy) {
		fprintf(stderr, ", not archived\n");
		fprintf(stderr, "\nspar: incomplete archive (creation was interrupted)\n");
		if (ofile != -1)
			fprintf(stderr, "extraction aborted\n");
		done(10);
	}

	/* turn it into a real inode structrue */
	iexpand(ibuf, dibuf);

	/* extract the file, sparsely */
	iarch(ibuf, ofile, FLOPTOHARD);

	if ((vflag) && ((ofile != -1) || (tflag))) {
		printf(", %lu data", Datablocks);
		if (Indblocks)
			printf(" + %lu indirect", Indblocks);
		printf(" blocks\n");
	} else if (tflag)
		printf("\n");
}

#define	SNMOUNT	8
#define	SNAMSIZ	32
#define DEFSYS	"/dev/hd0b"

int	Fsys;	/* file descriptor for the file system */

aspfile(longname, shortname, fild, buf)
char	*longname, *shortname, *buf;
{
	struct mtab {
		char	file[SNAMSIZ];
		char	spec[SNAMSIZ];
	} mtab[SNMOUNT];
	char	specbuf[2*SNAMSIZ];
	char	*p, *q, *r, *dev;
	char		dirbuf[60];
	int		dirfd;
	struct direct	dbuf;
	struct dinode	din;
	struct inode	in;
	register struct mtab *mp;
	int		mf;
	ino_t		myino;
	
	Indblocks = Datablocks = (long) 0;
	getwdir(dirbuf);

	/* if we have gotten an interrupt, make this an incomplete archive */
	/* by writing out an empty inode block; check now because the intr */
	/* might have fouled our getwdir */

	if (term) {
		for (p = buf; p < &buf[BSIZE]; *p++ = (char) 0)
			;
		writetape(buf);
		flushtape();
		done(10);
	}
		

	if ((dirfd = open(dirbuf, 0)) == -1) {
		fprintf(stderr, "can't open directory %s\n", dirbuf);
		done(9);
	}

	myino = 0;
	while (read(dirfd, (char *)&dbuf, sizeof(dbuf)) == sizeof(dbuf)) {
		if (strncmp(shortname, dbuf.d_name, DIRSIZ) == 0) {
			myino = dbuf.d_ino;
			break;
		}
	}
	if (myino <= 0) {
		fprintf(stderr, "can't find inode number for file %s\n", longname);
		done(9);
	}
	close(dirfd);

	if ((mf = open("/etc/mtab", 0)) == -1) {
		fprintf(stderr, "can't open mount file");
		done(9);
	}
	for (p = (char *)&mtab[SNMOUNT]; p >= (char *)mtab; *(--p) = 0)
		;
	if (read(mf, (char *)mtab, SNMOUNT*sizeof (struct mtab)) < 0) {
		fprintf(stderr, "\nspar: read error on mount table");
		done(9);
	}
	close(mf);
#ifdef SPDEBUG
	printf("mtab:\n%d\n\n", SNMOUNT*sizeof (struct mtab));
#endif
	dev = DEFSYS;
	for (mp = mtab; mp < &mtab[SNMOUNT]; mp++) {
		if (!mp->file[0])
		{
#ifdef SPDEBUG
			printf("null entry\n");
#endif
			continue;
		}
		if (strcmp("/", mp->file) == 0)
		{
#ifdef SPDEBUG
			printf("root entry\n");
#endif
			continue;
		}
		if (strncmp(dirbuf, mp->file, strlen(mp->file)) == 0) {
#ifdef SPDEBUG
			printf("match %s\n", mp->file);
#endif
			specbuf[0] = (char) 0;
			strcat(specbuf, "/dev/");
			strcat(specbuf, mp->spec);
			dev = specbuf;
			break;
		}
#ifdef SPDEBUG
		printf("no match %s %s\n", mp->file, mp->spec);
#endif
	}
	if ((Fsys= open(dev, 0)) == -1) {
		fprintf(stderr, "can't open root file %s\n", dev);
		done(9);
	}
#ifdef SPDEBUG
	printf("inode %d\n", myino);
#endif
	/* seek to the inode for this file */
	lseek(Fsys, (long) (myino - 1) * sizeof (struct dinode) + 2 * BSIZE, 0);
	if (read(Fsys, &din, sizeof (struct dinode)) != sizeof (struct dinode))
	{
		fprintf(stderr, "can't read inode\n");
		done(9);
	}
#ifdef DINDEBUG
	dirfd = creat("din", 0770);
	write(dirfd, &din, BSIZE);
	close(dirfd);
#endif

	/* last chance to abort this archive before doing this entire file */
	/* so if we've been intr'd, make this an incomplete archive */
	if (term) {
		for (p = buf; p < &buf[BSIZE]; *p++ = (char) 0)
			;
		writetape(buf);
		flushtape();
		done(10);
	}
		
	writetape(&din);

	iexpand(&in, &din);
	iarch(&in, fild, HARDTOFLOP);
	close(Fsys);
	if (vflag) {
		printf(", %lu data", Datablocks);
		if (Indblocks)
			printf(" + %lu indirect", Indblocks);
		printf(" blocks\n");
	}
}

/*
 * iarch - Archive or restore all the disk blocks associated
 * with the inode structure of a file.
 * Order is left to right depth first (preorder traverse).
 * The flag floptohard tells whether this is an archive or a restore.
 */

iarch(ip, fild, floptohard)
register struct inode *ip;
char floptohard;
{
	register i;
	daddr_t bn;

	for(i=0; i < NADDR; i++)
	{
		bn = ip->i_un.i_i.i_addr[i];

		switch(i) {
		default:
#ifdef SPDEBUG
			if (bn)
				printf("i=%d: bn=%O\n", i, bn);
#endif
			if (bn == (daddr_t) 0) {
				if (bytes)
					bytes -= (long) BSIZE;
				if (fild >= 0)
					lseek(fild, (long) BSIZE, 1);
			} else
				ario(fild, floptohard);
			break;

		case NADDR-3:
#ifdef SPDEBUG
			printf("ind bn=%O\n", bn);
#endif
			if (bn == (daddr_t) 0) {
				if (bytes)
					bytes -= (long) BSIZE*NINDIR;
				if (fild >= 0)
					lseek(fild, (long) BSIZE*NINDIR, 1);
			} else
				tloop(bn, (long) BSIZE, fild, floptohard);
			break;

		case NADDR-2:
#ifdef SPDEBUG
			printf("(ind)^2 bn=%O\n", bn);
#endif
			if (bn == (daddr_t) 0) {
				if (bytes)
					bytes -= (long) BSIZE*NINDIR*NINDIR;
				if (fild >= 0)
					lseek(fild, (long) BSIZE*NINDIR*NINDIR, 1);
			} else
				tloop(bn, (long) BSIZE*NINDIR, fild, floptohard);
			break;

		case NADDR-1:
#ifdef SPDEBUG
			printf("(ind)^3 bn=%O\n", bn);
#endif
			if (bn != (daddr_t) 0) {
					tloop(bn, (long) BSIZE*NINDIR*NINDIR, fild, floptohard);
			}
			break;
		}
	}
}

tloop(bn, skip, fild, floptohard)
daddr_t bn, skip;
char	floptohard;
{
	register i;
	register struct buf *bp;
	register daddr_t *bap;
	daddr_t nb;
	char	buf[BSIZE];

	Indblocks++;
	/* get next indirect block */
	if (floptohard)
		readtape(buf);
	else
	{
		if (lseek(Fsys, (long) bn * BSIZE, 0) == -1)
		{
			fprintf(stderr, "\nspar: seek error on indirect block");
			done(9);
		}
		if (read(Fsys, buf, BSIZE) == -1) {
			fprintf(stderr, "\nspar: read error on indirect block");
			done(9);
		}
		writetape(buf);
	}

	bap = (daddr_t *) buf;
	for (i = 0; i < NINDIR; i++) {
		if (bap[i] == (daddr_t) 0) {
			if (bytes)
				bytes -= skip;
			if (fild >= 0)
				if (lseek(fild, skip, 1) == -1)
					if (bytes > 0) {
						fprintf(stderr, "\nspar: seek error");
						done(9);
					}
		} else {
#ifdef SPDEBUG
			printf("<bn %O>", bap[i]);
#endif
			if (skip == (long) BSIZE)
				ario(fild, floptohard);
			else if (skip == (long) BSIZE * NINDIR)
				tloop(bap[i], (long) BSIZE, fild, floptohard);
			else if (skip == (long) BSIZE * NINDIR * NINDIR)
				tloop(bap[i], (long) BSIZE * NINDIR, fild, floptohard);
		}
	}
}

ario(fild, floptohard)
char floptohard;
{
	register int	did, l;

	Datablocks++;
	if (bytes >= BSIZE)
		l = BSIZE;
	else
		l = bytes;
	if (floptohard) {
		readtape(Spbuf);
		if (fild >= 0) {
			if ((did = write(fild, Spbuf, l)) != l) {
				fprintf(stderr, "\nspar: write error");
			done(9);
			}
		} else
			did = l;
	} else {
		if ((did = read(fild, Spbuf, l)) != l) {
			fprintf(stderr, "\nspar: read error");
			done(9);
		}
		writetape(Spbuf);
	}
	if (bytes > 0)
		bytes -= (long) did;
	else
		extra++;
}

iexpand(ip, dp)
register struct inode *ip;
register struct dinode *dp;
{
	register char *p1;
	char *p2;
	int i;

	ip->i_mode = dp->di_mode;
	ip->i_nlink = dp->di_nlink;
	ip->i_uid = dp->di_uid;
	ip->i_gid = dp->di_gid;
	ip->i_size = dp->di_size;
	p1 = (char *)ip->i_un.i_i.i_addr;
	p2 = (char *)dp->di_addr;
	for(i=0; i<NADDR; i++) {
	/* MACHINE DEPENDENCY HERE !!! ***********/
		*p1++ = *p2++;
		*p1++ = 0;
		*p1++ = *p2++;
		*p1++ = *p2++;
	}
}

