/* kbsignal.c -- signal handling
 * Copyright (C) 1995, 1996 Markus F.X.J. Oberhumer
 * For conditions of distribution and use, see copyright notice in kb.h 
 */

#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>

#include <kb.h>
#include "_kb.h"


/***********************************************************************
// wrapper for djgpp v1
************************************************************************/

#if defined(_KB_NO_SIGNALS)

const int _kb_nsig = 0;

#if !defined(SIG_ERR)
#  define SIG_ERR	NULL
#endif

int _kb_signal_install(void)
{
	return 0;				/* installation ok */
}

static void my_signal_handler(int signum)
{
	_kb_emergency_remove(1);
	exit(EXIT_SIG(signum));	
}
kb_sighandler_t kb_signal_handler = my_signal_handler;

kb_sighandler_t kb_get_chained_signal(int signum)
{
	return SIG_ERR;
}

kb_sighandler_t kb_set_chained_signal(int signum, kb_sighandler_t h)
{
	return SIG_ERR;
}

#endif /* _KB_NO_SIGNALS */


/***********************************************************************
// IMPORTANT NOTE: check your compiler flags so that no
//                 stack overflow checks will be inserted here !
************************************************************************/

#if !defined(_KB_NO_SIGNALS)

/* NSIG is the highest defined signal number + 1 */
#if !defined(NSIG)
#  if defined(_NSIG)
#    define NSIG	_NSIG
#  elif defined(SIGMAX)
#    define NSIG	(SIGMAX+1)
#  elif defined(_SIGMAX)
#    define NSIG	(_SIGMAX+1)
#  else
#    define NSIG	32
#  endif
#endif

const int _kb_nsig = NSIG;

/* sigaction() gives us more control than signal() */
#if defined(__KB_LINUX)
#  define _KB_USE_SIGACTION_GET
#elif defined(__EMX__)
#  define _KB_USE_SIGACTION_GET
#  define _KB_USE_SIGACTION_SET
#endif


/***********************************************************************
// wrappers for sigaction/signal
************************************************************************/

#if defined(_KB_USE_SIGACTION_GET)
static kb_sighandler_t kb_signal_get_sa(int signum, struct sigaction *sa)
{
	if (sa == NULL)
		return SIG_ERR;
	if (sigaction(signum,NULL,sa) != 0)
		return SIG_ERR;
	return sa->sa_handler;
}
#endif


#if defined(_KB_USE_SIGACTION_SET)
static kb_sighandler_t kb_signal_set_sa(int signum, kb_sighandler_t h,
									struct sigaction *sa)
{
	kb_sighandler_t oldh;

	if (sa == NULL)
		return SIG_ERR;
	if (sigaction(signum,NULL,sa) != 0)
		return SIG_ERR;
	oldh = sa->sa_handler;
	if (oldh == SIG_ERR || h == SIG_ERR)
		return SIG_ERR;
	sa->sa_handler = h;
#if defined(__EMX__)			/* use the emx signal processing model */
	sa->sa_flags &= ~(SA_SYSV);
	sa->sa_flags |= (SA_ACK);
#endif
	if (sigaction(signum,sa,NULL) != 0)
		return SIG_ERR;
	return oldh;
}
#endif


static kb_sighandler_t kb_signal_get(int signum)
{
#if defined(_KB_USE_SIGACTION_GET)
	struct sigaction sa;
	return kb_signal_get_sa(signum,&sa);
#else
	kb_sighandler_t h;

	h = signal(signum,SIG_DFL);					/* get and set handler */
	if (h == SIG_ERR)							/* error in signal() call */
		return SIG_ERR;
	if (h != SIG_DFL)
		if (signal(signum,h) == SIG_ERR)		/* restore handler */
			return SIG_ERR;
	return h;
#endif
}


static kb_sighandler_t kb_signal_set(int signum, kb_sighandler_t h)
{
#if defined(_KB_USE_SIGACTION_SET)
	struct sigaction sa;
	return kb_signal_set_sa(signum,h,&sa);
#else
	return signal(signum,h);
#endif
}


/***********************************************************************
// signal handler
************************************************************************/

#if defined(_KB_USE_SIGACTION_GET)
   static struct sigaction old_sigh[NSIG];
#  define OLD_SIGH(signum)	(old_sigh[signum].sa_handler)
#else
   static kb_sighandler_t old_sigh[NSIG];
#  define OLD_SIGH(signum)	(old_sigh[signum])
#endif
static unsigned char sig_installed[NSIG];


static void my_signal_handler(int signum)
{
	int err;

	/* remove keyboard handler before doing anything else */
	_kb_emergency_remove(1);

	/* sanity check */
	if (signum < 0 || signum >= NSIG)
		return;

#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	/* file I/O is somewhat dangerous within a signal handler ... */
	fprintf(stderr,"libkb: received signal %d\n",signum);
	fflush(stderr);
#endif

	/* restore old signal handler */
	if (sig_installed[signum] != 0x01)
		err = 1;
#if defined(_KB_USE_SIGACTION_GET)
	else if (sigaction(signum,&old_sigh[signum],NULL) != 0)
		err = 1;
#else
	else if (signal(signum,old_sigh[signum]) == SIG_ERR)
		err = 1;
#endif
	else
		err = 0;

#if defined(__EMX__)			/* emx signal processing model */
	signal(signum,SIG_ACK);		/* ack. signal */
#endif

	/* chain old signal handler - this should terminate the program */
	if (!err)
	{
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
		fprintf(stderr,"libkb: chaining signal %d\n",signum);
		fflush(stderr);
#endif
		sig_installed[signum] = 0;		/* no longer active */
		raise(signum);
	}

	/* if we return from here, the application should be aware
	 * that the keyboard handler was removed. */
}


/* improve source code readability */
#define MY_HANDLER		my_signal_handler

/* make accessible */
kb_sighandler_t kb_signal_handler = MY_HANDLER;


/***********************************************************************
// install a signal handler
************************************************************************/

kb_sighandler_t kb_signal(int signum, kb_sighandler_t handler)
{
	kb_sighandler_t h;

	if (signum < 0 || signum >= NSIG || handler == SIG_ERR)
		return SIG_ERR;
	h = kb_signal_get(signum);
	if (h == SIG_ERR)
		return SIG_ERR;

	if (h == MY_HANDLER && sig_installed[signum] != 0x01)
	{
		/* strange: someone installed my handler without my knowledge */
		sig_installed[signum] = 0x01;
	}

	if (h == handler)				/* no change */
		return h;

	if (handler != MY_HANDLER)		/* not my handler */
	{
		if (kb_signal_set(signum,handler) != h)
			return SIG_ERR;
		if (handler == SIG_IGN)
			sig_installed[signum] = 0x20;
		else
			sig_installed[signum] = 0x10;
		return h;
	}

	/* store current signal information */
#if defined(_KB_USE_SIGACTION_GET)
	if (kb_signal_get_sa(signum,&old_sigh[signum]) != h)
		return SIG_ERR;
#else
	old_sigh[signum] = h;
#endif

	/* if the signal is ignored, we ignore it as well */
	if (h == SIG_IGN)
	{
		sig_installed[signum] = 0x02;
		return h;
	}

	if (kb_signal_set(signum,handler) != h)
		return SIG_ERR;

	/* everthing ok */
	sig_installed[signum] = 0x01;
	return h;
}


/***********************************************************************
// install all signal handlers
// note: a handler cannot be installed for SIGKILL, so 
//       don't "kill -9" as this will lock the machine under Linux 
************************************************************************/

/* FIXME: which signals ? */
/* We catch all signals that cause an exit by default (aka almost all) */
static const short signals_to_catch[] = 
{
	SIGABRT,				/* ANSI */
#if defined(SIGALRM)
	SIGALRM,				/* POSIX.1 */
#endif
#if defined(SIGBREAK)
	SIGBREAK,
#endif
#if defined(SIGBUS)
	SIGBUS,		
#endif
	SIGFPE,					/* ANSI */
#if defined(SIGHUP)
	SIGHUP,					/* POSIX.1 */
#endif
	SIGILL,					/* ANSI */
	SIGINT,					/* ANSI */
#if defined(SIGIOT)
	SIGIOT,		
#endif
#if defined(SIGPIPE)
	SIGPIPE,				/* POSIX.1 */
#endif
#if defined(SIGQUIT)
	SIGQUIT,				/* POSIX.1 */
#endif
	SIGSEGV,				/* ANSI */
#if defined(SIGTERM)
	SIGTERM,				/* ANSI */
#endif
#if defined(SIGTRAP)
	SIGTRAP,		
#endif

#if defined(__KB_LINUX)
	SIGPROF, SIGPWR, SIGXCPU, SIGXFSZ, SIGVTALRM,
#endif
};


int _kb_signal_install(void)
{
	int i;
	int saved_errno;

	saved_errno = errno;
	
	for (i = 0; i < HIGH(signals_to_catch); i++)
		kb_signal(signals_to_catch[i],MY_HANDLER);

#if defined(KB_DEBUG)
	fprintf(stderr,"libkb info: installed a signal handler for these signals:\n");
	for (i = 0; i < NSIG; i++)
		if (sig_installed[i] == 0x01)
			fprintf(stderr,"%d ",i);
	fprintf(stderr,"\n");
#endif

	errno = saved_errno;

	/* return ok */
	return 0;
}


/***********************************************************************
//
************************************************************************/

kb_sighandler_t kb_get_chained_signal(int signum)
{
	kb_sighandler_t h;

	if (signum < 0 || signum >= NSIG)
		return SIG_ERR;
	if (sig_installed[signum] != 0x01)
		return SIG_ERR;
	
	h = OLD_SIGH(signum);
	/* sanity check - may not happen */
	if (h == SIG_IGN || h == SIG_ERR || h == MY_HANDLER)
		return SIG_ERR;
	return h;
}


kb_sighandler_t kb_set_chained_signal(int signum, kb_sighandler_t handler)
{
	kb_sighandler_t h;

	/* cannot chain to SIG_IGN */
	if (handler == SIG_IGN || handler == SIG_ERR)
		return SIG_ERR;
	/* cannot chain to the signal handler itself */
	if (handler == MY_HANDLER)
		return SIG_ERR;

	h = kb_get_chained_signal(signum);
	if (h == SIG_ERR)
		return SIG_ERR;

	OLD_SIGH(signum) = handler;
	return h;
}


#endif /* !_KB_NO_SIGNALS */

/*
vi:ts=4
*/

