/* _kblinux.h -- Linux keyboard handler installation
 * Copyright (C) Markus F.X.J. Oberhumer, Harm Hanemaayer and others
 * For conditions of distribution and use, see copyright notice in kb.h 
 */

/* WARNING: this file should *not* be used by applications. It is
   part of the implementation of the keyboard library and is
   subject to change. Applications should only use kb.h.
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/kd.h>
#include <linux/vt.h>


/* IMPORTANT NOTE: you have to update the keyboard manually under Linux 
 *                 !!! there are NO INTERRUPTS !!! 
 *
 * see also: svgalib 1.2.9: src/keyboard/keyboard.c
 * see also: kbd 0.89: src/showkey.c
 */


/***********************************************************************
// 
************************************************************************/

static int get_fd_info(int fd, int *mode, struct termios *t)
{
	int dummy_mode;
	struct termios dummy_termios;

	if (fd < 0)
		return -1;

	if (mode == NULL)
		mode = &dummy_mode;
	if (t == NULL)
		t = &dummy_termios;

	if (ioctl(fd, KDGKBMODE, mode) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: cannot get keyboard mode");
#endif
		return -1;
	}
#if defined(KB_DEBUG)
	fprintf(stderr,"fd: %d ioctl: kbmode=%d\n", fd, *mode);
#endif

	if (tcgetattr(fd, t) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: tcgetattr");
#endif
		return -1;
	}
#if defined(KB_DEBUG) 
	fprintf(stderr,"fd: %d tcgetattr: iflag=0x%04lx lflag=0x%04lx %d %d\n", 
		fd, t->c_iflag, t->c_lflag, t->c_cc[VMIN], t->c_cc[VTIME]);
#endif

	return 0;
}


/***********************************************************************
// 
************************************************************************/

static int _my_kbd_fd = -1;

static int oldkbmode = K_XLATE;
static struct termios oldkbdtermios;

static int keyboard_init_return_fd(void)
{
	struct termios newkbdtermios;
	int r = 0;

	if (_my_kbd_fd == -1)
		_my_kbd_fd = open("/dev/console", O_RDONLY);
	if (_my_kbd_fd == -1)
	{
#if defined(KB_DEBUG)
		perror("libkb: cannot open /dev/console");
#endif
		return -1;
	}
	if (_my_kbd_fd < 0)
		return -1;

	if (get_fd_info(_my_kbd_fd, &oldkbmode, &oldkbdtermios) != 0)
		return -1;

#if defined(KB_DEBUG)
	if (oldkbmode != K_XLATE)	/* what about K_UNICODE ? */
		fprintf(stderr,"libkb info: keyboard not in K_XLATE, strange\n");
#endif

	newkbdtermios = oldkbdtermios;
#if 1
	newkbdtermios.c_iflag = 0;
#else
	newkbdtermios.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
#endif
	newkbdtermios.c_lflag &= ~(ISIG | ICANON | ECHO);
	/* Making these 0 seems to have the desired effect. */
	newkbdtermios.c_cc[VMIN] = 0;
	newkbdtermios.c_cc[VTIME] = 0;

	if (tcsetattr(_my_kbd_fd, TCSAFLUSH, &newkbdtermios) != 0)
	{
		r = -2;
#if defined(KB_DEBUG)
		perror("libkb: tcsetattr");
#endif
	}

	if (ioctl(_my_kbd_fd, KDSKBMODE, K_RAW) != 0)
	{
		r = -2;
#if defined(KB_DEBUG)
		perror("libkb: ioctl KDSKBMODE K_RAW");
#endif
	}

	if (r < 0)
		return r;

#if defined(KB_DEBUG)
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the new settings */
#endif

	return _my_kbd_fd;	/* OK, return fd */
}


/***********************************************************************
// 
************************************************************************/

static void _kb_remove(int final)
{
	int r = 0;

	if (_my_kbd_fd < 0)
		return;

#if defined(KB_DEBUG)
	fprintf(stderr,"_kb_remove 1: fd: %d  final: %d\n", _my_kbd_fd, final);
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the current settings */
#endif

	if (ioctl(_my_kbd_fd, KDSKBMODE, oldkbmode) != 0)
		r |= 1;
	if (tcsetattr(_my_kbd_fd, 0, &oldkbdtermios) != 0)
		r |= 2;
#if defined(KB_DEBUG)
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the current settings */
#endif

	if (final)
	{
		if (close(_my_kbd_fd) != 0)
			r |= 4;
		_my_kbd_fd = -1;
	}

#if defined(KB_DEBUG)
	fprintf(stderr,"_kb_remove 2: error code: %d\n", r);
#endif
}


static int _kb_install(void)
{
	int fd = keyboard_init_return_fd();
	if (fd == -1)
		return -1;
	else if (fd < 0)
	{
		_kb_remove(1);
		return -1;
	}
	else
		return 0;
}



/***********************************************************************
// update the keyboard
************************************************************************/

void kb_update(void)
{
	unsigned char buf[128];
	unsigned char *p;
	int bytesread;

	if (!_kb_mode)
		return;
	
	if (_kb_flags & KB_FLAG_LINUX_NO_VT)
	{
		/* optimized version for no virtual terminal switching */
		while ((bytesread = read(_my_kbd_fd, buf, sizeof(buf))) > 0)
			for (p = buf; p < &buf[bytesread]; p++)
				_my_raw_handler(*p);
	}
	else
	{
		/* enable virtual terminal switching */
		int vt = 0;

		while ((bytesread = read(_my_kbd_fd, buf, sizeof(buf))) > 0)
			for (p = buf; p < &buf[bytesread]; p++)
			{
				int scan = *p & 0x7f;
				int press = !(*p & 0x80);

				/* exactly Alt out of Alt, AltGr */
				if (press && (KB_SHIFT_ALT == (_kb_shift & KB_SHIFT_ANY_ALT))
					&& scan >= KB_SCAN_F1)
				{
					if (scan >= KB_SCAN_F1 && scan <= KB_SCAN_F10)
					{
						vt = scan - KB_SCAN_F1 + 1;
						if (_kb_shift & KB_SHIFT_ANY_SHIFT)
							vt += 12;
						continue;				/* ignore this keypress */
					}
					if (scan >= KB_SCAN_F11 && scan <= KB_SCAN_F12)
					{
						vt = scan - KB_SCAN_F11 + 11;
						if (_kb_shift & KB_SHIFT_ANY_SHIFT)
							vt += 12;
						continue;				/* ignore this keypress */
					}
					if (scan == KB_SCAN_LAST_CONSOLE)
					{
						/* TODO: is there a system call ? */
					}
				}

				_my_raw_handler(*p);
			}

		if (vt > 0)
		{
			/* get number of virtual terminals */
			long first_non_opened = 0;
			if (ioctl(_my_kbd_fd, VT_OPENQRY, &first_non_opened) != 0)
				first_non_opened = 0;

			/* do the switch */
			if (vt < first_non_opened)
				/* This will also generate a signal catched by
 				 * svgalib to restore textmode.
 				 */
				ioctl(_my_kbd_fd, VT_ACTIVATE, vt);
		}
	}
	
	/* Control-C check */
	if (_kb_flags & KB_FLAG_SIGINT)
	{
		if (_kb_key[KB_SCAN_C] && KB_ANY_MASK(_kb_shift, KB_SHIFT_ANY_CONTROL))
			raise(SIGINT);
	}
}


/*
vi:ts=4
*/

