unit DBF;
(* ===========================================================================
 * dbf.dcu - tDBF : A custom data set which uses a flat binary
 *             structured datafile for single client usage only.
 *
 * Author:  Horacio Jamilis
 * Copyright (C) 1998, Terabyte Computacion
 *
 * ===========================================================================
 * ===========================================================================
 *
 * -- a number of bugfixes; 
 * -- BLOB (memo & generdl) datatype support;
 * -- international support; 
 * -- some speed and source optimization;
 *
 * (C) 1999, Alexander Eltsyn ae@ae.inc.ru
 *
 * ===========================================================================
 * v 0.91
 * - Fixed error on deleting records
 * - Added filtering capabilities (work wrong when there are no records within
 *   the filter expresion - Only support expresion with one field like
 *   "NUMFIELD>10" or "TEXTFIELD<='TEST'" or "DATEFIELD=19980626"
 *   (in yyyymmdd format)).
 * v ?.?? - see comments above (A. Eltsyn)
 * ===========================================================================
 *)

interface

uses
  SysUtils, Classes, Db, DsgnIntf;

type

  EDBFError = class (Exception);

  pDateTime = ^TDateTime;
  pBoolean = ^Boolean;
  pInteger = ^Integer;

  PRecInfo = ^TRecInfo;
  TRecInfo = record
    Bookmark: Longint;
    BookmarkFlag: TBookmarkFlag;
  end;

  TdbfHeader = record  { Dbase III + header definition        }
     VersionNumber     :byte;  { version number (03h or 83h ) }
     LastUpdateYear    :byte;  { last update YY MM DD         }
     LastUpdateMonth   :byte;
     LastUpdateDay     :byte;
     NumberOfRecords   :longint; { number of record in database }
     BytesInHeader     :smallint;{ number of bytes in header }
     BytesInRecords    :smallint;{ number of bytes in records }
     ReservedInHeader  :array[1..17] of char;   { reserved bytes in header }
     CPID              :Byte;
     Version           :Word;    { reserved bytes in header }
  end;

  TdbfField = record
     FieldName   :array[1..11] of char; { Name of this record             }
     FieldType   :char;           { type of record - C,N,D,L,etc.         }
     fld_addr    :longint;        { not used }
     Width       :byte;           { total field width of this record      }
     Decimals    :byte;           { number of digits to right of decimal  }
     MultiUser   :smallint;       { reserved for multi user }
     WorkAreaID  :byte;           { Work area ID }
     MUser       :smallint;       { reserved for multi_user }
     SetFields   :byte;           { SET_FIELDS flag }
     Reserved    :array[1..4] of byte;      { 8 bytes reserved }
  end;                           { record starts                         }

Type
  pRecordHeader = ^tRecordHeader;
  tRecordHeader = record
    DeletedFlag : char;
  end;

type
  TDBF = class(TDataSet)
  protected
    FStream     : TStream; // the physical table
    MemoStream  : TStream; // BLOB fields file
    FTableName  : String; // table path and file name
    // record data
    fRecordHeaderSize : Integer;   // The size of the record header
    FRecordCount,                  // current number of record
    FRecordSize,                   // the size of the actual data
    FRecordBufferSize,             // data + housekeeping (TRecInfo)
    FRecordInfoOffset,             // offset of RecInfo in record buffer
    FCurrentRecord,                // current record (0 to FRecordCount - 1)
    BofCrack,                      // before the first record (crack)
    EofCrack: Integer;             // after the last record (crack)
    FIsTableOpen: Boolean;         // status
    FFileWidth,                    // field widths in record
    FFileDecimals,                 // field decimals in record
    FFileOffset: TList;            // field offsets in record
    fReadOnly : Boolean;           // Enhancements
    fStartData : Integer;          // Position in file where data starts
    Function MemoFileName: String;
    Function IsOEMCPTable: Boolean;
    function FFieldType(F : char):TFieldType;
    function FFieldSize(FType:char;FWidth:integer):integer;
  protected
    // TDataSet virtual abstract method
    function AllocRecordBuffer: PChar; override;
    procedure FreeRecordBuffer(var Buffer: PChar); override;
    procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override;
    function GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;
    function GetRecord(Buffer: PChar; GetMode: TGetMode; DoCheck: Boolean): TGetResult; override;
    function GetRecordSize: Word; override;
    procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
    procedure InternalClose; override;
    procedure InternalDelete; override;
    function InternalGetFieldData( RecordBuffer: Pointer;
                                   Field: TField;
                                   Buffer: Pointer):Boolean;
    procedure InternalFirst; override;
    procedure InternalGotoBookmark(Bookmark: Pointer); override;
    procedure InternalHandleException; override;
    procedure InternalInitFieldDefs; override;
    procedure InternalInitRecord(Buffer: PChar); override;
    procedure InternalLast; override;
    procedure InternalOpen; override;
    procedure InternalPost; override;
    procedure InternalSetToRecord(Buffer: PChar); override;
    function IsCursorOpen: Boolean; override;
    procedure SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
    procedure SetBookmarkData(Buffer: PChar; Data: Pointer); override;
    procedure SetFieldData(Field: TField; Buffer: Pointer); override;
    // TDataSet virtual method (optional)
    function GetRecordCount: Integer; override;
    procedure SetRecNo(Value: Integer); override;
    function GetRecNo: Integer; override;
    Procedure WriteHeader;
  private
    Procedure _ReadRecord(Buffer:PChar;IntRecNum:Integer);
    Procedure _WriteRecord(Buffer:PChar;IntRecNum:Integer);
    Procedure _AppendRecord(Buffer:PChar);
    Procedure _SwapRecords(Rec1,REc2:Integer);
    Function _CompareRecords(SortFields:Array of String;Rec1,Rec2:Integer):Integer;
    Function _ProcessFilter(Buffer:PChar):boolean;
  public
    DBFHeader  : TdbfHeader;
    Constructor Create(AOwner:tComponent); override;
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
//    Procedure CreateTable;
    Procedure PackTable;
    Procedure EmptyTable;
    Procedure SortTable(SortFields : Array of String);
    Procedure UnsortTable;
    Function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
  published
    property TableName: string read FTableName write FTableName;
    property ReadOnly : Boolean read fReadOnly write fReadonly default False;
//    property DBFHeader : tDBFHeader read fDBFHeader write fDBFHeader;
    // redeclared data set properties
    property Active;
    property Filter;
    property Filtered;
    property BeforeOpen;
    property AfterOpen;
    property BeforeClose;
    property AfterClose;
    property BeforeInsert;
    property AfterInsert;
    property BeforeEdit;
    property AfterEdit;
    property BeforePost;
    property AfterPost;
    property BeforeCancel;
    property AfterCancel;
    property BeforeDelete;
    property AfterDelete;
    property BeforeScroll;
    property AfterScroll;
    property OnCalcFields;
    property OnDeleteError;
    property OnEditError;
    property OnNewRecord;
    property OnPostError;
  end;

  TDBFBlobStream = Class(TStream)
  private
    Field         : TBlobField;
    Mode          : TBlobStreamMode;
    StartPos      : Integer;
    MaxSize       : Integer;
    Modified      : Boolean;
    FPTSize       : Integer;
    _Size         : Integer;
    _Position     : Integer;
    Function DBF  : TDBF;
    Function DataPos: Integer;
    Procedure UpdateStartPos;
    Procedure UpdateDataHeader;
    Procedure UpdateFPTHeader;
  public
    constructor Create( _Field: TBlobField; _Mode: TBlobStreamMode);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure Truncate;
    property Size: Longint read _Size;
  end;

Var
  ActiveDBFTables : TStringList;

procedure Register;

implementation

uses
  TypInfo, Dialogs, Windows, Forms, Controls, IniFiles;

Const
  dfhVersionNumber = 13;

TYPE
  PBufArray = ^BufArray;
  BufArray = Array[0..0] of Char;

// ****************************************************************************
// Low Level Routines for accessing an internal record

// ____________________________________________________________________________
// TDBF._ReadRecord
Procedure TDBF._ReadRecord(Buffer:PChar;IntRecNum:Integer);
  {-Read a record based on the internal record number (absolute)}
BEGIN
  FStream.Position := FStartData + (FRecordSize * IntRecNum);
 try
  FStream.ReadBuffer(Buffer^, FRecordSize);
 except
 end;
END;

// ____________________________________________________________________________
// TDBF._WriteRecord

Procedure TDBF._WriteRecord(Buffer:PChar;IntRecNum:Integer);
  {-Write a record based on the internal record number (absolute)}
BEGIN
  FStream.Position := FStartData + (FRecordSize * IntRecNum);
  FStream.WriteBuffer (Buffer^, FRecordSize);
END;

// ____________________________________________________________________________
// TDBF._AppendRecord
Procedure TDBF._AppendRecord(Buffer:PChar);
BEGIN
  FStream.Position := FStartData + (FRecordSize * (FRecordCount{+FDeletedCount}));
  FStream.WriteBuffer (Buffer^, FRecordSize);
END;

/////////////////////////////////////////////////
////// Part I:
////// Initialization, opening, and closing
/////////////////////////////////////////////////

// ____________________________________________________________________________
Function TDBF.MemoFileName: String;
Var k : Integer;
  begin
    Result:=ExtractFileName(TableName);
    k:=Pos('.',Result);
    If k>0 then System.Delete(Result,k,Length(Result)-k+1);
    Result:=ExtractFilePath(TableName)+Result;
    If FileExists(Result+'.fpt') or Not FileExists(Result+'.dbt')
      then Result:=Result+'.fpt' 
      else Result:=Result+'.dbt';
  end;
Function TDBF.IsOEMCPTable: Boolean;
  begin
    Result:=(DBFHeader.CPID<>$57)
  end;
// I: open the table/file
procedure TDBF.InternalOpen;
var
  Field       : TField;
  i,j         : integer;
  d           : string;
  MemoExists  : Boolean;
begin
  // check if the file exists
  if not FileExists (FTableName) then
    raise eDBFError.Create ('Open: Table file not found');
  MemoStream:=Nil;
  // create a stream for the file
  if fReadOnly
    then FStream := TFileStream.Create( FTableName, fmOpenRead or fmShareDenyNone)
    else FStream := TFileStream.Create( FTableName, fmOpenReadWrite);
  fStream.ReadBuffer(DBFHeader,SizeOf(TDBFHeader));
  // sets cracks and record position
  BofCrack := -1;
  EofCrack := fRecordCount{+fDeletedCount};
  FCurrentRecord := BofCrack;

  // set the bookmark size
  BookmarkSize := sizeOf (Integer);

  if not (assigned(FFileOffset)) then
    FFileOffset := TList.Create;
  if not (assigned(FFileWidth)) then
    FFileWidth := TList.Create;
  if not (assigned(FFileDecimals)) then
    FFileDecimals := TList.Create;

  // initialize the field definitions
  // (another virtual abstract method of TDataSet)
  InternalInitFieldDefs;

  FRecordInfoOffset := FRecordSize;
  FRecordBufferSize := FRecordSize + sizeof (TRecInfo);

  // if there are no persistent field objects,
  // create the fields dynamically
  if DefaultFields then
    CreateFields;
  // connect the TField objects with the actual fields
  BindFields (True);

  for i := 0 to FieldCount-1 do
    begin
      Field := Fields[i];
      if (Field.DataType = ftFloat) and (Integer(FFileDecimals[i])>0) then
        begin
          d := '0.';
          for j := 1 to Integer(FFileDecimals[i]) do
            d := d + '0';
          (Field as TFloatField).DisplayFormat := d;
        end;
    end;

  // get the number of records and check size
  fRecordCount := DBFHeader.NumberOfRecords;


  // ShowMessage ('InternalOpen: RecCount: ' + IntToStr (FRecordCount));
  MemoExists:=False;
  for i := 0 to FieldDefs.Count - 1 do
    If (FieldDefs[i].DataType in [ftBlob..ftTypedBinary])
      then MemoExists:=True;

  if MemoExists then
    begin
      If ReadOnly then
        begin
          If FileExists(MemoFileName) then
            MemoStream := tFileStream.Create( MemoFileName, fmOpenRead or fmShareDenyNone)
        end else
        begin
          If Not FileExists(MemoFileName) then
            begin
              MemoStream := tFileStream.Create( MemoFileName, fmCreate);
              MemoStream.Free;
            end;
          MemoStream := tFileStream.Create( MemoFileName, fmOpenReadWrite)
        end;
    end;

  // everything OK: table is now open
  FIsTableOpen := True;
  ActiveDBFTables.AddObject(TableName,Self);
end;

// Returns the Type of the field
function TDBF.FFieldType(F : char):TFieldType;
begin
  if F = 'C' then
    FFieldType := ftString
  else if (F = 'N') or (F = 'F') then
    FFieldType := ftFloat
  else if F = 'L' then
    FFieldType := ftBoolean
  else if F = 'D' then
    FFieldType := ftDate
  else if (F='M') then 
    FFieldType := ftMemo
  else if (F='G') then
    FFieldType := ftBlob
  else
    FFieldType := ftUnknown;
end;

function TDBF.FFieldSize(FType:char;FWidth:integer):integer;
begin
  if FType = 'C' then
    FFieldSize := FWidth
  else if (FType = 'N') or (FType = 'F') then
    FFieldSize := 0
  else if FType = 'L' then
    FFieldSize := 0
  else if FType = 'D' then
    FFieldSize := 0
  else if FType = 'M' then
    FFieldSize := 0
//    FFieldSize := 8
  else
    FFieldSize := 0;
end;

// ____________________________________________________________________________
// TDBF.InternalInitFieldDefs
// I: define the fields
procedure TDBF.InternalInitFieldDefs;
var
  Il : Integer;
  TmpFileOffset : Integer;
  NumberOfFields : integer;
  Fld : TDBFField;
  FldName : PChar;
begin
  If fStream=Nil then
    begin
      InternalOpen;
      InternalClose;
      Exit;
    end;
  FieldDefs.Clear;
  FStream.Seek(SizeOf(TDbfHeader),soFromBeginning);
  NumberOfFields := ((DbfHeader.BytesInHeader-sizeof(DbfHeader))div 32);
  if not (assigned(FFileOffset)) then
    FFileOffset := TList.Create;
  FFileOffset.Clear;
  if not (assigned(FFileWidth)) then
    FFileWidth := TList.Create;
  FFileWidth.Clear;
  if not (assigned(FFileDecimals)) then
    FFileDecimals := TList.Create;
  FFileDecimals.Clear;
  TmpFileOffset := 0;
  if (NumberOfFields>0) then
    begin
      for Il:=0 to NumberOfFields-1 do
        begin
          FStream.Read(Fld,SizeOf(Fld));
          GetMem(FldName,Length(Fld.FieldName)+1);
          StrCopy(FldName, PChar(@Fld.FieldName));
          If IsOEMCPTable then OemToChar(FldName,FldName);
          TFieldDef.Create(FieldDefs, FldName,FFieldType(Fld.FieldType){DescribF.DataType},
                           FFieldSize(Fld.FieldType,Fld.Width){DescribF.Size},False,Il+1);
          FreeMem(FldName);
          FFileOffset.Add(Pointer(TmpFileOffset));
          FFileWidth.Add(Pointer(Fld.Width));
          FFileDecimals.Add(Pointer(Fld.Decimals));
          Inc(tmpFileOffset,Fld.Width);
        end;
      fRecordSize := tmpFileOffset+FrecordHeaderSize;
//      FStartData := FStream.Position+1;
      FStartData := DbfHeader.BytesInHeader;
    end;
end;

// ____________________________________________________________________________
// TDBF.InternalClose
// I: close the table/file
procedure TDBF.InternalClose;
Var k : Integer;
begin
  k:=ActiveDBFTables.IndexOfObject(Self);
  If k>=0 then ActiveDBFTables.Delete(k);
  // if required, save updated header
  if (DBFHeader.NumberOfRecords <> fRecordCount) or
    (DBFHeader.BytesInRecords = 0) then
    BEGIN
      DBFHeader.BytesInRecords := fRecordSize;
      DBFHeader.NumberOfRecords := fRecordCount;
      WriteHeader;
    END;

  // disconnet field objects
  BindFields(False);
  // destroy field object (if not persistent)
  if DefaultFields then
    DestroyFields;

  // free the internal list field offsets
  if Assigned(FFileOffset) then
    FFileOffset.Free;
  FFileOffset := nil;
  if Assigned(FFileWidth) then
    FFileWidth.Free;
  FFileWidth := nil;
  if Assigned(FFileDecimals) then
    FFileDecimals.Free;
  FFileDecimals := nil;
  FCurrentRecord := -1;

  // close the file
  FIsTableOpen := False;
  FStream.Free;
  FStream := nil;
  If MemoStream<>Nil then MemoStream.Free;
  MemoStream:=Nil;
end;

// ____________________________________________________________________________
// TDBF.IsCursorOpen
// I: is table open
function TDBF.IsCursorOpen: Boolean;
begin
  Result := FIsTableOpen;
end;

// ____________________________________________________________________________
// TDBF.WriteHeader
procedure TDBF.WriteHeader;
begin
  if fStream <> nil then
    begin
      FSTream.Seek(0,soFromBeginning);
      FStream.WriteBuffer(DBFHeader,SizeOf(TDbfHeader));
    end;
end;

// ____________________________________________________________________________
// TDBF.Create
constructor TDBF.Create(AOwner:tComponent);
BEGIN
  inherited create(aOwner);
  fRecordHeaderSize := SizeOf(tRecordHeader);
END;

// ____________________________________________________________________________
// TDBF.CreateTable
// I: Create a new table/file
(*
procedure TDBF.CreateTable;
var
  Ix : Integer;
  Offs : Integer;
  Fld : TDbfField;
  i : integer;
  MemoExists: Boolean;
begin
  CheckInactive;

  if FileExists (FTableName) and
    (MessageDlg ('File ' + FTableName +
      ' already exists. OK to override?',
      mtConfirmation, [mbYes, mbNo], 0) = mrNo) then
    Exit;
  if FieldDefs.Count = 0 then
    for Ix := 0 to FieldCount - 1 do
      with Fields[Ix] do if FieldKind = fkData then
          FieldDefs.Add(FieldName,DataType,Size,Required);

  MemoExists:=False;
  for Ix := 0 to FieldDefs.Count - 1 do
    If (FieldDefs[Ix].DataType in [ftBlob..ftTypedBinary])
      then MemoExists:=True;

  FStream := TFileStream.Create (FTableName,fmCreate or fmShareExclusive);
  If MemoExists then
   MemoStream := TFileStream.Create (MemoFileName,fmCreate or fmShareExclusive);

  try
    FillChar(fDBFHeader,SizeOf(TDbfHeader),0);
    fDBFHeader.BytesInRecords := 0; // filled later
    fDBFHeader.NumberOfRecords := 0; // empty
    WriteHeader;
    Offs:=0;
    for Ix:=0 to FieldDefs.Count-1 do
      begin
        begin
          FillChar(Fld,SizeOf(Fld),0);
          FillChar(Fld.FieldName, SizeOf(Fld.FieldName),' ');
          for i := 1 to Length(FieldDefs[Ix].Name) do
            Fld.FieldName[i] := FieldDefs[Ix].Name[i];
          If IsOEMCPTable then
            CharToOEMBuff(@Fld.FieldName,@Fld.FieldName,11)
          Case DataType of
            ftWord,
            begin
              Fld.FieldType:='G';
              Fld.Decimals:= 0;
              Fld.Width := Width;
            end;
            ftFloat,
            ftCurrency,
            ftBCD:
            begin
              Fld.FieldType:='G';
              Fld.Decimals:= 0;
              Fld.Width := Width;
            end;
            ftBoolean:
            begin
              Fld.FieldType:='L';
              Fld.Decimals:= 0;
              Fld.Width := 1;
            end;
            ftDate, ftTime, ftDateTime,

            ftBytes, ftVarBytes,
            ftBlob, ftGraphic, ftParadoxOle,
            ftDBaseOle, ftTypedBinary:
            begin
              Fld.FieldType:='G';
              Fld.Decimals:= 0;
              Fld.Width := 10;
            end;
            ftMemo, ftFmtMemo:
            begin
              Fld.FieldType:='M';
              Fld.Decimals:= 0;
              Fld.Width := 10;
            end;
            else
           end;
     Fld.Decimals: byte;
     Fld.Width := FieldDefs[Ix].Size;
          Inc(Offs,Fld.Width);
          FStream.Write(Fld,SizeOf(TDbfField));
        end;
      end;
    fStartData := FStream.Position;
    fDBFHeader.BytesInRecords := Offs;
    FRecordSize := Offs+FRecordHeaderSize;
    WriteHeader;
  finally
    // close the file
    fStream.Free;
    fStream := nil;
    If MemoExists then MemoStream.Free;
    MemoStream := nil;
  end;
end;
*)
// ____________________________________________________________________________
// TDBF.PackTable
Procedure TDBF.PackTable;
var
  TmpDBF      : TDBF;
  SrcBS,DstBS : TStream;
  i           : Integer;
  s           : String;
BEGIN
  CheckInactive;
  if fTableName = '' then
    raise EDBFError.Create('Table name not specified.');
  if not FileExists (FTableName) then
    raise EDBFError.Create('Table '+fTableName+' does not exist.');

  Open;
  DstBS:=TFileStream.Create(ExtractFilePath(TableName)+'$tmp$.dbf',fmCreate);
  SetLength(s,DBFHeader.BytesInHeader);
  fStream.Seek(0,soFromBeginning);
  fStream.Read(s[1],Length(s));
  i:=Integer(Addr(DBFHeader.NumberOfRecords))-Integer(Addr(DBFHeader))+1;
  FillChar(s[i],4,0);
  DstBS.Write(s[1],Length(s));
  DstBS.Free;
  s:='';
  TmpDBF:=TDBF.Create(Self);
  TmpDBF.TableName:=ExtractFilePath(TableName)+'$tmp$.dbf';
  TmpDBF.ReadOnly:=False;
  TmpDBF.Open;

  While Not EOF do
    begin
      TmpDBF.Append;
      For i:=0 to FieldCount-1 do
        If Fields[i].DataType in [ftString,ftFloat,ftBoolean,ftDate]
          then TmpDBF.Fields[i].Assign(Fields[i]) else
        If Fields[i].DataType in [ftBlob..ftTypedBinary] then
          try
            SrcBS:=CreateBlobStream(Fields[i], bmRead);
            DstBS:=TmpDBF.CreateBlobStream(TmpDBF.Fields[i],bmWrite);
            DstBS.CopyFrom(SrcBS,SrcBS.Size);
            DstBS.Free;
            SrcBS.Free;
          except end;
      TmpDBF.Post;
      Next;
    end;
  Close;
  TmpDBF.Close;
  TmpDBF.Free;

  DeleteFile(PChar(TableName));
  DeleteFile(PChar(MemoFileName));
  RenameFile(ExtractFilePath(TableName)+'$tmp$.dbf',TableName);
  RenameFile(ExtractFilePath(TableName)+'$tmp$.fpt',MemoFileName);
END;

Procedure TDBF.EmptyTable;
var
  F : File;
  h : TdbfHeader;
begin
  CheckInactive;
  if fTableName = '' then
    raise EDBFError.Create('Table name not specified.');
  if not FileExists (FTableName) then
    raise EDBFError.Create('Table '+fTableName+' does not exist.');
  try
    AssignFile(F,fTableName); ReSet(F,1);
    BlockRead(F, h, SizeOf(h));
    Seek(F, 0);
    h.NumberOfRecords:=0;
    BlockWrite(F, h, SizeOf(h));
    Seek(F, h.BytesInHeader);
    Truncate(F);
  except end;
  Try CloseFile(F) except end;

  If FileExists(MemoFileName) then SysUtils.DeleteFile(MemoFileName);
end;

function TDBF.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
begin
  Result := TDBFBlobStream.Create(Field as TBlobField, Mode);
end;

// ____________________________________________________________________________
// TDBF._SwapRecords
// Enhancement: Quick swap of two records.  Used primarily for sorting.
Procedure TDBF._SwapRecords(Rec1,REc2:Integer);
VAR
  Buffer1, Buffer2 : PChar;
  Bookmark1, BOokmark2 : TBookmarkFlag;
BEGIN
  Rec1 := Rec1 - 1;
  Rec2 := Rec2 - 1;
  if Rec1 < 0 then Exit;
  if Rec2 < 0 then Exit;
  Buffer1 := AllocRecordBuffer;
  Buffer2 := AllocRecordBuffer;
  _ReadRecord(Buffer1,Rec1);
  _ReadRecord(Buffer2,Rec2);
  Bookmark1 := GetBookmarkFlag(Buffer1);
  Bookmark2 := GetBookmarkFlag(Buffer2);
  SetBookmarkFlag(Buffer1,Bookmark2);
  SetBookmarkFlag(Buffer2,Bookmark1);
  _WriteRecord(Buffer1,Rec2);
  _WriteRecord(Buffer2,Rec1);
  StrDispose(Buffer1);
  StrDispose(Buffer2);
END;

// ____________________________________________________________________________
// TDBF._CompareRecords
// Compare two records.  Returns -1 if REC1 < REC2, 0 if REC1 = REC2, or
// 1 if REC1 > REC2.
Function TDBF._CompareRecords(SortFields:Array of String;Rec1,Rec2:Integer):Integer; FAR;
{-Compare the records Rec1, Rec2 and return -1 if Rec1 < Rec2, 0 if Rec1 = Rec2,
  1 if Rec1 > Rec2 }
VAR
  IX : Integer;

  Function CompareHelper(KeyId:String;Rec1,Rec2:Integer):Integer;
  VAR
    SKey1, SKey2 : String;
    IKey1, IKey2 : Integer;
    fKey1, fKey2 : Double;
    dKey1, dKey2 : tDateTime;
    CompareType : tFieldType;
    KeyField : tField;
  BEGIN
    KeyField := FieldByName(KeyID);
    CompareType := KeyField.DataType;
    Case CompareType of
      ftFloat,
      ftCurrency,
      ftBCD :
        BEGIN
          _ReadRecord(ActiveBuffer,Rec1-1);
          fKey1 := KeyField.AsFloat;
          _ReadRecord(ActiveBuffer,Rec2-1);
          fKey2 := KeyField.AsFloat;
          if fKey1 < fKey2 then
            Result := -1
          else
            if fKey1 > fKey2 then
              Result := 1
            else
              Result := 0;
        END;
      ftSmallInt,
      ftInteger,
      ftWord :
        BEGIN
          _ReadRecord(ActiveBuffer,Rec1-1);
          IKey1 := KeyField.AsInteger;
          _ReadRecord(ActiveBuffer,Rec2-1);
          IKey2 := KeyField.AsInteger;
          if IKey1 < IKey2 then
            Result := -1
          else
            if IKey1 > IKey2 then
              Result := 1
            else
              Result := 0;
        END;
      ftDate,
      ftTime,
      ftDateTime :
        BEGIN
          _ReadRecord(ActiveBuffer,Rec1-1);
          dKey1 := KeyField.AsDateTime;
          _ReadRecord(ActiveBuffer,Rec2-1);
          dKey2 := KeyField.AsDateTime;
          if dKey1 < dKey2 then
            Result := -1
          else
            if dKey1 > dKey2 then
              Result := 1
            else
              Result := 0;
        END;
      else
        BEGIN
          _ReadRecord(ActiveBuffer,Rec1-1);
          SKey1 := KeyField.AsString;
          _ReadRecord(ActiveBuffer,Rec2-1);
          SKey2 := KeyField.AsString;
          if SKey1 < SKey2 then
            Result := -1
          else
            if SKey1 > SKey2 then
              Result := 1
            else
              Result := 0;
        END;
    END;
  END;

BEGIN
  IX := 0;
  REPEAT // Loop through all available sortfields until not equal or no more sort fiels.
    Result := CompareHelper(SortFields[IX],Rec1,Rec2);
    Inc(IX);
  UNTIL (Result <> 0) or (IX > High(SortFields));
END;


// ____________________________________________________________________________
// TDBF.SortTable
// Enhancement: Sort the table by the fields passed.
Procedure TDBF.SortTable(SortFields : Array of String);

  { This is the main sorting routine. It is passed the number of elements and the
    two callback routines. The first routine is the function that will perform
    the comparison between two elements. The second routine is the procedure that
    will swap two elements if necessary } // Source: UNDU #8

  procedure QSort(uNElem: Integer);
  { uNElem - number of elements to sort }

    procedure qSortHelp(pivotP: Integer; nElem: word);
    label
      TailRecursion,
      qBreak;
    var
      leftP, rightP, pivotEnd, pivotTemp, leftTemp: word;
      lNum: Integer;
      retval: integer;
    begin
      TailRecursion:
        if (nElem <= 2) then
          begin
            if (nElem = 2) then

              begin
                rightP := pivotP +1;
                if (_CompareRecords(SortFields,pivotP, rightP) > 0) then
                  _SwapRecords(pivotP, rightP);
              end;
            exit;
          end;
        rightP := (nElem -1) + pivotP;
        leftP :=  (nElem shr 1) + pivotP;
        { sort pivot, left, and right elements for "median of 3" }
        if (_CompareRecords(SortFields,leftP, rightP) > 0) then _SwapRecords(leftP, rightP);
        if (_CompareRecords(SortFields,leftP, pivotP) > 0) then _SwapRecords(leftP, pivotP)

        else if (_CompareRecords(SortFields,pivotP, rightP) > 0) then _SwapRecords(pivotP, rightP);
        if (nElem = 3) then
          begin
            _SwapRecords(pivotP, leftP);
            exit;
          end;
        { now for the classic Horae algorithm }
        pivotEnd := pivotP + 1;
        leftP := pivotEnd;
        repeat
          retval := _CompareRecords(SortFields,leftP, pivotP);
          while (retval <= 0) do
            begin
              if (retval = 0) then
                begin
                  _SwapRecords(LeftP, PivotEnd);
                  Inc(PivotEnd);
                end;
              if (leftP < rightP) then
                Inc(leftP)
              else
                goto qBreak;
              retval := _CompareRecords(SortFields,leftP, pivotP);
            end; {while}
          while (leftP < rightP) do
            begin

              retval := _CompareRecords(SortFields,pivotP, rightP);
              if (retval < 0) then
                Dec(rightP)
              else
                begin
                  _SwapRecords(leftP, rightP);
                  if (retval <> 0) then
                    begin
                      Inc(leftP);
                      Dec(rightP);
                    end;
                  break;
                end;
            end; {while}

        until (leftP >= rightP);
      qBreak:
        if (_CompareRecords(SortFields,leftP, pivotP) <= 0) then Inc(leftP);
        leftTemp := leftP -1;
        pivotTemp := pivotP;
        while ((pivotTemp < pivotEnd) and (leftTemp >= pivotEnd)) do
          begin
            _SwapRecords(pivotTemp, leftTemp);
            Inc(pivotTemp);
            Dec(leftTemp);
          end; {while}
        lNum := (leftP - pivotEnd);
        nElem := ((nElem + pivotP) -leftP);

        if (nElem < lNum) then
          begin
            qSortHelp(leftP, nElem);
            nElem := lNum;
          end
        else
          begin
            qSortHelp(pivotP, lNum);
            pivotP := leftP;
          end;
        goto TailRecursion;
      end; {qSortHelp }

  begin
    if (uNElem < 2) then  exit; { nothing to sort }
    qSortHelp(1, uNElem);
  end; { QSort }


BEGIN
  CheckActive;
  if fReadOnly then
    raise eDBFError.Create ('Dataset must be opened for read/write to perform sort.');
//  if fDataFileHeader.DeletedCount > 0 then
//    BEGIN
//      Close;
//      PackTable;
//      Open;
//    END;
  QSort(FRecordCount {+ fDeletedCount});
  First;
END;

// ____________________________________________________________________________
// TDBF.UnsortTable
// Used to help test the sort routine.  Attempts to generate a random
// dispersment of the records in the dataset.
Procedure TDBF.UnsortTable;
Var
  IX : Integer;
BEGIN
  First;
  Randomize;
  for IX := 0 to RecordCOunt do
    BEGIN
      _SwapRecords(IX,Random(RecordCount+1));
    END;
  First;
END;

////////////////////////////////////////
////// Part II:
////// Bookmarks management and movement
////////////////////////////////////////

// ____________________________________________________________________________
// TDBF.InternalGotoBookmark
// II: set the requested bookmark as current record
procedure TDBF.InternalGotoBookmark (Bookmark: Pointer);
var
  ReqBookmark: Integer;
begin
  ReqBookmark := PInteger (Bookmark)^;
//  ShowMessage ('InternalGotoBookmark: ' +
//    IntToStr (ReqBookmark));
  if (ReqBookmark >= 0) and (ReqBookmark < FRecordCount {+ fDeletedCount}) then
    FCurrentRecord := ReqBookmark
//  else
//    raise eDBFError.Create ('Bookmark ' +
//      IntToStr (ReqBookmark) + ' not found');
end;

// ____________________________________________________________________________
// TDBF.InternalSetToRecord
// II: same as above (but passes a buffer)
procedure TDBF.InternalSetToRecord (Buffer: PChar);
var
  ReqBookmark: Integer;
begin
//  ShowMessage ('InternalSetToRecord');
  ReqBookmark := PRecInfo(Buffer + FRecordInfoOffset).Bookmark;
  InternalGotoBookmark (@ReqBookmark);
end;

// ____________________________________________________________________________
// TDBF.GetBookmarkFlag
// II: retrieve bookmarks flags from buffer
function TDBF.GetBookmarkFlag (
  Buffer: PChar): TBookmarkFlag;
begin
//  ShowMessage ('GetBookmarkFlag');
  Result := PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag;
end;

// ____________________________________________________________________________
// TDBF.SetBookmarkFlag
// II: change the bookmark flags in the buffer
procedure TDBF.SetBookmarkFlag (Buffer: PChar;
  Value: TBookmarkFlag);
begin
//  ShowMessage ('SetBookmarkFlag');
  PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag := Value;
end;

// ____________________________________________________________________________
// TDBF.GetBookmarkData
// II: read the bookmark data from record buffer
procedure TDBF.GetBookmarkData (
  Buffer: PChar; Data: Pointer);
begin
//  ShowMessage ('GetBookmarkData: ' + IntToStr (PRecInfo(Buffer + FRecordInfoOffset).Bookmark));
  PInteger(Data)^ :=
    PRecInfo(Buffer + FRecordInfoOffset).Bookmark;
end;

// ____________________________________________________________________________
// TDBF.SetBookmarkData
// II: set the bookmark data in the buffer
procedure TDBF.SetBookmarkData (
  Buffer: PChar; Data: Pointer);
begin
//  ShowMessage ('SetBookmarkData: ' + IntToStr (PInteger(Data)^));
  If Data<>Nil
    then PRecInfo(Buffer + FRecordInfoOffset).Bookmark := PInteger(Data)^
    else PRecInfo(Buffer + FRecordInfoOffset).Bookmark := 0;
end;

// ____________________________________________________________________________
// TDBF.InternalFirst
// II: Go to a special position before the first record
procedure TDBF.InternalFirst;
begin
  FCurrentRecord := BofCrack;
end;

// ____________________________________________________________________________
// TDBF.InternalLast
// II: Go to a special position after the last record
procedure TDBF.InternalLast;
begin
  EofCrack := FRecordCount {+ fDeletedCount};
  FCurrentRecord := EofCrack;
end;

// ____________________________________________________________________________
// TDBF.GetRecordCount
// II (optional): Record count
function TDBF.GetRecordCount: Longint;
begin
  CheckActive;
  Result := FRecordCount;
end;

// ____________________________________________________________________________
// TDBF.GetRecNo
// II (optional): Get the number of the current record
function TDBF.GetRecNo: Longint;
begin
  UpdateCursorPos;
  if FCurrentRecord < 0 then
    Result := 1
  else
    Result := FCurrentRecord + 1;
end;

// ____________________________________________________________________________
// TDBF.SetRecNo
// II (optional): Move to the given record number
procedure TDBF.SetRecNo(Value: Integer);
begin
  CheckBrowseMode;
  if (Value >= 1) and (Value <= (FRecordCount{+FDeletedCount})) then
  begin
    FCurrentRecord := Value - 1;
    Resync([]);
  end;
end;

//////////////////////////////////////////
////// Part III:
////// Record buffers and field management
//////////////////////////////////////////

// ____________________________________________________________________________
// TDBF.GetRecordSize
/// III: Determine the size of each record buffer in memory
function TDBF.GetRecordSize: Word;
begin
  Result := FRecordSize; // data only
end;

// ____________________________________________________________________________
// TDBF.AllocRecordBuffer
/// III: Allocate a buffer for the record
function TDBF.AllocRecordBuffer: PChar;
begin
  Result := StrAlloc(FRecordBufferSize+1);
end;

// ____________________________________________________________________________
// TDBF.InternalInitRecord
// III: Initialize the record (set to zero)
procedure TDBF.InternalInitRecord(Buffer: PChar);
begin
  FillChar(Buffer^, FRecordBufferSize, 32);
end;

// ____________________________________________________________________________
// TDBF.FreeRecordBuffer
// III: Free the buffer
procedure TDBF.FreeRecordBuffer (var Buffer: PChar);
begin
  StrDispose(Buffer);
end;

// ____________________________________________________________________________
// TDBF.GetRecord
// III: Retrieve data for current, previous, or next record
// (eventually moving to it) and return the status
function TDBF.GetRecord(Buffer: PChar;
  GetMode: TGetMode; DoCheck: Boolean): TGetResult;
var
  Acceptable : Boolean;
begin
  result := grOk;
  if FRecordCount < 1 then
    Result := grEOF
  else
    repeat
      case GetMode of
        gmCurrent :
          begin
            // ShowMessage ('GetRecord Current');
            if (FCurrentRecord >= FRecordCount{+fDeletedCount}) or
                (FCurrentRecord < 0) then
              Result := grError;
          end;
        gmNext :
          begin
            if (fCurrentRecord < (fRecordCount{+fDeletedCount})-1) then
              Inc (FCurrentRecord)
            else
              Result := grEOF;
          end;
        gmPrior :
          begin
           if (fCurrentRecord > 0) then
              Dec(fCurrentRecord)
           else
              Result := grBOF;
          end;
      end;
      // fill record data area of buffer
      if Result = grOK then
        begin
          _ReadRecord(Buffer, fCurrentRecord );
          ClearCalcFields(Buffer);
          GetCalcFields(Buffer);
          with PRecInfo(Buffer + FRecordInfoOffset)^ do
            begin
              BookmarkFlag := bfCurrent;
              Bookmark := FCurrentRecord;
            end;
        end
      else
        if (Result = grError) and DoCheck then
          raise eDBFError.Create('GetRecord: Invalid record');
      Acceptable := pRecordHeader(Buffer)^.DeletedFlag <> '*';
      if Filtered and Acceptable then
        Acceptable :=  _ProcessFilter(Buffer);
      if (GetMode=gmCurrent) and Not Acceptable then
        Result := grError;
    until (Result <> grOK) or Acceptable;
//  if ((Result=grEOF)or(Result=grBOF)) and Filtered and not (_ProcessFilter(Buffer)) then
//    Result := grError;

  if (Result in [grEOF, grBOF, grError]) then
    FillChar(Buffer^, FRecordSize,' ');

end;

// ____________________________________________________________________________
// TDBF.InternalPost
// III: Write the current data to the file
procedure TDBF.InternalPost;
begin
  CheckActive;
  if State = dsEdit then
    begin
      // replace data with new data
      _WriteRecord (ActiveBuffer, fCurrentRecord);
    end
  else
    begin
      // always append
      InternalLast;
      pRecordHeader(ActiveBuffer)^.DeletedFlag := ' ';
      _AppendRecord(ActiveBuffer);
      Inc (FRecordCount);
    end;
end;

// ____________________________________________________________________________
// TDBF.InternalAddRecord
// III: Add the current data to the file
procedure TDBF.InternalAddRecord(Buffer:Pointer; Append:Boolean);
begin
  // always append
  InternalLast;
  // add record at the end of the file
  pRecordHeader(ActiveBuffer)^.DeletedFlag := ' ';
  _AppendRecord(ActiveBuffer);
  Inc (FRecordCount);
end;

// ____________________________________________________________________________
// TDBF.InternalDelete
// III: Delete the current record
procedure TDBF.InternalDelete;
begin
  CheckActive;
  PChar(ActiveBuffer)^ := '*';
  _WriteRecord(ActiveBuffer,fCurrentRecord);
  Resync([]);
end;

// ____________________________________________________________________________
// TDBF.InternalGetFieldData
// III: Move data from RecordBuffer to field
function TDBF.InternalGetFieldData( RecordBuffer: Pointer;
                                    Field:TField;
                                    Buffer:Pointer):Boolean;
var
  FieldOffset: Integer;
  S : string;
  Buf2 : PChar;
  i,l : integer;
  D : Double;
  n : integer;
  T : TDateTime;
  j : integer;
  OldDateFormat : string;
  bf : TBooleanField;
  bool : Boolean;
begin
  Result := False;
  Buf2 := RecordBuffer;
  if (FRecordCount>0) and (Field.FieldNo > 0) and
     (Assigned(Buf2))  then
    begin
      FieldOffset := Integer(FFileOffset[Field.FieldNo-1])+FRecordHeaderSize;
      if Field.DataType = ftString then
        begin
          l := Integer(FFileWidth[Field.FieldNo-1]);
          S := '';
          i := 0;
          While (i<l) do
            begin
              S := S+pChar(Buf2+FieldOffset+i)^;
              inc(i);
            end;
          If Length(s)>l then SetLength(S,l);
          Result := (Trim(s)<>'');
          If Assigned(Buffer) then
            begin
              s := Trim(s)+#0; l := Length(s);
              If IsOEMCPTable
                then OEMToCharBuff(PChar(@S[1]), Buffer,l)
                else Move( S[1], Buffer^, l);
            end;
        end
      else if Field.DataType = ftFloat then
        begin
          n := Integer(FFileWidth[Field.FieldNo-1]);
          S := '';
          for i := FieldOffset to FieldOffset+n-1 do
            S := S+pChar(Buf2+i)^;
          S := Trim(S);
          if S='' then
            Result := False
          else
            begin
              if (Pos('.',S) > 0) and (DecimalSeparator <> '.') then
                S[Pos('.',S)] := DecimalSeparator;
              Result := (Trim(s)<>'');
              try
                D := StrToFloat(S);
              except
                D := 0;
                Result := False;
              end;
              If Assigned(Buffer) then PDouble(Buffer)^ := D;
            end;
        end
      else if Field.DataType = ftDate then
        begin
          S := '';
          for j := 0 to 7 do
            S := S + pChar(Buf2+FieldOffset+j);
          SetLength(S,8);
          if (trim(S) = '') or (S='00000000') then
            begin
              If Assigned(Buffer) then pInteger(Buffer)^ := Trunc(Now)+693594;
              Result := False;
            end
          else
            begin
              S := Copy(S,7,2)+DateSeparator+Copy(S,5,2)+DateSeparator+Copy(S,1,4);
              OldDateFormat := ShortDateFormat;
              ShortDateFormat := 'dd/mm/yyyy';
              t := StrToDate(S);
              ShortDateFormat := OldDateFormat;
              j := Trunc(pDouble(@t)^)+693594;
              If Assigned(Buffer) then pInteger(Buffer)^ := j;
              result := True;
            end;
        end
      else if Field.DataType = ftBoolean then
        begin
          bf:=TBooleanField.Create(Nil);
          If Assigned(Buffer) then FillChar(Buffer^,bf.DataSize,0);
          bf.Free;
          Result := True;  Bool:=False;
          if PChar(Buf2+FieldOffset)^ in ['S','T','Y']
            then Bool := True else
          if PChar(Buf2+FieldOffset)^ in ['N','F']
            then Bool := False
            else Result := False;
          If Assigned(Buffer) then pBoolean(Buffer)^ := Bool
        end
      else If Field.DataType in [ftBlob..ftTypedBinary] then
       begin
         Result := False;
       end
      else
        begin
          ShowMessage ('very bad error in get field data');
          Result := False;
        end;
    end;
end;

// ____________________________________________________________________________
// TDBF.GetFieldData
// III: Move data from record buffer to field
function TDBF.GetFieldData(Field:TField; Buffer:Pointer):Boolean;
begin
  Result:=InternalGetFieldData(ActiveBuffer, Field, Buffer);
end;

// ____________________________________________________________________________
// TDBF.SetFieldData
// III: Move data from field to record buffer
procedure TDBF.SetFieldData(Field: TField; Buffer: Pointer);
var
  FieldOffset: Integer;
  Buf2 : PChar;
  l,i,n:integer;
  S : string;
  D : TDateTime;
  j : integer;
begin
  Buf2 := ActiveBuffer;
  if (Field.FieldNo >= 0) and (Assigned(Buf2)) then
    begin
      FieldOffset := Integer(FFileOffset[Field.FieldNo-1])+FRecordHeaderSize;
      if Field.DataType = ftString then
        begin
          l := Integer(FFileWidth[Field.FieldNo-1]);
          If Assigned(Buffer) then S := Copy(StrPas(Buffer),1,l) else S:='';
          i:=Length(s); SetLength(s,l);
          For i:=i+1 to l do s[i]:=' ';
          If IsOEMCPTable
            then CharToOEMBuff(PChar(@S[1]),PChar(Buf2+FieldOffset),l)
            else Move(S[1],PChar(Buf2+FieldOffset)^,l);
        end
      else if Field.DataType = ftFloat then
        begin
          n := Integer(FFileWidth[Field.FieldNo-1]);
          If Assigned(Buffer)
            then Str(pDouble(Buffer)^:n:Integer(FFileDecimals[Field.FieldNo-1]),S)
            else s:='';
          while Length(S)<n do S :=  ' '+S;
          if Assigned(Buffer) and
             (Pos(DecimalSeparator,S) > 0) and (DecimalSeparator <> '.') then
            S[Pos(DecimalSeparator,S)] := '.';
          CopyMemory(Pchar(Buf2+FieldOffset),PChar(S),n);
        end
      else if Field.DataType = ftDate then
        begin
          If Assigned(Buffer) then
            begin
              j := pInteger(Buffer)^-693594;
              pDouble(@d)^ := j;
              S := FormatDateTime('yyyymmdd',d);
            end else S:='         ';
          Move( S[1], pChar(Buf2+FieldOffset)^, 8);
        end
      else if Field.DataType = ftBoolean then
        begin
          If Not Assigned(Buffer)
            then PChar(Buf2+FieldOffset)^ := ' '
            else if pBoolean(Buffer)^
                   then PChar(Buf2+FieldOffset)^ := 'T'
                   else PChar(Buf2+FieldOffset)^ := 'F';
        end
      else
        ShowMessage ('very bad error in setfield data');

      DataEvent (deFieldChange, Longint(Field));
    end;
end;

// ____________________________________________________________________________
// TDBF.InternalHandleException
// default exception handling
procedure TDBF.InternalHandleException;
begin
  // standard exception handling
  Application.HandleException(Self);
end;

Function TDBF._ProcessFilter(Buffer:PChar):boolean;
var
  PosComp   : integer;
  FieldPos  : integer;
  k         : integer;
  TestValue : String;
  FieldText : string;
  FieldR,TestR : Double;
  b            : Array[0..257] of Char;
begin
  Result:=True;
  k:=0;
  PosComp := Pos('>',Filter);
  if PosComp=0 then PosComp := Pos('<',Filter);
  if (PosComp>0) and (Length(Filter)>PosComp) and
     (Filter[PosComp+1]='=') then k:=PosComp+2;
  if PosComp=0 then PosComp := Pos('=',Filter);
  if PosComp=0 then Exit;
  If k=0 then k:=PosComp+1;

  FieldPos := FieldDefs.IndexOf(Trim(Copy(Filter,1,PosComp-1)));
  if FieldPos < 0 then Exit;

  TestValue := Trim(Copy(Filter,k,Length(Filter)-PosComp));
  If (TestValue>'') and (TestValue[1] in ['"','''']) then
    begin
      System.Delete(TestValue,1,1);
      k:=Length(TestValue);
      If k>0 then SetLength(TestValue, k-1);
    end;
try
  if FieldDefs.Items[FieldPos].DataType = ftString then
    begin
      If Not InternalGetFieldData(Buffer,Fields[FieldPos],@b) then b[0]:=#0;
      FieldText:=Trim(StrPas(b));

      if Filter[PosComp]='='
        then Result := (FieldText=TestValue) else
      if Filter[PosComp]='>' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldText>=TestValue)
            else Result := (FieldText>TestValue);
        end else
      if Filter[PosComp]='<' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldText<=TestValue)
            else Result := (FieldText<TestValue);
        end;
    end else
  if FieldDefs.Items[FieldPos].DataType = ftFloat then
    begin
      Val(TestValue,TestR,k);
      If k<>0 then TestR:=0;
      If Not InternalGetFieldData(Buffer,Fields[FieldPos],@FieldR) then
        begin Result:= (TestValue=''); Exit end;

      if Filter[PosComp]='='
         then Result := (FieldR=TestR) else
      if Filter[PosComp]='>' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldR>=TestR)
            else Result := (FieldR>TestR);
        end else
      if Filter[PosComp]='<' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldR<=TestR)
            else Result := (FieldR<TestR);
        end;
    end else
  if FieldDefs.Items[FieldPos].DataType = ftDate then
    begin
      If Not InternalGetFieldData(Buffer,Fields[FieldPos],@FieldR) then
        begin Result:= (TestValue=''); Exit end;
      FieldText := FormatDateTime('yyyymmdd',FieldR);
      if Filter[PosComp]='='
        then Result := (FieldText=TestValue) else
      if Filter[PosComp]='>' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldText>=TestValue)
            else Result := (FieldText>TestValue);
        end else
      if Filter[PosComp]='<' then
        begin
          if Filter[PosComp+1]='='
            then Result := (FieldText<=TestValue)
            else Result := (FieldText<TestValue);
        end;
    end;
except
end;
end;
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
Function TDBFBlobStream.DBF: TDBF;
begin
  Result := Field.DataSet as TDBF;
end;
//////////////////////////////////////////////////////////////////////////
Function TDBFBlobStream.DataPos: Integer;
begin
  Result:=StartPos*64+8;
end;
//////////////////////////////////////////////////////////////////////////
Procedure TDBFBlobStream.UpdateStartPos;
Var
   p   : PChar;
   i,k : Integer;
begin
  p := DBF.ActiveBuffer;
  if (Field.FieldNo > 0) and Assigned(p)  then
    begin
      Inc(p, DBF.FRecordHeaderSize);
      Inc(p, Integer(DBF.FFileOffset[Field.FieldNo-1]));
      If Size<=0 then FillChar( p^,10,' ') else
        begin
          Inc(p, 9 ); k:=StartPos;
          For i:=1 to 10 do
            begin
              p^:=Char((k mod 10)+48); k:=k div 10;
              Dec(p);
            end;
        end;
    end;
end;
//////////////////////////////////////////////////////////////////////////
Procedure TDBFBlobStream.UpdateDataHeader;
Var
   i, Sz : Integer;
   b     : Packed Array[0..7] of Byte;
begin
  If Not Assigned(DBF.MemoStream) then Exit;
  Sz:=Size;
  FillChar(b,8,0);
  b[3]:=1;
  For i:=7 downto 4 do
    begin
      b[i]:=Sz mod 256;
      Sz:=Sz div 256;
    end;
  DBF.MemoStream.Seek( DataPos-8, soFromBeginning );
  DBF.MemoStream.Write( b, 8 );
end;
//////////////////////////////////////////////////////////////////////////
Procedure TDBFBlobStream.UpdateFPTHeader;
Var
   Sz    : Integer;
   b     : Packed Array[0..511] of Byte;
begin
  If Not Assigned(DBF.MemoStream) then Exit;
  Sz:=(DBF.MemoStream.Size+63) div 64;
  If Sz<8 then Sz:=8;
  If Sz=FPTSize then Exit;
    begin
      FPTSize:=Sz;
      FillChar(b,SizeOf(b),0);
      Move(Sz,b,4);
      b[7]:=b[0]; b[0]:=b[3]; b[3]:=b[7];
      b[7]:=b[1]; b[1]:=b[2]; b[2]:=b[7];
      b[7]:=64;
      DBF.MemoStream.Seek( 0, soFromBeginning );
      DBF.MemoStream.Write( b, 512);
    end;
end;
//////////////////////////////////////////////////////////////////////////
constructor TDBFBlobStream.Create( _Field: TBlobField; _Mode: TBlobStreamMode);
Var
   p     : PChar;
   i     : Integer;
   b     : Packed Array[0..7] of Byte;
begin
  Inherited Create;
  Field := _Field;
  Mode:=_Mode;
  If Not DBF.Active then DatabaseError('Table is not opened');
  if Not (Field.DataType in [ftBlob..ftTypedBinary]) or
     DBF.ReadOnly and (Mode<>bmRead)
    then DatabaseError('Invalid blob field access mode ('+Field.DisplayName+')');

  p := DBF.ActiveBuffer;
  if (Field.FieldNo > 0) and Assigned(p) and Assigned(DBF.MemoStream) then
    begin
      Inc( p, DBF.FRecordHeaderSize);
      Inc( p, Integer(DBF.FFileOffset[Field.FieldNo-1]));
      StartPos:=0;
      For i:=1 to 10 do
        begin
          StartPos:=StartPos*10;
          If p^ in ['1'..'9'] then StartPos:=StartPos+Ord(p^)-48;
          Inc(p);
        end;
      FPTSize:=(DBF.MemoStream.Size+63) div 64;
      UpdateFPTHeader;
      If StartPos<=0 then
        begin
          StartPos:=FPTSize;
          _Size:=0; MaxSize:=2147483647;
          UpdateStartPos;
        end else
        begin
          DBF.MemoStream.Seek( DataPos-8, soFromBeginning );
          DBF.MemoStream.Read(b,8);
          _Size:=0;
          For i:=4 to 7 do _Size:=_Size*256+Integer(b[i]);
          If DataPos+Size+10<DBF.MemoStream.Size
            then MaxSize:=_Size else MaxSize:=2147483647;
        end;
    end;
  if Mode = bmWrite then Truncate;
end;
//////////////////////////////////////////////////////////////////////////
destructor TDBFBlobStream.Destroy;
begin
  if Modified and Assigned(DBF.MemoStream) then
  try
    UpdateFPTHeader;
    UpdateStartPos;
    If Size>0 then UpdateDataHeader;
    DBF.DataEvent(deFieldChange, Longint(Field));
  except
    Application.HandleException(Self);
  end;
  Inherited;
end;
//////////////////////////////////////////////////////////////////////////
function TDBFBlobStream.Read(var Buffer; Count: Longint): Longint;
begin
  if (Mode=bmWrite) then
    DatabaseError('Invalid blob field access mode ('+Field.DisplayName+')');
  If Assigned(DBF.MemoStream) then
    begin
      If Count>Size-Position then Count:=Size-Position;
      DBF.MemoStream.Seek( DataPos+Position, soFromBeginning );
      Result:=DBF.MemoStream.Read(Buffer, Count );
      Position:=Position+Result;
      if Field.Transliterate and DBF.IsOEMCPTable then
        OEMToCharBuff( @Buffer, @Buffer, Result);
    end else
    Result:=0;
end;
//////////////////////////////////////////////////////////////////////////
function TDBFBlobStream.Write(const Buffer; Count: Longint): Longint;
Var
  s   : ANSIString;
begin
  if (Mode=bmRead) then
    DatabaseError('Invalid blob field access mode ('+Field.DisplayName+')');
  If Not Assigned(DBF.MemoStream) then
    begin
      Result:=0;
      Exit;
    end;

  Modified:=True;
  If Count>MaxSize-Position then
    begin
      SetLength(s,Size);
      DBF.MemoStream.Seek( DataPos, soFromBeginning );
      DBF.MemoStream.Read(s[1],Size);
      StartPos:=(DBF.MemoStream.Size+63) div 64;
      DBF.MemoStream.Seek( DataPos, soFromBeginning );
      _Size:=DBF.MemoStream.Write(s[1],Size);
      s:='';
      MaxSize:=2147483647;
      UpdateStartPos;
      UpdateFPTHeader
    end;

  SetLength(s,Count);
  if Field.Transliterate and DBF.IsOEMCPTable
    then CharToOEMBuff( @Buffer, @s[1], Count)
    else Move( Buffer,s[1], Count);

  DBF.MemoStream.Seek( DataPos+Position, soFromBeginning );
  Result:=DBF.MemoStream.Write( s[1], Count );
  If Size<Position+Result then _Size:=Position+Result;

  Position:=Position+Result;
end;
//////////////////////////////////////////////////////////////////////////
function TDBFBlobStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  If Not Assigned(DBF.MemoStream) then _Position:=0 else
  case Origin of
    0: _Position := Offset;
    1: _Position := _Position+Offset;
    2: _Position := _Size + Offset;
  end;
  Result := _Position;
end;
//////////////////////////////////////////////////////////////////////////
procedure TDBFBlobStream.Truncate;
begin
  _Size:=Position; Modified:=True;
end;
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
procedure Register;
begin
  If ActiveDBFTables=Nil then ActiveDBFTables:=TStringList.Create;
  RegisterComponents('Data Access', [TDBF]);
end;


Initialization
  ActiveDBFTables:=TStringList.Create;
Finalization
  ActiveDBFTables.Free;
end.
