unit xqbase;

{*******************************************************}
{                                                       }
{       Base classes used in TxQuery dataset            }
{                                                       }
{       Copyright (c) 1999-2000 Alfonso moreno          }
{                                                       }
{     Written by:                                       }
{       Alfonso Moreno                                  }
{       Hermosillo, Sonora, Mexico.                     }
{       Internet:  gismap@hmo.megared.net.mx            }
{                  luisarvayo@yahoo.com                 }
{                  inconmap@prodigy.net.mx              }
{       http://www.sigmap.com/txquery.htm               }
{                                                       }
{*******************************************************}

{$I XQ_FLAG.INC}
interface

uses
  SysUtils,
  Windows,
  Messages,
  classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  StdCtrls,
  prExprQ,
  Db,
  xqmiscel,
  SparsArr
{$IFDEF WITHBDE}
  , DBTables
  , bde
{$endIF}
   ;

const
  SQuote = ['''', '"'];
  NBoolean: array[Boolean] of String[5] = ('FALSE', 'TRUE');
  MAX_INDEXED_FIELDS = 10;

type
{-------------------------------------------------------------------------------}
{                          Main exception                                       }
{-------------------------------------------------------------------------------}

  ExQueryError = class(Exception);

{-------------------------------------------------------------------------------}
{                          Some base types needed                               }
{-------------------------------------------------------------------------------}

  PFloat    = ^Double;
  PInteger  = ^Integer;
  PWordBool = ^WordBool;


{-------------------------------------------------------------------------------}
{                          Forward declarations                                 }
{-------------------------------------------------------------------------------}

  TMainExpr              = class;
  TCreateFields          = class;
  TColumnList            = class;
  TTableList             = class;
  TOrderByList           = class;
  TJoinOnExprList        = class;
  TJoinOnList            = class;
  TUpdateList            = class;
  TWhereOptimizeList     = class;
  TCreateTableList       = class;
  TInsertList            = class;
  TSrtFields             = class;
  TxqSortList            = class;
  TAggregateList         = class;
  TMemMapFile            = class;
  TReferencedDataSetList = class;
  TNestedList            = class;


{-------------------------------------------------------------------------------}
{                          Declare enumerations                                 }
{-------------------------------------------------------------------------------}

  TAggregateKind = ( akSUM,
                     akAVG,
                     akMIN,
                     akMAX,
                     akCOUNT );

  TRelationalOperator = ( ropBETWEEN,
                          ropGT,
                          ropGE,
                          ropLT,
                          ropLE,
                          ropNEQ );

  TJoinKind = ( jkInnerJoin,
                jkLeftOuterJoin,
                jkRightOuterJoin,
                jkFullOuterJoin );

  TSubQueryKind = ( skAny,
                    skAll );

  TSQLStatement = ( ssSelect,
                    ssUpdate,
                    ssDelete,
                    ssInsert,
                    ssMerge,
                    ssCreateTable,
                    ssCreateIndex,
                    ssDropTable,
                    ssDropIndex );
                    
  TMemorizeJoin = ( mjNone,
                    mjUsingMemory,
                    mjUsingFile );

{-------------------------------------------------------------------------------}
{                  Defines TAggregateItem                                       }
{-------------------------------------------------------------------------------}

  TAggregateItem = class
  private
     FAggregateList : TAggregateList;        { belongs to }
     FAggregateStr  : String;                { the expression as it is issued in the SQL statement }
     FColIndex      : Integer;               { the index in the ColumnList where this aggregate is temporary evaluated }
     FAggregate     : TAggregateKind;        { used if ColumnKind = ckAggregate                          }
     FIsDistinctAg  : Boolean;               { syntax is SELECT COUNT(distinct pricelist) FROM customer }
     FSparseList    : TAggSparseList;  { a sparse array for the aggregates values in every record }
  public
     constructor Create(AggregateList : TAggregateList);
     destructor Destroy; override;
     property AggregateStr: String read FAggregateStr write FAggregateStr;
     property ColIndex : Integer read FColIndex write FColIndex;
     property Aggregate: TAggregateKind read FAggregate write FAggregate;
     property IsDistinctAg: Boolean read FIsDistinctAg write FIsDistinctAg;
     property SparseList: TAggSparseList read FSparseList write FSparseList;
  end;

{-------------------------------------------------------------------------------}
{                  Defines TAggregateList                                       }
{-------------------------------------------------------------------------------}

  TAggregateList = class
  private
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TAggregateItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TAggregateItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     procedure Assign(AggregateList : TAggregateList);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TAggregateItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  SELECT section data                                          }
{-------------------------------------------------------------------------------}

  TColumnItem = class
  private
     FColumnList   : TColumnList;      { the column list that this column item belongs to }
     FColumnExpr   : String;           { the expression in this column }
     FAsAlias      : String;           { column Name (used later in where, group by, etc), default is FieldName   }
                                       { also is used as title in browse                                          }
     FIsAsExplicit  : Boolean;         { explicity defined in SQL (ex.: SELECT Sales+Bonus As TotalSales FROM...) }
     FResolver      : TMainExpr;       { object used to evaluate ColumnExpr                                       }
     FAutoFree      : Boolean;         { Auto free FResolver class                                                }
     FIsTemporaryCol: Boolean;         { Is a column used for some calculations (temporarily)                     }
     FCastType      : Word;            { column must be casted to this type (CAST expr AS DATE)                   }
     FCastLen       : Word;            { only used if casting to CHAR(n) n=CastLen                                }
     FAggregateList : TAggregateList;  { the list of aggregates for this column: SUM(expression) / SUM(expression) }
  public
     constructor Create( ColumnList: TColumnList );
     destructor Destroy; override;

     property ColumnExpr: String read FColumnExpr write FColumnExpr;
     property AsAlias: String read FAsAlias write FAsAlias;
     property IsAsExplicit: Boolean read FIsAsExplicit write FIsAsExplicit;
     property IsTemporaryCol: Boolean read FIsTemporaryCol write FIsTemporaryCol;
     property CastType: Word read FCastType write FCastType;
     property CastLen: Word read FCastLen write FCastLen;
     property Resolver: TMainExpr read FResolver write FResolver;
     property AutoFree: Boolean read FAutoFree write FAutoFree;
     property AggregateList: TAggregateList read FAggregateList write FAggregateList;
  end;

  TColumnList = class
  private
     FItems: TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TColumnItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TColumnItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     procedure DeleteAggregate(RecNo : Integer);
     procedure SortAggregateWithList(SortList : TxqSortList);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TColumnItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  FROM section data                                            }
{-------------------------------------------------------------------------------}

  TTableItem = class
  private
     FTableList   : TTableList;
     FTableName   : String;
     FAlias       : String;
     FDataSet     : TDataSet;       { the attached dataset }
  public
     constructor Create(TableList: TTableList);

     property TableName: String read FTableName write FTableName;
     property Alias: String read FAlias write FAlias;
     property DataSet: TDataSet read FDataSet write FDataSet;
  end;

  TTableList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TTableItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TTableItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     function IndexOfDataSet(DataSet: TDataSet): Integer;

     property Count: Integer read GetCount;
     property Items[Index: Integer]: TTableItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{   ORDER BY section data - used in ORDER BY and GROUP BY                       }
{-------------------------------------------------------------------------------}

  TOrderByItem = class
  private
     FOrderByList : TOrderByList;
     FColIndex    : Integer;
     FAlias       : String;          { field name used to order                                   }
     FDesc        : Boolean;          { Descending? default = false = Ascending;                   }
  public
     constructor Create(OrderByList: TOrderByList);
     property ColIndex: Integer read FColIndex write FColIndex ;
     property Alias: String read FAlias write FAlias ;
     property Desc: Boolean read FDesc write FDesc;
  end;

  TOrderByList = class
     FItems: TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TOrderByItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TOrderByItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TOrderByItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  JOIN ON section data                                         }
{-------------------------------------------------------------------------------}

  TJoinOnExprItem = class
  private
     FJoinOnExprList : TJoinOnExprList;  { belongs to }
     FExpression     : String;
     FResolver       : TMainExpr;
  public
     constructor Create(JoinOnExprList: TJoinOnExprList);
     destructor Destroy; override;

     property Expression: String read FExpression write FExpression;
     property Resolver: TMainExpr read FResolver write FResolver;
  end;

  TJoinOnExprList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TJoinOnExprItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TJoinOnExprItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TJoinOnExprItem read GetItem; default;
  end;

  TJoinOnItem = class
  private
     FJoinOnList     : TJoinOnList;
     FJoinKind       : TJoinKind;
     FRelOperator    : TRelationalOperator;
     FLeftJoinOn     : TJoinOnExprList;
     FRightJoinOn    : TJoinOnExprList;
     FAndJoinOn      : TJoinOnExprList;
     FWhereStr       : String;
     FWhereResolver  : TMainExpr;
     FMasterDataset  : TDataSet;
     FDetailDataset  : TDataSet;
  public
     constructor Create(JoinOnList: TJoinOnList);
     destructor Destroy; override;
     property JoinKind: TJoinKind read FJoinKind write FJoinKind;
     property RelOperator: TRelationalOperator read FRelOperator write FRelOperator;
     property LeftJoinOn: TJoinOnExprList read FLeftJoinOn write FLeftJoinOn;
     property RightJoinOn: TJoinOnExprList read FRightJoinOn write FRightJoinOn;
     property AndJoinOn: TJoinOnExprList read FAndJoinOn write FAndJoinOn;
     property WhereStr: String read FWhereStr write FWhereStr;
     property WhereResolver: TMainExpr read FWhereResolver write FWhereResolver;
     property MasterDataSet: TDataSet read FMasterDataSet write FMasterDataSet;
     property DetailDataSet: TDataSet read FDetailDataSet write FDetailDataSet;
  end;

  TJoinOnList = class
     FItems: TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TJoinOnItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TJoinOnItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TJoinOnItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  UPDATE statement section data                                }
{-------------------------------------------------------------------------------}

  TUpdateItem = class
  private
     FUpdateList  : TUpdateList;
     FColName     : String;
     FColExpr     : String;
     FResolver    : TMainExpr;
     FField       : TField;
  public
     constructor Create(UpdateList: TUpdateList);
     destructor Destroy; override;

     property ColName: String read FColName write FColName;
     property ColExpr: String read FColExpr write FColExpr;
     property Resolver: TMainExpr read FResolver write FResolver;
     property Field: TField read FField write FField;
  end;

  TUpdateList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TUpdateItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TUpdateItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TUpdateItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  WHERE optimization section data                              }
{-------------------------------------------------------------------------------}

  TWhereOptimizeItem = class
  private
     FWhereOptimizeList : TWhereOptimizeList;
     FDataSet           : TDataSet;
     FFieldNames        : String;
     FRangeStart        : String;
     FRangeEnd          : String;
     FRelOperator       : TRelationalOperator;
     FCanOptimize       : Boolean;        { Can optimize the result set generation with this config. }
  public
     constructor Create(WhereOptimizeList: TWhereOptimizeList);

     property DataSet: TDataSet read FDataSet write FDataSet;
     property FieldNames: String read FFieldNames write FFieldNames;
     property RangeStart: String read FRangeStart write FRangeStart;
     property Rangeend: String read FRangeend write FRangeend;
     property RelOperator: TRelationalOperator read FRelOperator write FRelOperator;
     property CanOptimize: Boolean read FCanOptimize write FCanOptimize;
  end;

  TWhereOptimizeList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TWhereOptimizeItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TWhereOptimizeItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TWhereOptimizeItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  CREATE TABLE section data                                    }
{-------------------------------------------------------------------------------}

  TCreateField = class
  private
    FCreateFields : TCreateFields;
    FFieldName    : String;
    FFieldType    : Integer;
    FScale        : Integer;
    FPrecision    : Integer;
    FSize         : Integer;
    FBlobType     : Integer;
  public
    constructor Create(CreateFields: TCreateFields);
    property FieldName: String read FFieldName write FFieldName;
    property FieldType: Integer read FFieldType write FFieldType;
    property Scale: Integer read FScale write FScale;
    property Precision: Integer read FPrecision write FPrecision;
    property Size: Integer read FSize write FSize;
    property BlobType: Integer read FBlobType write FBlobType;
  end;

  TCreateFields = class
  private
     FList  : TList;
     function Get(Index: Integer): TCreateField;
  public
     constructor Create;
     destructor Destroy; override;
     procedure Clear;
     procedure AddField( const AName: String;
                         AFieldType,
                         AScale,
                         APrecision,
                         ASize,
                         ABlobType: Integer);
     function Count: Integer;
     property Items[Index: Integer]: TCreateField read Get; default;
  end;

  TCreateTableItem = class
  private
    FCreateTableList : TCreateTableList;
    FFields          : TCreateFields;
    FTableName       : String;
    FPrimaryKey      : TStringList;
  public
    constructor Create(CreateTableList: TCreateTableList);
    destructor Destroy; override;
    function FieldCount: Integer;
    property Fields: TCreateFields read FFields;
    property TableName: String read FTableName write FTableName;
    property PrimaryKey: TStringList read FPrimaryKey;
  end;

  TCreateTableList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TCreateTableItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TCreateTableItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TCreateTableItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  INSERT INTO section data                                     }
{-------------------------------------------------------------------------------}

  TInsertItem = class
  private
     FInsertList   : TInsertList;
     FTableName    : String;
     FFieldNames   : TStringList;
     FExprList     : TStringList;
     FResolverList : TList;
     FDataSet      : TDataSet;
  public
     constructor Create(InsertList: TInsertList);
     destructor Destroy; override;

     property TableName: String read FTableName write FTableName;
     property FieldNames: TStringList read FFieldNames;
     property ExprList: TStringList read FExprList;
     property DataSet: TDataSet read FDataSet write FDataSet;
     property ResolverList: TList read FResolverList;
  end;

  TInsertList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TInsertItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TInsertItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     property Count: Integer read GetCount;
     property Items[Index: Integer]: TInsertItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  TSrtField to sort with variable type columns                 }
{-------------------------------------------------------------------------------}

  TSrtField = Class(TObject)
  private
     FFields          : TSrtFields;
     FDataType        : TExprType;
     FDataSize        : Integer;
     FDesc            : Boolean;
     FBufferOffset    : Integer;
     function GetData(Buffer: Pointer): Boolean;
     procedure SetData(Buffer: Pointer);
  protected
     function GetAsString: String; virtual; abstract;
     procedure SetAsString(const Value: String); virtual; abstract;
     function GetAsFloat: double; virtual; abstract;
     procedure SetAsFloat(Value: double); virtual; abstract;
     function GetAsInteger: Longint; virtual; abstract;
     procedure SetAsInteger(Value: Longint); virtual; abstract;
     function GetAsBoolean: Boolean; virtual; abstract;
     procedure SetAsBoolean(Value: Boolean); virtual; abstract;
     procedure SetDataType(Value: TExprType);
  public
     constructor Create(Fields: TSrtFields); virtual;

     property DataType: TExprType read FDataType write SetDataType;
     property DataSize: Integer read FDataSize write FDataSize;
     property Desc: Boolean read FDesc write FDesc;
     property BufferOffset: Integer read FBufferOffset write FBufferOffset;

     property AsString: String read GetAsString write SetAsString;
     property AsFloat: Double read GetAsFloat write SetAsFloat;
     property AsInteger: Longint read GetAsInteger write SetAsInteger;
     property AsBoolean: Boolean read GetAsBoolean write SetAsBoolean;
  end;

{-------------------------------------------------------------------------------}
{                  TSrtStringField                                              }
{-------------------------------------------------------------------------------}

  TSrtStringField = class(TSrtField)
  private
     function GetValue(var Value: string): Boolean;
  protected
     function GetAsString: String; override;
     procedure SetAsString(const Value: String); override;
     function GetAsFloat: double; override;
     procedure SetAsFloat(Value: double); override;
     function GetAsInteger: Longint; override;
     procedure SetAsInteger(Value: Longint); override;
     function GetAsBoolean: Boolean; override;
     procedure SetAsBoolean(Value: Boolean); override;
  public
     constructor Create(Fields: TSrtFields); override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TsrtFloatField                                        }
{-------------------------------------------------------------------------------}

  TSrtFloatField = class(TSrtField)
  protected
    function GetAsString: String; override;
    procedure SetAsString(const Value: String); override;
    function GetAsFloat: double; override;
    procedure SetAsFloat(Value: double); override;
    function GetAsInteger: Longint; override;
    procedure SetAsInteger(Value: Longint); override;
    function GetAsBoolean: Boolean; override;
    procedure SetAsBoolean(Value: Boolean); override;
  public
    constructor Create(Fields: TSrtFields); override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TsrtIntegerField                                      }
{-------------------------------------------------------------------------------}

  TSrtIntegerField = class(TSrtField)
  protected
    function GetAsString: String; override;
    procedure SetAsString(const Value: String); override;
    function GetAsInteger: Longint; override;
    procedure SetAsInteger(Value: Longint); override;
    function GetAsFloat: double; override;
    procedure SetAsFloat(Value: double); override;
    function GetAsBoolean: Boolean; override;
    procedure SetAsBoolean(Value: Boolean); override;
  public
    constructor Create(Fields: TSrtFields); override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TSrtBooleanField                                      }
{-------------------------------------------------------------------------------}

  TSrtBooleanField = class(TSrtField)
  protected
    function GetAsString: String; override;
    procedure SetAsString(const Value: String); override;
    function GetAsBoolean: Boolean; override;
    procedure SetAsBoolean(Value: Boolean); override;
    function GetAsInteger: Longint; override;
    procedure SetAsInteger(Value: Longint); override;
    function GetAsFloat: double; override;
    procedure SetAsFloat(Value: double); override;
  public
    constructor Create(Fields: TSrtFields); override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TSrtFields                                            }
{-------------------------------------------------------------------------------}

  TSrtFields = class
     FSortList : TxqSortList;
     FItems    : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TSrtField;
  public
     constructor Create(SortList: TxqSortList);
     destructor Destroy; override;
     function Add(DataType : TExprType): TSrtField;
     procedure Clear;

     property Count: Integer read GetCount;
     property Items[Index: Integer]: TSrtField read GetItem; default;
     property SortList: TxqSortList read FSortList;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqSortList                                           }
{-------------------------------------------------------------------------------}
  TxqSortList = class(TObject)
  private
     FFields           : TSrtFields;
     FRecNo            : Integer;
     FRecordBufferSize : Integer;
     function ActiveBuffer: PChar; virtual; abstract;
  protected
     function GetFieldData(Field: TSrtField; Buffer: Pointer): Boolean; virtual; abstract;
     procedure SetFieldData(Field: TSrtField; Buffer: Pointer); virtual; abstract;
     procedure SetRecno(Value: Integer);
     function GetRecno: Integer;
     procedure SetSourceRecno(Value: Integer); virtual; abstract;
     function GetSourceRecno: Integer; virtual; abstract;
     function GetRecordCount: Integer; virtual; abstract;
  public
     constructor Create;
     destructor Destroy; override;
     procedure AddField( pDataType   : TExprType;
                         pDataSize   : Integer;
                         pDescending : Boolean );
     procedure Insert; virtual; abstract;
     procedure Sort;
     procedure Exchange(Recno1,Recno2: Integer); virtual; abstract;
     procedure Clear; virtual; abstract;
     function IsEqual(Recno1, Recno2: Integer): Boolean;

     property Count: Integer read GetRecordCount;
     property Recno: Integer read GetRecno write SetRecno;
     property SourceRecno: Integer read GetSourceRecno write SetSourceRecno;
     property Fields: TSrtFields read FFields;
  end;

  TMemSortList = Class(TxqSortList)
  private
     FBufferList : TList;
     function ActiveBuffer: PChar; override;
  protected
     function GetFieldData(Field: TSrtField; Buffer: Pointer): Boolean; override;
     procedure SetFieldData(Field: TSrtField; Buffer: Pointer); override;
     function GetRecordCount: Integer; override;
     procedure SetSourceRecno(Value: Integer); override;
     function GetSourceRecno: Integer; override;
  public
     constructor Create;
     destructor Destroy; override;
     procedure Insert; override;
     procedure Exchange(Recno1,Recno2: Integer); override;
     procedure Clear; override;
  end;

  TFileSortList = Class(TxqSortList)
  private
     FBufferList : TList;
     FMemMapFile : TMemMapFile;
     FTmpFile    : String;
     FBuffer     : PChar;
     function ActiveBuffer: PChar; override;
  protected
     function GetFieldData(Field: TSrtField; Buffer: Pointer): Boolean; override;
     procedure SetFieldData(Field: TSrtField; Buffer: Pointer); override;
     function GetRecordCount: Integer; override;
     procedure SetSourceRecno(Value: Integer); override;
     function GetSourceRecno: Integer; override;
  public
     constructor Create(MapFileSize : Longint);
     destructor Destroy; override;
     procedure Insert; override;
     procedure Exchange(Recno1,Recno2: Integer); override;
     procedure Clear; override;
  end;


{-------------------------------------------------------------------------------}
{                  Define TReferencedDataSetItem                                }
{-------------------------------------------------------------------------------}

  { two next classes are used only for checking the number in that a dataset is
    referenced in an expression  }

  TReferencedDataSetItem = class
  private
     FDataSet : TDataSet;
     FCount   : Integer;
     FReferencedDataSets: TReferencedDataSetList;
  public
     constructor Create(RefDataSetList : TReferencedDataSetList);

     property DataSet : TDataSet read FDataSet write FDataSet;
     property Count: Integer read FCount write FCount;
  end;

{-------------------------------------------------------------------------------}
{                  Define TReferencedDataSetList                                }
{-------------------------------------------------------------------------------}
  TReferencedDataSetList = class
     FItems : TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TReferencedDataSetItem;
  public
     constructor Create;
     destructor Destroy; override;
     function Add: TReferencedDataSetItem;
     procedure Clear;
     procedure Delete(Index: Integer);
     function IndexOf(DataSet: TDataSet): Integer;

     property Count: Integer read GetCount;
     property Items[Index: Integer]: TReferencedDataSetItem read GetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  Define TMemMapFile                                           }
{-------------------------------------------------------------------------------}

  TMemMapFile = class(TObject)
  private
    FFileName: String;
    FSize: Longint;
    FFileSize: Longint;
    FFileMode: Integer;
    FFileHandle: Integer; 
    FMapHandle: Integer;  
    FData: PChar;
    FMapNow: Boolean;
    FPosition: Longint;
    FVirtualSize: Longint;

    procedure AllocFileHandle;
    procedure AllocFileMapping;
    procedure AllocFileView;
    function GetSize: Longint;
  public
    constructor Create(FileName: String; FileMode: integer;
                       Size: integer; MapNow: Boolean); virtual;
    destructor Destroy; override;
    procedure FreeMapping;
    function Read(var Buffer; Count: Longint): Longint;
    function Write(const Buffer; Count: Longint): Longint;
    function Seek(Offset: Longint; Origin: Word): Longint;

    property Data: PChar read FData;
    property Size: Longint read GetSize;
    property VirtualSize: Longint read FVirtualSize;
    property Position: Longint read FPosition;
    property FileName: String read FFileName;
    property FileHandle: Integer read FFileHandle;
    property MapHandle: Integer read FMapHandle;
  end;


{-------------------------------------------------------------------------------}
{        Expression evaluation section                                          }
{-------------------------------------------------------------------------------}

{-------------------------------------------------------------------------------}
{        Defines main expression evaluation class TMainExpr                     }
{-------------------------------------------------------------------------------}

  TCheckData = record
     Field           : TField;   { the field referenced }
     RefCount        : Integer;  { No of references to a field }
     FieldCount      : Integer;  { number of fields referenced in expression }
     Fields          : array[1..MAX_INDEXED_FIELDS] of TField; { the fields referenced (used for joining) }
     HasMorefunctions: Boolean;  { expression have more functions }
     //OtherRefCount   : Integer;  { this is used in order to know how many other datasets are referenced in the expression }
  end;

  TMainExpr = Class(TObject)
  private
     FDefaultDataSet : TDataSet;
     FAnalizer       : TObject;
     FCalcMaxWidth   : Boolean;
     FReferencedDataSets : TReferencedDataSetList;
     function IDFunc( const Identifier: string; ParameterList: TParameterList): TExpression;
  public
     SubqueryExpr : TExpression;        { used for subqueries only (special case) }
     CheckData    : TCheckData;         { used when checking expression }
     Expression   : TExpression;        { the real expression to evaluate }
     constructor Create( SqlAnalizer: TObject; DataSet: TDataSet );
     destructor Destroy; override;
     procedure ParseExpression(const ExprStr: string);
     function CheckExpression(const ExprStr: String): Boolean;

     property CalcMaxWidth : Boolean read FCalcMaxWidth write FCalcMaxWidth;
     property ReferencedDataSets : TReferencedDataSetList
        read  FReferencedDataSets
        write FReferencedDataSets;
  end;

{-------------------------------------------------------------------------------}
{        this class is used for filter property only                            }
{-------------------------------------------------------------------------------}

  TFilterExpr = Class(TObject)
  private
     FDataSet     : TDataSet;
     function IDFunc( const Identifier: string; ParameterList: TParameterList): TExpression;
  protected
  public
     Expression   : TExpression;          { the real expression to evaluate }
     constructor Create( DataSet: TDataSet );
     destructor Destroy; override;
     procedure ParseExpression(const ExprStr: string);
  end;

{-------------------------------------------------------------------------------}
{        This will be used for speeding up the process of evaluating subqueries }
{-------------------------------------------------------------------------------}

  TSubqueryExpr = Class(Tfunction)
  private
    FExprtype: TExprtype;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprtype: TExprtype; override;
  public
    Value: variant;
    constructor Create( ParameterList: TParameterList);
  end;

{-------------------------------------------------------------------------------}
{        Defines TNestedItem                                                    }
{-------------------------------------------------------------------------------}

  TNestedItem = class
  private
     FIdx1       : Integer;      { the joining left table index in FTableList  }
     FIdx2       : Integer;      { the joining right table index in FTableList }
     FJoinOnItem : TJoinOnItem;   { the TJoinOnItem that this belongs to        }
  public
     property Idx1 : Integer read FIdx1 write FIdx1;
     property Idx2 : Integer read FIdx2 write FIdx2;
     property JoinOnItem : TJoinOnItem read FJoinOnItem write FJoinOnItem;
  end;

  TNestedList = class
  private
     FList : TList;
     function Get(Index: Integer): TNestedItem;
  public
     constructor Create;
     destructor Destroy; override;
     procedure Add(JoinOnItem: TJoinOnItem; Idx1, Idx2: Integer);
     function Count: Integer;
     function IndexOf(Value: Integer): Integer;
     procedure Clear;

     property Items[Index: Integer]: TNestedItem read Get; default;
  end;

implementation

uses
   xquery, xqconsts ;

{-------------------------------------------------------------------------------}
{                  Implement TColumnItem                                        }
{-------------------------------------------------------------------------------}

constructor TColumnItem.Create( ColumnList: TColumnList );
begin
   inherited Create;
   FColumnList    := ColumnList;
   FAggregateList := TAggregateList.Create;
   FAutoFree      := True;
end;

destructor TColumnItem.Destroy;
begin
   FAggregateList.Free;
   if FAutoFree and Assigned(FResolver) then
      FResolver.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{                  Implement TColumnList                                        }
{-------------------------------------------------------------------------------}

constructor TColumnList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TColumnList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TColumnList.GetCount;
begin
   Result := FItems.Count;
end;

function TColumnList.GetItem(Index: Integer): TColumnItem;
begin
   Result := FItems[Index];
end;

function TColumnList.Add: TColumnItem;
begin
   Result := TColumnItem.Create(Self);
   FItems.Add(Result);
end;

procedure TColumnList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TColumnItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TColumnList.Delete(Index: Integer);
begin
   TColumnItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

procedure TColumnList.DeleteAggregate(RecNo : Integer);
var
   I, J : Integer;
begin
   for I := 0 to FItems.Count - 1 do
      with TColumnItem(FItems[I]) do
         for J := 0 to AggregateList.Count - 1 do
            AggregateList[J].SparseList.Delete(RecNo);
end;

procedure TColumnList.SortAggregateWithList(SortList : TxqSortList);
var
   I, J, K,
   Index      : Integer;
   SparseList : TAggSparseList;
begin
   for I := 0 to FItems.Count - 1 do
      with TColumnItem(FItems[I]) do
         for J := 0 to AggregateList.Count - 1 do
         begin
            SparseList := TAggSparseList.Create(1000);
            for K := 1 to SortList.Count do
            begin
               SortList.Recno := K;
               Index := SortList.SourceRecno;
               SparseList.Values[K] := AggregateList[J].SparseList.Values[Index];
               SparseList.Count[K]  := AggregateList[J].SparseList.Count[Index];
            end;
            AggregateList[J].SparseList.Free;
            AggregateList[J].SparseList:= SparseList;
         end;
end;

{-------------------------------------------------------------------------------}
{                  Implement TTableItem                                         }
{-------------------------------------------------------------------------------}

constructor TTableItem.Create(TableList: TTableList);
begin
   inherited Create;
   FTableList:= TableList;
end;

{-------------------------------------------------------------------------------}
{                  Implement TTableList                                         }
{-------------------------------------------------------------------------------}

constructor TTableList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TTableList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TTableList.GetCount;
begin
   Result := FItems.Count;
end;

function TTableList.GetItem(Index: Integer): TTableItem;
begin
   Result := FItems[Index];
end;

function TTableList.Add: TTableItem;
begin
   Result := TTableItem.Create(Self);
   FItems.Add(Result);
end;

procedure TTableList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TTableItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TTableList.Delete(Index: Integer);
begin
   TTableItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

function TTableList.IndexOfDataSet(DataSet: TDataSet): Integer;
var
   Idx : Integer;
begin
   Result := -1;
   for Idx := 0 to FItems.Count - 1 do
      if TTableItem(FItems[Idx]).DataSet = DataSet then
      begin
         Result := Idx;
         Exit;
      end;
end;

{-------------------------------------------------------------------------------}
{                  Implement TOrderByItem                                       }
{-------------------------------------------------------------------------------}

constructor TOrderByItem.Create(OrderByList: TOrderByList);
begin
   inherited Create;
   FOrderByList:= OrderByList;
end;

{-------------------------------------------------------------------------------}
{                  Implement TOrderByList                                       }
{-------------------------------------------------------------------------------}

constructor TOrderByList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TOrderByList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TOrderByList.GetCount;
begin
   Result := FItems.Count;
end;

function TOrderByList.GetItem(Index: Integer): TOrderByItem;
begin
   Result := FItems[Index];
end;

function TOrderByList.Add: TOrderByItem;
begin
   Result := TOrderByItem.Create(Self);
   FItems.Add(Result);
end;

procedure TOrderByList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TOrderByItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TOrderByList.Delete(Index: Integer);
begin
   TOrderByItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  Implement TJoinOnExprItem                                    }
{-------------------------------------------------------------------------------}

constructor TJoinOnExprItem.Create(JoinOnExprList: TJoinOnExprList);
begin
   inherited Create;
   FJoinOnExprList:= JoinOnExprList;
end;

destructor TJoinOnExprItem.Destroy;
begin
   if Assigned(FResolver) then
      FResolver.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{                  Implement TJoinOnExprList                                    }
{-------------------------------------------------------------------------------}

constructor TJoinOnExprList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TJoinOnExprList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TJoinOnExprList.GetCount;
begin
   Result := FItems.Count;
end;

function TJoinOnExprList.GetItem(Index: Integer): TJoinOnExprItem;
begin
   Result := FItems[Index];
end;

function TJoinOnExprList.Add: TJoinOnExprItem;
begin
   Result := TJoinOnExprItem.Create(Self);
   FItems.Add(Result);
end;

procedure TJoinOnExprList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TJoinOnExprItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TJoinOnExprList.Delete(Index: Integer);
begin
   TJoinOnExprItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;


{-------------------------------------------------------------------------------}
{                  Implement TJoinOnItem                                        }
{-------------------------------------------------------------------------------}

constructor TJoinOnItem.Create(JoinOnList: TJoinOnList);
begin
   inherited Create;
   FJoinOnList:= JoinOnList;
   FLeftJoinOn:= TJoinOnExprList.Create;
   FRightJoinOn:= TJoinOnExprList.Create;
   FAndJoinOn:= TJoinOnExprList.Create;
end;

destructor TJoinOnItem.Destroy;
begin
   FLeftJoinOn.Free;
   FRightJoinOn.Free;
   FAndJoinOn.Free;
   if Assigned(FWhereResolver) then
      FWhereResolver.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{                  Implement TJoinOnList                                        }
{-------------------------------------------------------------------------------}

constructor TJoinOnList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TJoinOnList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TJoinOnList.GetCount;
begin
   Result := FItems.Count;
end;

function TJoinOnList.GetItem(Index: Integer): TJoinOnItem;
begin
   Result := FItems[Index];
end;

function TJoinOnList.Add: TJoinOnItem;
begin
   Result := TJoinOnItem.Create(Self);
   FItems.Add(Result);
end;

procedure TJoinOnList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TJoinOnItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TJoinOnList.Delete(Index: Integer);
begin
   TJoinOnItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  Implement TUpdateItem                                        }
{-------------------------------------------------------------------------------}

constructor TUpdateItem.Create(UpdateList: TUpdateList);
begin
   inherited Create;
   FUpdateList:= UpdateList;
end;

destructor TUpdateItem.Destroy;
begin
   if Assigned(FResolver) then
      FResolver.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{                  Implement TUpdateList                                        }
{-------------------------------------------------------------------------------}

function TUpdateList.GetCount;
begin
   Result := FItems.Count;
end;

function TUpdateList.GetItem(Index: Integer): TUpdateItem;
begin
   Result := FItems[Index];
end;

constructor TUpdateList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TUpdateList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TUpdateList.Add: TUpdateItem;
begin
   Result := TUpdateItem.Create(Self);
   FItems.Add(Result);
end;

procedure TUpdateList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TUpdateItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TUpdateList.Delete(Index: Integer);
begin
   TUpdateItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  Implement TWhereOptimizeItem                                 }
{-------------------------------------------------------------------------------}

constructor TWhereOptimizeItem.Create(WhereOptimizeList: TWhereOptimizeList);
begin
   inherited Create;
   FWhereOptimizeList:= WhereOptimizeList;
end;

{-------------------------------------------------------------------------------}
{                  Implement TWhereOptimizeList                                 }
{-------------------------------------------------------------------------------}

function TWhereOptimizeList.GetCount;
begin
   Result := FItems.Count;
end;

function TWhereOptimizeList.GetItem(Index: Integer): TWhereOptimizeItem;
begin
   Result := FItems[Index];
end;

constructor TWhereOptimizeList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TWhereOptimizeList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TWhereOptimizeList.Add: TWhereOptimizeItem;
begin
   Result := TWhereOptimizeItem.Create(Self);
   FItems.Add(Result);
end;

procedure TWhereOptimizeList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TWhereOptimizeItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TWhereOptimizeList.Delete(Index: Integer);
begin
   TWhereOptimizeItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  Implement TCreateField                                       }
{-------------------------------------------------------------------------------}

constructor TCreateField.Create(CreateFields: TCreateFields);
begin
   inherited Create;
   FCreateFields:= CreateFields;
end;

{-------------------------------------------------------------------------------}
{                  Implement TCreateFields                                      }
{-------------------------------------------------------------------------------}

constructor TCreateFields.Create;
begin
   inherited Create;
   FList:= TList.Create;
end;

destructor TCreateFields.Destroy;
begin
   Clear;
   FList.Free;
   inherited;
end;

procedure TCreateFields.AddField( const AName : String;
                                  AFieldType,
                                  AScale,
                                  APrecision,
                                  ASize,
                                  ABlobType   : Integer);
var
   NewField: TCreateField;
begin
   NewField:= TCreateField.Create(Self);
   with NewField do
   begin
     FieldName := AName;
     FieldType := AFieldType;
     Scale     := AScale;
     Precision := APrecision;
     Size      := ASize;
     BlobType  := ABlobType;
   end;
   FList.Add(NewField);
end;

procedure TCreateFields.Clear;
var I: Integer;
begin
   for I:= 0 to FList.Count - 1 do
      TCreateField(FList[I]).Free;
   FList.Clear;
end;

function TCreateFields.Get(Index: Integer): TCreateField;
begin
   Result:= nil;
   if (Index<0) or (Index>FList.Count-1) then exit;
   Result:= TCreateField(FList[Index]);
end;

function TCreateFields.Count: Integer;
begin
   Result:= FList.Count;
end;

{-------------------------------------------------------------------------------}
{                  Implement TCreateTableItem                                   }
{-------------------------------------------------------------------------------}

constructor TCreateTableItem.Create(CreateTableList: TCreateTableList);
begin
   inherited Create;
   FCreateTableList:= CreateTableList;
   FFields:= TCreateFields.Create;
   FPrimaryKey:= TStringList.Create;
end;

destructor TCreateTableItem.Destroy;
begin
   FFields.Free;
   FPrimaryKey.Free;
   inherited Destroy;
end;

function TCreateTableItem.FieldCount: Integer;
begin
   Result:= FFields.Count;
end;

{-------------------------------------------------------------------------------}
{                  Implement TCreateTableList                                   }
{-------------------------------------------------------------------------------}

constructor TCreateTableList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TCreateTableList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TCreateTableList.GetCount;
begin
   Result := FItems.Count;
end;

function TCreateTableList.GetItem(Index: Integer): TCreateTableItem;
begin
   Result := FItems[Index];
end;

function TCreateTableList.Add: TCreateTableItem;
begin
   Result := TCreateTableItem.Create(Self);
   FItems.Add(Result);
end;

procedure TCreateTableList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TCreateTableItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TCreateTableList.Delete(Index: Integer);
begin
   TCreateTableItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;


{-------------------------------------------------------------------------------}
{                  Implement TInsertItem                                        }
{-------------------------------------------------------------------------------}
constructor TInsertItem.Create(InsertList: TInsertList);
begin
   inherited Create;
   FInsertList   := InsertList;
   FFieldNames   := TStringList.Create;
   FExprList     := TStringList.Create;
   FResolverList := TList.Create;
end;

destructor TInsertItem.Destroy;
var
   I        : Integer;
   Resolver : TMainExpr;
begin
   FFieldNames.Free;
   FExprList.Free;
   for I := 0 to FResolverList.Count - 1 do
   begin
      Resolver := TMainExpr(FResolverList[I]);
      if Assigned(Resolver) then
         Resolver.Free;
   end;
   FResolverList.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{                  Implement TInsertList                                        }
{-------------------------------------------------------------------------------}

constructor TInsertList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TInsertList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TInsertList.GetCount;
begin
   Result := FItems.Count;
end;

function TInsertList.GetItem(Index: Integer): TInsertItem;
begin
   Result := FItems[Index];
end;

function TInsertList.Add: TInsertItem;
begin
   Result := TInsertItem.Create(Self);
   FItems.Add(Result);
end;

procedure TInsertList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TInsertItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TInsertList.Delete(Index: Integer);
begin
   TInsertItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  implements TSrtField                                         }
{-------------------------------------------------------------------------------}

constructor TSrtField.Create(Fields: TSrtFields);
begin
   inherited Create;
   FFields  := Fields;
end;

function TSrtField.GetData(Buffer: Pointer): Boolean;
begin
   Result := FFields.FSortList.GetFieldData( Self, Buffer );
end;

procedure TSrtField.SetData(Buffer: Pointer);
begin
   FFields.FSortList.SetFieldData( Self, Buffer );
end;

procedure TSrtField.SetDataType(Value: TExprType);
begin
  FDataType := Value;
end;

{-------------------------------------------------------------------------------}
{                  implements TSrtStringField                                   }
{-------------------------------------------------------------------------------}
constructor TSrtStringField.Create(Fields: TSrtFields);
begin
   inherited Create(Fields);
   SetDataType(ttString);
end;

function TSrtStringField.GetValue(var Value: string): Boolean;
var
  Buffer: array[0..dsMaxStringSize] of Char;
begin
  Result := GetData(@Buffer);
  if Result then
    Value := Buffer;
end;

function TSrtStringField.GetAsString: String;
begin
  if not GetValue(Result) then Result := '';
end;

procedure TSrtStringField.SetAsString(const Value: String);
var
  Buffer: array[0..dsMaxStringSize] of Char;
  L : Integer;
begin
  FillChar(Buffer, FDataSize, 0);
  L := Length(Value);
  StrLCopy(Buffer, PChar(Value), L);
  SetData(@Buffer);
end;

function TSrtStringField.GetAsFloat: double;
begin
end;

procedure TSrtStringField.SetAsFloat(Value: double);
begin
end;

function TSrtStringField.GetAsInteger: Longint;
begin
end;

procedure TSrtStringField.SetAsInteger(Value: Longint);
begin
end;

function TSrtStringField.GetAsBoolean: Boolean;
begin
end;

procedure TSrtStringField.SetAsBoolean(Value: Boolean);
begin
end;

{-------------------------------------------------------------------------------}
{                  implements TSrtFloatField                                        }
{-------------------------------------------------------------------------------}
constructor TSrtFloatField.Create(Fields: TSrtFields);
begin
   inherited Create(Fields);
   SetDataType(ttFloat);
end;

function TSrtFloatField.GetAsFloat: double;
begin
  if not GetData(@Result) then Result := 0;
end;

procedure TSrtFloatField.SetAsFloat(Value: double);
begin
  SetData(@Value);
end;

function TSrtFloatField.GetAsString: string;
var
  F: Double;
begin
  if GetData(@F) then Result := FloatToStr(F) else Result := '';
end;

procedure TSrtFloatField.SetAsString(const Value: string);
var
  F: Extended;
begin
  if Value = '' then SetAsFloat(0) else
  begin
    if not TextToFloat(PChar(Value), F, fvExtended) then
      EXQueryError.CreateFmt(SIsInvalidFloatValue, [Value]);
    SetAsFloat(F);
  end;
end;

function TSrtFloatField.GetAsInteger: Longint;
begin
end;

procedure TSrtFloatField.SetAsInteger(Value: Longint);
begin
end;

function TSrtFloatField.GetAsBoolean: Boolean;
begin
end;

procedure TSrtFloatField.SetAsBoolean(Value: Boolean);
begin
end;

{-------------------------------------------------------------------------------}
{                  implements TsrtIntegerField                                      }
{-------------------------------------------------------------------------------}
constructor TSrtIntegerField.Create(Fields: TSrtFields);
begin
   inherited Create(Fields);
   SetDataType(ttInteger);
end;

function TSrtIntegerField.GetAsInteger: Longint;
begin
   if not GetData(@Result) then Result := 0;
end;

procedure TSrtIntegerField.SetAsInteger(Value: Longint);
begin
  SetData(@Value);
end;

function TSrtIntegerField.GetAsString: string;
var
  L: Longint;
begin
  if GetData(@L) then Str(L, Result) else Result := '';
end;

procedure TSrtIntegerField.SetAsString(const Value: string);
var
  E : Integer;
  L : Longint;
begin
  Val(Value, L, E);
  if E <> 0 then EXQueryError.CreateFmt(SIsInvalidIntegerValue, [Value]);
  SetAsInteger(L);
end;

function TSrtIntegerField.GetAsFloat: double;
begin
end;

procedure TSrtIntegerField.SetAsFloat(Value: double);
begin
end;

function TSrtIntegerField.GetAsBoolean: Boolean;
begin
end;

procedure TSrtIntegerField.SetAsBoolean(Value: Boolean);
begin
end;

{-------------------------------------------------------------------------------}
{                  implements TSrtBooleanField                                      }
{-------------------------------------------------------------------------------}
constructor TSrtBooleanField.Create(Fields: TSrtFields);
begin
   inherited Create(Fields);
   SetDataType(ttBoolean);
end;

function TSrtBooleanField.GetAsBoolean: Boolean;
var
  B: WordBool;
begin
  if GetData(@B) then Result := B else Result := False;
end;

procedure TSrtBooleanField.SetAsBoolean(Value: Boolean);
var
  B: WordBool;
begin
  if Value then Word(B) := 1 else Word(B) := 0;
  SetData(@B);
end;

function TSrtBooleanField.GetAsString: string;
var
  B: WordBool;
begin
  if GetData(@B) then Result := Copy(xqbase.NBoolean[B], 1, 1) else Result := '';
end;

procedure TSrtBooleanField.SetAsString(const Value: string);
var
  L : Integer;
begin
  L := Length(Value);
  if L = 0 then
  begin
    SetAsBoolean(False);
  end else
  begin
    if AnsiCompareText(Value, Copy(xqbase.NBoolean[False], 1, L)) = 0 then
      SetAsBoolean(False)
    else
      if AnsiCompareText(Value, Copy(xqbase.NBoolean[True], 1, L)) = 0 then
        SetAsBoolean(True)
      else
        EXQueryError.CreateFmt(SIsInvalidBoolValue, [Value]);
  end;
end;

function TSrtBooleanField.GetAsInteger: Longint;
begin
end;

procedure TSrtBooleanField.SetAsInteger(Value: Longint);
begin
end;

function TSrtBooleanField.GetAsFloat: double;
begin
end;

procedure TSrtBooleanField.SetAsFloat(Value: double);
begin
end;

{-------------------------------------------------------------------------------}
{                  implements TSrtFields                                            }
{-------------------------------------------------------------------------------}

constructor TSrtFields.Create(SortList: TxqSortList);
begin
   inherited Create;
   FSortList := SortList;
   FItems := TList.Create;
end;

destructor TSrtFields.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TSrtFields.GetCount: Integer;
begin
   Result := FItems.Count;
end;

function TSrtFields.GetItem(Index: Integer): TSrtField;
begin
   Result := FItems[Index];
end;

function TSrtFields.Add(DataType : TExprType): TSrtField;
begin
   case DataType of
      ttString  : Result := TSrtStringField.Create(Self);
      ttFloat   : Result := TSrtFloatField.Create(Self);
      ttInteger : Result := TSrtIntegerField.Create(Self);
      ttBoolean : Result := TSrtBooleanField.Create(Self);
   end;
   FItems.Add(Result);
end;

procedure TSrtFields.Clear;
var
   I: Integer;
begin
   for I := 0 TO FItems.Count - 1 do
      TSrtField(FItems[I]).Free;
   FItems.Clear;
end;

{-------------------------------------------------------------------------------}
{                  Define TxqSortList                                           }
{-------------------------------------------------------------------------------}
constructor TxqSortList.Create;
begin
   inherited Create;
   FFields := TSrtFields.Create(Self);
   FRecNo  := -1;
   FRecordBufferSize := SizeOf(Integer);  { first data is the SourceRecNo property }
end;

destructor TxqSortList.Destroy;
begin
   FFields.Free;
   inherited Destroy;
end;

procedure TxqSortList.SetRecno(Value: Integer);
begin
   FRecNo := Value;
   if Value = fRecno then Exit;
   if (Value < 1) or (Value > GetRecordCount) then
      raise EXQueryError.Create(SRecnoInvalid);
   FRecno := Value;
end;

function TxqSortList.GetRecno: Integer;
begin
   Result := FRecNo;
end;

procedure TxqSortList.AddField( pDataType   : TExprType;
                                pDataSize   : Integer;
                                pDescending : Boolean );
begin
   with FFields.Add(pDataType) do
   begin
      BufferOffset := FRecordBufferSize;
      DataType := pDataType;
      case DataType of
         ttString  : DataSize := pDataSize;
         ttFloat   : DataSize := SizeOf(Double);
         ttInteger : DataSize := SizeOf(Integer);
         ttBoolean : DataSize := SizeOf(WordBool);
      end;
      Desc := pDescending;
      Inc(FRecordBufferSize, DataSize);
   end;
end;

function TxqSortList.IsEqual(Recno1, Recno2: Integer): Boolean;
var
   Buffer  : PChar;
   Buffer1 : PChar;
   Buffer2 : PChar;
begin
   SetRecno( Recno1 );
   Buffer := ActiveBuffer;
   GetMem(Buffer1, FRecordBufferSize);
   Move( Buffer^, Buffer1^, FRecordBufferSize);

   SetRecno( Recno2 );
   Buffer := ActiveBuffer;
   GetMem(Buffer2, FRecordBufferSize);
   Move( Buffer^, Buffer2^, FRecordBufferSize);

   { the first SizeOf(Integer) bytes is the source recno and always is different }
   Result := Comparemem( (Buffer1 + SizeOf(Integer)),
                         (Buffer2 + SizeOf(Integer)),
                         FRecordBufferSize - SizeOf(Integer) );

   FreeMem(Buffer1, FRecordBufferSize);
   FreeMem(Buffer2, FRecordBufferSize);

end;

procedure TxqSortList.Sort;
var
  Idx      : Integer;
  Index    : Integer;
  Pivot    : Integer;
  DataType : TExprType;
  IsDesc   : Boolean;
  TempS    : String;

  function SortCompare_S(Recno: Integer; const Value: String): Integer;
  var
     s : String;
  begin
     SetRecno(Recno);
     s := FFields[Idx].AsString;
     if s = Value then
     begin
        Result := 0; Exit;
     end;
     if IsDesc then
     begin
        if s < Value then Result := 1
        else Result := -1;
     end else
     begin
        if s < Value then Result := -1
        else Result := 1;
     end;
  end;

  function SortCompare_F(Recno: Integer; const Value: Double): Integer;
  var
     f : Double;
  begin
     SetRecno(Recno);
     f := FFields[Idx].AsFloat;
     if f = Value then
     begin
        Result := 0; Exit;
     end;
     if IsDesc then
     begin
        if f < Value then Result := 1
        else Result := -1;
     end else
     begin
        if f < Value then Result := -1
        else Result := 1;
     end;
  end;

  function SortCompare_I(Recno: Integer; Value: Integer): Integer;
  var
     i : Integer;
  begin
     SetRecno(Recno);
     i := FFields[Idx].AsInteger;
     if i = Value then
     begin
        Result := 0; Exit;
     end;
     if IsDesc then
     begin
        if i < Value then Result := 1
        else Result := -1;
     end else
     begin
        if i < Value then Result := -1
        else Result := 1;
     end;
  end;

  function SortCompare_B(Recno: Integer; Value: Boolean): Integer;
  var
     b : Boolean;
  begin
     SetRecno(Recno);
     b := FFields[Idx].AsBoolean;
     if Ord(b) = Ord(Value) then
     begin
        Result := 0; Exit;
     end;
     if IsDesc then
     begin
        if Ord(b) < Ord(Value) then Result := 1
        else Result := -1;
     end else
     begin
        if Ord(b) < Ord(Value) then Result := -1
        else Result := 1;
     end;
  end;

  procedure QuickSort(L, R: Integer);
  var
    I, J, P : Integer;
    s1 : String;
    f1 : Double;
    i1 : Integer;
    b1 : Boolean;
  begin
    repeat
      I := L;
      J := R;
      P := (L + R) shr 1;
      SetRecno(P);
      case DataType of
         ttString  : s1 := FFields[Idx].AsString;
         ttFloat   : f1 := FFields[Idx].AsFloat;
         ttInteger : i1 := FFields[Idx].AsInteger;
         ttBoolean : b1 := FFields[Idx].AsBoolean;
      end;
      repeat
        case DataType of
           ttString  :
              begin
                while SortCompare_S(I, s1) < 0 do Inc(I);
              end;
           ttFloat   :
              begin
                while SortCompare_F(I, f1) < 0 do Inc(I);
              end;
           ttInteger :
              begin
                while SortCompare_I(I, i1) < 0 do Inc(I);
              end;
           ttBoolean :
              begin
                while SortCompare_B(I, b1) < 0 do Inc(I);
              end;
        end;

        case DataType of
           ttString  :
              begin
                while SortCompare_S(J, s1) > 0 do Dec(J);
              end;
           ttFloat   :
              begin
                while SortCompare_F(J, f1) > 0 do Dec(J);
              end;
           ttInteger :
              begin
                while SortCompare_I(J, i1) > 0 do Dec(J);
              end;
           ttBoolean :
              begin
                while SortCompare_B(J, b1) > 0 do Dec(J);
              end;
        end;
        if I <= J then
        begin
          Exchange(I, J);
          Inc(I);
          Dec(J);
        end;
      until I > J;
      if L < J then QuickSort(L, J);
      L := I;
    until I >= R;
  end;

begin
   if (FFields.Count = 0) or (GetRecordCount = 0) then Exit;
   Idx      := 0;
   DataType := FFields[0].DataType;
   IsDesc   := FFields[0].Desc;
   QuickSort(1, GetRecordCount);
   for Idx  := 1 to FFields.Count - 1 do
   begin
      SetRecno(1);
      DataType := FFields[Idx].DataType;
      IsDesc   := FFields[Idx].Desc;
      Index    := 1;
      Pivot    := 1;
      TempS    := FFields[Idx-1].AsString;
      while Index <= GetRecordCount do
      begin
         SetRecno(Index);
         if TempS <> FFields[Idx-1].AsString then
         begin
            if Index - 1 > Pivot then
               QuickSort(Pivot, Index - 1);
            Pivot := Index;
            SetRecno(Pivot);
            TempS := FFields[Idx-1].AsString;
            Index := Pivot - 1;
         end;
         Inc(Index);
      end;
   end;
end;

{-------------------------------------------------------------------------------}
{                  implements TMemSortList                                      }
{-------------------------------------------------------------------------------}

constructor TMemSortList.Create;
begin
  inherited Create;
  FBufferList := TList.Create;
end;

destructor TMemSortList.Destroy;
begin
   Clear;
   FBufferList.Free;
   inherited Destroy;
end;

procedure TMemSortList.Clear;
var
   I     : Integer;
   Buffer: PChar;
begin
   for I := 0 TO FBufferList.Count - 1 do
   begin
      Buffer := FBufferList[I];
      FreeMem(Buffer, FRecordBufferSize);
   end;
   FBufferList.Clear;
   FFields.Clear;
   FRecordBufferSize := SizeOf(Integer);
   FRecno := -1;
end;

function TMemSortList.ActiveBuffer: PChar;
begin
   Result := nil;
   if (FRecNo < 1) or (FRecNo > FBufferList.Count) then Exit;
   Result := FBufferList[FRecNo - 1];
end;

function TMemSortList.GetFieldData(Field: TSrtField; Buffer: Pointer): Boolean;
var
   RecBuf: PChar;
begin
   Result := False;
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( (RecBuf + Field.BufferOffset)^, Buffer^, Field.DataSize);
   Result := True;
end;

procedure TMemSortList.SetFieldData(Field: TSrtField; Buffer: Pointer);
var
   RecBuf: PChar;
begin
   RecBuf := ActiveBuffer;
   if (RecBuf = nil) or (Buffer = nil) then Exit;
   Move( Buffer^, (RecBuf + Field.BufferOffset)^, Field.DataSize);
end;

procedure TMemSortList.Insert;
var
   Buffer: PChar;
begin
   GetMem(Buffer, FRecordBufferSize);
   FillChar(Buffer^,FRecordBufferSize,0);
   FBufferList.Add( Buffer );
   FRecno := FBufferList.Count;
end;

function TMemSortList.GetRecordCount: Integer;
begin
   Result := FBufferList.Count;
end;

procedure TMemSortList.Exchange(Recno1,Recno2: Integer);
begin
   FBufferList.Exchange(Recno1-1,Recno2-1);
end;

function TMemSortList.GetSourceRecno: Integer;
var
   Buffer: PChar;
begin
   Result := 0;
   if (FRecno < 1) or (FRecno > GetRecordCount) then Exit;
   Buffer := PChar(FBufferList[FRecno - 1]);
   Move((Buffer + 0)^, Result, SizeOf(Integer));
end;

procedure TMemSortList.SetSourceRecno(Value: Integer);
var
   Buffer: PChar;
begin
   if (FRecno <1) or (FRecno >GetRecordCount) then Exit;
   Buffer := PChar(FBufferList[FRecno - 1]);
   Move(Value, (Buffer + 0)^, SizeOf(Integer));
end;

{-------------------------------------------------------------------------------}
{                  implements TFileSortList                                     }
{-------------------------------------------------------------------------------}

constructor TFileSortList.Create(MapFileSize : Longint);
begin
  inherited Create;
  FBufferList := TList.Create;
  FTmpFile    := GetTemporaryFileName('~dt');
  FMemMapFile := TMemMapFile.Create(FTmpFile, fmCreate, MapFileSize, True);
end;

destructor TFileSortList.Destroy;
begin
   Clear;
   FreeObject(FMemMapFile);
   SysUtils.DeleteFile(FTmpFile);
   FBufferList.Free;
   if Assigned(FBuffer) then
      FreeMem(FBuffer, FRecordBufferSize);
   inherited Destroy;
end;

procedure TFileSortList.Clear;
begin
   FMemMapFile.Seek(0,0);
   FBufferList.Clear;
   FFields.Clear;
   FRecordBufferSize := SizeOf(Integer);
   FRecno := -1;
end;

function TFileSortList.ActiveBuffer: PChar;
begin
   Result := nil;
   if (FRecNo < 1) or (FRecNo > FBufferList.Count) then Exit;
   if not Assigned(FBuffer) then
      GetMem(FBuffer, FRecordBufferSize);
   FMemMapFile.Seek(Longint(FBufferList[FRecno - 1]), 0);
   FMemMapFile.Read(FBuffer^, FRecordBufferSize);
   Result := FBuffer;
end;

function TFileSortList.GetFieldData(Field: TSrtField; Buffer: Pointer): Boolean;
var
   RecBuf: PChar;
begin
   Result := False;
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( (RecBuf + Field.BufferOffset)^, Buffer^, Field.DataSize);
   Result := True;
end;

procedure TFileSortList.SetFieldData(Field: TSrtField; Buffer: Pointer);
var
   RecBuf: PChar;
begin
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( Buffer^, (RecBuf + Field.BufferOffset)^, Field.DataSize);
   FMemMapFile.Seek(Longint(FBufferList[Recno - 1]), 0);
   FMemMapFile.Write(RecBuf^, FRecordBufferSize);
end;

procedure TFileSortList.Insert;
var
   Offset: Integer;
begin
   if not Assigned(FBuffer) then
      GetMem(FBuffer, FRecordBufferSize);
   FillChar(FBuffer^,FRecordBufferSize,0);
   Offset := FMemMapFile.VirtualSize;
   FMemMapFile.Seek(Offset, 0);
   FMemMapFile.Write(FBuffer^,FRecordBufferSize);
   FBufferList.Add(Pointer(Offset)); { the address in temp file is saved }
   FRecno := FBufferList.Count;
end;

function TFileSortList.GetRecordCount: Integer;
begin
   Result := FBufferList.Count;
end;

procedure TFileSortList.Exchange(Recno1,Recno2: Integer);
begin
   FBufferList.Exchange(Recno1-1,Recno2-1);
end;

procedure TFileSortList.SetSourceRecno(Value: Integer);
begin
   if (FRecno < 1) or (FRecno > GetRecordCount) then Exit;
   FMemMapFile.Seek(Longint(FBufferList[FRecno - 1]), 0);
   FMemMapFile.Write(Value, SizeOf(Integer));
end;

function TFileSortList.GetSourceRecno: Integer;
var
   RecBuf: PChar;
begin
   Result:= -1;
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( (RecBuf + 0)^, Result, SizeOf(Integer));
end;


{-------------------------------------------------------------------------------}
{ Implementation of TAggregateItem                                              }
{-------------------------------------------------------------------------------}

constructor TAggregateItem.Create(AggregateList : TAggregateList);
begin
   inherited Create;
   FAggregateList := AggregateList;

   FSparseList := TAggSparseList.Create(1000);
end;

destructor TAggregateItem.Destroy;
begin
   FSparseList.Free;
   inherited Destroy;
end;

{-------------------------------------------------------------------------------}
{ Implementation of TAggregateList                                              }
{-------------------------------------------------------------------------------}

constructor TAggregateList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TAggregateList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TAggregateList.GetCount;
begin
   Result := FItems.Count;
end;

function TAggregateList.GetItem(Index: Integer): TAggregateItem;
begin
   Result := FItems[Index];
end;

function TAggregateList.Add: TAggregateItem;
begin
   Result := TAggregateItem.Create(Self);
   FItems.Add(Result);
end;

procedure TAggregateList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TAggregateItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TAggregateList.Delete(Index: Integer);
begin
   TAggregateItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

procedure TAggregateList.Assign(AggregateList : TAggregateList);
var
   I         : Integer;
   Aggr : TAggregateItem;
begin
   Clear;
   for I := 0 to AggregateList.Count - 1 do
   begin
      Aggr := AggregateList[I];
      with Self.Add do
      begin
         AggregateStr := Aggr.AggregateStr;
         ColIndex     := Aggr.ColIndex;
         Aggregate    := Aggr.Aggregate;
         IsDistinctAg := Aggr.IsDistinctAg;
      end;
   end;
end;

{-------------------------------------------------------------------------------}
{  Implements TReferencedDataSetItem                                            }
{-------------------------------------------------------------------------------}
constructor TReferencedDataSetItem.Create(RefDataSetList : TReferencedDataSetList);
begin
   inherited Create;
   FReferencedDataSets := RefDataSetList;
end;

{-------------------------------------------------------------------------------}
{  Implements TReferencedDataSetList                                            }
{-------------------------------------------------------------------------------}

constructor TReferencedDataSetList.Create;
begin
   inherited Create;
   FItems:= TList.Create;
end;

destructor TReferencedDataSetList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TReferencedDataSetList.GetCount: Integer;
begin
   Result := FItems.Count;
end;

function TReferencedDataSetList.GetItem(Index: Integer): TReferencedDataSetItem;
begin
   Result := FItems[Index];
end;

function TReferencedDataSetList.Add: TReferencedDataSetItem;
begin
   Result := TReferencedDataSetItem.Create(Self);
   FItems.Add(Result);
end;

procedure TReferencedDataSetList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TReferencedDataSetItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TReferencedDataSetList.Delete(Index: Integer);
begin
   TReferencedDataSetItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

function TReferencedDataSetList.IndexOf(DataSet: TDataSet): Integer;
begin
   Result := FItems.IndexOf(DataSet);
end;

{-------------------------------------------------------------------------------}
{  Implements TMemMapFile                                                       }
{-------------------------------------------------------------------------------}

constructor TMemMapFile.Create(FileName: String; FileMode: integer;
                               Size: integer; MapNow: Boolean);
{ Creates Memory Mapped view of FileName file.
  FileName: Full pathname of file.
  FileMode: Use fmXXX constants.
  Size: size of memory map.  Pass zero as the size to use the
        file's own size.
}
begin

  { Initialize private fields }
  FMapNow := MapNow;
  FFileName := FileName;
  FFileMode := FileMode;

  AllocFileHandle;  // Obtain a file handle of the disk file.
  { Assume file is < 2 gig  }

  FFileSize := GetFileSize(FFileHandle, Nil);
  FSize := Size;

  try
    AllocFileMapping; // Get the file mapping object handle.
  except
    on ExQueryError do
    begin
      CloseHandle(FFileHandle);  // close file handle on error
      FFileHandle := 0;          // set handle back to 0 for clean up
      raise;                     // re-raise exception
    end;
  end;
  if FMapNow then
    AllocFileView;  // Map the view of the file
end;

destructor TMemMapFile.Destroy;
begin

  if FFileHandle <> 0 then
    CloseHandle(FFileHandle); // Release file handle.

  { Release file mapping object handle }
  if FMapHandle <> 0 then
    CloseHandle(FMapHandle);

  FreeMapping; { Unmap the file mapping view . }
  inherited Destroy;
end;

procedure TMemMapFile.FreeMapping;
{ This method unmaps the view of the file from this process's address
  space. }
begin
  if FData <> Nil then
  begin
    UnmapViewOfFile(FData);
    FData := Nil;
  end;
end;

function TMemMapFile.GetSize: Longint;
begin
  if FSize <> 0 then
    Result := FSize
  else
    Result := FFileSize;
end;

procedure TMemMapFile.AllocFileHandle;
{ creates or opens disk file before creating memory mapped file }
begin
  if FFileMode = fmCreate then
    FFileHandle := FileCreate(FFileName)
  else
    FFileHandle := FileOpen(FFileName, FFileMode);

  if FFileHandle < 0 then
    raise ExQueryError.Create(SFailOpenFile);
end;

procedure TMemMapFile.AllocFileMapping;
var
  ProtAttr: DWORD;
begin
  if FFileMode = fmOpenRead then  // obtain correct protection attribute
    ProtAttr := Page_ReadOnly
  else
    ProtAttr := Page_ReadWrite;
  { attempt to create file mapping of disk file.
    Raise exception on error. }
  FMapHandle := CreateFileMapping(FFileHandle, Nil, ProtAttr,
      0, FSize, Nil);
  if FMapHandle = 0 then
    raise ExQueryError.Create(SFailCreateMapping);
end;

procedure TMemMapFile.AllocFileView;
var
  Access: Longint;
begin
  if FFileMode = fmOpenRead then // obtain correct file mode
    Access := File_Map_Read
  else
    Access := File_Map_All_Access;
  FData := MapViewOfFile(FMapHandle, Access, 0, 0, FSize);
  if FData = Nil then
    raise ExQueryError.Create(SFailMapView);
end;

function TMemMapFile.Read(var Buffer; Count: Longint): Longint;
begin
  if FPosition + Count > GetSize then
    raise ExQueryError.Create(SBeyondEOF);
  Move((FData + FPosition)^, Buffer, Count);
  Inc(FPosition,Count);
end;

function TMemMapFile.Write(const Buffer; Count: Longint): Longint;
begin
  Move(Buffer, (FData + FPosition)^, Count);
  Inc(FPosition,Count);
  FVirtualSize := IMax(FPosition, FVirtualSize);
end;

function TMemMapFile.Seek(Offset: Longint; Origin: Word): Longint;
begin
  FPosition := Offset;      // only from beginning supported (Origin = 0)
end;


{-------------------------------------------------------------------------------}
{       Expression evaluator section                                            }
{-------------------------------------------------------------------------------}
    { This are the classes derived from prExprQ.Tfunction that will be used
       in TMainExpr... }
type

    {return value of a field in a database}
    TFieldExpr = Class(Tfunction)
    private
      FField      : TField;   {the field that the expression refers to}
      FxQuery     : TCustomxQuery;
      FIsMaxWidth : Boolean;  { calculate max width }
    protected
      function GetAsString: String; override;
      function GetAsFloat: Double; override;
      function GetAsInteger: prInteger; override;
      function GetAsBoolean: Boolean; override;
      function GetExprType: TExprtype; override;
    public
      constructor Create( ParameterList : TParameterList;
                          F             : TField;
                          xQuery        : TCustomxQuery ;
                          IsMaxWidth    : Boolean );
      property Field : TField read FField;
    end;

    {Evaluate FormatDateTime('dd/mmm/yyyy', NOW)}
    TFormatDateTimeExpr = Class(Tfunction)
    protected
       function GetAsString: string; override;
       function GetExprtype: TExprtype; override;
    end;

    {Evaluate FormatFloat('###,###,##0.00', 12345.567)}
    TFormatFloatExpr = Class(Tfunction)
    protected
       function GetAsString: string; override;
       function GetExprtype: TExprtype; override;
    end;

    {Evaluate Format(Format,Args)
     FORMAT('%s %d', FIRSTNAME, CUSTNO)}
    TFormatExpr = Class(Tfunction)
    protected
       function GetAsString: string; override;
       function GetExprtype: TExprtype; override;
    end;

    {returns Now() value}
    TNowExpr = Class(Tfunction)
    protected
       function GetAsFloat: Double; override;
       function GetExprtype: TExprtype; override;
    end;

    TStrToDateExpr = Class(Tfunction)
    protected
       function GetAsFloat: Double; override;
       function GetExprtype: TExprtype; override;
    end;

    {return LEFT(s,Count) value}
    TLeftExpr = Class(Tfunction)
    protected
       function GetAsString: String; override;
       function GetExprtype: TExprtype; override;
    end;

    {return RIGHT(s,Count) value}
    TRightExpr = Class(Tfunction)
    protected
       function GetAsString: String; override;
       function GetExprtype: TExprtype; override;
    end;

    { This function is used exclusively for the IN predicate in SQL SELECT
       something like this : SELECT * FROM customer WHERE CustNo IN (1,10,8) }
    TSQLInPredicateExpr = Class(Tfunction)
    private
      FIsNotIn: Boolean;
    protected
      function GetAsBoolean: Boolean; override ;
      function GetExprtype: TExprtype; override;
    public
      constructor Create(ParameterList: TParameterList; IsNotIn: Boolean);
    end;

    { This function is used exclusively for the LIKE predicate in SQL SELECT
       something like this : SELECT * FROM CUSTOMER WHERE NAME LIKE 'AL%' }
    TLikeCode = (lcNone, lcLeft, lcMiddle, lcRight);

    TLikeItem = class(TObject)
    public
      LikeText: String;       { text to find }
      LikeCode: TLikeCode;    { text must go at left, middle, right or on a column }
    end;

    TLikeList = class(TObject)
    private
       FItems: TList;
       function GetCount: Integer;
       function GetItem(Index: Integer): TLikeItem;
    public
       constructor Create;
       destructor Destroy; override;
       function Add: TLikeItem;
       procedure Clear;
       procedure Delete(Index: Integer);
       property Count: Integer read GetCount;
       property Items[Index: Integer]: TLikeItem read GetItem; default;
    end;

    TSQLLikeExpr = Class(Tfunction)
    private
      LikeList    : TLIKEList;
      FIsNotLike  : Boolean;
      function SQLPos(var Start: Integer; const Substr, Str: String): Integer;
    protected
      function GetAsBoolean: Boolean; override ;
      function GetExprtype: TExprtype; override;
    public
      constructor Create(ParameterList: TParameterList; IsNotLike: Boolean);
      destructor Destroy; override;
      procedure AddToList(Like: TLikeItem);
    end;

    { is used for TRIM SQL like function
       syntax is : TRIM([LEADING|TRAILING|BOTH] [trimmed_char] FROM column_reference)
    }
    TSQLTrimExpr = Class(Tfunction)
    protected
      function GetAsString: String; override;
      function GetExprtype: TExprtype; override;
    end;

    { is used for EXTRACT SQL like function
       syntax is : EXTRACT(YEAR|MONTH|DAY FROM column_reference)
    }
    TSQLExtractExpr = Class(Tfunction)
    protected
      function GetAsFloat: Double; override;
      function GetExprtype: TExprtype; override;
    end;

    {  MINOF(arg1,arg2, ..., argn), MAXOF(ARG1,ARG2, ... ,argn)
        hint by: Fady Geagea
    }
    TMinMaxOfExpr = Class(Tfunction)
    private
      FIsMin: Boolean;
    protected
      function GetAsFloat: Double; override;
      function GetExprtype: TExprtype; override;
    public
      constructor Create(ParameterList: TParameterList; IsMin: Boolean);
    end;

    {for solving extra functions defined with property TxQuery.Exfunctions}
    TExfunctionsExpr = Class(Tfunction)
    private
      FxQuery     : TCustomXQuery;
      FIdentifier : string;
      FResulttype : TExprtype;
      {I will save here the ParameterList pointer because declaration is
       private in prExprQ.pas and I don't want to modify it}
      FParams     : TParameterList;
    protected
      function GetAsString: String; override;
      function GetAsFloat: Double; override;
      function GetAsInteger: prInteger; override;
      function GetAsBoolean: Boolean; override;
      function GetExprtype: TExprtype; override;
    public
      constructor Create( ParameterList: TParameterList; xQuery: TCustomXQuery;
         const Identifier: string; Resulttype: TExprtype );
    end;

    TISNULLExpr = Class(Tfunction)
    protected
      function GetAsBoolean: Boolean; override;
      function GetExprtype: TExprtype; override;
    end;

{ TFieldExpr - class implementation}
constructor TFieldExpr.Create( ParameterList : TParameterList;
                               F             : TField;
                               xQuery        : TCustomxQuery;
                               IsMaxWidth    : Boolean );
begin
  inherited Create( ParameterList );
  FField      := F;
  FxQuery     := xQuery;
  FIsMaxWidth := IsMaxWidth;
end;

function TFieldExpr.GetExprType: TExprtype;
begin
  if FField.Datatype in ftNonTexttypes then
     Result := ttInteger
  else
     case FField.Datatype of
        ftString{$IFDEF LEVEL4},ftFixedChar,ftWideString{$endIF} : Result := ttString;
        ftFloat,ftCurrency,ftBCD,ftDate,ftTime,ftDateTime: Result := ttFloat;
        ftAutoInc,ftSmallInt,ftInteger,ftWord
        {$ifndef LEVEL3},ftLargeInt{$endif} : Result := ttInteger;
        ftBoolean: Result := ttBoolean;
     else
        result := ttString;
     end;
end;

function TFieldExpr.GetAsString: string;
begin
  Result := '';
  if FxQuery.IsDataSetDisabled(FField.DataSet) then Exit;
  if not (FField.DataType in ftNonTextTypes) then
  begin
     if FIsMaxWidth and
        (FField.DataType in [ftString{$IFDEF LEVEL4},ftFixedChar,ftWideString{$endif}]) then
     begin
        SetLength(Result, FField.Size);
        FillChar(Result[1], FField.Size, #32);
     end
     else
        Result := FField.AsString;
  end;
end;

function TFieldExpr.GetAsFloat: Double;
begin
  Result := 0;
  if FxQuery.IsDataSetDisabled(FField.DataSet) then Exit;
  Result := FField.AsFloat;
end;

function TFieldExpr.GetAsInteger: prInteger;
begin
  Result := 0;
  if FxQuery.IsDataSetDisabled( FField.DataSet ) then Exit;
  Result := FField.AsInteger;
end;

function TFieldExpr.GetAsBoolean: Boolean;
begin
  Result := False;
  if FxQuery.IsDataSetDisabled( FField.DataSet ) then Exit;
  Result := FField.AsBoolean;
end;


{TStrToDate class implementation}
function TStrToDateExpr.GetAsFloat: Double;
begin
  try
     Result := StrToDate(Param[0].AsString);
  except
     on E:Exception do
     begin
        MessageToUser(E.Message,mtError);
        Result := 0;
     end;
  end;
end;

function TStrToDateExpr.GetExprtype: TExprtype;
begin
   Result := ttFloat;
end;

{ TNowExpr class implementation }
function TNowExpr.GetAsFloat: Double;
begin
  Result := Now;
end;

function TNowExpr.GetExprtype: TExprtype;
begin
  Result := ttFloat;
end;


{TFormatDateTimeExpr - class implementation}
function TFormatDateTimeExpr.GetAsString: string;
begin
  Result := FormatDateTime(Param[0].AsString, Param[1].AsFloat);
end;

function TFormatDateTimeExpr.GetExprtype: TExprtype;
begin
  Result := ttString;
end;

{ TFormatFloatExpr - class implementation }
function TFormatFloatExpr.GetAsString: string;
begin
  Result := FormatFloat(Param[0].AsString, Param[1].AsFloat);
end;

function TFormatFloatExpr.GetExprtype: TExprtype;
begin
  Result := ttString;
end;

{ TFormatExpr - class implementation
  Format('%d %s ...', 1234, 'ABC', ..., etc) }
function TFormatExpr.GetAsString: string;
const
   MAXARGS = 20;  {maximum number of arguments allowed (increase if needed)}
var
   cnt,n : integer;
   ss  : array[0..MAXARGS] of ShortString;
   ea  : array[0..MAXARGS] of Extended;
   vars: array[0..MAXARGS] of TvarRec;
begin
   n := imin( ParameterCount - 1, MAXARGS );
   { first parameter is the format string and the rest are the args}
   for cnt :=1 to n do
   begin
      case Param[cnt].Exprtype of
         ttString:
           begin
           vars[cnt-1].Vtype := vtString;
           ss[cnt-1] := Param[cnt].AsString;
           vars[cnt-1].VString := @ss[cnt-1];
           end;
         ttFloat:
           begin
           vars[cnt-1].Vtype := vtExtended;
           ea[cnt-1] := Param[cnt].AsFloat;
           vars[cnt-1].VExtended := @ea[cnt-1];
           end;
         ttInteger:
           begin
           vars[cnt-1].Vtype := vtInteger;
           vars[cnt-1].VInteger := Param[cnt].AsInteger;
           end;
         ttBoolean:
           begin
           vars[cnt-1].Vtype := vtBoolean;
           vars[cnt-1].VBoolean := Param[cnt].AsBoolean;
           end;
      end;
   end;
   result := Format(Param[0].AsString, vars);
end;

function TFormatExpr.GetExprtype: TExprtype;
begin
   result := ttString;
end;


{TSQLTrimExpr - class implementation }
function TSQLTrimExpr.GetAsString: String;

   function SQLTrim(trimmed_char: char; const S: string): string;
   var
     I, L: Integer;
   begin
     L := Length(S);
     I := 1;
     while (I <= L) and (S[I] = trimmed_char) do Inc(I);
     if I > L then
       Result := ''
     else
     begin
       while S[L] = trimmed_char do Dec(L);
       Result := Copy(S, I, L - I + 1);
     end;
   end;

   function SQLTrimLeft(trimmed_char: char; const S: string): string;
   var
     I, L: Integer;
   begin
     L := Length(S);
     I := 1;
     while (I <= L) and (S[I] = trimmed_char) do Inc(I);
     Result := Copy(S, I, Maxint);
   end;

   function SQLTrimRight(trimmed_char: char; const S: string): string;
   var
     I: Integer;
   begin
     I := Length(S);
     while (I > 0) and (S[I] = trimmed_char) do Dec(I);
     Result := Copy(S, 1, I);
   end;

begin
   case Param[2].AsInteger of
      0: // leading
         Result := SQLTrimLeft(Param[0].AsString[1], Param[1].AsString);
      1: // trailing
         Result := SQLTrimRight(Param[0].AsString[1], Param[1].AsString);
      2: // both
         Result := SQLTrim(Param[0].AsString[1], Param[1].AsString);
   end;
end;

function TSQLTrimExpr.GetExprtype: TExprtype;
begin
   Result := ttString;
end;


{TSQLExtractExpr - class implementation }
function TSQLExtractExpr.GetAsFloat: Double;
var
   year,month,day,hour,min,sec,msec:Word;
begin
   if Param[1].AsInteger in [0..2] then begin
      DecodeDate(Param[0].AsFloat,year,month,day);
      case Param[1].AsInteger of
         0: Result := year;
         1: Result := month;
         2: Result := day;
      else
         Result := 0.0;
      end;
   end else if Param[1].AsInteger in [3..5] then begin
      DecodeTime(Param[0].AsFloat,hour,min,sec,msec);
      case Param[1].AsInteger of
         3: Result := hour;
         4: Result := min;
         5: Result := sec;
      else
         Result := 0.0;
      end;
   end;
end;

function TSQLExtractExpr.GetExprtype: TExprtype;
begin
   Result := ttFloat;
end;

{TMinMaxOfExpr - class implementation }
constructor TMinMaxOfExpr.Create(ParameterList: TParameterList; IsMin: Boolean);
begin
   inherited Create( ParameterList );
   FIsMin := IsMin;
end;

function TMinMaxOfExpr.GetAsFloat: Double;
var
   i: Integer;
begin
   Result := Param[0].AsFloat;
   for i := 1 to ParameterCount - 1 do
   begin
      if FIsMin then
         Result := Min(Result, Param[i].AsFloat)
      else
         Result := Max(Result, Param[i].AsFloat);
   end;
end;

function TMinMaxOfExpr.GetExprtype: TExprtype;
begin
   Result := ttFloat;
end;

{ TISNULLExpr implementation }
function TISNULLExpr.GetAsBoolean: Boolean;
begin
   Result:= (Param[0] as TFieldExpr).Field.IsNull;
   if Param[1].AsBoolean = False then
      Result := Not Result;
end;

function TISNULLExpr.GetExprtype: TExprtype;
begin
   Result:= ttBoolean;
end;

{TLeftExpr - class implementation }
function TLeftExpr.GetAsString: String;
begin
   Result := Copy(Param[0].AsString, 1, Param[1].AsInteger);
end;

function TLeftExpr.GetExprtype: TExprtype;
begin
   Result := ttString;
end;

{TRightExpr - class implementation }
function TRightExpr.GetAsString: String;
var
   p: Integer;
begin
   p := IMax(1, Length(Param[0].AsString) - Param[1].AsInteger + 1);
   Result := Copy(Param[0].AsString, p, Param[1].AsInteger);
end;

function TRightExpr.GetExprtype: TExprtype;
begin
   Result := ttString;
end;

{ TSQLInPredicateExpr - class implementation}
constructor TSQLInPredicateExpr.Create(ParameterList: TParameterList; IsNotIn: Boolean);
begin
   inherited Create(ParameterList);
   FIsNotIn := IsNotIn;
end;

function TSQLInPredicateExpr.GetAsBoolean: Boolean;
var
   t: Integer;
   s: String;
   f: Double;
   i: Integer;
   b: Boolean;
begin
   Result := False;
   { `We'll compare expressions like
       COUNTRY IN ('USA','SPAIN','MEXICO','ENGLAND')
       CUSTID not IN (1,10,25)
       ISMARRIED IN (TRUE)
       Combination of parameters like:
       CUSTID IN ('USA', 2, 'MEXICO', 2.54)
       where CUSTID is integer, is invalid
   }
   case Param[0].Exprtype of
      ttString:
         begin
            s := Param[0].AsString;
            for t := 1 to ParameterCount - 1 do
               if s = Param[t].AsString then begin
                  Result := True; Break;
               end;
         end;
      ttFloat:
         begin
            f := Param[0].AsFloat;
            for t := 1 to ParameterCount - 1 do
               if f = Param[t].AsFloat then begin
                  Result := True; Break;
               end;
         end;
      ttInteger:
         begin
            i := Param[0].AsInteger;
            for t := 1 to ParameterCount - 1 do
               if i = Param[t].AsInteger then begin
                  Result := True; Break;
               end;
         end;
      ttBoolean:
         begin
            b := Param[0].AsBoolean;
            for t := 1 to ParameterCount - 1 do
               if b = Param[t].AsBoolean then begin
                  Result := True; Break;
               end;
         end;
   end;
   if FIsNotIn then Result := not Result;
end;

function TSQLInPredicateExpr.GetExprtype: TExprtype;
begin
   Result := ttBoolean;
end;


{ TSQLLikeExpr - class implementation
  evaluate LIKE expressions as:
  SELECT * FROM CUSTOMER WHERE NAME LIKE 'ROBERT%';
  SELECT * FROM CUSTOMER WHERE NAME LIKE '%ROB%BROWN%';
  SELECT * FROM CUSTOMER WHERE NAME LIKE 'BROWN%';
  SELECT * FROM CUSTOMER WHERE NAME LIKE '%ROBERT%BR_WN%';
  }

{ TLikeList implementaton}
constructor TLikeList.Create;
begin
   inherited Create;
   FItems := TList.Create;
end;

destructor TLikeList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TLikeList.GetCount;
begin
   Result := FItems.Count;
end;

function TLikeList.GetItem(Index: Integer): TLikeItem;
begin
   Result := FItems[Index];
end;

function TLikeList.Add: TLikeItem;
begin
   Result := TLikeItem.Create;
   FItems.Add(Result);
end;

procedure TLikeList.Clear;
var
   I: Integer;
begin
   for I:= 0 to FItems.Count - 1 do
      TLikeItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TLikeList.Delete(Index: Integer);
begin
   TLikeItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{ TSQLLikeExpr implementation }
constructor TSQLLikeExpr.Create(ParameterList: TParameterList; IsNotLike: Boolean);
var
   s,Work     : String;
   p, n       : Integer;
   Previous   : Char;
   EscapeChar : Char;
   Accept     : Boolean;
begin
   inherited Create(ParameterList);
   LIKEList := TLikeList.Create;
   FIsNotLike := IsNotLike;
   if Length(Param[2].AsString) > 0 then
      EscapeChar := Param[2].AsString[1]
   else
      EscapeChar := #0;

   s := Param[1].AsString;
   if (Length(s) = 0) or (Pos('%',s) = 0) then
   begin
      with LikeList.Add do begin
         LikeText := s;
         LikeCode := lcNone;
      end;
   end else
   begin
      work := ''; p := 1; n := 0; Previous := #0;
      while p <= Length(s) do
      begin
         Accept := ((s[p] = '%') and (EscapeChar =#0)) or
                   ((s[p] = '%') and (Previous<>EscapeChar));
         if Accept then
         begin
            if (Length(Work) > 0) then
            begin
               if n = 0 then
               begin
                  // text must start with Work
                  with LikeList.Add do begin
                     LikeText := Work;
                     LikeCode := lcLeft;
                  end;
               end else
               begin
                  // el texto debe tener en medio work
                  with LikeList.Add do begin
                     LikeText := Work;
                     LikeCode := lcMiddle;
                  end;
               end;
            end;
            work := '';
            inc(n);
         end else
         begin
            if (EscapeChar=#0) or not(s[p]=EscapeChar) then
               work := work + s[p];
         end;
         Previous := s[p];

         Inc(p);
      end;
      if Length(work) > 0 then
      begin
         { texto deber terminar en Work }
         with LikeList.Add do begin
            LikeCode := lcRight;
            LikeText := Work;
         end;
      end;
   end;
end;

destructor TSQLLikeExpr.Destroy;
begin
   LIKEList.Free;
   inherited Destroy;
end;

function TSQLLikeExpr.SQLPos(var Start: Integer; const Substr, Str: String): Integer;
var
   I, Pivot, NumValid, L1, L2: Integer;
   Accept : Boolean;
begin
   Result := 0;
   L1 := Length(Str);
   L2 := Length(Substr);
   if (L1 = 0) or (L2 = 0) or (L2 > L1) then Exit;
   if (Start=1) and (Pos('_', Substr) = 0) then
   begin
      Result := Pos(Substr, Str);    // speed up result
      if Result > 0 then
         Inc(Start, Length(Substr));
   end else
   begin
      for I := Start to L1 do
      begin
         NumValid := 0;
         Pivot := 1;
         Accept := true;
         while Accept and (I + Pivot - 1 <= L1) and (Pivot <= L2) and
            ( (Substr[Pivot] = '_') or (Str[I + Pivot - 1] = Substr[Pivot]) ) do
         begin
            Inc(NumValid);
            Inc(Pivot);
         end;
         if NumValid = L2 then
         begin
            Inc(Start, Length(Substr));
            Result := I; Exit;
         end;
      end;
   end;
end;

procedure TSQLLikeExpr.AddToList(Like: TLikeItem);
begin
   with LikeList.Add do begin
      LikeCode := Like.LikeCode;
      LikeText := Like.LikeText;
   end;
end;

function TSQLLikeExpr.GetAsBoolean: Boolean;
var
   I, n, Start, p : Integer;
   Like           : TLikeItem;
   s0,s1          : String;
   Accept         : Boolean;
begin
   n := 0;
   s0 := Param[0].AsString;
   Start := 1;
   Accept := False; //Basri
   for I := 0 to LIKEList.Count - 1 do
   begin
      Like := LIKEList[I];
      s1 := Like.LikeText;
      case Like.LikeCode of
         lcNone: Accept := (s0 = s1);
         lcLeft:
            begin
            Start := 1;
            Accept := (SQLPos(Start, s1, s0) = 1);
            end;
         lcMiddle: Accept := (SQLPos(Start, s1, s0) > 0);
         lcRight:
            begin
            p := Length(s0) - Length(s1) + 1;
            if Start <= p then
            begin
               Start := p;
               Accept := (SQLPos(Start, s1, s0) = p);
            end else
               Accept := False;
            end;
      end;
      if Accept then Inc(n);
   end;
   Result := (n = LIKEList.Count);
   if FIsNotLike then Result := not Result;
end;

function TSQLLikeExpr.GetExprtype: TExprtype;
begin
   Result := ttBoolean;
end;


{TSubqueryExpr - class implementation}
constructor TSubqueryExpr.Create( ParameterList: TParameterList);
var
   i: TExprtype;
begin
   inherited Create(ParameterList);
   // find the expression type
   for i := Low(TExprtype) to High(TExprtype) do
      if AnsiCompareText(Param[0].AsString, prExprQ.NExprtype[i])=0 then
      begin
         FExprtype := i;
         Break;
      end;
   Value := Null;
end;

function TSubqueryExpr.GetAsString: String;
begin
   Result := Value;
end;

function TSubqueryExpr.GetAsFloat: Double;
begin
   Result := Value;
end;

function TSubqueryExpr.GetAsInteger: prInteger;
begin
   Result := Value;
end;

function TSubqueryExpr.GetAsBoolean: Boolean;
begin
   Result := Value;
end;

function TSubqueryExpr.GetExprtype: TExprtype;
begin
   Result := FExprtype;
end;

{functions defined in Exfunctions property}
constructor TExfunctionsExpr.Create( ParameterList: TParameterList; xQuery: TCustomXQuery;
   const Identifier: string; Resulttype: TExprtype );
begin
   inherited Create( ParameterList );
   FxQuery := xQuery;
   FIdentifier := Identifier;
   FParams := ParameterList;
   FResulttype := Resulttype;
end;

function TExfunctionsExpr.GetExprtype: TExprtype;
begin
  Result := FResulttype;
end;

function TExfunctionsExpr.GetAsString: string;
var
  Value: variant;
begin
  Value := Null;
  Result := '';
  FxQuery.OnfunctionSolve(FxQuery, FIdentifier, FParams, Value );
  if vartype(Value) <> varNull then
     Result := Value;    // varAstype(Value, varString);
end;

function TExfunctionsExpr.GetAsFloat: Double;
var
  Value: variant;
begin
  Value := Null;
  Result := 0;
  FxQuery.OnfunctionSolve(FxQuery, FIdentifier, FParams, Value );
  if vartype(Value) <> varNull then
     Result := Value;    // varAstype(Value, varDouble);
end;

function TExfunctionsExpr.GetAsInteger: prInteger;
var
  Value: variant;
begin
  Value := Null;
  Result := 0;
  FxQuery.OnfunctionSolve(FxQuery, FIdentifier, FParams, Value );
  if vartype(Value) <> varNull then
     Result := Value;    // varAstype(Value, varInteger);
end;

function TExfunctionsExpr.GetAsBoolean: Boolean;
var
  Value: variant;
begin
  Value := Null;
  Result := False;
  FxQuery.OnfunctionSolve(FxQuery, FIdentifier, FParams, Value );
  if vartype(Value) <> varNull then
     Result := Value;    // varAstype(Value, varBoolean);
end;


{TMainExpr - clas implementation}
constructor TMainExpr.Create( SqlAnalizer: TObject; DataSet: TDataSet );
begin
  inherited Create;
  FDefaultDataSet := DataSet;
  FAnalizer := SqlAnalizer;
end;

destructor TMainExpr.Destroy;
begin
  Expression.Free;
  inherited Destroy;
end;

function TMainExpr.IDFunc( const Identifier: string;
  ParameterList: TParameterList): TExpression;
var
  TmpDataSet   : TDataSet;
  FieldName,
  TableName    : String;
  I, NumError,
  Idx          : Integer;
  F            : TField;
  Accept       : Boolean;
  Item         : TExfunctionItem;
begin
  Result :=nil;
  NumError := 0;
  TmpDataSet := FDefaultDataSet;
  if Identifier='FIELD' then
  begin
     if Assigned(ParameterList) and (ParameterList.Count=2) and
        (ParameterList.Exprtype[0] = ttString) and
        (ParameterList.Exprtype[1] = ttString) then
     begin
        TableName := ParameterList.AsString[0];
        if Length(TableName) > 0 then
        begin
           TmpDataSet := (FAnalizer as TSqlAnalizer).FindDataSetByName(TableName);
           if not Assigned(TmpDataSet) then
              raise EExpression.Create(Format(SWrongDataSetname,[TableName]))
        end;
        FieldName := ParameterList.AsString[1];
        if TmpDataSet = FDefaultDataSet then
           Inc(CheckData.RefCount) ;             // number of primary dataset referenced
        (*else                                    // number of other datasets referenced in the expression
           Inc(CheckData.OtherRefCount);*)
        { create the list of datasets referenced in the expression }
        if Assigned(FReferencedDataSets) then
        begin
           Idx := FReferencedDataSets.IndexOf(TmpDataSet);
           if Idx >= 0 then
              FReferencedDataSets[Idx].Count := FReferencedDataSets[Idx].Count + 1
           else
              with FReferencedDataSets.Add do
              begin
                 DataSet := TmpDataSet;
                 Count := 1;
              end;
        end;
        // Create expression
        F := TmpDataSet.FindField(FieldName);
        if Assigned(F) then
        begin
           Result := TFieldExpr.Create( ParameterList,
                                        F,
                                        (FAnalizer as TSqlAnalizer).xQuery,
                                        Self.FCalcMaxWidth );
           if CheckData.RefCount = 1 then
              CheckData.Field := F;
           Inc(CheckData.FieldCount);   // used in multi-key joining
           if CheckData.FieldCount <= MAX_INDEXED_FIELDS then
              CheckData.Fields[CheckData.FieldCount] := F; // used in multi-key joining
        end;
     end else
        NumError := 1;
  end else if (Identifier='MINOF') or (Identifier='MAXOF') then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count > 1) then
     begin
        Accept := True;
        for i := 0 to ParameterList.Count - 1 do
            if not(ParameterList.Exprtype[i] in [ttFloat,ttInteger]) then
            begin
               Accept := False; Break;
            end;
        if Accept then
           Result := TMinMaxOfExpr.Create( ParameterList, (Identifier='MINOF'))
        else
           NumError := 1;
     end else
        NumError := 1;
  end else if Identifier='FORMAT' then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count>1) then
     begin
        if ParameterList.Exprtype[0] = ttString then
           Result := TFormatExpr.Create( ParameterList )
        else
           raise EExpression.CreateFmt(SWrongFirstArg, [Identifier]);
     end else
        NumError := 1;
  end else if Identifier='SQLEXTRACT' then   // SQL EXTRACT function
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 2) then
     begin
        if not( (ParameterList.Exprtype[0] in [ttFloat, ttInteger]) and
           (ParameterList.Exprtype[1] in [ttFloat, ttInteger]) ) then
           raise EExpression.Create(SWrongParamsInExtract);
        Result := TSQLExtractExpr.Create( ParameterList )
     end else
        NumError := 1;
  end else if Identifier='SQLTRIM' then  // SQL TRIM function
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 3) then
     begin
        if Length(ParameterList.AsString[0]) <>1 then
           raise EExpression.Create(SWrongLengthInTrim);
        Result := TSQLTrimExpr.Create( ParameterList )
     end else
        NumError := 1;
  end else if Identifier='SQLLIKE' then  // for use in sql select only
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 3) then
        Result := TSQLLikeExpr.Create( ParameterList, False )
     else
        NumError := 1;
  end else if Identifier='SQLNOTLIKE' then  // for use in sql select only
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 3) then
        Result := TSQLLikeExpr.Create( ParameterList, True )
     else
        NumError := 1;
  end else if Identifier='SQLIN' then  // for use in sql select only
  begin
     CheckData.HasMorefunctions := True;
     { not enough checking for now }
     if Assigned(ParameterList) and (ParameterList.Count > 1) then
        Result := TSQLInPredicateExpr.Create( ParameterList, False )
     else
        NumError := 1;
  end else if Identifier='SQLNOTIN' then  // for use in sql select only
  begin
     CheckData.HasMorefunctions := True;
     { not enough checking for now }
     if Assigned(ParameterList) and (ParameterList.Count > 1) then
        Result := TSQLInPredicateExpr.Create( ParameterList, True )
     else
        NumError := 1;
  end else if Identifier='ISNULL' then  // for use in sql select only
  begin
     CheckData.HasMorefunctions := True;
     { not enough checking for now }
     if Assigned(ParameterList) and (ParameterList.Count = 2 ) then
        Result := TISNULLExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='LEFT' then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 2) and
        (ParameterList.Exprtype[0] = ttString) and
        (ParameterList.Exprtype[1] = ttInteger) then
        Result := TLeftExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='RIGHT' then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 2) and
        (ParameterList.Exprtype[0] = ttString) and
        (ParameterList.Exprtype[1] = ttInteger) then
        Result := TRightExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='NOW' then
  begin
     CheckData.HasMorefunctions := True;
     if not Assigned(ParameterList) then
        Result := TNowExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='FORMATDATETIME' then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count = 2) then
        Result := TFormatDateTimeExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='FORMATFLOAT' then
  begin
     CheckData.HasMorefunctions := True;
     if Assigned(ParameterList) and (ParameterList.Count=2) then
        Result := TFormatFloatExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='SUBQUERY' then
  begin     { auxiliary for evaluating subqueries (not a real function) }
     Result := TSubqueryExpr.Create(ParameterList);
     Self.SubqueryExpr := Result;      { save the the pointer for be used in TSqlAnalizer.CreateResultSet }
  end else with (FAnalizer as TSqlAnalizer) do
  begin     { is this an extra function ? }
     for i := 0 to xQuery.Exfunctions.Count - 1 do
     begin
        Item := xQuery.Exfunctions[i];
        if AnsiCompareText(Item.Name, Identifier) = 0 then
        begin
           Accept := True;
           if Assigned(xQuery.OnfunctionCheck) then
              xQuery.OnfunctionCheck(xQuery, Identifier, ParameterList, Accept);
           if Accept and Assigned(xQuery.OnfunctionSolve) then
           begin
              CheckData.HasMorefunctions := True;
              Result := TExfunctionsExpr.Create(ParameterList, xQuery, Identifier,
                 Item.Resulttype);
           end else
              NumError := 1;      { wrong number of params }
           Break;
        end;
     end;
  end;

  if NumError = 1 then
     raise EExpression.CreateFmt(SWrongParameters, [Identifier]);
end;

procedure TMainExpr.ParseExpression(const ExprStr: string);
var
  s: string;
begin
  s := ExprStr;
  TrimCRLF(s);
  FreeObject(Expression);
  try
    if Length(s)>0 then
    begin
       Expression := CreateExpression(s, IDFunc);
       CheckData.HasMorefunctions := CheckData.HasMorefunctions or prExprQ.Hasfunctions;
    end;
  except
    FreeObject(Expression);
    raise;
  end;
end;

function TMainExpr.CheckExpression(const ExprStr: String): Boolean;
var
  s: String;
begin
  FillChar(CheckData, SizeOf(TCheckData), 0);
  Result := false;
  s := ExprStr;
  TrimCRLF(s);
  FreeObject(Expression);
  try
    if Length(s) > 0 then
    begin
       Expression := CreateExpression(s, IDFunc);
       CheckData.HasMorefunctions := CheckData.HasMorefunctions or prExprQ.Hasfunctions;
       Result := CheckData.RefCount > 0;
    end;
  except
    FreeObject(Expression);
    raise;
  end;
end;

{for reading fields in the dataset in TFilterExpr class}
type
    TFilterFieldExpr = Class(Tfunction)
    private
      FField: TField;   {the field that the expression refers to}
    protected
      function GetAsString: String; override;
      function GetAsFloat: Double; override;
      function GetAsInteger: prInteger; override;
      function GetAsBoolean: Boolean; override;
      function GetExprtype: TExprtype; override;
    public
      constructor Create( ParameterList: TParameterList; F: TField);
    end;

{ TFilterFieldExpr - class implementation}
constructor TFilterFieldExpr.Create(ParameterList: TParameterList; F: TField);
begin
  inherited Create( ParameterList );
  FField := F;
end;

function TFilterFieldExpr.GetExprtype: TExprtype;
begin
  if FField.Datatype in ftNonTexttypes then
     Result := ttInteger
  else
     case FField.Datatype of
        ftString{$IFDEF LEVEL4},ftFixedChar,ftWideString{$endIF} : Result := ttString;
        ftFloat,ftCurrency,ftBCD,ftDate,ftTime,ftDateTime: Result := ttFloat;
        ftAutoInc,ftSmallInt,ftInteger,ftWord
        {$ifndef LEVEL3},ftLargeInt{$endif} : Result := ttInteger;
        ftBoolean: Result := ttBoolean;
     else
        result :=ttString;
     end;
end;

function TFilterFieldExpr.GetAsString: string;
begin
  Result := '';
  if not (FField.Datatype in ftNonTexttypes) then
     Result := FField.AsString;
end;

function TFilterFieldExpr.GetAsFloat: Double;
begin
  Result := FField.AsFloat;
end;

function TFilterFieldExpr.GetAsInteger: prInteger;
begin
  Result := FField.AsInteger;
end;

function TFilterFieldExpr.GetAsBoolean: Boolean;
begin
  Result := FField.AsBoolean;
end;


{ TFilterExpr - class implementation
  This class is used for Filter property in TxQuery}
constructor TFilterExpr.Create( DataSet: TDataSet );
begin
  inherited Create;
  FDataSet := DataSet;
end;

destructor TFilterExpr.Destroy;
begin
  Expression.Free;
  inherited Destroy;
end;

function TFilterExpr.IDFunc( const Identifier: string;
  ParameterList: TParameterList): TExpression;
var
  NumError  : Integer;
  F         : TField;
begin
  Result :=nil;
  NumError := 0;
  if Identifier='LIKE' then  // for use in sql select only
  begin
     if Assigned(ParameterList) and (ParameterList.Count = 3) then
        Result := TSQLLikeExpr.Create( ParameterList, False )
     else
        NumError := 1;
  end else if Identifier='IN' then  // for use in sql select only
  begin
     { not enough checking for now }
     if Assigned(ParameterList) and (ParameterList.Count > 1) then
        Result := TSQLInPredicateExpr.Create( ParameterList, False )
     else
        NumError := 1;
  end else if Identifier='LEFT' then
  begin
     if Assigned(ParameterList) and (ParameterList.Count = 2) and
        (ParameterList.Exprtype[0] = ttString) and
        (ParameterList.Exprtype[1] = ttInteger) then
        Result := TLeftExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='RIGHT' then
  begin
     if Assigned(ParameterList) and (ParameterList.Count = 2) and
        (ParameterList.Exprtype[0] = ttString) and
        (ParameterList.Exprtype[1] = ttInteger) then
        Result := TRightExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='ISNULL' then
  begin
     if Assigned(ParameterList) and (ParameterList.Count = 2) and
        (ParameterList.Exprtype[1] = ttBoolean) then
        Result := TISNULLExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='STRTODATE' then
  begin
     if Assigned(ParameterList) and (ParameterList.Count = 1) and
        (ParameterList.Exprtype[0] = ttString) then
        Result := TStrToDateExpr.Create( ParameterList )
     else
        NumError := 1;
  end else if Identifier='NOW' then
  begin
     if not Assigned(ParameterList) then
        Result := TNowExpr.Create( ParameterList )
     else
        NumError := 1;
  end else
  begin
     // now look in the FDataSet fields
     F := FDataSet.FindField(Identifier);
     if Assigned(F) then
     begin
        if not Assigned(ParameterList) then
           Result := TFilterFieldExpr.Create(ParameterList,F)
        else
           NumError :=1;
     end;
  end;

  if NumError = 1 then
     raise EExpression.CreateFmt(SWrongParameters, [Identifier]);

end;

procedure TFilterExpr.ParseExpression(const ExprStr: string);
var
  s: string;
begin
  s := ExprStr;
  TrimCRLF(s);
  FreeObject(Expression);
  try
    if Length(s)>0 then
       Expression := CreateExpression(s, IDFunc);
  except
    FreeObject(Expression);
    raise;
  end;
end;

{-------------------------------------------------------------------------------}
{                  Implement TNestedList                                        }
{-------------------------------------------------------------------------------}
constructor TNestedList.Create;
begin
   inherited Create;
   FList := TList.Create;
end;

destructor TNestedList.Destroy;
begin
   Clear;
   FList.Free;
   inherited Destroy;
end;

procedure TNestedList.Clear;
var
   I : Integer;
begin
   for I := 0 to FList.Count - 1 do
      TNestedItem(FList[I]).Free;
   FList.Clear;
end;

function TNestedList.Get(Index: Integer): TNestedItem;
begin
   if (Index < 0) or (Index > FList.Count - 1) then
      raise ExQueryError.Create(SListError);
   Result:= TNestedItem(FList[Index]);
end;

procedure TNestedList.Add(JoinOnItem: TJoinOnItem; Idx1, Idx2: Integer);
var
   Item: TNestedItem;
begin
   Item := TNestedItem.Create;
   Item.JoinOnItem := JoinOnItem;
   Item.Idx1 := Idx1;
   Item.Idx2 := Idx2;
   FList.Add(Item);
end;

function TNestedList.Count: Integer;
begin
   Result := FList.Count;
end;

function TNestedList.IndexOf(Value: Integer): Integer;
var
   I : Integer;
begin
   Result := -1;
   for I := 0 to FList.Count - 1 do
      if (TNestedItem(FList[I]).Idx1 = Value) or
         (TNestedItem(FList[I]).Idx2 = Value) then
         begin
            Result := I;
            Exit;
         end;
end;

end.
