unit prExprQ;

{$I XQ_FLAG.INC}
interface
{main documentation block just before implementation
Copyright:   1997 Production Robots Engineering Ltd, all rights reserved.
Version:     1.03 31/1/98
Status:      Free for private or commercial use subject to following restrictions:
             * Use entirely at your own risk
             * Do not resdistribute without this note
             * Any redistribution to be free of charges
any questions to Martin Lafferty   robots@enterprise.net}

uses
  Classes,
  SysUtils;

type
{$IFDEF VER90}
  prInteger = Longint;
{$ELSE}
  prInteger = Integer;
{$ENDIF}

  TExprType = (ttString, ttFloat, ttInteger, ttBoolean, ttUnknown); //Basri ttUnknown added

  TExpression =
  class
  private
  protected
    function GetAsString: String; virtual;
    function GetAsFloat: Double; virtual;
    function GetAsInteger: prInteger; virtual;
    function GetAsBoolean: Boolean; virtual;
    function GetExprType: TExprType; virtual; abstract;
  public
    property AsString: String read GetAsString;
    property AsFloat: Double read GetAsFloat;
    property AsInteger: prInteger read GetAsInteger;
    property AsBoolean: Boolean read GetAsBoolean;
    property ExprType: TExprType read GetExprType;
    function CanReadAs(aExprType: TExprType): Boolean;
      {means 'can be interpreted as'. Sort of}
    constructor Create;
    destructor Destroy; override;
  end;

  TStringLiteral =
  class(TExpression)
  private
    FAsString: String;
  protected
    function GetAsString: String; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aAsString: String);
  end;

  TFloatLiteral =
  class(TExpression)
  private
    FAsFloat: Double;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aAsFloat: Double);
  end;

  TIntegerLiteral =
  class(TExpression)
  private
    FAsInteger: prInteger;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aAsInteger: prInteger);
  end;

  TBooleanLiteral =
  class(TExpression)
  private
    FAsBoolean: Boolean;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aAsBoolean: Boolean);
  end;

  TParameterList =
  class(TList)
  private
    function GetAsString(i: prInteger): String;
    function GetAsFloat(i: prInteger): Double;
    function GetAsInteger(i: prInteger): prInteger;
    function GetAsBoolean(i: prInteger): Boolean;
    function GetExprType(i: prInteger): TExprType;
    function GetParam(i: prInteger): TExpression;
  public
    destructor Destroy; override;
    property Param[i: prInteger]: TExpression read GetParam;
    property ExprType[i: prInteger]: TExprType read GetExprType;
    property AsString[i: prInteger]: String read GetAsString;
    property AsFloat[i: prInteger]: Double read GetAsFloat;
    property AsInteger[i: prInteger]: prInteger read GetAsInteger;
    property AsBoolean[i: prInteger]: Boolean read GetAsBoolean;
  end;

  TFunction =
  class(TExpression)
  private
    FParameterList: TParameterList;
    function GetParam(n: prInteger): TExpression;
  public
    constructor Create( aParameterList: TParameterList);
    destructor Destroy; override;
    function ParameterCount: prInteger;
    property Param[n: prInteger]: TExpression read GetParam;
  end;

  EExpression = class(Exception);

  TIdentifierFunction = function( const Identifier: String;
                                  ParameterList: TParameterList): TExpression of Object;

function CreateExpression( const S: String;
                IdentifierFunction: TIdentifierFunction): TExpression;
const
  MaxStringLength = 255; {why?}

  {to get a string representation of TExprType use NExprType[ExprType] }
  NExprType: array[TExprType] of String =
      ('String', 'Float', 'Integer', 'Boolean', 'Unknown');

{for debugging version, checking memory leaks}
{$IFDEF VER80}
const
{$ELSE}
var
{$ENDIF}
  InstanceCount: prInteger = 0;
  HasFunctions: Boolean = False;

{This unit comprises a mixed type expression evaluator which follows pascal
syntax (reasonably accurately) and approximates standard pascal types.

Feedback
--------
If you have any questions/comments I would be pleased to hear from you.

If you discover any bugs in this software I would be VERY pleased to
hear from you.

If you want to offer me work implementing extensions or applications using this
software then I will be moderately pleased to hear from you (if I am busy) or
VERY VERY pleased to hear from you (if I am not)

in any event, my address is:   robots@enterprise.net

I have found this code very useful and surprisingly robust. I sincerely hope you do too.

For detailed explanation as to how to effectively use this unit please refer to prExpr.txt.

Compatibility
-------------
This code developed with Delphi 3.0, but I can't offhand think of any reason why it wouldn't
work with Delphi 2.0. I have a 16 bit (Delphi 1.0) version somewhere: email me if you are
interested in that and I will dig it out. It is not well documented though.
(16 bit version merged as of v1.03- mgl)

Additional Resources
--------------------
This archive includes a help file prExpr.hlp, which you can incorporate into your
help system to provide your users with a definition of expression syntax. If you want
the rtf file from which this is compiled then download

http://homepages.enterprise.net/robots/downloads/exprhelp.zip

That package includes
  prExpr.rtf - The source file (inline graphics)
  prExpr.hpj - The help project file.

I have also made available a WINWORD v7.0 file giving the same reference information.

http://homepages.enterprise.net/robots/downloads/exprdoc.zip

Version History
---------------
latest version should be available from:
  http://homepages.enterprise.net/robots/downloads/expreval.zip


31/1/98 v1.03
(a) Unit Name changed from 'Expressions' to 'prExpr'.
    reasons:
     1. Merging 16/32 versions into one unit means name must be
        8.3 compliant
     2. Name should be 8.3 compliant anyway. Long filenames are
        still a pain in the neck.
     3. 'Expressions' is a term with too many meanings. Better to
        use an arbitary, mostly meaningless name.

(b) Incorporation of 16 & 32 bit versions in one unit.

(c) Modification by Markus Stephany
    (http://home.t-online.de/home/mirbir.st)
    Support for Hex literals added. Marked (mst) in source.

(d) Reverse 'Decimal Separator' mod made in v1.02. This was a silly
    change as using , as decimal Separator while using , as parameter
    Separator is a syntactical nonsense. Instead I have defined
    constants DecSeparator and ParamSeparator as constants. T

(e) Significant structural changes to rationalise by
    eliminating repeated code. Introduced concept of 'Expression
    chain' which means that functions Factor, Term, Simple,
    and Expression now have a common implementation (Chain)
    The source is now a lot shorter, and, I hope, clearer. These
    changes should have eliminated Ken Friesen's bug in a more
    structured way.

(f) I have added another 'tier' to the syntax hierachy. The basic
    syntax element is now the 'SimpleFactor' - (was Factor). A
    factor now consists of a string of SimpleFactors linked by
    ^ the exponention operator. This change allows the ^ operator
    to be supported. prExpr.hlp updated.

(g) Archive structure changes:
      1. Expr.hlp has been renamed prExpr.hlp and is included with the
         issue archive.

      2. Tutorial documentation removed from this file to a separate file
         prExpr.txt. (hint - right click on filename then choose 'Open file
         at cursor')

      3. Form unit name changed to Main.pas

      4. Now includes 16 bit example files (tester16.dpr, Main16.pas, Main16.dfm)


20/1/98 v1.02
companion help material issued.

Structure of comment blocks rationalised. Or derationalised,
depending on your point of view.
Juha Hartikainen suggested:
"Can you change decimal separation? You have used
 hardcoded . You can use DecimalSeparator variable
 from SysUtils.Pas and it acts always right in different
 countries."
This seems reasonable enough to me, except there may be
those in foreign parts who wish to retain pascal syntax for
their floating point numbers. I have therefore introduced a
conditional define PASCALFLOAT which forces the parser to
accept only '.' as a decimal Separator. Otherwise it uses
SysUtils.DecimalSeparator. So if you want a '.', thus
rebelliously defying your national convention $DEFINE PASCALFLOAT.
(NOTE This change reversed at version 1.03)


9/1/98 v1.01
Bugs reported by Ken Friesen

1) (1+2))-1=3
(this is a bug, but known. See comment right at end
of source. Function EoE (EndOfExpression) returns true for all of
 ')' ',' or #0. This is necessary for handing functions and
 parameters but irritating if your expression has an extra ).
 Fixed.

2) 1+( )= Access Violation
Oversight. Parser did not check for null subexpresssion (fixed)

3) 0-2+2=-4
4) 1-0+1=0
5) -2-2+2 = -6
I cannot believe that this has not been picked up before now!
There is an awful lot of recursion about and this was caused
by the fact that the function SIMPLE called itself to in order
to generate a string of TERMS. The result of this is that any
simple expressions containing more than two terms were constructed
as if they were bracketed from the end of the expression.

i.e a+b+c+d was evaluated as a + (b + (c + d))

This was an elegant construct and I fell for it regardless of the fact
that it was completely wrong. This problem also affected the function
Term, but because a*b*c*d = a*(b*(c*d)) I got away with it.

I have made a (rough) fix. Which works but may have introduced other
problems. The structure of simple is now (approximately)

function Simple: TExpression;
Result:= Term
while NextOperator in [+, -, or, xor] do
  Result:= Binary(NextOperator, Result, Term)

As opposed to the previous (incorrect) way of doing things which was:

function Simple: TExpression;
Result:= Term
if NextOperator in [+, -, or, xor] then
  Result:= Binary(NextOperator, Result, Simple)


I have also made this modification to  TERM, in order to be consistent,
and because it was a  fluke that it worked before.

The unit now passes Ken's tests. I cannot be sure I have not introduced
other problems. I should devise a proper test routine, when I have some
time.

30/12/97 v1.00
Released to http://homepages.enterprise.net/robots/downloads/expreval.zip
Some slight restructing. Added more comprehensive documentation. Removed
a few calls to StrPas which are redundant under D2/D3

11/11/97
Bug caused mishandling of function lists. Fixed.

5/11/97
Slight modifications for first issue of Troxler.exe

16/9/97
realised (whilst lying in the bath) that the way this unit
handles parameters is a bit daft. It should be possible to
pass the parameter stack to the identifier function. The only
problem with this approach is how to handle disposal of the stack.

We could require that the identifier function disposes of the stack...
I don't really like this (I can't think why at the moment). Another
approach would be to define a 'placeholder' expression which does nothing
but hold the parameter list and the <clients> expression.

Compromise solution:
  The parser constructs an instance of TParameter list and passes it to
  the 'user' via a call to IdentifierFunction. There are four possible
  mechanisms for disposal of the parameter list.
     a) If the Identifier function returns NIL the parser disposes
        of the parameter list then raises 'Unknown identifier'.
     b) If the Identifier function raises an exception then the parser
        catches this exception (in a 'finally' clause) and disposes
        of the parameter list.
     c) If the Identifier function returns an expression then it must
        dispose of the parameter list if it does not wish to keep it.
     d) If the Identifier function returns an expression which is
        derived from TFunction, then it may pass the parameter list to
        its result. The result frees the parameter list when it is freed.
        (i.e. ParameterList passed to TFunction.Create is freed by
        TFunction.Destroy)

Simple rule - if IdentFunction returns Non-nil then parameters are
responsiblity of the object returned. Otherwise caller will handle. OK?


7/9/97
function handling completely changed.

added support for Integers including support for the following operators
  bitwise not
  bitwise and
  bitwise or
  bitwise xor
  shl
  shr
  div

now support std functions:

arithmetic...
  TRUNC, ROUND, ABS, ARCTAN, COS, EXP, FRAC, INT,
     LN, PI, SIN, SQR, SQRT, POWER

string...
  UPPER, LOWER, COPY, POS, LENGTH

Fixed a couple of minor bugs. Forgotten what they are.


18/6/97
Written for Mark Page's troxler thing - as part of the report definition language,
but might be needed for Robot application framework. Not tested much.
Loosely based on syntax diagrams in BP7 Language Guide pages 66 to 79.
This is where the nomenclature Term, Factor, SimpleExpression, Expression is
derived.

I have written spreadsheet-like applications using this unit and it might be useful to
implement more Spreadsheet type functions as standard.
}

implementation
type
  TOperator = ( opNot,
                opExp,
                opMult, opDivide, opDiv, opMod, opAnd, opShl, opShr,
                opPlus, opMinus, opOr, opXor,
                opEq, opNEQ, opLT, opGT, opLTE, opGTE);

  TOperators = set of TOperator;

  TUnaryOp =
  class(TExpression)
  private
    Operand: TExpression;
    OperandType: TExprType;
    Operator: TOperator;
  protected
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aOperator: TOperator; aOperand: TExpression);
    destructor Destroy; override;
  end;

  TBinaryOp =
  class(TExpression)
  private
    Operand1, Operand2: TExpression;
    Operator: TOperator;
    OperandType: TExprType;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aOperator: TOperator; aOperand1, aOperand2: TExpression);
    destructor Destroy; override;
  end;

  TRelationalOp =
  class(TExpression)
  private
    Operand1, Operand2: TExpression;
    Operator: TOperator;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aOperator: TOperator; aOperand1, aOperand2: TExpression);
    destructor Destroy; override;
  end;

const
  NOperator: array[TOperator] of String =
              ( 'opNot',
                'opExp',
                'opMult', 'opDivide', 'opDiv', 'opMod', 'opAnd', 'opShl', 'opShr',
                'opPlus', 'opMinus', 'opOr', 'opXor',
                'opEq', 'opNEQ', 'opLT', 'opGT', 'opLTE', 'opGTE');

  UnaryOperators = [opNot];
  ExpOperator = [opExp];
  MultiplyingOperators = [opMult, opDivide, opDiv, opMod, opAnd, opShl, opShr];
  AddingOperators = [opPlus, opMinus, opOr, opXor];
  RelationalOperators = [opEQ, opNEQ, opLT, opGT, opLTE, opGTE];

  NBoolean: array[Boolean] of String[5] = ('FALSE', 'TRUE');


function ResultType( Operator: TOperator; OperandType: TExprType): TExprType;
procedure NotAppropriate;
begin
  Result:= ttString;
  raise EExpression.CreateFmt( 'Operator %s incompatible with %s',
                               [NOperator[Operator], NExprType[OperandType]])
end;

begin
  case OperandType of
    ttString:
    case Operator of
      opPlus: Result:= ttString;
      opEq..opGTE: Result:= ttBoolean;
    else
      NotAppropriate;
    end;
    ttFloat:
    case Operator of
      opExp, opMult, opDivide, opPlus, opMinus: Result:= ttFloat;
      opEq..opGTE: Result:= ttBoolean;
    else
      NotAppropriate;
    end;
    ttInteger:
    case Operator of
      opNot, opMult, opDiv, opMod, opAnd, opShl, opShr, opPlus, opMinus,
      opOr, opXor: Result:= ttInteger;
      opExp, opDivide: Result:= ttFloat;
      opEq..opGTE: Result:= ttBoolean;
    else
      NotAppropriate;
    end;
    ttBoolean:
    case Operator of
      opNot, opAnd, opOr, opXor, opEq, opNEQ: Result:= ttBoolean;
    else
      NotAppropriate;
    end;
  end
end;

function CommonType( Op1Type, Op2Type: TExprType): TExprType;
begin
  if Op1Type < Op2Type then
    Result:= Op1Type else
    Result:= Op2Type
end;

procedure Internal( Code: prInteger);
begin
  raise EExpression.CreateFmt('Internal parser error. Code %d', [Code])
end;

constructor TExpression.Create;
begin
  inherited Create;
  Inc(InstanceCount)
end;

destructor TExpression.Destroy;
begin
  Dec(InstanceCount);
  inherited Destroy
end;


function TExpression.GetAsString: String;
begin
  case ExprType of
    ttString: raise EExpression.CreateFmt('Cannot read %s as String',
                                              [NExprType[ExprType]]);
    ttFloat: Result:= FloatToStr(AsFloat);
    ttInteger: Result:= IntToStr(AsInteger);
    ttBoolean: Result:= NBoolean[AsBoolean];
  end
end;

function TExpression.GetAsFloat: Double;
begin
  Result:= 0;
  case ExprType of
    ttString, ttFloat:
      raise EExpression.CreateFmt('Cannot read %s as Float',
                                   [NExprType[ExprType]]);
    ttInteger, ttBoolean: Result:= AsInteger;
  end
end;

function TExpression.GetAsInteger: prInteger;
begin
  Result:= 0;
  case ExprType of
    ttString, ttFloat, ttInteger:
       raise EExpression.CreateFmt('Cannot read %s as prInteger',
                               [NExprType[ExprType]]);
    ttBoolean: Result:= prInteger(AsBoolean);
  end;
end;

function TExpression.GetAsBoolean: Boolean;
begin
  raise EExpression.CreateFmt('Cannot read %s as boolean',
                               [NExprType[ExprType]])
end;

function TExpression.CanReadAs(aExprType: TExprType): Boolean;
begin
  Result:= Ord(ExprType) >= Ord(aExprType)
end;

function TStringLiteral.GetAsString: String;
begin
  Result:= FAsString
end;

function TStringLiteral.GetExprType: TExprType;
begin
  Result:= ttString
end;

constructor TStringLiteral.Create( aAsString: String);
begin
  inherited Create;
  FAsString:= aAsString
end;

function TFloatLiteral.GetAsString: String;
begin
  Result:= FloatToStr(FAsFloat)
end;

function TFloatLiteral.GetAsFloat: Double;
begin
  Result:= FAsFloat
end;

function TFloatLiteral.GetExprType: TExprType;
begin
  Result:= ttFloat
end;

constructor TFloatLiteral.Create( aAsFloat: Double);
begin
  inherited Create;
  FAsFloat:= aAsFloat
end;

function TIntegerLiteral.GetAsString: String;
begin
  Result:= FloatToStr(FAsInteger)
end;

function TIntegerLiteral.GetAsFloat: Double;
begin
  Result:= FAsInteger
end;

function TIntegerLiteral.GetAsInteger: prInteger;
begin
  Result:= FAsInteger
end;

function TIntegerLiteral.GetExprType: TExprType;
begin
  Result:= ttInteger
end;

constructor TIntegerLiteral.Create( aAsInteger: prInteger);
begin
  inherited Create;
  FAsInteger:= aAsInteger
end;

function TBooleanLiteral.GetAsString: String;
begin
  Result:= NBoolean[FAsBoolean]
end;

function TBooleanLiteral.GetAsFloat: Double;
begin
  Result:= GetAsInteger
end;

function TBooleanLiteral.GetAsInteger: prInteger;
begin
  Result:= prInteger(FAsBoolean)
end;

function TBooleanLiteral.GetAsBoolean: Boolean;
begin
  Result:= FAsBoolean
end;

function TBooleanLiteral.GetExprType: TExprType;
begin
  Result:= ttBoolean
end;

constructor TBooleanLiteral.Create( aAsBoolean: Boolean);
begin
  inherited Create;
  FAsBoolean:= aAsBoolean
end;

function TUnaryOp.GetAsFloat: Double;
begin
  case Operator of
    opMinus: Result:= -Operand.AsFloat;
    opPlus: Result:= Operand.AsFloat;
  else
    Result:= inherited GetAsFloat;
  end
end;

function TUnaryOp.GetAsInteger: prInteger;
begin
  Result:= 0;
  case Operator of
    opMinus: Result:= -Operand.AsInteger;
    opPlus: Result:= Operand.AsInteger;
    opNot:
    case OperandType of
      ttInteger: Result:= not Operand.AsInteger;
      ttBoolean: Result:= prInteger(AsBoolean);
    else
      Internal(6);
    end;
  else
    Result:= inherited GetAsInteger;
  end
end;

function TUnaryOp.GetAsBoolean: Boolean;
begin
  case Operator of
    opNot: Result:= not(Operand.AsBoolean)
  else
    Result:= inherited GetAsBoolean;
  end
end;

function TUnaryOp.GetExprType: TExprType;
begin
  Result:= ResultType(Operator, OperandType)
end;

constructor TUnaryOp.Create( aOperator: TOperator; aOperand: TExpression);
begin
  inherited Create;
  Operand:= aOperand;
  Operator:= aOperator;
  OperandType:= Operand.ExprType;
  if not (Operator in [opNot, opPlus, opMinus]) then
    raise EExpression.CreateFmt('%s is not simple unary operator',
                                [NOperator[Operator]])
end;

destructor TUnaryOp.Destroy;
begin
  Operand.Free;
  inherited Destroy
end;

function TBinaryOp.GetAsString: String;
begin
  Result:= '';
  case ExprType of
    ttString:
      case Operator of
        opPlus: Result:= Operand1.AsString + Operand2.AsString;
      else
        Internal(10);
      end;
    ttFloat:
      Result:= FloatToStr(AsFloat);
    ttInteger:
      Result:= IntToStr(AsInteger);
    ttBoolean:
      Result:= NBoolean[AsBoolean];
  end
end;

function TBinaryOp.GetAsFloat: Double;
begin
  Result:= 0;
  case ExprType of
    ttFloat:
      case Operator of
        opExp: Result:= Exp(Operand2.AsFloat * Ln(Operand1.AsFloat));
        opPlus: Result:= Operand1.AsFloat + Operand2.AsFloat;
        opMinus: Result:= Operand1.AsFloat - Operand2.AsFloat;
        opMult: Result:= Operand1.AsFloat * Operand2.AsFloat;
        opDivide: Result:= Operand1.AsFloat / Operand2.AsFloat;
      else
        Internal(11);
      end;
    ttInteger:
        Result:= AsInteger;
    ttBoolean:
       Result:= prInteger(AsBoolean);
  end
end;


function TBinaryOp.GetAsInteger: prInteger;
begin
  Result:= 0;
  case ExprType of
    ttInteger:
    case Operator of
      opPlus: Result:= Operand1.AsInteger + Operand2.AsInteger;
      opMinus: Result:= Operand1.AsInteger - Operand2.AsInteger;
      opMult: Result:= Operand1.AsInteger * Operand2.AsInteger;
      opDiv: Result:= Operand1.AsInteger div Operand2.AsInteger;
      opMod: Result:= Operand1.AsInteger mod Operand2.AsInteger;
      opShl: Result:= Operand1.AsInteger shl Operand2.AsInteger;
      opShr: Result:= Operand1.AsInteger shr Operand2.AsInteger;
      opAnd: Result:= Operand1.AsInteger and Operand2.AsInteger;
      opOr: Result:= Operand1.AsInteger or Operand2.AsInteger;
      opXor: Result:= Operand1.AsInteger xor Operand2.AsInteger;
    else
      Internal(12);
    end;
    ttBoolean:
      Result:= prInteger(GetAsBoolean);
  end
end;

function TBinaryOp.GetAsBoolean: Boolean;
begin
  Result:= false;
  case Operator of
    opAnd: Result:= Operand1.AsBoolean and Operand2.AsBoolean;
    opOr: Result:= Operand1.AsBoolean or Operand2.AsBoolean;
    opXor: Result:= Operand1.AsBoolean xor Operand2.AsBoolean;
  else
    Internal(13);
  end
end;

function TBinaryOp.GetExprType: TExprType;
begin
  GetExprType:= ResultType(Operator, OperandType)
end;

constructor TBinaryOp.Create( aOperator: TOperator; aOperand1, aOperand2: TExpression);
begin
  inherited Create;
  Operator:= aOperator;
  Operand1:= aOperand1;
  Operand2:= aOperand2;
  OperandType:= CommonType(Operand1.ExprType, Operand2.ExprType);
  if not (Operator in [opExp, opMult..opXor]) then
    raise EExpression.CreateFmt('%s is not a simple binary operator',
              [NOperator[Operator]])
end;

destructor TBinaryOp.Destroy;
begin
  Operand1.Free;
  Operand2.Free;
  inherited Destroy
end;

function TRelationalOp.GetAsString: String;
begin
  Result:= NBoolean[AsBoolean]
end;

function TRelationalOp.GetAsFloat: Double;
begin
  Result:= prInteger(AsBoolean)
end;

function TRelationalOp.GetAsInteger: prInteger;
begin
  Result:= prInteger(AsBoolean)
end;

function TRelationalOp.GetAsBoolean: Boolean;
begin
  Result:= false;
  case CommonType(Operand1.ExprType, Operand2.ExprType) of
    ttBoolean:
    case Operator of
      opEQ: Result:= Operand1.AsBoolean = Operand2.AsBoolean;
      opNEQ: Result:= Operand1.AsBoolean <> Operand2.AsBoolean;
    else
      raise EExpression.CreateFmt('cannot apply %s to boolean operands',
                                  [NOperator[Operator]]);
    end;

    ttInteger:
    case Operator of
      opLT: Result:= Operand1.AsInteger < Operand2.AsInteger;
      opLTE: Result:= Operand1.AsInteger <= Operand2.AsInteger;
      opGT: Result:= Operand1.AsInteger > Operand2.AsInteger;
      opGTE: Result:= Operand1.AsInteger >= Operand2.AsInteger;
      opEQ: Result:= Operand1.AsInteger = Operand2.AsInteger;
      opNEQ: Result:= Operand1.AsInteger <> Operand2.AsInteger;
    end;

    ttFloat:
    case Operator of
      opLT: Result:= Operand1.AsFloat < Operand2.AsFloat;
      opLTE: Result:= Operand1.AsFloat <= Operand2.AsFloat;
      opGT: Result:= Operand1.AsFloat > Operand2.AsFloat;
      opGTE: Result:= Operand1.AsFloat >= Operand2.AsFloat;
      opEQ: Result:= Operand1.AsFloat = Operand2.AsFloat;
      opNEQ: Result:= Operand1.AsFloat <> Operand2.AsFloat;
    end;

    ttString:
    case Operator of
      opLT: Result:= Operand1.AsString < Operand2.AsString;
      opLTE: Result:= Operand1.AsString <= Operand2.AsString;
      opGT: Result:= Operand1.AsString > Operand2.AsString;
      opGTE: Result:= Operand1.AsString >= Operand2.AsString;
      opEQ: Result:= Operand1.AsString = Operand2.AsString;
      opNEQ: Result:= Operand1.AsString <> Operand2.AsString;
    end;
  end
end;

function TRelationalOp.GetExprType: TExprType;
begin
  Result:= ttBoolean
end;

constructor TRelationalOp.Create( aOperator: TOperator; aOperand1, aOperand2: TExpression);
begin
  inherited Create;
  Operator:= aOperator;
  Operand1:= aOperand1;
  Operand2:= aOperand2;
  if not (Operator in RelationalOperators) then
    raise EExpression.CreateFmt('%s is not relational operator',
                                 [NOperator[Operator]])
end;

destructor TRelationalOp.Destroy;
begin
  Operand1.Free;
  Operand2.Free;
  inherited Destroy
end;

function TParameterList.GetAsString(i: prInteger): String;
begin
  Result:= Param[i].AsString
end;

function TParameterList.GetAsFloat(i: prInteger): Double;
begin
  Result:= Param[i].AsFloat
end;

function TParameterList.GetAsInteger(i: prInteger): prInteger;
begin
  Result:= Param[i].AsInteger
end;

function TParameterList.GetAsBoolean(i: prInteger): Boolean;
begin
  Result:= Param[i].AsBoolean
end;

function TParameterList.GetExprType(i: prInteger): TExprType;
begin
  Result:= Param[i].ExprType
end;

function TParameterList.GetParam(i: prInteger): TExpression;
begin
  Result:= TExpression(Items[i])
end;

destructor TParameterList.Destroy;
var
  i: prInteger;
begin
  for i:= 0 to (Count - 1) do
    TObject(Items[i]).Free;
  inherited Destroy
end;

function TFunction.GetParam(n: prInteger): TExpression;
begin
  Result:= FParameterList.Param[n]
end;

function TFunction.ParameterCount: prInteger;
begin
  if Assigned(FParameterList) then
    ParameterCount:= FParameterList.Count
  else
    ParameterCount:= 0
end;

constructor TFunction.Create( aParameterList: TParameterList);
begin
  inherited Create;
  FParameterList:= aParameterList
end;

destructor TFunction.Destroy;
begin
  FParameterList.Free;
  inherited Destroy
end;

type
  TTypeCast =
  class(TFunction)
  private
    Operator: TExprType;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aParameterList: TParameterList;
                        aOperator: TExprType);
  end;

  TMF =
    (mfTrunc, mfRound, mfAbs, mfArcTan, mfCos, mfExp, mfFrac, mfInt,
     mfLn, mfPi, mfSin, mfSqr, mfSqrt, mfPower);

  TMathExpression =
  class(TFunction)
  private
    Operator: TMF;
    procedure CheckParameters;
  protected
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aParameterList: TParameterList;
                        aOperator: TMF);
  end;

  TSF =
    (sfUpper, sfLower, sfCopy, sfPos, sfLength);

  TStringExpression =
  class(TFunction)
  private
    Operator: TSF;
    procedure CheckParameters;
  protected
    function GetAsString: String; override;
    function GetAsInteger: prInteger; override;
    function GetExprType: TExprType; override;
  public
    constructor Create( aParameterList: TParameterList;
                        aOperator: TSF);
  end;


  TConditional =
  class(TFunction)
  private
    procedure CheckParameters;
    function Rex: TExpression;
  protected
    function GetAsString: String; override;
    function GetAsFloat: Double; override;
    function GetAsInteger: prInteger; override;
    function GetAsBoolean: Boolean; override;
    function GetExprType: TExprType; override;
  public
  end;

const
  NTypeCast: array[TExprType] of PChar =
    ('STRING', 'FLOAT', 'prInteger', 'BOOLEAN', 'UNKNOWN');
  NMF: array[TMF] of PChar =
    ('TRUNC', 'ROUND', 'ABS', 'ARCTAN', 'COS', 'EXP', 'FRAC', 'INT',
     'LN', 'PI', 'SIN', 'SQR', 'SQRT', 'POWER');
  NSF: array[TSF] of PChar = ('UPPER', 'LOWER', 'COPY', 'POS', 'LENGTH');

function TStringExpression.GetAsString: String;
begin
  CheckParameters;
  case Operator of
    sfUpper: Result:= AnsiUpperCase(Param[0].AsString);
    sfLower: Result:= AnsiLowerCase(Param[0].AsString);
    sfCopy: Result:=  Copy(Param[0].AsString, Param[1].AsInteger, Param[2].AsInteger);
  else
    Result:= inherited GetAsString;
  end
end;

function TStringExpression.GetAsInteger: prInteger;
begin
  CheckParameters;
  case Operator of
    sfPos: Result:= Pos(Param[0].AsString, Param[1].AsString);
    sfLength: Result:= Length(Param[0].AsString);
  else
    Result:= inherited GetAsInteger
  end
end;

function TStringExpression.GetExprType: TExprType;
begin
  case Operator of
    sfUpper, sfLower, sfCopy: Result:= ttString;
  else
    Result:= ttInteger;
  end
end;

procedure TStringExpression.CheckParameters;
var
  OK: Boolean;
begin
  OK:= false;
  case Operator of
    sfUpper, sfLower, sfLength:
      OK:= (ParameterCount = 1) and
           (Param[0].ExprType >= ttString);
    sfCopy:
      OK:= (ParameterCount = 3) and
           (Param[0].ExprType >= ttString) and
           (Param[1].ExprType >= ttInteger) and
           (Param[2].ExprType >= ttInteger);
    sfPos:
      OK:= (ParameterCount = 2) and
           (Param[0].ExprType >= ttString) and
           (Param[1].ExprType >= ttString);
  end;
  if not OK then
    raise EExpression.CreateFmt('Invalid parameter to %s',
                                [NSF[Operator]])
end;

constructor TStringExpression.Create( aParameterList: TParameterList;
                                      aOperator: TSF);
begin
  inherited Create(aParameterList);
  Operator:= aOperator
end;

function TMathExpression.GetAsFloat: Double;
begin
  CheckParameters;
  case Operator of
    mfAbs: Result:= Abs(Param[0].AsFloat);
    mfArcTan: Result:= ArcTan(Param[0].AsFloat);
    mfCos: Result:= Cos(Param[0].AsFloat);
    mfExp: Result:= Exp(Param[0].AsFloat);
    mfFrac: Result:= Frac(Param[0].AsFloat);
    mfInt: Result:= Int(Param[0].AsFloat);
    mfLn: Result:= Ln(Param[0].AsFloat);
    mfPi: Result:= Pi;
    mfSin: Result:= Sin(Param[0].AsFloat);
    mfSqr: Result:= Sqr(Param[0].AsFloat);
    mfSqrt: Result:= Sqrt(Param[0].AsFloat);
    mfPower: Result:=  Exp(Param[1].AsFloat * Ln(Param[0].AsFloat))
  else
    Result:= inherited GetAsFloat;
  end
end;

function TMathExpression.GetAsInteger: prInteger;
begin
  CheckParameters;
  case Operator of
    mfTrunc: Result:= Trunc(Param[0].AsFloat);
    mfRound: Result:= Round(Param[0].AsFloat);
    mfAbs: Result:= Abs(Param[0].AsInteger);
  else
    Result:= inherited GetAsInteger;
  end
end;

procedure TMathExpression.CheckParameters;
var
  OK: Boolean;
begin
  OK:= True;
  case Operator of
    mfTrunc, mfRound, mfArcTan, mfCos, mfExp, mfFrac, mfInt,
    mfLn, mfSin, mfSqr, mfSqrt, mfAbs:
    begin
      OK:= (ParameterCount = 1) and
           (Param[0].ExprType >= ttFloat);
    end;
    mfPower:
    begin
      OK:= (ParameterCount = 2) and
           (Param[0].ExprType >= ttFloat) and
           (Param[1].ExprType >= ttFloat);
    end;
  end;
  if not OK then
    raise EExpression.CreateFmt('Invalid parameter to %s',
                                [NMF[Operator]])
end;

function TMathExpression.GetExprType: TExprType;
begin
  case Operator of
    mfTrunc, mfRound: Result:= ttInteger;
  else
    Result:= ttFloat;
  end
end;

constructor TMathExpression.Create( aParameterList: TParameterList;
                                    aOperator: TMF);
begin
  inherited Create(aParameterList);
  Operator:= aOperator
end;


function TTypeCast.GetAsString: String;
begin
  Result:= Param[0].AsString
end;

function TTypeCast.GetAsFloat: Double;
begin
  Result:= Param[0].AsFloat
end;

function TTypeCast.GetAsInteger: prInteger;
begin
  Result:= Param[0].AsInteger
end;

function TTypeCast.GetAsBoolean: Boolean;
begin
  Result:= Param[0].AsBoolean
end;

function TTypeCast.GetExprType: TExprType;
begin
  Result:= Operator
end;

constructor TTypeCast.Create( aParameterList: TParameterList;
                              aOperator: TExprType);
begin
  inherited Create(aParameterList);
  Operator:= aOperator
end;

function TConditional.Rex: TExpression;
begin
  CheckParameters;
  if Param[0].AsBoolean then
    Result:= Param[1] else
    Result:= Param[2]
end;


procedure TConditional.CheckParameters;
begin
  if not ((ParameterCount = 3) and
          (Param[0].ExprType = ttBoolean)) then
    raise EExpression.Create('Invalid parameters to If')
end;

function TConditional.GetAsString: String;
begin
  Result:= Rex.AsString
end;

function TConditional.GetAsFloat: Double;
begin
  Result:= Rex.AsFloat
end;
function TConditional.GetAsInteger: prInteger;
begin
  Result:= Rex.AsInteger
end;
function TConditional.GetAsBoolean: Boolean;
begin
  Result:= Rex.AsBoolean
end;
function TConditional.GetExprType: TExprType;
begin
  Result:= Rex.ExprType
end;

function StandardFunctions (const Ident: String; PL: TParameterList): TExpression;
var
  i: TExprType;
  j: TMF;
  k: TSF;
  Found: Boolean;
begin
  Found:= false;
  if Ident = 'IF' then
  begin
    Result:= TConditional.Create(PL)
  end else
  begin
    for i:= Low(TExprType) to High(TExprType) do
    begin
{$IFDEF VER80}
      if Ident = StrPas(NTypeCast[i]) then
{$ELSE}
      if Ident = NTypeCast[i] then
{$ENDIF}
      begin
        Found:= true;
        Break
      end;
    end;
    if Found then
    begin
      Result:= TTypeCast.Create(PL, i)
    end else
    begin
      for j:= Low(TMF) to High(TMF) do
      begin
{$IFDEF VER80}
        if Ident = StrPas(NMF[j]) then
{$ELSE}
        if Ident = NMF[j] then
{$ENDIF}
        begin
          Found:= true;
          break
        end
      end;
      if Found then
      begin
        Result:= TMathExpression.Create(PL, j)
      end else
      begin
        for k:= Low(TSF) to High(TSF) do
        begin
{$IFDEF VER80}
          if Ident = StrPas(NSF[k]) then
{$ELSE}
          if Ident = NSF[k] then
{$ENDIF}
          begin
            Found:= true;
            break
          end
        end;
        if Found then
        begin
          Result:= TStringExpression.Create(PL, k)
        end else
        begin
          Result:= nil
        end
      end
    end
  end
end;

{parser...}
const
{note: These two cannot be the same}
  DecSeparator = '.';
  ParamDelimiter = ',';

  OpTokens: array[TOperator] of PChar =
              ( 'NOT',
                '^',
                '*', '/', 'DIV', 'MOD', 'AND', 'SHL', 'SHR',
                '+', '-', 'OR', 'XOR',
                '=', '<>', '<', '>', '<=', '>=');

  Whitespace = [#$1..#$20];
  Digits = ['0'..'9'];
  SignChars = ['+', '-'];
  RelationalChars = ['<', '>', '='];
  OpChars = SignChars + ['^', '/', '*'] + RelationalChars;

  OpenSub = '(';
  CloseSub = ')';
  SQuote = ['''', '"'];
  PrimaryIdentChars = ['a'..'z', 'A'..'Z', #128..#255, '_'];
  IdentChars = PrimaryIdentChars + Digits;

  ExprDelimiters = [#0, CloseSub, ParamDelimiter];

  {mst}
  SHex = '$';
  HexDigs = Digits+['a'..'f','A'..'F'];
  {mst}

procedure SwallowWhitespace( var P: PChar);
begin
  while P^ in Whitespace do inc(P)
end;

function EoE( var P: PChar): Boolean;
begin
  Result:= (P^ in ExprDelimiters)
end;


function GetOperator( var P: PChar; var Operator: TOperator): Boolean;
{this leaves p pointing to next char after operator}
var
  Buf: array[0..3] of Char;
  lp: PChar;
  i: prInteger;

function tt( op: TOperator): Boolean;
begin
  if StrLComp(Buf, OpTokens[Op], i) = 0 then
  begin
    Operator:= op;
    Result:= true
  end else
  begin
    Result:= false
  end
end;

begin
  Result:= false;
  if P^ in OpChars then
  begin
    Result:= true;
    Buf[0]:= P^;
    Inc(P);
    case Buf[0] of
      '*': Operator:= opMult;
      '+': Operator:= opPlus;
      '-': Operator:= opMinus;
      '/': Operator:= opDivide;
      '<': if P^ = '=' then
           begin
             Operator:= opLTE;
             Inc(P)
           end else
           if P^ = '>' then
           begin
             Operator:= opNEQ;
             Inc(P)
           end else
           begin
             Operator:= opLT
           end;
       '=': Operator:= opEq;
       '>': if P^ = '=' then
            begin
              Operator:= opGTE;
              Inc(P)
            end else
            begin
              Operator:= opGT
            end;
      '^': Operator:= opExp;
    end
  end else
  if UpCase(P^) in ['A', 'D', 'M', 'N', 'O', 'S', 'X'] then
  begin  {check for 'identifer' style operators. We can ignore NOT}
    lp:= P;
    i:= 0;
    while (i <= 3) and (lp^ in IdentChars) do
    begin
      Buf[i]:= UpCase(lp^);
      inc(lp);
      inc(i)
    end;
    if i in [2,3] then
    begin
      if tt(opNot) then
        Result:= true
      else
      if tt(opDiv) then
        Result:= true
      else
      if tt(opMod) then
        Result:= true
      else
      if tt(opAnd) then
        Result:= true
      else
      if tt(opShl) then
        Result:= true
      else
      if tt(opShr) then
        Result:= true
      else
      if tt(opOr) then
        Result:= true
      else
      if tt(opXor) then
        Result:= true
    end;
    if Result then
      inc(P, i)
  end
end;

type
  TExprFunc = function( var P: PChar; IDF: TIdentifierFunction): TExpression;

function Chain(var P: PChar; IDF: TIdentifierFunction;
                   NextFunc: TExprFunc; Ops: TOperators): TExpression;
{this function is used to construct a chain of expressions}
var
  NextOpr: TOperator;
  StopF: Boolean;
  lp: PChar;
begin
  StopF:= false;
  Result:= NextFunc(P, IDF);
  try
    repeat
      SwallowWhitespace(P);
      lp:= P;
      if not EoE(P) and GetOperator(lp, NextOpr) and (NextOpr in Ops) then
      begin
        P:= lp;
        if NextOpr in RelationalOperators then
          Result:= TRelationalOp.Create(NextOpr, Result, NextFunc(P, IDF))
        else
          Result:= TBinaryOp.Create(NextOpr, Result, NextFunc(P, IDF))
      end else
      begin
        StopF:= true
      end
    until StopF
  except
    Result.Free;
    raise
  end
end;

function Expression( var P: PChar; IDF: TIdentifierFunction): TExpression;
{$IFDEF VER80}
  far;
{$ENDIF}
  forward;

function SimpleFactor( var P: PChar; IDF: TIdentifierFunction): TExpression;
{$IFDEF VER80}
  far;
{$ENDIF}

function UnsignedNumber: TExpression;
type
  TNScan = (nsMantissa, nsDPFound, nsExpFound, nsFound);
var
  S: String[30];
  State: TNScan;
  Int: Boolean;
  SaveSep: Char;

procedure Bomb;
begin
  raise EExpression.Create('Bad numeric format')
end;

begin
  S:= '';
  Int:= false;
  State:= nsMantissa;
  repeat
    if P^ in Digits then
    begin
      S:= S + P^;
      inc(P)
    end else
    if P^ = DecSeparator then
    begin
      if State = nsMantissa then
      begin
        S:= S + P^;
        inc(P);
        State:= nsDPFound
      end else
      begin
        Bomb
      end;
    end else
    if (P^ = 'e') or (P^ = 'E') then
    begin
      if (State = nsMantissa) or
         (State = nsDPFound) then
      begin
        S:= S + 'E';
        inc(P);
        if P^ = '-' then
        begin
          S:= S + P^;
          inc(P);
        end;
        State:= nsExpFound;
        if not (P^ in Digits) then
          Bomb
      end else
      begin
        Bomb
      end
    end else
    begin
      Int:= (State = nsMantissa);
      State:= nsFound
    end;
    if Length(S) > 28 then
      Bomb
  until State = nsFound;
  if Int then
  begin
    Result:= TIntegerLiteral.Create(StrToInt(S))
  end else
  begin
    {WATCH OUT if you are running another thread
     which might refer to DecimalSeparator &&&}
    SaveSep:= SysUtils.DecimalSeparator;
    SysUtils.DecimalSeparator:= DecSeparator;
    try
      Result:= TFloatLiteral.Create(StrToFloat(S))
    finally
      SysUtils.DecimalSeparator:= SaveSep
    end
  end
end;

function CharacterString(EndStrChar: Char): TExpression;
var
  SR: String;
begin
  SR:= '';
  repeat
    inc(P);
    if P^ = EndStrChar then
    begin
     inc(P);
     if (P^ <> EndStrChar) then
        break;
    end;
    if P^ = #0 then
      raise EExpression.Create('Unterminated string');
    if Length(SR) > MaxStringLength then
      raise EExpression.Create('String too long');
    SR:= SR + P^;
  until false;
  Result:= TStringLiteral.Create(SR)
end;

{mst}
function HexValue : Texpression;
var
  SR: String;
begin
  SR:= '';
  repeat
    inc(P);
    if Length(SR) > MaxStringLength then
      raise EExpression.Create('Hex string too long');
    if not (P^ in HexDigs) then break;
      SR:= SR + P^
  until False;
  try
    Result:= TintegerLiteral.Create(StrToInt(SHex+SR))
  except
    raise EExpression.Create('Invalid char in hex number')
  end;
end;
{mst}

var
  Identifier: String;
  Operator: TOperator;
  PList: TParameterList;
  MoreParameters: Boolean;

begin {simple factor}
  Result:= nil;
  try
    SwallowWhitespace(P);
    if GetOperator(P, Operator) then
    begin
      case Operator of
        opPlus:
          Result:= TUnaryOp.Create(opPlus, SimpleFactor(P, IDF));
        opMinus:
          Result:= TUnaryOp.Create(opMinus, SimpleFactor(P, IDF));
        opNot:
          Result:= TUnaryOp.Create(opNot, SimpleFactor(P, IDF));
      else
        raise EExpression.CreateFmt('%s not allowed here', [NOperator[Operator]]);
      end;
    end else
    if P^ in SQuote then
    begin
      Result:= CharacterString(P^);
    end else
    {mst}
    if P^ = SHex then
    begin
      Result:= HexValue;
    end else
    {mst}
    if P^ in Digits then
    begin
      Result:= UnsignedNumber;
    end else
    if P^ = OpenSub then
    begin
      Inc(P);
      Result:= Expression(P, IDF);
      {K Friesen's bug 2. Expression may be nil if
      factor = (). Note: this may also apply to
      parameters i.e. Func(x ,, y)}
      if Result = nil then
        raise EExpression.Create('invalid sub-expression');
      if P^ = CloseSub then
        inc(P)
      else
        raise EExpression.Create(' ) expected')
    end else
    if P^ in PrimaryIdentChars then
    begin
      Identifier:= '';
      while P^ in IdentChars do
      begin
        Identifier:= Identifier + UpCase(P^);
        inc(P)
      end;
      if Identifier = 'TRUE' then
      begin
        Result:= TBooleanLiteral.Create(true)
      end else
      if Identifier = 'FALSE' then
      begin
        Result:= TBooleanLiteral.Create(false)
      end else
      begin
        PList:= nil;
        try
          SwallowWhitespace(P);
          MoreParameters:= P^ = OpenSub;
          if MoreParameters then
          begin
            PList:= TParameterList.Create;
            while MoreParameters do
            begin
              inc(P);
              PList.Add(Expression(P, IDF));
              MoreParameters:= P^ = ParamDelimiter;
            end;
            {bug fix 11/11/97}
            if P^ = CloseSub then
              Inc(P)
            else
              raise EExpression.Create('Incorrectly formed parameters')
          end;         
          Result:= StandardFunctions(Identifier, PList);
          if Result<>nil then HasFunctions:= True;
          if (Result = nil) and Assigned(IDF) then
            Result:= IDF(Identifier, PList);
          if Result = nil then
            raise EExpression.CreateFmt('Unknown Identifier %s', [Identifier]);
        finally
          if Result = nil then
            PList.Free          
        end
      end
    end else
    if EoE(P) then
    begin
      raise EExpression.Create('Unexpected end of factor')
    end else
    begin
      raise EExpression.Create('Syntax error') {leak here ?}
    end
  except
    Result.Free;
    raise
  end
end;  {Simplefactor}

function Factor( var P: PChar; IDF: TIdentifierFunction): TExpression;
{$IFDEF VER80}
  far;
{$ENDIF}
begin
  Result:= Chain(P, IDF, SimpleFactor, [opExp])
end;

function Term( var P: PChar; IDF: TIdentifierFunction): TExpression;
{$IFDEF VER80}
  far;
{$ENDIF}
begin
  Result:= Chain(P, IDF, Factor, [opMult, opDivide, opDiv, opMod, opAnd, opShl, opShr])
end;

function Simple( var P: PChar; IDF: TIdentifierFunction): TExpression;
{$IFDEF VER80}
  far;
{$ENDIF}
begin
  Result:= Chain(P, IDF, Term, [opPlus, opMinus, opOr, opXor])
end;

function Expression( var P: PChar; IDF: TIdentifierFunction): TExpression;
begin
  Result:= Chain(P, IDF, Simple, RelationalOperators)
end;

function CreateExpression( const S: String;
                IdentifierFunction: TIdentifierFunction): TExpression;

var
{$IFDEF VER80}
  Buf: array[0..255] of Char;
{$ENDIF}
  P:PChar;

begin
{$IFDEF VER80}
  P:= StrPCopy(Buf, S);
{$ELSE}
  P:= PChar(S);
{$ENDIF}
  HasFunctions:= False;
  Result:= Expression(P, IdentifierFunction);
  {Friesen bug 1. I knew about this one...}
  if P^ <> #0 then
  begin
    Result.Free;
    raise EExpression.CreateFmt('%s not appropriate', [P^])
  end
end;

end.

{
Scrap Comments (MGL)
--------------------

String is superset of Float is superset of prInteger is Superset of Boolean
this is not quite like pascal...

String - Float - Integer - Boolean
<-upcast                 downcast->

Boolean can always be read as prInteger (True = 1 false = 0)
But prInteger can never be read as Boolean.

Float can always be read as String but string can NEVER be read as Float -
even if string forms valid Float.

Often explicit casts are not required

Enforcement of type compatibility is a great deal less strict than Pascal.

If an operator requires a particular type of operand then both operands
are upcast to the nearest compatible type.

I have arbitrarily asserted that both parties to a relational operator must
be of identical (not compatible) type. This may be a bad decision and
perhaps implicit Upcasts (like that above) should be allowed. Not difficult
to do... Can always use specific upcast. I think a downcast always fails.

Client defined identifiers sort of supported.
}



