unit HalcnQry;

{-------------------------------------------------------------------------------}
{   (C) 2000 Alfonso moreno                                                     }
{   THalcyonxQuery class implementation                                         }
{   You need Halcyon 6 in order to use this unit                                }
{-------------------------------------------------------------------------------}

{$I XQ_FLAG.INC}
interface
//{$R HalcnQry.dcr}

uses
  SysUtils,
  Windows,
  Messages,
  classes,
  Graphics,
  Controls,
  forms,
  Dialogs,
  StdCtrls,
  prExprQ,
  xqmiscel,
  xquery,
  halcn6DB,
  gs6_shel,
  Db,
  IniFiles,
  xqbase;

type
{-------------------------------------------------------------------------------}
{                  forward declarations                                         }
{-------------------------------------------------------------------------------}
  TDataList          = class;
  THalcyonxQuery     = class;
  TCustomxQueryclass = class(TCustomxQuery);

{-------------------------------------------------------------------------------}
{                  Defines TDataItem                                            }
{-------------------------------------------------------------------------------}

  TDataItem = class
  private
     FDataList    : TDataList;   { belongs to                                          }
     FDataSet     : TDataSet;    { the THalcyonDataset                                 }
     FFileName    : String;      { the original filename c:\mydb\file1.dbf             }
     FAlias       : String;      { the alias assigned (to be passed to THalcyonxQuery) }
     FIndexFiles  : TStringList; { The list of index files to use in FDataSet          }
  public
     constructor Create(DataList : TDataList);
     destructor Destroy; override;
     procedure Open;

     property FileName: String read FFileName write FFileName;
     property Alias: String read FAlias write FAlias;
     property DataSet: TDataSet read FDataSet write FDataSet;
     property IndexFiles: TStringList read FIndexFiles write FIndexFiles;
  end;

{-------------------------------------------------------------------------------}
{                  Defines TDataList                                            }
{-------------------------------------------------------------------------------}

  TDataList = class
  private
    FItems          : TList;
    FUseDeleted     : Boolean;
    FConfigFileName : String;
    FInMemResultSet : Boolean;
    FMapFileSize    : Longint;
    FDateFormat     : String;
    function GetCount: Integer;
    function GetItem(Index: Integer): TDataItem;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(const pFileName, pAlias: String; pIndexFiles: TStringList): TDataItem;
    procedure Clear;
    procedure Delete(Index: Integer);
    function IndexOf(const S: String): Integer;
    procedure LoadFromFile(const ConfigFileName: String);
    procedure SaveToFile(const ConfigFileName: String);
    procedure OpenDataSets;
    procedure CloseDataSets;

    property Count: Integer read GetCount;
    property Items[Index: Integer]: TDataItem read GetItem; default;
    property UseDeleted: Boolean read FUseDeleted write FUseDeleted;
    property ConfigFileName: String read FConfigFileName write FConfigFileName;
    property InMemResultSet : Boolean read FInMemResultSet write FInMemResultSet;
    property MapFileSize: Longint read FMapFileSize write FMapFileSize;
    property DateFormat: String read FDateFormat write FDateFormat;
  end;

{-------------------------------------------------------------------------------}
{                  Defines THalcyonxQuery                                       }
{-------------------------------------------------------------------------------}

  THalcyonxQuery = class(TCustomxQuery)
  private
     FType: gsDBFTypes;
     FAutoOver: boolean;
     FUseDeleted: Boolean;
     FSaveUseDeleted : TBits;
     { this is only a reference to a global object and must not be created }
     FDataList : TDataList;
  protected
     procedure IndexNeededfor(Sender: TObject; DataSet: TDataSet;
        const FieldNames: String; ActivateIndex: Boolean; var Accept: Boolean); override;
     procedure SetRange(Sender: TObject; RelOperator: TRelationalOperator;
        DataSet: TDataSet; const FieldNames, StartValues, endValues: String); override;
     procedure CancelRange(Sender: TObject; DataSet: TDataSet);  override;
     procedure FixDummiesForFilter(var Filter: String); override;

     procedure CreateTable(Sender: TObject; CreateTable: TCreateTableItem);
     procedure CreateIndex(Sender: TObject; Unique, Descending: Boolean;
        const TableName, IndexName: String; ColumnExprList: TStringList);
     procedure DropTable(Sender: TObject; const TableName: String);
     procedure DropIndex(Sender: TObject; const TableName, IndexName: String);
     procedure BeforeQuery(Sender : TObject);
     procedure AfterQuery(Sender : TObject);
     procedure SetDataList(Value: TDataList);
     procedure SetFilter(Sender : TObject; DataSet: TDataSet; const Filter: String;
        var Handled : Boolean);
     procedure CancelFilter(Sender : TObject; DataSet : TDataSet);

  public
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;

     procedure SaveToDBF(const FileName: String);
     procedure Loaded; override;

     property DataList : TDataList read FDataList write SetDataList;
  published
     { properties }
     property DBFType: gsDBFTypes read FType write FType;
     property AutoOverwrite: Boolean read FAutoOver write FAutoOver;

     property UseDeleted: Boolean read FUseDeleted write FUseDeleted;

     { inherited properties }
     property DataSets;
  end;

  procedure Register;

implementation

uses
  xqyacc;

procedure Register;
begin
  RegisterComponents('Halcyon6', [THalcyonxQuery]);
end;

resourcestring
  hqErrOverwriteTable    = 'Table exists.  Do you want to overwrite?';
  xqTablenameNotFound    = 'Table name does not exists.';

const
  IDXExtns : array[0..3] of string[4] = ('.NTX', '.NDX', '.MDX', '.CDX');


{-------------------------------------------------------------------------------}
{                  Implementes TDataItem                                        }
{-------------------------------------------------------------------------------}

constructor TDataItem.Create(DataList: TDataList);
begin
   inherited Create;
   FDataList   := DataList;
   FIndexFiles := TStringList.Create;
   { the dataset belong to the DataList }
   FDataSet    := THalcyonDataSet.Create( nil );
end;

destructor TDataItem.Destroy;
begin
   FDataSet.Free;
   FIndexFiles.Free;
   inherited Destroy;
end;

procedure TDataItem.Open;
begin
   FDataSet.Close;
   with (FDataSet as THalcyonDataSet) do
   begin
      DatabaseName   := ExtractFilePath( FFileName );
      Tablename      := ExtractFileName( FFileName );
      IndexFiles.Assign(Self.FIndexFiles);
      UseDeleted     := FDataList.UseDeleted;
      Open;
   end;
end;

{-------------------------------------------------------------------------------}
{                  Implement TDataList                                          }
{-------------------------------------------------------------------------------}
constructor TDataList.Create;
begin
   inherited Create;
   FItems := TList.Create;
end;

destructor TDataList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TDataList.IndexOf(const S: String): Integer;
var
   I: Integer;
begin
   result := -1;
   for I := 0 to FItems.Count - 1 do
   begin
      if AnsiCompareText(Items[I].FileName, S) = 0 then
      begin
         Result := I;
         Exit;
      end;
   end;
end;

function TDataList.GetCount;
begin
   Result := FItems.Count;
end;

function TDataList.GetItem(Index: Integer): TDataItem;
begin
   Result := FItems[Index];
end;

function TDataList.Add(const pFileName, pAlias: String; pIndexFiles: TStringList): TDataItem;
var
   I : Integer;
begin
   Result := TDataItem.Create(Self);
   try
      with TDataItem(Result) do
      begin
         DataSet.Close;
         with THalcyonDataSet(DataSet) do
         begin
            DatabaseName := ExtractFilePath(pFileName);
            TableName := ExtractFileName(pFileName);
            IndexFiles.Clear;
            for I := 0 to pIndexFiles.Count - 1 do
               IndexFiles.Add( ExtractFileName( pIndexFiles[I] ) );
         end;
         IndexFiles.Assign(pIndexFiles);
         if Length(pAlias) > 0 then
            Alias := pAlias
         else
            Alias := ChangeFileExt(ExtractFileName(pFileName), '');
         FileName := pFileName;
      end;
   except
      Result.Free;
      raise;
   end;
   FItems.Add(Result);
end;

procedure TDataList.Clear;
var
   I: Integer;
begin
   for I := 0 to FItems.Count - 1 do
      TDataItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TDataList.Delete(Index: Integer);
begin
   TDataItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

procedure TDataList.LoadFromFile(const ConfigFileName: String);
var
   IniFile     : TIniFile;
   NumFiles    : Integer;
   NumIndexes  : Integer;
   I           : Integer;
   J           : Integer;
   IndexFiles  : TStringList;
   FileName    : String;
   Alias       : String;
begin
   Clear;
   IniFile    := TIniFile.Create(ConfigFileName);
   IndexFiles := TStringList.Create;
   try
      { this is the configuration for the file :
      [General]
      NumFiles=3
      File1=C:\MyDatabase\File1.Dbf
      File2=C:\MyDatabase\File2.Dbf
      File3=C:\MyDatabase\File3.Dbf
      Alias1=Customer
      Alias2=Orders
      Alias3=Items
      ...
      UseDeleted=1 or 0
      FInMemResultSet : Boolean;
      FMapFileSize    : Longint;
      FDateFormat     : String;
      FUseDeleted     : Boolean;


      [File1]
      NumIndexes=1
      Index1=File1.Cdx
      }
      NumFiles         := IniFile.ReadInteger('General', 'NumFiles', 0);
      FUseDeleted      := IniFile.ReadBool('General', 'UseDeleted', False);
      FInMemResultSet  := IniFile.ReadBool('General', 'InMemResultSet', True);
      FMapFileSize     := IniFile.ReadInteger('General', 'MapFileSize', 2000000);
      FDateFormat      := IniFile.ReadString('General', 'DateFormat', 'm/d/yyyy');

      for I := 1 to NumFiles DO
      begin
         IndexFiles.Clear;
         NumIndexes := IniFile.ReadInteger('File' + IntToStr(I), 'NumIndexes', 0);
         for J := 1 to NumIndexes do
         begin
            FileName := IniFile.ReadString('File' + IntToStr(I), 'Index' + IntToStr(J), '');
            if Length(FileName) > 0 then
               IndexFiles.Add( FileName );
         end;
         FileName := IniFile.ReadString('General', 'File' + IntToStr(I), '');
         if not FileExists(FileName) then Continue;
         Alias := IniFile.ReadString('General', 'Alias' + IntToStr(I), '');
         Add(FileName, Alias, IndexFiles);
      end;
   finally
      IniFile.Free;
      IndexFiles.Free;
   end;
   FConfigFileName := ConfigFileName;
end;

procedure TDataList.SaveToFile(const ConfigFileName: String);
var
   IniFile: TIniFile;
   NumFiles, NumIndexes, I, J: Integer;
   IndexFiles: TStringList;
   FileName, Alias: String;
   Item : TDataItem;
begin
   IniFile := TIniFile.Create(ConfigFileName);
   IndexFiles:= TStringList.Create;
   try
      NumFiles := FItems.Count;
      IniFile.WriteInteger('General', 'NumFiles', NumFiles);
      IniFile.WriteBool('General', 'UseDeleted', FUseDeleted);
      IniFile.WriteBool('General', 'InMemResultSet', FInMemResultSet);
      IniFile.WriteInteger('General', 'MapFileSize', FMapFileSize);
      IniFile.WriteString('General', 'DateFormat', FDateFormat);
      for I := 0 to NumFiles - 1 DO
      begin
         Item := Items[I];
         IndexFiles.Assign(THalcyonDataSet(Item.DataSet).IndexFiles);
         NumIndexes := IndexFiles.Count;
         IniFile.WriteInteger('File' + IntToStr(I + 1), 'NumIndexes', NumIndexes);
         for J := 0 to IndexFiles.Count - 1 do
         begin
            FileName := IndexFiles[J];
            IniFile.writeString('File' + IntToStr(I+1), 'Index' + IntToStr(J+1), FileName);
         end;
         FileName := Item.FileName;
         IniFile.writeString('General', 'File' + IntToStr(I+1), FileName);
         Alias := Item.Alias;
         IniFile.writeString('General', 'Alias' + IntToStr(I+1), Alias);
      end;
   finally
      IniFile.Free;
      IndexFiles.Free;
   end;
end;

procedure TDataList.OpenDataSets;
var
   I : Integer;
begin
   Screen.Cursor := crHourglass;
   try
      for I := 0 to FItems.Count - 1 do
         TDataItem(FItems[I]).Open;
   finally
      Screen.Cursor := crDefault;
   end;
end;

procedure TDataList.CloseDataSets;
var
   I : Integer;
begin
   for I := 0 to FItems.Count - 1 do
      TDataItem(FItems[I]).DataSet.Close;
end;

{-------------------------------------------------------------------------------}
{                  Implementes THalcyonxQuery                                   }
{-------------------------------------------------------------------------------}

constructor THalcyonxQuery.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   DataSets.DataSetClass   := THalcyonDataSet;
   AllSequenced            := False;
   FSaveUseDeleted         := TBits.Create;
end;

destructor THalcyonxQuery.Destroy;
begin
   FSaveUseDeleted.Free;
   inherited Destroy;
end;

procedure THalcyonxQuery.Loaded;
begin
   inherited Loaded;
   OnIndexNeededfor:= IndexNeededfor;
   OnSetRange      := SetRange;
   OnCancelRange   := CancelRange;
   OnCreateTable   := CreateTable;
   OnCreateIndex   := CreateIndex;
   OnDropTable     := DropTable;
   OnDropIndex     := DropIndex;
   OnBeforeQuery   := BeforeQuery;
   OnAfterQuery    := AfterQuery;
   OnSetFilter     := SetFilter;
   OnCancelFilter  := CancelFilter;
end;

procedure THalcyonxQuery.IndexNeededfor(Sender: TObject; DataSet: TDataSet;
   const FieldNames: String; ActivateIndex: Boolean; var Accept: Boolean);
var
  i      : integer;
  fNames : String;
begin
  Accept := False;
  if not Optimize then
    exit;

  fNames := AnsiUpperCase(FieldNames);
  { warning: only simple index expressions accepted:
    FIRSTNAME+LASTNAME}
  ReplaceString(fNames, ';', '+');
  with THalcyonDataSet(DataSet) do
  begin
     for i := 1 to IndexCount do
        if AnsiCompareText(fNames, AnsiUpperCase(IndexExpression(i))) = 0 then
        begin
           Accept := True;
           if ActivateIndex then
              SetTagTo( IndexTagName( i ) );
        end;
  end;

end;

procedure THalcyonxQuery.SetRange(Sender: TObject; RelOperator: TRelationalOperator;
   DataSet: TDataSet; const FieldNames, StartValues, endValues: String);

  // DecStr - Decrements a string value (ex. 'Hello' -> 'Helln')-------------//
  function DecStr( sStr: String ): String;
	var
  	iLen: Integer;
  begin
		iLen := Length(sStr);
    if iLen > 0 then
    begin
      if sStr[iLen] > #0 then
        sStr := Copy( sStr, 1, iLen-1) + Char(Ord(sStr[iLen])-1) // Minus one
      else
        sStr := Copy( sStr, 1, iLen-1);  // Chop off last Char
		end;
  	Result := sStr;
  end; //--------------------------------------------------------------------//

  // IncStr - Increments a string value (ex. 'Hello' -> 'Hellp')-------------//
  function IncStr( sStr: String ): String;
	var
  	iLen: Integer;
  begin
	 iLen := Length(sStr);
    if iLen > 0 then
    begin
      if sStr[iLen] < #255 then
        sStr := Copy( sStr, 1, iLen-1) + Char(Ord(sStr[iLen])+1) // Add one
      else
        sStr := sStr + #1;  // Add one character
	 end;
  	Result := sStr;
  end; //--------------------------------------------------------------------//

begin
  with (DataSet as THalcyonDataSet) do
  begin
     case RelOperator of
        ropBETWEEN  : (DataSet as THalcyondataSet).SetRange(StartValues, endValues);
        ropGT       : (DataSet as THalcyondataSet).SetRange(IncStr(StartValues), #255);
        ropGE       : (DataSet as THalcyondataSet).SetRange(StartValues, #255);
        ropLT       : (DataSet as THalcyondataSet).SetRange(#0, DecStr(endValues));
        ropLE       : (DataSet as THalcyondataSet).SetRange(#0, endValues);
        ropNEQ      : ; // how?
     end;
  end;
end;

procedure THalcyonxQuery.CancelRange(Sender: TObject; DataSet: TDataSet);
begin
  (DataSet as THalcyonDataSet).SetRange('','');
  (DataSet as THalcyonDataSet).Filtered := False;
end;

procedure THalcyonxQuery.CreateTable(Sender: TObject; CreateTable: TCreateTableItem);
var
   FieldList      : TStringList;
   s,
   FileName,
   FieldName,
   IndexFileName  : String;
   FieldType      : Char;
   I,
   FieldSize,
   FieldDec       : Integer;
   Halc           : THalcyonDataSet;
   IndexFiles     : TStringList;
begin
   { if not datalist is assigned then I cannot save to a global configuration file }
   if not Assigned(FDataList) then Exit;

   FileName := CreateTable.TableName;
   if FileExists(FileName) and not FAutoOver then
   begin
      if Application.MessageBox(PChar(hqErrOverwriteTable), 'Warning', MB_OKCANCEL) = IDCANCEL then
         Exit;
   end;
   FieldList := TStringList.Create;
   IndexFiles := TStringList.Create;
   try
     for I := 0 to CreateTable.FieldCount - 1 do
     begin
        FieldName:= CreateTable.Fields[I].FieldName;
        case CreateTable.Fields[I].FieldType of
           // list of possible types accepted in TxQuery parser
           RW_CHAR     :
              begin
                 FieldType:= 'C';
                 FieldSize:= CreateTable.Fields[I].Size;
                 FieldDec := 0;
              end;
           RW_INTEGER, RW_AUTOINC  :
              begin
                 FieldType:= 'N';
                 FieldSize:= 11;
                 FieldDec := 0;
              end;
           RW_SMALLINT :
              begin
                 FieldType:= 'N';
                 FieldSize:= 6;
                 FieldDec := 0;
              end;
           RW_BOOLEAN  :
              begin
                 FieldType:= 'L';
                 FieldSize:= 1;
                 FieldDec := 0;
              end;
           RW_DATE, RW_TIME, RW_DATETIME :
              begin
                 FieldType:= 'D';
                 FieldSize:= 10;
                 FieldDec := 0;
              end;
           RW_MONEY, RW_FLOAT    :
              begin
                 FieldType:= 'N';
                 if CreateTable.Fields[I].Scale= 0 then
                 begin
                   FieldSize:= 20;
                   FieldDec := 4;
                 end else
                 begin
                   FieldSize:= CreateTable.Fields[I].Scale;
                   FieldDec := CreateTable.Fields[I].Precision;
                 end;
              end;
           RW_BLOB     :
              begin
              // use BlobType property here
              case CreateTable.Fields[I].BlobType of
                 1, 3: // Memo, formatted Memo
                   FieldType:= 'M';
                 2,4 : // Binary, OLE
                   FieldType:= 'B';
                 5:  // Graphic/Binary
                   FieldType:= 'G';
              end;
              FieldSize:= 8;
              FieldDec:= 0;
              end;
        end;
        FieldList.Add(format('%s;%s;%d;%d',[FieldName,FieldType,FieldSize,FieldDec]));
     end;
     gs6_shel.CreateDBF( FileName, '', FType, FieldList);
     Halc:= THalcyonDataSet.Create(nil);
     Halc.DatabaseName:= AddSlash(ExtractFilePath(FileName));
     Halc.TableName:= ExtractFileName(FileName);
     try
        Halc.Open;
     except
        Halc.Free;
        raise;
     end;
     { add the new created table to the list of datasets
     Self.AddDataSet(Halc, ChangeFileExt(Halc.TableName,''));}
     if CreateTable.PrimaryKey.Count > 0 then
     begin
       S := CreateTable.PrimaryKey[0];
       for I := 1 to CreateTable.PrimaryKey.Count - 1 do
            S := S + '+' + CreateTable.PrimaryKey[I];
       IndexFileName:= ChangeFileExt(FileName, IDXExtns[Ord(FType)]);
       Halc.IndexOn(IndexFileName, 'PRIMARY', S, '.NOT.DELETED()',  // optional
          Halcn6DB.Unique, Halcn6DB.Ascending);
       IndexFiles.Add(IndexFileName);
     end;
     { add to the list }
     FDataList.Add(FileName, ChangeFileExt(Halc.TableName,''), IndexFiles);
     FDataList.SaveToFile( FDataList.ConfigFileName );
   finally
     FieldList.Free;
     IndexFiles.Free;
   end;
end;

procedure THalcyonxQuery.CreateIndex(Sender: TObject; Unique, Descending: Boolean;
   const TableName, IndexName: String; ColumnExprList: TStringList);
var
  Temps: String;
  j, Index: integer;
  IndexUnique: TgsIndexUnique;
  SortStatus: TgsSortStatus;
  Halc: THalcyonDataSet;
begin
  if not Assigned(FDataList) then Exit;

  Index := FDataList.IndexOf(TableName);
  if Index < 0 then Exit;
  Temps := ColumnExprList[0];
  for j := 1 to ColumnExprList.Count - 1 do
     Temps := Temps + '+' + ColumnExprList[j];
  if Unique then IndexUnique := Halcn6DB.Unique else IndexUnique := Halcn6DB.Duplicates;
  if Descending then SortStatus:= Halcn6DB.Descending else SortStatus:= Halcn6DB.Ascending;
  Halc := FDataList[Index].DataSet as THalcyonDataSet;
  { supposed to add to a primary index .cdx, .mdx }
  Halc.IndexOn(ChangeFileExt(Halc.TableName, IDXExtns[Ord(FType)]),
     IndexName, Temps, '.NOT.DELETED()',  // optional
     IndexUnique, SortStatus);
  FDataList.SaveToFile(FDataList.ConfigFileName);
end;

procedure THalcyonxQuery.DropTable(Sender: TObject; const TableName: String);
var
  Index: integer;
begin
  if not Assigned(FDataList) then Exit;

  Index := FDataList.IndexOf(TableName);
  if Index < 0 then Exit;
  SysUtils.DeleteFile(TableName);
  FDataList.Delete(Index);
end;

procedure THalcyonxQuery.DropIndex(Sender: TObject; const TableName, IndexName: String);
var
   Halc: THalcyonDataSet;
   Index: integer;
begin
   if not Assigned(FDataList) then Exit;

   Index := FDataList.IndexOf(TableName);
   if Index < 0 then Exit;
   Halc := FDataList[Index].DataSet as THalcyonDataSet;
   if Halc.Active  then
      Halc.IndexTagRemove(ChangeFileExt(Halc.TableName, IDXExtns[Ord(FType)]), IndexName);
end;

procedure THalcyonxQuery.SaveToDBF(const FileName: String);
var
  I: Integer;
  Field,SrcField,DestField: TField;
  FieldList, NewFieldNamesList: TStringList;
  FieldName: String;
  FieldType: Char;
  FieldSize, FieldDec: Integer;
  Halc: THalcyonDataSet;
  bm: TBookmark;

  function CheckDuplicate(const fname: String): String;
  var
     NumTry: Integer;
     Found: Boolean;
  begin
     Result:= fname;
     NumTry:= 0;
     repeat
        Found:= NewFieldNamesList.IndexOf( Result ) >= 0;
        if Found then
        begin
           Inc(NumTry);
           Result:= Copy(fname,1,8) + '_' + IntToStr(NumTry);
        end;
     until not Found;
  end;

begin
  if FileExists(FileName) and not FAutoOver then
  begin
     if Application.MessageBox(PChar(hqErrOverwriteTable), 'Warning', MB_OKCANCEL) = IDCANCEL then
        exit;
  end;
  FieldList          := TStringList.Create;
  NewFieldNamesList  := TStringList.Create;
  Halc               := THalcyonDataSet.Create(nil);
  Halc.DatabaseName  := AddSlash(ExtractFilePath(FileName));
  Halc.TableName     := ExtractFileName(FileName);
  DisableControls;
  bm:= GetBookmark;
  try
     for I:= 0 to Self.FieldCount - 1 do
     begin
        Field:= Self.Fields[I];
        // warning: SQL statement must have a valid DBF file name in as AS clause
        // this is not valid: SELECT CustNo As VeryLongFieldName FROM Customer
        // due to that DBF only accepts field names of 10 chars max length
        FieldName:= Field.FieldName;
        ReplaceString(FieldName, ' ', '');
        FieldName:= CheckDuplicate(Copy(FieldName,1,10));
        NewFieldNamesList.Add(FieldName);
        case Field.DataType of
           ftMemo, ftFmtMemo:
              begin
              FieldType:= 'M';
              FieldSize:= 8;
              FieldDec:= 0;
              end;
           ftGraphic:
              begin
              FieldType:= 'G';
              FieldSize:= 8;
              FieldDec:= 0;
              end;
           ftBytes, ftvarBytes, ftBlob,
           ftParadoxOle, ftDBaseOle, ftTypedBinary, ftCursor
           {$IFDEF LEVEL4}, ftADT, ftArray, ftReference, ftDataSet{$endIF}:
              begin
              FieldType:= 'B';
              FieldSize:= 8;
              FieldDec:= 0;
              end;
           ftString{$IFDEF LEVEL4},ftFixedChar,ftWideString{$endIF}:
              begin
              FieldType:= 'C';
              FieldSize:= ResultSet.Fields[I].ColWidth;
              FieldDec:= 0;
              end;
           ftFloat,ftCurrency,ftBCD:
              begin
              FieldType:= 'N';
              FieldSize:= 20;     // you can change this
              FieldDec:= 4;
              end;
           ftDate,ftTime,ftDateTime:
              begin
              FieldType:= 'D';
              FieldSize:= 10;
              FieldDec:= 0;
              end;
           ftAutoInc,ftSmallInt,ftInteger,ftWord
           {$ifndef LEVEL3},ftLargeInt{$endif} :
              begin
              FieldType:= 'N';
              FieldSize:= 11;      // configure this also to your needs
              FieldDec:= 0;
              end;
           ftBoolean:
              begin
              FieldType:= 'L';
              FieldSize:= 1;
              FieldDec:= 0;
              end;
        end;
       FieldList.Add(format('%s;%s;%d;%d',[FieldName,FieldType,FieldSize,FieldDec]));
     end;
     gs6_shel.CreateDBF( FileName, '', FoxPro2, FieldList);  // change FoxPro2 to your choice
     Halc.Open;
     // after creating dbf, insert records from source DBF to dest DBF
     Self.First;
     while not Self.EOF do
     begin
        Halc.Insert;
        for I:= 0 to Self.FieldCount - 1 do
        begin
           SrcField:= Self.Fields[I];
           DestField:= Halc.Fields[I];
           DestField.Assign(SrcField);
        end;
        Halc.Post;

        Self.Next;
     end;
  finally
     if Bookmarkvalid(bm) then GotoBookmark(bm);
     FreeBookmark(bm);
     EnableControls;
     FieldList.Free;
     NewFieldNamesList.Free;
     Halc.Free;
  end;
end;

procedure THalcyonxQuery.BeforeQuery(Sender : TObject);
var
   I : Integer;
begin
   for I := 0 to DataSets.Count - 1 do
   begin
      with (DataSets[I].DataSet as THalcyonDataSet) do
      begin
         FSaveUseDeleted[ I ] := UseDeleted;
         UseDeleted := Self.FUseDeleted;
      end;
   end;
end;

procedure THalcyonxQuery.AfterQuery(Sender : TObject);
var
   I : Integer;
begin
   for I := 0 to DataSets.Count - 1 do
   begin
      with (DataSets[I].DataSet as THalcyonDataSet) do
         { restore previous states of the dataset }
         UseDeleted := FSaveUseDeleted[ I ];
   end;
end;

procedure THalcyonxQuery.SetDataList(Value: TDataList);
var
   I : Integer;
begin
   DataSets.Clear;
   if not Assigned(Value) then Exit;
   { feed the datasets from the data list }
   for I := 0 to Value.Count - 1 do
      AddDataSet(Value[I].DataSet, Value[I].Alias);
   UseDeleted  := Value.UseDeleted;

   FDataList   := Value;
end;

procedure THalcyonxQuery.FixDummiesForFilter(var Filter: String);
var
   Ps : Integer;
   I  : Integer;
   Dt : Double;
begin
   { this method called in the WHERE clause is a filter in
     order to fix some flags:
     - working flag now is the date in the format: 'DummyDate(32445.6566)'
     - another is the handling of True and False in the expression parser }
   ReplaceString(Filter, 'DummyBoolean(True)', 'True');
   ReplaceString(Filter, 'DummyBoolean(False)', 'False');
   Ps := AnsiPos('DummyDate(', Filter);
   while Ps > 0 do
   begin
      if Ps > 0 then
      begin
         { by default, the date is left as it is but in descendant classes
           the date can be changed to meet the dataset filter implementation}
         for I := Ps + 1 to Length(Filter) do
            if Filter[I] = ')' then
            begin
               Dt := StrToFloat(Copy(Filter, Ps + 10, I - (Ps + 10)));
               ReplaceString(Filter, Copy(Filter, Ps, (I - Ps) + 1), '{' + DateToStr(Dt) + '}');
            end;
      end;
      Ps := AnsiPos('DummyDate(', Filter);
   end;
end;

procedure THalcyonxQuery.SetFilter(Sender : TObject; DataSet: TDataSet; const Filter: String;
   var Handled : Boolean);
begin
   { this is only called for the WHERE expression }
   try
      (DataSet as THalcyonDataSet).Filtered := False;
      (DataSet as THalcyonDataSet).Filter := Filter;
      (DataSet as THalcyonDataSet).Filtered := True;
      Handled := True;
   except
      Handled := False;
      (DataSet as THalcyonDataSet).Filtered := False;
   end;
end;

procedure THalcyonxQuery.CancelFilter(Sender : TObject; DataSet : TDataSet);
begin
   (DataSet as THalcyondataSet).Filter := '';
   (DataSet as THalcyondataSet).Filtered := False;
end;

end.
