unit FFxQuery;

{
================================================================================
   (c) 2000 Alfonso moreno
   TFFxQuery class implementation
   Most of this code due (and thanks) to : David G. Stern (60% :-))
================================================================================
}
interface

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  Db,
  IniFiles,
  xquery,
  xqmiscel,
  xqbase;

//{$R FFxQUERY.DCR}

type
{-------------------------------------------------------------------------------}
{                  forward declarations                                         }
{-------------------------------------------------------------------------------}
  TDataList          = class;

{-------------------------------------------------------------------------------}
{                  Defines TDataItem                                            }
{-------------------------------------------------------------------------------}

  TDataItem = class
  private
     FDataList     : TDataList;   { belongs to                                          }
     FDataSet      : TDataSet;    { the TFFxTable                                       }
     FDatabaseName : String;
     FTableName    : String;
     FAlias        : String;      { the alias assigned (to be passed to TFFxQuery)      }
     function GetFileName: String;
  public
     constructor Create(DataList : TDataList);
     destructor Destroy; override;
     procedure Open;

     property DatabaseName: String read FDatabaseName write FDatabaseName;
     property TableName: String read FTableName write FTableName;
     property Alias: String read FAlias write FAlias;
     property DataSet: TDataSet read FDataSet write FDataSet;
     property FileName: String read GetFileName;
  end;

{-------------------------------------------------------------------------------}
{                  Defines TDataList                                            }
{-------------------------------------------------------------------------------}

  TDataList = class
  private
    FItems          : TList;
    FConfigFileName : String;
    FInMemResultSet : Boolean;
    FMapFileSize    : Longint;
    FDateFormat     : String;
    FUseDisplayLabel: Boolean;
    function GetCount: Integer;
    function GetItem(Index: Integer): TDataItem;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(const pDatabaseName, pTableName, pAlias: String): TDataItem;
    procedure Clear;
    procedure Delete(Index: Integer);
    procedure LoadFromFile(const ConfigFileName: String);
    procedure SaveToFile(const ConfigFileName: String);
    procedure OpenDataSets;
    procedure CloseDataSets;

    property Count: Integer read GetCount;
    property Items[Index: Integer]: TDataItem read GetItem; default;
    property ConfigFileName: String read FConfigFileName write FConfigFileName;
    property InMemResultSet : Boolean read FInMemResultSet write FInMemResultSet;
    property MapFileSize: Longint read FMapFileSize write FMapFileSize;
    property DateFormat: String read FDateFormat write FDateFormat;
    property UseDisplayLabel: Boolean read FUseDisplayLabel write FUseDisplayLabel;
  end;

  TFFxQuery = class(TCustomxQuery)
  private
    fIndexList: array [1..2] of TStringList;
    fDatabaseName: string;
    { this is only a reference to a global object and must not be created }
    FDataList : TDataList;

    procedure SetDataList(Value: TDataList);
  protected
    { overriden methods}
    procedure IndexNeededFor(Sender: TObject; DataSet: TDataSet;
              const FieldNames: String; ActivateIndex: Boolean; var Accept: Boolean); override;
    procedure SetRange(Sender: TObject; RelOperator: TRelationalOperator;
              DataSet: TDataSet; const FieldNames, StartValues, EndValues: String); override;
    procedure CancelRange(Sender: TObject; DataSet: TDataSet); override;

    { not overrides }
    procedure CreateTable(Sender: TObject; CreateTable: TCreateTableItem);
    procedure CreateIndex(Sender: TObject; Unique, Descending: Boolean;
       const TableName, IndexName: String; ColumnExprList: TStringList);
    procedure DropTable(Sender: TObject; const TableName: String);
    procedure DropIndex(Sender: TObject; const TableName, IndexName: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Loaded; override;

    property DataList : TDataList read FDataList write SetDataList;
  published
    property DatabaseName: string read fDatabaseName write fDatabaseName;
  end;

procedure Register;

implementation

uses
  FFDB;

procedure Register;
begin
  RegisterComponents('FlashFiler', [TFFxQuery]);
end;


{-------------------------------------------------------------------------------}
{                  Implementes TDataItem                                        }
{-------------------------------------------------------------------------------}

constructor TDataItem.Create(DataList: TDataList);
begin
   inherited Create;
   FDataList   := DataList;
   { the dataset belong to the DataList }
   FDataSet    := TffTable.Create( nil );
end;

destructor TDataItem.Destroy;
begin
   FDataSet.Free;
   inherited Destroy;
end;

function TDataItem.GetFileName: String;
begin
   Result := AddSlash(FDatabaseName) + FTableName;
end;

procedure TDataItem.Open;
begin
   FDataSet.Close;
   with (FDataSet as TffTable) do
   begin
      DatabaseName := Self.FDatabaseName;
      Tablename    := Self.FTableName;
      Open;
   end;
end;

{-------------------------------------------------------------------------------}
{                  Implement TDataList                                          }
{-------------------------------------------------------------------------------}
constructor TDataList.Create;
begin
   inherited Create;
   FItems := TList.Create;
end;

destructor TDataList.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TDataList.GetCount;
begin
   Result := FItems.Count;
end;

function TDataList.GetItem(Index: Integer): TDataItem;
begin
   Result := FItems[Index];
end;

function TDataList.Add(const pDatabaseName, pTableName, pAlias: String): TDataItem;
begin
   Result := TDataItem.Create(Self);
   try
      with TDataItem(Result) do
      begin
         DataSet.Close;
         TffTable(DataSet).DatabaseName := pDatabaseName;
         TffTable(DataSet).TableName := pTableName;
         if Length(pAlias) > 0 then
            Alias := pAlias
         else
            Alias := ChangeFileExt(ExtractFileName(pTableName), '');
         DatabaseName := pDatabaseName;
         TableName    := pTableName;
      end;
   except
      Result.Free;
      raise;
   end;
   FItems.Add(Result);
end;

procedure TDataList.Clear;
var
   I: Integer;
begin
   for I := 0 to FItems.Count - 1 do
      TDataItem(FItems[I]).Free;
   FItems.Clear;
end;

procedure TDataList.Delete(Index: Integer);
begin
   TDataItem(FItems[Index]).Free;
   FItems.Delete(Index);
end;

procedure TDataList.LoadFromFile(const ConfigFileName: String);
var
   IniFile     : TIniFile;
   NumFiles    : Integer;
   I           : Integer;
   DatabaseName: String;
   Tablename   : String;
   Alias       : String;
begin
   Clear;
   IniFile    := TIniFile.Create(ConfigFileName);
   try
      { this is the configuration for the file :
      [General]
      NumFiles=3
      DatabaseName1=Examples
      Tablename1=ExCust
      DatabaseName2=Examples
      Tablename2=ExOrders
      Alias1=Customer
      Alias2=Orders
      ...
      FInMemResultSet : Boolean;
      FMapFileSize    : Longint;
      FDateFormat     : String;
      }
      NumFiles         := IniFile.ReadInteger('General', 'NumFiles', 0);
      FInMemResultSet  := IniFile.ReadBool('General', 'InMemResultSet', True);
      FMapFileSize     := IniFile.ReadInteger('General', 'MapFileSize', 2000000);
      FDateFormat      := IniFile.ReadString('General', 'DateFormat', 'm/d/yyyy');

      for I := 1 to NumFiles DO
      begin
         DatabaseName := IniFile.ReadString('General', 'DatabaseName' + IntToStr(I), '');
         TableName := IniFile.ReadString('General', 'TableName' + IntToStr(I), '');
         Alias := IniFile.ReadString('General', 'Alias' + IntToStr(I), '');
         Add(DatabaseName, TableName, Alias);
      end;
   finally
      IniFile.Free;
   end;
   FConfigFileName := ConfigFileName;
end;

procedure TDataList.SaveToFile(const ConfigFileName: String);
var
   IniFile: TIniFile;
   NumFiles, I: Integer;
   DatabaseName, TableName, Alias: String;
   Item : TDataItem;
begin
   IniFile := TIniFile.Create(ConfigFileName);
   try
      NumFiles := FItems.Count;
      IniFile.WriteInteger('General', 'NumFiles', NumFiles);
      IniFile.WriteBool('General', 'InMemResultSet', FInMemResultSet);
      IniFile.WriteInteger('General', 'MapFileSize', FMapFileSize);
      IniFile.WriteString('General', 'DateFormat', FDateFormat);
      for I := 0 to NumFiles - 1 DO
      begin
         Item := Items[I];
         DatabaseName := Item.DatabaseName;
         TableName := Item.TableName;
         IniFile.writeString('General', 'DatabaseName' + IntToStr(I+1), DatabaseName);
         IniFile.writeString('General', 'TableName' + IntToStr(I+1), TableName);
         Alias := Item.Alias;
         IniFile.writeString('General', 'Alias' + IntToStr(I+1), Alias);
      end;
   finally
      IniFile.Free;
   end;
end;

procedure TDataList.OpenDataSets;
var
   I : Integer;
begin
   Screen.Cursor := crHourglass;
   try
      for I := 0 to FItems.Count - 1 do
         TDataItem(FItems[I]).Open;
   finally
      Screen.Cursor := crDefault;
   end;
end;

procedure TDataList.CloseDataSets;
var
   I : Integer;
begin
   for I := 0 to FItems.Count - 1 do
      TDataItem(FItems[I]).DataSet.Close;
end;

{ Startt of TFFxQuery implementation }
function filterL(line, from_ch, to_ch: string): string;
var
  posn,
  locn:  integer;
begin
  posn := 1;
  while posn <= length(line) do begin
    locn := pos(line[posn],from_ch);
    if locn = 0 then
      inc(posn)
    else if locn > length(to_ch) then
      delete(line,posn,1)
    else begin
      line[posn] := to_ch[locn];
      inc(posn);
    end;
  end;
  result := line;
end;

function extractL(line: string; sep: char; locn: integer): string;
var
  cntr,
  spos,
  posn:    integer;
begin
  spos := 1;
  for cntr := 1 to locn-1 do begin
    posn := pos(sep,copy(line,spos,$FFFF));
    if posn = 0 then
      spos := $FFFF
    else
      inc(spos,posn);
  end;
  posn := pos(sep,copy(line,spos,$FFFF));
  if posn = 0 then
    posn := $FFFF;
  result := copy(line,spos,posn-1);
end;

constructor TFFxQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fIndexList[1] := TStringList.Create;
  fIndexList[2] := TStringList.Create;
  DataSets.DataSetClass:= TffTable;     // don't accept other datasets
end;

destructor TFFxQuery.Destroy;
begin
  fIndexList[1].Free;
  fIndexList[2].Free;
  inherited Destroy;
end;

procedure TFFxQuery.Loaded;
begin
  inherited Loaded;
  OnIndexNeededFor := IndexNeededFor;
  OnSetRange := SetRange;
  OnCancelRange := CancelRange;
  OnCreateTable := CreateTable;
  OnCreateIndex := CreateIndex;
  OnDropTable:= DropTable;
  OnDropIndex:= DropIndex;
end;

procedure TFFxQuery.IndexNeededFor(Sender: TObject;
  DataSet: TDataSet; const FieldNames: String; ActivateIndex: Boolean;
  var Accept: Boolean);
var
  fcntr,
  xcntr,
  cntr: integer;
  fNames,
  xNames: string;
begin
  Accept := False;
  if not Optimize then
    exit;

  fNames := AnsiUpperCase(FieldNames);
  fcntr := 1;
  for cntr := 1 to length(fNames) do
    if fNames[cntr] = ';' then
      inc(fcntr);
  with TffTable(DataSet) do begin
    if IndexDefs.Count = 0 Then
      IndexDefs.Update;
    for xcntr := fcntr downto 1 do begin
      for cntr := 0 to IndexDefs.Count-1 do begin
        xNames := AnsiUpperCase(TIndexDef(IndexDefs[cntr]).Fields);
        if fNames+';' = copy(xNames+';',1,length(fNames)+1) then begin
          Accept := True;
          if fIndexList[1].IndexOf(DataSet.Name+'|'+FieldNames) = -1 then begin
            fIndexList[1].Add(DataSet.Name+'|'+FieldNames);
            fIndexList[2].Add( filterL(IntToStr(xcntr)+';'+fNames,';',#13) );
          end;
          IndexFieldNames := fNames;
          break;
        end;
      end;
      if Accept then
        break;
      if fcntr <> 1 then begin
        for cntr := length(fNames) downto 1 do begin
          if fNames[cntr] = ';' then begin
            fNames := copy(fNames,1,cntr-1);
            break;
          end;
        end;
      end;
    end;
  end;
end;

procedure TFFxQuery.SetRange(Sender: TObject; RelOperator: TRelationalOperator;
  DataSet: TDataSet; const FieldNames, StartValues, EndValues: String);
var
  ix: integer;
  fc: integer;
  fNames: TStringList;

  procedure LoadRangeValues(const rvals: string);
  var
    cntr: integer;
    valstr: string;
  begin
    for cntr := 1 to fc do begin
      valstr := extractL(rvals,';',cntr);
      try
        TffTable(DataSet).FieldByName(fNames[cntr]).Value := valstr;
      except
        TffTable(DataSet).FieldByName(fNames[cntr]).Value := StrToFloat(valstr);
      end;
    end;
  end;

begin
  ix := fIndexList[1].IndexOf(DataSet.Name+'|'+FieldNames);
  if ix <> -1 then begin
    fNames := TStringList.Create;
    fNames.Text := fIndexList[2][ix];
    with TffTable(DataSet) do begin
      fc := StrToInt(fNames[0]);
      SetRangeStart;
      KeyFieldCount := fc;
      if RelOperator in [ropBETWEEN, ropGT, ropGE] then
        LoadRangeValues(StartValues);
      SetRangeEnd;
      KeyFieldCount := fc;
      if RelOperator in [ropBETWEEN, ropLT, ropLE] then
        LoadRangeValues(EndValues);
      ApplyRange;
    end;
    fNames.Free;
  end;
end;

procedure TFFxQuery.CancelRange(Sender: TObject; DataSet: TDataSet);
begin
  TffTable(DataSet).CancelRange;
end;

procedure TFFxQuery.CreateTable(Sender: TObject; CreateTable: TCreateTableItem);
begin
end;

procedure TFFxQuery.CreateIndex(Sender: TObject; Unique, Descending: Boolean;
   const TableName, IndexName: String; ColumnExprList: TStringList);
begin
end;

procedure TFFxQuery.DropTable(Sender: TObject; const TableName: String);
begin
end;

procedure TFFxQuery.DropIndex(Sender: TObject; const TableName, IndexName: String);
begin
end;

procedure TFFxQuery.SetDataList(Value: TDataList);
var
   I : Integer;
begin
   DataSets.Clear;
   if not Assigned(Value) then Exit;
   { feed the datasets from the data list }
   for I := 0 to Value.Count - 1 do
      AddDataSet(Value[I].DataSet, Value[I].Alias);

   FDataList   := Value;
end;

end.
