/* exercise <pthread.h> threads and mutexes */

/* Threads 1 and 2 test argument passing and time slicing.  Each spins on
 * the time() function, waiting for the number of seconds passed to it as
 * an argument to elapse, then prints "thread[12]" to stdout.  Both threads
 * loop forever doing this.
 *
 * Threads 3 and 4 test mutex locking and unlocking by implementing a
 * lock-stepped producer/consumer relationship using 3 mutex variables.
 * Thread 3 "produces" (by incrementing a variable), then waits for thread
 * 4 to "consume" (by decrementing the variable).  Conversely, thread 4
 * waits for 3 to "produce", then "consumes".  Each wait is acheived by
 * locking a mutex held by the other thread.  In the initial state, thread
 * 3 has mutexes 1 and 2 locked, and thread 4 has mutex 3 locked.  Each
 * then goes through a sequence involving three steps: a lock, an unlock,
 * and a produce or consume.  After three sequences, each thread is back in
 * its initial state, and a loop is formed to repeat the sequences forever.
 * A fourth mutex and a counting variable are needed to ensure that both
 * threads initialize themselves before starting their producer/consumer
 * relationship.
 */

#include <sys/types.h> /* must be first for VxWorks */
#include <errno.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
char Ccover_off[] = "C-Cover off";
#include <pthread.h>
char Ccover_on[] = "C-Cover on";

#if !defined(PTHREAD_LINK)
#	define PTHREAD_LINK
#endif

/*------------------------- Private Variable ---------------------------
 | Name:    Check
 | Purpose: True when batch mode testing (via check.ksh).
 */
static int Check = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    Run
 | Purpose: False to stop all threads.
 */
static volatile int Run = 1;

/* These are set true when the respective threads have executed at least
 * one loop.  The first element is not used.
 */
static volatile int running[5] = { 0, 0, 0, 0, 0 };

/* mutexes used by threads 3 and 4 */
static pthread_mutex_t start = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t lock1 = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t lock2 = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t lock3 = PTHREAD_MUTEX_INITIALIZER;

static volatile int not_ready = 2;  /* decremented to 0 when both threads 3
									   and 4 have initialized themselves */
static volatile int product = 0;    /* incremented by thread 3 ("producer"),
									   decremented by thread 4 ("consumer") */
static int mutex_error = 0;         /* set true if a mutex lock/unlock
									   failure is detected */

/* Thread return values */
static int r1;
static int r2;

static void print(char * s)
{
	if (!Check) {
		write(1, s, strlen(s));
	}
}

static void * PTHREAD_LINK thread1(void * arg)
{
	time_t t0;
	print("thread1: starting\n");
	while (Run) {
		t0 = time(0) + 2;
		while (time(0) < t0) {
			sched_yield();
		}
		print("thread1\n");
		running[1] = 1;
	}
	print("thread1: exiting\n");
	return &r1;
}

static void * PTHREAD_LINK thread2(void * arg)
{
	time_t t0;
	print("thread2: starting\n");
	while (Run) {
		t0 = time(0) + 3;
		while (time(0) < t0) {
			sched_yield();
		}
		print("thread2\n");
		running[2] = 1;
	}
	print("thread2: exiting\n");
	return &r2;
}

/* producer */
static void * PTHREAD_LINK thread3(void * arg)
{
	int rc;
	arg = arg;

	print("thread3: starting\n");
	/* get initialization lock */
	if ((rc = pthread_mutex_lock(&start)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}
	if ((rc = pthread_mutex_lock(&lock1)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}
	if ((rc = pthread_mutex_lock(&lock2)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}
	not_ready--;

	/* release initialization lock */
	if ((rc = pthread_mutex_unlock(&start)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}

	/* wait for other thread to finish initializing */
	"C-Cover off";
	while (not_ready) {
		sched_yield();
	}
	"C-Cover on";

	/* begin producing */
	while (Run) {
		/* have 1 2 */
		if (product++ != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock1)) != 0) {
			break;
		}
		if ((rc = pthread_mutex_lock(&lock3)) != 0) {
			break;
		}

		/* have 2 3 */
		if (product++ != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock2)) != 0) {
			break;
		}
		if ((rc = pthread_mutex_lock(&lock1)) != 0) {
			break;
		}

		/* have 1 3 */
		if (product++ != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock3)) != 0) {
			break;
		}
		if ((rc = pthread_mutex_lock(&lock2)) != 0) {
			break;
		}
		print("t3 ");
		running[3] = 1;
	}

	if (rc == 0) {
		rc = pthread_mutex_unlock(&lock1);
	}
	if (rc == 0) {
		rc = pthread_mutex_unlock(&lock2);
	}

	if (rc != 0) {
		mutex_error = rc;
		print("thread3: error\n");
	}
	print("thread3: exiting\n");
	return 0;
}

/* consumer */
static void * PTHREAD_LINK thread4(void * arg)
{
	int rc;
	arg = arg;

	print("thread4: starting\n");
	/* get initialization lock */
	if ((rc = pthread_mutex_lock(&start)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}
	if ((rc = pthread_mutex_lock(&lock3)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}
	not_ready--;

	/* release initialization lock */
	if ((rc = pthread_mutex_unlock(&start)) != 0) {
		mutex_error = rc;
		return 0; /* exit thread */
	}

	/* wait for other thread to finish initializing */
	"C-Cover off";
	while (not_ready) {
		sched_yield();
	}
	"C-Cover on";

	/* begin consuming */
	while (1) {
		/* have 3 */
		if ((rc = pthread_mutex_lock(&lock1)) != 0) {
			break;
		}
		if (--product != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock3)) != 0) {
			break;
		}

		/* have 1 */
		if ((rc = pthread_mutex_lock(&lock2)) != 0) {
			break;
		}
		if (--product != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock1)) != 0) {
			break;
		}

		/* have 2 */
		if ((rc = pthread_mutex_lock(&lock3)) != 0) {
			break;
		}
		if (--product != 0) {
			if (Run) {
				rc = 1;
			}
			break;
		}
		if ((rc = pthread_mutex_unlock(&lock2)) != 0) {
			break;
		}
		print("t4 ");
		running[4] = 1;
	}

	if (rc == 0) {
		rc = pthread_mutex_unlock(&lock3);
	}

	if (rc != 0) {
		mutex_error = rc;
		print("thread4: error\n");
	}
	print("thread4: exiting\n");
	return 0;
}

static volatile int Done = 0;

static void * PTHREAD_LINK trylock_test1(void* p)
{
	if (pthread_mutex_trylock((pthread_mutex_t*)p) == 0) {
		write(1, "trylock not working\n", 20);
	}
	Done = 1;
	return 0;
}

static int trylock_test(void)
{
	static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	int rc = pthread_mutex_trylock(&mutex);
	if (rc == 0) {
		pthread_t thread;
		if (pthread_create(&thread, NULL, trylock_test1, &mutex) == 0) {
			while (!Done) {
				sched_yield();
			}
		} else {
			/* Assume our unsafe implementation */
			trylock_test1(&mutex);
		}
		pthread_mutex_unlock(&mutex);
		pthread_mutex_destroy(&mutex);
	}
	return rc;
}

int main(int argc, char * argv[])
{
	pthread_t t1, t2, t3, t4;
	int rc;
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

	if (argc > 1 && strcmp(argv[1], "-c") == 0) {
		Check = 1;
	}

	/* Simple mutex test */
	rc = (
		pthread_mutex_lock(&mutex) != 0 &&
		pthread_mutex_unlock(&mutex) != 0 &&
		pthread_mutex_destroy(&mutex) != 0);
	if (rc != 0) {
		write(1, "main: simple mutex failure\n", 27);
	} else {
		rc = trylock_test();
		if (rc != 0) {
			write(1, "trylock test failed\n", 20);
		}
	}
	if (rc == 0) {
		/* start threads */
		rc = pthread_create(&t1, NULL, thread1, NULL);
		if (rc == 0) {
			rc = pthread_create(&t2, NULL, thread2, NULL);
		}
		if (rc == 0) {
			rc = pthread_create(&t3, NULL, thread3, NULL);
		}
		if (rc == 0) {
			rc = pthread_create(&t4, NULL, thread4, NULL);
		}
		if (rc != 0) {
			if (Check) {
				/* threads not supported, only mutex */
				rc = 0;
			} else {
				print("main: thread creation failed\n");
#				if __convex__
					print("On Exemplar, use -min 6 -max 6 a.out\n");
#				endif
			}
		} else {
			int i;
			/* Number of seconds to run this test */
			const unsigned limit = 5;
#			if SYS_UNIX || unix
#				if __convex__
					cnx_thread_block(limit * 1000000, NULL);
#				else
					sleep(limit);
#				endif
#			endif
#			if SYS_NT
				Sleep(limit * 1000);
#			endif
#			if 0
				time_t t0 = time(NULL);
				while (time(NULL) - t0 < limit)
					sched_yield();
#			endif
			for (i = 1; i <= 4; i++) {
				static char nums[] = "01234";
				if (!running[i]) {
					write(1, "main: thread ", 13);
					write(1, &nums[i], 1);
					write(1, " stuck\n", 7);
					rc = 1;
				}
			}
			if (mutex_error) {
				write(1, "main: mutex failure\n", 20);
				rc = mutex_error;
			}
			Run = 0;
			{
			void* p;
			if (pthread_join(t1, &p) != 0) {
				write(1, "join 1 failure\n", 15);
			}
			if (p != &r1) {
				write(1, "join 1 child status wrong\n", 26);
			}
			if (pthread_join(t2, &p) != 0) {
				write(1, "join 2 failure\n", 15);
			}
			if (p != &r2) {
				write(1, "join 2 child status wrong\n", 26);
			}
			if (pthread_join(t3, &p) != 0) {
				write(1, "join 3 failure\n", 15);
			}
			if (p != NULL) {
				write(1, "join 3 child status wrong\n", 26);
			}
			if (pthread_join(t4, NULL) != 0) {
				write(1, "join 4 failure\n", 15);
			}
			}
		}
	}
	print("main: exiting\n");
	return rc;
}
