#!/bin/sh
# $Id: check.ksh,v 1.510 2000/05/10 23:15:58 paul Exp $
# Copyright (c) Bullseye Testing Technology

# Usage
#   check [-n N] [-r N M] [-u]
#   -n N	- Run test #N only
#	-r N M  - Run tests #N thru #M
#   -u		- Regenerate expected output, written to stdout

list_fail=  # List of test numbers which failed
if [ '~' != ~ ]
then
	typeset -i n_fail n_select n_total
fi
n_fail=0
n_select=0
n_total=0
n_only=""
rm -f calc*.def
unset COLUMNS
unset COVC
rm -f cov*.cfg
COVFILE=test.cov
export COVFILE
unset COVSRCDIR
mode=testing
PROMPT="% "   # for buildrt
unset MAKEFLAGS
TMP=tmp
TMPDIR=tmp
export TMP
export TMPDIR
unset tmp
unset tmpdir
WCL="$WCL -zq"
WCL386="$WCL386 -zq"
nul=nul
if [ -x /bin/pwd ]
then
	cwd=`/bin/pwd`
else
	cwd=`pwd`
fi
Cwd=`echo $cwd | tr "[a-z]" "[A-Z]"`
# force Win32 API to report cwd the same as Korn shell does
cd $cwd
which=which

# Parse options
while [ $# -gt 0 ]
do
	case "$1" in
	-u) mode=updating ;;
	-n) shift; n_only="$n_only $1 " ;;
	-r) shift
		i=$1
		while [ $i -le $2 ]
		do
			n_only="$n_only $i "
			if [ '~' = ~ ]
			then
				i=`expr $i + 1`
			else
				let i=i+1
			fi
		done
		shift
		;;
	*) echo $0: unknown option $1 ;;
	esac
	shift
done

rm -fr test.cov *-
[ ! -d $TMPDIR ] && mkdir $TMPDIR
t1=$TMPDIR/check1.tmp
t2=$TMPDIR/check2.tmp
uname=`uname`
case $uname in
	Windows_*)
		sys=n; cc="cl -GX -nologo"; c2="$cc"; o=-Fe
		;;
	*)
		sys=u; cc="gcc -I../src/libc3"; c2="c++ -ansi -I../src/libc3"; o="-o "
		;;
esac

if [ $sys = n ]; then
	LIB="../lib;$LIB"
	# Remove installed C-Cover directories from LIB and PATH
	LIB=${LIB/[cC]:'\Program Files\ccover\lib;'}
	PATH=${PATH/[cC]:'\Program Files\ccover\bin;'}
fi
if [ $sys = u ]
then
	nul=/dev/null
	which=../src/which
	bindir=`$which covc | sed 's,/covc$,,g'`
	if [ -d "$bindir" ]
	then
		true
	else
		echo Cannot find covc.  Add C-Cover bin directory to PATH.
		exit 1
	fi
	PATH=.:$PATH
	if [ ! -f init ]
	then
		ln -f ../sample/* .
		>init
	fi
	if ($cc -o calc1 calc1.c && ./calc1 </dev/null >/dev/null)
	then
		true
	else
		echo This script requires $cc.
		# return success since no problems detected
		exit 0
	fi
	if ($c2 -o calc3 calc3*.cpp && ./calc3 </dev/null >/dev/null)
	then
		true
	else
		echo This script requires $c2 to compile C++ programs.
		# return success since no problems detected
		exit 0
	fi
else
	bindir=none-none-none
	for i in \
		dirent.h \
		fnmatch.h \
		iso646.h \
		libcdef.h \
		pthread.h \
		unistd.h
	do
		if [ ! -f /usr/lib/$i -o $i -nt /usr/lib/$i ]; then
			cp -f $i /usr/lib/
		fi
	done
	rm -rf src-tmp; mkdir src-tmp
	cp ../lib/*.* src-tmp
	rm -f src-tmp/*.lib src-tmp/*.obj
fi

br() {
	covbr -c0 -e1 -q -w47 $* | sed \
		-e 's/[ ]*-->  [ ]*[0-9][0-9]*/./' \
		-e 's/^[ ][ ]*[0-9][0-9]*\([a-z]\)/K\1/' \
		-e 's/[ ]*-->\([TFtf]\)[ ]*[0-9][0-9]*/\1/' \
		-e 's/X[ ]*[0-9][0-9]*/X/' \
		-e 's/TF[ ]*[0-9][0-9]*/x/' \
		-e 's/  tf[ ]*[0-9][0-9]*/y/' \
		-e 's/[ ]*\([&(),;<=>?{|}]\)/\1/g' \
		-e     's/\([&(),;<=>?{|}]\)[ ]*/\1/g' \
		-e 's/ [ ]*/ /'
}

#-------------------------- Private Function ---------------------------
# Name:    clean
# Args:      none
# Return:    success
# Purpose: Remove version-specific and platform-specific information
#          from the output of a C-Cover command.

clean() {
	sed -e 's/^[	]*//' \
		-e 's/ [ ]*$//' \
		-e 's/
$//' \
		-e 's/\\/\//g' \
		-e "s,[cC]:/[a-zA-Z0-9/~]*/[bB][iI][nN][a-zA-Z]*/$cc ,%cc ,g" \
		-e 's,[cC]:/[a-zA-Z0-9/~]*/[bB][iI][nN][a-zA-Z]*/\([a-z0-9]*\) ,%.../\1 ,' \
		-e 's,"[cC]:/[pP][rR][oO][gG][rR][aA][mM] [fF][iI][lL][eE][sS]/.*/cl" -GX -nologo ,%cc ,g' \
		-e "s, [-a-zA-Z0-9/_:.]*/$cc , %cc ,g" \
		-e 's, [-a-zA-Z0-9/_:.]*/gcc , %cc ,g' \
		-e 's, [-a-zA-Z0-9/_:.]*/c++ , %cc ,g' \
		-e "s, $cc , %cc ,g" \
		-e 's/ gcc / %cc /g' \
		-e 's/ c++ / %cc /g' \
		-e 's,"[cC]:/[pP][rR][oO][gG][rR][aA][mM] [fF][iI][lL][eE][sS]/.*/\([a-z0-9]*\)" ,%.../\1 ,' \
		-e "s,$bindir[/]*,%bindir/,g" \
		-e "s,$cwd/tmp/covc[0-9]*/,tmp/covc/,g" \
		-e "s,$cwd,%cwd,g" \
		-e "s,$Cwd,%cwd,g" \
		-e 's/ [3-9]\.[0-9]\.[0-9a-z][0-9a-z]* [-\/a-zA-Z0-9]*/ %ver/g' \
		-e 's/=[3-9][0-9][0-9][0-9]/=%ver/g' \
		-e 's/cov_version_[0-9_a-z]*/cov_version_%ver/g' \
		-e 's/, Serial [0-8][0-9]*//' \
		-e 's/[, ]*Copyright (c) Bullseye Testing Technology [-0-9]*//g' \
		-e 's/.*Borland International//g' \
		-e 's,.*Inprise Corporation,,' \
		-e 's/Microsoft (R).*//g' \
		-e 's/Copyright.*//g' \
		-e 's/^Object Modules .*//g' \
		-e 's/^Run File .*//g' \
		-e 's/List File .*//g' \
		-e 's/Libraries .*//g' \
		-e 's/Definitions File .*//g' \
		-e 's/.*OPTLINK (R) for Win32.*//g' \
		-e 's/^Generating Code.*//' \
		-e 's,^tmp/covc[0-9]*/[a-zA-Z0-9._:]*,,' \
		-e 's/@..nm[0-9]*.$/@%file/g' \
		-e 's/^bounce[a-z.]*[:]*$//' \
		-e 's/^calc[a-z0-9.]*[:]*$//' \
		-e 's/^t_[a-zA-Z0-9._ ]*[:]*$//' \
		-e 's/^ space.c:$//' \
		-e 's, -lexc,,' \
		-e 's, -lposix4,,' \
		-e 's, -lrt,,' \
		-e 's/ -lpthread//' \
		-e 's/ [a-z0-9\/:]*\/echo/ %echo/g' \
		-e '/[aA]bort.*[cC]ore *[dD]ump/d' \
		-e 's/ ([0-9a-f][0-9a-f]*)\.  / %object-id.  /' \
		-e 's/Error occurred at [0-9,]*/Error occurred at %.../' \
		-e 's, /lib/libpthread.so.0,,' \
		-e 's,ccoverEntryUser-[^.]*,ccoverEntryUser,g' \
		-e 's,cov_probe_[0-9]_[0-9]_[0-9][a-z]*,cov_probe,g' \
		-e 's, -Ob1,,' \
		-e 's,^covxCheckTmp.cpp$,,' \
		-e 's,^checkTmp.*\.c[p]*$,,' \
		-e "s,.*Borland C++.*Win32.*,," \
		-e "s,Turbo Incremental Link.*,," \
		-e '/^$/d' \
		-e "s,tmp/covc[0-9]*/,tmp/covc/,g" \
		-e "s, -ladvapi32 -luser32,," \
		-e "s,\./\.\./lib/libcov.a,%bindir/../lib/libcov.a," \
		-e "/^ld: WARNING 84: /d" \
		-e 's,^,	,' |
			if [ $sys = u ]
			then
				sed -e '/License expires in [0-9]* days/d'
			else
				cat
			fi
}

#-------------------------- Private Function ---------------------------
# Name:    definitions
# Args:      none
# Return:    success
# Purpose: Retrieve from stdin the C-Cover definition code.

definitions() {
	sed -n "/C-Cover definitions/,/end of definitions/p"
}

#-------------------------- Private Function ---------------------------
# Name:    run
# Args:      $* - a command and its arguments
# Return:    success
# Purpose: Run a command and check its output against stdin.

run() {
	if [ ~ = '~' ]
	then
		n_total=`expr $n_total + 1`
	else
		let n_total=n_total+1
	fi
	option=
	# determine if test selected for this platform
	if [ ."$1" = .-n -o ."$1" = .-u -o ."$1" = .-0 ]
	then
		enable=0
		while [ ."$1" = .-n -o ."$1" = .-u -o ."$1" = .-0 ]
		do
			option="$option $1"
			if [ $1 = -$sys ]
			then
				enable=1
			fi
			shift
		done
	else
		enable=1
	fi

	# If test numbers specified and this test enabled
	if [ "$n_only" != "" -a $enable = 1 ]
	then
		# If this test number was specified
		enable=0
		for i in $n_only
		do
			if [ $i = $n_total ]
			then
				enable=1
				break
			fi
		done
	fi

	if [ $enable = 1 ]
	then
		if [ ~ = '~' ]
		then
			n_select=`expr $n_select + 1`
		else
			let n_select=n_select+1
		fi

		if [ $mode = testing ]
		then
			# print the command
			echo $n_total "$*"
		fi

		if eval "( $* )"
		then
			true
		else
			echo status 1
		fi >$TMPDIR/run1.tmp 2>&1
		clean <$TMPDIR/run1.tmp >$TMPDIR/run2.tmp

		if [ $mode = testing ]
		then
			# The redundant sed command below works around a glitch on Win9x
			if sed -e "s,^[	]*,	," | diff - $TMPDIR/run2.tmp
			then
				true
			else
				if [ $mode = testing ]
				then
					echo "*** test #$n_total failed"
				fi
				n_fail=`expr $n_fail + 1`
				list_fail="$list_fail #$n_total"
			fi
		else
			echo "#$n_total"
			echo run $option "'$*'<<."
			cat $TMPDIR/run2.tmp
			echo .
		fi
	else
		if [ $mode = updating ]
		then
			echo "#$n_total"
			echo run $option "'$*'<<."
			cat
			echo .
		fi
	fi
}

#-------------------------- Private Function ---------------------------
# Name:    finish
# Args:      none
# Return:    success
# Purpose: Perform termination tasks.

finish() {
	rm -fr \
		$TMP/covc \
		a.out \
		bounce*.def \
		bounce*.obj \
		bounce*.pdb \
		bounce.ex? \
		bouncedl.dll \
		bouncedl.exp \
		bouncedl.lib \
		calc3stk.o \
		calc[123].exe \
		check*.tmp \
		checkTmp* \
		core \
		covxCheckTmp* \
		ctype-ccover.h \
		ro \
		src-tmp \
		t_*.cov \
		t_*.dll \
		t_*.exe \
		t_*.exp \
		t_*.o \
		t_*.obj \
		t_empty.c \
		t_pragma.cpp \
		t_retain.c \
		t_tmp* \
		test.cov \
		tmp \
		tmp.cov
	if [ $mode = testing ]
	then
		echo Total: $n_total
		echo Selected: $n_select
		if [ $n_fail = 0 ]
		then
			echo Passed: ALL
		else
			echo Failed: $n_fail \( $list_fail \)
			exit 1
		fi
	else
		echo finish
	fi
	cov01 -q -0
}

#-------------------------- Main ---------------------------------------

# make sure C-Cover is installed
if ($which which >$t1 2>&1 && $which covc >$t1 2>&1)
then
	true
else
	echo C-Cover not installed.
	# return success since no problems detected
	exit 0
fi

if [ $mode = updating ]
then
	if [ $sys = u ]
	then
		program=$0
	else
		program=check.ksh
	fi
	sed -n "1,/^# Begin tests/p" <$program
	echo ""
fi

# Begin tests

#1
run '# Basic sanity check
	covc -q $cc ${o}calc1 calc1.c
	echo 1= | calc1
	covbr -q -c0 calc1.c:pop
	covc -q $c2 calc3.cpp calc3stk.cpp
'<<.
	Enter reverse-polish expressions, q to quit
	1
	X        21  static int pop(Pvoid)
.
#2
run 'cov01; cov01 -q -h'<<.
	C-Cover Coverage Build Enable/Disable %ver
	Usage: cov01 -0|--off [options]
	       cov01 -1|--on [options]
	       cov01 -h|--help
	       cov01 -s|--status [options]
	Actions:
	 -0 --off        Disable Coverage Build
	 -1 --on         Enable Coverage Build
	 -h --help       Display this usage summary
	 -s --status     Show pseudo compiler command status
	Options:
	 -q --quiet      Quiet
	 -v --verbose    Verbose
	 --no-banner     Suppress startup banner
	Usage: cov01 -0|--off [options]
	       cov01 -1|--on [options]
	       cov01 -h|--help
	       cov01 -s|--status [options]
	Actions:
	 -0 --off        Disable Coverage Build
	 -1 --on         Enable Coverage Build
	 -h --help       Display this usage summary
	 -s --status     Show pseudo compiler command status
	Options:
	 -q --quiet      Quiet
	 -v --verbose    Verbose
	 --no-banner     Suppress startup banner
	status 1
.
#3
run -n '
	covc -h -q
'<<.
	Usage: covc [options] compiler ... source ...
	-C     --cond         Measure the conditional operator (?:)
	-dDir  --srcdir Dir   Source file directory is Dir
	-fPath --file Path    Coverage data file is Path
	-g     --debug        Allow interactive debugging. Bigger, slower
	-h     --help         Display this usage summary
	-iList --ignore List  Do not preprocess sources in comma separated List
	-l     --leave        Leave intermediate source file with probes
	-n     --function     Measure function invocation coverage only
	-q     --quiet        Quiet
	-s     --shell        Spawn compiler using command processor
	-tList --select List  Analyze selected source regions in comma separated List
	-v     --verbose      Verbose
	-xList --suffix List  Analyze files with suffixes in comma separated List
	--c++                 Source files with suffix .c are C++
	--comment             Preserve comments during preprocessing
	--delay-io            Delay run-time file i/o
	--no-banner           Suppress startup banner
	--no-cpp              Do not use smart preprocessor
	--no-lib              Do not automatically link with C-Cover library
	--no-Ob1              Do not add to Microsoft compile commands
	--retain              Retain previous measurements when possible
	status 1
.
#4
run -n '
	echo >t_tmp.cc "
		#if !__cplusplus
			#error __cplusplus not defined
		#endif
		int main() { return 0; }
	"
	covc -v bcc32 -c -P -R -R- -n. -Dinc t_tmp.cc 2>&1 | grep cov90[249]
	echo -P- -P >bcc32.cfg
	covc -v bcc32 -c t_tmp.cc 2>&1 | grep cov90[249]
	cp bcc32.cfg cfg
	rm bcc32.cfg
	covc -v bcc32 +cfg -c t_tmp.cc 2>&1 | grep cov90[249]
	echo -P -R -R- -n. -ot_tmp.obj -Din_cfg >cfg
	covc -v bcc32 +cfg -c -H t_tmp.cc 2>&1 | grep cov90[249]
	rm cfg; covmgr -r -q t_tmp.cc
'<<.
	note cov902: running %.../cpp32 -DCCOVER=%ver -c -P -Dinc -otmp/covc/t_tmp.i -P -w- -D__cplusplus=1 t_tmp.cc
	note cov904: processing source t_tmp.cc
	note cov902: running %.../bcc32 -c -P -R -R- -n. tmp/covc/t_tmp.cc
	note cov909: exit status 0
	note cov902: running %.../cpp32 -DCCOVER=%ver -c -otmp/covc/t_tmp.i -P -w- -D__cplusplus=1 t_tmp.cc
	note cov904: processing source t_tmp.cc
	note cov902: running %.../bcc32 -c tmp/covc/t_tmp.cc
	note cov909: exit status 0
	note cov902: running %.../cpp32 -DCCOVER=%ver +cfg -c -otmp/covc/t_tmp.i -P -w- -D__cplusplus=1 t_tmp.cc
	note cov904: processing source t_tmp.cc
	note cov902: running %.../bcc32 +cfg -c tmp/covc/t_tmp.cc
	note cov909: exit status 0
	note cov902: running %.../cpp32 -DCCOVER=%ver -Din_cfg -c -H -otmp/covc/t_tmp.i -P -w- -D__cplusplus=1 t_tmp.cc
	note cov904: processing source t_tmp.cc
	note cov902: running %.../bcc32 +cfg -c -H tmp/covc/t_tmp.cc
	note cov909: exit status 0
.
#5
run -n '
	covc -v cl -c -Y calc1.c 2>&1 | grep 90[249]
	covc -v cl -c -Yu -Tpcalc1.c 2>&1 | grep 90[249]
'<<.
	note cov902: running internalCpp -DCCOVER=%ver -c -Y calc1.c -+cc cl
	note cov904: processing source calc1.c
	note cov902: running %.../cl -c -Y tmp/covc/calc1.c
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -c -Tp calc1.c -+cc cl
	note cov904: processing source calc1.c
	note cov902: running %.../cl -c -Tp tmp/covc/calc1.c
	note cov909: exit status 0
.
#6
run -n '
	covc -q -s cl -c calc1.c
'<<.
.
#7
run -n '
	cd src-tmp
	unset TMP
	buildrt nt cl
	cd -
'<<.
	% cl -c -DSYS_nt=1 -nologo -W4 -Zl -Og -Oi -Ot -Oy -Gs -Gr  covrt.c
	covrt.c
	% cl -c -DSYS_nt=1 -nologo -W3 -Zl -Og -Oi -Ot -Oy -Gs -Gr  covrt-nt-entryUser.c
	covrt-nt-entryUser.c
	% lib -nologo -out:covntm.lib covrt.obj covrt-nt-entryUser.obj
	% del covrt-nt-entryUser.obj
	% cl -c -DSYS_nt=1 -nologo -W4 -Zl -Og -Oi -Ot -Oy -Gs -Gr  covrt-nt-atexit.c
	covrt-nt-atexit.c
	buildrt: successfully built covntm.lib
.
#8
run -n '
	rm -f bounce*.obj bounce.exe bouncedl.dll bouncedl.lib
	cp covc.exe cl.exe-
	cp covlink.exe link.exe-
	cov01 -1q &&
	unset CFLAGS
	nmake -nologo -f bounce.mak CFG="bounce - Win32 Debug" 2>&1 |
		sed -e "/link\.exe @/d" -e s,NMAKE,nmake, &&
	cov01 -0q &&
	covfn -qk bounce*.c
	covmgr -q -r bounce*.c
'<<.
	nmake /L                  /F "./bounce.mak" CFG="bouncedl - Win32 Debug"
	cl.exe /nologo /MLd /W3 /Z7 /Od /c "bouncedl.c"
	C-Cover Compile C/C++ %ver
	C-Cover Link %ver
	ccoverEntryUser.c
	   Creating library ./bouncedl.lib and object ./bouncedl.exp
	cl.exe /nologo /MLd /W3 /Z7 /Od /c "bounce.c"
	C-Cover Compile C/C++ %ver
	C-Cover Link %ver
	ball_update  bouncedl.c      11  0 / 1       0 /     8 =   0%
	WndProc      bounce.c         9  0 / 1       0 /     5 =   0%
	WinMain      bounce.c        46  0 / 1       0 /     4 =   0%
	ball_change  bouncedl.c      36  0 / 1       0 /     0
.
#9
run -n -u '# gcc
	g++ -c calc3stk.cpp
	covc --shell -v gcc -S calc1.c 2>&1 | grep "cov90[249]"
	covc -p- -v gcc -c calc1.c 2>&1 | grep "cov90[249]"
	covc -p- -v gcc -I. -c calc1.c 2>&1 | grep "cov90[249]"
	rm calc1.[os]
'<<.
	note cov902: running %cc -DCCOVER=%ver -E calc1.c >tmp/covc/calc1.i
	note cov904: processing source calc1.c
	note cov902: running %cc -S tmp/covc/calc1.c
	note cov909: exit status 0
	note cov904: processing source calc1.c
	note cov902: running %cc -I. -c tmp/covc/calc1.c
	note cov909: exit status 0
	note cov904: processing source calc1.c
	note cov902: running %cc -I. -c tmp/covc/calc1.c
	note cov909: exit status 0
.
#10
run -u 'cp $bindir/../lib/libcov.a l.a;
  covc -v c++ calc3.cpp calc3stk.o l.a 2>&1 | grep cov90[249]
  rm l.a'<<.
	note cov902: running %cc -DCCOVER=%ver -E calc3.cpp >tmp/covc/calc3.i
	note cov904: processing source calc3stk.h
	note cov904: processing source calc3.cpp
	note cov902: running %cc tmp/covc/calc3.cpp calc3stk.o l.a %bindir/../lib/libcov.a
	note cov909: exit status 0
.
#11
run -n '# pseudo compilers
	# make sure correct config file used
  cp covc.exe cl.exe;
  echo -z >covc.cfg; cl calc1.c; rm covc.cfg;
	# compile using each possible pseudo compiler
  for i in bcc32 cl; do;
    cp covc.exe $i.exe;
    $i -c calc1.c
    rm $i.exe;
  done'<<.
	C-Cover Compile C/C++ %ver
	error cov137: unknown option "-z"
	C-Cover Compile C/C++ %ver
	C-Cover Compile C/C++ %ver
.
#12
run -n 'cp covc.exe cl.exe; cl -c; cl -c calc1.c'<<.
	C-Cover Compile C/C++ %ver
	Command line error D2003 : missing source filename
	warning cov816: original unmodified compile command fails
	C-Cover Compile C/C++ %ver
.
#13
run -n 'rm -f $(echo ????????.exe | sed s/cov......exe//g) at_least_one;
  cp covc.exe cl.exe;
  (
  cov01 -0;    echo -0 cl.exe*;
  cov01 -1;    echo -1 cl.exe*;
  cov01 -0 -q; echo -0 cl.exe*;
  cov01 -1 -q; echo -1 cl.exe*;
  cov01 -0 -v; echo -0 cl.exe*;
  cov01 -0 -v; echo -0 cl.exe*;
  cov01 -1 -v; echo -1 cl.exe*;
  cov01 -1 -v; echo -1 cl.exe*;
  chmod -w cl.exe;
  cov01 -0;    echo -0 cl.exe*;
  cov01 -1;    echo -1 cl.exe*;
  ) 2>&1 |
  grep -i -e "cov[1-9][0-9][0-9]" -e cl |
  sed -e s/CL.EXE/cl.exe/g -e "s|directory \..\$|directory .|g";
  chmod +w cl.exe && rm cl.exe'<<.
	-0 cl.exe-
	-1 cl.exe
	-0 cl.exe-
	-1 cl.exe
	note cov925: bin directory .
	disabled: ./cl.exe
	-0 cl.exe-
	note cov925: bin directory .
	disabled: ./cl.exe
	-0 cl.exe-
	note cov925: bin directory .
	enabled: ./cl.exe
	-1 cl.exe
	note cov925: bin directory .
	enabled: ./cl.exe
	-1 cl.exe
	-0 cl.exe-
	-1 cl.exe
.
#14
run -u '
	(
	cov01 -1 -v; echo -1 $bindir/gcc*
	cov01 -0 -v; echo -0 $bindir/gcc*
	cov01 -1 -v; echo -1 $bindir/gcc*
	cov01 -0 -v; echo -0 $bindir/gcc*
	) >$t1 2>&1
	# have to use full path due to weird std{out,err} shell bug
	cov01 -1q &&
	$bindir/gcc -c calc1.c &&
	cov01 -0q &&
	grep "cov[1-9][0-9][0-9]" $t1 &&
	grep gcc $t1
'<<.
	C-Cover Compile C/C++ %ver
	note cov925: bin directory %bindir/
	note cov925: bin directory %bindir/
	note cov925: bin directory %bindir/
	note cov925: bin directory %bindir/
	enabled: %bindir/gcc
	-1 %bindir/gcc
	disabled: %bindir/gcc
	-0 %bindir/gcc-
	enabled: %bindir/gcc
	-1 %bindir/gcc
	disabled: %bindir/gcc
	-0 %bindir/gcc-
.
#15
run 'rm -f tmp.cov; covc -ftmp.cov -q $cc -c calc1.c && test -f tmp.cov'<<.
.
#16
run 'covc -q "`$which $cc`" -c calc1.c'<<.
.
#17
run '# Indirect file
	echo $cc -c calc1.c >$t1 && covc -q @$t1 2>&1 |
		grep -v "^cl "
		true
'<<.
.
#18
run -n '
	echo $cc -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c -c \
		-c -c -c -c -c -c -c calc1.c >$t1 && covc -q @$t1 2>&1 |
	grep -v -e "^cl " -e "^[ ]* -c"
	true
'<<.
.
#19
run '# Nested argument files
	echo -q -, >$t1; echo @$t1 >$t2; covc @$t2;
	d=`$which covdir`; d=`dirname $d`;
	cp $t2 $d/covdir.cfg; covdir; rm $d/covdir.cfg
'<<.
	error cov137: unknown option "-,"
	error cov137: unknown option "-,"
.
#20
run -n 'head=$(which head); mv=$(which mv);
  d=$(dirname $(which covc)); unset PATH;
  if [ -f covx.exe ]; then; $mv covx.exe covx; fi;
  $d/covc 2>$t1 || $head -n 1 $t1;
  if [ -f covx ]; then; $mv covx covx.exe; fi'<<.
	./covc: cannot run covx.exe : The system cannot find the file specified.
.
#21
run 'covc -q $cc -De1=1 t_error.c || covc -q $cc -De2=1 t_error.c ||
	covc -q $cc -De3=1 t_error.c ||
	covc -q $cc -De4=1 t_error.c ||
	covc -q $cc -De5=1 t_error.c
'<<.
	t_error.c(15): error cov118: unbalanced {
	t_error.c(12): error cov118: unbalanced }
	t_error.c(19): error cov116: more than 64 nested parenthesis
	t_error.c(19): error cov117: syntax error
	t_error.c(28): error cov118: unbalanced (
	t_error.c(31): error cov118: unbalanced )
	status 1
.
#22
run 'covc -q $c2 -c t_class.cpp && br t_class.cpp &&
		covfn -c -q t_class.cpp'<<.
	. int class2::operator[](const char
	. int class2::operator()(){retur
	. char class2::operator()(int){retur
	. void* class2::operator new(size_t)
	. class2::operator void *(){retur
	. int class2::operator++(){return
	. int class2::operator--(){re
	. class2::class2(){}
	. class2::~class2(){}
	. int class2::name21()const{retur
	. inline class class1 *class2::name2
	. class2::operator void*()const{
	. if(b)return 0;return(void *)t
	. class3(){}
	. virtual ~class3(){}
	. int name32(int arg){return 0;}
	. int name33(int arg)const{retur
	. int operator()(){return 0;
	. int operator [ ](int i){retu
	. operator long(){return 0;}
	. operator unsigned long(){return
	. int regular(){return 0;}
	. int name41(const class2&r){ret
	. int name42(void){return 0;}
	. Class5(int a): class4(a){}
	. class6(){}
	. void function1(){}
	. extern void function2(){}
	. class8(){}
	. template<class t>void class8<t>:
	. class9(){}
	. class10(){}
	. void function3(){}
	. extern void function4(){}
	. void function5(){}
	. extern void function6(){}
	.a int function12(short a){return a
	.b
	. void name111(){}
	. void name1111(){}
	. void name1121(){}
	. void name11211(){}
	. void name1122(){}
	. void name112(){}
	. void class11::class112::class1121:
	. if(v)v++;
	. void name9(){}
	. extern "C" void extern_c1(){}
	. extern "C"{void extern_c2(){}
	. template<class T>void Class12<T>
	. void f(){m.f();}
	. int main()
	"Function","Source","Line","FnCov","C/D Coverage","out of","%"
	"class2::operator void*","t_class.cpp",34,0,0,2,0%
	"class11::class112::class1121::name11212","t_class.cpp",131,0,0,2,0%
	"class2::operator[]","t_class.cpp",23,0,0,0,""
	"operator ()","t_class.cpp",24,0,0,0,""
	"operator()","t_class.cpp",25,0,0,0,""
	"class2::operator new","t_class.cpp",26,0,0,0,""
	"class2::operator void *","t_class.cpp",27,0,0,0,""
	"class2::operator++","t_class.cpp",28,0,0,0,""
	"class2::operator--","t_class.cpp",29,0,0,0,""
	"class2::class2","t_class.cpp",30,0,0,0,""
	"class2::~class2","t_class.cpp",31,0,0,0,""
	"class2::name21","t_class.cpp",32,0,0,0,""
	"class2::name22","t_class.cpp",33,0,0,0,""
	"class3::class3","t_class.cpp",45,0,0,0,""
	"class3::~class3","t_class.cpp",46,0,0,0,""
	"class3::name32","t_class.cpp",48,0,0,0,""
	"class3::name33","t_class.cpp",49,0,0,0,""
	"class3::operator ()","t_class.cpp",50,0,0,0,""
	"class3::operator [ ]","t_class.cpp",51,0,0,0,""
	"class3::operator long","t_class.cpp",52,0,0,0,""
	"class3::operator unsigned long","t_class.cpp",53,0,0,0,""
	"regular","t_class.cpp",56,0,0,0,""
	"class4::name41","t_class.cpp",61,0,0,0,""
	"class4::name42","t_class.cpp",62,0,0,0,""
	"Class5::Class5","t_class.cpp",66,0,0,0,""
	"class6::class6","t_class.cpp",72,0,0,0,""
	"function1","t_class.cpp",81,0,0,0,""
	"function2","t_class.cpp",82,0,0,0,""
	"class8::class8","t_class.cpp",86,0,0,0,""
	"class8::name81","t_class.cpp",89,0,0,0,""
	"class9::class9","t_class.cpp",92,0,0,0,""
	"class10::class10","t_class.cpp",98,0,0,0,""
	"function3","t_class.cpp",101,0,0,0,""
	"function4","t_class.cpp",102,0,0,0,""
	"function5","t_class.cpp",105,0,0,0,""
	"function6","t_class.cpp",106,0,0,0,""
	"function12","t_class.cpp",110,0,0,0,""
	"function12","t_class.cpp",110,0,0,0,""
	"class11::name111","t_class.cpp",114,0,0,0,""
	"class11::class111::name1111","t_class.cpp",117,0,0,0,""
	"class11::class112::name1121","t_class.cpp",121,0,0,0,""
	"class11::class112::class1121::name11211","t_class.cpp",124,0,0,0,""
	"class11::class112::name1122","t_class.cpp",127,0,0,0,""
	"class11::name112","t_class.cpp",129,0,0,0,""
	"c1::c2::c3::c4::c5::c6::c7::c8::c9::name9","t_class.cpp",139,0,0,0,""
	"extern_c1","t_class.cpp",152,0,0,0,""
	"extern_c2","t_class.cpp",153,0,0,0,""
	"Class12::f","t_class.cpp",165,0,0,0,""
	"Class13::f","t_class.cpp",169,0,0,0,""
	"main","t_class.cpp",171,0,0,0,""
	"Total","","",0.00,0,4,0%
.
#23
run 'covc -C -q $cc -c t_condop.c && br t_condop.c;
	covmgr -r -q t_condop.c'<<.
	. int main(void)
	. i=1?2 : 3;
	. printf("%d %d %d/n",i,1?2 : 3
	. i=0?(2,3): 4;
	. i=1?(2,3):(4,5);
	. i=0
	. :(1?3 : 4);
	.a i=1?0 :
	.b 2?3 : 4;
	.a i=(1?0 : 2
	.b)?3 : 4;
	.a i=1?0 :(
	.b 2?3 : 4);
	.a for(1?2 : 3;
	.d 7?8
	.a for(1?2 : 3;;
	.b 7?8 : 9)
	. i=(1)?(int)2:(int)3;
.
#24
run '# Column numbers without cpp.exe
	if [ $sys = n ]
	then
		c2="bcc32 -w-"
	fi
	covc -q $c2 -c t_const.cpp
	br t_const.cpp
		covmgr -r -q t_const.cpp'<<.
	. int f01()
	. int f02()
	. int f03()
	. int f04()
	. int f05()
	. int f06()
	. int f07()
	. int f08()
	. int f09()
	. int f11()
	. int f12()
	. int f13()
	. int f14()
	. int f15()
	. int f16()
	. int f17()
	. int f18()
	. void f19()
	. int f20()
	. void f21()
	.{if(i==1);}
	. void f22()
	.a{if(
	.b i&&
	. void f23()
	.{if(1==i);}
	. void f24()
	.a{if(
	.c i);}
	. int f25()
	. int f26()
	. int f27()
	. operator char(){return '0';}
	. void f29()
	. void f30()
	. void f31()
	.{int __i=0;if(__i);}
	. operator int(){return 0;}
	. void f(){
	. if(*this);
	.a if(
	.c i0);
	. if(sizeof i0==i0);
	.a if(
	.c i0);
	.a if(
	.c i0)
	. void f32(){
	.a if(
	.c m1(a));
	.a if(
	.d
	.a if(
	.c m2(a,0));
	.a if(
	.b a&&
	.a if(
	.b a&&
	.a if(
	.c
	.a if(
	.b m1(a)&&0);
	. if(m1(a));
	.a if(
	.d
	.a if(
	.c
	.a if(
	.b m2(a,0));
	. void f33(){
.
#25
run -n '
	covc -q $c2 -c t_const.cpp
	br t_const.cpp
	covmgr -r -q t_const.cpp
'<<.
	. int f01()
	. int f02()
	. int f03()
	. int f04()
	. int f05()
	. int f06()
	. int f07()
	. int f08()
	. int f09()
	. int f11()
	. int f12()
	. int f13()
	. int f14()
	. int f15()
	. int f16()
	. int f17()
	. int f18()
	. void f19()
	. int f20()
	. void f21()
	.{if(i==1);}
	. void f22()
	.a{if(
	.b i&&
	. void f23()
	.{if(1==i);}
	. void f24()
	.a{if(
	.c i);}
	. int f25()
	. int f26()
	. int f27()
	. operator char(){return '0';}
	. void f29()
	. void f30()
	. void f31()
	.{int __i=0;if(__i);}
	. operator int(){return 0;}
	. void f(){
	. if(*this);
	.a if(
	.c i0);
	. if(sizeof i0==i0);
	.a if(
	.c i0);
	.a if(
	.c i0)
	. void f32(){
	.a if(
	.c m1(a));
	.a if(
	.c m2(0,a));
	.a if(
	.c m2(a,0));
	.a if(
	.b a&&
	.a if(
	.b a&&
	.a if(
	.c a);
	.a if(
	.b m1(a)&&
	. if(m1(a));
	.a if(
	.c a);
	. if(m2(0,a));
	. if(m2(a,0));
	. void f33(){
.
#26
run -u '
	covc -q gcc -o t_gcc t_gcc.c
	t_gcc
	br t_gcc.c
	covmgr -q -r t_gcc.c
'<<.
	X int main()
	F if(({k0;}));
	Fa if(
	tb({k1;})&&
	fc({k0;}));
	F if(({{k0;}}));
	Ta if(
	fb({{k0;}})||
	tc({k1;}));
	F for(;({k0;}););
	Fa for(;
	fb({k0;})&&
	.c({k1;}););
	X case 0: break;
	. default: break;
.
#27
run 'covc -q -p- cp t_name.cpp $t1 && br t_name.cpp;
	covfn -q -k -s t_name.cpp
  covmgr -q -r t_name.cpp'<<.
	. void f(){i++;}
	. void g(){i++;}
	. int g(int a){return a;}
	. int k(){}
	. void q(){}
	. int g(char a){return l+a;}
	. int k(){return g(i);}
	. void V::f(){}
	. void g(){}
	. void y_f(){}
	. void x_f(){}
	. void f(X){}
	. void g(){f(x);}
	. void h(){
	. void p(){
	. void m(){q();}
	. void q(){}
	. void q(){}
	. void f(char){}
	. void f(int){}
	. void f(int){}
	. void f(){
	. void f1(){}
	. void f2(){}
	. void f3(){}
	. void f()
	. if(i);
	f             t_name.cpp       3  0 / 1       0 /     0
	A::g          t_name.cpp       9  0 / 1       0 /     0
	N::g          t_name.cpp      17  0 / 1       0 /     0
	N::k          t_name.cpp      18  0 / 1       0 /     0
	N::q          t_name.cpp      19  0 / 1       0 /     0
	N::g          t_name.cpp      23  0 / 1       0 /     0
	N::k          t_name.cpp      25  0 / 1       0 /     0
	Q::V::f       t_name.cpp      34  0 / 1       0 /     0
	Q::V::g       t_name.cpp      36  0 / 1       0 /     0
	A::X::Y::y_f  t_name.cpp      46  0 / 1       0 /     0
	A::X::x_f     t_name.cpp      48  0 / 1       0 /     0
	A::f          t_name.cpp      50  0 / 1       0 /     0
	A::g          t_name.cpp      52  0 / 1       0 /     0
	h             t_name.cpp      55  0 / 1       0 /     0
	X::p          t_name.cpp      61  0 / 1       0 /     0
	X::m          t_name.cpp      64  0 / 1       0 /     0
	X::q          t_name.cpp      65  0 / 1       0 /     0
	q             t_name.cpp      67  0 / 1       0 /     0
	D::f          t_name.cpp      79  0 / 1       0 /     0
	E::f          t_name.cpp      85  0 / 1       0 /     0
	D::f          t_name.cpp      90  0 / 1       0 /     0
	f             t_name.cpp      92  0 / 1       0 /     0
	N9::f1        t_name.cpp     103  0 / 1       0 /     0
	N9::C9::f2    t_name.cpp     105  0 / 1       0 /     0
	f3            t_name.cpp     108  0 / 1       0 /     0
	f             t_name.cpp     113  0 / 1       0 /     2 =   0%
.
#28
run '
	covc -q $cc ${o}t_parse t_parse.c
	t_parse
	br t_parse.c
'<<.
	X int f1(int i){return 0;}
	X int f2(int i,int j){return 0;}
	X int main(void)
	T if(e);
	F while(k0);
	T if(e){}
	F if(p);
	T if(!p);
	T if(e);
	Fa if(e1)
	.b if(e2);
	Fa if(
	fb e1&&
	. e2)
	F if(
	x for(e=0;e<10;e++);
	x for(e=0;e<10;e++);
	x for((e=0);(e<10);(e++));
	x for((e=0);(e<10);(e++))
	T e<10;
	F for(;k0;)
	T while(k1)
	T if(k1)
	. case 0:
	. case 2:
	. case 3:
	. case 4:
	. case 5:
	. case 6:
	. L1:     case 7:
	. case 8:
	. case 9:
	. case(10):
	X default:
	. case 11:
	. if(k1)
	. case 12:
	.a case 13: for(;
	.b k1;)
	. case 1:
	. case(1?2 : 3):
	. case(1?4 : 5):
	. default:
	.a case(14):
	.b case(15):
	Ta if(k1)
	Fb if(k0);
	Fa for(;k0;)
	.b if(k1);
	Fa for(;k0;)
	.b if(k1)for(;
	.c k0;);
	x for(i=0;i<=4;i++){
	X case 0:
	. case 100:
	X case 1:
	X case 2:
	. case 101:
	X case 3:;/* null statement */
	.a case 102:break;
	.b case 103:break;
	Xa case 4:;
	.b case 104:break;
	.}
	X case 100000L: break;
	. default: break;
	X case 0xffffffff: break;
	. default: break;
	F if((int)k0);
	F for(;(int)k0;);
	Fa if(
	tb k1&&
	fc k0);
	Fa if(
	fb k0&&
	.c k1);
	Ta if(
	fb k0||
	tc k1);
	Ta if(
	tb k1||
	.c k0);
	T if((k1)==k1);
	F if(!(k1)==k1);
	Fa if(
	tb(k1)&&
	fc(k0));
	Fa if(
	tb(k1)&&
	fc(k0));
	Fa if(
	fb k1==2&&(
	Fa if(
	fb k1==2&&(
	Fa if(
	tb k1==1&&(
	Fa if(
	tb k1==1&&(
	Ta if(
	tb k1==1&&(
	Fa if((
	fb k1==2&&
	Fa if((
	tb k1==1&&
	Fa if((
	tb k1==1&&
	Ta if((
	tb k1==1&&
	Fa if(
	tb k1==1&&
	Fa for(;
	tb k1&&
	fc k0;);
	Fa for(;
	fb k0&&
	.c k1;);
	Fa for(;(
	tb k1&&
	fc k0););
	Fa for(;(
	fb k0&&
	.c k1););
	Fa for(;
	tb(k1)&&
	fc(k0););
	Fa for(;
	fb(k0)&&
	.c(k1););
	Fa if(
	tb f1(1)==f1(1)&&f1(
	tc k1&&
	fd k0
	Fa if(
	fb f1(1)!=f1(1)&&f1(
	.c k1&&
	.d k0
	Fa if(
	tb f1(1)==f1(1)&&f1(
	fc k0&&
	.d k1
	Fa if(
	tb f1(1)==f1(1)&&f1(
	fc k0||
	td k1
	Ta if(e=1);
	Fb if(k0);
	T if(e=2);
	Ta if(e=
	tc k1);
	Fa if(e=
	tb k1&&
	fc k0);
	Fd if(e);
	Fa if(e=(
	tb k1&&
	fc k0));
	Fd if(e);
	Fa if(e|=
	tb k1&&
	fc k0);
	Fd if(e);
	Fa if(e&=
	tb k1&&
	fc k0);
	Fd if(e);
	F if(k1,k0);
	T if(k0,k1);
	Fa if(k1,
	fb k0&&
	.c k0);
	F if(f2(1,0));
	F if(f2(0,1));
	Fa if((
	fb k0||
	tc k1),(
	fd k0||
	fe k0));
	Fa if(
	fb k0||
	tc k1,
	fd k0||
	fe k0);
	Fa if(f2(
	fb k0||
	tc k1,0));
	Fa if(f2(
	fb k0||
	tc k1,1));
	Fa if(f2(0,
	tb k1&&
	fc k0));
	Fa if(f2(
	fb k0||
	tc k1,
	fd k0||
	fe k0));
	Fa if(
	fb k0&&
	.c k0,k0)
	.d if(k0);
	Fa if(
	tb k1&&
	fc k0,k0)
	.d if(k0);
	Fa if(
	fb k0&&
	.c k1,k0)
	.d if(k0);
	Fa if(
	tb k1&&
	tc k1,k0)
	.d if(k0);
	Ta if(
	tb k1&&
	tc k1,k1)
	Fd if(k0);
	Ta if(
	fb k0&&
	.c k0,k1)
	Fd if(k0);
	Ta if(
	tb k1&&
	fc k0,k1)
	Fd if(k0);
	Ta if(
	fb k0&&
	.c k1,k1)
	Fd if(k0);
	Fa if(
	tb k1?k0 : k1);
	Ta if((
	fb k0&&
	.c k1)?k0 : k1);
	Ta if(
	fb k0&&
	.c k1?k0 : k1);
	Fa if(
	fb k0&&(
	.c k1?k0 : k1));
	Fa if(
	tb k1?
	fc k0&&
	.d k1 : k1);
	Fa if(
	tb k1?
	tc k1&&
	fd k0 : k1);
	Fa if(
	tb k1&&("s"[
	. f3(int(*fp)())
	. f4(arg)
	. f5(arg1,arg2,arg3)
	. static void f7(void){}
	. void *function6(int i){return 0
	. void(function8)(int i){}
	. void *(function9)(int i){return
	. function11(p)
	.()
	. void catch(){}
	. void switch_select()
	. case 2:;
	.}
	. case 1:;
	. case 1:;
	. void f8()
	. if(
	.a a&&
	.b a)
.
#29
run 'echo >t_retain.c "#if part1"
  echo >>t_retain.c "int main() { return 0; }"
  echo >>t_retain.c "#endif"
  echo >>t_retain.c "#if part2"
  echo >>t_retain.c "int main() { ; return 0; }"
  echo >>t_retain.c "#endif"
  covc -q --retain $cc -Dpart1=1 ${o}t_retai1 t_retain.c &&
  covc -q --retain $cc -Dpart2=1 ${o}t_retai2 t_retain.c &&
  t_retai1 && t_retai2; br t_retain.c; covmgr -q -r t_retain.c'<<.
	X int main(){return 0;}
	X int main(){;return 0;}
.
#30
run -n '
  f1=$TMPDIR/t1; f2=$TMPDIR/t2
	covc -q $cc -c -Dpart1=1 -Fo$f1.obj t_retain.c &&
	covc -q $cc $o$f1 $f1.obj &&
	covc -q $cc -c -Dpart2=1 -Fo$f2.obj t_retain.c &&
	covc -q $cc $o$f2 $f2.obj &&
  $f1 && $f2; br t_retain.c; covmgr -q -r t_retain.c
'<<.
	X int main(){return 0;}
	X int main(){;return 0;}
.
#31
run -u '
  f1=$TMPDIR/t1; f2=$TMPDIR/t2
  covc -q $cc -c -Dpart1=1 -o $f1.o t_retain.c && covc -q $cc -o $f1 $f1.o &&
  covc -q $cc -c -Dpart2=1 -o $f2.o t_retain.c && covc -q $cc -o $f2 $f2.o &&
  $f1 && $f2; br t_retain.c; covmgr -q -r t_retain.c'<<.
	X int main(){return 0;}
	X int main(){;return 0;}
.
#32
run 'covc -v $cc ${o}t_setjmp t_setjmp.c >$t1 2>&1
	grep "91[45]" $t1 |
		sed \
			-e "s, [_]*setjmp, setjmp," \
			-e "s, [_]*sigsetjmp, setjmp,"
	./t_setjmp
	br t_setjmp.c
  rm -f t_setjmp.exe; covmgr -q -r t_setjmp.c'<<.
	t_setjmp.c(9): note cov914: avoided call to setjmp
	t_setjmp.c(12): note cov914: avoided call to setjmp
	t_setjmp.c(18): note cov914: avoided call to setjmp
	t_setjmp.c(28): note cov914: avoided call to setjmp
	first
	second
	. void f()
	. if(c);
	. if(c);
	. if(c);
	. if(c);
	X int main()
	F if(c);
	F if(c);
.
#33
run -n 'echo >t_afxAssert.c "int AfxAssertFailedLine(char*s, int l){return 0;}";
  echo >>t_afxAssert.c "int main() {";
  echo >>t_afxAssert.c " int c = 0;";
  echo >>t_afxAssert.c " if (c) ;";
  echo >>t_afxAssert.c " if (0 && AfxAssertFailedLine(__FILE__, __LINE__)) ;";
  echo >>t_afxAssert.c " if (c) ;";
  echo >>t_afxAssert.c " return 0; }";
  covc -v  cl t_afxAssert.c >$t1 2>&1 && grep "91[45]" $t1 && br t_afxAssert.c &&
	covmgr -q -r t_afxAssert.c
	rm t_afxAssert*
'<<.
	t_afxAssert.c(5): note cov914: avoided call to AfxAssertFailedLine
	. int AfxAssertFailedLine(char*s,in
	. int main(){
	. if(c);
	. if(c);
.
#34
run 'covc -q $cc ${o}t_term t_term.c && t_term && sleep 1 &&
	t_term exit1
	t_term exit2
	t_term exit3
	sh -c "t_term abort1" >$t1 2>&1
	sh -c "t_term abort2" >$t1 2>&1
	br t_term.c; covmgr -r -q t_term.c'<<.
	X int main(int argc,char * * argv)
	x if(argc<=1)s
	x if(strcmp(argv[1],"1")==0)s
	x if(strcmp(argv[1],"2")==0)s
	x if(strcmp(argv[1],"3")==0){
	x if(strcmp(argv[1],"4")==0){
	x if(strcmp(argv[1],"5")==0){
	x if(strcmp(argv[1],"6")==0){
	x if(strcmp(argv[1],"abort1")==
	x if(strcmp(argv[1],"abort2")==
	x if(strcmp(argv[1],"exit1")==
	x if(strcmp(argv[1],"exit2")==
	T if(strcmp(argv[1],"exit3")==
.
#35
run -n 'covc -q cl -nologo -DSYS_NT=1 t_term.c covntm.lib && t_term Exit &&
		covbr -q t_term.c | grep ExitProcess; covmgr -r -q t_term.c'<<.
	-->T     37          if (strcmp(argv[1], "Exit") == 0) ExitProcess(0);
.
#36
run 'covc -q $cc -c t_token.c && br t_token.c && covfn -c -q t_token.c;
	covmgr -r -q t_token.c'<<.
	. int main(int argc,char * argv[])
	. for(i=0;i<=10;i++)
	. while((c=fgetc(stdin))!=EOF)
	.a case '0':
	.b case '1':
	.c case '2':
	.d ca
	.e
	.a case '5':
	.b case '6':
	.c case '7':
	.d ca
	.e
	. case ' ':
	. case '/n':
	. case '/t':
	. default:
	. for(i=0;i<10;i++)
	. if(strcmp("/""," ")==0)
	. if(strcmp("a/"b"," ")==0)
	. if(strcmp("//"," ")==0)
	. if(strcmp("///""," ")==0)
	. if(strcmp("/"//"," ")==0)
	. if(ptr)
	. while(ptr)
	. for(;ptr;)
	.a case '0':
	.b case '1':
	.c case '2':
	.d ca
	.e
	. case ' ':
	. case 1: i=i;break;
	. case 2: break;
	.}
	. default:
	. static void a123456789012345678901
	. static void b123456789012345678901
	. void function12(operator){}
	. void operator(){}
	. void function13(){
	. void f14(){}
	. void f15()/
	"Function","Source","Line","FnCov","C/D Coverage","out of","%"
	"main","t_token.c",11,0,0,46,0%
	"a123456789012345678901234567890","t_token.c",214,0,0,0,""
	"b1234567890123456789012345678901","t_token.c",215,0,0,0,""
	"function12","t_token.c",218,0,0,0,""
	"operator","t_token.c",219,0,0,0,""
	"function13","t_token.c",220,0,0,0,""
	"f14","t_token.c",237,0,0,0,""
	"f15","t_token.c",240,0,0,0,""
	"Total","","",0.00,0,46,0%
.
#37
run -n '
	covc -p- -q cl -nologo -c t_token2.cpp && br t_token2.cpp &&
		covfn -c -q t_token2.cpp; covmgr -r -q t_token2.cpp'<<.
	. static void a012345678901234567890
	. static void b012345678901234567890
	. void f1(void){
	"Function","Source","Line","FnCov","C/D Coverage","out of","%"
	"a012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789","t_token2.cpp",14,0,0,0,""
	"b012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789","t_token2.cpp",15,0,0,0,""
	"f1","t_token2.cpp",17,0,0,0,""
	"Total","","",0.00,0,0,""
.
#38
run -n '
	covc -q --comment cl -nologo -c -Tp t_token3.c && br t_token3.c
	covc -q --comment bcc32 -c -P t_token3.c && br t_token3.c
'<<.
	. void f1(){}
	. void f3(){}
	. void f1(){}
	. void f3(){}
.
#39
run -n '
	covc -q cl -nologo -c t_token3.c && br t_token3.c
'<<.
	. void f1(){}
	. void f2(){}
	. void f3(){}
	. void class(){}
.
#40
run -n 'covc -q bcc32 -c -P t_token3.c && br t_token3.c'<<.
	. void f1(){}
	. void f2(){}
	. void f3(){}
.
#41
run -u '
	covc -q --comment $cc -c -x c++ t_token3.c && br t_token3.c &&
	covc -q $cc -c t_token3.c && br t_token3.c
'<<.
	. void f1(){}
	. void f3(){}
	. void f1(){}
	. void f2(){}
	. void f3(){}
	. void class(){}
.
#42
run 'covmgr -q -r t_token3.c'<<.
.
#43
run -n -u '
	yacc t_yacc.y && mv y*tab.c t_yacc.c &&
  covc -q -tmain,yyerror,yylex $cc ${o}t_yacc t_yacc.c &&
	t_yacc
	covbr -q -c0 -e1 t_yacc.y:main yyerror yylex
	covmgr -r -q t_yacc.y;
  rm t_yacc.c'<<.
	action 1
	-->       9  void yyerror(const char * string)
	X        14  int yylex(void)
	TF       18   if (first) {
	X        47  int main()
.
#44
run -n '
	covc -q $cc t_excpt1.cpp && t_excpt1 && br t_excpt1.cpp
	covfn -q -c -k -s t_excpt1.cpp
	covmgr -r -q t_excpt1.cpp
'<<.
	C++ try
	C++ catch
	X void f(void)throw(int)
	X void g()
	. catch(int e){}
	.a catch(long e){}
	.b catch(...){
	X void h()
	. catch(int i){
	. if(i==0){
	X void i()
	. case 1:
	. case 2:
	X}
	X int main(void)
	X catch(...){
	"f","t_excpt1.cpp",3,1,0,0,""
	"g","t_excpt1.cpp",7,1,0,3,0%
	"h","t_excpt1.cpp",16,1,0,3,0%
	"i","t_excpt1.cpp",30,1,1,3,33%
	"main","t_excpt1.cpp",41,1,1,1,100%
.
#45
run -n 'covc -q cl -nologo t_excpt2.c covntm.lib && t_excpt2 &&
		br t_excpt2.c; covmgr -r -q t_excpt2.c'<<.
	__try (for __except outer)
	__try (for __except inner)
	__except inner
	__except outer
	__try (for __finally)
	__finally
	X void seh1(void)
	X __except(GetExceptionCode()==
	X __except(GetExceptionCode()==0x
	X void seh2(void)
	X __finally{
	X int main(void)
.
#46
run -n '
    cl -nologo -DSYS_NT=1 t_pthrea.c
	t_pthrea -c
'<<.
.
#47
run 'covc -n -q $cc -c calc1.c && br calc1.c'<<.
	. static void push(int data)
	. static int pop(Pvoid)
	. static int Input(Pvoid)
	. int main()
.
#48
run '# covc option -x
	cp calc1.c calc1.abc; cp calc2.c calc2.d
	covc -xabc,.d,!c -p- -v $cc -c -I. calc1.abc calc1.c calc2.d >$t1 2>&1;
		grep cov902: $t1; rm calc1.abc calc2.d
		covmgr -r -q calc1.abc calc2.d'<<.
	note cov902: running %cc -c -I. tmp/covc/calc1.abc calc1.c
	note cov902: running %cc -c -I. calc1.c tmp/covc/calc2.d
.
#49
run '
	unset TMP
	unset TMPDIR
	covc -v $cc -c calc1.c 2>&1 |
		sed "s,[cC]:.tmp,/tmp," |
		sed "s,covc[0-9]*,covc," |
		grep "cov902.*covc.calc1.c"
'<<.
	note cov902: running %cc -c /tmp/covc/calc1.c
.
#50
run '
	if [ ! -d $TMP/covc ]
	then
		mkdir $TMP/covc
	fi
	>$TMP/covc/calc1.c; chmod 000 $TMP/covc/calc1.c;
	covc -lv $cc -c calc1.c 2>&1 | grep cov125
	rm -f $TMP/covc/calc1.c
'<<.
	error cov125: cannot open tmp/covc/calc1.c: Permission denied
.
#51
run 'covc -p- -q $cc -c no_file.c'<<.
	error cov125: cannot open no_file.c: No such file or directory
	status 1
.
#52
run 'echo int a\; >t_empty.c
	covc -tnone -v $cc -c t_empty.c 2>&1 |
		grep cov815
'<<.
	warning cov815: function "none" not seen
.
#53
run 'rm -f *.i $TMPDIR/*.i;
  covc -q -pecho $cc -c calc1.c || covc -q no-file.c || covc -q covc'<<.
	error cov114: preprocessor failed echo
	error cov110: no compiler specified
	error cov119: recursive use of covc
	status 1
.
#54
run -n '
	cl
	echo $?
	covc -q cl
	echo $?
'<<.
	usage: cl [ option... ] filename... [ /link linkoption... ]
	0
	usage: cl [ option... ] filename... [ /link linkoption... ]
	0
.
#55
run -n '
	$c2 -c calc3.cpp && covc -q $c2 calc3.obj calc3stk.cpp
'<<.
.
#56
run 'covc -q $cc ${o}calc1 calc1.c && echo "1=" | calc1'<<.
	Enter reverse-polish expressions, q to quit
	1
.
#57
run '
	covc -q -g $cc ${o}calc1 calc1.c && echo "1=" | calc1
'<<.
	Enter reverse-polish expressions, q to quit
	1
.
#58
run '
	covc -v $cc ${o}calc2 calc2.c 2>&1 | grep 90[49]
	echo 1+2 | calc2
'<<.
	note cov904: processing source calc2.c
	note cov909: exit status 0
	Enter infix expressions, q to quit
	3
.
#59
run '
	covc -v $c2 ${o}calc3 calc3.cpp calc3stk.cpp 2>&1 | grep "cov90[49]"
	echo "1=" | ./calc3
'<<.
	note cov904: processing source calc3stk.h
	note cov904: processing source calc3.cpp
	note cov904: processing source calc3stk.h
	note cov904: processing source calc3stk.cpp
	note cov909: exit status 0
	Enter reverse-polish expressions, q to quit
	1
.
#60
run 'covbr calc1.c'<<.
	C-Cover Coverage by Branch %ver
	  ...
	          8
	          9  #if __STDC__ || __cplusplus || _MSC_VER || __BORLANDC__ || __IBMC_
	         10  #define Pvoid void
	X        11  static void push(int data)
	         12  #else
	         13  #define Pvoid
	         14  static void push(data)
	  ...
	         18          stack[top++] = data;
	         19  }
	         20
	X        21  static int pop(Pvoid)
	         22  {
	         23          return (top > 0) ? stack[--top] : 0;
	         24  }
	         25
	X        26  static int Input(Pvoid)
	         27  {
	         28          return getchar();
	         29  }
	         30
	X        31  int main()
	         32  {
	         33          int c = 0;
	         34          int temp;
	  ...
	         36
	         37          printf("Enter reverse-polish expressions, q to quit/n");
	         38          quit = 0;
	TF       39a         while (
	  -->t   39b                !quit &&
	  tf     40                  (c = Input()) != EOF)
	         41          {
	         42                  switch (c) {
	         43
	-->      44                  case ' ':
	X        45                  case '/n':
	         46                          break;
	         47
	-->      48a                 case '0':
	X        48b                           case '1':
	-->      48c                                     case '2':
	-->      48d                                               case '3':
	-->      48e                                                         case '4':
	-->      49a                 case '5':
	-->      49b                           case '6':
	-->      49c                                     case '7':
	-->      49d                                               case '8':
	-->      49e                                                         case '9':
	         50                          temp = 0;
	TF       51                          for (; isdigit(c); c = Input()) {
	         52                                  temp = temp * 10 + (c - '0');
	         53                          }
	         54                          ungetc(c, stdin);
	         55                          push(temp);
	         56                          break;
	         57
	-->      58                  case '+': push(pop() + pop()); break;
	         59
	-->      60                  case '*': push(pop() * pop()); break;
	         61
	-->      62                  case '-': temp = pop(); push(pop() - temp); break;
	         63
	-->      64                  case '/':
	         65                          temp = pop();
	-->      66                          if (temp == 0)
	         67                                  printf("division by zero/n");
	         68                          else
	         69                                  push(pop() / temp);
	         70                          break;
	         71
	X        72                  case '=':
	         73                          temp = pop();
	         74                          push(temp);
	         75                          printf("%d/n", temp);
	         76                          break;
	         77
	-->      78                  case 's':
	-->      79                          if (top != 0) {
	         80                                  temp = 0;
	         81                                  do {
	         82                                          printf("%d ", stack[temp])
	         83                                          temp++;
	-->      84                                  } while (temp < top);
	         85                                  printf("/n");
	         86                          }
	         87                          break;
	         88
	-->      89                  case 'q':
	         90                          quit = 1;
	         91                          break;
	         92
	-->      93                  default:
	         94                          printf("unrecognized: %c/n", c);
	         95                          break;
	         96                  }
	  ...
.
#61
run 'covbr -h -q'<<.
	Usage: covbr [options] [region ...]
	-a     --all         Show all source lines
	-cN    --context N   Show N source lines before or after a probe [N=3]
	-dDir  --srcdir Dir  Source file directory is Dir
	-eN    --tab N       Expand tabs to every N column [N=8]
	-fPath --file Path   Coverage data file is Path
	-h     --help        Display this usage summary
	-i     --no-time     Ignore source file date/time
	-q     --quiet       Quiet
	-r     --wrap        Wrap lines rather than truncate
	-t     --trunc       Truncate lines rather than wrap [default]
	-u     --uncover     Suppress fully covered probes (with -c)
	-v     --verbose     Verbose
	-wN    --width N     Report width is N columns [N=79]
	--csv                Output comma separated values
	--html               Output in HTML format
	--no-banner          Suppress startup banner
	status 1
.
#62
run 'covbr -r -u -q -w55 calc1.c'<<.
	  ...
	             ssions, q to quit/n");
	         38          quit = 0;
	TF       39a         while (
	  -->t   39b                !quit &&
	  tf     40                  (c = Input()) != EOF)
	         41          {
	         42                  switch (c) {
	         43
	-->      44                  case ' ':
	X        45                  case '/n':
	         46                          break;
	         47
	-->      48a                 case '0':
	X        48b                           case '1':
	-->      48c                                     case '
	             2':
	-->      48d
	                 case '3':
	-->      48e
	                           case '4':
	-->      49a                 case '5':
	-->      49b                           case '6':
	-->      49c                                     case '
	             7':
	-->      49d
	                 case '8':
	-->      49e
	                           case '9':
	         50                          temp = 0;
	TF       51                          for (; isdigit(c);
	  ...
	         55                          push(temp);
	         56                          break;
	         57
	-->      58                  case '+': push(pop() + pop
	             ()); break;
	         59
	-->      60                  case '*': push(pop() * pop
	             ()); break;
	         61
	-->      62                  case '-': temp = pop(); pu
	             sh(pop() - temp); break;
	         63
	-->      64                  case '/':
	         65                          temp = pop();
	-->      66                          if (temp == 0)
	         67                                  printf("di
	             vision by zero/n");
	         68                          else
	  ...
	             p);
	         76                          break;
	         77
	-->      78                  case 's':
	-->      79                          if (top != 0) {
	         80                                  temp = 0;
	         81                                  do {
	         82                                          pr
	             intf("%d ", stack[temp]);
	         83                                          te
	             mp++;
	-->      84                                  } while (t
	             emp < top);
	         85                                  printf("/n
	             ");
	         86                          }
	         87                          break;
	         88
	-->      89                  case 'q':
	         90                          quit = 1;
	         91                          break;
	         92
	-->      93                  default:
	         94                          printf("unrecogniz
	             ed: %c/n", c);
	         95                          break;
	  ...
.
#63
run 'COLUMNS=56; export COLUMNS; covbr -r -u -q calc1.c; unset COLUMNS'<<.
	  ...
	             ssions, q to quit/n");
	         38          quit = 0;
	TF       39a         while (
	  -->t   39b                !quit &&
	  tf     40                  (c = Input()) != EOF)
	         41          {
	         42                  switch (c) {
	         43
	-->      44                  case ' ':
	X        45                  case '/n':
	         46                          break;
	         47
	-->      48a                 case '0':
	X        48b                           case '1':
	-->      48c                                     case '
	             2':
	-->      48d
	                 case '3':
	-->      48e
	                           case '4':
	-->      49a                 case '5':
	-->      49b                           case '6':
	-->      49c                                     case '
	             7':
	-->      49d
	                 case '8':
	-->      49e
	                           case '9':
	         50                          temp = 0;
	TF       51                          for (; isdigit(c);
	  ...
	         55                          push(temp);
	         56                          break;
	         57
	-->      58                  case '+': push(pop() + pop
	             ()); break;
	         59
	-->      60                  case '*': push(pop() * pop
	             ()); break;
	         61
	-->      62                  case '-': temp = pop(); pu
	             sh(pop() - temp); break;
	         63
	-->      64                  case '/':
	         65                          temp = pop();
	-->      66                          if (temp == 0)
	         67                                  printf("di
	             vision by zero/n");
	         68                          else
	  ...
	             p);
	         76                          break;
	         77
	-->      78                  case 's':
	-->      79                          if (top != 0) {
	         80                                  temp = 0;
	         81                                  do {
	         82                                          pr
	             intf("%d ", stack[temp]);
	         83                                          te
	             mp++;
	-->      84                                  } while (t
	             emp < top);
	         85                                  printf("/n
	             ");
	         86                          }
	         87                          break;
	         88
	-->      89                  case 'q':
	         90                          quit = 1;
	         91                          break;
	         92
	-->      93                  default:
	         94                          printf("unrecogniz
	             ed: %c/n", c);
	         95                          break;
	  ...
.
#64
run 'covbr -q --csv calc1.c calc2.c'<<.
	calc1.c,11,,function,X,push
	calc1.c,21,,function,X,pop
	calc1.c,26,,function,X,Input
	calc1.c,31,,function,X,main
	calc1.c,39,a,decision,TF,
	calc1.c,39,b,condition,t,
	calc1.c,40,,condition,tf,
	calc1.c,44,,switch-label,,
	calc1.c,45,,switch-label,X,
	calc1.c,48,a,switch-label,,
	calc1.c,48,b,switch-label,X,
	calc1.c,48,c,switch-label,,
	calc1.c,48,d,switch-label,,
	calc1.c,48,e,switch-label,,
	calc1.c,49,a,switch-label,,
	calc1.c,49,b,switch-label,,
	calc1.c,49,c,switch-label,,
	calc1.c,49,d,switch-label,,
	calc1.c,49,e,switch-label,,
	calc1.c,51,,decision,TF,
	calc1.c,58,,switch-label,,
	calc1.c,60,,switch-label,,
	calc1.c,62,,switch-label,,
	calc1.c,64,,switch-label,,
	calc1.c,66,,decision,,
	calc1.c,72,,switch-label,X,
	calc1.c,78,,switch-label,,
	calc1.c,79,,decision,,
	calc1.c,84,,decision,,
	calc1.c,89,,switch-label,,
	calc1.c,93,,switch-label,,
	calc2.c,54,,function,X,lex
	calc2.c,57,,decision,TF,
	calc2.c,66,a,switch-label,,
	calc2.c,66,b,switch-label,X,
	calc2.c,66,c,switch-label,X,
	calc2.c,66,d,switch-label,,
	calc2.c,66,e,switch-label,,
	calc2.c,67,a,switch-label,,
	calc2.c,67,b,switch-label,,
	calc2.c,67,c,switch-label,,
	calc2.c,67,d,switch-label,,
	calc2.c,67,e,switch-label,,
	calc2.c,72,,switch-label,,
	calc2.c,74,,decision,,
	calc2.c,75,,decision,,
	calc2.c,77,,switch-label,,
	calc2.c,79,,decision,,
	calc2.c,80,,decision,,
	calc2.c,83,a,switch-label,,
	calc2.c,83,b,decision,,
	calc2.c,84,a,switch-label,,
	calc2.c,84,b,decision,,
	calc2.c,85,a,switch-label,,
	calc2.c,85,b,decision,,
	calc2.c,86,a,switch-label,,
	calc2.c,86,b,decision,,
	calc2.c,88,,switch-label,X,
	calc2.c,92,,switch-label,X,
	calc2.c,99,,function,,name
	calc2.c,105,,switch-label,,
	calc2.c,108,,switch-label,,
	calc2.c,109,,switch-label,,
	calc2.c,110,,switch-label,,
	calc2.c,111,,switch-label,,
	calc2.c,112,,switch-label,,
	calc2.c,113,,switch-label,,
	calc2.c,114,,switch-label,,
	calc2.c,115,,switch-label,,
	calc2.c,116,,switch-label,,
	calc2.c,119,,switch-label,,
	calc2.c,128,,function,,match
	calc2.c,130,,decision,,
	calc2.c,131,,decision,,
	calc2.c,143,,function,,z
	calc2.c,145,,decision,,
	calc2.c,146,,decision,,
	calc2.c,158,,function,X,prs
	calc2.c,166,,switch-label,X,
	calc2.c,172,,switch-label,,
	calc2.c,179,,switch-label,,
	calc2.c,180,,switch-label,,
	calc2.c,181,,switch-label,,
	calc2.c,182,,switch-label,,
	calc2.c,185,,switch-label,,
	calc2.c,186,,decision,,
	calc2.c,194,,decision,T,
	calc2.c,199,a,decision,F,
	calc2.c,199,b,condition,t,
	calc2.c,199,c,condition,f,
	calc2.c,200,a,decision,F,
	calc2.c,200,b,condition,t,
	calc2.c,200,c,condition,f,
	calc2.c,201,a,decision,F,
	calc2.c,201,b,condition,t,
	calc2.c,201,c,condition,f,
	calc2.c,202,a,decision,TF,
	calc2.c,202,b,condition,tf,
	calc2.c,202,c,condition,tf,
	calc2.c,203,a,decision,F,
	calc2.c,203,b,condition,tf,
	calc2.c,203,c,condition,f,
	calc2.c,204,a,decision,F,
	calc2.c,204,b,condition,tf,
	calc2.c,204,c,condition,f,
	calc2.c,205,a,decision,F,
	calc2.c,205,b,condition,tf,
	calc2.c,205,c,condition,f,
	calc2.c,206,a,decision,F,
	calc2.c,206,b,condition,tf,
	calc2.c,206,c,condition,f,
	calc2.c,207,a,decision,F,
	calc2.c,207,b,condition,tf,
	calc2.c,207,c,condition,f,
	calc2.c,208,a,decision,F,
	calc2.c,208,b,condition,tf,
	calc2.c,208,c,condition,f,
	calc2.c,209,a,decision,F,
	calc2.c,209,b,condition,tf,
	calc2.c,209,c,condition,f,
	calc2.c,210,a,decision,F,
	calc2.c,210,b,condition,tf,
	calc2.c,210,c,condition,f,
	calc2.c,211,a,decision,F,
	calc2.c,211,b,condition,tf,
	calc2.c,211,c,condition,f,
	calc2.c,212,a,decision,F,
	calc2.c,212,b,condition,tf,
	calc2.c,212,c,condition,f,
	calc2.c,213,a,decision,F,
	calc2.c,213,b,condition,tf,
	calc2.c,213,c,condition,f,
	calc2.c,214,a,decision,F,
	calc2.c,214,b,condition,tf,
	calc2.c,214,c,condition,f,
	calc2.c,217,a,decision,F,
	calc2.c,217,b,condition,tf,
	calc2.c,218,,condition,f,
	calc2.c,224,a,decision,F,
	calc2.c,224,b,condition,tf,
	calc2.c,225,,condition,f,
	calc2.c,231,a,decision,F,
	calc2.c,231,b,condition,tf,
	calc2.c,232,,condition,f,
	calc2.c,252,,function,X,eval
	calc2.c,261,a,decision,F,
	calc2.c,261,b,condition,t,
	calc2.c,262,,condition,f,
	calc2.c,271,,function,X,main
	calc2.c,277,,decision,TF,
	calc2.c,280,,decision,F,
	calc2.c,283,a,decision,TF,
	calc2.c,283,b,condition,tf,
	calc2.c,284,,condition,t,
.
#65
run -n '
	covbr -q no-file.c || cp calc1.c t_br.c; chmod +w t_br.c
  touch t_br.c;
  covc -q $cc -c t_br.c && (sleep 2; touch t_br.c; covbr -q t_br.c) ||
		(touch -t 199501010000 t_br.c; covbr -q t_br.c) ||
		covbr -i -c0 -q t_br.c:pop && rm t_br.c;
		covbr -q t_br.c || covmgr -r t_br.c
'<<.
	error cov139: no coverage information for source no-file.c.  Specify regions as they are shown, without -b/--base
	error cov141: source modified: t_br.c, rebuild t_br.obj
	error cov142: date/time mismatch: t_br.c, rebuild t_br.obj
	-->      21  static int pop(Pvoid)
	error cov124: cannot find t_br.c
	C-Cover Manager %ver
.
#66
run 'covbr -e0 -w0 -c0 -q calc1.c:pop'<<.
	X        21  static int pop(Pvoid)
.
#67
run '# covbr -u shows nothing when there are no uncovered probes
	echo "" >t_full.c
	echo "" >>t_full.c
	echo "int main() { return 0; }" >>t_full.c
	echo "" >>t_full.c
	echo "" >>t_full.c
	covc -q $cc ${o}t_full t_full.c
	./t_full
	covbr -u -v t_full.c | sed "s/^/:/"
	covmgr -r -q t_full.c
	rm t_full*
'<<.
	C-Cover Coverage by Branch %ver
	note cov911: data file test.cov
	:t_full.c:
	:  Fully covered
.
#68
run '# covbr --html
	covbr -q --html calc1.c
'<<.
	<HTML><HEAD><TITLE>covbr</TITLE></HEAD><BODY><PRE>
	  ...
	       <FONT COLOR=GRAY>   8 </FONT>
	       <FONT COLOR=GRAY>   9 </FONT> #if __STDC__ || __cplusplus || _MSC_VER || __BORLANDC__ || __IBMC_
	       <FONT COLOR=GRAY>  10 </FONT> #define Pvoid void
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  11 </FONT> static void push(int data)
	       <FONT COLOR=GRAY>  12 </FONT> #else
	       <FONT COLOR=GRAY>  13 </FONT> #define Pvoid
	       <FONT COLOR=GRAY>  14 </FONT> static void push(data)
	  ...
	       <FONT COLOR=GRAY>  18 </FONT>         stack[top++] = data;
	       <FONT COLOR=GRAY>  19 </FONT> }
	       <FONT COLOR=GRAY>  20 </FONT>
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  21 </FONT> static int pop(Pvoid)
	       <FONT COLOR=GRAY>  22 </FONT> {
	       <FONT COLOR=GRAY>  23 </FONT>         return (top &gt; 0) ? stack[--top] : 0;
	       <FONT COLOR=GRAY>  24 </FONT> }
	       <FONT COLOR=GRAY>  25 </FONT>
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  26 </FONT> static int Input(Pvoid)
	       <FONT COLOR=GRAY>  27 </FONT> {
	       <FONT COLOR=GRAY>  28 </FONT>         return getchar();
	       <FONT COLOR=GRAY>  29 </FONT> }
	       <FONT COLOR=GRAY>  30 </FONT>
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  31 </FONT> int main()
	       <FONT COLOR=GRAY>  32 </FONT> {
	       <FONT COLOR=GRAY>  33 </FONT>         int c = 0;
	       <FONT COLOR=GRAY>  34 </FONT>         int temp;
	  ...
	       <FONT COLOR=GRAY>  36 </FONT>
	       <FONT COLOR=GRAY>  37 </FONT>         printf(&quot;Enter reverse-polish expressions, q to quit/n&quot;);
	       <FONT COLOR=GRAY>  38 </FONT>         quit = 0;
	<FONT COLOR=BLUE>TF</FONT>     <FONT COLOR=GRAY>  39a</FONT>         while (
	  <FONT COLOR=RED>--&gt;</FONT><FONT COLOR=BLUE>t</FONT> <FONT COLOR=GRAY>  39b</FONT>                !quit &amp;&amp;
	  <FONT COLOR=BLUE>tf</FONT>   <FONT COLOR=GRAY>  40 </FONT>                 (c = Input()) != EOF)
	       <FONT COLOR=GRAY>  41 </FONT>         {
	       <FONT COLOR=GRAY>  42 </FONT>                 switch (c) {
	       <FONT COLOR=GRAY>  43 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  44 </FONT>                 case ' ':
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  45 </FONT>                 case '/n':
	       <FONT COLOR=GRAY>  46 </FONT>                         break;
	       <FONT COLOR=GRAY>  47 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  48a</FONT>                 case '0':
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  48b</FONT>                           case '1':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  48c</FONT>                                     case '2':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  48d</FONT>                                               case '3':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  48e</FONT>                                                         case '4':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  49a</FONT>                 case '5':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  49b</FONT>                           case '6':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  49c</FONT>                                     case '7':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  49d</FONT>                                               case '8':
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  49e</FONT>                                                         case '9':
	       <FONT COLOR=GRAY>  50 </FONT>                         temp = 0;
	<FONT COLOR=BLUE>TF</FONT>     <FONT COLOR=GRAY>  51 </FONT>                         for (; isdigit(c); c = Input()) {
	       <FONT COLOR=GRAY>  52 </FONT>                                 temp = temp * 10 + (c - '0');
	       <FONT COLOR=GRAY>  53 </FONT>                         }
	       <FONT COLOR=GRAY>  54 </FONT>                         ungetc(c, stdin);
	       <FONT COLOR=GRAY>  55 </FONT>                         push(temp);
	       <FONT COLOR=GRAY>  56 </FONT>                         break;
	       <FONT COLOR=GRAY>  57 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  58 </FONT>                 case '+': push(pop() + pop()); break;
	       <FONT COLOR=GRAY>  59 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  60 </FONT>                 case '*': push(pop() * pop()); break;
	       <FONT COLOR=GRAY>  61 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  62 </FONT>                 case '-': temp = pop(); push(pop() - temp); break;
	       <FONT COLOR=GRAY>  63 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  64 </FONT>                 case '/':
	       <FONT COLOR=GRAY>  65 </FONT>                         temp = pop();
	<FONT COLOR=RED>--&gt;</FONT><FONT COLOR=BLUE> </FONT>   <FONT COLOR=GRAY>  66 </FONT>                         if (temp == 0)
	       <FONT COLOR=GRAY>  67 </FONT>                                 printf(&quot;division by zero/n&quot;);
	       <FONT COLOR=GRAY>  68 </FONT>                         else
	       <FONT COLOR=GRAY>  69 </FONT>                                 push(pop() / temp);
	       <FONT COLOR=GRAY>  70 </FONT>                         break;
	       <FONT COLOR=GRAY>  71 </FONT>
	<FONT COLOR=BLUE>X</FONT>      <FONT COLOR=GRAY>  72 </FONT>                 case '=':
	       <FONT COLOR=GRAY>  73 </FONT>                         temp = pop();
	       <FONT COLOR=GRAY>  74 </FONT>                         push(temp);
	       <FONT COLOR=GRAY>  75 </FONT>                         printf(&quot;%d/n&quot;, temp);
	       <FONT COLOR=GRAY>  76 </FONT>                         break;
	       <FONT COLOR=GRAY>  77 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  78 </FONT>                 case 's':
	<FONT COLOR=RED>--&gt;</FONT><FONT COLOR=BLUE> </FONT>   <FONT COLOR=GRAY>  79 </FONT>                         if (top != 0) {
	       <FONT COLOR=GRAY>  80 </FONT>                                 temp = 0;
	       <FONT COLOR=GRAY>  81 </FONT>                                 do {
	       <FONT COLOR=GRAY>  82 </FONT>                                         printf(&quot;%d &quot;, stack[temp])
	       <FONT COLOR=GRAY>  83 </FONT>                                         temp++;
	<FONT COLOR=RED>--&gt;</FONT><FONT COLOR=BLUE> </FONT>   <FONT COLOR=GRAY>  84 </FONT>                                 } while (temp &lt; top);
	       <FONT COLOR=GRAY>  85 </FONT>                                 printf(&quot;/n&quot;);
	       <FONT COLOR=GRAY>  86 </FONT>                         }
	       <FONT COLOR=GRAY>  87 </FONT>                         break;
	       <FONT COLOR=GRAY>  88 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  89 </FONT>                 case 'q':
	       <FONT COLOR=GRAY>  90 </FONT>                         quit = 1;
	       <FONT COLOR=GRAY>  91 </FONT>                         break;
	       <FONT COLOR=GRAY>  92 </FONT>
	<FONT COLOR=RED>--&gt;</FONT>    <FONT COLOR=GRAY>  93 </FONT>                 default:
	       <FONT COLOR=GRAY>  94 </FONT>                         printf(&quot;unrecognized: %c/n&quot;, c);
	       <FONT COLOR=GRAY>  95 </FONT>                         break;
	       <FONT COLOR=GRAY>  96 </FONT>                 }
	  ...
	</PRE></BODY></HTML>
.
#69
run '# covclear
	COVFILE=checkTmp.cov
	export COVFILE
	covc -q $cc ${o}checkTmp calc3.cpp calc3stk.cpp
	echo 1 | checkTmp
	covdir -q -k
	covclear
	covdir -q -k
'<<.
	Enter reverse-polish expressions, q to quit
	.               6 /     6 = 100%      12 /    33 =  36%
	C-Cover Clear %ver
	.               0 /     6 =   0%       0 /    33 =   0%
.
#70
run '
	rm -f checkTmp.cov
	covmgr -q -c -l -f checkTmp.cov
	covclear -f checkTmp.cov -q source.c
'<<.
	error cov101: extra arguments
	status 1
.
#71
run 'covclass'<<.
	C-Cover Coverage by Class %ver
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class1::                                0 /     5 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#72
run 'covclass -c -q'<<.
	"Class","Function Coverage","out of","%","C/D Coverage","out of","%"
	"class2::",0,10,0%,0,2,0%
	"class3::",0,8,0%,0,0,""
	"class1::",0,5,0%,0,0,""
	"class11::class112::class1121::",0,2,0%,0,2,0%
	"class11::",0,2,0%,0,0,""
	"class11::class112::",0,2,0%,0,0,""
	"class4::",0,2,0%,0,0,""
	"class8::",0,2,0%,0,0,""
	"Class12::",0,1,0%,0,0,""
	"Class13::",0,1,0%,0,0,""
	"Class5::",0,1,0%,0,0,""
	"c1::c2::c3::c4::c5::c6::c7::c8::c9::",0,1,0%,0,0,""
	"class10::",0,1,0%,0,0,""
	"class11::class111::",0,1,0%,0,0,""
	"class6::",0,1,0%,0,0,""
	"class9::",0,1,0%,0,0,""
	"stack::",5,5,100%,3,4,75%
	"Total",5,46,10%,3,8,37%
.
#73
run 'covclass -h -q'<<.
	Usage: covclass [options] [region ...]
	-c     --csv        Output comma separated values
	-fPath --file Path  Coverage data file is Path
	-h     --help       Display this usage summary
	-k     --no-head    Suppress column headings
	-m     --by-name    Sort by class name
	-p     --by-pcnt    Sort by percent exercised [default]
	-q     --quiet      Quiet
	-r     --reverse    Reverse sort
	-u     --by-uncov   Sort by number of uncovered functions/branches
	-v     --verbose    Verbose
	-wN    --width N    Report width is N columns [N=79]
	--decision          Show decision (branch) coverage.  Not recommended
	--no-banner         Suppress startup banner
	status 1
.
#74
run 'covclass -k -q'<<.
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class1::                                0 /     5 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
.
#75
run 'covclass -m -q'<<.
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::                               0 /     2 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class1::                                0 /     5 =   0%       0 /     0
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#76
run 'covclass -p -q'<<.
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class1::                                0 /     5 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#77
run 'covclass -r -q'<<.
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	class9::                                0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class1::                                0 /     5 =   0%       0 /     0
	class3::                                0 /     8 =   0%       0 /     0
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#78
run 'covclass -u -q'<<.
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class1::                                0 /     5 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#79
run 'covclass -v'<<.
	C-Cover Coverage by Class %ver
	note cov911: data file test.cov
	Class                                  Function Coverage           C/D Coverage
	---------------------------------  ---------------------  ---------------------
	class2::                                0 /    10 =   0%       0 /     2 =   0%
	class3::                                0 /     8 =   0%       0 /     0
	class1::                                0 /     5 =   0%       0 /     0
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	class4::                                0 /     2 =   0%       0 /     0
	class8::                                0 /     2 =   0%       0 /     0
	Class12::                               0 /     1 =   0%       0 /     0
	Class13::                               0 /     1 =   0%       0 /     0
	Class5::                                0 /     1 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class10::                               0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
	class6::                                0 /     1 =   0%       0 /     0
	class9::                                0 /     1 =   0%       0 /     0
	stack::                                 5 /     5 = 100%       3 /     4 =  75%
	---------------------------------  ---------------------  ---------------------
	Total                                   5 /    46 =  10%       3 /     8 =  37%
.
#80
run 'covclass -q -k "class11*::" "*::c9::"'<<.
	class11::class112::class1121::          0 /     2 =   0%       0 /     2 =   0%
	class11::                               0 /     2 =   0%       0 /     0
	class11::class112::                     0 /     2 =   0%       0 /     0
	...::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
	class11::class111::                     0 /     1 =   0%       0 /     0
.
#81
run 'covclass -w1 -q'<<.
	Class      Function Coverage           C/D Coverage
	-----  ---------------------  ---------------------
	...::       0 /    10 =   0%       0 /     2 =   0%
	...::       0 /     8 =   0%       0 /     0
	...::       0 /     5 =   0%       0 /     0
	...::       0 /     2 =   0%       0 /     2 =   0%
	...::       0 /     2 =   0%       0 /     0
	...::       0 /     2 =   0%       0 /     0
	...::       0 /     2 =   0%       0 /     0
	...::       0 /     2 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       0 /     1 =   0%       0 /     0
	...::       5 /     5 = 100%       3 /     4 =  75%
	-----  ---------------------  ---------------------
	Total       5 /    46 =  10%       3 /     8 =  37%
.
#82
run 'covclass -w54 -q'<<.
	Class         Function Coverage           C/D Coverage
	--------  ---------------------  ---------------------
	class2::       0 /    10 =   0%       0 /     2 =   0%
	class3::       0 /     8 =   0%       0 /     0
	class1::       0 /     5 =   0%       0 /     0
	...121::       0 /     2 =   0%       0 /     2 =   0%
	...s11::       0 /     2 =   0%       0 /     0
	...112::       0 /     2 =   0%       0 /     0
	class4::       0 /     2 =   0%       0 /     0
	class8::       0 /     2 =   0%       0 /     0
	...s12::       0 /     1 =   0%       0 /     0
	...s13::       0 /     1 =   0%       0 /     0
	Class5::       0 /     1 =   0%       0 /     0
	...:c9::       0 /     1 =   0%       0 /     0
	...s10::       0 /     1 =   0%       0 /     0
	...111::       0 /     1 =   0%       0 /     0
	class6::       0 /     1 =   0%       0 /     0
	class9::       0 /     1 =   0%       0 /     0
	stack::        5 /     5 = 100%       3 /     4 =  75%
	--------  ---------------------  ---------------------
	Total          5 /    46 =  10%       3 /     8 =  37%
.
#83
run 'COLUMNS=55; export COLUMNS; covclass -q; unset COLUMNS'<<.
	Class         Function Coverage           C/D Coverage
	--------  ---------------------  ---------------------
	class2::       0 /    10 =   0%       0 /     2 =   0%
	class3::       0 /     8 =   0%       0 /     0
	class1::       0 /     5 =   0%       0 /     0
	...121::       0 /     2 =   0%       0 /     2 =   0%
	...s11::       0 /     2 =   0%       0 /     0
	...112::       0 /     2 =   0%       0 /     0
	class4::       0 /     2 =   0%       0 /     0
	class8::       0 /     2 =   0%       0 /     0
	...s12::       0 /     1 =   0%       0 /     0
	...s13::       0 /     1 =   0%       0 /     0
	Class5::       0 /     1 =   0%       0 /     0
	...:c9::       0 /     1 =   0%       0 /     0
	...s10::       0 /     1 =   0%       0 /     0
	...111::       0 /     1 =   0%       0 /     0
	class6::       0 /     1 =   0%       0 /     0
	class9::       0 /     1 =   0%       0 /     0
	stack::        5 /     5 = 100%       3 /     4 =  75%
	--------  ---------------------  ---------------------
	Total          5 /    46 =  10%       3 /     8 =  37%
.
#84
run 'covclass -w55 -q'<<.
	Class          Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	class2::        0 /    10 =   0%       0 /     2 =   0%
	class3::        0 /     8 =   0%       0 /     0
	class1::        0 /     5 =   0%       0 /     0
	...1121::       0 /     2 =   0%       0 /     2 =   0%
	class11::       0 /     2 =   0%       0 /     0
	...s112::       0 /     2 =   0%       0 /     0
	class4::        0 /     2 =   0%       0 /     0
	class8::        0 /     2 =   0%       0 /     0
	Class12::       0 /     1 =   0%       0 /     0
	Class13::       0 /     1 =   0%       0 /     0
	Class5::        0 /     1 =   0%       0 /     0
	...::c9::       0 /     1 =   0%       0 /     0
	class10::       0 /     1 =   0%       0 /     0
	...s111::       0 /     1 =   0%       0 /     0
	class6::        0 /     1 =   0%       0 /     0
	class9::        0 /     1 =   0%       0 /     0
	stack::         5 /     5 = 100%       3 /     4 =  75%
	---------  ---------------------  ---------------------
	Total           5 /    46 =  10%       3 /     8 =  37%
.
#85
run '# Maximum width
	covclass -q -w0 -k c1::*
'<<.
	c1::c2::c3::c4::c5::c6::c7::c8::c9::       0 /     1 =   0%       0 /     0
.
#86
run 'covdir .'<<.
	C-Cover Coverage by Directory %ver
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#87
run 'covdir -c -q .'<<.
	"Directory","Function Coverage","out of","%","C/D Coverage","out of","%"
	".",17,87,19%,313,734,42%
	"Total",17,87,19%,313,734,42%
.
#88
run 'covdir -h -q'<<.
	Usage: covdir [options] [region ...]
	-c     --csv         Output comma separated values
	-dDir  --srcdir Dir  Source file directory is Dir
	-fPath --file Path   Coverage data file is Path
	-h     --help        Display this usage summary
	-k     --no-head     Suppress column headings
	-m     --by-name     Sort by directory name
	-p     --by-pcnt     Sort by percent exercised [default]
	-q     --quiet       Quiet
	-r     --reverse     Reverse sort
	-u     --by-uncov    Sort by number of uncovered functions/branches
	-v     --verbose     Verbose
	-wN    --width N     Report width is N columns [N=79]
	--decision           Show decision (branch) coverage.  Not recommended
	--no-banner          Suppress startup banner
	status 1
.
#89
run 'covdir -k -q .'<<.
	.              17 /    87 =  19%     313 /   734 =  42%
.
#90
run 'covdir -m -q .'<<.
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#91
run 'covdir -p -q .'<<.
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#92
run 'covdir -r -q .'<<.
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#93
run 'covdir -u -q .'<<.
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#94
run 'covdir -v .'<<.
	C-Cover Coverage by Directory %ver
	note cov911: data file test.cov
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#95
run 'covdir -w1 -q .'<<.
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	.              17 /    87 =  19%     313 /   734 =  42%
	---------  ---------------------  ---------------------
	Total          17 /    87 =  19%     313 /   734 =  42%
.
#96
run 'covmgr -r -q t_class.cpp'<<.
.
#97
run 'covfn -! .'<<.
	C-Cover Coverage by Function %ver
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#98
run 'covfn -c -q .'<<.
	"Function","Source","Line","FnCov","C/D Coverage","out of","%"
	"name","calc2.c",99,0,0,11,0%
	"f8","t_parse.c",342,0,0,6,0%
	"match","calc2.c",128,0,0,4,0%
	"z","calc2.c",143,0,0,4,0%
	"switch_select","t_parse.c",308,0,0,4,0%
	"f3","t_parse.c",234,0,0,0,""
	"f4","t_parse.c",240,0,0,0,""
	"f5","t_parse.c",247,0,0,0,""
	"f7","t_parse.c",256,0,0,0,""
	"function6","t_parse.c",266,0,0,0,""
	"function8","t_parse.c",272,0,0,0,""
	"function9","t_parse.c",273,0,0,0,""
	"function11","t_parse.c",276,0,0,0,""
	"function14","t_parse.c",283,0,0,0,""
	"catch","t_parse.c",306,0,0,0,""
	"push","calc1.c",11,1,0,0,""
	"pop","calc1.c",21,1,0,0,""
	"Input","calc1.c",26,1,0,0,""
	"stack::selftest","calc3stk.cpp",15,1,0,0,""
	"stack::stack","calc3stk.h",8,1,0,0,""
	"stack::push","calc3stk.h",15,1,0,0,""
	"f1","t_parse.c",2,1,0,0,""
	"f2","t_parse.c",3,1,0,0,""
	"lex","calc2.c",54,1,6,36,16%
	"main","calc1.c",31,1,10,34,29%
	"main","calc3.cpp",7,1,10,29,34%
	"main","t_parse.c",4,1,196,457,42%
	"eval","calc2.c",252,1,3,6,50%
	"stack::pop","calc3stk.h",23,1,1,2,50%
	"prs","calc2.c",158,1,77,125,61%
	"main","calc2.c",271,1,8,10,80%
	"stack::~stack","calc3stk.cpp",5,1,2,2,100%
	"Total","","",0.53,313,730,42%
.
#99
run 'covfn -h -q'<<.
	Usage: covfn [options] [region ...]
	-b     --base        Show file basename rather than full path
	-c     --csv         Output comma separated values
	-dDir  --srcdir Dir  Source file directory is Dir
	-fPath --file Path   Coverage data file is Path
	-h     --help        Display this usage summary
	-k     --no-head     Suppress column headings
	-m     --by-name     Sort by function name
	-n     --no-source   Suppress source file and line number
	-p     --by-pcnt     Sort by percent exercised [default]
	-q     --quiet       Quiet
	-r     --reverse     Reverse sort
	-s     --by-source   Sort by source filename and line
	-u     --by-uncov    Sort by number of uncovered branches
	-v     --verbose     Verbose
	-wN    --width N     Report width is N columns [N=79]
	--decision           Show decision (branch) coverage.  Not recommended
	--no-banner          Suppress startup banner
	status 1
.
#100
run 'covfn -k -q .'<<.
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
.
#101
run 'covfn -m -q .'<<.
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	Input            calc1.c         26  1 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	pop              calc1.c         21  1 / 1       0 /     0
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	push             calc1.c         11  1 / 1       0 /     0
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#102
run 'covfn -p -q .'<<.
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#103
run 'covfn -r -q .'<<.
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	f2               t_parse.c        3  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	push             calc1.c         11  1 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f3               t_parse.c      234  0 / 1       0 /     0
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#104
run 'covfn -s -q .'<<.
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#105
run 'covfn -u -q .'<<.
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#106
run 'covfn -v .'<<.
	C-Cover Coverage by Function %ver
	note cov911: data file test.cov
	Function         Source        Line  FnCov           C/D Coverage
	---------------  ------------------  -----  ---------------------
	name             calc2.c         99  0 / 1       0 /    11 =   0%
	f8               t_parse.c      342  0 / 1       0 /     6 =   0%
	match            calc2.c        128  0 / 1       0 /     4 =   0%
	z                calc2.c        143  0 / 1       0 /     4 =   0%
	switch_select    t_parse.c      308  0 / 1       0 /     4 =   0%
	f3               t_parse.c      234  0 / 1       0 /     0
	f4               t_parse.c      240  0 / 1       0 /     0
	f5               t_parse.c      247  0 / 1       0 /     0
	f7               t_parse.c      256  0 / 1       0 /     0
	function6        t_parse.c      266  0 / 1       0 /     0
	function8        t_parse.c      272  0 / 1       0 /     0
	function9        t_parse.c      273  0 / 1       0 /     0
	function11       t_parse.c      276  0 / 1       0 /     0
	function14       t_parse.c      283  0 / 1       0 /     0
	catch            t_parse.c      306  0 / 1       0 /     0
	push             calc1.c         11  1 / 1       0 /     0
	pop              calc1.c         21  1 / 1       0 /     0
	Input            calc1.c         26  1 / 1       0 /     0
	stack::selftest  calc3stk.cpp    15  1 / 1       0 /     0
	stack::stack     calc3stk.h       8  1 / 1       0 /     0
	stack::push      calc3stk.h      15  1 / 1       0 /     0
	f1               t_parse.c        2  1 / 1       0 /     0
	f2               t_parse.c        3  1 / 1       0 /     0
	lex              calc2.c         54  1 / 1       6 /    36 =  16%
	main             calc1.c         31  1 / 1      10 /    34 =  29%
	main             calc3.cpp        7  1 / 1      10 /    29 =  34%
	main             t_parse.c        4  1 / 1     196 /   457 =  42%
	eval             calc2.c        252  1 / 1       3 /     6 =  50%
	stack::pop       calc3stk.h      23  1 / 1       1 /     2 =  50%
	prs              calc2.c        158  1 / 1      77 /   125 =  61%
	main             calc2.c        271  1 / 1       8 /    10 =  80%
	stack::~stack    calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	---------------  ------------------  -----  ---------------------
	Total                                  53%     313 /   730 =  42%
.
#107
run 'covfn -w1 -q .'<<.
	Function  Source        Line  FnCov           C/D Coverage
	--------  ------------------  -----  ---------------------
	name      calc2.c         99  0 / 1       0 /    11 =   0%
	f8        t_parse.c      342  0 / 1       0 /     6 =   0%
	match     calc2.c        128  0 / 1       0 /     4 =   0%
	z         calc2.c        143  0 / 1       0 /     4 =   0%
	swi...ct  t_parse.c      308  0 / 1       0 /     4 =   0%
	f3        t_parse.c      234  0 / 1       0 /     0
	f4        t_parse.c      240  0 / 1       0 /     0
	f5        t_parse.c      247  0 / 1       0 /     0
	f7        t_parse.c      256  0 / 1       0 /     0
	fun...n6  t_parse.c      266  0 / 1       0 /     0
	fun...n8  t_parse.c      272  0 / 1       0 /     0
	fun...n9  t_parse.c      273  0 / 1       0 /     0
	fun...11  t_parse.c      276  0 / 1       0 /     0
	fun...14  t_parse.c      283  0 / 1       0 /     0
	catch     t_parse.c      306  0 / 1       0 /     0
	push      calc1.c         11  1 / 1       0 /     0
	pop       calc1.c         21  1 / 1       0 /     0
	Input     calc1.c         26  1 / 1       0 /     0
	sta...st  calc3stk.cpp    15  1 / 1       0 /     0
	sta...ck  calc3stk.h       8  1 / 1       0 /     0
	sta...sh  calc3stk.h      15  1 / 1       0 /     0
	f1        t_parse.c        2  1 / 1       0 /     0
	f2        t_parse.c        3  1 / 1       0 /     0
	lex       calc2.c         54  1 / 1       6 /    36 =  16%
	main      calc1.c         31  1 / 1      10 /    34 =  29%
	main      calc3.cpp        7  1 / 1      10 /    29 =  34%
	main      t_parse.c        4  1 / 1     196 /   457 =  42%
	eval      calc2.c        252  1 / 1       3 /     6 =  50%
	sta...op  calc3stk.h      23  1 / 1       1 /     2 =  50%
	prs       calc2.c        158  1 / 1      77 /   125 =  61%
	main      calc2.c        271  1 / 1       8 /    10 =  80%
	sta...ck  calc3stk.cpp     5  1 / 1       2 /     2 = 100%
	--------  ------------------  -----  ---------------------
	Total                           53%     313 /   730 =  42%
.
#108
run 'echo "*push" >$t1; covfn -q -k @$t1'<<.
	push         calc1.c         11  1 / 1       0 /     0
	stack::push  calc3stk.h      15  1 / 1       0 /     0
.
#109
run 'covsrc .'<<.
	C-Cover Coverage by Source %ver
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#110
run 'covsrc -c -q .'<<.
	"Source","Function Coverage","out of","%","C/D Coverage","out of","%"
	"t_parse.c",3,15,20%,196,467,41%
	"calc2.c",4,7,57%,94,196,47%
	"calc1.c",4,4,100%,10,34,29%
	"calc3.cpp",1,1,100%,10,29,34%
	"calc3stk.h",3,3,100%,1,2,50%
	"calc3stk.cpp",2,2,100%,2,2,100%
	"Total",17,32,53%,313,730,42%
.
#111
run 'covsrc -h -q'<<.
	Usage: covsrc [options] [region ...]
	-b     --base        Show file basename rather than full path
	-c     --csv         Output comma separated values
	-dDir  --srcdir Dir  Source file directory is Dir
	-fPath --file Path   Coverage data file is Path
	-h     --help        Display this usage summary
	-k     --no-head     Suppress column headings
	-m     --by-name     Sort by source file name
	-p     --by-pcnt     Sort by percent exercised [default]
	-q     --quiet       Quiet
	-r     --reverse     Reverse sort
	-u     --by-uncov    Sort by number of uncovered functions/branches
	-v     --verbose     Verbose
	-wN    --width N     Report width is N columns [N=79]
	--decision           Show decision (branch) coverage.  Not recommended
	--no-banner          Suppress startup banner
	status 1
.
#112
run 'covsrc -k -q .'<<.
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
.
#113
run 'covsrc -m -q .'<<.
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#114
run 'covsrc -p -q .'<<.
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#115
run 'covsrc -r -q .'<<.
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#116
run 'covsrc -u -q .'<<.
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#117
run 'covsrc -v .'<<.
	C-Cover Coverage by Source %ver
	note cov911: data file test.cov
	Source            Function Coverage           C/D Coverage
	------------  ---------------------  ---------------------
	t_parse.c          3 /    15 =  20%     196 /   467 =  41%
	calc2.c            4 /     7 =  57%      94 /   196 =  47%
	calc1.c            4 /     4 = 100%      10 /    34 =  29%
	calc3.cpp          1 /     1 = 100%      10 /    29 =  34%
	calc3stk.h         3 /     3 = 100%       1 /     2 =  50%
	calc3stk.cpp       2 /     2 = 100%       2 /     2 = 100%
	------------  ---------------------  ---------------------
	Total             17 /    32 =  53%     313 /   730 =  42%
.
#118
run 'covsrc -w1 -q .'<<.
	Source      Function Coverage           C/D Coverage
	------  ---------------------  ---------------------
	...e.c       3 /    15 =  20%     196 /   467 =  41%
	...2.c       4 /     7 =  57%      94 /   196 =  47%
	...1.c       4 /     4 = 100%      10 /    34 =  29%
	...cpp       1 /     1 = 100%      10 /    29 =  34%
	...k.h       3 /     3 = 100%       1 /     2 =  50%
	...cpp       2 /     2 = 100%       2 /     2 = 100%
	------  ---------------------  ---------------------
	Total       17 /    32 =  53%     313 /   730 =  42%
.
#119
run 'echo "*.c" >$t1; covsrc -q -k @$t1'<<.
	t_parse.c       3 /    15 =  20%     196 /   467 =  41%
	calc2.c         4 /     7 =  57%      94 /   196 =  47%
	calc1.c         4 /     4 = 100%      10 /    34 =  29%
.
#120
run 'covmerge'<<.
	C-Cover Merge %ver
	Usage: covmerge [options] input.cov ...
	-c     --create     Create result coverage data file
	-fPath --file Path  Result coverage data file is Path
	-h     --help       Display this usage summary
	-q     --quiet      Quiet
	-v     --verbose    Verbose
	--no-banner         Suppress startup banner
	status 1
.
#121
run 'covmerge -h -q'<<.
	Usage: covmerge [options] input.cov ...
	-c     --create     Create result coverage data file
	-fPath --file Path  Result coverage data file is Path
	-h     --help       Display this usage summary
	-q     --quiet      Quiet
	-v     --verbose    Verbose
	--no-banner         Suppress startup banner
	status 1
.
#122
run 'covbr -q >$t1 && covmerge -c -ftmp.cov -q test.cov;
  covbr -q -ftmp.cov | diff $t1 -'<<.
.
#123
run 'cp test.cov tmp.cov && covmerge -ftmp.cov -q test.cov test.cov;
  covbr -q -ftmp.cov | diff $t1 -'<<.
.
#124
run 'covc -q $cc ${o}calc1 calc1.c && echo 1 | calc1 &&
  covbr -q calc1.c >$t1 && rm -f tmp1.cov tmp2.cov &&
  export COVFILE;
  COVFILE=tmp1.cov; covc -q $cc ${o}calc1 calc1.c && echo 9 | calc1 &&
  COVFILE=tmp2.cov; covc -q $cc ${o}calc1 calc1.c && echo q | calc1 &&
  COVFILE=test.cov;
  covmerge tmp1.cov tmp2.cov && covbr -q calc1.c | diff $t1 -;
  rm -f tmp1.cov tmp2.cov'<<.
	Enter reverse-polish expressions, q to quit
	Enter reverse-polish expressions, q to quit
	Enter reverse-polish expressions, q to quit
	C-Cover Merge %ver
	33c33
	<   -->t   39b                !quit &&
	---
	>   tf     39b                !quit &&
	51c51
	< -->      49e                                                         case '9':
	---
	> X        49e                                                         case '9':
	91c91
	< -->      89                  case 'q':
	---
	> X        89                  case 'q':
.
#125
run 'covmerge -q test.cov && covmgr -lq calc1.c | sed s/calc/-calc/'<<.
	-calc1.c
.
#126
run -n '
	covmgr -r -q calc1.obj
	covc -q bcc32 -c -w- -ocalc1.bcc calc1.c &&
  covc -q cl -c -Focalc1.cl calc1.c &&
  covmgr -q -l calc1 | sed s/calc/-calc/ &&
	rm calc1.bcc calc1.cl
'<<.
	-calc1.bcc
	-calc1.c
	-calc1.cl
	-calc1.c
.
#127
run -u '
	covmgr -r -q calc1.c
	covc -q gcc -c -o calc1.gc calc1.c
  covmgr -q -l calc1 | sed s/calc/-calc/ && rm calc1.gc'<<.
	-calc1.gc
	-calc1.c
.
#128
run 'covmgr -q -r calc1.c'<<.
.
#129
run 'covmerge -ftmp.cov -q no-file'<<.
	covmerge: cannot open no-file: No such file or directory
	status 1
.
#130
run 'rm -f tmp.cov; covclear -ftmp.cov -q test.cov'<<.
	covclear: cannot open tmp.cov: No such file or directory
	status 1
.
#131
run 'covmgr'<<.
	C-Cover Manager %ver
	Usage: covmgr -aN|--autosave N [options]
	       covmgr -h|--help
	       covmgr -l|--list [options] [sub-string ...]
	       covmgr -m|--compact
	       covmgr -r|--remove [options] source ...
	       covmgr -s|--rmsub [options]
	Actions:
	 -aN    --autosave N  Automatically save data older than N seconds
	 -h     --help        Display this usage summary
	 -l     --list        List coverage data file contents
	 -m     --compact     Compact the file
	 -r     --remove      Remove coverage structure of specified sources
	 -s     --rmsub       Remove substitution list
	Options:
	 -c     --create      Create coverage data file if it does not exist
	 -dDir  --srcdir Dir  Source file directory is Dir
	 -fPath --file Path   Coverage data file is Path
	 -q     --quiet       Quiet
	 -v     --verbose     Verbose
	 --no-banner          Suppress startup banner
	status 1
.
#132
run 'covmgr -a123 && covmgr -a1 && covmgr -a2 -q && covmgr -a0 &&
		covmgr -a || covmgr -a.'<<.
	C-Cover Manager %ver
	autosave interval set to 123 seconds
	C-Cover Manager %ver
	autosave interval set to 1 second
	C-Cover Manager %ver
	autosave disabled
	C-Cover Manager %ver
	error cov136: option -a requires an argument
	C-Cover Manager %ver
	error cov135: invalid number
	status 1
.
#133
run 'covmgr -h -q'<<.
	Usage: covmgr -aN|--autosave N [options]
	       covmgr -h|--help
	       covmgr -l|--list [options] [sub-string ...]
	       covmgr -m|--compact
	       covmgr -r|--remove [options] source ...
	       covmgr -s|--rmsub [options]
	Actions:
	 -aN    --autosave N  Automatically save data older than N seconds
	 -h     --help        Display this usage summary
	 -l     --list        List coverage data file contents
	 -m     --compact     Compact the file
	 -r     --remove      Remove coverage structure of specified sources
	 -s     --rmsub       Remove substitution list
	Options:
	 -c     --create      Create coverage data file if it does not exist
	 -dDir  --srcdir Dir  Source file directory is Dir
	 -fPath --file Path   Coverage data file is Path
	 -q     --quiet       Quiet
	 -v     --verbose     Verbose
	 --no-banner          Suppress startup banner
	status 1
.
#134
run 'covmgr -l calc | sed s/calc/-calc/ | sed s/obj$/o/'<<.
	C-Cover Manager %ver
	-calc2.o
	-calc2.c
	-calc3.o
	-calc3stk.h
	-calc3.cpp
	-calc3stk.o
	-calc3stk.h
	-calc3stk.cpp
.
#135
run 'covmgr -r -q'<<.
	error cov102: no arguments
	status 1
.
#136
run 'cp test.cov tmp.cov; covmgr -r -ftmp.cov -q calc1.c'<<.
	error cov139: no coverage information for source calc1.c.  Specify regions as they are shown, without -b/--base
	status 1
.
#137
run '# option error
	covbr -q -Z
	covc -q -Z
	covclass -q -Z
	covclear -q -Z
	covdir -q -Z
	covfn -q -Z
	covmerge -q -Z
	covmgr -q -Z
	covsrc -q -Z
'<<.
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	error cov137: unknown option "-Z"
	status 1
.
#138
run 'rm -f no; covbr -fno -q; covclass -fno -q; covclear -fno -q;
		covdir -fno -q; covfn -fno -q
		covmerge -fno -q test.cov; covmgr -fno -l -q; covsrc -fno -q'<<.
	covbr: cannot open no: No such file or directory
	covclass: cannot open no: No such file or directory
	covclear: cannot open no: No such file or directory
	covdir: cannot open no: No such file or directory
	covfn: cannot open no: No such file or directory
	covmerge: cannot open no: No such file or directory
	covmgr: cannot open no: No such file or directory
	covsrc: cannot open no: No such file or directory
	status 1
.
#139
run -n '
	covbr -f. -q || covclass -f. -q || covdir -f. -q ||
		covfn -f. -q || covsrc -f. -q'<<.
	covbr: . is a directory. Specify a filename for the coverage file
	covclass: . is a directory. Specify a filename for the coverage file
	covdir: . is a directory. Specify a filename for the coverage file
	covfn: . is a directory. Specify a filename for the coverage file
	covsrc: . is a directory. Specify a filename for the coverage file
	status 1
.
#140
run -u '>no; chmod 000 no; covbr -fno -q || covclass -fno -q ||
		covdir -fno -q || covfn -fno -q || covsrc -fno -q || rm -f no'<<.
	covbr: cannot open no: Permission denied
	covclass: cannot open no: Permission denied
	covdir: cannot open no: Permission denied
	covfn: cannot open no: Permission denied
	covsrc: cannot open no: Permission denied
.
#141
run 'rm -f ro; cp test.cov ro; chmod -w ro;
  covc -fro -q cc s.c || covc -fro -q cc s.c ||
  covclear -fro -q ||
  covmgr -fro -c -a1 -q || covmgr -fro -l -c -q || covmgr -fro -l -q'<<.
	covc: cannot open ro: Permission denied
	covc: cannot open ro: Permission denied
	covclear: cannot open ro: Permission denied
	covmgr: cannot open ro: Permission denied
	covmgr: cannot open ro: Permission denied
.
#142
run -n '
	covmerge -fro -q test.cov || covmerge -fro -c -q test.cov
'<<.
	error cov146: error removing ro: Permission denied
	error cov146: error removing ro: Permission denied
	status 1
.
#143
run 'echo covmeraa.*'<<.
	covmeraa.*
.
#144
run -n '
		x="$(which covx)"
		license 987654 $x &&
		covc $cc -c calc1.c && covmgr -r calc1.c &&
		license -e 1 12345 $x &&
		covc $cc -c calc1.c && covmgr -r calc1.c &&
		license 0 $x &&
		covc $cc -c calc1.c && covmgr -r calc1.c'<<.
	C-Cover Compile C/C++ %ver, Serial 987654
	C-Cover Manager %ver, Serial 987654
	C-Cover Compile C/C++ %ver
	License expires in 30 days
	C-Cover Manager %ver
	License expires in 30 days
	C-Cover Compile C/C++ %ver
	C-Cover Manager %ver
.
#145
run -n 'covx abcde || covc fghijk/lmnop'<<.
	C-Cover Compile C/C++ %ver
	error cov147: unable to locate compiler program abcde.  Check the PATH and make sure your compiler bin directory does not contain covc
	warning cov816: original unmodified compile command fails
	C-Cover Compile C/C++ %ver
	error cov113: cannot run program fghijk/lmnop: No such file or directory
	warning cov816: original unmodified compile command fails
	status 1
.
#146
run -n 'cl -c -Facovrt_2.asm -Zp2 -DSYS_nt=1 ../lib/covrt.c &&
	cl -c -Facovrt_4.asm -Zp4 -DSYS_nt=1 ../lib/covrt.c && diff covrt_?.asm;
	rm covrt_?.asm
'<<.
	covrt.c
	covrt.c
.
#147
run -n '
	nmake -f t_meta.mk sys_$sys
'<<.
.
#148
run -u '
	if (file "`$which $cc`" 2>&1 | grep -v "commands text" >/dev/null)
  then
	make -f t_meta.mk sys_$sys 2>&1 |
		# Remove "Entering directory ..." and "Leaving directory ..."
		grep -v "ing directory"
		true
  fi'<<.
.
#149
run -n '
	cd src-tmp
	# Sub-shell avoids here-document glitch (missing first char) on Win9x
	( buildrt VxWorks gcc -DCPU -DCovrt_nanosleep -I/cc/VxWorks/include )
	nm -u libcov-VxWorks.a | sed s,.*_,_,
	cd -
'<<.
	% %cc -c -DSYS_VxWorks=1 -o covrt.obj -O2 -Wall -DCPU -DCovrt_nanosleep -I/cc/VxWorks/include covrt.c
	% ar -cr libcov-VxWorks.a covrt.obj
	buildrt: successfully built libcov-VxWorks.a
	covrt.obj:
	_errno
	_close
	_exit
	_getenv
	_lseek
	_open
	_read
	_semBCreate
	_semDelete
	_semFlush
	_semTake
	_taskDelay
	_taskDeleteHookAdd
	_taskDeleteHookDelete
	_taskIdSelf
	_taskLock
	_taskSafe
	_taskSpawn
	_taskTcb
	_taskUnlock
	_taskUnsafe
	_vxTas
	_write
.
#150
run 'covc -q -t,f, $cc -c calc1.c'<<.
.
#151
run -n '
	covc -v cl -nologo -Zg calc1.c >$t1 2>&1 && grep 902 $t1
	covc -v cl -c -nologo /Dm /Un /Idir /D o calc1.c >$t1 2>&1 && grep 902 $t1;
	covc -v cl -c -nologo /D >$t1 2>&1 || grep 902 $t1'<<.
	note cov902: running %.../cl -nologo -Zg calc1.c
	note cov902: running internalCpp -DCCOVER=%ver -c -nologo /Dm /Un /Idir /D o calc1.c -+cc cl
	note cov902: running %.../cl -c -nologo /Idir tmp/covc/calc1.c
	note cov902: running %.../cl -c -nologo /D
.
#152
run -u '
  covc -v gcc -c -Dm -Un -Idir -Wall calc1.c >$t1 2>&1 && grep cov902 $t1'<<.
	note cov902: running %cc -DCCOVER=%ver -Dm -Un -Idir -Wall -E calc1.c >tmp/covc/calc1.i
	note cov902: running %cc -c -Idir -Wall tmp/covc/calc1.c
.
#153
run -u '
	(
	covc -v $cc calc1.c
	covc -v --no-lib $cc calc1.c
	covc -v $cc calc1.c -lcov
	) 2>&1 | grep cov902.*tmp/covc[0-9]*/calc1.c
'<<.
	note cov902: running %cc tmp/covc/calc1.c %bindir/../lib/libcov.a
	note cov902: running %cc tmp/covc/calc1.c
	note cov902: running %cc tmp/covc/calc1.c -lcov -L%bindir/../lib/
.
#154
run -n '
	echo >t_pragma.cpp "#pragma strings(readonly)" &&
	echo >>t_pragma.cpp "int main() { return 0; }" &&
	covc -q -l $cc -c t_pragma.cpp 2>&1 |
		grep -v C4068
	head -n 20 $TMPDIR/covc/t_pragma.cpp
'<<.
	#if _MSC_VER || __BORLANDC__ || __SC__
	#define cov_link_ __cdecl
	#endif
	#if __IBMC__ || __IBMCPP__
	#define cov_link_ _Optlink
	#endif
	#if !defined(cov_link_)
	#define cov_link_
	#endif
	#ifdef __cplusplus
	extern "C" void cov_link_ cov_count();
	extern "C" void cov_link_ cov_term();
	class cov_class_ { public:
	cov_class_() { cov_count(); }
	~cov_class_() { cov_term(); }
	};
	static cov_class_ cov_static_;
	#endif
	#line 1 "t_pragma.cpp"
	#pragma strings(readonly)
.
#155
run '
	rm -f -r "D I R"; mkdir "D I R" "D I R"/tmp && cd "D I R" &&
	if [ $sys = u ]
	then
		no_cpp=
		[ -f ../../ccoverLicense.txt ] && cp ../../ccoverLicense.txt .
		ln ../../src/covc .
	else
		no_cpp=--no-cpp
		cp ../cov*.exe .
	fi &&
	cp ../calc1.c . &&
	covc -q -f../test.cov $no_cpp $cc -c calc1.c &&
	cd .. && rm -f -r "D I R" &&
	covmgr -l -q calc1.c && covsrc -q | grep -i "D I R"'<<.
	D I R/calc1.c
	D I R/calc1.c       0 /     4 =   0%       0 /    34 =   0%
.
#156
run 'echo >t_switch.c "int main() { int i = 1;
  switch (i) { case (char*)1: break; } return 0; }";
  covc -q $cc -c t_switch.c && rm t_switch.c'<<.
.
#157
run -n '
	$COMSPEC /c covc none
	$COMSPEC /c covbr none
	true
'<<.
	C-Cover Compile C/C++ %ver
	error cov147: unable to locate compiler program none.  Check the PATH and make sure your compiler bin directory does not contain covc
	warning cov816: original unmodified compile command fails
	C-Cover Coverage by Branch %ver
	error cov139: no coverage information for function none.  Specify regions as they are shown, without -b/--base
.
#158
run -n 'covc -q cl -c -nologo t_atexit.cpp'<<.
.
#159
run -n '
	covc -q $cc t_atexit.cpp
	rm -f t_atexit.exe
'<<.
.
#160
run -n -u '
  echo >"t_ space.c" "#include <stdio.h>" &&
  echo >>"t_ space.c" "void f() { }" &&
  echo \"t_ space.c\" >$t1
  covc $cc -c @$t1 && covsrc -ckq @$t1;
  rm "t_ space.c"; covmgr -q -r @$t1'<<.
	C-Cover Compile C/C++ %ver
	"t_ space.c",0,1,0%,0,0,""
.
#161
run -n '
	export CL="-Da -Db"
	covc -v cl -c t_empty.c >$t1 2>&1
	unset CL
	grep -e 902 -e "Command line" $t1'<<.
	note cov902: running internalCpp -DCCOVER=%ver -Da -Db -c t_empty.c -+cc cl
	note cov902: running %.../cl -c tmp/covc/t_empty.c
.
#162
run -n '
  integer i=0
  list=""
  echo "void f(){}" >t_many.cpp
  while [ $i -le 299 ]; do
	list="$list t_many.cpp"
	i=i+1
  done
  mkdir t_many
  covc -q -p- cp $list t_many
  rm -f -r t_many*'<<.
.
#163
run 'covc -q $c2 -c t_term.cpp && covmgr -q -r t_term.cpp'<<.
.
#164
run 'echo >t_path.c "#pragma C-Cover off";
  echo >>t_path.c "int main() {";
  echo >>t_path.c "int success = cov_file(\"test1.cov\");";
  echo >>t_path.c "#pragma C-Cover on";
  echo >>t_path.c "if(success);";
  echo >>t_path.c "return 0; }";
  covc -q -ftest1.cov $cc ${o}t_path t_path.c && ./t_path &&
  br -ftest1.cov t_path.c && rm t_path.c test1.cov'<<.
	???2 int main(){
	T if(success);
.
#165
run '# runtime errors
	echo >t_tmp.c "int main(int argc, char ** argv) {";
	echo >>t_tmp.c "if (argc == 2) cov_o_.signature_ = 0;"
	echo >>t_tmp.c "return 0; }"
	covc -q $cc ${o}t_tmp t_tmp.c &&
	cp t_tmp.c t_tmp2.c
	covc -q $cc -c t_tmp2.c
	COVERR=t_error && export COVERR && rm -f t_error && ./t_tmp 1 &&
	COVFILE=none && ./t_tmp &&
	COVFILE=$t1 && >$t1 && ./t_tmp &&
	unset COVFILE && mv test.cov $t1 && ./t_tmp && mv $t1 test.cov &&
	covmgr -q -r t_tmp.c
	./t_tmp
	covmgr -q -r t_tmp2.c
	sed s/t_tmp.obj/t_tmp.o/ t_error; rm t_error t_tmp.c
'<<.
	C-Cover %ver  1: Memory corrupted for object t_tmp.o.  (Error occurred at %...).
	C-Cover %ver 16: Cannot open none.  COVFILE is probably set to an incorrect path.  (Error occurred at %...).
	C-Cover %ver 18: Not a C-Cover file or wrong version: tmp/check1.tmp.  (Error occurred at %...).
	C-Cover %ver 15: Cannot find %cwd/test.cov.  COVFILE is not set.  (Error occurred at %...).
	C-Cover %ver 12: Object missing or out of date.  Check coverage data file is same one created when building.  Or, try covc option --retain.  Path is %cwd/test.cov.  Recompile object t_tmp.o %object-id.  (Error occurred at %...).
.
#166
run 'covc -q $cc -c calc1.c &&
  echo "*" >$t1 && covmgr -q -r @$t1 &&
  covmgr -q -l'<<.
.
#167
run -n '# Argument file line length near 256
	echo >$t1 ./././././././././././././././././././././././././././././././././././././././././././././././././././././././././././calc3.cpp ./././././././././././././././././././././././././././././././././././././././././././././././././././././././././calc3stk.cpp
	covc -q cl -nologo @$t1
'<<.
.
#168
run 'rm -f test.cov; covmgr -q -c -l &&
  covbr && covclass && covdir && covfn && covsrc'<<.
	C-Cover Coverage by Branch %ver
	warning cov814: report is empty
	C-Cover Coverage by Class %ver
	warning cov814: report is empty
	Class      Function Coverage           C/D Coverage
	-----  ---------------------  ---------------------
	-----  ---------------------  ---------------------
	Total       0 /     0              0 /     0
	C-Cover Coverage by Directory %ver
	warning cov814: report is empty
	Directory      Function Coverage           C/D Coverage
	---------  ---------------------  ---------------------
	---------  ---------------------  ---------------------
	Total           0 /     0              0 /     0
	C-Cover Coverage by Function %ver
	warning cov814: report is empty
	Function  Source        Line  FnCov           C/D Coverage
	--------  ------------------  -----  ---------------------
	--------  ------------------  -----  ---------------------
	Total                            0%       0 /     0
	C-Cover Coverage by Source %ver
	warning cov814: report is empty
	Source      Function Coverage           C/D Coverage
	------  ---------------------  ---------------------
	------  ---------------------  ---------------------
	Total        0 /     0              0 /     0
.
#169
run -n '# DLL plain
	covc -q cl -nologo -c -Fot_dll_dll.obj -MLd t_dll.cpp &&
	covlink -v link -nologo -dll t_dll_dll.obj 2>&1 |
		grep 902
	covc -q cl -MLd -nologo -Dexe t_dll.cpp t_dll_dll.lib &&
	t_dll && br t_dll.cpp
'<<.
	note cov902: running %.../cl -c -nologo -Zl -Fotmp/ccoverEntryUser.obj tmp/ccoverEntryUser.c
	note cov902: running %.../link -libpath:./../lib/ -nologo -dll t_dll_dll.obj -entry:cov_DllMain@12 tmp/ccoverEntryUser.obj -include:__DllMainCRTStartup@12 covntm.lib covrt-nt-link.obj
	X int main(){
	X static void atexit1(){}
	X static void atexit2(){}
	X void __declspec(dllexport)dll(vo
	T if(Attach_raw)
	X C(){
	X ~C(){
	X BOOL WINAPI DllMain(HINSTANCE h,
	x if(why==DLL_PROCESS_ATTACH)
	. BOOL WINAPI user(HINSTANCE h,DWO
	. if(why==DLL_PROCESS_ATTACH)
.
#170
run -n '# DLL with -entry
	covc -q cl -nologo -c -Fot_dll_dll.obj -MLd -Dentry t_dll.cpp &&
	covlink -v link -nologo -dll t_dll_dll.obj -entry:user@12 2>&1 |
		grep 902
	t_dll && br t_dll.cpp
'<<.
	note cov902: running %.../cl -c -nologo -Zl -Fotmp/ccoverEntryUser.obj tmp/ccoverEntryUser.c
	note cov902: running %.../link -libpath:./../lib/ -nologo -dll t_dll_dll.obj -entry:cov_DllMain@12 tmp/ccoverEntryUser.obj -include:__DllMainCRTStartup@12 covntm.lib covrt-nt-link.obj
	X int main(){
	X void __declspec(dllexport)dll(vo
	F if(Attach_raw)
	. BOOL WINAPI DllMain(HINSTANCE h,
	. if(why==DLL_PROCESS_ATTACH)
	X BOOL WINAPI user(HINSTANCE h,DWO
	x if(why==DLL_PROCESS_ATTACH)
.
#171
run -n '# DLL with -noentry
	covc -q cl -nologo -c -Fot_dll_dll.obj -MLd -Dentry t_dll.cpp &&
 	covlink -v link -nologo -dll t_dll_dll.obj -noentry 2>&1 |
		grep 902
	t_dll && br t_dll.cpp
'<<.
	note cov902: running %.../link -libpath:./../lib/ -nologo -dll t_dll_dll.obj -entry:cov_DllMain@12 -include:__DllMainCRTStartup@12 covntm.lib covrt-nt-link.obj
	X int main(){
	X void __declspec(dllexport)dll(vo
	F if(Attach_raw)
	. BOOL WINAPI DllMain(HINSTANCE h,
	. if(why==DLL_PROCESS_ATTACH)
	. BOOL WINAPI user(HINSTANCE h,DWO
	. if(why==DLL_PROCESS_ATTACH)
.
#172
run -n '# DLL with -MD
	covc -q cl -nologo -c -Fot_dll_dll.obj -MDd t_dll.cpp &&
	covlink -v link -nologo -dll t_dll_dll.obj 2>&1 |
		grep 902
	t_dll && br t_dll.cpp
'<<.
	note cov902: running %.../cl -c -nologo -Zl -Fotmp/ccoverEntryUser.obj tmp/ccoverEntryUser.c
	note cov902: running %.../link -libpath:./../lib/ -nologo -dll t_dll_dll.obj -entry:cov_DllMain@12 tmp/ccoverEntryUser.obj -include:__DllMainCRTStartup@12 covntm.lib covrt-nt-link.obj
	X int main(){
	X static void atexit1(){}
	X static void atexit2(){}
	X void __declspec(dllexport)dll(vo
	T if(Attach_raw)
	X C(){
	X ~C(){
	X BOOL WINAPI DllMain(HINSTANCE h,
	x if(why==DLL_PROCESS_ATTACH)
	. BOOL WINAPI user(HINSTANCE h,DWO
	. if(why==DLL_PROCESS_ATTACH)
.
#173
run -n '# -nod
	covc -q cl -nologo -c calc1.c &&
	covlink -v link -nod calc1.obj libc.lib 2>&1 |
		grep 902
'<<.
	note cov902: running %.../link -libpath:./../lib/ -nod calc1.obj libc.lib covntm.lib kernel32.lib user32.lib covrt-nt-link.obj
.
#174
run -n '# covlink as link.exe
	covc -q cl -nologo -c calc1.c &&
	cp covlink.exe link.exe &&
	echo -v >covlink.cfg &&
	link calc1.obj 2>&1 | grep 90[27]
	rm covlink.cfg link.exe
'<<.
	note cov902: running %.../link -libpath:./../lib/ calc1.obj covrt-nt-link.obj
	note cov907: program successful C:/Program Files/Microsoft Visual Studio/VC98/bin/link
.
#175
run -n '# cl -Fc
	covc -q cl -nologo -c -Fc calc1.c &&
	rm calc1.cod
'<<.
.
#176
run -n 'covlink -q covlink'<<.
	error cov119: recursive use of covlink
	status 1
.
#177
run -n '# Incorrect dependence on precompiled header option
	echo  >t_pch.h    "#define pch 1"
	echo  >t_pch_Yc.c "#include \"t_pch.h\""
	covc -q cl -c -nologo -Yct_pch.h t_pch_Yc.c

	# compile error
	echo  >t_pch_Yu.c "#define i 1"
	echo >>t_pch_Yu.c "#include \"t_pch.h\""
	echo >>t_pch_Yu.c "int i;"
	cl -c -nologo -Yut_pch.h t_pch_Yu.c
	echo exit 1: $?
	covc -q --no-cpp cl -c -nologo -Yut_pch.h t_pch_Yu.c
	echo exit 2: $?

	# preprocessor error
	echo  >t_pch_Yu.c "#include \"t_pch_error.h\""
	echo >>t_pch_Yu.c "#include \"t_pch.h\""
	cl -c -nologo -Yut_pch.h t_pch_Yu.c
	echo exit 3: $?
	covc -q --no-cpp cl -c -nologo -Yut_pch.h t_pch_Yu.c
	echo exit 4: $?
'<<.
	exit 1: 0
	t_pch_Yu.c(3) : error C2059: syntax error : 'constant'
	error cov149: detected precompiled header inconsistency, source t_pch_Yu.c option -Yut_pch.h.  IMPORTANT: To diagnose this, see the description in C-Cover Help under Troubleshooting
	exit 2: 1
	exit 3: 0
	t_pch_Yu.c(1) : fatal error C1083: Cannot open include file: 't_pch_error.h': No such file or directory
	error cov149: detected precompiled header inconsistency, source t_pch_Yu.c option -Yut_pch.h.  IMPORTANT: To diagnose this, see the description in C-Cover Help under Troubleshooting
	exit 4: 1
.
#178
run -n '# Class definition with declspec(...) on the class
	echo >t_declspec.cpp "class __declspec(dllexport) c {"
	echo >>t_declspec.cpp "void f() { } };"
	covc -q cl -c -nologo t_declspec.cpp &&
	br t_declspec.cpp
	rm t_declspec.cpp; covmgr -q -r t_declspec.cpp
'<<.
	. void f(){}};
.
#179
run '# change directory at runtime with COVFILE not set
	echo >t_tmp.c "int main(void) {"
	echo >>t_tmp.c "chdir(\"t_tmp_d\");"
	echo >>t_tmp.c "return 0; }"
	mkdir t_tmp_d
	covc -q $cc ${o}t_tmp t_tmp.c
	unset COVFILE
	COVERR=t_tmperr
	export COVERR
	./t_tmp
	[ -f t_tmp/t_tmperr ] && cat t_tmp_d/t_tmperr
	br t_tmp.c
	covmgr -q -r t_tmp.c
	rm -rf t_tmp*
'<<.
	X int main(void){
.
#180
run -n '# library external references
	nm -u ../lib/covntm.lib
'<<.
	covrt-nt-entryUser.obj:
	covrt.obj:
	    0:0000000 U __imp__CloseHandle@4
	    0:0000000 U __imp__CreateFileA@28
	    0:0000000 U __imp__CreateMutexA@12
	    0:0000000 U __imp__CreateThread@24
	    0:0000000 U __imp__EnterCriticalSection@4
	    0:0000000 U __imp__ExitThread@4
	    0:0000000 U __imp__GetCurrentDirectoryA@8
	    0:0000000 U __imp__GetEnvironmentVariableA@12
	    0:0000000 U __imp__GetExitCodeThread@8
	    0:0000000 U __imp__GetLastError@0
	    0:0000000 U __imp__GetModuleFileNameA@12
	    0:0000000 U __imp__GetProcAddress@8
	    0:0000000 U __imp__GetTempPathA@8
	    0:0000000 U __imp__InterlockedExchange@8
	    0:0000000 U __imp__IsBadWritePtr@8
	    0:0000000 U __imp__LeaveCriticalSection@4
	    0:0000000 U __imp__LoadLibraryA@4
	    0:0000000 U __imp__LockFile@20
	    0:0000000 U __imp__LockFileEx@24
	    0:0000000 U __imp__MessageBoxA@16
	    0:0000000 U __imp__MoveFileExA@12
	    0:0000000 U __imp__ReadFile@20
	    0:0000000 U __imp__ReleaseMutex@4
	    0:0000000 U __imp__SetFilePointer@16
	    0:0000000 U __imp__Sleep@4
	    0:0000000 U __imp__UnlockFile@20
	    0:0000000 U __imp__UnlockFileEx@20
	    0:0000000 U __imp__WaitForSingleObject@8
	    0:0000000 U __imp__WriteFile@20
	    0:0000000 U __imp__lstrcatA@8
	    0:0000000 U __imp__lstrcpyA@8
	    0:0000000 U _atexit
	    0:0000000 U _ccover_Link_with_coverage_build_enabled
	    0:0000000 U _cov_DllMain_user
.
#181
run -n '# unused template warning
	echo >t_tmp.cpp "template<class T> void func(T t) {}"
	covc -q cl -c -W4 -WX t_tmp.cpp &&
	rm t_tmp.cpp
'<<.
.
#182
run '# local and global config files using $HOME
	unset HOMEPATH
	HOME=$TMPDIR
	export HOME
	d=`$which covdir`; d=`dirname $d`
	echo >$d/covdir.cfg -,
	echo >$HOME/covdir.cfg -_
	covdir
	echo >$d/covdir.cfg -q
	covdir
	rm $d/covdir.cfg $HOME/covdir.cfg
'<<.
	C-Cover Coverage by Directory %ver
	error cov137: unknown option "-,"
	error cov137: unknown option "-_"
.
#183
run -n '# local and global config files using $HOMEDRIVE and $HOMEPATH
	if [ "$HOMEDRIVE" = "" ]; then; HOMEDRIVE=$ROOTDIR; fi
	HOMEPATH=$(echo $TMPDIR | sed s/^[a-zA-Z]://)
	mkdir $TMPDIR/tmp
	HOME=$TMPDIR/tmp
	export HOME HOMEDRIVE HOMEPATH
	d=`which covdir`; d=`dirname $d`
	echo >$d/covdir.cfg -,
	echo >$HOMEDRIVE$HOMEPATH/covdir.cfg -_
	echo >$HOME/covdir.cfg -:
	covdir
	echo >$d/covdir.cfg -q
	covdir
	rm -r $d/covdir.cfg $TMPDIR/tmp $HOMEDRIVE$HOMEPATH/covdir.cfg
'<<.
	C-Cover Coverage by Directory %ver
	error cov137: unknown option "-,"
	error cov137: unknown option "-_"
.
#184
run -n '# -Fo after the source
	rm -f t_tmp.obj
	covc -q cl -nologo -c calc1.c -Fot_tmp.obj
	covmgr -qr t_tmp.obj
	rm t_tmp.obj
'<<.
.
#185
run -n '# cl -FI
	echo >t_fi.h "int i = 1;"
	echo >t_fi.c "void main(){}"
	covc -q cl -c -nologo -FIt_fi.h t_fi.c
	covc -q cl -c -nologo -FI t_fi.h t_fi.c
	rm t_fi.*
'<<.
.
#186
run -u '# GNU pragma implementation
	echo > t_tmp.cxx "#pragma implementation \"t1.h\""
	echo >>t_tmp.cxx "#include <stdio.h>"
	echo >>t_tmp.cxx "#pragma implementation \"t2.h\""
	echo >>t_tmp.cxx "int main() { return 0; }"
	covc -q c++ -c t_tmp.cxx
	rm t_tmp.*
	covmgr -qr t_tmp.cxx
'<<.
.
#187
run -n '# C++ syntax
	covc -q cl -c -nologo t_parse.cpp
	covbr -q -c0 -e2 t_parse.cpp
	covmgr -q -r t_parse.cpp
	rm t_parse.obj
'<<.
	-->       4    static int f() { return 0; }
	-->       7  int main()
	-->      11a   if (
	  -->    11b       a &&
	  -->    11c            C<int,int>::f()) {
	-->      17    case 1:
	-->      19    }
.
#188
run -n '# exclude NT ddk functions
	covc -q cl -c -nologo t_specialNTk.c
	covsrc -qkc t_specialNTk.c
	covmgr -qr t_specialNTk.c
'<<.
	"t_specialNTk.c",0,1,0%,0,0,""
.
#189
run -n '# Build kernel mode driver
	covc -q cl -Gi- -Gm- -Gz -Od -QIf -W3 -Zl -Zp8 -c -cbstring -nologo t_ntk.c
	covlink -v link -nologo \
		-align:0x20 \
		-base:0x10000 \
		-debug:notmapped,full \
		-debugtype:both \
		-driver \
		-entry:DriverEntry@8 \
		-fullbuild \
		-ignore:4078 \
		-incremental:no \
		-machine:i386 \
		-merge:.rdata=.text \
		-merge:_PAGE=PAGE \
		-merge:_TEXT=.text \
		-nodefaultlib \
		-optidata \
		-osversion:4.00 \
		-out:t_ntk.sys \
		-pdb:none \
		-release \
		-subsystem:native,4.00 \
		-version:4.00 \
		t_ntk.obj \
		2>&1 | fgrep -v -e cov907 -e cov910 -e cov923 -e cov926
	rm t_ntk.sys
	covmgr -qr t_ntk.*
'<<.
	C-Cover Link %ver
	note cov925: bin directory .
	note cov902: running %.../link -libpath:./../lib/ -nologo -align:0x20 -base:0x10000 -debug:notmapped,full -debugtype:both -driver -entry:cov_DriverEntry@8 -fullbuild -ignore:4078 -incremental:no -machine:i386 -merge:.rdata=.text -merge:_PAGE=PAGE -merge:_TEXT=.text -nodefaultlib -optidata -osversion:4.00 -out:t_ntk.sys -pdb:none -release -subsystem:native,4.00 -version:4.00 t_ntk.obj -nodefaultlib:covntm.lib covntkm.lib ntoskrnl.lib covrt-nt-link.obj
	note cov909: exit status 0
.
#190
run -n '# Build kernel mode driver in C++ and without -driver
	covc -q cl -c -Gz -nologo -TP t_ntk.c
	covlink -v link -nologo \
		-base:0x10000 \
		-debug:notmapped,full \
		-debugtype:both \
		-entry:DriverEntry@8 \
		-fullbuild \
		-ignore:4078 \
		-incremental:no \
		-machine:i386 \
		-merge:.rdata=.text \
		-merge:_PAGE=PAGE \
		-merge:_TEXT=.text \
		-nodefaultlib \
		-optidata \
		-osversion:4.00 \
		-out:t_ntk.sys \
		-pdb:none \
		-release \
		-subsystem:native,4.00 \
		-version:4.00 \
		t_ntk.obj \
		2>&1 | fgrep -v -e cov907 -e cov910 -e cov923 -e cov926
	rm t_ntk.sys
	covmgr -qr t_ntk.*
'<<.
	C-Cover Link %ver
	note cov925: bin directory .
	note cov902: running %.../link -libpath:./../lib/ -nologo -base:0x10000 -debug:notmapped,full -debugtype:both -entry:cov_DriverEntry@8 -fullbuild -ignore:4078 -incremental:no -machine:i386 -merge:.rdata=.text -merge:_PAGE=PAGE -merge:_TEXT=.text -nodefaultlib -optidata -osversion:4.00 -out:t_ntk.sys -pdb:none -release -subsystem:native,4.00 -version:4.00 t_ntk.obj -nodefaultlib:covntm.lib covntkm.lib ntoskrnl.lib covrt-nt-link.obj
	note cov909: exit status 0
.
#191
run -n '# Graphics driver
	cl -GF -Gz -Od -Oi -Oy- -QIf -QIfdiv- -W3 -Z7 -Zl -Zp8 -c -cbstring -nologo t_ntGraphic.c
	covlink -v link \
		-align:0x20 \
		-base:0x10000 \
		-debug:notmapped,full \
		-debugtype:both \
		-dll \
		-driver \
		-entry:DrvEnableDriver@12 \
		-force:multiple \
		-fullbuild \
		-ignore:4078 \
		-incremental:no \
		-machine:i386 \
		-merge:.rdata=.text \
		-merge:_PAGE=PAGE \
		-merge:_TEXT=.text \
		-nodefaultlib \
		-opt:ref \
		-optidata \
		-osversion:4.00 \
		-out:t_ntGraphic.dll \
		-pdb:none \
		-release \
		-subsystem:native,4.00 \
		-version:4.00 \
		win32k.lib \
		t_ntGraphic.obj \
		2>&1 | fgrep -v -e cov907 -e cov910 -e cov923 -e cov926
'<<.
	C-Cover Link %ver
	note cov925: bin directory .
	note cov902: running %.../link -libpath:./../lib/ -align:0x20 -base:0x10000 -debug:notmapped,full -debugtype:both -dll -driver -entry:cov_DrvEnableDriver@12 -force:multiple -fullbuild -ignore:4078 -incremental:no -machine:i386 -merge:.rdata=.text -merge:_PAGE=PAGE -merge:_TEXT=.text -nodefaultlib -opt:ref -optidata -osversion:4.00 -out:t_ntGraphic.dll -pdb:none -release -subsystem:native,4.00 -version:4.00 win32k.lib t_ntGraphic.obj -nodefaultlib:covntm.lib covntGraphicm.lib covrt-nt-link.obj
	note cov909: exit status 0
.
#192
run -n '# Kernel driver entry not supported
	covlink link \
		-force:multiple \
		-release \
		-base:0x20000000 \
		-debug \
		-entry:entryPoint@12 \
		-machine:I386 \
		-nodefaultlib \
		-osversion:4.00 \
		-pdb:none \
		-subsystem:native,4.00 \
		-version:4.0 \
		win32k.lib \
		t_ntGraphic.obj
'<<.
	C-Cover Link %ver
	warning cov818: driver entry entryPoint not supported
.
#193
run -n '# covlink warning 817
	covlink link \
		-force:multiple \
		-release \
		-base:0x20000000 \
		-debug \
		-machine:I386 \
		-nodefaultlib \
		-osversion:4.00 \
		-pdb:none \
		-subsystem:native,4.00 \
		-version:4.0 \
		win32k.lib \
		t_ntGraphic.obj
	rm -f t_ntGraphic.dll t_ntGraphic.obj 
'<<.
	C-Cover Link %ver
	warning cov817: no driver entry point specified
	LINK : error LNK2001: unresolved external symbol _NtProcessStartup
	t_ntGraphic.exe : fatal error LNK1120: 1 unresolved externals
.
#194
run -n '# All compiler include files
	covc -q cl -nologo -c -GX t_include.cpp
'<<.
.
#195
run -n '# Cygnus cygwin32
	covc -v gcc -o calc1.exe calc1.c 2>&1 | grep -e 902 -e 909
	covmgr -q -r calc1.c
'<<.
	note cov902: running %cc -DCCOVER=%ver -E calc1.c >tmp/covc/calc1.i
	note cov902: running %cc -o calc1.exe tmp/covc/calc1.c %bindir/../lib/libcov.a
	note cov909: exit status 0
.
#196
run -u '# Column position computation
	covc -q $cc ${o}t_column t_column.c
	t_column
	covbr -q -c0 -e1 t_column.c | sed "s,  [0-9][0-9]*,,"
	covfn -q -c -k -s t_column.c
	covmgr -q -r t_column.c
	rm -f t_column.exe t_column.obj
'<<.
	X         void macro_complex()
	-->F   a  if (
	  -->t b      macro1(True) && False) ;
	  -->f c
	-->F   a  if (
	  -->t b      True &&
	  -->t c              macro1(True) && False) ;
	  -->f d
	-->F   a  if (macro2(True, False) && False) ;
	  -->t b
	  -->f c
	  -->  d
	-->F   a  if (
	  -->t b      True && macro2(True, False)) ;
	  -->t c
	  -->f d
	-->F   a  if (
	  -->t b      True &&
	  -->t c              True && macro2(True, False)) ;
	  -->t d
	  -->f e
	-->F   a  if (macro3(False) && False) ;
	  -->f b
	  -->f c
	-->T   a  if (
	  -->t b      True && macro3(False)) ;
	  -->f c
	-->T   a  if (
	  -->t b      True &&
	  -->t c              True && macro3(False)) ;
	  -->f d
	X        void macro_iso646()
	-->F   a  if (
	  -->t b      True and
	  -->t c               True and
	  -->f d                        False) ;
	-->F   a  if ((
	  -->t b       not False or
	  -->  c                    not False) and
	  -->f d                                   not True) ;
	-->F   a  if ((
	  -->f b       False or
	  -->f c                False) &&
	  -->  d                          True) ;
	-->F   a  if ((
	  -->f b       False ||
	  -->f c                False) and
	  -->  d                           True) ;
	X        void condition_const()
	-->F   a  if (
	  -->t b      True &&
	  -->f d                   False) ;
	-->F   a  if (
	  -->t b      (True) == (1) &&
	  -->f d                              (False)) ;
	-->F   a  if (
	  -->t b      True &&
	  -->f     False) ;
	-->T   a  if (
	  -->t b      True &&
	-->T   a  if (
	  -->t c           True) ;
	X        void spaceBeforeNewLine()
	-->F   a  if (
	  -->f b      a && /* space here */
	  -->      b)
	-->F   a  if (
	  -->f b      a && ( /* space here */
	  -->  a   b ||
	  -->  b        c))
	-->F      if (a = /* space here */
	  -->f a   b &&
	  -->  b        c)
	-->F      if (a, /* space here */
	  -->f a   b &&
	  -->  b        c)
	-->      void macro_null()
	-->    a  if (
	  -->  b      p == 0 &&
	  -->  c                False)
	-->    a  if (
	  -->  b      p == NULL &&
	  -->  c                   False)
	-->    a  if (
	  -->  b      p == NULL and
	  -->  c                    False)
	-->    a  if (
	  -->  b      p == NULL && False)
	  -->  c
	X        int main()
	"macro_complex","t_column.c",8,1,29,60,48%
	"macro_iso646","t_column.c",20,1,13,32,40%
	"condition_const","t_column.c",33,1,13,26,50%
	"spaceBeforeNewLine","t_column.c",46,1,8,26,30%
	"macro_null","t_column.c",65,0,0,24,0%
	"main","t_column.c",81,1,0,0,""
.
#197
run -n '# Column position computation
	covc -q $cc ${o}t_column t_column.c
	t_column
	covbr -q -c0 -e1 t_column.c | sed "s,  [0-9][0-9]*,,"
	covfn -q -c -k -s t_column.c
	covmgr -q -r t_column.c
	rm -f t_column.exe t_column.obj
'<<.
	X         void macro_complex()
	-->F   a  if (
	  -->t b      macro1(True) &&
	  -->f c                      False) ;
	-->F   a  if (
	  -->t b      True &&
	  -->t c              macro1(True) &&
	  -->f d                              False) ;
	-->F   a  if (
	  -->f b      macro2(True, False) &&
	  -->  c                             False) ;
	-->F   a  if (
	  -->t b      True &&
	  -->f c              macro2(True, False)) ;
	-->F   a  if (
	  -->t b      True &&
	  -->t c              True &&
	  -->f d                      macro2(True, False)) ;
	-->F   a  if (
	  -->t b      macro3(False) &&
	  -->f c                       False) ;
	-->T   a  if (
	  -->t b      True &&
	  -->t c              macro3(False)) ;
	-->T   a  if (
	  -->t b      True &&
	  -->t c              True &&
	  -->t d                      macro3(False)) ;
	X        void macro_iso646()
	-->F   a  if (
	  -->t b      True and
	  -->t c               True and
	  -->f d                        False) ;
	-->F   a  if ((
	  -->t b       not False or
	  -->  c                    not False) and
	  -->f d                                   not True) ;
	-->F   a  if ((
	  -->f b       False or
	  -->f c                False) &&
	  -->  d                          True) ;
	-->F   a  if ((
	  -->f b       False ||
	  -->f c                False) and
	  -->  d                           True) ;
	X        void condition_const()
	-->F   a  if (
	  -->t b      True &&
	  -->f d                   False) ;
	-->F   a  if (
	  -->t b      (True) == (1) &&
	  -->f d                              (False)) ;
	-->F   a  if (
	  -->t b      True &&
	  -->f     False) ;
	-->T   a  if (
	  -->t b      True &&
	-->T   a  if (
	  -->t c           True) ;
	X        void spaceBeforeNewLine()
	-->F   a  if (
	  -->f b      a && /* space here */
	  -->      b)
	-->F   a  if (
	  -->f b      a && ( /* space here */
	  -->  a   b ||
	  -->  b        c))
	-->F      if (a = /* space here */
	  -->f a   b &&
	  -->  b        c)
	-->F      if (a, /* space here */
	  -->f a   b &&
	  -->  b        c)
	-->      void macro_null()
	-->    a  if (
	  -->  b      p == 0 &&
	  -->  c                False)
	-->    a  if (
	  -->  b      p == NULL &&
	  -->  c                   False)
	-->    a  if (
	  -->  b      p == NULL and
	  -->  c                    False)
	-->    a  if (
	  -->  b      p == NULL &&
	  -->  c                   False)
	X        int main()
	"macro_complex","t_column.c",8,1,26,54,48%
	"macro_iso646","t_column.c",20,1,13,32,40%
	"condition_const","t_column.c",33,1,13,26,50%
	"spaceBeforeNewLine","t_column.c",46,1,8,26,30%
	"macro_null","t_column.c",65,0,0,24,0%
	"main","t_column.c",81,1,0,0,""
.
#198
run '# Last pragma between function header and body
	echo >t_tmp.c "void f(){}"
	echo >>t_tmp.c "#pragma"
	echo >>t_tmp.c "void g()"
	echo >>t_tmp.c "#pragma"
	echo >>t_tmp.c "{}"
	covc -q -t!t_tmp.c $cc -c t_tmp.c | grep -v "unknown pragma"
	rm t_tmp*
'<<.
.
#199
run '# Error 17, empty file
	rm -f $COVFILE
	covc -q $cc calc1.c
	rm $COVFILE
	covmgr -q -c -l
	COVERR=t_tmperr
	export COVERR
	echo q | calc1
	cat $COVERR
	rm $COVERR
'<<.
	Enter reverse-polish expressions, q to quit
	C-Cover %ver 17: Coverage file test.cov is empty.  (Error occurred at %...).
.
#200
run -n '# Multiple -Tp
	cp calc3.cpp calc3.c
	cp calc3stk.cpp calc3stk.c
	covc -v cl -c -nologo -Tp calc3.c -Tp calc3stk.c 2>&1 | grep -e 902 -e 909
	covc -v cl    -nologo -Tp calc3.c -Tp calc3stk.c 2>&1 | grep -e 902 -e 909
	covmgr -q -r calc3.c calc3stk.c
	rm -r calc3.c calc3stk.c
'<<.
	note cov902: running internalCpp -DCCOVER=%ver -c -nologo -Tp calc3.c -+cc cl
	note cov902: running %.../cl -c -nologo -Tp tmp/covc/calc3.c
	note cov902: running internalCpp -DCCOVER=%ver -c -nologo -Tp calc3stk.c -+cc cl
	note cov902: running %.../cl -c -nologo -Tp tmp/covc/calc3stk.c
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -nologo -Tp calc3.c -+cc cl
	note cov902: running internalCpp -DCCOVER=%ver -nologo -Tp calc3stk.c -+cc cl
	note cov902: running %.../cl -nologo -Tp tmp/covc/calc3.c -Tp tmp/covc/calc3stk.c -link covrt-nt-link.obj
	note cov909: exit status 0
.
#201
run '# autosave
	rm -f test.cov
	covc -q $cc -Dsys_$sys=1 ${o}t_sleep t_sleep.c
	covmgr -lv 2>&1 | grep autosave
	t_sleep&
	sleep 2
	covfn -qk t_sleep.c
	wait

	covclear -q
	covmgr -q -a0
	covmgr -lv 2>&1 | grep autosave
	t_sleep&
	sleep 2
	covfn -qk t_sleep.c
	wait

	covclear -q
	covmgr -q -a1
	covmgr -lv 2>&1 | grep autosave
	t_sleep&
	sleep 2
	covfn -qk t_sleep.c
	wait
	covmgr -qr t_sleep.c
'<<.
	autosave interval set to default (1 second)
	main      t_sleep.c       10  1 / 1       1 /     2 =  50%
	autosave disabled
	main      t_sleep.c       10  0 / 1       0 /     2 =   0%
	autosave interval set to 1 second
	main      t_sleep.c       10  1 / 1       1 /     2 =  50%
.
#202
run -n '# Windows CE link
	echo >t_tmp.cpp "#include <windows.h>"
	echo >>t_tmp.cpp "int WINAPI WinMain(HINSTANCE h1, HINSTANCE h2, wchar_t* cl, int cs) { return 0; }"
	covc -q cl -c -DUNDER_CE -DUNICODE -D_WIN32_WCE_EMULATION -Dx86=1 -I"/Program Files/Windows CE Tools/wce211/ms palm size pc/include" -nologo t_tmp.cpp
	covlink -v link -subsystem:windows -windowsce:emulation t_tmp.obj "c:/Program Files/Windows CE Tools/wce211/ms palm size pc/lib/x86em/coredll.lib" 2>&1 |
		grep -e 902 -e 909
	rm t_tmp.*
	covmgr -q -r t_tmp.cpp
'<<.
	note cov902: running %.../link -libpath:./../lib/ -subsystem:windows -windowsce:emulation t_tmp.obj "c:/Program Files/Windows CE Tools/wce211/ms palm size pc/lib/x86em/coredll.lib" -nodefaultlib:covntm.lib covWindowsCEm.lib covrt-nt-link.obj
	note cov909: exit status 0
.
#203
run -n '# Windows CE symbols
	nm -u ../lib/covWindowsCEm.lib
'<<.
	covrt.obj:
	    0:0000000 U _CloseHandle@4
	    0:0000000 U _CreateFileW@28
	    0:0000000 U _CreateMutexW@12
	    0:0000000 U _CreateThread@24
	    0:0000000 U _EnterCriticalSection@4
	    0:0000000 U _ExitThread@4
	    0:0000000 U _GetExitCodeThread@8
	    0:0000000 U _GetLastError@0
	    0:0000000 U _InitializeCriticalSection@4
	    0:0000000 U _LeaveCriticalSection@4
	    0:0000000 U _MessageBoxW@16
	    0:0000000 U _ReadFile@20
	    0:0000000 U _ReleaseMutex@4
	    0:0000000 U _SetFilePointer@16
	    0:0000000 U _Sleep@4
	    0:0000000 U _WaitForSingleObject@8
	    0:0000000 U _WriteFile@20
	    0:0000000 U _atexit
.
#204
run -u '# Sun ProWorks C++ templates
	if [ $uname = SunOS ] && $which CC >/dev/null
	then
		rm -fr Templates.DB
		covc -q CC t_class.cpp 2>&1 | fgrep -v "template manager :"
		rm a.out
		covmgr -q -r t_class.cpp
	fi
'<<.
.
#205
run -u '# Solaris CC shared library
	if [ $uname = SunOS ] && $which CC >/dev/null
	then
		covc -q CC -c -Kpic calc1.c
		covc -q CC -G -o libcalc1.so calc1.o
		rm calc1.o libcalc1.so
		covmgr -q -r calc1.c
	fi
'<<.
.
#206
run -n '# Run regsvr32 on a C-Cover DLL
	echo >t_tmp.c "
		long __declspec(dllexport) __stdcall DllRegisterServer() { return 0; }"
	echo >t_tmp.def "
		LIBRARY t_tmp.dll
		EXPORTS
			DllRegisterServer @1 PRIVATE"
	covc -q cl -c -nologo t_tmp.c
	covlink -q link -def:t_tmp.def -dll -nologo t_tmp.obj
	regsvr32 -s t_tmp.dll
	echo $?
	covfn -q -c -k t_tmp.c
	rm t_tmp.*
	covmgr -q -r t_tmp.c
'<<.
	ccoverEntryUser.c
	   Creating library t_tmp.lib and object t_tmp.exp
	0
	"DllRegisterServer","t_tmp.c",2,1,0,0,""
.
#207
run '# Constant probe is last
	echo >t_tmp.c "int main() { if (0) ; return 0; }"
	covc -q $cc ${o}t_tmp t_tmp.c
	t_tmp
	rm t_tmp*
'<<.
.
#208
run -n '# Create precompiled header with space in path
	echo "#include \"t_pch.h\"" >"t_ pch.cpp"
	echo >t_pch.h
	covc -q cl -c -nologo -Yct_pch.h "t_ pch.cpp"
	rm t_pch* "t_ pch"*
'<<.
.
#209
run -n '# Disable C4505
	echo >t_tmp.cpp "template<class T> void f() { int i = 0; if (i) ; }"
	echo >>t_tmp.cpp "int main() { return 0; }"
	echo >>t_tmp.cpp "#pragma warning(default:4505)"
	covc -q cl -c -nologo -W4 -WX t_tmp.cpp
	rm t_tmp.*
'<<.
.
#210
run -n '# autosave with DLLs
	echo >t_tmp1.c "#include <windows.h>"
	echo >>t_tmp1.c "__declspec(dllimport) void dll2();"
	echo >>t_tmp1.c "__declspec(dllimport) void dll3();"
	echo >>t_tmp1.c "int main() { dll2(); dll3(); dll4(); Sleep(3500); return 0; }"
	echo >t_tmp2.c "#include <windows.h>"
	echo >>t_tmp2.c "__declspec(dllexport) void dll2() { }"
	echo >>t_tmp2.c "BOOL WINAPI DllMain(HINSTANCE h, DWORD why, void* p) {"
	echo >>t_tmp2.c "if (why == DLL_PROCESS_DETACH) ;"
	echo >>t_tmp2.c "return 1; }"
	sed s,dll2,dll3,g t_tmp2.c >t_tmp3.c
	sed s,dll2,dll4,g t_tmp2.c >t_tmp4.c

	# Set autosave to default
	rm -f $COVFILE
	# Build some DLLs with covlink
	covc -q cl -c -nologo t_tmp2.c
	covlink -q link -dll -nologo t_tmp2.obj -entry:DllMain@12
	covc -q cl -c -nologo t_tmp3.c
	covlink -q link -dll -nologo t_tmp3.obj
	# Build one DLL without covlink
	covc -q cl -LD -nologo t_tmp4.c

	# EXE not compiled with C-Cover
	cl -nologo t_tmp1.c t_tmp2.lib t_tmp3.lib t_tmp4.lib
	t_tmp1&
	sleep 3
	covbr -q -c0 !t_tmp[34].c:DllMain
	wait
	echo "-"
	covbr -q -c0 !t_tmp[34].c:DllMain

	# EXE compiled with C-Cover
	covclear -q
	covc -q cl -nologo t_tmp1.c t_tmp2.lib t_tmp3.lib t_tmp4.lib
	t_tmp1&
	sleep 2
	echo "-"
	covbr -q -c0 !t_tmp[34].c:DllMain
	wait
	echo "-"
	covbr -q -c0 !t_tmp[34].c:DllMain
	rm t_tmp*
'<<.
	ccoverEntryUser.c
	   Creating library t_tmp2.lib and object t_tmp2.exp
	ccoverEntryUser.c
	   Creating library t_tmp3.lib and object t_tmp3.exp
	error cov153: DLL created by cl.exe not safe; use link.exe -dll
	   Creating library t_tmp4.lib and object t_tmp4.exp
	X         2  __declspec(dllexport) void dll2() { }
	X         3  BOOL WINAPI DllMain(HINSTANCE h, DWORD why, void* p) {
	-->F      4  if (why == DLL_PROCESS_DETACH) ;
	X         2  __declspec(dllexport) void dll3() { }
	X         2  __declspec(dllexport) void dll4() { }
	-
	X         2  __declspec(dllexport) void dll2() { }
	X         3  BOOL WINAPI DllMain(HINSTANCE h, DWORD why, void* p) {
	TF        4  if (why == DLL_PROCESS_DETACH) ;
	X         2  __declspec(dllexport) void dll3() { }
	X         2  __declspec(dllexport) void dll4() { }
	-
	X         4  int main() { dll2(); dll3(); dll4(); Sleep(3500); return 0; }
	X         2  __declspec(dllexport) void dll2() { }
	X         3  BOOL WINAPI DllMain(HINSTANCE h, DWORD why, void* p) {
	-->F      4  if (why == DLL_PROCESS_DETACH) ;
	X         2  __declspec(dllexport) void dll3() { }
	X         2  __declspec(dllexport) void dll4() { }
	-
	X         4  int main() { dll2(); dll3(); dll4(); Sleep(3500); return 0; }
	X         2  __declspec(dllexport) void dll2() { }
	X         3  BOOL WINAPI DllMain(HINSTANCE h, DWORD why, void* p) {
	TF        4  if (why == DLL_PROCESS_DETACH) ;
	X         2  __declspec(dllexport) void dll3() { }
	X         2  __declspec(dllexport) void dll4() { }
.
#211
run -n '# Resource-only DLL
	echo >t_tmp.res
	rc t_tmp.res
	covlink -q link -dll -machine:ix86 -noentry t_tmp.res
	rm t_tmp*
'<<.
.
#212
run -n '# covlink DLL object not compiled with C-Cover
	cl -c -nologo t_dll.cpp
	covlink -q link -dll -ignore:4089 -machine:ix86 t_dll.obj
'<<.
	ccoverEntryUser.c
	   Creating library t_dll.lib and object t_dll.exp
.
#213
run '# Empty source file
	echo >t_tmp.c
	covc -q $cc -c t_tmp.c
	rm t_tmp.*
'<<.
.
#214
run -n '# cl option -EH
	echo >t_tmp.c
	covc -v $cc -c -EHa t_tmp.c 2>&1 | grep 902
	rm t_tmp.*
'<<.
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -EHa t_tmp.c -+cc cl
	note cov902: running %cc -c -EHa tmp/covc/t_tmp.c
.
#215
run -n '# ATL with -D_ATL_MIN_CRT
	echo >t_tmp.cpp "
		#include <atlbase.h>
		CComModule _Module;
		#include <atlimpl.cpp>
		void __declspec(dllexport) dll() { }"
	covc -q cl -c -D_ATL_MIN_CRT -D_USRDLL -nologo -MT t_tmp.cpp
	covlink -v link -dll -implib:t_tmp.lib -nologo t_tmp.obj 2>&1 |
		fgrep -e 902 -e 909 -e 930
	echo >t_tmpMain.cpp "
		void __declspec(dllimport) dll();
		int main() {
			dll();
			return 0;
		}" &&
	cl -nologo t_tmpMain.cpp t_tmp.lib &&
	t_tmpMain &&
	br t_tmp.cpp
'<<.
	note cov902: running %.../cl -c -nologo -Zl -Fotmp/ccoverEntryUser.obj tmp/ccoverEntryUser.c
	note cov902: running %.../cl -c -nologo -Zl -Fotmp/ccoverEntryUser.obj tmp/ccoverEntryUser.c
	note cov930: automatic workaround for atexit
	note cov902: running %.../link -libpath:./../lib/ -dll -implib:t_tmp.lib -nologo t_tmp.obj -entry:cov_DllMain@12 tmp/ccoverEntryUser.obj covrt-nt-atexit.obj covntm.lib covrt-nt-link.obj
	note cov909: exit status 0
	X void __declspec(dllexport)dll()
.
#216
run '# "#line" with line number out of range
	echo "
			int Begin000;
		#line 0
			int line0;
		#line 9
			int line9;
		#line 11
			int line11;
			int line12;
		#line 10
			int line10a;
		#line 10
			int line10b;
		#line 32767
			int line32767;
		#line 32768
			int line32768;
		#line 32770
			int line32770;
		#line 32776
			int line32776;
		#line 32768
			int line32768;
		#line 32767
			int End000;
		" > t_tmp.c
	if [ $sys = n ]
	then
		# Make sure no warning C4112
		w4=-W4
	else
		w4=
	fi
	covc -l -p- -q $cc -c $w4 t_tmp.c
	sed -n "/Begin000/,/End000/p" tmp/covc/t_tmp.c | sed "s,^[ 	]*,^,"
'<<.
	^int Begin000;
	^int line0;
	^#line 9
	^int line9;
	^
	^int line11;
	^int line12;
	^#line 10
	^int line10a;
	^#line 10
	^int line10b;
	^#line 32767
	^int line32767;
	^int line32768;
	^
	^int line32770;
	^
	^
	^
	^
	^
	^int line32776;
	^int line32768;
	^#line 32767
	^int End000;
.
#217
run '# "#line" with source not given
	echo "
		int Begin000;
		#line 1 \"a\"
		#line 2
			int line_a_2;
		#line 1 \"b\"
		#line 2
			int line_b_2;
		int End000;
		" > t_tmp.c
	covc -l -p- -q $cc -c t_tmp.c
	sed -n "/Begin000/,/End000/p" tmp/covc/t_tmp.c | sed "s,^[ 	]*,^,"
'<<.
	File open error: a: No such file or directory
	File open error: b: No such file or directory
	^int Begin000;
	^#line 2 "a"
	^int line_a_2;
	^#line 2 "b"
	^int line_b_2;
	^int End000;
.
#218
run -n '# FreeLibrary and friends
	echo "
		#include <windows.h>
		int main() {
			HINSTANCE handle = LoadLibraryEx(NULL, NULL, DONT_RESOLVE_DLL_REFERENCES);
			FreeLibrary(handle);
			FreeLibraryAndExitThread(handle, 0);
			return 0;
		}
		" > t_tmp.cpp
		covc -q cl -c -DSTRICT -nologo -W3 t_tmp.cpp
'<<.
.
#219
run -n '# cl with -LD
	echo "int main() { return 0; }" >t_tmp.c
	covc cl -LD -nologo t_tmp.c
	covc cl -LDd -nologo t_tmp.c
'<<.
	C-Cover Compile C/C++ %ver
	error cov153: DLL created by cl.exe not safe; use link.exe -dll
	C-Cover Compile C/C++ %ver
	error cov153: DLL created by cl.exe not safe; use link.exe -dll
.
#220
run -n '# Backend precompiled header
	mkdir pch.tmp
	(
	covc -v $cc -c -YXctype.h calc1.c
	covc -v $cc -c -Ycctype.h calc1.c
	covc -v $cc -c -Yuctype.h calc1.c
	covc -v $cc -c -Ycctype.h -Fppch.tmp/x.pch calc1.c
	covc -v $cc -c -Yuctype.h -Fppch.tmp/x.pch calc1.c
	covc -v --no-cpp $cc -c -Yuctype.h -Fppch.tmp/x.pch calc1.c
	rm pch.tmp/x-ccover.pch ctype-ccover.h
	covc -v $cc -c -Yuctype.h -Fppch.tmp/x.pch calc1.c
	) 2>&1 | grep -e "90[29]" -e 820
	rm -r pch.tmp ctype-ccover.pch ctype.pch
'<<.
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c
	note cov909: exit status 0
	note cov902: running %cc -c -Ycctype.h calc1.c
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Ycctype.h calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c -I. -Yc%cwd/ctype-ccover.h -Fpctype-ccover.pch
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Yuctype.h calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c -Yu%cwd/ctype-ccover.h -Fpctype-ccover.pch
	note cov909: exit status 0
	note cov902: running %cc -c -Ycctype.h -Fppch.tmp/x.pch calc1.c
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Ycctype.h -Fppch.tmp/x.pch calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c -I. -Yc%cwd/pch.tmp/ctype-ccover.h -Fppch.tmp/x-ccover.pch
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Yuctype.h -Fppch.tmp/x.pch calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c -Yu%cwd/pch.tmp/ctype-ccover.h -Fppch.tmp/x-ccover.pch
	note cov909: exit status 0
	note cov902: running %.../cl -nologo -P -DCCOVER=%ver -GX -nologo -c calc1.c
	note cov902: running %cc -c tmp/covc/calc1.c
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c calc1.c -+cc cl
	note cov902: running %cc -c tmp/covc/calc1.c
	note cov909: exit status 0
.
#221
run -n '# Backend pch with pragma hdrstop
	echo >covxCheckTmpC.cpp "
		#define macro value
		#pragma hdrstop
		"
	echo >covxCheckTmpU.cpp "
		#pragma hdrstop
		void macro() { }
		"
	(
	covc -v $cc -c -Yc -FpcovxCheckTmp.pch covxCheckTmpC.cpp
	covc -v $cc -c -Yu -FpcovxCheckTmp.pch covxCheckTmpU.cpp
	) 2>&1 | grep -e "90[29]" -e 820
	covfn -q -k covxCheckTmpU.cpp
'<<.
	note cov902: running %cc -c -Yc -FpcovxCheckTmp.pch covxCheckTmpC.cpp
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Yc -FpcovxCheckTmp.pch covxCheckTmpC.cpp -+cc cl
	note cov902: running %cc -c tmp/covc/covxCheckTmpC.cpp -I. -Yc%cwd/covxCheckTmpC-ccover.h -FpcovxCheckTmp-ccover.pch
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Yu -FpcovxCheckTmp.pch covxCheckTmpU.cpp -+cc cl
	note cov902: running %cc -c tmp/covc/covxCheckTmpU.cpp -Yu%cwd/covxCheckTmpU-ccover.h -FpcovxCheckTmp-ccover.pch
	note cov909: exit status 0
	value     covxCheckTmpU.cpp     3  0 / 1       0 /     0
.
#222
run -n '# Backend precompiled header with pch #include not first
	echo > t_pch.cpp "#include \"t_pchEmpty.h\""
	echo >>t_pch.cpp "#include \"t_pch.h\""
	echo >t_pch.h "int a;"
	echo >t_pchEmpty.h
	echo > t_pchTmp.cpp "#include \"t_pch.h\""
	echo >>t_pchTmp.cpp "int* b = &a;"
	covc -q $cc -c -Yct_pch.h t_pch.cpp
	covc -q $cc -c -Yut_pch.h t_pchTmp.cpp
	rm t_pch.pch t_pch-ccover.pch
'<<.
.
#223
run -n '# Backend precompiled header with pch in -I directory
	mkdir pch.tmp
	mv t_pch.h pch.tmp
	(
	covc -v $cc -c -Ipch.tmp -Yct_pch.h t_pch.cpp
	covc -v $cc -c -I pch.tmp -Yct_pch.h t_pch.cpp
	covc -v $cc -c -Ipch.tmp -Yut_pch.h t_pchTmp.cpp
	) 2>&1 | grep -e "90[29]" -e 820
	rm t_pch.pch t_pch-ccover.pch
'<<.
	note cov902: running %cc -c -Ipch.tmp -Yct_pch.h t_pch.cpp
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Ipch.tmp -Yct_pch.h t_pch.cpp -+cc cl
	note cov902: running %cc -c -Ipch.tmp tmp/covc/t_pch.cpp -I. -Yc%cwd/t_pch-ccover.h -Fpt_pch-ccover.pch
	note cov909: exit status 0
	note cov902: running %cc -c -I pch.tmp -Yct_pch.h t_pch.cpp
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -I pch.tmp -Yct_pch.h t_pch.cpp -+cc cl
	note cov902: running %cc -c -I pch.tmp tmp/covc/t_pch.cpp -I. -Yc%cwd/t_pch-ccover.h -Fpt_pch-ccover.pch
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Ipch.tmp -Yut_pch.h t_pchTmp.cpp -+cc cl
	note cov902: running %cc -c -Ipch.tmp tmp/covc/t_pchTmp.cpp -Yu%cwd/t_pch-ccover.h -Fpt_pch-ccover.pch
	note cov909: exit status 0
.
#224
run -n '# Backend precompiled header with pch in subdirectory
	echo > t_pch.cpp "#include \"t_pchEmpty.h\""
	echo >>t_pch.cpp "#include \"pch.tmp/t_pch.h\""
	echo > t_pchTmp.cpp "#include \"pch.tmp/t_pch.h\""
	echo >>t_pchTmp.cpp "int* b = &a;"
	(
	covc -v $cc -c -Ycpch.tmp/t_pch.h t_pch.cpp
	covc -v $cc -c -Yupch.tmp/t_pch.h t_pchTmp.cpp
	) 2>&1 | grep -e "90[29]" -e 820
	rm -r pch.tmp t_pch*
'<<.
	note cov902: running %cc -c -Ycpch.tmp/t_pch.h t_pch.cpp
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Ycpch.tmp/t_pch.h t_pch.cpp -+cc cl
	note cov902: running %cc -c tmp/covc/t_pch.cpp -I. -Yc%cwd/pch.tmp/t_pch-ccover.h -Fppch.tmp/t_pch-ccover.pch
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c -Yupch.tmp/t_pch.h t_pchTmp.cpp -+cc cl
	note cov902: running %cc -c tmp/covc/t_pchTmp.cpp -Yu%cwd/pch.tmp/t_pch-ccover.h -Fppch.tmp/t_pch-ccover.pch
	note cov909: exit status 0
.
#225
run -n '# covc option -i
	(
	covc -v -icalc1.c $cc -c calc1.c
	covc -v -icalc1.c,calc3.cpp $c2 calc3.cpp calc3stk.cpp
	covc -v -icalc1.c -icalc3.cpp $c2 -c calc3.cpp calc3stk.cpp
	) 2>&1 | fgrep -e 902 -e 909
'<<.
	note cov902: running %cc -c calc1.c
	note cov909: exit status 0
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo calc3stk.cpp -+cc cl
	note cov902: running %cc calc3.cpp tmp/covc/calc3stk.cpp -link covrt-nt-link.obj
	note cov909: exit status 0
	note cov902: running %cc -c calc3.cpp
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c calc3stk.cpp -+cc cl
	note cov902: running %cc -c tmp/covc/calc3stk.cpp
	note cov909: exit status 0
.
#226
run -n '# Backend precompiled header found in correct directory
	mkdir t_a
	echo >t_tmp.h "int i;"
	echo >t_tmp.c "#include \"t_tmp.h\""
	echo >>t_tmp.c "int* p = &i;"
	covc -q $cc -c -Fpt_a/p.pch -Yct_tmp.h t_tmp.c
	mkdir t_b
	mv -r t_a t_b
	echo >t_tmp.h "int j;"
	echo >t_tmp.c "#include \"t_tmp.h\""
	echo >>t_tmp.c "int* p = &j;"
	mkdir t_a
	covc -q $cc -c -Fpt_a/p.pch -Yct_tmp.h t_tmp.c -It_b
	rm -r t_a t_b
'<<.
.
#227
run '# Avoid C2362
	echo >t_tmp.cpp "
		void f() {
			int i = 0;
			goto L1;
			switch (i) {
		L1:
			case 1:
				break;
			}
		}
		"
	covc -q $c2 -c t_tmp.cpp
'<<.
.
#228
run -n '# Compiler option -Tc,-Tp with unusual suffix
	cp calc1.c t_tmp.src
	covc -q $cc -c -Tct_tmp.src
	covsrc -q -k t_tmp.src
	covc -q $cc -c -Ycctype.h -Tc t_tmp.src
	covsrc -q -k t_tmp.src
'<<.
	t_tmp.src       0 /     4 =   0%       0 /    34 =   0%
	t_tmp.src       0 /     4 =   0%       0 /    34 =   0%
.
#229
run -n '# cl -u
	covc $cc -c -u calc1.c
'<<.
	C-Cover Compile C/C++ %ver
	warning cov822: removing incompatible cl option -u
.
#230
run -n '# #import
	covc -q $cc -c t_import.cpp
	rm hhctrl.tl?
'<<.
.
#231
run -n '# Backend precompiled header does not cause C4049
	echo >t_tmp.cpp "
		#include <stdio.h>
		int main() {
			int nl = 10;
			for (int i = 0; i <= 0x10000; i++) {
				putchar(nl);
			}
			printf(\"#if 0\");
			putchar(nl);
			printf(\"#endif\");
			putchar(nl);
			return 0; }"
	$cc t_tmp.cpp
	t_tmp >t_tmp.h
	echo >t_tmp.c "
		#include \"t_tmp.h\"
		"
	covc -q $cc -c -Yct_tmp.h t_tmp.c
'<<.
.
#232
run -n '# Backend precompiled header does not cause C4112
	echo >t_tmp.h "
		#line 32767
		int i1;
		#include \"nul\"
		int i2;"
	echo >t_tmp.c "
		#include \"t_tmp.h\"
		int i3;"
	covc -q $cc -c -W4 -Yct_tmp.h t_tmp.c
'<<.
.
#233
run -n '# Option looks like source /..source.c
	echo "#include \"t_tmp.c\"" >t_tmp.cpp
	echo >t_tmp.c
	covc -q $cc -c /Yct_tmp.c t_tmp.cpp
'<<.
.
#234
run -n '# Variable defined in Microsoft C __except clause
	echo >t_tmp.c "
		void f() {
			__try { }
			__except(1) { int i; }
			/* Nested */
			__try { }
			__except (1) {
				__try { }
				__except (1) { }
			}
		}"
	covc -q $cc -c t_tmp.c
	br t_tmp.c
'<<.
	. void f(){
	. __except(1){int i;}
	. __except(1){
	. __except(1){}
.
#235
run '# Verify failure analysis does not leave object
	echo >t_tmp1.c "
		#if CCOVER
			/* Preprocess time error */
			#error
		#endif
		"
	cp t_tmp1.c t_tmp2.c
	echo >t_tmp3.c "
		#if CCOVER
			/* Compile time error */
			error error
		#endif
		"
	cp t_tmp3.c t_tmp4.c
	covc -q $cc -c t_tmp1.c          >$nul 2>&1
	covc -q $cc -c t_tmp1.c t_tmp2.c >$nul 2>&1
	covc -q $cc -c t_tmp3.c          >$nul 2>&1
	covc -q $cc -c t_tmp3.c t_tmp4.c >$nul 2>&1
	for i in t_tmp?.o*; do
		if [ -f $i ]; then
			echo exist $i
		fi
	done
'<<.
.
#236
run -n '# Backend precompiled header with -FI
	echo >t_tmp_Yc.c "#include \"t_tmp.h\""
	echo >t_tmp_Yu.c "
			an error
			#include \"t_tmp.h\"
			int* p = &i;
			int* q = &j;
			"
	echo >t_tmp.h "int i;"
	echo >t_tmp_fi.h "int j;"
	rm -f t_tmp.pch
	covc -q $cc -c -FIt_tmp_fi.h -Yct_tmp.h t_tmp_Yc.c
	covc -q $cc -c -FIt_tmp_fi.h -Yut_tmp.h t_tmp_Yu.c
'<<.
.
#237
run -n '# switch with __int64
	echo >t_tmp.c "
		int main() {
			__int64 i = 0x123456789abcdef;
			switch (i) {
			case 0x123456789abcdef: break;
			default: break;
			}
			return 0;
		}
		"
	covc -q $cc t_tmp.c
	t_tmp
	br t_tmp.c
'<<.
	X int main(){
	X case 0x123456789abcdef: break;
	. default: break;
.
#238
run -n -u '# gcc __attribute__
	echo >t_tmp.c "
		void f() __attribute__((__noreturn__));
		void g() { }
		"
	covc -q gcc -c t_tmp.c
	br t_tmp.c
'<<.
	. void g(){}
.
#239
run -n '# Macro expansion does not cause probes
	echo >t_tmp.cpp "
		#define m1 if (a) ;
		#define m2 for (;a;) ;
		#define m3 switch (a) { case 1: break; }
		#define m4 try { } catch (...) { }
		#define m5 case 1: break;
		#define m6 void f() { }
		int main() {
			int a = 0;
			m1
			m2
			m3
			m4
			switch (a) {
			m5
			}
			return 0;
		}
		m6
		"
	covc -q $cc -c t_tmp.cpp
	covbr -c0 -e1 -q t_tmp.cpp
'<<.
	-->       8    int main() {
.
#240
run -n '# Structure packing
	for i in 1 2 4 8 16; do
		covmgr -q -r calc1.c >nul 2>&1
		rm -f calc1.exe
		covc -q $cc -Zp$i calc1.c
		echo 1= | calc1 >nul
		covfn -q -k calc1.c:main
	done
'<<.
	main      calc1.c         31  1 / 1      10 /    34 =  29%
	main      calc1.c         31  1 / 1      10 /    34 =  29%
	main      calc1.c         31  1 / 1      10 /    34 =  29%
	main      calc1.c         31  1 / 1      10 /    34 =  29%
	main      calc1.c         31  1 / 1      10 /    34 =  29%
.
#241
run -n '# Space in object path
	covmgr -q -r t_tmp.c 2>nul
	mkdir "d i r"
	cp -f calc1.c t_tmp.c
	covc -q $cc -c -Fo"d i r"/ t_tmp.c
	touch t_tmp.c
	covc -q $cc -c -Fo"d i r"/ t_tmp.c
	covbr -q t_tmp.c >nul
	rm -r "d i r"
	covmgr -q -r "d i r"/+
'<<.
.
#242
run -n '# Build run-time in C++ mode
	cd ../lib
	[ -d tmp ] || mkdir tmp
	mv covntm.lib check.tmp
	buildrt nt cl -TP
	mv check.tmp covntm.lib
	rmdir tmp
'<<.
	% cl -c -DSYS_nt=1 -nologo -W4 -Zl -Og -Oi -Ot -Oy -Gs -Gr -TP covrt.c
	covrt.c
	% cl -c -DSYS_nt=1 -nologo -W3 -Zl -Og -Oi -Ot -Oy -Gs -Gr -TP covrt-nt-entryUser.c
	covrt-nt-entryUser.c
	% lib -nologo -out:covntm.lib covrt.obj covrt-nt-entryUser.obj
	% del covrt-nt-entryUser.obj
	% cl -c -DSYS_nt=1 -nologo -W4 -Zl -Og -Oi -Ot -Oy -Gs -Gr -TP covrt-nt-atexit.c
	covrt-nt-atexit.c
	buildrt: successfully built covntm.lib
.
#243
run -n '# cl -ZI does not force source names to lowercase
	rm -f t_Tmp.c
	echo >t_Tmp.c "void f() {}"
	covc -q $cc -ZI -c t_Tmp.c
	covsrc -qk t_Tmp.c
'<<.
	t_Tmp.c       0 /     1 =   0%       0 /     0
.
#244
run -u '# HP-UX shl_unload
	echo >t_tmp.c
	if [ -f /usr/include/dl.h ]
	then
		echo >>t_tmp.c "#include <dl.h>"
	else
		echo >>t_tmp.c "typedef struct dynamic *shl_t;"
		echo >>t_tmp.c "int shl_unload(shl_t handle);"
	fi
	echo >>t_tmp.c "
		void f() {
			shl_t handle = 0;
			shl_unload(handle);
			if (shl_unload(handle) == 0) ;
			if (1 && shl_unload(handle) == 0) ;
			if (1 && (shl_unload(handle) == 0)) ;
		}
		"
	covc -q -l $cc -c t_tmp.c
	grep "cov_write.*shl_unload" $TMPDIR/covc/t_tmp.c
'<<.
	(cov_write(),cov_reset(),shl_unload(handle));
	if (cov_c_(1, (cov_write(),cov_reset(),shl_unload(handle) == 0) ? 1 : 0)) ;
	if (cov_c_(2, (1 && cov_c_(3, (cov_write(),cov_reset(),shl_unload(handle) == 0) ? 1 : 0)) ? 1 : 0)) ;
	if (cov_c_(4, (1 && cov_c_(5, (cov_write(),cov_reset(),(shl_unload(handle) == 0)) ? 1 : 0)) ? 1 : 0)) ;
.
#245
run -n '# DLL without C run-time library
	echo >t_tmp.c "
		#include <windows.h>
		BOOL WINAPI f(HINSTANCE h, DWORD d, void* v) { return 1; }
		"
	covc -q cl -nologo -c -Zl t_tmp.c
	covlink link -nologo -dll -entry:f -nodefaultlib t_tmp.obj
'<<.
	C-Cover Link %ver
	ccoverEntryUser.c
	ccoverEntryUser.c
.
#246
run -n '# DLL entry point in C++
	echo >t_tmp.cpp "
		#include <windows.h>
		BOOL WINAPI f(HINSTANCE, DWORD, void*) { return 1; }
		"
	covc -q cl -nologo -c -DSTRICT t_tmp.cpp
	covlink link -nologo -dll -entry:f t_tmp.obj
'<<.
	C-Cover Link %ver
	ccoverEntryUser.c
	ccoverEntryUser.c
	ccoverEntryUser.cpp
.
#247
run -n '# DLL without C run-time library and entry point in C++
	echo >t_tmp.cpp "
		#include <windows.h>
		BOOL WINAPI f(HINSTANCE, DWORD, void*) { return 1; }
		"
	covc -q cl -nologo -c -Zl t_tmp.cpp
	covlink link -nologo -dll -entry:f -nodefaultlib t_tmp.obj
'<<.
	C-Cover Link %ver
	ccoverEntryUser.c
	ccoverEntryUser.c
	ccoverEntryUser.cpp
	ccoverEntryUser.cpp
.
#248
run '# covc option -t/--select with environment variable
	covc_select=main
	export covc_select
	covc -q -t\$covc_select $cc -c calc1.c
	covfn -q -k calc1.c
	covc -q -t%covc_select% $cc -c calc1.c
	covfn -q -k calc1.c
'<<.
	main      calc1.c         31  0 / 1       0 /    34 =   0%
	main      calc1.c         31  0 / 1       0 /    34 =   0%
.
#249
run -n '# covc -i with -Yu
	covc -q $cc -c -Ycctype.h calc1.c
	covc -q -icalc1.c $cc -c -Yuctype.h calc1.c
'<<.
.
#250
run -n '# link -lib
	covlink -v link -lib -out:t_tmp.lib calc1.obj 2>&1 | fgrep -e 902 -e 909
'<<.
	note cov902: running %.../link -lib -out:t_tmp.lib calc1.obj
	note cov909: exit status 0
.
#251
run '# case label with comments, like cpp macro markings
	echo >t_tmp.c "
		void f() {
			int i = 0;
			switch (i) {
			case /* */0/* */: break;
			}
		}
		"
	covc -q -p- $cc -c t_tmp.c
	covbr -q -c0 -e1 t_tmp.c
'<<.
	-->       2    void f() {
	-->       5     case /* */0/* */: break;
	-->       6     }
.
#252
run -u '# HP-UX pragma align
	echo >t_tmp.c "
		#pragma align 64
		struct { int i, j, k; } value1 = { 1, 2, 3 };
		#pragma		align	64
		struct { int i, j, k; } value2 = { 1, 2, 3 };
		"
	covc -q -l $cc -c t_tmp.c
	egrep "end of definitions|align" tmp/covc/t_tmp.c
'<<.
	/* C-Cover end of definitions */
	#pragma align 64
	#pragma		align	64
.
#253
run -n '# K+R struct parameter
	echo >t_tmp.c "
		void f(a)
			struct { int i; } *a;
		{ }
		void g(a,b,c,d,e)
			int a;
			struct { int i; } *b;
			int c;
			struct tag { int i; } *d;
			int e;
		{ }
		void h(a)
			struct tag *a;
		{ }
		"
	covc -q $cc -c t_tmp.c
	covbr -q -c0 -e1 t_tmp.c
'<<.
	-->       2    void f(a)
	-->       5    void g(a,b,c,d,e)
	-->      12    void h(a)
.
#254
run '# Preserve comments of the form /**/ and /*NN*/, in literals
	echo >t_tmp.c "const char* string = \"/**/ /*00*/\";"
	covc -q -l $cc -c t_tmp.c
	fgrep string tmp/covc/t_tmp.c
'<<.
	const char* string = "/**/ /*00*/";
.
#255
run -n '# cl with -link
	covc -q cl -nologo -c calc1.c
	covc -q cl -nologo calc1.obj -link -nologo
'<<.
.
#256
run -n '# cpp and non-cpp style column info together
	echo >t_tmp.h "
		#define NULL 0
		void f() {
			int* p = NULL;
			int a = 0;
			if (p == NULL && a) ;
		}
		"
	echo >t_tmp1.c "
		#include \"t_tmp.h\"
		int main() {
			f();
			return 0;
		}
		"
	cp t_tmp1.c t_tmp2.c
	covc -q --no-cpp $cc t_tmp1.c
	t_tmp1
	cp $COVFILE t_old.cov
	covc -q $cc t_tmp2.c
	covbr -q -e1 t_tmp.h
	covmgr -q -r t_tmp1.c
	covmerge -q -c -f t_new.cov t_old.cov $COVFILE
	covbr -q -e1 -f t_new.cov t_tmp.h
'<<.
	          1
	          2    #define NULL 0
	X         3    void f() {
	          4     int* p = NULL;
	          5     int a = 0;
	-->F      6     if (p == NULL && a) ;
	          7    }
	          8
	          1
	          2    #define NULL 0
	X         3    void f() {
	          4     int* p = NULL;
	          5     int a = 0;
	-->F      6     if (p == NULL && a) ;
	          7    }
	          8
.
#257
run -n '# kernel32.lib/user32.lib in link command with -nod
	PATH="$PATH;$LIB"
	k32=$(which kernel32.lib)
	u32=$(which user32.lib)
	libc=$(which libc.lib)
	unset LIB
	covc -q cl -nologo -c -Za calc1.c
	covlink -q link -nologo -nodefaultlib calc1.obj "$k32" "$u32" "$libc"
'<<.
.
#258
run '# source timestamp differs by exactly one hour
	COVFILE=$TMP/1.cov
	export COVFILE
	cp calc1.c t_tmp.c
	touch -t 01011200 t_tmp.c
	covc -q $cc -c t_tmp.c
	COVFILE=$TMP/2.cov
	touch -t 01011100 t_tmp.c
	covc -q $cc -c t_tmp.c
	COVFILE=$TMP/3.cov
	touch -t 01011300 t_tmp.c
	covc -q $cc -c t_tmp.c
	covmerge -c -f$TMP/4.cov $TMP/1.cov $TMP/2.cov $TMP/3.cov
'<<.
	C-Cover Merge %ver
.
#259
run '# Multiple coverage files
	echo >t_tmp1.c "
		void f();
		int main() { f(); return 0; }
		"
	echo >t_tmp2.c "void f() { }"
	rm -f $COVFILE test2.cov
	covc -q $cc -c t_tmp1.c
	COVFILE=test2.cov
	# Wait a second so that the two coverage files have different object IDs
	sleep 1
	covc -q $cc -c t_tmp2.c
	covc -q $cc ${o}t_tmp t_tmp[12].o*
	COVERR=checkTmpError
	export COVERR
	./t_tmp
	unset COVERR
	covsrc -q -k
	covsrc -q -k -f test.cov
	covclear -q
	COVFILELIST="test.cov,test2.cov"
	export COVFILELIST
	./t_tmp
	covsrc -q -k
	covsrc -q -k -f test.cov
	rm test2.cov
'<<.
	t_tmp2.c       1 /     1 = 100%       0 /     0
	t_tmp1.c       0 /     1 =   0%       0 /     0
	t_tmp2.c       1 /     1 = 100%       0 /     0
	t_tmp1.c       1 /     1 = 100%       0 /     0
.
#260
run '# Complex C++ switch case expression
	echo >t_tmp.cpp "
		class C {
		public:
			enum { name };
		};
		void f() {
			int a = 1;
			switch (a) {
			case int(C::name): ;
			}
		}
		"
	covc -q $cc -c t_tmp.cpp
'<<.
.
#261
run -n '# ANSI mode
	covc -q bcc32 -A -w calc1.c
	rm -f t_tmp.c
	echo >t_tmp.c "
		void f() {
			__asm { }
		}
		"
	covc -q bcc32 -A -c t_tmp.c
	covc -q cl -nologo -W4 -Za calc1.c
'<<.
	Warning W8002 t_tmp.c 3: Restarting compile using assembly in function f
	Assembling file:   t_tmp.ASM
	Error messages:    None
	Warning messages:  None
	Passes:            1
.
#262
run -n '# macro containing ?:
	echo >t_tmp.c "
		#define macro a?2:3
		void f() {
			int a = 0;
			if (macro) ;
			if (a && macro) ;
		}
		"
	covc -q $cc -c t_tmp.c
	covbr -q -e1 t_tmp.c
'<<.
	          1
	          2    #define macro a?2:3
	-->       3    void f() {
	          4     int a = 0;
	-->       5     if (macro) ;
	-->       6a    if (
	  -->     6b        a &&
	  -->     6c             macro) ;
	          7    }
	          8
.
#263
run -n '# EXE without CRT
	echo >t_tmp.c "
		#define WIN32_LEAN_AND_MEAN 1
		#include <windows.h>
		void __cdecl mainCRTStartup(void) {
			ExitProcess(0);
		}
		int __cdecl main(void) {
			return 0;
		}
		"
	covc -q $cc -c t_tmp.c
	covlink --no-banner link -nod t_tmp.obj kernel32.lib
	t_tmp
	covbr -q -c0 -e1 t_tmp.c
'<<.
	X         4    void __cdecl mainCRTStartup(void) {
	-->       7    int __cdecl main(void) {
.
#264
run -0 '# C++ variable definition inside decision
	echo >t_tmp.cpp "
		int main()
		{
			int i = 0;
			int* n = 0;
			typedef int T;
			#if __GNUC__
				for (; T    j = i; ) ;
				for (; T*   j = n; ) ;
				for (; int  j = i; ) ;
				for (; int* j = n; ) ;
			#endif
			if (T    j = i) ;
			if (T*   j = n) ;
			if (int  j = i) ;
			if (int* j = n) ;
			switch (int  j = i) { case 0:; }
			#if __GNUC__
				switch (T j = i) { case 0:; }
			#endif
			#if _MSC_VER
				switch (__int64 j = i) { case 0:; }
			#endif
			while (int j = i) ;
			return 0;
		}
		"
	covc -q cl -nologo t_tmp.cpp && t_tmp
	covc -q gcc -ot_tmp t_tmp.cpp && t_tmp
'<<.
.
#265
run -n '# Function entry excluded by macro
	echo >t_tmp.cpp "
		#define m void h() {
		m
			int i = 0; if (i) ;
		}
		"
	covc -v $cc ${o}t_tmp t_tmp.cpp 2>&1 | grep 933
'<<.
	t_tmp.cpp(4): note cov933: function entry excluded by macro
.
#266
run '# Unknown function
	echo >t_tmp.cpp "
		void ((f))() {
			int i = 0; if (i) ;
		}
		void ((g))() {
			int i = 0; if (i) ;
		}
		class C {
		public:
			static void f();
			static void known();
		};
		#pragma C-Cover off
		void C::f()
		{
		#pragma C-Cover on
			int i = 0;
			if (i) ;
		}
		void C::known() { }
		#pragma C-Cover off
		int main() {
		#pragma C-Cover on
			C::f();
			C::known();
			int j = 1;
			if (j) ;
			return 0;
		}
		"
	covc -v $cc ${o}t_tmp t_tmp.cpp 2>&1 | egrep "cov(810|825|933)"
	t_tmp
	covbr -q -e1 -c0 t_tmp.cpp
	covclass -q -k t_tmp.cpp
	covdir -q -k t_tmp.cpp
	covfn -qks t_tmp.cpp
	covsrc -q -k t_tmp.cpp
'<<.
	t_tmp.cpp(3): warning cov825: function head not recognized
	t_tmp.cpp(6): warning cov825: function head not recognized
	t_tmp.cpp(18): warning cov810: function entry excluded
	t_tmp.cpp(27): warning cov810: function entry excluded
	???       2    void ((f))() {
	-->       3     int i = 0; if (i) ;
	???       5    void ((g))() {
	-->       6     int i = 0; if (i) ;
	???      14    void C::f()
	-->F     18     if (i) ;
	X        20    void C::known() { }
	???      22    int main() {
	-->T     27     if (j) ;
	C::         1 /     1 = 100%       1 /     2 =  50%
	.               1 /     1 = 100%       2 /     8 =  25%
	(Unknown)  t_tmp.cpp        2  0 / 0       0 /     2 =   0%
	(Unknown)  t_tmp.cpp        5  0 / 0       0 /     2 =   0%
	C::f       t_tmp.cpp       14  0 / 0       1 /     2 =  50%
	C::known   t_tmp.cpp       20  1 / 1       0 /     0
	main       t_tmp.cpp       22  0 / 0       1 /     2 =  50%
	t_tmp.cpp       1 /     1 = 100%       2 /     8 =  25%
.
#267
run '# GNU -include
	echo >t_tmp.cpp "
		#if !macro2
			#error
		#endif
		"
	echo >t_tmp.h "
		#if !macro1
			#error
		#endif
		#define macro2 1
		"
	covc -q gcc -c -Dmacro1 -include t_tmp.h t_tmp.cpp
	covc -q gcc -c -Dmacro1 -imacros t_tmp.h t_tmp.cpp
'<<.
.
#268
run -n '# C-Cover always uses -Ob1
	echo >checkTmp1.cpp "
		#include \"checkTmp.h\"
		void f();
		int main() { f(); return 0; }
		void g() { h(); }
		"
	echo >checkTmp2.cpp "
		#include \"checkTmp.h\"
		void f() { h(); }
		"
	echo >checkTmp.h "
		inline void h() { }
		"
	covc -q -icheckTmp1.cpp cl -nologo checkTmp1.cpp checkTmp2.cpp
	checkTmp1
	covfn -qk checkTmp.h
	echo With -Ob0
	covc -q -icheckTmp1.cpp cl -Ob0 -nologo checkTmp1.cpp checkTmp2.cpp
	checkTmp1
	covfn -qk checkTmp.h
	echo With -O1
	covc -q -icheckTmp1.cpp cl -O1 -nologo checkTmp1.cpp checkTmp2.cpp
	checkTmp1
	covfn -qk checkTmp.h
'<<.
	h         checkTmp.h       2  1 / 1       0 /     0
	With -Ob0
	h         checkTmp.h       2  1 / 1       0 /     0
	With -O1
	h         checkTmp.h       2  1 / 1       0 /     0
.
#269
run '# covmgr -m
	covc -q $cc ${o}calc1 calc1.c
	covmgr -q -m
	echo 1 | calc1
	covfn -q -k calc1.c
'<<.
	Enter reverse-polish expressions, q to quit
	pop       calc1.c         21  0 / 1       0 /     0
	push      calc1.c         11  1 / 1       0 /     0
	Input     calc1.c         26  1 / 1       0 /     0
	main      calc1.c         31  1 / 1       9 /    34 =  26%
.
#270
run -n '# Environment variable CL
	mkdir covxCheckTmpDir
	echo >covxCheckTmpDir/t.h "int i;"
	echo >covxCheckTmp.c "
		#include \"t.h\"
		int main() { return 0; }
		"
	export CL=-IcovxCheckTmpDir
	covc -q $cc -c covxCheckTmp.c
	covc -q $cc covxCheckTmp.c
	rm -r covxCheckTmpDir
'<<.
	covxCheckTmp.c
	covxCheckTmp.c
.
#271
run -n '# Function declaration with "throw"
	echo >covxCheckTmp.cpp "
		void f() throw (); 
		struct s { };
		extern \"C\" { }
		void g() throw () { }
		"
	covc -q $cc -c covxCheckTmp.cpp
	covbr -q -c0 -e1 covxCheckTmp.cpp
'<<.
	covxCheckTmp.cpp:
	-->       5    void g() throw () { }
.
#272
run '# Macro CCOVER defined with -E
	echo >covxCheckTmp.c "
		#if CCOVER
		=true
		#else
		=false
		#endif
		"
	covc -q $cc -E covxCheckTmp.c 2>&1 | grep =
'<<.
	=true
.
#273
run '# Static destructor before pragma and first function
	echo >covxCheckTmp.cpp "
		class C {
		public:
			~C();
		};
		C c;
		#pragma
		C::~C() { }
		int main() { return 0; }
		"
	covc -q $cc ${o}covxCheckTmp covxCheckTmp.cpp
	covxCheckTmp
	covfn -kq covxCheckTmp.cpp
'<<.
	C::~C     covxCheckTmp.cpp     8  1 / 1       0 /     0
	main      covxCheckTmp.cpp     9  1 / 1       0 /     0
.
#274
run -n '# Automatically exclude functions in Windows headers
	windows_h=$(PATH="$PATH;$INCLUDE"; which windows.h)
	incdir=$(dirname "$windows_h")
	mkdir checkTmpInclude
	cp -r "$incdir"/win* checkTmpInclude
	echo >checkTmp.c "#include \"windows.h\""
	covc -q $cc -c -IcheckTmpInclude checkTmp.c
	covfn -kq checkTmpInclude/
	rm -fr checkTmpInclude
'<<.
	error cov139: no coverage information for directory checkTmpInclude/.  Specify regions as they are shown, without -b/--base
.
#275
run -n '# __asm block and braces
	# Microsoft treats semicolon comment to end of line
	echo >covxCheckTmp.c "
		void f() {
			int a = 0;
			__asm {
			;	{
			}
			if (a) ;
		}
		"
	covc -q $cc -c covxCheckTmp.c
	covbr -q -c0 -e1 covxCheckTmp.c
	# Borland treats semicolon as statement separator
	echo >covxCheckTmp.c "
		void f() {
			int a = 0;
			__asm { ; }
			if (a) ;
		}
		"
	covc -q bcc32 -c -w- covxCheckTmp.c
	covbr -q -c0 -e1 covxCheckTmp.c
'<<.
	covxCheckTmp.c
	covxCheckTmp.c:
	-->       2    void f() {
	-->       7     if (a) ;
	covxCheckTmp.c:
	Assembling file:   covxCheckTmp.ASM
	Error messages:    None
	Warning messages:  None
	Passes:            1
	covxCheckTmp.c:
	-->       2    void f() {
	-->       5     if (a) ;
.
#276
run '# -DCCoverDisable
	echo >checkTmp.c "void f() { }"
	covc -v $cc -c -DCCoverDisable checkTmp.c 2>&1 | fgrep cov902
	covc -v $cc -c -D CCoverDisable checkTmp.c 2>&1 | fgrep cov902
'<<.
	note cov902: running %cc -c -DCCoverDisable checkTmp.c
	note cov902: running %cc -c -D CCoverDisable checkTmp.c
.
#277
run '# namespace containing extern "C" containing pragma
	echo >checkTmp.cpp "
		namespace n {
			extern \"C\" {
				#pragma 
			}
		};
		void f() { }
		"
	covc -q $cc -c checkTmp.cpp
	covbr -q -c0 -e1 checkTmp.cpp
'<<.
	checkTmp.cpp:
	-->       7    void f() { }
.
#278
run -n '# covc only adds -Ob1 for regular compile commands
	echo >checkTmp.c "int main() { return 0; }"
	(
	covc -v $cc -c checkTmp.c
	covc -v $cc -E checkTmp.c
	covc -v $cc -Zg checkTmp.c
	) 2>&1 | grep 90[29]: | sed "s,-Ob1,(Ob1),"
'<<.
	note cov902: running internalCpp -DCCOVER=%ver -GX -nologo -c checkTmp.c -+cc cl
	note cov902: running %cc -c tmp/covc/checkTmp.c (Ob1)
	note cov909: exit status 0
	note cov902: running %.../cl -DCCOVER=%ver -GX -nologo -E checkTmp.c
	note cov909: exit status 0
	note cov902: running %cc -Zg checkTmp.c
	note cov909: exit status 0
.
#279
run '# gcc -M family
	(
	covc -v gcc -M -ocalc1 calc1.c
	covc -v gcc -MM -ocalc1 calc1.c
	covc -v gcc -MD -ocalc1 calc1.c
	covc -v gcc -MMD -ocalc1 calc1.c
	) 2>&1 | grep cov902
	rm calc1.d
'<<.
	note cov902: running %cc -M -ocalc1 calc1.c
	note cov902: running %cc -MM -ocalc1 calc1.c
	note cov902: running %cc -DCCOVER=%ver -MD -ocalc1 -E calc1.c >tmp/covc/calc1.i
	note cov902: running %cc -DCCOVER=%ver -MMD -ocalc1 -E calc1.c >tmp/covc/calc1.i
.
#280
run '# template class with multiple type parameters
	echo >checkTmp.cpp "
		template<class T1, class T2> class C {
		public:
			void f();
			void g() { }
		};
		template<class T1, class T2> void C<T1,T2>::f() { }
	"
	covc -q $cc -c checkTmp.cpp
	covbr -q -c0 -e1 checkTmp.cpp
	covfn -kq checkTmp.cpp
'<<.
	checkTmp.cpp:
	-->       5     void g() { }
	-->       7    template<class T1, class T2> void C<T1,T2>::f() { }
	C::g      checkTmp.cpp     5  0 / 1       0 /     0
	C::f      checkTmp.cpp     7  0 / 1       0 /     0
.
#281
run '# template specialization
	echo >checkTmp.cpp "
		// Taken from ISO/IEC 14882 14.7
		template<class T = int> struct A {
			static void x();
		};
		template<class U> void g(U) { }
		template<> struct A<double> { };
		template<> struct A<> { };
		template<> void g(char) { }
		template<> void g<int>(int) { }
		template<> void A<char>::x() { }
		template<class T = int> struct B {
			static void x();
		};
		template<> void B<>::x() { }
		// 14.7.3
		template<class T> class Array { };
		template<class T> void sort(Array<T>&) { }
		template<> void sort<char*>(Array<char*>&) { }
		// From RogueWave
		class C1 { };
		class C2 { };
		template<class T> class C3;
		template<> class C3<int>: public C1, public C2 {
			void f() { }
		};
	"
	covc -q $cc -c checkTmp.cpp
	covbr -q -c0 -e1 checkTmp.cpp
	covfn -kq checkTmp.cpp
'<<.
	checkTmp.cpp:
	-->       6    template<class U> void g(U) { }
	-->       9    template<> void g(char) { }
	-->      10    template<> void g<int>(int) { }
	-->      11    template<> void A<char>::x() { }
	-->      15    template<> void B<>::x() { }
	-->      18    template<class T> void sort(Array<T>&) { }
	-->      19    template<> void sort<char*>(Array<char*>&) { }
	-->      25     void f() { }
	g         checkTmp.cpp     6  0 / 1       0 /     0
	g         checkTmp.cpp     9  0 / 1       0 /     0
	g         checkTmp.cpp    10  0 / 1       0 /     0
	A::x      checkTmp.cpp    11  0 / 1       0 /     0
	B::x      checkTmp.cpp    15  0 / 1       0 /     0
	sort      checkTmp.cpp    18  0 / 1       0 /     0
	sort      checkTmp.cpp    19  0 / 1       0 /     0
	C3::f     checkTmp.cpp    25  0 / 1       0 /     0
.
#282
run '# covc -t with directory containing slash but not ending in slash
	mkdir checkTmpDir
	echo >checkTmpDir/checkTmp.c "void f() { }"
	covc -v -t!./checkTmpDir $cc -c checkTmpDir/checkTmp.c 2>&1 | grep 932
	rm -r checkTmpDir
'<<.
	note cov932: checkTmpDir/checkTmp.c:f exclude directory ./checkTmpDir
.
#283
run '# covc -p- should not remove the input file
	cp calc1.c checkTmp.c
	chmod +w checkTmp.c
	covc -q -p- $cc -c checkTmp.c
	ls checkTmp.c
'<<.
.
#284
run -n '# covc recognizes pragma directive generated from macro
	echo >checkTmp.c "
		#if CCOVER
			#define OFF C-Cover off
			#define ON C-Cover on
		#else
			#define OFF
			#define ON
		#endif
		#pragma OFF
		int main() { return 0; }
		#pragma ON
		void f() { }
		"
	covc -q $cc -c checkTmp.c
	covfn -q -k checkTmp.c
'<<.
	f         checkTmp.c      12  0 / 1       0 /     0
.
#285
run -n '# -Yc with no argument and no pragma hdrstop
	echo >checkTmpC.c "
		#define a 1234
		"
	echo >checkTmpU.c "
		#include \"pch.h\"
		int b = a;
		"
	covc -q $cc -c -FpcheckTmp.pch -Yc checkTmpC.c
	covc -q $cc -c -FpcheckTmp.pch -Yupch.h checkTmpU.c
'<<.
.
#286
run -n '# Preprocessor error
	echo >checkTmp.c "#include <noSuchFile.h>"
	covc -q $cc -c checkTmp.c
'<<.
	checkTmp.c(1): fatal error c1083: cannot open include file: 'noSuchFile.h': No such file or directory
	status 1
.
#287
run '# covc option --leave
	rm -fr $TMPDIR/covc
	covc -q -l $cc -c calc1.c
	ls $TMPDIR/covc/* | sed s,tmp/,,
'<<.
	covc/calc1.c
	covc/calc1.i
	covc/ccoverLog2
.
#288
run -n '# long input line
	echo >checkTmp.c "struct { char* s; } a[] = {"
	echo >checkTmp1.cpp "
		#include <stdio.h>
		int main() {
			for (int i = 0; i < 800; i++) {
				printf(\"{ \x22%u\x22 }, \", i);
			}
			return 0;
		}
		"
	$cc ${o}checkTmp1 checkTmp1.cpp
	checkTmp1 >>checkTmp.c
	echo >>checkTmp.c "};"
	echo >>checkTmp.c "void f() { }"
	cp checkTmp.c t.c
	covc -q $cc -c checkTmp.c
	covfn -qk checkTmp.c
'<<.
	f         checkTmp.c       3  0 / 1       0 /     0
.
#289
run '# Exception specification
	echo >checkTmp.cpp "
		void f() throw();
		namespace N {
			void g() { }
		}
		"
	covc -q $cc -c checkTmp.cpp
	covfn -qk checkTmp.cpp
'<<.
	N::g      checkTmp.cpp     4  0 / 1       0 /     0
.
#290
run '# Scope of friend function defined in class granting friendship
	echo >checkTmp.cpp "
		namespace N {
			class C1 {
				class C2 {
					friend void f() { }
				};
			};
		};
		int main()
		{
			N::f();
			return 0;
		}
		"
	covc -q $cc -c checkTmp.cpp
	covfn -qk checkTmp.cpp
'<<.
	N::f      checkTmp.cpp     5  0 / 1       0 /     0
	main      checkTmp.cpp     9  0 / 1       0 /     0
.
#291
run -n '# FreeLibrary with autosave off
	echo >checkTmp.cpp "
		#include <stdio.h>
		#include <windows.h>
		int main() {
				const HMODULE module = LoadLibrary(\"checkTmpDll.dll\");
				if (module == 0) printf(\"LoadLibrary failure %d\", GetLastError());
				const BOOL success = FreeLibrary(module);
				if (!success) printf(\"FreeLibrary failure %d\", GetLastError());
				puts(\"main\");
				return success ? 0 : 1;
		}
		"
	echo >checkTmpDll.cpp "
		#include <stdio.h>
		class C {
		public:
			// Ctor is needed to cause C-Cover run-time initalization
			C() { }
			~C() { puts(\"~C\"); }
		};
		static C c;
		"
	$cc checkTmp.cpp
	$cc -c checkTmpDll.cpp
	link -dll checkTmpDll.obj
	checkTmp
	covc -q $cc -c checkTmpDll.cpp
	covlink -q link -dll checkTmpDll.obj
	covmgr -a1 -q
	checkTmp
	covmgr -a0 -q
	checkTmp
'<<.
	~C
	main
	ccoverEntryUser.c
	main
	~C
	~C
	main
.
#292
run '# covsrc, covfn width reduction
	mkdir check.dir
	cp calc[12].c check.dir
	rm -f $COVFILE
	covc -q $cc -c check.dir/calc[12].c
	covfn -q -b check.dir/
	covsrc -q -b check.dir/
	covfn -q -n check.dir/
	rm -fr check.dir
'<<.
	Function  Source        Line  FnCov           C/D Coverage
	--------  ------------------  -----  ---------------------
	prs       calc2.c        158  0 / 1       0 /   125 =   0%
	lex       calc2.c         54  0 / 1       0 /    36 =   0%
	main      calc1.c         31  0 / 1       0 /    34 =   0%
	name      calc2.c         99  0 / 1       0 /    11 =   0%
	main      calc2.c        271  0 / 1       0 /    10 =   0%
	eval      calc2.c        252  0 / 1       0 /     6 =   0%
	match     calc2.c        128  0 / 1       0 /     4 =   0%
	z         calc2.c        143  0 / 1       0 /     4 =   0%
	push      calc1.c         11  0 / 1       0 /     0
	pop       calc1.c         21  0 / 1       0 /     0
	Input     calc1.c         26  0 / 1       0 /     0
	--------  ------------------  -----  ---------------------
	Total                            0%       0 /   230 =   0%
	Source       Function Coverage           C/D Coverage
	-------  ---------------------  ---------------------
	calc2.c       0 /     7 =   0%       0 /   196 =   0%
	calc1.c       0 /     4 =   0%       0 /    34 =   0%
	-------  ---------------------  ---------------------
	Total         0 /    11 =   0%       0 /   230 =   0%
	Function  FnCov           C/D Coverage
	--------  -----  ---------------------
	prs       0 / 1       0 /   125 =   0%
	lex       0 / 1       0 /    36 =   0%
	main      0 / 1       0 /    34 =   0%
	name      0 / 1       0 /    11 =   0%
	main      0 / 1       0 /    10 =   0%
	eval      0 / 1       0 /     6 =   0%
	match     0 / 1       0 /     4 =   0%
	z         0 / 1       0 /     4 =   0%
	push      0 / 1       0 /     0
	pop       0 / 1       0 /     0
	Input     0 / 1       0 /     0
	--------  -----  ---------------------
	Total        0%       0 /   230 =   0%
.
#293
run -n '# Borland pragma resource
	echo >checkTmp.c "
		#pragma resource \"*.dfm\"
		int i;
		"
	covc -q bcc32 -c checkTmp.c
'<<.
	checkTmp.c:
.
#294
run -n '# COMSPEC not set to cmd.exe
	COMSPEC=$SHELL
	covc -q gcc -c calc1.c
'<<.
.
#295
run '# Unknown GNU cross compiler
	mkdir checkBin
	cp "`$which gcc`" checkBin/x-gcc.exe
	covc -q checkBin/x-gcc.exe -c calc1.c
	rm -fr checkBin
'<<.
.
#296
run '# switch value is signed, but high bit is set
	echo >checkTmp.c "
		int main()
		{
			unsigned i = 0x8000000 << 4;
			switch (i) {
			case 0x8000000 << 4:
				puts(\"a\");
				break;
			default:
				puts(\"b\");
				break;
			}
			return 0;
		}
		"
	covc -q $cc ${o}checkTmp checkTmp.c
	checkTmp
'<<.
	a
.
#297
run '# #line directive short form with trailing space
	rm -f $COVFILE
	echo >checkTmp1.c "
		#line 1 \"checkTmp2.c\"
		#line 20 
		void f(){}
		"
	covc -q -p- cp checkTmp1.c checkTmp3
	covfn -q -k
'<<.
.
#298
run -n '# GNU option -fhandle-signatures
	# NT-only because older versions of GCC do not produce the warning below
	covc -q g++ -c -fhandle-signatures calc3.cpp
'<<.
	cc1plus.exe: warning: -fhandle-signatures is deprecated.
	cc1plus.exe: warning: Please see the documentation for details.
.
#299
run -n '# switch case with expression text shorter than instrumentation
	# and with no space following "case"
	echo >checkTmp.cpp "
		int main(int argc, char**) {
			switch (argc) {
			case(-1):
			case 0: case 1: case 2: case 3: case 4:
			case 5: case 6: case 7: case 8: case 9:
				break;
			}
			return 0;
		}
		"
	covc -q $cc -W4 ${o}checkTmp checkTmp.cpp
	checkTmp 2 3 4 5 6 7 8 9
	covbr -q -c0 -e1 checkTmp.cpp
'<<.
	checkTmp.cpp:
	X         2    int main(int argc, char**) {
	-->       4     case(-1):
	-->       5a    case 0:
	-->       5b            case 1:
	-->       5c                    case 2:
	-->       5d                            case 3:
	-->       5e                                    case 4:
	-->       6a    case 5:
	-->       6b            case 6:
	-->       6c                    case 7:
	-->       6d                            case 8:
	X         6e                                    case 9:
	-->       8     }
.
#300
run -u '# Compiler name contains dot
	# Unix only because cmd.exe cannot run gcc with name containing dots
	mkdir checkDir
	cp "`which gcc`" checkDir/gcc-x.y.z
	covc -q checkDir/gcc-x.y.z -c calc1.c
	PATH="$PATH:checkDir"
	export PATH
	covc -q gcc-x.y.z -c calc1.c
	ln ../src/covc covc
	ln covc gcc-x.y.z
	gcc-x.y.z -c calc1.c
	ln ../src/cov01 cov01
	cov01 -0v 2>&1 | fgrep gcc-x.y.z
	cov01 -1v 2>&1 | fgrep gcc-x.y.z
	rm -fr checkDir cov01 covc gcc-x.y.z
'<<.
	C-Cover Compile C/C++ %ver
	disabled: ./gcc-x.y.z
	enabled: ./gcc-x.y.z
.
#301
run -u '# Avoid Sun ild
	if [ "`which CC 2>/dev/null`" = /opt/SUNWspro/bin/CC ]
	then
		CC -c calc1.c
		covc -q CC -g -v calc1.o 2>&1 | fgrep /ild
	fi
	true
'<<.
.
#302
run -n '# Multiple source files with options following
	echo >checkTmp.c "
		#if !macro
		#error
		#endif
		"
	covc -q $cc -c checkTmp.c checkTmp.c -Dmacro
'<<.
.
#303
run -n '# Empty result of preprocessing
	echo >checkTmp.c
	debug/covx covc -q $cc -c checkTmp.c
'<<.
.
#304
run '# Link non-shared
	if [ $uname = SunOS -o $uname = IRIX ]
	then
		true
	else
		covc -q gcc -static calc1.c
	fi
'<<.
.
finish
