/* $Id: pthread.h,v 1.76 1999/10/15 17:21:59 steve Exp $
 * Copyright (c) Bullseye Testing Technology
 *
 * Purpose
 *    Subset of POSIX Threads for C-Cover.
 *    ISO/IEC 9945-1: 1996
 *    ANSI/IEEE P1003.1: 1996
 *
 * Platforms
 *    Win32
 *    Exemplar
 *    NT Kernel Mode
 *    NT Kernel Mode Graphics Driver
 *    pSOS
 *    Solaris
 *    VxWorks
 *    All Others
 *
 * Declarations
 *    PTHREAD_MUTEX_INITIALIZER
 *    pthread_create
 *    pthread_exit
 *    pthread_join
 *    pthread_mutex_destroy
 *    pthread_mutex_lock
 *    pthread_mutex_t
 *    pthread_mutex_trylock
 *    pthread_mutex_unlock
 *    pthread_t
 *    sched_yield
 */

#if !_PTHREAD_H
#define _PTHREAD_H 1

/*======================================================================
 * Win32
 */
#if SYS_NT || SYS_nt || SYS_WindowsCE
#	include <windows.h>
#	define PTHREAD_LINK WINAPI

	typedef HANDLE pthread_t;

	static int pthread_create(
		pthread_t * thread,
		const void * attr,
		void * (PTHREAD_LINK * start_routine)(void *),
		void * arg)
	{
		int status;
		DWORD unused = (DWORD)attr; /* quiet warning */
		*thread = CreateThread(NULL, 0,
			(LPTHREAD_START_ROUTINE)start_routine, arg, 0, &unused);
		if (*thread == NULL) {
			status = (int)GetLastError();
		} else {
			status = 0;
		}
		return status;
	}

	static int pthread_join_isDllDetach = 0;
#	define pthread_exit(status) ExitThread((DWORD)(status))
	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		DWORD status;
		int success = GetExitCodeThread(thread, &status);
		if (success && status == STILL_ACTIVE) {
			if (pthread_join_isDllDetach) {
				/* All threads except this one were killed by system */
			} else {
				success = WaitForSingleObject(thread, INFINITE) != WAIT_FAILED;
				if (success) {
					success = GetExitCodeThread(thread, &status);
				}
			}
		}
		if (value_ptr != NULL) {
			DWORD* p = (DWORD*)value_ptr;
			*p = status;
		}
		return success ? 0 : (int)GetLastError();
	}

	typedef struct {
		volatile long init;
		HANDLE handle;
		DWORD error;
	} pthread_mutex_t;

#	define PTHREAD_MUTEX_INITIALIZER {0, 0, 0}

	static DWORD _Pthread_mutex_init(pthread_mutex_t* mutex)
	{
		DWORD status = 0;
		switch ((int)InterlockedExchange((long *)&mutex->init, 1)) {
		default:
		case 0: /* uninitialized */
			mutex->handle = CreateMutex(NULL, FALSE, NULL);
			if (mutex->handle == NULL) {
				status = GetLastError();
				if (status == 0) {
					status = 1;
				}
				mutex->error = status;
				mutex->init = 0;
			} else {
				mutex->init = 2;
			}
			break;
		case 1: /* initializing */
			while (mutex->init != 2) {
				if (mutex->init == 0) {
					status = mutex->error;
					break;
				}
				Sleep(0);
			}
			break;
		case 2: /* initialized */
			mutex->init = 2;
			break;
		}
		return status;
	}

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		DWORD status = 0;
		if (mutex->init != 2) {
			status = _Pthread_mutex_init(mutex);
		}
		if (status == 0 &&
			WaitForSingleObject(mutex->handle, INFINITE) == WAIT_FAILED)
		{
			status = GetLastError();
			if (status == 0) {
				status = 1;
			}
		}
		return (int)status;
	}

	static int pthread_mutex_trylock(pthread_mutex_t* mutex)
	{
		DWORD status = 0;
		if (mutex->init != 2) {
			status = _Pthread_mutex_init(mutex);
		}
		if (status == 0) {
			status = WaitForSingleObject(mutex->handle, 0);
			if (status == WAIT_TIMEOUT) {
				status = EBUSY;
			} else if (status == WAIT_FAILED) {
				status = GetLastError();
				if (status == 0) {
					status = 1;
				} else if (status == ERROR_INVALID_HANDLE) {
					/* The data segment is shared (-section:,s or .def SHARED)
					 * and the mutex handle was set by another process
					 */
					status = 9999;
				}
			} else {
				status = 0;
			}
		}
		return (int)status;
	}

	static int pthread_mutex_unlock(pthread_mutex_t* mutex)
	{
		int status = 0;
		if (!ReleaseMutex(mutex->handle)) {
			status = (int)GetLastError();
		}
		return status;
	}
	static int pthread_mutex_destroy(pthread_mutex_t* mutex)
	{
		int status = 0;
		if (!CloseHandle(mutex->handle)) {
			status = (int)GetLastError();
		}
		return status;
	}

#	define sched_yield() Sleep(0)

#	define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * Convex Exemplar
 */
#if !defined(_PTHREAD_DEFINED) && PTHREAD_exemplar
	#include <cps.h>
	#include <errno.h>

	#pragma _CNX sync_routine(pthread_mutex_lock, pthread_mutex_unlock)
	#pragma _CNX sync_routine(_Pthread_mutex_init, pthread_mutex_trylock)

	typedef int pthread_t;

	typedef void (* _Pthread_fn)(void*);
	static int pthread_create(
		pthread_t* thread,
		const void* attr,
		void* (* start_routine)(void*),
		void* arg)
	{
		int any = CPS_ANY_NODE;
		*thread = cps_thread_create(&any, (_Pthread_fn)start_routine, arg);
		return *thread == -1 ? errno : 0;
	}

	#define pthread_exit(status) cps_thread_exit()

	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		thread = thread;
		value_ptr = value_ptr;
		return 1;
	}

	typedef cps_mutex_t pthread_mutex_t;
	#define PTHREAD_MUTEX_INITIALIZER {0}

	static int _Pthread_mutex_init(pthread_mutex_t* mutex)
	{
		int status = 0;
		#pragma _CNX critical_section
			if (*mutex == 0) {
				status = cps_mutex_alloc(mutex);
			}
		#pragma _CNX end_critical_section
		return status;
	}

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		int status = 0;
		if (*mutex == 0) {
			status = _Pthread_mutex_init(mutex);
		}
		if (status == 0) {
			const int errno_save = errno;
			status = cps_mutex_lock(mutex);
			errno = errno_save;
		}
		return status;
	}

	static int pthread_mutex_trylock(pthread_mutex_t* mutex)
	{
		int status = 0;
		if (*mutex == 0) {
			status = _Pthread_mutex_init(mutex);
		}
		if (status == 0) {
			const int errno_save = errno;
			errno = 0;
			status = cps_mutex_trylock(mutex);
			if (status == -1) {
				if (errno == 0) {
					status = EBUSY;
				} else {
					status = errno;
				}
			}
			errno = errno_save;
		}
		return status;
	}

	#define pthread_mutex_unlock cps_mutex_unlock
	#define pthread_mutex_destroy cps_mutex_free

	#define sched_yield()

#	define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * NT Kernel Mode
 *
 * We use KMUTEX rather than FAST_MUTEX because on NT 4.0 FAST_MUTEX
 * causes a mysterious hang with file i/o in between lock and unlock.
 */
#if !_PTHREAD_DEFINED && SYS_ntk

#	include <ntddk.h>
#	define PTHREAD_LINK __stdcall

	typedef HANDLE pthread_t;

#	define pthread_create(thread, attr, start_routine, arg) \
		(int)PsCreateSystemThread(thread, THREAD_ALL_ACCESS, NULL, \
		NULL, NULL, (PKSTART_ROUTINE)start_routine, arg)
#	define pthread_exit(status) PsTerminateSystemThread((long)(status))
	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		void* object;
		NTSTATUS status = ObReferenceObjectByHandle(thread, THREAD_ALL_ACCESS,
			NULL, KernelMode, &object, NULL);
		if (status == 0) {
			status = KeWaitForSingleObject(object, Executive, KernelMode,
				FALSE, NULL);
		}
		value_ptr = value_ptr;
		return (int)status;
	}

	typedef struct {
		volatile long init;
		KMUTEX* m;
	} pthread_mutex_t;

#	define PTHREAD_MUTEX_INITIALIZER {0}    /* Incomplete initializer */
	static int _Pthread_0 = 0;

	static void _Pthread_mutex_init(pthread_mutex_t* mutex)
	{
		switch ((int)InterlockedExchange((long*)&mutex->init, 1)) {
		default:
		case 0: /* Uninitialized */
			mutex->m = ExAllocatePool(NonPagedPool, sizeof(*(mutex->m)));
			KeInitializeMutex(mutex->m, LONG_MAX);
			mutex->init = 2;
			break;
		case 1: /* Initializing */
			while (mutex->init != 2) {
				LARGE_INTEGER largeInteger;
				largeInteger.QuadPart = -1;
				KeDelayExecutionThread(KernelMode, FALSE, &largeInteger);
			}
			break;
		case 2: /* Initialized */
			mutex->init = 2;
			break;
		}
	}

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		if (mutex->init != 2) {
			_Pthread_mutex_init(mutex);
		}
		return (int)KeWaitForMutexObject(mutex->m, Executive, KernelMode,
			FALSE, NULL);
	}

	static int pthread_mutex_trylock(pthread_mutex_t* mutex)
	{
		int status;
		LARGE_INTEGER largeInteger;
		largeInteger.QuadPart = 0;
		if (mutex->init != 2) {
			_Pthread_mutex_init(mutex);
		}
		status = (int)KeWaitForMutexObject(mutex->m, Executive, KernelMode,
			FALSE, &largeInteger);
		return status;
	}

#	define pthread_mutex_unlock(mutex) \
		(KeReleaseMutex((mutex)->m, FALSE), (_Pthread_0 = _Pthread_0))
#	define pthread_mutex_destroy(mutex) \
		(ExFreePool((mutex)->m), (_Pthread_0 = _Pthread_0))

#	define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * NT Kernel Mode Graphics Driver
 */
#if !_PTHREAD_DEFINED && SYS_ntGraphic
	#include <winerror.h>
	typedef int pthread_t;

	/* Using a function rather than a macro causes fewer warnings */
	static int pthread_create(
		pthread_t* thread,
		const void* attr,
		void* (* start_routine)(void*),
		void* arg)
	{
		thread = thread;
		attr = attr;
		start_routine = start_routine;
		arg = arg;
		return ERROR_NOT_SUPPORTED;
	}

	#define pthread_exit(status)

	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		thread = thread;
		value_ptr = value_ptr;
		return ERROR_NOT_SUPPORTED;
	}

	typedef struct {
		volatile HSEMAPHORE semaphore;
	} pthread_mutex_t;

	#define PTHREAD_MUTEX_INITIALIZER {NULL}

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		int status = 1;
		if (mutex->semaphore == NULL) {
			mutex->semaphore = EngCreateSemaphore();
		}
		if (mutex->semaphore != NULL) {
			EngAcquireSemaphore(mutex->semaphore);
			status = 0;
		}
		return status;
	}

	#define pthread_mutex_trylock pthread_mutex_lock
	static int _Pthread_0 = 0;
	#define pthread_mutex_unlock(mutex) \
		(EngReleaseSemaphore((mutex)->semaphore), (_Pthread_0 = _Pthread_0))
	#define pthread_mutex_destroy(mutex) \
		(EngDeleteSemaphore((mutex)->semaphore), (_Pthread_0 = _Pthread_0))

	#define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * pSOS
 */
#if !_PTHREAD_DEFINED && SYS_pSOS
#	include <psos.h>

	typedef unsigned long pthread_t;

	static void _Pthread_create_glue(unsigned long * targs) {
		void (* start_routine)(void *) = (void (*)(void *))targs[0];
		start_routine((void *)targs[1]);
		t_delete(0);
	}

	static int pthread_create(
		pthread_t * thread,
		const void * attr,
		void * (* start_routine)(void *),
		void * arg)
	{
		unsigned long status = t_create("covt", 1, 1024, 1024,
			T_LOCAL | T_NOFPU, thread);
		if (status == 0) {
			unsigned long targs[4] = { (unsigned long)start_routine,
				(unsigned long)arg, 0, 0 };
			status = t_start(*thread, T_PREEMPT | T_TSLICE | T_NOASR |
				T_USER | T_LEVELMASK0, _Pthread_create_glue, targs);
		}
		return (int)status;
	}

#	define pthread_exit(status)

	typedef struct {
		int init;
		unsigned long sm;
	} pthread_mutex_t;
#	define PTHREAD_MUTEX_INITIALIZER {0,0}

	static int pthread_mutex_lock(pthread_mutex_t * mutex)
	{
		unsigned long status = 0;
		if (!mutex->init) {
			const unsigned long mask = T_LEVELMASK7 |
				T_PREEMPT | T_NOPREEMPT |
				T_TSLICE  | T_NOTSLICE |
				T_ASR     | T_NOASR;
			unsigned long mode;
			t_mode(mask, T_NOPREEMPT | T_NOTSLICE | T_NOASR | T_LEVELMASK7,
				&mode);
			if (!mutex->init) {
				status = sm_create("covs", 1, SM_LOCAL | SM_PRIOR, &mutex->sm);
				if (status == 0) {
					mutex->init = 1;
				}
			}
			t_mode(mask, mode, &mode);
		}
		if (status == 0) {
			status = sm_p(mutex->sm, SM_WAIT, 0);
		}
		return (int)status;
	}

	static int pthread_mutex_unlock(pthread_mutex_t * mutex)
	{
		return (int)sm_v(mutex->sm);
	}

#	define pthread_mutex_destroy(mutex) \
		(sm_delete((mutex)->sm), (mutex)->init = 0)

#	define sched_yield() tm_wkafter(0)

#	define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * Solaris
 */
#if !_PTHREAD_DEFINED && PTHREAD_lwp
#	include <signal.h>
#	include <synch.h>
#	include <sys/lwp.h>

	/* The stack is inside pthread_t - it must live as long as the thread. */
	typedef struct {
		lwpid_t tid;
		ucontext_t context;
		char stack[1000];
	} pthread_t;

	typedef lwp_mutex_t pthread_mutex_t;

#	define PTHREAD_MUTEX_INITIALIZER {0}    /* incomplete initializer */

	static int pthread_create(
		pthread_t * thread,
		const void * attr,
		void * (*start_routine)(void *),
		void * arg)
	{
		sigprocmask(SIG_SETMASK, NULL, &thread->context.uc_sigmask);
		_lwp_makecontext(&thread->context, (void(*)(void*))start_routine,
			arg, NULL, thread->stack, sizeof(thread->stack));
		return _lwp_create(&thread->context, 0, &thread->tid);
	}

#	define pthread_exit(status)
	#define pthread_join(thread, value_ptr) _lwp_wait((thread).tid, NULL)

#	define pthread_mutex_lock(mutex)    _lwp_mutex_lock(mutex)
#	define pthread_mutex_trylock(mutex) _lwp_mutex_trylock(mutex)
#	define pthread_mutex_unlock(mutex)  _lwp_mutex_unlock(mutex)
	static int _Pthread_0 = 0;
#	define pthread_mutex_destroy(mutex) (_Pthread_0 = _Pthread_0)

#	define sched_yield yield

#	define _PTHREAD_DEFINED 1
#endif /* SYS__Solaris */

/*======================================================================
 * VxWorks 5.2
 */
#if !_PTHREAD_DEFINED && SYS_VxWorks

#	include <vxWorks.h>
#	include <private/tasklibp.h>
#	include <semLib.h>
#	include <taskLib.h>
#	include <vxLib.h>

	typedef int pthread_t;

	static int _Pthread_create_glue(int arg1, int arg2, int arg3, int arg4, int arg5, int arg6, int arg7, int arg8, int arg9, int arg10)
	{
		void* (*start_routine)(void*) = (void* (*)(void*))arg1;
		return (int)start_routine((void*)arg2);
	}

	static int pthread_create(
		pthread_t * thread,
		const void * attr,
		void * (* start_routine)(void *),
		void * arg)
	{
		attr = attr;    /* Quiet warning */
		*thread = taskSpawn("tC-Cover", 255, 0, sizeof(WIND_TCB) + 2048,
			_Pthread_create_glue, (int)start_routine, (int)arg, 0, 0, 0, 0, 0, 0, 0, 0);
		return (*thread == ERROR) ? 1 : 0;
	}

#	define pthread_exit(status) exit((int)status)
	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		for (;;) {
			WIND_TCB* tcb = taskTcb(thread);
			if (tcb == NULL) {
				break;
			}
			if ((tcb->status & WIND_DEAD) != 0) {
				break;
			}
			taskDelay(1);
		}
		return 0;
	}

	typedef struct {
		int init;
		int is_taken;
		SEM_ID sem;
	} pthread_mutex_t;

#	define PTHREAD_MUTEX_INITIALIZER { 0, 0, NULL }

	static int _Pthread_mutex_init(pthread_mutex_t* mutex)
	{
		int status = 0;
		/* Invalid to call VxWorks with interrupts disabled */
		taskLock();
		/* if init == PTHREAD_MUTEX_INITIALIZER */
		if (mutex->init == 0) {
			mutex->sem = semBCreate(SEM_Q_FIFO, SEM_EMPTY);
			if (mutex->sem == 0) {
				status = 1;
			} else {
				mutex->init = 1;
			}
		}
		taskUnlock();
		return status;
	}

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		int status = 0;
		/* if init == PTHREAD_MUTEX_INITIALIZER */
		if (mutex->init == 0) {
			status = _Pthread_mutex_init(mutex);
		}
		if (status == 0) {
			/* while semaphore is taken */
			while (!vxTas(&mutex->is_taken)) {
				/* wait for broadcast (via semFlush) that it is available */
				semTake(mutex->sem, WAIT_FOREVER);
			}
			taskSafe();
		}
		return status;
	}

	static int pthread_mutex_trylock(pthread_mutex_t* mutex)
	{
		int status = 0;
		if (vxTas(&mutex->is_taken)) {
			taskSafe();
		} else {
			status = EBUSY;
		}
		return status;
	}

	int pthread_mutex_unlock(pthread_mutex_t* mutex)
	{
		mutex->is_taken = 0;
		taskUnsafe();
		/* wake up anyone waiting for semaphore */
		return (int)semFlush(mutex->sem);
	}

#	define pthread_mutex_destroy(mutex) semDelete((mutex)->sem)

#	define _PTHREAD_DEFINED 1
#endif /* VxWorks */

/*======================================================================
 * User Defined
 */
#if !_PTHREAD_DEFINED && SYS_user
	/* Defined in covrt-user.c */
	#define _PTHREAD_DEFINED 1
#endif

/*======================================================================
 * All Other Platforms
 */
#if !_PTHREAD_DEFINED
	typedef int pthread_t;

	/* Using a function rather than a macro causes fewer warnings */
	static int pthread_create(
		pthread_t* thread,
		const void* attr,
		void* (* start_routine)(void*),
		void* arg)
	{
		thread = thread;
		attr = attr;
		start_routine = start_routine;
		arg = arg;
		/* Return value !=0 other than 1, used for this error elsewhere */
		return 2;
	}

#	define pthread_exit(status)

	static int pthread_join(pthread_t thread, void** value_ptr)
	{
		thread = thread;
		value_ptr = value_ptr;
		return 1;
	}

	/* This mutex implementation is unsafe, since it simulates a test-and-set.
	 */
	/* Use a struct to contain the volatile modifier */
	typedef struct {
		volatile unsigned is_taken;
	} pthread_mutex_t;
#	define PTHREAD_MUTEX_INITIALIZER { 0 }

	static int pthread_mutex_lock(pthread_mutex_t* mutex)
	{
		for (;;) {
			/* fake a "test and set" operation */
			if (mutex->is_taken == 0) {
				mutex->is_taken = 1;
				break;
			}
		}
		return 0;
	}

	static int pthread_mutex_trylock(pthread_mutex_t* mutex)
	{
		int status = 0;
		/* fake a "test and set" operation */
		if (mutex->is_taken == 0) {
			mutex->is_taken = 1;
		} else {
			status = EBUSY;
		}
		return status;
	}

#	define pthread_mutex_unlock(mutex) ((mutex)->is_taken = 0)
	static int _Pthread_0 = 0;
#	define pthread_mutex_destroy(mutex) (_Pthread_0 = _Pthread_0)

#	define sched_yield()

#endif /* !_PTHREAD_DEFINED */

#endif /* _PTHREAD_H */

/* C-Cover on */
