/***********************************************************************
 * $Id: covrtvxw.h,v 2.2 1999/10/14 22:34:06 steve Exp $
 * Copyright (c) Bullseye Testing Technology
 *
 * Purpose
 *    C-Cover run-time definitions for VxWorks 5.2 or later.
 *
 * POSIX Functions
 *    atexit
 *    nanosleep
 *
 * Platform Support
 *    error_write_screen
 */

/* Does not support fcntl */
#if !defined(COVRT_NOFCNTL)
#	define COVRT_NOFCNTL 1
#endif

/*========== POSIX.1 Interface =======================================*/

#include <errno.h>
#include <timers.h>
#include <unistd.h>

/* atexit */
#include <taskHookLib.h>
#include <taskLib.h>
static struct {
	WIND_TCB* tcb;
	void (*function)(void);
} atexit_array[32];
static unsigned atexit_count = 0;
static void atexit_hook(WIND_TCB * p)
{
	unsigned i;
	for (i = 0; i < sizeof(atexit_array) / sizeof(atexit_array[0]); i++) {
		if (p == atexit_array[i].tcb) {
			atexit_array[i].tcb = NULL;
			atexit_array[i].function();
			/* Delete our hook if this is the last function */
			atexit_count--;
			if (atexit_count == 0) {
				taskDeleteHookDelete((FUNCPTR)atexit_hook);
			}
		}
	}
}
#undef atexit
#define atexit COVRTVXW_atexit
static int atexit(void (*function)(void))
{
	/* Search for an available slot in atexit_array */
	int status = 1;
	unsigned i;
	for (i = 0; i < sizeof(atexit_array) / sizeof(atexit_array[0]); i++) {
		if (atexit_array[i].tcb == NULL) {
			status = 0;
			break;
		}
	}
	/* If found */
	if (status == 0) {
		/* Add this function to the array */
		atexit_array[i].tcb = taskTcb(taskIdSelf());
		atexit_array[i].function = function;
		/* Add our hook if this is the first function */
		if (atexit_count == 0) {
			status = taskDeleteHookAdd((FUNCPTR)atexit_hook);
		}
		atexit_count++;
	}
	return status;
}

#if Covrt_nanosleep
	/* nanosleep */
	/* Although VxWorks 5.2 documents this function, customer 7157 does not
	 * have it available on the mbx860 Board Support Package */
	#undef nanosleep
	#define nanosleep Covrt_nanosleep_
	int nanosleep(const struct timespec* rqtp, const struct timespec* rmtp)
	{
		/* Nanoseconds per clock tick */
		/*   Denominator is ticks per second */
		const long k = 1000000000 / 60;
		taskDelay(rqtp->tv_nsec / k);
		rmtp = rmtp;
		return 0;
	}
#endif

static void error_write_file(int fd, const char* s);
static void error_write_screen(const char* s)
{
	/* Write to stderr */
	error_write_file(2, s);
}
