/***********************************************************************
 * $Id: covrt.c,v 5.83 2000/04/07 01:46:16 paul Exp $
 *
 * CONFIDENTIAL
 *    This source file contains copyrighted, proprietary information of
 *    Bullseye Testing Technology.
 *
 * Purpose
 *    C-Cover run-time probe support.
 *
 * Compile Option Summary
 *    Define the macros listed below as in:
 *
 *      cc -c -Dmacro=value covrt.c
 *
 *    define            when
 *    ----------------------------------------------------------------
 *    COVRT_NOATX=1     neither atexit nor on_exit is in C library
 *    COVRT_NOENV=1     to not use function getenv
 *    COVRT_NOFCNTL=1   to not use file locking (function fcntl)
 *    SYS_LabW=1        to run on National Instruments LabWindows/CVI
 *    SYS_UNIX=1        to run on Unix, POSIX
 *    SYS_user=1        to run on user defined operating system
 *    SYS_VxWorks=1     to run on VxWorks
 *    SYS_hpemul=1      to run on HP emulator
 *    SYS_nt=1          to run on MS Windows NT or Windows 9x or Win32
 *    SYS_ntGraphic=1   to run on Windows NT kernel mode graphics driver
 *    SYS_ntk=1         to run on Windows NT kernel mode
 *    SYS_pharlap=1     to run on Pharlap TNT Embedded ToolSuite
 *    atexit=on_exit    C library contains on_exit but not atexit
 *
 * Public Functions
 *    cov_file          - set coverage data file path
 *    cov_init          - initialize this module
 *    cov_probe         - handle a probe event
 *    cov_reset         - reset measurements
 *    cov_term          - terminate C-Cover run-time
 *    cov_write         - update data to coverage data file
 *
 * Private Functions
 *    path_is_absolute
 *    path_is_slash
 *    strcat            - same as standard C
 *    strcat_size
 *    strtoul           - same as standard C
 *    ultostr           - convert unsigned long to string
 *    cov_path          - get coverage data filename
 *    error_set         - set current error
 *    error_n_format    - convert number to string
 *    error_write_file  - write line of text to a file descriptor
 *    error_write       - write a string to screen or log file
 *    trace             - diagnostic trace
 *    error_format      - format and output current error
 *    file_read         - helper to read from file
 *    file_read_ulong   - helper to read number from file
 *    file_seek         - helper to seek in file
 *    file_lock,unlock  - helper to lock, unlock file region
 *    buf_fill          - begin buffered i/o
 *    buf_flush         - end buffered i/o
 *    buf_getc          - get buffered char
 *    buf_putc          - put buffered char
 *    cov_open          - open the coverage data file
 *    cov_close         - close coverage data file
 *    cov_obj_next      - read next object in coverage data file directory
 *    object_check      - test coverage structure consistency
 *    object_write_event- write an event
 *    object_write      - write a coverage structure
 *    init_force_bool   - copy boolean force info from force list to object
 *    init_option       - load forced branch data and other options
 */

#if CCOVER
	#error Disable Coverage Build before compiling this source (cov01 -0)
#endif

/*lint -e1924 */

#if _CVI_
#	define SYS_LabW 1
#endif

/* make sure exactly one operating system is targeted */
#if \
	SYS_hpKernel + \
	SYS_LabW + \
	SYS_nt + \
	SYS_UNIX + \
	SYS_user + \
	SYS_VxWorks + \
	SYS_hpemul + \
	SYS_ntGraphic + \
	SYS_ntk + \
	SYS_pharlap + \
	SYS_WindowsCE + \
	0 != 1
	#error must define one operating system
#endif

#include "ccoverVr.h"
#include "ccoverLibVersion.h"

/*========== System Headers ==========================================*/

#if (__hpux || SYS__HPUX) && !defined(_HPUX_SOURCE)
	#define _HPUX_SOURCE 1
#endif
#if SYS__OSF1
	#define _XOPEN_SOURCE 1
#endif
#if SYS__Solaris || SYS__IRIX64
#	define __EXTENSIONS__ 1
#endif
#if !SYS__Solaris
#	define _POSIX_C_SOURCE 199506
#endif
/*lint -esym(750,_POSIX_C_SOURCE) */

#include <stdio.h>
#if SYS_LabW
#	include <lowlvlio.h>
#endif
#if !SYS_LabW && !SYS_hpemul
#	include <sys/types.h> /* for sys/stat.h, near first due to size_t */
#	include <sys/stat.h>
#	include <fcntl.h>
#endif
#include <stdlib.h>
#include <limits.h>
#include <time.h>

#if !SYS_UNIX && !defined(off_t)
#	define off_t long
#endif

#define true  1
#define false 0

#if (!defined(S_IRUSR) || !defined(S_IWUSR))
#	define S_IRUSR S_IREAD
#	define S_IWUSR S_IWRITE
#endif

#if PATH_MAX < 127
#	undef PATH_MAX
#	define PATH_MAX 127
#endif

/*========== Platform-Specific Definitions ===========================*/

/*----------------------------- Constant -------------------------------
 | Name:    Fcntl_timeout
 | Purpose: Number of seconds before giving up a file lock.
 */
#define Fcntl_timeout 40
/*lint -esym(750,Fcntl_timeout) */

/*------------------------- Private Function ---------------------------
 | Name:    unused
 | Args:      p - a parameter or local variable
 | Return:    none
 | Purpose: Reference an unused parameter or local variable to quiet warning.
 */
static void* Std_unused_p;
#define Std_unused(p) (Std_unused_p = (void*)&p)

#if SYS_hpKernel
	#include "covrt-hpKernel.c"
#endif
#if SYS_hpemul
#	include "covrt-hpemul.h"
#endif
#if SYS_LabW
#	define COVRT_NOFCNTL 1
#	include <errno.h>
	int cov_write(void);
	static void error_write_file(int fd, const char * s);
	static void error_write_screen(const char* s)
	{
		error_write_file(2, s);
	}
#endif
#if SYS_ntGraphic
	#include "covrt-ntGraphic.h"
#endif
#if SYS_ntk
#	include "covrtntk.h"
#endif
#if SYS_nt
	#include "covrt-nt.c"
#endif
#if SYS_pharlap
	#include "covrt-pharlap.c"
#endif
#if SYS_UNIX
#	include "covrtunx.h"
#endif
#if SYS_user
	#include "covrt-user.c"
#endif
#if SYS_VxWorks
#	include "covrtvxw.h"
#endif
#if SYS_WindowsCE
	#include "covrt-WindowsCE.c"
#endif

#if !SYS_hpKernel
	#include <signal.h>     /* For SGI IRIX */
	#include "pthread.h" /* not <>, so that -I. not needed */
#endif
#if !defined(PTHREAD_LINK)
#	define PTHREAD_LINK
#endif

/*========== Constants ===============================================*/

/* The product name and version */
static const char Title[] = VERSION_title;

/* The copyright string */
static const char Copyright[] = VERSION_copyright;

/* For the Unix command "what" */
static const char whatString[] = "@(#)" VERSION_title;
/*lint -esym(528 752,whatString) */

/*----------------------------- Constant -------------------------------
 | Name:    Data_...
 | Purpose: Bits used to represent run-time probe information.
 */
#define Data_event          0x0f  /* mask for events only */
#define Data_event_false    0x01  /* condition false occurred */
#define Data_event_true     0x02  /* condition true occurred */
#define Data_event_target   0x04  /* switch label or C++ catch */
#define Data_event_func     0x08  /* function invoked */
#define Data_force          0xf0  /* mask for force only */
#define Data_force_false_11 0x10  /* force false once */
#define Data_force_true_11  0x20  /* force true once */
#define Data_force_false_1n 0x30  /* force false always */
#define Data_force_true_1n  0x40  /* force true always */
#define Data_force_false_21 0x50  /* force on second hit false once */
#define Data_force_true_21  0x60  /* force on second hit true once */
#define Data_force_false_2n 0x70  /* force on second hit false always */
#define Data_force_true_2n  0x80  /* force on second hit true always */

/*----------------------------- Constant -------------------------------
 | Name:    Err_...
 | Purpose: Error codes.  Returned by function cov_write.
 */
#define Err_mem_obj   1  /* memory object structure corrupt */
#define Err_lock      3  /* file locking timeout or interrupt */
#define Err_io        5  /* error on read, write, lseek, or close */
#define Err_invalid   6  /* data file content erroneous */
#define Err_mismatch  7  /* data file does not match program */
#define Err_atexit    8  /* atexit failed */
#define Err_unknown   9  /* unknown or unexpected error */
#define Err_static   10  /* static variables corrupt */
#define Err_mutex    11  /* multi-thread mutual exclusion failed */
#define Err_file_obj 12  /* object not found in file */
#define Err_autosave 13  /* autosave not available */
#define Err_none     14  /* No measurements to report */
#define Err_open1    15  /* cannot open coverage data file (COVFILE not set) */
#define Err_open2    16  /* cannot open coverage data file (COVFILE set) */
#define Err_empty    17  /* Coverage file is empty */
#define Err_identity 18  /* Not a C-Cover file or wrong version */
#define Err_maximum  19  /* invalid error code, numerically greatest */

/*----------------------------- Constant -------------------------------
 | Name:    Msg_...
 | Purpose: Error messages and a table to look them up by error code.
 |          Prefix "1" means mention the coverage filename.
 |          Prefix "2" means mention the object source filename.
 */
static const char Msg_1 [] = "Memory corrupted for object %2";
static const char Msg_2 [] = "";  /* obsolete */
static const char Msg_3 [] = "File locking failure on %1 at %4";
static const char Msg_4 [] = "";  /* obsolete */
static const char Msg_5 [] = "File i/o error on %1 at %4";
static const char Msg_6 [] = "Invalid file %1.  Check COVFILE, rebuild";
static const char Msg_7 [] = "Object mismatch.  Check coverage data file is same one created when building.  Or, try covc option --retain.  Path is %1.  Recompile object %2 (%3)";
static const char Msg_8 [] = "Failure in atexit";
static const char Msg_9 [] = "Unknown error.  Contact technical support";
static const char Msg_10[] = "";  /* message is useless */
static const char Msg_11[] = "Thread sync failed.  Contact technical support";
static const char Msg_12[] = "Object missing or out of date.  Check coverage data file is same one created when building.  Or, try covc option --retain.  Path is %1.  Recompile object %2 (%3)";
static const char Msg_13[] = "Autosave not available";
static const char Msg_14[] = "";
static const char Msg_15[] = "Cannot find %1.  COVFILE is not set";
static const char Msg_16[] = "Cannot open %1.  COVFILE is probably set to an incorrect path";
static const char Msg_17[] = "Coverage file %1 is empty";
static const char Msg_18[] = "Not a C-Cover file or wrong version: %1";
static const char * Msg[] = {
	Msg_1, Msg_2, Msg_3, Msg_4, Msg_5, Msg_6, Msg_7, Msg_8, Msg_9,
	Msg_10, Msg_11, Msg_12, Msg_13,
	Msg_14,
	Msg_15,
	Msg_16,
	Msg_17,
	Msg_18,
};

/*----------------------------- Constant -------------------------------
 | Name:    File_id
 | Purpose: The coverage data file identification string.
 */
static const char File_id[] = "C-Cover  4.3.5\n";

/*----------------------------- Constant -------------------------------
 | Name:    Path_default
 | Purpose: The default coverage data file name.
 */
static const char Path_default[] = "test.cov";

/* Directory seperator
 */
#if SYS_UNIX
#	define Slash "/"
#else
	static const char Slash[] = "\\";
#endif

/*----------------------------- Constant -------------------------------
 | Name:    Force_max
 | Purpose: Maximum possible number of forced branches.
 |          Feel free to increase this value.
 */
#define Force_max 16

/*----------------------------- Constant -------------------------------
 | Name:    Offset_...
 | Purpose: Offset of values in header.
 */
#define Offset_dir      (sizeof(File_id)-1)
#define Offset_option   (Offset_dir + 3 * sizeof("12345678"))

/*========== Types ===================================================*/

#undef bool
#undef uchar
#undef ushort
#undef ulong
#define bool   int
#define uchar  unsigned char
#define ushort unsigned short
#define ulong  unsigned long

#if 1
	#define __int64 long
#endif

/*------------------------------- Type ---------------------------------
 | Name:    OBJECT
 | Purpose: The the run-time data collection structure.
 |          Corresponds 1-to-1 with an object code module.
 */
typedef struct t {
	struct t* next;    /* pointer to next struct in chain */
	unsigned id;        /* Unique object identifier */
	short       signature;  /* integrity check value */
	ushort      data_n;     /* number of entries in data[] */
	char        is_linked;  /* if this struct in chain */
	char        is_changed; /* if data modified since save */
	char        is_found;   /* if this object found during save */
	char        delay_io;   /* true to avoid calling initialize */
	uchar       data[30000];/* events, size > 1 in case bounds checked */
} OBJECT;
#define Object_signature 0x5a6b

/*------------------------------- Type ---------------------------------
 | Name:    STD_LINK
 | Purpose: The function linkage for public functions and atexit callback.
 */
#if !defined(STD_LINK)
#	define STD_LINK
#endif

/*========== Variables ===============================================*/

/*------------------------- Private Variable ---------------------------
 | Name:    Autosave_interval
 | Purpose: The autosave interval, in 1/10 seconds.
 */
static ulong Autosave_interval = 10;

/* True if autosave interval is set to default */
static bool Autosave_isDefault = true;

/*------------------------- Private Variable ---------------------------
 | Name:    Autosave_pthread
 |          Autosave_pthread_valid
 | Purpose: The pthread object for autosave_thread.
 |          ..._valid tells if it has been set.
 */
static pthread_t Autosave_pthread;
static bool Autosave_pthread_valid = false;

/* Set false to stop the autosave thread */
static volatile char Autosave_run = true;

/*------------------------- Private Variable ---------------------------
 | Name:    Dir_next
 | Purpose: The offset of the next file directory entry.
 */
static ulong Dir_next;

/*------------------------- Private Variable ---------------------------
 | Name:    Dir_lock
 | Purpose: The offset of the current file region we have locked.
 */
static ulong Dir_lock;

/* The current error status, never reset. */
static int Error_fatal = 0;
static int Error_info = 0;
static int Error_recoverable = 0;
static int Pthread_errno = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    Error_line
 | Purpose: The line number in this file where the last error occurred.
 */
static unsigned Error_line = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    Error_object
 | Purpose: Pointer to the object which was current where error occurred.
 */
static OBJECT* Error_object = NULL;

static unsigned long Error_offset = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    File
 | Purpose: Low level file i/o information.
 |          Member <limit> prevents reading beyond what we need which
 |          allows us to not bother with file locking at this level.
 */
static struct {
	ulong    offset;    /* seek position of this buffer */
	ulong    limit;     /* maximum number of bytes to read beyond <offset> */
	int fd;             /* file descriptor */
	unsigned buf_i;     /* position of next read in <buf> */
	unsigned buf_n;     /* number of bytes read into buffer */
	char     buf[4*1024+1];
	char     pad[sizeof(int) - 1];
} File;

/*------------------------- Private Variable ---------------------------
 | Name:    Force
 |          Force_n
 | Purpose: Force[0..Force_n-1] is the list of forced branches.
 */
static struct {
	ulong object;   /* the object module, corresponds to OBJECT.id */
	ushort probe;   /* probe number of the branch */
	char action;    /* one of XTFxtf */
} Force[Force_max];
static ushort Force_n = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    Init
 | Purpose: Tells if <initialize> has been called.
 */
static bool Init = false;

/* True when any coverage data has changed since last save */
#if SYS_hpKernel
	#if __cplusplus
		extern "C"
	#endif
#else
	static
#endif
bool cov_isChange = false;

/* Points to chain of OBJECT structures */
#if SYS_hpKernel
	#if __cplusplus
		extern "C"
	#endif
#else
	static
#endif
OBJECT* cov_list = NULL;

/*------------------------- Private Variable ---------------------------
 | Name:    Mutex
 | Purpose: Mutual exclusion variable controls entry into this library.
 |          Ignore warnings such as "missing braces around initializer".
 */
static pthread_mutex_t Mutex = PTHREAD_MUTEX_INITIALIZER;

/* The coverage file path currently being used */
static char path_current[PATH_MAX];

/*------------------------- Private Variable ---------------------------
 | Name:    Path_user
 | Purpose: The coverage data file path set by cov_file, else NULL.
 */
static char* Path_user = NULL;

/* Default fully qualified coverage data file path
 */
static char Path_default_full[PATH_MAX];

/*------------------------- Private Variable ---------------------------
 | Name:    Path_is_set
 | Purpose: True if coverage data file path is set (not allowed to change).
 */
static bool Path_is_set = false;

/* The number of calls to cov_term expected. */
static unsigned Term_count = 0;

/*------------------------- Private Variable ---------------------------
 | Name:    Trace
 | Purpose: Whether or not we are tracing.
 */
static bool Trace = true;

/*========== Private Functions =======================================*/

#if COVRT_NOENV
#	undef getenv
	#define getenv(s) NULL
#endif

#if __cplusplus
extern "C" {
#endif
void STD_LINK cov_reset(void);
void STD_LINK cov_term (void);
int  STD_LINK cov_write(void);
#if __cplusplus
}
#endif

/* Determine if given letter is valid path seperator.
 * Args:
 *   c      - character to test
 */
#if SYS_UNIX
#	define path_is_slash(c) ((c) == '/')
#else
#	define path_is_slash(c) ((c) == '/' || (c) == '\\')
#endif

/* Determine if given path is absolute.
 * Args:
 *   s      - string to test
 */
#if SYS_UNIX
#	define path_is_absolute(s) path_is_slash((s)[0])
#else
#	define path_is_absolute(s) (path_is_slash((s)[0]) || ((s)[1] == ':'))
#endif

/* We define this function to make this module independent of the C library.
 */
#undef strcat
#define strcat COVRT_strcat
static void COVRT_strcat(
	char* s1,
	const char* s2)
{
	while (*s1 != '\0') {
		s1++;
	}
	do {
		*s1++ = *s2;
	} while (*s2++ != '\0');
}

/* Same as strcat except this one takes a buffer size
 */
static void strcat_size(
	char* s1,
	unsigned size,
	const char* s2)
{
	unsigned i = 0;

	while (s1[i] != '\0') {
		i++;
	}
	do {
		if (i >= size) {
			s1[size - 1] = '\0';
			break;
		}
		s1[i++] = *s2;
	} while (*s2++ != '\0');
}

/*------------------------- Private Function ---------------------------
 | Name:    strtoul
 | Args:      standard
 | Return:    standard
 | Purpose: Same as the standard C function.  We define this function to
 |          make this module independent of the C library.
 */
#undef strtoul
#define strtoul(s, endptr, base) COVRT_strtoul(s, endptr)
static ulong COVRT_strtoul(
	const char* s,
	char** endptr)
{
	ulong value;
	value = 0L;
	/* assert(base == 16); */
	while (*s == ' ') {
		s++;
	}
	for (;; s++) {
		unsigned digit;
		if (*s >= '0' && *s <= '9') {
			digit = (uchar)(*s - '0');
		} else if (*s >= 'a' && *s <= 'f') {
			digit = (uchar)(*s - 'a') + 10;
		} else {
			break;
		}
		value = (value << 4) + digit;
	}
	if (endptr != NULL) {
		*endptr = (char*)s;
	}
	return value;
}

/*------------------------- Private Function ---------------------------
 | Name:    ultostr
 | Args:      ul    - unsigned long input
 | Return:    str   - string output
 | Purpose: Convert an unsigned long to a string of 8 hex characters.
 */
static char* COVRT_ultostr(ulong ul)
{
	static char str[8+1];
	static const char hex[16+1] = "0123456789abcdef";
	int i;
	for (i = 7; i >= 0; i--) {
		str[i] = hex[ul & 0xf];
		ul /= 0x10;
	}
	return str;
}

/* Get the index'th coverage file path
 */
static bool cov_path(unsigned index)
{
	int success = false;
	/* Set pathList to comma separated list of paths */
	const char* pathList;
	pathList = Path_user;
	if (pathList == NULL) {
		pathList = getenv("COVFILELIST");
	}
	if (pathList == NULL && index == 0) {
		pathList = getenv("COVFILE");
		if (pathList == NULL) {
			pathList = Path_default_full;
		}
	}
	Path_is_set = true;

	if (pathList != NULL) {
		/* Begin index of substring */
		unsigned begin = 0;
		/* Number of ',' seen */
		unsigned count = 0;
		/* Index to path */
		unsigned i;

		/* Find begin, at index'th comma */
		for (i = 0; count < index && pathList[i] != '\0'; i++) {
			if (pathList[i] == ',') {
				begin = i + 1;
				count++;
			}
		}
		if (count == index) {
			/* Copy until nul or ',' */
			for (i = 0;
				pathList[begin + i] != '\0' &&
				pathList[begin + i] != ',';
				i++)
			{
				path_current[i] = pathList[begin + i];
				success = true;
			}
			path_current[i] = '\0';
		}
	}
	return success;
}

/*------------------------- Private Function ---------------------------
 | Name:    error_write_file
 | Args:      int        fd  - file descriptor
 |            const char *  s   - a string with no new-lines
 | Return:    none
 | Purpose: Write a string followed by a new-line to a file.
 */
static void error_write_file(
	int fd,
	const char* s)
{
	char c;
	unsigned i;
	for (i = 0; s[i] != '\0'; i++)
		;
	write(fd, (char*)s, i);
	/* write newline */
#if !SYS_UNIX
	c = '\r'; (void)write(fd, &c, 1);
#endif
	c = '\n'; (void)write(fd, &c, 1);
}

/*------------------------- Private Function ---------------------------
 | Name:    error_set
 | Args:      int   error   - error number, one of Err_...
 | Return:    none
 | Purpose: Set an error if not already set.
 */
#define error_set(error) error_set_line(error, __LINE__)
static void error_set_line(
	int error,
	unsigned line)
{
	switch (error) {
	case Err_atexit:
	case Err_autosave:
	case Err_none:
		if (Error_info == 0) {
			Error_info = error;
		}
		break;
	case Err_empty:
	case Err_identity:
	case Err_invalid:
	case Err_io:
	case Err_lock:
	case Err_mem_obj:
	case Err_open1:
	case Err_open2:
	case Err_static:
	case Err_unknown:
		if (Error_fatal == 0) {
			Error_fatal = error;
			Error_line = line;
		}
		break;
	default:
		if (Error_recoverable == 0) {
			Error_recoverable = error;
		}
		if (Error_line == 0) {
			Error_line = line;
		}
		break;
	}
}

/* Concatenate the printable representation of <n> to <buf>.
 * Args:
 *   buf    - string buffer
 *   buf_size   - size of buffer
 *   n      - number to convert
 */
static void error_n_format(
	char* buf,
	unsigned buf_size,
	int n)
{
	char buf_n[21];   /* 21 = log10(2^63) + 2 */
	unsigned i;  /* index to <buf_n> */
	int is_negative = 0;

	if (n < 0) {
		is_negative = 1;
		n = -n;
	}
	i = sizeof(buf_n) - 1;
	buf_n[i] = '\0';
	do {
		buf_n[--i] = (char)('0' + n % 10);
		n = n / 10;
	} while (n != 0);
	if (is_negative) {
		buf_n[--i] = '-';
	}
	strcat_size(buf, buf_size, buf_n + i);
}

/* Append the line number to the diagnostic log file named by
 * environment variable COVDIAG.  If COVDIAG is not set, do nothing.
 */
#define trace() if (Trace) trace_line(__LINE__)
static void trace_line(unsigned line)
{
	char* diag;
	diag = getenv("COVDIAG");
	if (diag == NULL) {
		Trace = false;
	} else {
		int fd = open(diag, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
		if (fd == -1) {
			static bool first = true;
			if (first) {
				static char m[] = "Cannot open diag log.  Check COVDIAG";
				first = false;
				error_write_screen(m);
			}
		} else {
			char buf[sizeof("1234")];
			(void)lseek(fd, 0L, SEEK_END);    /* ignore errors */
			buf[0] = '\0';
			error_n_format(buf, sizeof(buf), (int)line);
			error_write_file(fd, buf);
			close(fd);
		}
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    error_write
 | Args:      const char *  s   - a string
 | Return:    none
 | Purpose: Write an error message to the screen or, if COVERR is set,
 |          to the log file.  Customize error_write_screen (in covrt*.h)
 |          rather than this function.
 */
static void error_write(const char* s)
{
	const char* log;
	log = getenv("COVERR");
	if (log == NULL) {
		/* display message to screen */
		error_write_screen(s);
		trace();
	} else {
		/* append message to file */
		int fd = open(log, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
		if (fd == -1) {
			static char m[] = "Cannot open error log.  Check COVERR";
			error_write_screen(m);
			trace();
		} else {
			(void)lseek(fd, 0L, SEEK_END);    /* ignore errors */
			error_write_file(fd, s);
			close(fd);
			trace();
		}
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    error_format
 | Args:      none
 | Return:    none
 | Purpose: Format and write the current error.
 */
static void error_format(void)
{
	static bool once = false;
	char buf[512 + 2 * PATH_MAX];
	if (!once) {
		static const char space[] = " ";
		const char* m;
		int error = Error_fatal;
		if (error == 0) {
			error = Error_recoverable;
		}
		if (error == 0) {
			error = Error_info;
		}
		once = true;
		trace();

		if (error == 0 || error >= Err_maximum) {
			error = Err_unknown;
		}
		buf[0] = '\0';
		/* version identification */
		strcat_size(buf, sizeof(buf), Title);
		strcat_size(buf, sizeof(buf), space);

		/* error code */
		{
			static char error_code[] = "error __: ";
			if (error < 10) {
				error_code[6] = ' ';
				error_code[7] = (char)(error + '0');
			} else {
				error_code[6] = '1';
				error_code[7] = (char)(error + '0' - 10);
			}
			strcat_size(buf, sizeof(buf), error_code);
		}

		/* Format message */
		for (m = Msg[error - 1]; *m != '\0'; m++) {
			if (*m == '%') {
				m++;
				switch (*m) {
				case '1':
					strcat_size(buf, sizeof(buf), path_current);
					break;
				case '2':
					if (Error_object == NULL) {
						Error_object = cov_list;
					}
					strcat_size(buf, sizeof(buf), (char*)(Error_object->data) +
						Error_object->data_n);
					break;
				case '3':
					strcat_size(buf, sizeof(buf), COVRT_ultostr(Error_object->id));
					break;
				case '4':
					strcat_size(buf, sizeof(buf), COVRT_ultostr(Error_offset));
					break;
				default:
					break;
				}
			} else {
				char c[2];
				c[0] = *m;
				c[1] = '\0';
				strcat_size(buf, sizeof(buf), c);
			}
		}

		/* diagnostic line number */
		strcat_size(buf, sizeof(buf), ".  (Error occurred at ");
		error_n_format(buf, sizeof(buf), (int)Error_line);

		/* errno value */
		if (errno != 0 || Pthread_errno != 0) {
			strcat_size(buf, sizeof(buf), ",");
			error_n_format(buf, sizeof(buf),
				error == Err_mutex || error == Err_autosave ?
				Pthread_errno : errno);
		}

		/* Close paren */
		strcat_size(buf, sizeof(buf), ").");

		error_write(buf);
		trace();
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    file_read
 | Args:      char *    buf     - buffer to read into
 |            unsigned  length  - number of bytes to read
 | Return:    int               - same as read()
 | Purpose: Read from the coverage data file.
 |          Store a nul at the end of the buffer.
 */
static void file_read(
	char* buf,
	unsigned length)
{
	if (Error_fatal == 0) {
		int n;
		n = read(File.fd, buf, length);
		if (n < 0) {
			error_set(Err_io); trace();
			Error_offset = (unsigned long)lseek(File.fd, 0, SEEK_CUR);
			buf[0] = '\0';
		} else {
			buf[n] = '\0';
		}
	} else {
		buf[0] = '\0';
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    file_read_ulong
 | Args:      none
 | Return:    ulong n   - a number or 0 if error
 | Purpose: Read a 8 digit hexadecimal number at the current file position.
 */
static ulong file_read_ulong(void)
{
	ulong n;
	char buf[sizeof("12345678")];
	char* end = NULL;
	file_read(buf, sizeof(buf)-1);
	n = strtoul(buf, &end, 16);
	if (end != buf + sizeof(buf)-1) {
		error_set(Err_invalid); trace();
	}
	return n;
}

/* Seek in the coverage data file.  */
static void file_seek(ulong offset)
{
	if (Error_fatal == 0 &&
		lseek(File.fd, (off_t)offset, SEEK_SET) == (off_t)-1)
	{
		error_set(Err_io); trace();
		Error_offset = offset;
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    file_lock
 | Name:    file_unlock
 | Args:      ulong offset  - file position
 | Return:    none
 | Purpose: Lock or unlock one byte at the specified position for
 |          synchronized access.
 */
static int fcntl_isReadLock = true;
#if COVRT_NOFCNTL
#define file_lock(offset, length)      (void)0
#define file_lock_read(offset, length) (void)0
#define file_unlock(offset, length)    (void)0
#else
#define file_lock(offset, length)      file_lock_unlock(offset, length, F_WRLCK)
#define file_lock_read(offset, length) file_lock_unlock(offset, length, F_RDLCK)
#define file_unlock(offset, length)    file_lock_unlock(offset, length, F_UNLCK)
static void file_lock_unlock(
	ulong offset,
	unsigned length,
	short type)
{
	if (Error_fatal == 0) {
		int status;
		struct flock fl;
		fl.l_whence = SEEK_SET;
		fl.l_start = (off_t)offset;
		fl.l_len = (off_t)length;
		if (type == F_RDLCK && !fcntl_isReadLock) {
			fl.l_type = F_WRLCK;
		} else {
			fl.l_type = type;
		}
		status = fcntl(File.fd, F_SETLKW, &fl);
		if (status != 0 && type == F_RDLCK && errno == EINVAL) {
			fcntl_isReadLock = false;
			fl.l_type = F_WRLCK;
			status = fcntl(File.fd, F_SETLKW, &fl);
		}
		if (status != 0) {
			error_set(Err_lock); trace();
			Error_offset = offset;
		}
	}
}
#endif

/*------------------------- Private Function ---------------------------
 | Name:    buf_fill
 | Args:      none
 | Return:    none
 | Purpose: Read a new buffer of data.  Requires that <File.offset> be
 |          set to the file position to access.
 */
static void buf_fill(void)
{
	File.buf_n = sizeof(File.buf) - 1;
	if ((ulong)File.buf_n > File.limit) {
		File.buf_n = (unsigned)File.limit;
	}
	file_seek(File.offset);
	file_read(File.buf, File.buf_n);
	File.buf_i = 0;
	File.buf[File.buf_n] = '\0';
	File.limit -= (unsigned)File.buf_n;
}

/*------------------------- Private Function ---------------------------
 | Name:    buf_flush
 | Args:      none
 | Return:    none
 | Purpose: Write the current buffer of data.
 |          Only write as much of the buffer as we changed.
 |          Advance <File.offset> to the position following the buffer.
 */
static void buf_flush(void)
{
	file_seek((ulong)File.offset);
	/* if buffer has been changed */
	if (Error_fatal == 0 && File.buf_n > 0) {
		if (write(File.fd, File.buf, File.buf_n) != (int)File.buf_n) {
			error_set(Err_io); trace();
			Error_offset = (unsigned long)lseek(File.fd, 0, SEEK_CUR);
		}
	}
	File.offset += (unsigned)File.buf_n;
}

/*------------------------- Private Function ---------------------------
 | Name:    buf_getc
 | Args:      none
 | Return:    int   c   - a char or -1
 | Purpose: Read the next character from the current buffer <File.buf>.
 |          Return -1 if there is an error condition.
 */
static int buf_getc(void)
{
	int c;  /* return value */
	/* if not EOF */
	if (Error_fatal == 0 && File.buf_n > 0) {
		/* check to see if we need to go to the next buffer */
		if (File.buf_i >= File.buf_n) {
			buf_flush();
			buf_fill();
		}
	}
	/* if not EOF */
	if (Error_fatal == 0 && File.buf_n > 0) {
		/* get next char */
		c = File.buf[File.buf_i];
		File.buf_i++;
	} else {
		c = -1;
	}
	return c;
}

/*------------------------- Private Function ---------------------------
 | Name:    buf_putc
 | Args:      char  c   - a char
 | Return:    none
 | Purpose: Write a char, buffered.  Only one call to this function is
 |          allowed between calls to buf_getc.
 */
static void buf_putc(int c)
{
	if (Error_fatal == 0) {
		File.buf[File.buf_i - 1] = (char)c;
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    cov_open
 | Args:      none
 | Return:    none
 | Purpose: Open the coverage data file.  Set <Cov> to the file path.
 */
static bool cov_open(unsigned index)
{
	bool success = false;
	trace();
	if (Error_fatal == 0) {
		if (cov_path(index)) {
			File.fd = open(path_current, O_RDWR, 0);
			if (File.fd == -1) {
				if (getenv("COVFILE") == NULL) {
					error_set(Err_open1);
					trace();
				} else {
					error_set(Err_open2);
					trace();
				}
			} else {
				unsigned i;
				trace();
				file_lock_read(0, sizeof(File_id) - 1);
				File.offset = 0;
				File.limit = sizeof(File_id) - 1;
				buf_fill();
				file_unlock(0, sizeof(File_id) - 1);
				/* check file identification line */
				for (i = 0; i < sizeof(File_id) - 1; i++) {
					if (buf_getc() != File_id[i]) {
						error_set(Err_identity); trace();
						break;
					}
				}
			}
			success = true;
		}
	}
	Dir_next = 0;
	trace();
	return success;
}

/*------------------------- Private Function ---------------------------
 | Name:    cov_close
 | Args:      none
 | Return:    none
 | Purpose: Close the coverage data file.
 */
static void cov_close(void)
{
	if (File.fd != -1) {
		if (close(File.fd) != 0) {
			error_set(Err_io); trace();
		}
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    cov_obj_next
 | Args:      ulong *   id_p        - where to put object id
 |            ulong *   offset_p    - where to put object offset
 | Return:    bool      success     - false after last entry
 | Purpose: Read the coverage file directory to get the next object entry.
 */
static int cov_obj_next(
	unsigned* id_p,
	unsigned* offset_p)
{
	static unsigned directory;
	static unsigned directory_size;
	char ent[sizeof("o12 12345678 12345678")];
	unsigned len;
	char* end = NULL;
	trace();
	*offset_p = 0;

	if (Dir_next == 0) {
		/* first call, lock and read pointer to directory */
		Dir_lock = Offset_dir;
		file_lock_read(Dir_lock, 1);
		file_seek(Dir_lock);
		directory = file_read_ulong();
		if (directory == 0) {
			/* no directory - empty coverage file */
			error_set(Err_empty);
		}
		Dir_next = (unsigned)directory + sizeof("Dir 12345678");
		if (fcntl_isReadLock) {
			/* Lock whole directory */
			file_seek((unsigned)directory + sizeof("Dir ") - 1);
			directory_size = file_read_ulong();
			file_lock_read(directory, directory_size);
			/* Unlock pointer to directory */
			file_unlock(Offset_dir, 1);
		}
	}

	if (!fcntl_isReadLock) {
		/* lock next entry, then unlock previous */
		file_lock_read(Dir_next, 1);
		file_unlock(Dir_lock, 1);
		Dir_lock = Dir_next;
	}
	file_seek(Dir_next);
	/* read length then read entry so we do not read too much - locking */
	file_read(ent, 4);  /* o12, s123, or End\n */
	len = (unsigned)strtoul(ent + 1, &end, 16);
	Dir_next += len;

	if (ent[0] == 'o') {
		/* read object entry */
		file_seek(Dir_next - len);
		file_read(ent, len);
		*id_p = strtoul(end, &end, 16);
		*offset_p = strtoul(end, NULL, 16);
	} else {
		/* found Section_end or error */
		Dir_next = 0;
		/* Unlock directory */
		if (fcntl_isReadLock) {
			file_lock_read(directory, directory_size);
		} else {
			file_unlock(Dir_lock, 1);
		}
		if (ent[0] != 'E') {
			error_set(Err_invalid); trace();
		}
		*id_p = 0;
		*offset_p = 0;
	}
	trace();
	return (Error_fatal == 0 && ent[0] == 'o');
}

/*------------------------- Private Function ---------------------------
 | Name:    object_check
 | Args:      OBJECT *  p       - ptr to memory object
 | Return:    none
 | Purpose: Perform integrity checks on coverage data memory structure
 |          looking for signs that it has been overwritten by spurious
 |          memory writes from the program under test.
 */
static void object_check(OBJECT* p)
{
	trace();

	/* check for wild pointer <p> */
	if (p == NULL || p->signature != Object_signature) {
		error_set(Err_mem_obj); trace();
	}
	/* check structure members */
	if ((p->next != NULL && p->next->signature != Object_signature) ||
		p->data_n == 0 ||
		p->is_linked + p->is_changed > 2)
	{
		error_set(Err_mem_obj); trace();
	}
	trace();
}

/*------------------------- Private Function ---------------------------
 | Name:    object_write_event
 | Args:      unsigned  data        - OBJECT.data[i]
 |            unsigned  event       - a bit, one of Data_event_...
 |            char      char_new    - char for event
 | Return:    none
 | Purpose: Update and check an event character in the file.
 |          This function reads and then may write one char in its place.
 */
static void object_write_event(
	unsigned data,
	unsigned event,
	int char_new)
{
	int c;
	c = buf_getc();
	/* if the <event> is set in <data> */
	if ((data & event) != 0) {
		/* if this is the first time this event occurred */
		if (c == ' ') {
			buf_putc(char_new);
		}
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    object_write
 | Args:      OBJECT *  p       - ptr to memory object
 |            ulong     offset  - where to write
 | Return:    none
 | Purpose: Update an object structure in the file at <offset>.
 */
static void object_write(
	OBJECT* p,
	ulong offset)
{
	int c;      /* input buffer */
	int kind;   /* type of probe in file */
	unsigned i; /* index to p->data[] */

	trace();
	if (p->is_changed) {
		/* Lock file for writing */
		file_lock(offset, 1);
		/* get size */
		file_seek(offset + sizeof("Obj"));
		File.limit = file_read_ulong();
		File.offset = offset;
		buf_fill();
		/* skip object header */
		do {
			c = buf_getc();
		} while (c != -1 && c != '\n');

		/* for each probe */
		for (i = 0; Error_fatal == 0 && i < (unsigned)p->data_n; i++) {
			const unsigned data = p->data[i];

			/* go to next probe */
			kind = buf_getc();
			while (kind == 's' || kind == 'k') {
				/* skip source header or constant */
				do {
					c = buf_getc();
				} while (c != -1 && c != '\n');
				kind = buf_getc();
			}
			/* update events */
			switch (kind) {
			case 'n':
				object_write_event(data, Data_event_func, 'X');
				break;
			case 'c':
				object_write_event(data, Data_event_true,  't');
				object_write_event(data, Data_event_false, 'f');
				break;
			case 'd':
				object_write_event(data, Data_event_true,  'T');
				object_write_event(data, Data_event_false, 'F');
				break;
			case 'h':
			case 'l':
				object_write_event(data, Data_event_target, 'X');
				break;
			case 'E':
				/* more probes in memory than in file */
				error_set(Err_mismatch); trace();
				break;
			default:
				/* invalid char in object */
				error_set(Err_invalid); trace();
				break;
			}
			/* advance to beginning of next line */
			do {
				c = buf_getc();
			} while (c != -1 && c != '\n');
		}
		c = buf_getc();
		if (c != 'E' && c != 'k') {
			/* fewer probes in memory than in file */
			error_set(Err_mismatch); trace();
		}
		buf_flush();
		trace();
		p->is_changed = false;
		/* Release write lock */
		file_unlock(offset, 1);
	}
}

/*------------------------- Private Function ---------------------------
 | Name:    init_force_bool
 | Args:      OBJECT *  object  - current memory object
 | Return:    none
 | Purpose: Transfer forcing information from the force list <Force[]>
 |          to the object <object->data[]>.
 */
static void init_force_bool(OBJECT* object)
{
	ushort f;
	trace();
	for (f = 0; f < Force_n; f++) {
		if (object->id == Force[f].object) {
			/* forced branch found */
			if (Force[f].probe >= object->data_n) {
				/* forced probe is out of range */
				error_set(Err_mismatch); trace();
			} else {
				/* store action in object probe data */
				uchar action = 0;
				switch (Force[f].action) {
				case 'f': action = Data_force_false_11; break;
				case 't': action = Data_force_true_11;  break;
				case 'F': action = Data_force_false_1n; break;
				case 'T': action = Data_force_true_1n;  break;
				case '0': action = Data_force_false_21; break;
				case '1': action = Data_force_true_21;  break;
				case '2': action = Data_force_false_2n; break;
				case '3': action = Data_force_true_2n;  break;
				case 'x': break;
				case 'X': break;
				case 'y': break;
				case 'Y': break;
				default:
					error_set(Err_invalid); trace();
					break;
				}
				object->data[Force[f].probe] |= action;
			}
		}
	}
	trace();
}

/*------------------------- Private Function ---------------------------
 | Name:    init_option
 | Args:      none
 | Return:    none
 | Purpose: Load the list of forced branches from the coverage file into
 |          <Force[]>.
 */
static void init_option(void)
{
	ulong offset;
	bool done;
	trace();
	file_lock_read(Offset_option, 8);
	file_seek(Offset_option);
	offset = file_read_ulong();
	if (offset != 0) {
		trace();
		offset += sizeof("Opt 12345678");
		/* for each option */
		done = false;
		while (Error_fatal == 0 && !done) {
			char line[999];
			unsigned len;
			trace();
			/* read length then read entry to not read too far - locking */
			file_seek(offset);
			file_read(line, sizeof("End"));
			len = (unsigned)strtoul(line + 1, NULL, 16);
			file_seek(offset);
			switch (line[0]) {
			case 'a':
				trace();
				len = sizeof("a12345678");
				file_read(line, len);
				Autosave_interval = (unsigned)strtoul(line + 1, NULL, 16);
				Autosave_isDefault = Autosave_interval == 0xffffffffUL;
				if (Autosave_isDefault) {
					Autosave_interval = 1;
				}
				Autosave_interval *= 10;
				break;
			case 'f':
				trace();
				if (Force_n < sizeof(Force) / sizeof(Force[0])) {
					char* p1 = NULL;
					char* p2 = NULL;
					if (len >= sizeof(line)) {
						len = sizeof(line) - 1;
					}
					file_read(line, len);
					len = (unsigned)strtoul(line + 1, &p1, 16);
					Force[Force_n].object = strtoul(p1, &p2, 16);
					if (p2 == p1) {
						error_set(Err_invalid); trace();
					}
					Force[Force_n].probe = (ushort)strtoul(p2, &p1, 16);
					if (p1 == p2) {
						error_set(Err_invalid); trace();
					}
					Force[Force_n].action = p1[1];
					Force_n++;
				}
				break;
			case 'E':
				done = true;
				break;
			default:
				trace();
				break;
			}
			offset += len;
		}
	}
	file_unlock(Offset_option, 8);
	trace();
}

/*------------------------- Private Function ---------------------------
 | Name:    autosave_thread
 | Args:      none
 | Return:    none
 | Purpose: Write coverage data every <Autosave_interval>.
 |          This function is passed to pthread_create.
 */
#if __cplusplus
extern "C" {
#endif
static void* PTHREAD_LINK autosave_thread(void* arg)
{
	int status = 0;
	ulong count = 0;
	Std_unused(arg);
	do {
		/* Sleep 0.1 seconds */
		struct timespec rqt;
		rqt.tv_sec = 0;
		rqt.tv_nsec = 100000000L;
		nanosleep(&rqt, NULL);
		count++;
		if (count >= Autosave_interval && Autosave_run) {
			status = cov_write();
			count = 0;
		}
	} while (Autosave_run && status == 0);
	Autosave_pthread_valid = false;
	pthread_exit(NULL);
	return NULL;
}
#if __cplusplus
}
#endif

/*------------------------- Private Function ---------------------------
 | Name:    initialize
 | Args:      none
 | Return:    none
 | Purpose: Initialize the C-Cover run-time.
 */
static void initialize(void)
{
	Init = true;

	#if SYS_nt || SYS_pharlap || SYS_UNIX
		/* Preserve initial working directory so coverage data file can be
		 * found there later, but do not if default path has been
		 * hard coded to an absolute path.
		 */
		if (!path_is_absolute(Path_default)) {
			if (getcwd(Path_default_full, sizeof Path_default_full) == NULL) {
				Path_default_full[0] = '\0';
			} else {
				unsigned i;
				for (i = 0; Path_default_full[i] != '\0'; i++)
					;
				if (i > 0 && !path_is_slash(Path_default_full[i-1])) {
					strcat(Path_default_full, Slash);
				}
			}
		}
#	endif

	strcat(Path_default_full, Path_default);
	trace();

	cov_open(0);
	/* load options and force data */
	init_option();
	if (Error_fatal == 0 && Autosave_interval != 0) {
		int status;
		#if SYS_nt && _MSC_VER
			dllLoadLibrary();
		#endif
		status = pthread_create(&Autosave_pthread, NULL, autosave_thread,
			NULL);
		if (status == 0) {
			Autosave_pthread_valid = true;
		} else {
			trace();
		}
		/*lint -save -e774 */
		if (status != 0 && !Autosave_isDefault) {
			Pthread_errno = status;
			error_set(Err_autosave); trace();
		}
		/*lint -restore */
	}

	cov_close();
	trace();

	Std_unused(Std_unused_p);
}

/*========== Public Functions ========================================*/

#if __cplusplus
extern "C" {
#endif

/*------------------------- Public Function ----------------------------
 | Name:    cov_file
 | Args:      path      - string
 | Return:    success   - true if successful, false if already set by probe
 | Purpose: Set the path (name) of the coverage data file.
 |          You must call this function before any C-Cover probe occurs.
 |          This setting overrides environment variable COVFILE.
 */
int STD_LINK cov_file(const char* path)
{
	int success = !Path_is_set;
	if (success) {
		static char buf[PATH_MAX];
		buf[0] = '\0';
		strcat(buf, path);
		Path_user = buf;
	}
	return success;
}

/*------------------------- Public Function ----------------------------
 | Name:    cov_init
 | Args:      none
 | Return:    none
 | Purpose: Initialize this library.
 */
void STD_LINK cov_init(void)
{
	if (Title[0] == 'C' && Copyright[0] == 'C') {
		const int pthread_status = pthread_mutex_lock(&Mutex);
		if (pthread_status != 0) {
			if (Pthread_errno == 0) {
				Pthread_errno = pthread_status;
			}
			error_set(Err_mutex); trace();
		} else {
			trace();
			if (!Init) {
				initialize();
			}
			trace();
			pthread_mutex_unlock(&Mutex);
		}
	}
}

void STD_LINK cov_count(void)
{
	const int pthread_status = pthread_mutex_lock(&Mutex);
	if (pthread_status == 0) {
		Term_count++;
		pthread_mutex_unlock(&Mutex);
	} else {
		if (Pthread_errno == 0) {
			Pthread_errno = pthread_status;
		}
		error_set(Err_mutex); trace();
	}
}

/*------------------------- Public Function ----------------------------
 | Name:    cov_term
 | Args:      none
 | Return:    none
 | Purpose: Terminate the C-Cover run-time.  This function is called
 |          by atexit and then maybe by static destructors.  Termination
 |          is not permanent.  Calls into the C-Cover run-time may continue.
 */
void STD_LINK cov_term(void)
{
	unsigned count = 1;
	const int pthread_status = pthread_mutex_lock(&Mutex);

	if (pthread_status == 0) {
		/* Track termination count and copy to stack */
		Term_count--;
		count = Term_count;
		pthread_mutex_unlock(&Mutex);
	} else {
		if (Pthread_errno == 0) {
			Pthread_errno = pthread_status;
		}
		error_set(Err_mutex); trace();
	}

	/* If last call and auto save thread running */
	if (count == 0 && Autosave_pthread_valid) {
		/* Terminate autosave_thread */
		Autosave_run = false;
		pthread_join(Autosave_pthread, NULL);
	}

	cov_write();

	/* If last call */
	if (count == 0) {
		pthread_mutex_destroy(&Mutex);
	}
}

/*------------------------- Public Function ----------------------------
 | Name:    cov_probe
 | Args:      OBJECT *  p       - ptr to memory object
 |            unsigned  probe   - data array index
 |            unsigned  event   - bit to be set in data minus 1
 | Return:    unsigned  is_true - 1 or 0 for condition event
 | Purpose: Record a probe event.
 */
#define cov_probe VERSION_symbol
int STD_LINK cov_probe(
	void* p_void,
	int probe,
	int event)
{
	/* Avoid recursion into this function as possible from the C library
	 * such as from atexit and getenv which we call
	 */
	static int isInProbe = 0;
	OBJECT* p = (OBJECT*)p_void;
	/* Make sure we can access our static variables. */
	if (Title[0] == 'C' && Copyright[0] == 'C' && !isInProbe) {
		/* Whether the object at "p" is writable.  COM proxy stub code
		 * callback functions get called with the C-Cover data read-only.
		 * Do not write the object when false, to avoid access violation */
		bool isWritable = true;
		isInProbe = 1;
		trace();
		if (p->signature != Object_signature) {
			error_set(Err_mem_obj); trace();
		} else {
			/* if this run-time data structure is not in our chain */
			if (!p->is_linked) {
				const int pthread_status = pthread_mutex_trylock(&Mutex);
				trace();
				if (pthread_status == 0) {
					trace();
					if (!Init && !p->delay_io) {
						initialize();
						#if !COVRT_NOATX
							/* Register termination with atexit */
							trace();
							if (atexit(cov_term) == 0) {
								Term_count++;
							} else {
								error_set(Err_atexit); trace();
							}
						#endif
					}
					object_check(p);
					#if SYS_nt || SYS_pharlap
						isWritable = !IsBadWritePtr(p, 16);
					#endif
					/* Check is_linked again because we first checked it
					 * outside our mutex */
					if (!p->is_linked && Error_fatal == 0 && isWritable) {
						/* link structure into the chain */
						p->is_linked = true;
						p->next = cov_list;
						cov_list = p;
						/* setup for branch forcing */
						init_force_bool(p);
					}
					pthread_mutex_unlock(&Mutex);
				} else if (pthread_status != EBUSY) {
					trace();
					if (Pthread_errno == 0) {
						Pthread_errno = pthread_status;
					}
					error_set(Err_mutex);
				}
			}
			/* make sure probe index is valid */
			if ((unsigned)probe >= (unsigned)p->data_n) {
				error_set(Err_mem_obj); trace();
			}
		}
		if (Error_fatal == 0 && isWritable) {
			volatile uchar* data_p = &p->data[probe];
			const uchar data = *data_p;
			/* check for force */
			if ((data & Data_force) != 0) {
				switch (data & Data_force) {
				case Data_force_false_11:
					event = Data_event_false - 1;   /* force the branch */
					*data_p &= Data_event;          /* remove the force */
					break;
				case Data_force_true_11:
					event = Data_event_true - 1;    /* force the branch */
					*data_p &= Data_event;          /* remove the force */
					break;
				case Data_force_false_1n:
					event = Data_event_false - 1;   /* force the branch */
					break;
				case Data_force_true_1n:
					event = Data_event_true - 1;    /* force the branch */
					break;
				case Data_force_false_21:
					if (data & Data_event) {
						event = Data_event_false - 1;   /* force the branch */
						*data_p &= Data_event;          /* remove the force */
					}
					break;
				case Data_force_true_21:
					if (data & Data_event) {
						event = Data_event_true - 1;    /* force the branch */
						*data_p &= Data_event;          /* remove the force */
					}
					break;
				case Data_force_false_2n:
					if (data & Data_event) {
						event = Data_event_false - 1;   /* force the branch */
						*data_p = (uchar)
							((uchar)(data & Data_event) /* remove the force */
							| Data_force_false_1n);     /* begin 1n force */
					}
					break;
				case Data_force_true_2n:
					if (data & Data_event) {
						event = Data_event_true - 1;    /* force the branch */
						*data_p = (uchar)
							((uchar)(data & Data_event) /* remove the force */
							| Data_force_true_1n);      /* begin 1n force */
					}
					break;
				default:
					break;
				}
			}
			/* if first time this event happened */
			if ((data & (uchar)(event + 1)) == 0) {
				trace();
				/* record the event */
				*data_p |= (uchar)(event + 1);
				cov_isChange = true;
				p->is_changed = true;
			}
		}
		if (Error_fatal + Error_recoverable != 0) {
			if (Error_object == NULL) {
				Error_object = p;
			}
			error_format();
		}
		trace();
		isInProbe = 0;
	}
	/* if event is condition-true then return 1 else return 0 */
	return event;
}

/*------------------------- Public Function ----------------------------
 | Name:    cov_reset
 | Args:      none
 | Return:    none
 | Purpose: Reset measurements to zero coverage.  Callable by user.
 */
void STD_LINK cov_reset(void)
{
	if (Title[0] == 'C' && Copyright[0] == 'C') {
		const int pthread_status = pthread_mutex_lock(&Mutex);
		if (pthread_status != 0) {
			if (Pthread_errno == 0) {
				Pthread_errno = pthread_status;
			}
			error_set(Err_mutex); trace();
		} else {
			OBJECT* p;
			trace();
			if (!Init) {
				initialize();
			}
			for (p = cov_list; Error_fatal == 0 && p != NULL; p = p->next) {
				unsigned i;
				trace();
				object_check(p);
				if (Error_fatal == 0) {
					for (i = 0; i < (unsigned)p->data_n; i++) {
						const uchar data = (uchar)(p->data[i] & Data_force);
						if (p->data[i] != data) {
							/* Mark as change in case file is reset too */
							p->is_changed = true;
							cov_isChange = true;
							p->data[i] = data;
						}
					}
				}
			}

			#if SYS_hpKernel || Covrt_sharedLibrary
				/* Disconnect chain, to avoid pointers into unloaded
				 * shared library */
				for (;;) {
					p = cov_list;
					if (p == NULL) {
						break;
					}
					cov_list = p->next;
					p->next = NULL;
					p->is_linked = false;
				}
			#endif

			if (Error_fatal + Error_recoverable != 0) {
				error_format();
			}
			trace();
			pthread_mutex_unlock(&Mutex);
		}
	}
}

/*------------------------- Public Function ----------------------------
 | Name:    cov_write
 | Args:      none
 | Return:    int     status  - error code or 0
 | Purpose: Write coverage data.
 |          Documented user callable.
 */
int STD_LINK cov_write(void)
{
	int status = 0; /* not static */
#if !SYS_hpKernel
	/* Make sure we can access our static variables. */
	if (Title[0] != 'C' || Copyright[0] != 'C') {
		status = Err_static;
	} else {
		trace();
		if (cov_isChange) {
			const int pthread_status = pthread_mutex_lock(&Mutex);
			if (pthread_status != 0) {
				if (Pthread_errno == 0) {
					Pthread_errno = pthread_status;
				}
				error_set(Err_mutex); trace();
			} else {
				unsigned index = 0;
				OBJECT* p = NULL;
				trace();
				if (!Init) {
					initialize();
				}
				while (cov_open(index)) {
					unsigned obj_id = 0;
					unsigned obj_offset = 0;
					index++;
					/* for each object in directory */
					while (Error_fatal == 0 && cov_obj_next(&obj_id, &obj_offset)) {
						/* search chain for the object */
						for (p = cov_list; Error_fatal == 0 && p != NULL; p = p->next) {
							if (p->id == obj_id) {
								trace();
								p->is_found = true;
								object_check(p);
								object_write(p, obj_offset);
								break;
							}
							else if (p->next == cov_list) {
								/* cycle in linked list */
								error_set(Err_mem_obj); trace();
							}
						}
					}
					cov_close();
				}
				if (Error_fatal + Error_recoverable != 0 &&
					Error_object == NULL)
				{
					Error_object = p;
				}
				/* check that all objects were found */
				for (p = cov_list; Error_fatal == 0 && p != NULL; p = p->next) {
					object_check(p);
					if (!p->is_found) {
						error_set(Err_file_obj); trace();
						if (Error_object == NULL) {
							Error_object = p;
						}
					}
					p->is_found = false;
				}
				cov_isChange = false;
				if (Error_fatal + Error_recoverable != 0) {
					error_format();
				}
				trace();
				pthread_mutex_unlock(&Mutex);
			}
			status = Error_fatal;
			if (status == 0) {
				status = Error_recoverable;
			}
			if (status == 0) {
				status = Error_info;
			}
		}
		if (status == 0 && cov_list == NULL) {
			status = Err_none;
		}
	}
#endif
	return status;
}

#if __cplusplus
}
#endif
