/* $Id: covrt-hpemul.h,v 1.4 1997/07/17 19:11:42 steve Exp $
 * Copyright (c) Bullseye Testing Technology
 *
 * Purpose
 *    C-Cover run-time definitions for HP emulator.
 *
 * POSIX Functions
 *    open, close, read, write, lseek
 *
 * Platform Support
 *    error_write_screen
 */

/*========== POSIX.1 Interface =======================================*/

static int errno;
#define nanosleep(rqt)

#define O_CREAT		0x04
#define O_RDWR		0x02
#define O_WRONLY	0x01
#define S_IREAD		0
#define S_IWRITE	0

#define S_OPEN 		0x90
#define S_CLOSE 	0x91
#define S_READ 		0x92
#define S_WRITE 	0x93
#define S_POS_FILE 	0x95
#define S_RESET	 	0x9A

unsigned char control_addr[4097];
unsigned char *simio_addr = control_addr;

/*lint -e516 -e704 -e732 -e734 -e746 -e774 */

#define open(p, o, m) covrt_open(p, o)
int
covrt_open(const char* path, int open_option)

/*
 *	RETURNS		file descriptor >= 0 -- if the open succeeded.
 *			-1 if the open failed, errno is set the the
 *			error code.
 */

{
	unsigned char *addr_ptr;

	addr_ptr = simio_addr + 1;

	*addr_ptr = open_option;
	addr_ptr++;

	*addr_ptr = 255;
	addr_ptr++;

	while (*path != '\0')
	{
		*addr_ptr = *path;
		addr_ptr++;
		path++;
	}

	*addr_ptr = '\0';

	*simio_addr = S_OPEN;

	while (*simio_addr == S_OPEN)
	{
		/* Empty loop - wait for results */
	}

	if (*simio_addr == 0)
	{
		return (unsigned int) *(simio_addr + 1);
	}
	else
	{
		errno = (unsigned int) *simio_addr;
		return -1;
	}
}

int
close(file_des)
int file_des;

/*
 *	RETURNS		0 for no error.
 *		       -1 for error and the global errno is set to the error code.
 */

{
	*(simio_addr + 1) = file_des;
	*simio_addr = S_CLOSE;

	while (*simio_addr == S_CLOSE)
	{
		/* Empty loop - wait for results */
	}

	if (*simio_addr == 0)
	{
		return 0;
	}
	else
	{
		errno = (unsigned int) *simio_addr;
		return -1;
	}
}

int
read(fd, buffer, nbytes)
int fd;
unsigned char *buffer;
int nbytes;

/*
 *	RETURNS		number of bytes read (>= 0) if sucessfull.
 *		       -1 for error and the global errno is set to the error code.
 */

{
	unsigned char *addr_ptr;
	unsigned char index;

	addr_ptr = simio_addr + 1;
	*addr_ptr = fd;
	addr_ptr++;
	*addr_ptr = nbytes;
	addr_ptr++;

	*simio_addr = S_READ;
	
	while (*simio_addr == S_READ)
	{
		/* Empty loop - wait for results */
	}

	addr_ptr = simio_addr + 3;

	for (index = 0; index < (*(simio_addr + 2)); index++)
	{
		*buffer = *addr_ptr;
		addr_ptr++;
		buffer++;
	}

	if (*simio_addr == 0)
	{
		return (unsigned int) *(simio_addr + 2);
	}
	else
	{
		errno = (unsigned int) *simio_addr;
		return -1;
	}
}

int
write(int fd, const void* buffer, unsigned nbytes)

/*
 *	RETURNS		number of bytes written (>= 0) if sucessfull.
 *		       -1 for error and the global errno is set to the error code.
 */

{
	unsigned char *addr_ptr;
	int index;

	addr_ptr = simio_addr + 1;
	*addr_ptr = fd;
	addr_ptr++;
	*addr_ptr = nbytes;
	addr_ptr++;

	for (index = 0; index < nbytes; index++)
	{
		*addr_ptr = *buffer;
		addr_ptr++;
		buffer++;
	}

	*simio_addr = S_WRITE;
	
	while (*simio_addr == S_WRITE)
	{
		/* Empty loop - wait for results */
	}

	if (*simio_addr == 0)
	{
		return (unsigned int) *(simio_addr + 2);
	}
	else
	{
		errno = (unsigned int) *simio_addr;
		return -1;
	}
}

long
lseek(fd, offset, whence)
int fd;
long offset; 
int whence;

/*
 *	RETURNS		non-negative integer indicating file pointer if sucessfull.
 *		       -1 for error and the global errno is set to the error code.
 */

{
	unsigned char *addr_ptr;

	addr_ptr = simio_addr + 1;
	*addr_ptr = fd;
	addr_ptr++;

	*addr_ptr = offset & 0xff;
	addr_ptr++;
	*addr_ptr = (offset >> 8) & 0xff;
	addr_ptr++;

	/* The following code checks to see if the compiler for this
	 * processor supports 32 bit (4 byte) longs.  If so, the offset
	 * parameter is used.  Otherwise only 16 bits are available and
	 * the offset is sign extended to created a 32 bit integer.  This
	 * code must be altered if files greater than 64K bytes long must
	 * be supported with compilers which do not support 32 bit integers.
	 */

	if (sizeof(long) >= 4)
	{
		*addr_ptr = (offset >> 16) & 0xff;
		addr_ptr++;
		*addr_ptr = (offset >> 24) & 0xff;
		addr_ptr++;
	}
	else  if (offset >= 0)
	{
		/* Compiler does not support 32 bit longs - sign extend.      */

		*addr_ptr = 0;
		addr_ptr++;
		*addr_ptr = 0;
		addr_ptr++;
	}
	else
	{
		*addr_ptr = 0xff;
		addr_ptr++;
		*addr_ptr = 0xff;
		addr_ptr++;
	}

	*addr_ptr = whence;

	*simio_addr = S_POS_FILE;

	while (*simio_addr == S_POS_FILE)
	{
		/* Empty loop - wait for results */
	}

	if (*simio_addr == 0)
	{
		if (sizeof(long) >= 4)
		{
			/* Compiler supports 32 bit longs -- the offset 
			 * is already guarenteed to be non-negitive.
			 */

			return (((long)*(simio_addr + 2) & 0xff) 
				| (((long)*(simio_addr + 3) << 8) & 0xff) 
				| (((long)*(simio_addr + 4) << 16) & 0xff) 
				| (((long)*(simio_addr + 5) << 24) & 0xff));
		}
		else
		{
			/* Truncate most significant bits of absolute offset 
                         * to 15 bits to guarentee that number is non-negative.
			 */

			return (*(simio_addr + 2) & 0xff) | ((*(simio_addr + 3) << 8) & 0x7f);
		}
	}
	else
	{
		errno = (unsigned int) *simio_addr;
		return -1;
	}
}

/* Does not support atexit, fcntl, getenv */
#if !defined(COVRT_NOATX)
#	define COVRT_NOATX 1
#endif
#if !defined(COVRT_NOFCNTL)
#	define COVRT_NOFCNTL 1
#endif
#if !defined(COVRT_NOENV)
#	define COVRT_NOENV 1
#endif

static void error_write_screen P1(const char*, s)
{
	int fd = open("stderr", O_WRONLY, 0);
	if (fd != -1) {
		unsigned length;
		char nl = '\n';
		for (length = 0; s[length] != '\0'; length++)
			;
		write(fd, s, length);
		write(fd, &nl, 1);
		close(fd);
	}
}
