/*
 * Based heavily on code by David Brownell <david-b@pacbell.net>
 * Modifications by Compaq Corporate Research
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
 
/*
 * USB driver for Compaq Personal Jukebox
 */

/*
 * HISTORY
 *
 * 25 December 1999 -- Code taken from dc2xx.c driver
 *
 * Thanks to:  David Brownell for providing this easy to modify example!
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/random.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/malloc.h>
#include <linux/module.h>

#include "usb.h"


// #define	PJB_DEBUG


/* XXX need to get registered minor number, cdev 10/MINOR */
/* XXX or: cdev USB_MAJOR(180)/USB_PJB_MINOR */
#define	USB_PJB_MINOR	176


/* Application protocol limit is 0x8002; USB has disliked that limit! */
#define	MAX_PACKET_SIZE		0x2000		/* e.g. image downloading */

#define	MAX_READ_RETRY		5		/* times to retry reads */
#define	MAX_WRITE_RETRY		5		/* times to retry writes */
#define	RETRY_TIMEOUT		(HZ)		/* sleep between retries */


/* table of pjbs that work through this driver */
static const struct pjb {
	short		idVendor;
	short		idProduct;
	/* plus hooks for pjb-specific info if needed */
} pjbs [] = {
    { 1183, 0x504A },           // Compaq Personal Jukebox
};


struct pjb_state {
	/* these fields valid (dev != 0) iff pjb connected */
	struct usb_device	*dev;		/* USB device handle */
	char			inEP;		/* read endpoint */
	char			outEP;		/* write endpoint */
	const struct pjb	*info;		/* PJB-100, etc */

	/* valid iff isOpen */
	int			isOpen;		/* device opened? */
	int			isActive;	/* I/O taking place? */
	char			*buf;		/* buffer for I/O */

	/* always valid */
	wait_queue_head_t	wait;		/* for timed waits */
};


/* For now, we only support one pjb at a time: there's one
 * application-visible device (e.g. /dev/cpqpjb) and the second
 * (to Nth) pjb detected on the bus is ignored.
 */
static struct pjb_state static_pjb_state;


static ssize_t pjb_read (struct file *file,
	char *buf, size_t len, loff_t *ppos)
{
	struct pjb_state	*pjb;
	int			retries;

	pjb = (struct pjb_state *) file->private_data;
	if (len > MAX_PACKET_SIZE)
		return -EINVAL;
	if (pjb->isActive++)
		return -EBUSY;

	/* Big reads are common, for image downloading.  Smaller ones
	 * are also common (even "directory listing" commands don't
	 * send very much data).  We preserve packet boundaries here,
	 * they matter in the application protocol.
	 */
	for (retries = 0; retries < MAX_READ_RETRY; retries++) {
		unsigned long		count;
		int			result;

		if (signal_pending (current)) {
			pjb->isActive = 0;
			return -EINTR;
		}
		if (!pjb->dev) {
			pjb->isActive = 0;
			return -ENODEV;
		}

		result = usb_bulk_msg (pjb->dev,
			  usb_rcvbulkpipe (pjb->dev, pjb->inEP),
			  pjb->buf, len, &count, HZ*10);

#ifdef	PJB_DEBUG
		printk ("pjb.r (%d) - 0x%x %ld\n", len, result, count);
#endif
		if (!result) {
			if (copy_to_user (buf, pjb->buf, count))
				return -EFAULT;
			pjb->isActive = 0;
			return count;
		}
		if (result != USB_ST_TIMEOUT)
			break;
		interruptible_sleep_on_timeout (&pjb->wait, RETRY_TIMEOUT);
#ifdef	PJB_DEBUG
		printk ("pjb.r (%d) - retry\n", len);
#endif
	}
	pjb->isActive = 0;
	return -EIO;
}

static ssize_t pjb_write (struct file *file,
	const char *buf, size_t len, loff_t *ppos)
{
	struct pjb_state	*pjb;
	ssize_t			bytes_written = 0;

	pjb = (struct pjb_state *) file->private_data;
	if (len > MAX_PACKET_SIZE)
		return -EINVAL;
	if (pjb->isActive++)
		return -EBUSY;
	
	/* most writes will be small: simple commands, sometimes with
	 * parameters.  putting images (like borders) into the pjb
	 * would be the main use of big writes.
	 */
	while (len > 0) {
		char		*obuf = pjb->buf;
		int		maxretry = MAX_WRITE_RETRY;
		unsigned long	copy_size, thistime;

		/* it's not clear that retrying can do any good ... or that
		 * fragmenting application packets into N writes is correct.
		 */
		thistime = copy_size = len;
		if (copy_from_user (obuf, buf, copy_size)) {
			bytes_written = -EFAULT;
			break;
		}
		while (thistime) {
			int		result;
			unsigned long	count;

			if (signal_pending (current)) {
				if (!bytes_written)
					bytes_written = -EINTR;
				goto done;
			}
			if (!pjb->dev) {
				if (!bytes_written)
					bytes_written = -ENODEV;
				goto done;
			}

			result = usb_bulk_msg (pjb->dev,
				 usb_sndbulkpipe (pjb->dev, pjb->outEP),
				 obuf, thistime, &count, HZ*10);
#ifdef	PJB_DEBUG
			if (result)
				printk ("pjb.w USB err - %x\n", result);
#endif
			if (count) {
				obuf += count;
				thistime -= count;
				maxretry = MAX_WRITE_RETRY;
				continue;
			} else if (!result)
				break;
				
			if (result == USB_ST_TIMEOUT) {	/* NAK - delay a bit */
				if (!maxretry--) {
					if (!bytes_written)
						bytes_written = -ETIME;
					goto done;
				}
                                interruptible_sleep_on_timeout (&pjb->wait,
					RETRY_TIMEOUT);
				continue;
			} 
			if (!bytes_written)
				bytes_written = -EIO;
			goto done;
		}
		bytes_written += copy_size;
		len -= copy_size;
		buf += copy_size;
	}
done:
	pjb->isActive = 0;
#ifdef	PJB_DEBUG
	printk ("pjb.w %d\n", bytes_written); 
#endif
	return bytes_written;
}

static int pjb_open (struct inode *inode, struct file *file)
{
	struct pjb_state *pjb = &static_pjb_state;

	/* ignore pjb->dev so it can be turned on "late" */

	if (pjb->isOpen++)
		return -EBUSY;
	if (!(pjb->buf = (char *) kmalloc (MAX_PACKET_SIZE, GFP_KERNEL))) {
		pjb->isOpen = 0;
		return -ENOMEM;
	}
#ifdef	PJB_DEBUG
	printk ("pjb.open\n"); 
#endif
	
	/* Keep driver from being unloaded while it's in use */
	MOD_INC_USE_COUNT;

	pjb->isActive = 0;
	file->private_data = pjb;
	return 0;
}

static int pjb_release (struct inode *inode, struct file *file)
{
	struct pjb_state *pjb;

	pjb = (struct pjb_state *) file->private_data;
	kfree (pjb->buf);
	pjb->isOpen = 0;
	MOD_DEC_USE_COUNT;
#ifdef	PJB_DEBUG
	printk ("pjb.close\n"); 
#endif

	return 0;
}

	/* XXX should define some ioctls to expose pjb type
	 * to applications ... what USB exposes should suffice.
	 * apps should be able to see the pjb type.
	 */
static /* const */ struct file_operations usb_pjb_fops = {
	NULL,		/* llseek */
	pjb_read,
	pjb_write,
	NULL, 		/* readdir */
	NULL,		/* poll */
	NULL,		/* ioctl */
	NULL,		/* mmap */
	pjb_open,
	NULL,		/* flush */
	pjb_release,
	NULL,		/* async */
	NULL,		/* fasync */
	NULL,		/* check_media_change */
	NULL,		/* revalidate */
	NULL,		/* lock */
};

static struct miscdevice usb_pjb = {
	USB_PJB_MINOR,
	"USB PJB (HanGo PJB-100)",
	&usb_pjb_fops
	// next, prev
};



static void * pjb_probe(struct usb_device *dev, unsigned int ifnum)
{
	int				i;
	const struct pjb		*pjb_info = NULL;
	struct usb_interface_descriptor	*interface;
	struct usb_endpoint_descriptor	*endpoint;
	int				direction, ep;

	struct pjb_state		*pjb = &static_pjb_state;

	printk(KERN_INFO "pjb_probe: looking for devices\n");

	/* Is it a supported pjb? */
	for (i = 0; i < sizeof (pjbs) / sizeof (struct pjb); i++) {
		if (pjbs [i].idVendor != dev->descriptor.idVendor)
			continue;
		if (pjbs [i].idProduct != dev->descriptor.idProduct)
			continue;
		pjb_info = &pjbs [i];
		break;
	}
	if (pjb_info == NULL)
		return NULL;

	/* these have one config, one interface */
	if (dev->descriptor.bNumConfigurations != 1
			|| dev->config[0].bNumInterfaces != 1) {
		printk (KERN_INFO "Bogus PJB config info\n");
		return NULL;
	}

	/* models differ in how they report themselves */
	interface = &dev->actconfig->interface[ifnum].altsetting[0];

	if (interface->bNumEndpoints != 4) {
		printk(KERN_INFO "Bogus PJB interface info\n");
		return NULL;
	}

	/* can only show one pjb at a time through /dev ... */
	if (!pjb->dev) {
		pjb->dev = dev;
		printk(KERN_INFO "USB PJB is connected\n");
	} else {
		printk(KERN_INFO "Ignoring additional USB PJB\n");
		return NULL;
	}

	/* get input and output endpoints (either order) */
	endpoint = interface->endpoint;
	pjb->outEP = pjb->inEP =  -1;

	ep = endpoint [2].bEndpointAddress & USB_ENDPOINT_NUMBER_MASK;
	direction = endpoint [2].bEndpointAddress & USB_ENDPOINT_DIR_MASK;
	if (direction == USB_DIR_IN)
		pjb->inEP = ep;
	else
		pjb->outEP = ep;

	ep = endpoint [3].bEndpointAddress & USB_ENDPOINT_NUMBER_MASK;
	direction = endpoint [3].bEndpointAddress & USB_ENDPOINT_DIR_MASK;
	if (direction == USB_DIR_IN)
		pjb->inEP = ep;
	else
		pjb->outEP = ep;

	if (pjb->outEP == -1 || pjb->inEP == -1
			|| endpoint [2].bmAttributes != USB_ENDPOINT_XFER_BULK
			|| endpoint [3].bmAttributes != USB_ENDPOINT_XFER_BULK
			) {
		printk (KERN_INFO "Bogus PJB endpoints\n");
		pjb->dev = NULL;
		return NULL;
	}


	if (usb_set_configuration (dev, dev->config[0].bConfigurationValue)) {
		printk (KERN_INFO "Failed usb_set_configuration: pjb\n");
		pjb->dev = NULL;
		return NULL;
	}

	pjb->info = pjb_info;
	return pjb;
}

static void pjb_disconnect(struct usb_device *dev, void *ptr)
{
	struct pjb_state	*pjb = (struct pjb_state *) ptr;

	if (pjb->dev != dev)
		return;

	/* Currently not reflecting this up to userland; at one point
	 * it got called on bus reconfig, which we clearly don't want.
	 * A good consequence is the ability to remove PJB for
	 * a while without apps needing to do much more than ignore
	 * some particular error returns.  On the bad side, if one
	 * PJB is swapped for another one, we won't be telling.
	 */
	pjb->info = NULL;
	pjb->dev = NULL;

	printk (KERN_INFO "USB PJB disconnected\n");
}

static /* const */ struct usb_driver pjb_driver = {
	"cpqpjb",
	pjb_probe,
	pjb_disconnect,
	{ NULL, NULL },

	NULL,	/* &usb_pjb_fops, */
	0	/* USB_PJB_MINOR */
};


#ifdef MODULE
static __init
#endif
int usb_cpqpjb_init(void)
{
	struct pjb_state *pjb = &static_pjb_state;

	pjb->dev = NULL;
	pjb->isOpen = 0;
	pjb->isActive = 0;
	init_waitqueue_head (&pjb->wait);

	printk(KERN_INFO "Initializing PJB driver\n");

	if (usb_register (&pjb_driver) < 0)
		return -1;
	misc_register (&usb_pjb);

	return 0;
}

#ifdef MODULE
static __exit
#endif
void usb_cpqpjb_cleanup(void)
{
	usb_deregister (&pjb_driver);
	misc_deregister (&usb_pjb);
}


#ifdef MODULE

MODULE_AUTHOR("Compaq Corporate Research");
MODULE_DESCRIPTION("USB Driver for Compaq Personal Jukebox");

module_init (usb_cpqpjb_init);
module_exit (usb_cpqpjb_cleanup);

#endif	/* MODULE */
