/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Implementation of the TOC class		File: TOC.C		**
   **									**
   **  This module contains routines to manipulate the table-of-	**
   **  contents of the Personal Jukebox.				**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "toc.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#ifdef __WIN32__
#define strcasecmp strcmpi
#endif


TOC_Ctx	*MakeTocCtx(int tocType)
{
    TOC_Ctx *ctx = (TOC_Ctx *) malloc(sizeof(struct _TOC_Ctx));
    ctx->root = NULL;
    ctx->tocType = tocType;
    ctx->cdUIDs = NULL;
    ctx->uidCur = 0;
    ctx->uidMax = 0;
    return (ctx);
}

void	FreeTocCtx(TOC_Ctx *ctx)
{
    int i;
    if (ctx->root != NULL) {
	FreeNode(ctx->root);
	}
    for (i=0;i<ctx->uidCur;i++) {
	free((void *) ctx->cdUIDs[i]);
	}
    free((void *) ctx->cdUIDs);
    free((void *) ctx);
}

void InitIOBuf(TOC_IOBuf *bf, LPCTSTR s)
{
    bf->s = (char *) s;
    bf->pos = 0;
    bf->maxlen = (s == NULL) ? 0 : strlen(s);
}

int InputTocLine(TOC_IOBuf *bf, char *pc, char *ss)
{
    int			pos = bf->pos;
    int			bpos = pos;
    char		*s = bf->s;

    if (s == NULL)
	return (0);
	
    while (s[pos] == '\r' || s[pos] == '\n')
	pos++;

    if (s[pos] == '\0' || pos >= bf->maxlen) {
	bf->pos = pos;
	return 0;
	}

    *pc = s[pos++];
    bpos = pos;
    while (s[pos] != '\0' && s[pos] != '\r' && s[pos] != '\n')
	pos++;

    if ((pos-bpos) > MAX_TOC_STR)
	return (-1);
			
    memcpy(ss, s+bpos, pos - bpos);
    ss[pos - bpos] = '\0';
    bf->pos = pos;
    return (1);
}

void OutputTocLine(TOC_IOBuf *bf, char c, LPCTSTR t)
{
    int len = strlen(t);
    int pos = bf->pos;
    // +3 in next line for "c" + "\n\0"
    if (len + 3 >= (bf->maxlen - bf->pos)) {
	bf->maxlen = (bf->maxlen == 0) ? 1024 : bf->maxlen *2;
	bf->s = (char *) realloc(bf->s, bf->maxlen);
	}
    bf->s[pos++] = c;
    strcpy(bf->s + pos, t);
    pos += len;
    bf->s[pos++] = '\n';
    bf->s[pos] = '\0';
    bf->pos = pos;
}

JNode MakeNode(JNodeType type, LPCTSTR text)
{
    JNode	n = (JNode) malloc (sizeof(struct _JNodeRec));
    n->parent = NULL;
    n->child = NULL;
    n->sibling = NULL;
    n->type = type;
    n->text = (LPCTSTR) malloc(strlen(text)+1);
    strcpy((char *) n->text, text);
    memset(&n->trackAddr, 0, sizeof(n->trackAddr));
    n->time = 0;
    n->iCdUID = -1;
    n->iTrack = -1;
    n->bit_rate = 1;
    return n;
}

JNode CopyNode(JNode node)
{
    JNode	n = MakeNode(node->type, node->text);
    n->child = CopyChildren(node->child, n);
    n->enc_type = node->enc_type;
    n->trackAddr = node->trackAddr;
    n->time = node->time;
    n->iCdUID = node->iCdUID;
    n->iTrack = node->iTrack;
    n->bit_rate = node->bit_rate;
    return n;
}

JNode CopyChildren(JNode node, JNode parent)
{
    JNode nn = NULL;
    JNode prev;
    while (node != NULL) {
	JNode	n = MakeNode(node->type, node->text);
	n->parent = parent;
	n->child = CopyChildren(node->child, n);
	n->sibling = NULL;
	n->enc_type = node->enc_type;
	n->trackAddr = node->trackAddr;
	n->time = node->time;
	n->iCdUID = node->iCdUID;
	n->iTrack = node->iTrack;
	n->bit_rate = node->bit_rate;
	if (nn != NULL)
	    prev->sibling = n;
	else
	    nn = n;
	prev = n;
	node = node->sibling;
	}
    return nn;
}

void UnlinkNode(JNode node)
{
    JNode parent = node->parent;
    //ASSERT(parent != NULL);

    JNode *pNode = &parent->child;
    while (node != *pNode) {
	//ASSERT(*pNode != NULL);
	pNode = &(*pNode)->sibling;
	}
    *pNode = node->sibling;
    node->sibling = NULL;
}

void SetNodeText(JNode n, LPCTSTR text)
{
    free((void *) n->text);
    n->text = (char *) malloc(strlen(text)+1);
    strcpy((char *) n->text, text);
}

void FreeNode(JNode node)
{
    JNode n = node->child;
    while (n != NULL) {
	JNode next = n->sibling;
	FreeNode(n);
	n = next;
	}
    free((void *) node->text);
    free((void *) node);
}

int		InputToc(TOC_Ctx *ctx, TOC_IOBuf *bf)
{
    int					done = 0;
    int					first = 1;
    JNode				parent = NULL;
    int					n, major_ver, minor_ver;
    long				bitRate;
    int					rc = 0;

    while (!done) {
		
	JNodeType	type = JNODE_TYPE_NULL;
	char		ss[MAX_TOC_STR + 1];
	char		tc;

	int x = InputTocLine(bf, &tc, ss);
	if (x < 0)
	    return TOC_ERROR_FORMAT;
	if (x == 0) {
	    if (first) {
		ctx->root = MakeNode(
		    JNODE_TYPE_ROOT, DEFAULT_NEW_JUKEBOX_NAME);
		return (0);
		}
	    return TOC_ERROR_FORMAT;
	    }
	first = 0;

	switch (tc) {
	    case TC_Version:
		if (sscanf(ss, TC_VERSION_FORMAT, &major_ver, &minor_ver) != 2)
		    return TOC_ERROR_VERSION;
		if (major_ver > TC_MAJ_VERSION)
		    return TOC_ERROR_VERSION;
		break;
	    case TC_EndSegment:
		done = 1;
		break;
	    case TC_Block:
		if (parent == NULL || parent->type != JNODE_TYPE_TRACK)
		    return TOC_ERROR_FORMAT;

		switch (ctx->tocType) {
		    case TOC_TYPE_PJB:
			n = sscanf(ss, TC_PJB_TRACKADDR_FORMAT,
				   &parent->enc_type,
				   &parent->trackAddr.pjb.start_block,
				   &parent->trackAddr.pjb.start_block_offset,
				   &parent->trackAddr.pjb.end_block,
				   &parent->trackAddr.pjb.end_block_bytes);
			if (n != 5)
			    return (TOC_ERROR_FORMAT);
			break;
		    default:
			//ASSERT(FALSE);
			break;
		    }
		break;
	    case TC_ID:
		if (parent == NULL || parent->type != JNODE_TYPE_TRACK)
		    return TOC_ERROR_FORMAT;
			
		if (sscanf(ss, TC_TRACK_ID_FORMAT,
			   &parent->iCdUID, &parent->iTrack,
			   &parent->time, &bitRate) != 4) {
		    if (sscanf(ss, TC_TRACK_ID_FORMAT,
			       &parent->iCdUID, &parent->iTrack,
			       &parent->time) != 3) {
			return TOC_ERROR_FORMAT;
			}
		    bitRate = TC_BR_128;
		    }
		if (bitRate < 1024)
		    parent->bit_rate = 8000 * bitRate;
		else
		    parent->bit_rate = bitRate;
		break;
	    case TC_Set:
		type = JNODE_TYPE_SET;
		break;
	    case TC_Disk:
		type = JNODE_TYPE_DISK;
		break;
	    case TC_Track:
		type = JNODE_TYPE_TRACK;
		break;
	    case TC_Root:
		type = JNODE_TYPE_ROOT;
		break;
	    case TC_UidMap:
		(void) IndexFromCdUID(ctx, ss);
		break;
	    case '\0':
		return TOC_ERROR_FORMAT;
		break;
	    default:
		break;
	    }
	if (type != JNODE_TYPE_NULL) {
	    JNode		nn;
	    JNodeType   ltype;
	    ltype = (parent == NULL) ?
		JNODE_TYPE_NULL : parent->type;
	    if (type > ltype + 1)
		return TOC_ERROR_FORMAT;
	    while (type <= ltype) {
		parent = parent->parent;
		ltype = (parent == NULL) ?
		    JNODE_TYPE_NULL : parent->type;
		}
	    nn = MakeNode(type, ss);
	    if (ctx->root == NULL) {
		if (type != JNODE_TYPE_ROOT)
		    return TOC_ERROR_FORMAT;
		ctx->root = nn;
		} 
	    else {
		JNode *pnn = &parent->child;
		while (*pnn != NULL)
		    pnn = &(*pnn)->sibling;
		*pnn = nn;
		nn->parent = parent;
		nn->sibling = NULL;
		}
	    parent = nn;
	    }
	}
    return (0);
}

void	OutputToc(TOC_Ctx *ctx, TOC_IOBuf *bf)
{
    char		s[MAX_TOC_STR];
    JNode		curr = ctx->root;
    long		br_enc;
    int			i;

    sprintf(s, TC_VERSION_FORMAT, TC_MAJ_VERSION, TC_MIN_VERSION);
    OutputTocLine(bf, TC_Version, s);
    while (curr != NULL) {
	switch (curr->type) {
	    case JNODE_TYPE_ROOT:
		OutputTocLine(bf, TC_Root, curr->text);
		break;
	    case JNODE_TYPE_SET:
		OutputTocLine(bf, TC_Set, curr->text);
		break;
	    case JNODE_TYPE_DISK:
		OutputTocLine(bf, TC_Disk, curr->text);
		break;
	    case JNODE_TYPE_TRACK:
		OutputTocLine(bf, TC_Track, curr->text);
		if (curr->bit_rate % 8000 == 0 && curr->bit_rate < 8000 * TC_BR_ESCAPE)
		    br_enc = curr->bit_rate / 8000;
		else
		    br_enc = curr->bit_rate;
		switch (ctx->tocType) {
		    case TOC_TYPE_PJB:
			sprintf(s, TC_PJB_TRACKADDR_FORMAT,
				curr->enc_type,
				curr->trackAddr.pjb.start_block,
				curr->trackAddr.pjb.start_block_offset,
				curr->trackAddr.pjb.end_block,
				curr->trackAddr.pjb.end_block_bytes);
			break;
		    default:
			//ASSERT(FALSE);
			break;
		    }
		OutputTocLine(bf, TC_Block, s);
		sprintf(s, TC_TRACK_ID_FORMAT, curr->iCdUID, curr->iTrack, curr->time, br_enc);
		OutputTocLine(bf, TC_ID, s);
		break;
	    default:
		break;
	    }
	if (curr->child != NULL) {
	    curr = curr->child;
	    } 
	else {
	    do {
		if (curr->sibling != NULL) {
		    curr = curr->sibling;
		    break;
		    }
		curr = curr->parent;
		} while (curr != NULL);
	    }
	}
		
    // write the UID map
    for (i=0;i<ctx->uidCur;i++) {
	OutputTocLine(bf, TC_UidMap, ctx->cdUIDs[i]);
	}
    OutputTocLine(bf, TC_EndSegment, "");
}

int IndexFromCdUID(TOC_Ctx *ctx, LPCTSTR uid)
{
    int	iCdUID = -1;
    int i;
    for (i=0;i<ctx->uidCur;i++) {
	if (strcmp(uid, ctx->cdUIDs[i]) == 0) {
	    iCdUID = i;
	    break;
	    }
	}
    if (iCdUID < 0) {
	if (ctx->uidCur == ctx->uidMax) {
	    ctx->uidMax = (ctx->uidMax == 0) ? 16 : 2 * ctx->uidMax;
	    ctx->cdUIDs = (LPCTSTR *) realloc(
		(void *) ctx->cdUIDs, ctx->uidMax * sizeof (LPCTSTR));
	    }
	iCdUID = ctx->uidCur++;
	ctx->cdUIDs[iCdUID] = (LPCTSTR) malloc(strlen(uid)+1);
	strcpy((char *) ctx->cdUIDs[iCdUID], uid);
	}
    return (iCdUID);
}

LPCTSTR CdUIDFromIndex(TOC_Ctx *ctx, int i)
{
    return (ctx->cdUIDs[i]);
}

void FormatTrackAddr(char *str, TOC_Ctx *ctx, JNode n)
{
    LPCTSTR ext = "";
    switch (ctx->tocType) {
	case TOC_TYPE_PJB:
	    sprintf(str, "%ld.%ld .. %ld.%ld",
		    n->trackAddr.pjb.start_block,
		    n->trackAddr.pjb.start_block_offset,
		    n->trackAddr.pjb.end_block,
		    n->trackAddr.pjb.end_block_bytes);
	    break;
	default:
	    //ASSERT(FALSE);
	    break;
	}
}


static JNode FindChild(JNode parent, LPCTSTR name)
{
    JNode node;

    if (parent == NULL) return NULL;

    node = parent->child;

    while (node != NULL) {
	if (strcasecmp(node->text, name) == 0)
	    return node;
	node = node->sibling;
	}

    return NULL;
}


// XXX Note: after calling DeleteNode the map must be rebuilt!

int DeleteNode(JNode node)
{
    UnlinkNode(node);
    FreeNode(node);
    return (0);
}

void InsertNode(JNode jSrc, JInsertPt *pIns,
		int bReplace, int bCopy)
{
    JNode parent,pred,rep;
    int i;
    char *dname,*name;
    char tempstr[1024];

    parent = pIns->parent;
    pred = pIns->pred;
    jSrc->parent = parent;

    assert(parent != NULL);
	
    //make sure we have a unique name
    i = 0;
    dname = (char *) jSrc->text;
    rep = NULL;

    for (;;) {
	if (i == 0) {
	    name = dname;
	    } 
	else if (!bCopy) {
	    sprintf(tempstr,"%s (%d)", dname, i+1);
	    name = tempstr;
	    } 
	else if (i == 1) {
	    sprintf(tempstr,"Copy of %s", dname);
	    name = tempstr;
	    } 
	else {
	    sprintf(tempstr,"Copy (%d) of %s", i, dname);
	    name = tempstr;
	    }
	rep = FindChild(parent, name);
	if (rep == NULL || bReplace)
	    break;
	i++;
	}
    if (i != 0)
	SetNodeText(jSrc, name);

    if (rep != NULL) {
	// replace existing node -- find predecessor
	JNode xx = parent->child;
	pred = JNODE_FIRST;
	while (xx != rep) {
	    pred = xx;
	    xx = xx->sibling;
	    }
	}

    if (pred == JNODE_FIRST) {
	jSrc->sibling = parent->child;
	parent->child = jSrc;
	} else if (pred != JNODE_LAST) {
	    jSrc->sibling = pred->sibling;
	    pred->sibling = jSrc;
	    } 
    else {
	JNode *pnn = &parent->child;
	while (*pnn != NULL)
	    pnn = &(*pnn)->sibling;
	*pnn = jSrc;
	jSrc->sibling = NULL;
	}
    if (rep != NULL)
	DeleteNode(rep);
}
