/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  High-level PJB operations		File: PJBTOC.C		**
   **									**
   **  This is the high level interface to the PJB.  From this		**
   **  library, you can manipulate a PJB's TOC, add files to the	**
   **  PJB, delete files, etc.						**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>

#include <fcntl.h>

#ifdef __WIN32__
#include <io.h>
#endif

#include "pjbtoc.h"

/*  *********************************************************************
    *  Macros								*
    ********************************************************************* */

#define MAXPJBIDS 10

#ifdef __WIN32__
#define strcasecmp strcmpi
#endif

#ifdef __linux__
#define O_BINARY 0
#endif

/*
 * Note: the routines that start PJBTOC_ are the "documented"
 * entry points to the library.   The other routine should
 * be static and are used internally.
 */

/*  *********************************************************************
    *  Forward declarations						*
    ********************************************************************* */

static int RebuildMap(PJBTOC *t);


/*  *********************************************************************
    *  24-bit helper routines						*
    *  									*
    *  These routines are useful for stuffing 24-bit numbers		*
    *  into buffers.  The PJB's DSP uses 24-bit ints.			*
    ********************************************************************* */

static int ReadBE24(u8 *p)
    // read a 24-bit big endian value from buffer
{
    int x = *p++ << 16;
    x |= *p++ << 8;
    x |= *p;
    if ((x & 0x800000) != 0)
	x |= 0xff000000;
    return (x);
}

static void WriteBE24(u8 *p, int x)
    // write a 24-bit big endian value to buffer
{
    *p++ = ((x >> 16) & 0xFF) ; *p++ = ((x >> 8) & 0xFF) ; *p = (x & 0xFF); 
}

static int ReadBE32(u8 *p)
{
    int x = *p++ << 24;
    x |= *p++ << 16;
    x |= *p++ << 8;
    x |= *p;
    return (x);
}

static void WriteBE32(u8 *p, unsigned long x)
{ 
    *p++ = (u8) ((x >> 24) & 0xFF) ; *p++ = (u8) ((x >> 16) & 0xFF) ;
    *p++ = (u8) ((x >> 8) & 0xFF) ; *p = (u8) (x & 0xFF); 
}


/*  *********************************************************************
    *  PJBTOC_OpenPjb(idx,tptr,readtoc)					*
    *  									*
    *  Open a PJB and read its table of contents.  This routie		*
    *  is used to create a PJBTOC structure and read and parse		*
    *  the TOC.  Once this is called, you can manipulate		*
    *  the data on a PJB.						*
    *  									*
    *  Input parameters: 						*
    *  	   idx - index of which PJB to open (0 opens the first PJB,	*
    *  	         1 opens the second, ...)				*
    *  	   tptr - pointer to a pointer to a PJBTOC to be returned	*
    *  	   readtoc - if FALSE, does not read the TOC.  This can		*
    *  	        be useful to connect to a PJB for flash update		*
    *  	        operations only, where reading the TOC might not	*
    *  	        be possible						*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error						*
    ********************************************************************* */

int PJBTOC_OpenPjb(int idx,PJBTOC **tptr,int readtoc)
{
    PJB_INFO ids[MAXPJBIDS];
    int cnt;
    int err;
    PJBTOC *t;
    TOC_IOBuf tociobuf;
    char *tocstr;

    *tptr = NULL;

    cnt = PJB_Enumerate(ids,MAXPJBIDS);
    if (idx >= cnt) return PJB_ERR_NODEVICE;

    t = calloc(1,sizeof(PJBTOC));
    if (!t) return PJB_ERR_NO_RESOURCES;

    err = PJB_Open(&(ids[idx].id),&(t->pjb));
    if (err || (t->pjb == NULL)) {
	free(t);
	return PJB_ERR_TIMEOUT;
    }

    memcpy(&(t->info),&ids[idx],sizeof(PJB_INFO));

    /* this special hack is used for flash updates.  For the
       most part, 'readtoc' should always be TRUE */

    if (!readtoc) {
	*tptr = t;
	return 0;	
	}

    err = PJB_GetDiskInfo(t->pjb,&(t->diskinfo));
    if (err) {
	PJB_Close(t->pjb);
	free(t);
	return err;
    }

    tocstr = malloc(t->diskinfo.curclicksintoc*PJB_ClickSize + 1);
    if (!tocstr) {
	PJB_Close(t->pjb);
	free(t);
	return PJB_ERR_NO_RESOURCES;
    }

    err = 0;
    for (cnt = 0; cnt < t->diskinfo.curclicksintoc; cnt++) {
	memset(tocstr+cnt*PJB_ClickSize,0,PJB_ClickSize);
	err = PJB_ReadTOCClick(t->pjb,cnt,tocstr+cnt*PJB_ClickSize);
	if (err < 0) break;
	}

    if (err < 0) {
	PJB_Close(t->pjb);
	free(tocstr);
	free(t);
	return PJB_ERR_NO_RESOURCES;
    }

    t->toc = MakeTocCtx(TOC_TYPE_PJB);

    t->allocmap = AMapCreate(t->diskinfo.numallocblocks);

    InitIOBuf(&tociobuf,tocstr);

    err = InputToc(t->toc,&tociobuf);
    if (err == 0) {
	err = AMapInput(t->allocmap,&tociobuf);
    }

    // We no longer need tocstr or the TocIOBuf.
    free(tocstr);

    if (err == 0) {
	*tptr = t;
	return 0;
    }


    PJBTOC_Close(t);

    return PJB_ERR_BADPARM;
}


/*  *********************************************************************
    *  PJBTOC_Close(t)							*
    *  									*
    *  Close an open PJBTOC structure, freeing allocated memory		*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure pointer					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

void PJBTOC_Close(PJBTOC *t)
{
    if (t->allocmap) AMapDestroy(t->allocmap);
    if (t->toc) FreeTocCtx(t->toc);
    PJB_Close(t->pjb);
    free(t);
}


/*  *********************************************************************
    *  WriteAllocBlock(t,buffer,blk)					*
    *  									*
    *  Write 128Kbytes of data to a PJB					*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   buffer - address of 128KB block				*
    *  	   blk - block # to write on the PJB				*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error code					*
    ********************************************************************* */

static int WriteAllocBlock(PJBTOC *t,u8 *buffer,AllocMap_Block blk)
{
    int i;
    int prc = 0;

    for (i=0; i<AB_NClicks; i++) {
	prc = PJB_WriteClick(t->pjb,(char *) buffer+i*PJB_ClickSize,i,blk);
	if (prc != 0) {
	    break;
	}
    }

    return prc;
}


/*  *********************************************************************
    *  WriteToc(t,toc)							*
    *  									*
    *  Write a table-of-contents to the PJB, then commit the TOC	*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   toc - table of contents string (note: could be 1MB long!)	*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error code					*
    ********************************************************************* */
static int WriteToc(PJBTOC *t,LPCTSTR toc)
{
    int prc = 0;
    int i;
    int n = strlen(toc);
    int nclicks = n / PJB_ClickSize;

#if 0			/* uncomment to write the toc to a local temp file */
    FILE *str;
    str = fopen("new.toc","wt");
    fprintf(str,"%s",toc);
    fclose(str);
#endif

    for (i=0; i<nclicks; i++) {
		
	prc = PJB_WriteTOCClick(t->pjb, i, (char *) toc + i*PJB_ClickSize);
	if (prc != 0) {
	    break;
	}
    }

    if (prc == 0) {
	int nlast = (n % PJB_ClickSize);
	PJB_Checksum chk;
	memset(chk.b, 0, sizeof(chk.b));
	cksum((void *) toc, n-nlast, chk.b);
	if (nlast != 0) {
	    // pad the TOC to even click size
	    char lastClick[PJB_ClickSize];
	    memcpy(lastClick, toc+n-nlast, nlast);
	    while (nlast != PJB_ClickSize)
		lastClick[nlast++] = '\n';
	    cksum((void *) lastClick, PJB_ClickSize, chk.b);
	    prc = PJB_WriteTOCClick(t->pjb, nclicks++, lastClick);
	}
	if (prc == 0) {
	    prc = PJB_CommitTOC(t->pjb, nclicks, &chk);
	}
    }

    return prc;

}

/*  *********************************************************************
    *  OutputTocCtx(t,toc,ctx)						*
    *  									*
    *  Create a large string containing the complete PJB TOC		*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   toc - pointer to pointer to string				*
    *  	   ctx - TOC context structure					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void OutputTocCtx(PJBTOC *t,LPCTSTR *toc, TOC_Ctx *ctx)
{
    TOC_IOBuf bf;
    InitIOBuf(&bf, NULL);
  
    OutputToc(ctx, &bf);
    AMapOutput(t->allocmap,&bf);
    *toc = bf.s;
}


/*  *********************************************************************
    *  PJBTOC_FlushJukebox(t)						*
    *  									*
    *  Flush the modified TOC to the PJB and recompute the map.		*
    *  This is the call you should make after adding data to		*
    *  a PJB.								*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok else error code					*
    ********************************************************************* */

int PJBTOC_FlushJukebox(PJBTOC *t)
{
    LPCTSTR s = NULL;
    int res;

    // no need to do this if nothing has changed.
    if (t->modified == FALSE) return 0;

    // Scan the TOC to reconstruct the free list
    RebuildMap(t);

    // Create the text version of the TOC
    OutputTocCtx(t,&s,t->toc);

    // Write the TOC to the jukebox
    res = WriteToc(t,s);

    // Free the string we just allocated
    free((char  *) s);

    t->modified = FALSE;

    return res;

}


/*  *********************************************************************
    *  RebuildMapFunc(t,n,arg)						*
    *  									*
    *  This is a tree-walker callback used for rebuilding the		*
    *  allocation map.  It is used when we think the map		*
    *  has changed and needs to be rebuilt.				*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   n - the current JNode					*
    *  	   arg - passed in from the tree walker, the ALLOCMAP struct	*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep going						*
    *  	   !=0 to stop							*
    ********************************************************************* */

static int RebuildMapFunc(PJBTOC *t,JNode n,void *arg)
{
    ALLOCMAP *newmap = (ALLOCMAP *) arg;
    AllocMap_Block b;
    AllocMap_Block start, link;

    if (n->type != JNODE_TYPE_TRACK)
	return TRUE;
	
    b = n->trackAddr.pjb.end_block;
    do {
	if (AMapFollowChain(t->allocmap,b, &start, &link) < 0) {
	    return -1;
	}
	if (n->trackAddr.pjb.start_block >= start &&
	    b >= n->trackAddr.pjb.start_block) {
	    start = n->trackAddr.pjb.start_block;
	    link = -1;
	}
	if (AMapRebuild(newmap,start, b-start+1, link) < 0) {
	    return -1;
	}
	b = link;
    } while (start != n->trackAddr.pjb.start_block);

    return 0;

}

/*  *********************************************************************
    *  RebuildMap(t)							*
    *  									*
    *  Rebuild the TOC's allocation map					*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error code					*
    ********************************************************************* */
static int RebuildMap(PJBTOC *t)
{
    ALLOCMAP *newmap;
    int res;

    newmap = AMapCreate(t->diskinfo.numallocblocks);
    res = PJBTOC_WalkTree(t,RebuildMapFunc,newmap);

    if (res) {
	AMapDestroy(newmap);
	return FALSE;             // failed to scan map
    }

    // Put the new map in place of the old one, and
    // then delete the old node.

    AMapFinishRebuild(newmap);
    AMapDestroy(t->allocmap);
    t->allocmap = newmap;    

    return TRUE;
}

/*  *********************************************************************
    *  PJBTOC_DeleteNode(node)						*
    *  									*
    *  Delete the specified node from the tree.				*
    *  XXX should this also delete the children?			*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC							*
    *  	   node - JNode to delete					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   else error code						*
    ********************************************************************* */
int PJBTOC_DeleteNode(PJBTOC *t,JNode node)
{
    int res;

    // Unlink the node from the tree
    UnlinkNode(node);

    // Make a new allocation map
    res = RebuildMap(t);

    // Free the node
    FreeNode(node);

    // Mark the TOC modified
    t->modified = TRUE;

    return res;
}


/*  *********************************************************************
    *  WalkTree(t,func,n,arg)						*
    *  									*
    *  Inner loop of the tree walker.  					*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   func - function to call for each node in the tree		*
    *  	   n - node we're walking now					*
    *  	   arg - passed to the function for context info		*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep going						*
    *  	   !=0 to stop							*
    ********************************************************************* */

static int WalkTree(PJBTOC *t,int (*func)(PJBTOC *t,JNode n,void *arg),
		    JNode n,
		    void *arg)
{
    int res = 0;
    JNode nn;

    while (n) {
	nn = n->sibling;
	res = (*func)(t,n,arg);
	if (n->child != NULL) {
	    res = WalkTree(t,func,n->child,arg);
	    if (res != 0) break;
	}
	if (res != 0) break;
	n = nn;
    }

    return res;
}


/*  *********************************************************************
    *  PJBTOC_WalkTree(t,func,arg)					*
    *  									*
    *  Walk the entire tree, starting at the root.  This is 		*
    *  the main API call for performing an operation over the		*
    *  entire tree.							*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   func - function to call					*
    *  	   arg - argument to pass to the function			*
    *  	   								*
    *  Return value:							*
    *  	   0 if all nodes were walked					*
    *  	   !=0 if we stopped somewhere					*
    ********************************************************************* */

int PJBTOC_WalkTree(PJBTOC *t,int (*func)(PJBTOC *t,JNode n,void *arg),
		     void *arg)
{
    JNode n = t->toc->root;

    if (n->child) return WalkTree(t,func,n->child,arg);
    return 0;
}



/*  *********************************************************************
    *  PJBTOC_WriteTrack(t,n,ins,readfunc,arg,tag)			*
    *  									*
    *  Write an MP3 file to the PJB.					*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   n - new JNode, must be a TRACK node				*
    *  	   ins - insert point, identifies place in the tree for		*
    *  	         the new node.  (should be a DISC node)			*
    *  	   readfunc - callback for reading MP3 data			*
    *  	   arg - context arg to pass t readfunc				*
    *  	   id3tag - ID3V2 (128byte) string of tag information		*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   else error code						*
    ********************************************************************* */

int PJBTOC_WriteTrack(PJBTOC *t,JNode n,JInsertPt *ins,
		      int (*readfunc)(PJBTOC *t,u8 *buffer,long len,
				      long *numread,void *arg),
		      void *arg,
		      char *id3Tag)
{
    int rc;
    int blkInTrack = 0;
    long bytesInBlock = 0;
    long totBytes = 0;
    AllocMap_Block current_block;
    AllocMap_Block prev_block;
    AllocMap_Block next_block;
    AllocMap_Block start_block;
    AllocMap_Block end_block;
    AllocMap_Block b;
    long start_offset;
    long current_pos;
    long end_bytes;
    long trackOffset;
    u8 *buffer;
    PJB_Checksum chk;

    buffer = (u8 *) malloc(AB_Size);

    if (!buffer) return -1;

    memset(buffer,0,AB_Size);

    current_block = AMapAlloc(t->allocmap);
    if (current_block < 0) {
	return(PJB_ERR_NO_RESOURCES);
    }
    prev_block = -1;
    next_block = -1;
    current_pos = 0;
	
    start_block = current_block;
    start_offset = current_pos;
    end_block = start_block;
    end_bytes = 0;

    for (;;) {
	long bytesread;
	long readsz = AB_PayloadSize - current_pos;

	// EOF is a good read with 0 bytes read.
	rc = (*readfunc)(t,buffer+AB_PayloadStart+current_pos, 
			 readsz, &bytesread,arg);
	if (rc != 0) break;

	// note end of track
	if ((bytesread == 0)) {
	    n->trackAddr.pjb.start_block = start_block;
	    n->trackAddr.pjb.start_block_offset = start_offset;
	    n->trackAddr.pjb.end_block = end_block;
	    n->trackAddr.pjb.end_block_bytes = end_bytes;
	    if (start_block == end_block && end_bytes != 0) {
		n->trackAddr.pjb.end_block_bytes += start_offset;
	// advance to next buffer when start of track is in the same buffer
		prev_block = current_block;
		current_block = next_block;
		next_block = -1;
		current_pos = 0;
	    }
	    break;
	}

	if (next_block < 0) {
	    next_block = AMapAlloc(t->allocmap);
	    if (next_block < 0) {
		rc = PJB_ERR_NO_RESOURCES;
		break;
	    }
	}

	// the following marks the beginning of the next track
	trackOffset = (current_pos != 0) ? current_pos : -1L;
	bytesInBlock += bytesread;
	totBytes += bytesread;

	WriteBE24(buffer+AB_NextLink, next_block);
	WriteBE24(buffer+AB_NextLink1, next_block);
	WriteBE24(buffer+AB_AltNextLink, next_block);
	WriteBE24(buffer+AB_PrevLink, prev_block);
	WriteBE24(buffer+AB_PrevLink1, prev_block);
	buffer[AB_HdrVersion] = AB_CURR_HDR_VERSION;
	buffer[AB_Origin] = id3Tag ? TC_Origin_ID3 : TC_Origin_Unk;
	buffer[AB_Encoding] = n->enc_type;
	WriteBE24(buffer+AB_Bitrate, n->bit_rate);
	WriteBE32(buffer+AB_TrackOffset, trackOffset);
	WriteBE32(buffer+AB_BlockInTrack, blkInTrack);
	WriteBE32(buffer+AB_BytesInBlock, bytesInBlock);

	if (id3Tag) strcpy((char*) buffer+AB_ID3_TAG, id3Tag);

	memset(chk.b, 0, sizeof(chk.b));
	cksum((void *)buffer, AB_HdrCRCSize, chk.b);
	*((PJB_Checksum *) (buffer+AB_HdrCRC)) = chk;

	memset(chk.b, 0, sizeof(chk.b));
	cksum((void *)buffer, AB_CRCSize, chk.b);
	*((PJB_Checksum *) (buffer+AB_CRC)) = chk;

	current_pos += bytesread;
	end_block = current_block;
	end_bytes = bytesInBlock;

	// write the buffer here
	rc = WriteAllocBlock(t,buffer,current_block);
	if (rc != 0)
	    break;

	// advance to next buffer when full
	if (current_pos == AB_PayloadSize) {
	    current_pos = 0;
	    prev_block = current_block;
	    current_block = next_block;
	    next_block = -1;
	    blkInTrack++;
	    bytesInBlock = 0;
	}
    }
    if (rc != 0) {
	if (next_block >= 0)
	    end_block = next_block;
	else
	    end_block = current_block;
	AMapFreeChain(t->allocmap,end_block, start_block,
		     (end_block != start_block || (start_offset == 0)),
		     (start_offset == 0));
	current_block = -1;
    }


    InsertNode(n,ins,FALSE,FALSE);
    t->modified = TRUE;

    // We don't need the allocblock anymore.
    free(buffer);

    // free up the single block that might be
    // allocated but unused between (after) calls to WriteTrack
    if (current_block < 0) {
	return rc;
    }
	
    b = next_block;
    if (b < 0 && current_pos == 0) {
	b = current_block;
    }
    if (b >= 0) {
	AMapFreeChain(t->allocmap,b, b, TRUE, TRUE);
    }
    current_block = -1;

    return(rc);
}




typedef struct FindJNodeInfo {
    char *text;
    JNode parent;
    JNode result;
} FINDJNODEINFO;

/*  *********************************************************************
    *  FindJNodeFunc(t,n,arg)						*
    *  									*
    *  Tree-walker callback for finding JNodes				*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   n - current place in the tree				*
    *  	   arg - FINDJNDOEINFO structure				*
    *  	   								*
    *  Return value:							*
    *  	   JNode we found, or NULL					*
    ********************************************************************* */
static int FindJNodeFunc(PJBTOC *t,JNode n,void *arg)
{
    FINDJNODEINFO *info = (FINDJNODEINFO *) arg;

    if ((n->parent == info->parent) && 
	(strcasecmp(n->text,info->text) == 0)) {
	info->result = n;
	return -1;          /* stop looking */
    }

    return 0;               /* keep looking */
}

/*  *********************************************************************
    *  FindJNode(t,parent,text)						*
    *  									*
    *  find a JNode by text						*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   parent - starting place in the tree				*
    *  	   text - text to find						*
    *  	   								*
    *  Return value:							*
    *  	   JNode we found, or NULL					*
    ********************************************************************* */
JNode PJBTOC_FindJNode(PJBTOC *t,JNode parent,char *text)
{
    FINDJNODEINFO info;
    int res;

    info.parent = parent;
    info.text = text;
    info.result = NULL;

    res = PJBTOC_WalkTree(t,FindJNodeFunc,&info);

    return info.result;
}



/*  *********************************************************************
    *  PJBTOC_CreateSet(t,ins,text)					*
    *  									*
    *  Create a new set in the TOC.  The new set is inserted		*
    *  in the tree. This function assumes the set does not already	*
    *  exist.								*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   ins - insert point - the parent is assumed to be the root	*
    *  	   text - name of new set					*
    *  	   								*
    *  Return value:							*
    *  	   JNode or NULL if we could not create one			*
    ********************************************************************* */

JNode PJBTOC_CreateSet(PJBTOC *t,JInsertPt *ins,char *text)
{
    JNode n;

    n = MakeNode(JNODE_TYPE_SET,text);
    if (!n) return NULL;

    ins->parent = t->toc->root;		/* sets always go in root */

    InsertNode(n,ins,FALSE,FALSE);

    t->modified = TRUE;

    return n;
}

/*  *********************************************************************
    *  PJBTOC_CreateDisc(t,ins,text)					*
    *  									*
    *  Create a new Disc in the TOC.  The new Disc is inserted		*
    *  in the tree. This function assumes the Disc does not already	*
    *  exist.								*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   ins - insert point - the parent should be a Set.		*
    *  	   text - name of new Disc					*
    *  	   								*
    *  Return value:							*
    *  	   JNode or NULL if we could not create one			*
    ********************************************************************* */
JNode PJBTOC_CreateDisc(PJBTOC *t,JInsertPt *ins,char *text)
{
    JNode n;

    /* insert point must be a SET */

    if (ins->parent->type != JNODE_TYPE_SET) return NULL;

    n = MakeNode(JNODE_TYPE_DISK,text);
    if (!n) return NULL;

    InsertNode(n,ins,FALSE,FALSE);
    t->modified = TRUE;

    return n;
}



/*  *********************************************************************
    *  PJBTOC_FindSet(t,text,create)					*
    *  									*
    *  Locate a Set in the TOC.						*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   text - set to find						*
    *  	   create - create the set if it does not exist.		*
    *  	   								*
    *  Return value:							*
    *  	   JNode of set we found (or created)				*
    ********************************************************************* */

JNode PJBTOC_FindSet(PJBTOC *t,char *text,int create)
{
    JNode node;

    node = PJBTOC_FindJNode(t,t->toc->root,text);

    if (create && (node == NULL)) {
	JInsertPt ins;

	ins.parent = t->toc->root;	// insert new one at the end
	ins.pred = JNODE_LAST;
	
	node = PJBTOC_CreateSet(t,&ins,text);
	}

    return node;
}

/*  *********************************************************************
    *  PJBTOC_FindDisc(t,setnode,text,create)			       	*
    *  									*
    *  Locate a Disc in the TOC.					*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *      setnode - JNode of the set to search in			*
    *  	   text - Disc to find						*
    *  	   create - create the Disc if it does not exist.		*
    *  	   								*
    *  Return value:							*
    *  	   JNode of Disc we found (or created)				*
    ********************************************************************* */
JNode PJBTOC_FindDisc(PJBTOC *t,JNode setnode,char *text,int create)
{
    JNode node;


    node = PJBTOC_FindJNode(t,setnode,text);

    if (create && (node == NULL)) {
	JInsertPt ins;

	ins.parent = setnode;
	ins.pred = JNODE_LAST;	// insert new one at the end
	
	node = PJBTOC_CreateDisc(t,&ins,text);
	}

    return node;
}



/*  *********************************************************************
    *  PJBTOC_Reboot(t,maincode)					*
    *  									*
    *  Reboot the PJB to the main program or to flash update mode	*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   maincode - if FALSE, go to flash update mode.  		*
    *  	              if TRUE, go to the main player program		*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error						*
    ********************************************************************* */

int PJBTOC_Reboot(PJBTOC *t,int maincode)
{
    int res;

    res = PJB_Reboot(t->pjb,maincode ? BOOTMETHOD_MAIN : BOOTMETHOD_RECOVERY);

    return res;
}


/*  *********************************************************************
    *  PJBTOC_UpdateFlash(t,filename,execute)				*
    *  									*
    *  Update the program file in the PJB's flash			*
    *  Note: the PJB must be in flash update mode already.		*
    *  									*
    *  Input parameters: 						*
    *  	   t - PJBTOC structure						*
    *  	   filename - name of flash image file to write			*
    *  	   execute - if TRUE, run the program without writing to flash	*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error code					*
    ********************************************************************* */

int PJBTOC_UpdateFlash(PJBTOC *t,char *filename,int execute)
{
    int fh;
    PJBFLASHIMAGE hdr;
    unsigned char buffer[1024];
    PJB_Checksum checksum;
    int mode = FLASHREC_MAINCODE;
    int res;
    int total;
    int blknum;

    if (PJB_Ping(t->pjb,64,0) < 0) return PJB_ERR_TIMEOUT;

    if (!filename) return PJB_ERR_BADPARM;

    fh = open(filename,O_BINARY | O_RDONLY);
    if (fh < 0) {
	return PJB_ERR_BADPARM;
	}

    res = read(fh,&hdr,sizeof(hdr));
    if (res != sizeof(hdr)) {
	goto done;
	}

    if (memcmp(hdr.seal,"PJB\x1A",4) != 0) {
	goto done;
	}

    res = PJB_OpenSoft(t->pjb);

    if (res < 0) {
	return PJB_ERR_TIMEOUT;
	}

    total = 0;
    blknum = 0;
    for (;;) {
	res = read(fh,buffer,sizeof(buffer));
	if (res < 0) {
	    goto done;
	    }
	if (res == 0) break;
	total += res;
	res = PJB_WriteSoft(t->pjb,(u16)blknum,buffer,res);
	if (res < 0) {
	    goto done;
	    }
	blknum++;
	}

    if (execute) mode = FLASHREC_EXECUTE;

    memcpy(checksum.b,hdr.crc,sizeof(checksum.b));
    res = PJB_CommitSoft(t->pjb,&checksum,mode);

    close(fh);

    return 0;

done:
    close(fh);

    return PJB_ERR_BADPARM;
}


/*
 * Things to do:
 *
 * Delete track:  Basically, find the TRACK JNode you're interested in,
 * and call PJBTOC_DeleteNode().
 *
 * Delete Set, Disc: You need to dig down to the tracks and delete
 * the children first, then the disc or set nodes.  There is no
 * code in TOC.C to do this yet.
 *
 * To move a node, unlink it and then re-insert it.
 *
 * In most cases, you should always rebuild the map before flushing
 * the TOC out.
 *
 */
