/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  MP3 File Reader module			File: MP3READER.C	**
   **									**
   **  This library file contains routines to walk through MP3 files 	**
   **  and extract key information such as ID3 tags, bitrate, etc.	**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#ifdef __WIN32__
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#ifdef __linux__
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#define INVALID_HANDLE_VALUE -1
#define __int64 long long
#define strcmpi strcasecmp
#endif

#include "pjbtoc.h"

/*#define _DUMP_*/

/*  *********************************************************************
    *  Constants							*
    ********************************************************************* */

#define MAX_GENRE 80
#define NO_GENRE 255

#define MP3_BUFFER_SIZE 94080
#define MP3_FRAME_HDR_SIZE 4
#define ID3_TAG_SIZE 128


/*  *********************************************************************
    *  Static data							*
    ********************************************************************* */

static char *genres[MAX_GENRE + 1] = {
    "Blues", "Classic Rock", "Country", "Dance", "Disco",
    "Funk", "Grunge", "Hip-Hop", "Jazz", "Metal", "New Age",
    "Oldies", "Other", "Pop", "R&B", "Rap", "Reggae", "Rock",
    "Techno", "Industrial", "Alternative", "Ska", "Death Metal",
    "Pranks", "Soundtrack", "Euro-Techno", "Ambient", "Trip-Hop",
    "Vocal", "Jazz+Funk", "Fusion", "Trance", "Classical",
    "Instrumental", "Acid", "House", "Game", "Sound Clip",
    "Gospel", "Noise", "AlternRock", "Bass", "Soul", "Punk",
    "Space", "Meditative", "Instrumental Pop", "Instrumental Rock",
    "Ethnic", "Gothic", "Darkwave", "Techno-Industrial",
    "Electronic", "Pop-Folk", "Eurodance", "Dream", "Southern Rock",
    "Comedy", "Cult", "Gangsta", "Top 40", "Christian Rap",
    "Pop/Funk", "Jungle", "Native American", "Cabaret",
    "New Wave", "Psychadelic", "Rave", "Showtunes", "Trailer",
    "Lo-Fi", "Tribal", "Acid Punk", "Acid Jazz", "Polka",
    "Retro", "Musical", "Rock & Roll", "Hard Rock", "Unknown",
};

static char *no_genre = "";

/*  *********************************************************************
    *  MP3 Bitrate information						*
    ********************************************************************* */


static unsigned int bitrates[2][4][16] =
{
    { // version == 0 for MPEG 2 lower bit rates
		{0 }, // reserved
		// Layer 1
		{0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0},
		// Layer 2 and 3 are the same
		{0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0},
		{0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0}
    },
    { // version == 1 for MPEG 1 bit rates
		{0 }, // reserved
		// Layer 1
		{0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0},
		// Layer 2
		{0, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, 0},
		// Layer 3
		{0, 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 0}
    }
};

static unsigned int s_freq[2][4] =
{
    {22050, 24000, 16000, 0},
    {44100, 48000, 32000, 0}
};


static void MP3CheckForTags(MP3READER *rdr);

/*  *********************************************************************
    *  CMP3Reader() 							*
    *  									*
    *  Constructor for this class					*
    ********************************************************************* */


MP3READER *MP3Create(void)
{
    MP3READER *rdr;

    rdr = calloc(1,sizeof(MP3READER));

    rdr->membase = NULL;
#ifdef __WIN32__
    rdr->maphandle = INVALID_HANDLE_VALUE;
#endif
    rdr->filehandle = INVALID_HANDLE_VALUE;
    rdr->bitRate = 0;

    rdr->tag_genre = NULL;
    rdr->tag_album = NULL;
    rdr->tag_artist = NULL;
    rdr->tag_title = NULL;

    return rdr;
}

/*  *********************************************************************
    *  ~CMP3Reader()							*
    *  									*
    *  Destructor for this class					*
    ********************************************************************* */

void MP3Destroy(MP3READER *rdr)
{
    if (rdr->open) MP3Close(rdr);
    free(rdr);
}


/*  *********************************************************************
    *  MP3Open(filename,noparse)					*
    *  									*
    *  Opens an MP3 file.  This routine maps the file into		*
    *  memory and attempts to parse it.  It also extracts		*
    *  the tags from both ID3V1 and ID3V2/3 formats.			*
    *  									*
    *  Input parameters: 						*
    *  	   filename - name of mp3 to open				*
    *  	   noparse - skip the file scan (tags will still work)		*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok else error code					*
    ********************************************************************* */

int MP3Open(MP3READER *rdr,char *filename, int noparse)
{
    if (rdr->open) MP3Close(rdr);

#ifdef __WIN32__
    rdr->filehandle = CreateFile(
                filename,
                GENERIC_READ,
                0,
                NULL,
                OPEN_EXISTING,
                FILE_ATTRIBUTE_NORMAL,
                0);

    if (rdr->filehandle == INVALID_HANDLE_VALUE) return MP3READER_ERR_OPEN;

    // get the file size

    rdr->filelength = GetFileSize(rdr->filehandle,NULL);

    // Okay, file is open.  Create a mapped view of it.

    rdr->maphandle = CreateFileMapping(
                rdr->filehandle,
                NULL,
                PAGE_READONLY,
                0,              // entire file specified
                0,              // by putting 0,0 here
                NULL);          // no map name

    if (rdr->maphandle == NULL) {
        CloseHandle(rdr->filehandle);
        return MP3READER_ERR_OPEN;
        }

    // Actually map the file.

    rdr->membase = (u8 *) MapViewOfFile(
                rdr->maphandle,
                FILE_MAP_READ,
                0,
                0,              // start at offset 0,0
                0);             // map entire file

    if (rdr->membase == NULL) {
        CloseHandle(rdr->maphandle);
        CloseHandle(rdr->filehandle);
        return MP3READER_ERR_OPEN;
        }
#endif

#ifdef __linux__

    rdr->filehandle = open(filename,O_RDONLY);
    if (rdr->filehandle < 0) return MP3READER_ERR_OPEN;

    rdr->filelength = lseek(rdr->filehandle,0L,SEEK_END);
    lseek(rdr->filehandle,0L,SEEK_SET);

    rdr->membase = mmap(NULL,rdr->filelength,PROT_READ,MAP_PRIVATE,rdr->filehandle,0);

    if (rdr->membase == (void *) 0xFFFFFFFF) {
	close(rdr->filehandle);
	rdr->filehandle = INVALID_HANDLE_VALUE;
	return MP3READER_ERR_OPEN;	
	}
#endif

    rdr->open = TRUE;

    //
    // Set things up for reading
    //

    rdr->mp3start = rdr->membase;
    rdr->mp3end = rdr->mp3start + rdr->filelength;
    rdr->mp3length = rdr->filelength;
    rdr->mp3ptr = rdr->mp3start;

    //
    // Now scan for tags
    //

    MP3CheckForTags(rdr);

    if (noparse) return 0;

    return MP3ParseMP3(rdr);

}


/*  *********************************************************************
    *  Close()								*
    *  									*
    *  close the MP3 file.						*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

void MP3Close(MP3READER *rdr)
{
    if (!rdr->open) return;

    //
    // First, unmap the shared memory area.
    //

#ifdef __WIN32__
    UnmapViewOfFile((PVOID) rdr->membase);

    // Close the map handle.

    CloseHandle(rdr->maphandle);

    // Close the file handle.

    CloseHandle(rdr->filehandle);

    // Just for grins, set the fields back to their initial values.

    rdr->maphandle = INVALID_HANDLE_VALUE;
    rdr->filehandle = INVALID_HANDLE_VALUE;
#endif

#ifdef __linux__
    munmap(rdr->membase,rdr->filelength);
    close(rdr->filehandle);
    rdr->filehandle = INVALID_HANDLE_VALUE;
#endif

    rdr->membase = NULL;

    if (rdr->tag_genre) free(rdr->tag_genre);
    if (rdr->tag_album) free(rdr->tag_album);
    if (rdr->tag_artist) free(rdr->tag_artist);
    if (rdr->tag_title) free(rdr->tag_title);
    rdr->tag_album = NULL;
    rdr->tag_artist = NULL;
    rdr->tag_title = NULL;
    rdr->tag_genre = NULL;

    rdr->open = FALSE;

}


/*  *********************************************************************
    *  Helper routines for parsing tags					*
    ********************************************************************* */

static unsigned int read28(u8 *ptr)
{
    return (unsigned int) (ptr[3]) +
	((unsigned int) ptr[2] << 7) +
	((unsigned int) ptr[1] << 14) +
	((unsigned int) ptr[0] << 21);
}

static unsigned int read32(u8 *ptr)
{
    return (unsigned int) (ptr[3]) +
	((unsigned int) ptr[2] << 8) +
	((unsigned int) ptr[1] << 16) +
	((unsigned int) ptr[0] << 24);
}

static int ReSync(u8 *binarySourceData,unsigned int sourceSize)
{
    u8	*source, *dest;
    u8	temp;
    unsigned int	destinationSize;

    source = binarySourceData;
    destinationSize = sourceSize;
	
    while ( source < binarySourceData + sourceSize ) {
	temp = *source++;

	if ( temp == 0xFF ) {
	    if	( ( temp = *source++ ) == 0x00 )
		destinationSize--;
	    }
	}

    dest = source = binarySourceData;
		
    while ( ( source < binarySourceData + sourceSize ) && 
	    ( dest < binarySourceData + sourceSize ) ) {
	*dest++ = temp = *source++;

	if ( temp == 0xFF ) {
	    if ( ( temp = *source++ ) != 0x00 )
		*dest++ = temp;
	    }
	}

    return destinationSize;
}


/*  *********************************************************************
    *  makestring(buf,len)						*
    *  									*
    *  Given a buffer and length malloc a string.			*
    *  									*
    *  Input parameters: 						*
    *  	   buf,len - buffer and length					*
    *  	   								*
    *  Return value:							*
    *  	   string, allocated via malloc					*
    ********************************************************************* */

static char *makestring(char *buf,int len)
{
    char *str;

    str = malloc(len+1);
    memcpy(str,buf,len);
    str[len] = '\0';

    while (len > 0) {
	len--;
	if (str[len] != ' ') break;
	str[len] = '\0';
	}	

    if (len == 0) {
	free(str);
	return NULL;
	}

    return str;
}

/*  *********************************************************************
    *  ParseID3v2(taghdr,tagdata,length)				*
    *  									*
    *  Internal routine to parse ID3V2 information. 			*
    *  									*
    *  Input parameters: 						*
    *  	   taghdr - pointer to ID3 tag header				*
    *  	   tagdata - pointer to tag data				*
    *  	   length - length of tag data					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE if valid ID3 tag is present and has useful data		*
    *  	   FALSE otherwise						*
    ********************************************************************* */


static int MP3ParseID3v2(MP3READER *rdr,u8 *taghdr,u8 *tagdata,int length)
{
    int reclen;
    u8 *ptr;
    unsigned int tagtype;
    u8 flags;
    int foundtag = FALSE;

    if (memcmp(taghdr,"ID3",3) != 0) return FALSE;
    if ((taghdr[3] != 2) && (taghdr[3] != 3)) return FALSE;

    flags = taghdr[5];

    if (flags & 0x40) {			/* skip tags w/extended hdrs */
	return FALSE;
	}
    if (flags & 0x80) {			/* Resync if necessary */
	length = ReSync(tagdata,length);
	}

    ptr = tagdata;

    memset(rdr->id3v1tag,' ',128);
    memcpy(rdr->id3v1tag,"TAG",3);
    rdr->id3v1tag[128] = '\0';

    while (length > 0) {
	reclen = (int) read32(&ptr[4]);
	if (reclen > length) break;
	if (!reclen || !(ptr[0]|ptr[1]|ptr[2]|ptr[3])) break;
	tagtype = read32(ptr);

	// Tag format: CC CC CC CC LL LL LL LL FF FF

	if (reclen > 2) {
	    int slen = reclen - 1;
	    int idx;

	    switch (tagtype) {
		case 'TCON':		/* content type */
		    rdr->tag_genre = makestring(ptr+11,slen);
		    rdr->id3v1tag[127] = NO_GENRE;
		    for (idx = 0; idx < MAX_GENRE; idx++) {
			if (strcmpi(genres[idx],rdr->tag_genre) == 0) {
			    rdr->id3v1tag[127] = (char) idx;
			    break;
			    }
			}
		    foundtag = TRUE;
		    break;
		case 'TALB':		/* album name */
		    rdr->tag_album = makestring(ptr+11,slen);
		    if (slen > 30) slen = 30;
		    memcpy(&(rdr->id3v1tag[63]),ptr+11,slen);
		    foundtag = TRUE;
		    break;
		case 'TPE1':		/* performer */
		    rdr->tag_artist = makestring(ptr+11,slen);
		    if (slen > 30) slen = 30;
		    memcpy(&(rdr->id3v1tag[33]),ptr+11,slen);
		    foundtag = TRUE;
		    break;
		case 'TIT2':		/* title of song */
		    rdr->tag_title = makestring(ptr+11,slen);
		    if (slen > 30) slen = 30;
		    memcpy(&(rdr->id3v1tag[3]),ptr+11,slen);
		    foundtag = TRUE;
		    break;
		}
	    }
	ptr += reclen + 10;
	length -= reclen + 10;
	}
 
    return foundtag;

}

/*  *********************************************************************
    *  CheckFortags()							*
    *  									*
    *  Checks various tag formats.					*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static parsetag(MP3READER *rdr)
{
    int tagpresent = FALSE;

    if (memcmp(rdr->id3v1tag,"TAG",3) == 0) {

	if (rdr->tag_genre) free(rdr->tag_genre);
	if (rdr->tag_album) free(rdr->tag_album);
	if (rdr->tag_artist) free(rdr->tag_artist);
	if (rdr->tag_title) free(rdr->tag_title);
	rdr->tag_album = NULL;
	rdr->tag_artist = NULL;
	rdr->tag_title = NULL;
	rdr->tag_genre = NULL;

	rdr->tag_title  = makestring(&(rdr->id3v1tag[3]),30);
	rdr->tag_artist = makestring(&(rdr->id3v1tag[33]),30);
	rdr->tag_album  = makestring(&(rdr->id3v1tag[63]),30);

	if (rdr->id3v1tag[127] >= MAX_GENRE) {
	    rdr->tag_genre = strdup("Unknown");
	    }
	else {
	    rdr->tag_genre = strdup(genres[rdr->id3v1tag[127]]);
	    }

	tagpresent = TRUE;
	}

    return(tagpresent);
}


/*  *********************************************************************
    *  MP3CheckForTags()						*
    *  									*
    *  Check for tags in the open mp3 file				*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void MP3CheckForTags(MP3READER *rdr)
{
    unsigned int length;
    u8 *tagdata;

    //
    // Skip ID3 tag
    //

    rdr->tagtypes = 0;

    if (memcmp(rdr->mp3start,"ID3",3) == 0) {

	length = read28(&rdr->mp3start[6]);

	if (length < 256*1024) {

	    tagdata = (u8 *) malloc(length);
	    if (tagdata) {
		memcpy(tagdata,rdr->mp3start+10,length);
		if (MP3ParseID3v2(rdr,rdr->mp3start,tagdata,length)) {
		    rdr->tagtypes |= MP3TAG_ID3V2;
		    }
		free(tagdata);
		}

	    rdr->mp3start += (length+10);
	    rdr->mp3length -= (length+10);
	    }
	}

    //
    // Look for ID3V1 tag
    //

    if (rdr->mp3length >= 128) {
	if (memcmp(rdr->mp3end-128,"TAG",3) == 0) {
	    rdr->mp3end -= 128;
	    rdr->mp3length -= 128;
	    rdr->tagtypes |= MP3TAG_ID3V1;
	    if (!(rdr->tagtypes & MP3TAG_ID3V2)) {
		memcpy(rdr->id3v1tag,rdr->mp3end,128);
		parsetag(rdr);
		}
	    }
	}

}

/*  *********************************************************************
    *  MP3ParseMP3()							*
    *  									*
    *  Walk through frames in the mp3 file to make sure			*
    *  we can play it in the Jukebox					*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   else error code						*
    ********************************************************************* */

static int layers[4] = {0,3,2,1};    /* header layer to layer name */
static int versions[4] = {0,0,0,1};  /* header version to array index */
static int lengths[4] = {144,72,72,144}; /* header version to bitrate numerator */


typedef struct MpegHeader {
    unsigned char sync1 : 8;

    unsigned char crc : 1;
    unsigned char layer : 2;
    unsigned char ver : 2;
    unsigned char sync2 : 3;

    unsigned char extension : 1;
    unsigned char padding : 1;
    unsigned char sampfreq : 2;
    unsigned char bitrate : 4;

    unsigned char emphasis : 2;
    unsigned char original : 1;
    unsigned char copyright : 1;
    unsigned char modeext : 2;
    unsigned char mode : 2;
} MPEGHEADER;

#define MPEG_VER_25 0
#define MPEG_VER_RSV 1
#define MPEG_VER_20 2
#define MPEG_VER_10 3

#define MPEG_LYR_RSV 0
#define MPEG_LYR_3 1
#define MPEG_LYR_2 2
#define MPEG_LYR_1 3

#define MPEG_CRC 0
#define MPEG_NOCRC 1

#define ISSYNC(m) ((m->sync1 == 0xFF) & (m->sync2 == 7))

#ifdef _DUMP_

static char *mpegversions[4] = {"2.5","VRS","2.0","1.0"};
static char *mpeglayers[4] = {"LRS","III","II ","I  "};
static char *mpegerrors[2] = {"CRC  ","NOCRC"};
static char *mpegpads[2] = {"NOPAD","PAD  "};
static char *mpegmodes[4] = {"STR ","JS  ","DCH ","MONO"};
static char *mpegmodeexts[4] = {"LRLR","LRI ","MSLR", "MSL "};
static char *mpegcopyright[2] = {"NCPY","CRGT"};
static char *mpegorig[2] = {"NORG","ORIG"};


static int DumpHeader(MPEGHEADER *h)
{
    if (ISSYNC(h)) printf("SYNC ");
    else printf("!SYN ");


    printf("%s %s %s %3d %5d %s %d %s %s %s %s %d",
	   mpegversions[h->ver],
	   mpeglayers[h->layer],
	   mpegerrors[h->crc],
	   bitrates[versions[h->ver]][layers[h->layer]][h->bitrate],
	   s_freq[versions[h->ver]][h->sampfreq],
	   mpegpads[h->padding],
	   h->extension,
	   mpegmodes[h->mode],
	   mpegmodeexts[h->modeext],
	   mpegcopyright[h->copyright],
	   mpegorig[h->original],
	   h->emphasis);
    return 0;
	   
}
#define DBGDUMP(x) x
#else
#define DBGDUMP(X)
#endif


int MP3ParseMP3(MP3READER *rdr)
{
    u8 *buffer;
    MPEGHEADER *h;
    int framesize;
    int bitrate;
    int sampling_freq;
    int framenum = 0;
    double playingtime = 0;

    buffer = rdr->mp3start;
    
    while (buffer < rdr->mp3start + rdr->mp3length) {
	h = (MPEGHEADER *) buffer;

	/* if there's anything we don't like about the header,
	   skip it and move on to try to find another header. */

	if (!ISSYNC(h) ||		/* if no sync */ 
	    (h->ver == MPEG_VER_RSV) || /* header version is "reserved" */
	    (h->layer != MPEG_LYR_3)) {	/* not layer 3 */
	    DBGDUMP(printf("#"));


	    /* advance our pointer and continue */

	    buffer++;
	    continue;
	    }


	/* If it is otherwise valid but is a layer2, bomb out now */

	if ((framenum == 0) && (h->layer != MPEG_LYR_3)) {
	    return MP3READER_ERR_FORMAT;
	    }

	bitrate = bitrates[versions[h->ver]][layers[h->layer]][h->bitrate];
	sampling_freq = s_freq[versions[h->ver]][h->sampfreq];

	/* Something is wrong if the bitrate or sampling frequency are wrong */

	if ((bitrate == 0) || (sampling_freq == 0)) {
	    DBGDUMP(printf("$"));
	    buffer++;
	    continue;
	    }


	DBGDUMP((printf("[%08X] ",buffer-rdr->mp3start),DumpHeader(h)));

	/* If the bitrate has changed at least once, it's a VBR file */

        if (rdr->bitRate != 0) {
	    if (rdr->bitRate != bitrate) rdr->vbr = TRUE;
	    }

	framenum++;
	rdr->bitRate = bitrate;

	/* calculate frame length and cumulative playing time */
		
	framesize = (lengths[h->ver]*bitrate*1000)/sampling_freq;

	if (h->padding) framesize++;
	if (h->crc == 0) framesize += 2;

	buffer += framesize;

	DBGDUMP(printf(" FS=%d\n",framesize));

	playingtime += (double)(framesize * 8) / (double)(bitrate * 1000);

	}

    /* If it was a VBR file, change the bitrate to the
       average bitrate */

    if (rdr->vbr) {
	rdr->bitRate = (rdr->mp3length * 8 / (int) playingtime) / 1000;
	}

    /* if the frame count is very small, meaning a very
       short playing time, the file is probably corrupted
       and we were parsing garbage. */

    if (framenum < 20) return MP3READER_ERR_MALFORMED;

    rdr->playtime75 = (int) (playingtime * 75.0);

    return 0;
    
}

#ifdef __notused__
int OldMP3ParseMP3(MP3READER *rdr)
{
    u8 *buffer;
    int iBitrate;
    int version;
    int layer = 0;
    int bitrate = 0;
    int iSamplingRate;
    int bitrateTotal = 0;
    int slotLength = 1;
    int frames = 0;
    int bytesPerFrame = 0;
    int rc = 0;
    int mpeg25 = FALSE;
    int padding = FALSE;
    int possHeaderFound = FALSE;
    int headerFound = FALSE;
    u8 synchbyte;
    __int64 bytesToEnd = 0;
    double playingTime = 0;
    int sampling_freq = 0;

    rdr->vbr = FALSE;

    bytesToEnd = rdr->mp3length;

    buffer = rdr->mp3start;
 
    /* Scan through the block looking for a header */
    while (buffer < rdr->mp3end-MP3_FRAME_HDR_SIZE) {
	if ((buffer[0] == 0xFF) && ((buffer[1] & 0xE0) == 0xE0)) {
	    layer = layers[(buffer[1] >> 1) & 3];
	    /* two headers in a row that ar non-layer3 means reject */
	    if ((layer != 3)  && possHeaderFound) return MP3READER_ERR_LAYER;
	    if (layer != 0) possHeaderFound = TRUE;
	    if (layer == 3) {
		headerFound = TRUE;
		break;
		}
	    }
	buffer++;
	}

printf("Layer = %d\n",layer);

    if (!headerFound) {
	if (possHeaderFound) {
	    return MP3READER_ERR_LAYER;
	    }
	if (buffer > rdr->mp3end-MP3_FRAME_HDR_SIZE) {
	    return MP3READER_ERR_FORMAT;
	    }
	}
    version = (buffer[1] & 8) >> 3;

    if ((version == 0) && ((buffer[1] & 24) == 0)) {
	mpeg25 = TRUE;
	synchbyte = 0xE0;
	}
    else synchbyte = 0xF0;	

printf("Header found at %d\n",buffer-rdr->mp3start);

    iSamplingRate = (buffer[2] & 0x0C) >> 2;
    sampling_freq = s_freq[version][iSamplingRate];
    iBitrate = (buffer[2] & 0xF0) >> 4;
    bitrate = bitrates[version][layer][iBitrate];
    padding = (buffer[2] & 0x02)>>1;

    // 1152 slots/packet, 8 bits/slot = 144

    bytesPerFrame = 144*bitrate*1000/sampling_freq;

    if (bitrate != 0) {
	frames++;
	if (padding) {
	    // playing time is in 75ths, so we multiply frame size by 75
	    // 125 is 1000 / 8 -- bitrate doesn't include the 3 trailing
	    // zeroes in the table and there are 8 bits per byte.
	    playingTime = (75 * bytesPerFrame + slotLength);
	    playingTime /= (bitrate * 125);
	    } 
	else {
	    playingTime = (75 * bytesPerFrame);
	    playingTime /= (bitrate * 125);
	    }
	}

    bitrateTotal = bitrate;
    buffer += bytesPerFrame;
    if (padding) buffer += slotLength;

    for(;;) {
	// 56K, 48K, and 40K encode rates have either claimed to have 
	// padding when they didn't, claimed not to have padding when
	// they did (two bytes!?!), or used one byte less than 
	// bytesPerFrame. How many ways of mucking up the encoding 
	// will we find?
	if (buffer[0] != 0xFF) {
	    if (buffer[1] == 0xFF) buffer++;
	    else if ((buffer-1)[0] == 0xFF) buffer--;
	    if ((buffer[0] != 0xFF) && (buffer[2] == 0xFF) 
		&& (bytesToEnd > 0) && ((buffer[3]&synchbyte) == synchbyte)
		&& (version == ((buffer[3]&8)>>3))
		&& (layer == (-(buffer[3]>>1)&3))) {
		return MP3READER_ERR_MALFORMED;
		}
	    }
	if ((buffer[0] == 0xFF) && ((buffer[1] & synchbyte) == synchbyte)) {
	    if ((version == ((buffer[1] & 8) >> 3)) && 
		(layer == (-(buffer[1] >> 1) & 3))) {
		if (iSamplingRate != (buffer[2] & 0x0C) >> 2) {
		    // miscoded frame, treat it as though it was the same
		    // as the previous frame, with padding
		    double addedTime;

		    padding = TRUE;
		    bitrateTotal += bitrate;
		    addedTime = 75*bytesPerFrame + slotLength;
		    addedTime /= (bitrate*125);
		    playingTime += addedTime;
		    frames++;
		    rdr->bitRate = bitrateTotal/frames;
		    } 
		else {
		    if (!mpeg25 && (iBitrate != ((buffer[2] & 0xF0)>>4))) {
			rdr->vbr = TRUE;
			}
		    if (!mpeg25 || (mpeg25 && ((buffer[1] & 24) == 0))) {
			iBitrate = (buffer[2] & 0xF0) >> 4;
			bitrate = bitrates[version][layer][iBitrate];

			bitrateTotal += bitrate;
			padding = (buffer[2] & 0x02)>>1;
			if (bitrate != 0) {
			    frames++;
			    // 1152 slots/packet, 8 bits/slot = 144
			    bytesPerFrame = 144*bitrate*1000/sampling_freq;
			    if (padding) {
				double addedTime = 75*bytesPerFrame + slotLength;
				addedTime /= (bitrate*125);
				playingTime += addedTime;
				} 
			    else {
				double addedTime = 75*bytesPerFrame;
				addedTime /= (bitrate*125);
				playingTime += addedTime;
				}
			    }
			rdr->bitRate = bitrateTotal/frames;
			}
		    }
		}
	    }

	printf("%d ",bytesPerFrame);
	buffer += bytesPerFrame;

	if (padding) buffer += slotLength;

	if (buffer > rdr->mp3end-MP3_FRAME_HDR_SIZE) {
	    break;
	    }

	}

    rdr->playtime75 = (unsigned long) (playingTime);
printf("%d frames\n",frames);

    return (0);
}
#endif


/*  *********************************************************************
    *  Read (buf,length)						*
    *  									*
    *  Read data from the mp3 file					*
    *  									*
    *  Input parameters: 						*
    *  	   buf - dest buffer						*
    *  	   length - size of dest buffer					*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes read, or 0 if no bytes available		*
    ********************************************************************* */

int MP3Read(MP3READER *rdr,unsigned char *buf,int length)
{
    int copylen;

    copylen = rdr->mp3length;
    if (copylen > length) copylen = length;

    memcpy(buf,rdr->mp3ptr,copylen);

    rdr->mp3ptr += copylen;
    rdr->mp3length -= copylen;

    return copylen;	   
}





#ifdef _TESTPROG_
void main(int argc,char *argv[])
{
    int res;
    int idx;
    MP3READER *reader;

    reader = MP3Create();

    for (idx = 1; idx < argc; idx++) {
	printf("--------------------------------------------\n");
	printf("** Opening: %s\n",argv[idx]);

	res = MP3Open(reader,argv[idx],FALSE);

	if (res != 0) {
	    printf("Could not open %s: %d\n",argv[idx],res);
	    }

	printf("BitRate:      %d kbps\n",reader->bitRate);
	printf("Playing time: %d sec\n",reader->playtime75/75);
	printf("VBR:          %s\n",reader->vbr ? "YES" : "NO");
	printf("MP3 Length:   %d\n",reader->mp3length);

	if (reader->tagtypes) {
	    printf("\n");
	    printf("Title:    %s\n",reader->tag_title);
	    printf("Artist:   %s\n",reader->tag_artist);
	    printf("Album:    %s\n",reader->tag_album);
	    printf("Genre:    %s\n",reader->tag_genre);
	    printf("Tag:      %s\n",reader->tagtypes & MP3TAG_ID3V2 ? "ID3V2/3" : "ID3V1");
	    printf("ID3V1 tag %s\n",reader->id3v1tag);
	    }

	MP3Close(reader);
	}
    MP3Destroy(reader);
}
#endif

