/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  "C" implementation of Bitmap class  	File: BITMAP.C		**
   **									**
   **  This module implements operations on an array of bits		**
   **  and is used to represent free space on the PJB.			**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/


#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <assert.h>
#include "bitmap.h"

static int TocBitmapExpand(TOCBITMAP *bm,int iElem);
static void TocBitmapInit(TOCBITMAP *,int nbits);

#ifndef VERIFY
#define VERIFY(x) assert(x)
#endif
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif


TOCBITMAP *TocBitmapCreate(int nblocks)
{
    TOCBITMAP *bm;

    bm = calloc(1,sizeof(TOCBITMAP));
    if (!bm) return NULL;

    bm->nMaxBits = 0;
    bm->nElems = 0;
    bm->map = NULL;
    bm->allocHint = 0;

    TocBitmapInit(bm,nblocks);

    return bm;
}


void TocBitmapDestroy(TOCBITMAP *bm)
{
    if (bm->map) free(bm->map);
    free(bm);
}

static void TocBitmapInit(TOCBITMAP *bm,int maxbits)
{
    if (bm->map != NULL) free (bm->map);
    bm->nMaxBits = maxbits;
    bm->nElems = 0;
    bm->map = NULL;
}

void TocBitmapReset(TOCBITMAP *bm)
{
    if (bm->map) {
	memset(bm->map, 0, bm->nElems*sizeof(TocBitmapElem));
	}
    bm->allocHint = 0;
}

int TocBitmapAlloc(TOCBITMAP *bm)
{
    int i = bm->allocHint / BITS_PER_ELEM;
    TocBitmapElem x;
    TocBitmapElem y;
    int j;

    for (;;) {
	if (i >= bm->nElems) {
	    if (! TocBitmapExpand(bm,i))
		return (-1);
	    }
	if (bm->map[i] != ~(0))
	    break;
	i++;
	}

    x = 1;
    y = bm->map[i];
    j = i * BITS_PER_ELEM;

    while ((x & y) != 0) {
	x <<= 1;
	j++;
	}

    if (j >= bm->nMaxBits && bm->nMaxBits > 0)
	return (-1);
    bm->map[i] |= x;
    bm->allocHint = j+1;
    return (j);
}

void TocBitmapSet(TOCBITMAP *bm,int n)
{
    int i = n / BITS_PER_ELEM;
    if (i >= bm->nElems)
	VERIFY(TocBitmapExpand(bm,i));
    bm->map[i] |= (1 << (n%BITS_PER_ELEM));
}

void TocBitmapClear(TOCBITMAP *bm,int n)
{
    int i = n / BITS_PER_ELEM;
    if (i < bm->nElems) {
	bm->map[i] &= ~(1 << (n%BITS_PER_ELEM));
	if (n < bm->allocHint)
	    bm->allocHint = n;
	}
}

int TocBitmapIsSet(TOCBITMAP *bm,int n)
{
    int i = n / BITS_PER_ELEM;
    int m;

    if (i >= bm->nElems)
	return FALSE;
    m = 1 << (n%BITS_PER_ELEM);
    return ((bm->map[i] & m) != 0);
}

#define INIT_ELEMS			4
#define EXPANSION_FACTOR	4

static int TocBitmapExpand(TOCBITMAP *bm,int iElem)
{
    int new_n;

    if (bm->nMaxBits > 0 && iElem > (bm->nMaxBits-1)/((int) BITS_PER_ELEM))
	return (FALSE);

    new_n = (bm->nElems == 0) ? INIT_ELEMS : bm->nElems;
    while (iElem >= new_n)
	new_n *= EXPANSION_FACTOR;
    bm->map = (TocBitmapElem *) realloc(bm->map, new_n * sizeof(TocBitmapElem));
    memset(bm->map + bm->nElems, 0, (new_n-bm->nElems)*sizeof(TocBitmapElem));
    bm->nElems = new_n;
    return (TRUE);
}
