/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Allocation Map operations		File: ALLOCMAP.CPP	**
   **									**
   **  This module contains code to calculate and manipulate the	**
   **  allocated space bitmap on the PJB's file system.			**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include "ptrarray.h"
#include "toc.h"
#include "allocmap.h"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef VERIFY
#define VERIFY(x) assert(x)
#endif



static ChainSegment *AMapNewSegment(ALLOCMAP *a,
				    AllocMap_Block b, int nblks, 
				    AllocMap_Block link);
static ChainSegment *AMapFindSegment(ALLOCMAP *a,AllocMap_Block b, 
				     int *ich);

ALLOCMAP *AMapCreate(long nblocks)
{
    ALLOCMAP *a;

    a = calloc(1,sizeof(ALLOCMAP));
    if (!a) return NULL;

    a->lastSeg = NULL; 
    a->nTotalBlocks = nblocks;
    a->nFreeBlocks = nblocks;
    a->bitmap = TocBitmapCreate(nblocks);
    a->chain = PtrArrayCreate();

    return a;
}

void AMapDestroy(ALLOCMAP *a)
{
    int i;

    for (i = 0; i < PtrArrayGetSize(a->chain); i++) {
	free(PtrArrayGetAt(a->chain,i));
	}

    TocBitmapDestroy(a->bitmap);
    PtrArrayDestroy(a->chain);
}

void AMapReset(ALLOCMAP *a)
{
    int i;

    for (i = 0; i < PtrArrayGetSize(a->chain); i++) {
	free(PtrArrayGetAt(a->chain,i));
	}

    PtrArrayRemoveAll(a->chain);
    TocBitmapReset(a->bitmap);

    a->nFreeBlocks = a->nTotalBlocks;
    a->lastSeg = NULL;
}

int AMapRebuild (ALLOCMAP *a,AllocMap_Block blk, int n, AllocMap_Block prev)
{
    // there may be an existing entry spanning this range.
    // if so, it should cover the complete range and the prev
    // pointer should be consistent
    // however, we must allow for an overlap including the first
    // or last block ... where tracks share a block
    int i = 0;

    while (i < PtrArrayGetSize(a->chain)) {
	ChainSegment *ch = (ChainSegment *) PtrArrayGetAt(a->chain,i);
	if (ch->start + ch->nblocks > blk) {
	    if (ch->start < blk + n) {
		// test block has at least partial overlap
		if (ch->start <= blk && blk + n <= ch->start + ch->nblocks) {
		    // complete overlap here (segment already added?)
		    if (prev >= 0 && (blk != ch->start ||
				      (ch->link >= 0 && prev != ch->link))) {
			return (-1);
			}	
		    return (0);
		    } 
		// check for partial overlap
		if (blk == ch->start + ch->nblocks - 1) {
		    // 1-overlap at top of ch
		    // check that our link points to top of existing block
		    if (prev >= 0 && prev != blk - 1) 
			return (-1);
		    // check that there is not 1-overlap with next ch
		    if (i+1 < PtrArrayGetSize(a->chain)) {
			ChainSegment *ch1 = (ChainSegment *) PtrArrayGetAt(a->chain,i+1);
			if (ch1->start == blk + n - 1) {
			    if (ch1->link >= 0 && ch1->link != ch1->start - 1)
				return (-1);
			    ch->nblocks += ch1->nblocks;
			    PtrArrayRemoveAt(a->chain,i+1);
			    free(ch1);
			    n--;
			    }
			}
		    ch->nblocks += n - 1;
		    a->nFreeBlocks -= n - 1;
		    for (i=1; i < n; i++) {
			TocBitmapSet(a->bitmap,blk+i);
			}
		    return (0);
		    }
		if (blk + n - 1 == ch->start) {
		    // 1-overlap at bottom of ch
		    // check that link points to top of our block
		    if (ch->link >= 0 && ch->link != ch->start - 1)
			return (-1);
		    ch->start = blk;
		    ch->nblocks += n - 1;
		    ch->link = prev;
		    a->nFreeBlocks -= n - 1;
		    for (i=0; i < n-1; i++) {
			TocBitmapSet(a->bitmap,blk+i);
			}
		    return (0);
		    }
		// overlap of more than one block is illegal
		return (-1);
		}
	    break;
	    }
	i++;
	}	
    (void) AMapNewSegment(a,blk, n, prev);
    a->nFreeBlocks -= n;
    for (i=0; i < n; i++) {
	TocBitmapSet(a->bitmap,blk+i);
	}
    return (0);
}

void AMapFinishRebuild(ALLOCMAP *a)
{
    // coalesce touching elements in the map
    int i = 0;

    while (i < PtrArrayGetSize(a->chain)-1) {
	ChainSegment *s1 = (ChainSegment *) PtrArrayGetAt(a->chain,i);
	ChainSegment *s2 = (ChainSegment *) PtrArrayGetAt(a->chain,i+1);
	if (s1->start + s1->nblocks == s2->start &&
	    (s2->link < 0 || s2->link == s2->start - 1)) {
	    s1->nblocks += s2->nblocks;
	    PtrArrayRemoveAt(a->chain,i+1);
	    free(s2);
	    } 
	else {
	    i++;
	    }
	}
    a->lastSeg = NULL;
}

int AMapInput(ALLOCMAP *a,TOC_IOBuf *bf)
{
    int	done = FALSE;
    int	first = TRUE;
    AllocMap_Block	fblk, nblk, backlink;
    int i;
	
    AMapReset(a);

    while (!done) {

	char		tc;
	char		ss[MAX_TOC_STR + 1];

	int x = InputTocLine(bf, &tc, ss);

	if (x < 0) {
	    return TOC_ERROR_FORMAT;
	    }

	if (x == 0) {
	    if (first) return (0);
	    return TOC_ERROR_FORMAT;
	    }

		
	switch (tc) {
	    case TC_AllocMap:
		if (sscanf(ss, TC_AM_FORMAT, &fblk, &nblk, &backlink) != 3) {
		    return TOC_ERROR_ALLOCMAP;
		    } 
		else {
		    (void) AMapNewSegment(a,fblk, nblk, backlink);
		    for (i=0; (i < nblk); i++) {
			TocBitmapSet(a->bitmap,fblk + i);
			a->nFreeBlocks--;
			}
		    }
		break;

	    case TC_EndSegment:
		done = TRUE;
		break;

	    default:
		break;
	    }
	}

    a->lastSeg = NULL;

    return 0;
}
	
void AMapOutput(ALLOCMAP *a,TOC_IOBuf *bf)
{
    int i;
    char buffer[512];

    for (i=0; i<PtrArrayGetSize(a->chain); i++) {
	ChainSegment *ch = (ChainSegment *) PtrArrayGetAt(a->chain,i);
	sprintf(buffer,TC_AM_FORMAT,ch->start, ch->nblocks, ch->link);
	OutputTocLine(bf, TC_AllocMap, buffer);
	}

    OutputTocLine(bf, TC_EndSegment, "");
}

AllocMap_Block AMapAlloc(ALLOCMAP *a)
{
    int i;
    AllocMap_Block link;

    i = TocBitmapAlloc(a->bitmap);
    if (i < 0) return -1;
    a->nFreeBlocks--;

    link = -1;
    if (a->lastSeg == NULL && i != 0) {
	//VERIFY(firstInExtent);
	a->lastSeg = AMapFindSegment(a,i - 1,NULL);
	VERIFY(a->lastSeg != NULL);
	}
    if (a->lastSeg != NULL) {
	link = a->lastSeg->start + a->lastSeg->nblocks - 1;
	if (i == link + 1) {
	    a->lastSeg->nblocks++;
	    return (i);
	    }
	}
    a->lastSeg = AMapNewSegment(a,i, 1, link);
    return (i);
}

int AMapFollowChain(	ALLOCMAP *a,
			AllocMap_Block blk,
			AllocMap_Block* start,
			AllocMap_Block* nextSegTail)
{
    ChainSegment *ch = AMapFindSegment(a,blk,NULL);
    if (ch == NULL || blk >= ch->start + ch->nblocks)
	return (-1);
    *start = ch->start;
    *nextSegTail = ch->link;
    return (0);
}

int AMapFreeChain(ALLOCMAP *a,
		AllocMap_Block end, AllocMap_Block start,
		int free_end, int free_start)
{
    int ich;
    ChainSegment *ch = NULL;
    AllocMap_Block b = end;
    AllocMap_Block link;

    for (;;) {
	if (ch == NULL) {
	    ch = AMapFindSegment(a,b, &ich);
	    if (ch == NULL) return (-1);
	    link = ch->link;
	    }

	if ((free_end || b != end) && (free_start || b != start)) {

	    if (b != ch->start + ch->nblocks - 1) {
		// not end of segment, we must split this one
		ChainSegment *nch = calloc(1,sizeof(ChainSegment));
		nch->start = b+1;
		nch->nblocks = ch->nblocks - (b - ch->start) - 1;
		nch->link = -1;
		PtrArrayInsertAt(a->chain,ich+1,nch);
		ch->nblocks -= nch->nblocks;
		}

	    TocBitmapClear(a->bitmap,b);
	    a->nFreeBlocks++;
	    if (--ch->nblocks == 0) {
		if (ch == a->lastSeg) {
		    a->lastSeg = NULL;
		    }
		PtrArrayRemoveAt(a->chain,ich);
		free(ch);
		ch = NULL;
		}
	    }

	if (b == start) break;

	if (ch == NULL) b = link;
	else b--;

	}

    return (0);
}

ChainSegment *AMapNewSegment(ALLOCMAP *a,
	AllocMap_Block b, int nblks, AllocMap_Block link)
{
    int i;

    ChainSegment *ch;

    for (i=0; i<PtrArrayGetSize(a->chain);i++) {
	ch = (ChainSegment *) PtrArrayGetAt(a->chain,i);
	if (b < ch->start) {
	    // check here that there is no overlap with next block
	    VERIFY(b + nblks <= ch->start);
	    break;
	    }
	}
    // check here that there is no overlap with previous block
    if (i != 0) {
	ChainSegment *xch = (ChainSegment *) PtrArrayGetAt(a->chain,i-1);
	VERIFY(b >= (xch->start + xch->nblocks));
	}
    // finally ... add the new segment
    ch = calloc(1,sizeof(ChainSegment));
    ch->start = b;
    ch->nblocks = nblks;
    ch->link = link;
    PtrArrayInsertAt(a->chain,i,ch);
    return (ch);
}

ChainSegment *AMapFindSegment(ALLOCMAP *a,AllocMap_Block b, int *ich)
{
    int i;

    ChainSegment *ch;

    for (i=0; i<PtrArrayGetSize(a->chain); i++) {
	ch = (ChainSegment *) PtrArrayGetAt(a->chain,i);
	if (b >= ch->start && b < ch->start + ch->nblocks) {
	    if (ich != NULL) *ich = i;
	    return ch;
	    }
	}
    return (NULL);
}
