/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB API test program			File: PJBTEST.C		**
   **									**
   **  this program demonstrates how to call the PJB API.		**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#ifdef __WIN32__
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>

#ifdef __linux__
#include <unistd.h>
#include <sys/time.h>
#endif
#ifdef __WIN32__
#include <time.h>
#endif

#include "swparse.h"
#include "parsetab.h"

#include "pjbapi.h"
#include "cksum.h"

#ifdef __linux__
#define O_BINARY 0               /* Linux doesn't need this distinction */
#endif


/*  *********************************************************************
    *  Constants                                                        *
    ********************************************************************* */

#define TRUE 1
#define FALSE 0

#define CMDERR 0xFFFFFFFF

extern void setswitch(int,char *);

extern int scan_switch(int argc,char *argv[]);
extern int switch_set(char *);

/*  *********************************************************************
    *  Types                                                            *
    ********************************************************************* */

typedef struct ErrMap {
    int err;
    char *str;
    } ERRMAP;


ERRMAP pcli_errors[] = {
    {PJB_ERR_NODISK,	"No disk at PJB"},
    {PJB_ERR_DISK,      "Disk I/O error"},
    {PJB_ERR_ALLOCBLOCK,"alloc block number out of range"},
    {PJB_ERR_OFFSET,    "offset out of range"},
    {PJB_ERR_CHECKSUM,  "checksum mismatch"},
    {PJB_ERR_NOTOC,     "no table of contents"},
    {PJB_ERR_COMM,      "communications error"},
    {PJB_ERR_TIMEOUT,	 "timeout"},
    {PJB_ERR_NO_RESOURCES,"out of resources"},
    {PJB_ERR_NODEVICE,   "no ethernet device"},
    {PJB_ERR_DATACORRUPT,"network data is corrupt"},
    {PJB_ERR_PROTOCOLERR,"protocol error"},
    {PJB_ERR_NOBIND,	   "could not bind to adapter"},
    {PJB_ERR_BADPARM, "invalid parameter"},
    {PJB_ERR_PJBINUSE,"PJB is in use by another jukebox manager"},
    {0,NULL}};




/*  *********************************************************************
    *  Prototypes							*
    ********************************************************************* */

/*  *********************************************************************
    *  Command Table							*
    ********************************************************************* */

extern PARSESET cmdtab;

/*  *********************************************************************
    *  Globals								*
    ********************************************************************* */

int running = TRUE;

PJB_HANDLE pjb = NULL;

FS_ALLOCBLOCK *ablk = NULL;


#ifdef __linux__
int kbhit()
{
    fd_set rfds;
    struct timeval tv;

    FD_ZERO(&rfds);
    FD_SET(fileno(stdin),&rfds);
    tv.tv_sec = 0; tv.tv_usec = 0;
    select(fileno(stdin)+1,&rfds,NULL,NULL,&tv);

    return FD_ISSET(fileno(stdin),&rfds);
}
#endif

void pjbsleep(int seconds)
{
#ifdef __linux__
    sleep(seconds);
#endif
#ifdef __WIN32__
    Sleep(seconds*1000);
#endif
}

/*  *********************************************************************
    *  error_text(tab,error)                                            *
    *                                                                   *
    *  Map error number to text                                         *
    *                                                                   *
    *  Input Parameters:                                                *
    *      error - error #                                              *
    *                                                                   *
    *  Return Value:                                                    *
    *      text                                                         *
    ********************************************************************* */
char *error_text(ERRMAP *emap,int error)
{
    static char errbuf[100];

    while (emap->str) {
        if (emap->err == error) break;
        emap++;
        }

    if (!emap->str) {
        sprintf(errbuf,"Unknown error %u",error);
        return errbuf;
        }

    return emap->str;
 
}


char *addr_text(u8 *addr)
{
    static char text[50];
    sprintf(text,"%02X-%02X-%02X-%02X-%02X-%02X",
	    addr[0],addr[1],addr[2],addr[3],addr[4],addr[5]);
    return text;
}




/*  *********************************************************************
    *  writelog(a,b,c,d,e,f,g)                                          *
    *                                                                   *
    *  Writes log file                                                  *
    *                                                                   *
    *  Input Parameters:                                                *
    *      a..g - args                                                  *
    *                                                                   *
    *  Return Value:                                                    *
    *      nothing.                                                     *
    ********************************************************************* */
void writelog(char *a,...)
{
        struct tm *tms;
        long l;
	va_list mark;

        time(&l);
        tms = localtime(&l);

        printf("%02d/%02d/%04d %02d:%02d:%02d  ",
                tms->tm_mon+1,tms->tm_mday,tms->tm_year+1900,
                tms->tm_hour,tms->tm_min,tms->tm_sec);

	va_start(mark,a);
        vprintf(a,mark);
	va_end(mark);
}


/*  *********************************************************************
    *  debugcallback(flg,arg,text)					*
    *  									*
    *  Display debug callback data					*
    *  									*
    *  Input parameters: 						*
    *  	   flg - 0 for diagnostic data					*
    *      arg - what we set in the PJB_SetDebugCallback call		*
    *  	   str - string to display					*
    *  	   								*
    *  Return value:							*
    *  	   0								*
    ********************************************************************* */
int debugcallback(int flg,void *arg,char *text)
{
    if (flg == 0) {
	printf("# %s\n",text);
	}

    return 0;
}


/*  *********************************************************************
    *  dumpdata(str,buffer,buflen)					*
    *  									*
    *  dump data to the screen in hex format.				*
    *  									*
    *  Input Parameters: 						*
    *      str - file for output					*
    *      buffer - pointer to buffer					*
    *      buflen - length of buffer					*
    *      								*
    *  Return Value:							*
    *      nothing.							*
    ********************************************************************* */

void dumpdata(str,buffer,buflen)
FILE *str;
unsigned char *buffer;
int buflen;
{
	int i,j;

	for (i=0; i<buflen; i+=16) {
		fprintf(str,"%04X: ",i);
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str,"   ");
			else fprintf(str,"%02X ",buffer[i+j]);
			}
		fprintf(str," | ");
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str," ");
			else fprintf(str,"%c",isprint(buffer[i+j]) ? buffer[i+j] : ' ');
			}
		fprintf(str,"\n");
		}
}

/*  *********************************************************************
    *  docommandfile(x)                                                 *
    *                                                                   *
    *  Execute commands from a command file                             *
    *                                                                   *
    *  Input Parameters:                                                *
    *      x - file name                                                *
    *                                                                   *
    *  Return Value:                                                    *
    *      FALSE if error, TRUE if ok                                   *
    ********************************************************************* */
int docommandfile(x)
char *x;
{
        FILE *str;
        char *errormsg;
        int res;
        char line[200];

        str = fopen(x,"rt");
        if (!str) {
                writelog("Could not open command file %s\n",x);
                return 0;
                }
        else {
                writelog("Processing commands from file %s\n",x);
                }

        while (!feof(str)) {
                if (!fgets(line,sizeof(line),str)) break;
                if (x = strchr(line,'\n')) *x = '\0';
                x = line;
                while (*x && isspace(*x)) x++;
                if (line[0] == '\0') continue;
                if (line[0] == ';') continue;
                if (line[0] == '!') continue;
                printf("PJBTEST> %s\n",x);
                res = swparse(x,&cmdtab,&errormsg);
                if (res == SWPBLANK) continue;
                if (res < 0) {
                        writelog("%s\n",errormsg);
                        continue;
                        }
                res = cmdtab.objects[res].value;
                res = (*cmdtab.functions[res])();
                if (!running) break;
                }

        fclose(str);

        writelog("Command file finished\n");

        return 1;       
}


/*  *********************************************************************
    *  docmd(cmd)                                                       *
    *                                                                   *
    *  Process a single command                                         *
    *                                                                   *
    *  Input Parameters:                                                *
    *      cmd - command line to process                                *
    *                                                                   *
    *  Return Value:                                                    *
    *      result                                                       *
    ********************************************************************* */
unsigned int docmd(char *cmd)
{
        int res;
        char *errormsg;

        if (cmd[0] == '@') {
                docommandfile(cmd+1);
                return 0;
                }
        
        res = swparse(cmd,&cmdtab,&errormsg);
        if (res == SWPBLANK) return 0;
        if (res < 0) {
                printf("%s\n",errormsg);
                return 0;
                }
        res = cmdtab.objects[res].value;
        return (*cmdtab.functions[res])();
}


/*  *********************************************************************
    *  scan_hwaddr(txt,addr)						*
    *  									*
    *  Scan a hardware address						*
    *  									*
    *  Input parameters: 						*
    *  	   txt								*
    *  	   addr								*
    * 									*
    *  Return value:							*
    *  	   0 ok								*
    *  	   nonzero - error						*
    ********************************************************************* */
int scan_hwaddr(char *txt,unsigned char *addr)
{
    int a,b,c,d,e,f;
    int res;

    res = sscanf(txt,"%x-%x-%x-%x-%x-%x",
		 &a,&b,&c,&d,&e,&f);

    if (res != 6) {
	res = sscanf(txt,"%2x%2x%2x%2x%2x%2x",
		     &a,&b,&c,&d,&e,&f);
	if (res != 6) {
	    return -1;
	    }
	}

    *addr++ = (unsigned char) a;
    *addr++ = (unsigned char) b;
    *addr++ = (unsigned char) c;
    *addr++ = (unsigned char) d;
    *addr++ = (unsigned char) e;
    *addr++ = (unsigned char) f;

    return 0;
}



unsigned int pjb_dump()
{
    char *x;

    if (x = getarg(0,"dump mode (1=on,0=off): ")) {
        PJB_SetDebugMode(atoi(x));
        }
    return 0;
}



unsigned int pjb_exit()
{
    writelog("Exiting PJBTEST\n");
    running = FALSE;
    return 0;
}

unsigned int pjb_pjb()
{
    char *x;
    int idx;
    PJB_ID *id;
    PJB_ID newid;
    int err;

    if (pjb) {
	id = PJB_GetID(pjb);
	printf("Current PJB address is: %s\n",addr_text(id->b));
	}
    else {
	printf("Current PJB has not been set\n");
	}

    if ((x = getarg(0,NULL)) == NULL) return 0;

    if (scan_hwaddr(x,newid.b) != 0) {
	printf("invalid hardware address: %s\n",x);
	return CMDERR;
	}

    if (pjb) {
	PJB_Close(pjb);
	pjb = NULL;
	}
    err = PJB_Open(&newid,&pjb);

    if (err != 0) {
	printf("PJB_Open failed, could not open pjb, error %d\n",err);
	return CMDERR;
	}

    printf("PJB address has been set to: %s\n",addr_text(newid.b));
    return 0;
}

unsigned int pjb_ping()
{
    int num;
    int res;
    int idx;
    char *x;
    int verify;
    int len;
    int cmdlen = 0;
    int forever;
    int count = 0;

    forever = switchset(sFOREVER);
    if (x = getarg(0,NULL)) num = atoi(x); else num = 1;
    if (x = getarg(1,NULL)) cmdlen = atoi(x); else cmdlen = -1;

    verify = switchset(sVERIFY);

    for (idx = 0; forever || (idx < num); idx++) {	
	len = (cmdlen < 0) ? rand() % 1024 : cmdlen;
	res = PJB_Ping(pjb,len,verify);
	if (res < 0) {
	    printf("PJB_Ping: error %d (%s)\n",res,
		   error_text(pcli_errors,res));
	    break;
	    }
	count++;
	if ((count % 100) == 0) {
	    printf("%8d\n",count);
	    }
	}

    if (res == 0) {
	if (idx == 0) printf("PJB didn't respond\n");
	else printf("PJB is alive, %u of %u pings responded\n",idx,num);
	}

    return 0;

}


unsigned int pjb_help()
{
    printf("commands:\n");
    printf("\n");
    printf("SOLICIT                   Obtain list of connected PJBs\n");
    printf("PJB xx-xx-xx-xx-xx-xx     Set target PJB address\n");
    printf("EXIT                      Outta here!\n");
    printf("WAIT [sec]                Pause for [sec] seconds\n");
    printf("\n");
    printf("* Options that work on all firmware images\n");
    printf("\n");
    printf("PING [count] [/VERIFY] [/FOREVER]  Ping PJB and verify data\n");
    printf("REBOOT [/MAIN]            Reboot to flash update or main program\n");
    printf("READTOC [filename]        Read TOC to screen or store in file\n");
    printf("WRITETOC filename         Write TOC from local disk file\n");
    printf("READBLKHDR                Read block header\n");
    printf("FSINFO                    Request filesystem info\n");
    printf("\n");
    printf("* Options for flash recovery mode only\n");
    printf("\n");
    printf("FLASH image-name [/EXECUTE] Write or execute an image file\n");
    printf("\n");
    return CMDERR;
}



void showblock(FS_ALLOCBLOCK *blk,int firstonly)
{
    PJB_Checksum c;

    printf("Next block (hex): %02X%02X%02X\n",
	   blk->h.h.h_next.addr[0],
	   blk->h.h.h_next.addr[1],
	   blk->h.h.h_next.addr[2]);
    printf("Prev block (hex): %02X%02X%02X\n",
	   blk->h.h.h_prev.addr[0],
	   blk->h.h.h_prev.addr[1],
	   blk->h.h.h_prev.addr[2]);

    printf("Alt. Next block (hex): %02X%02X%02X\n",
	   blk->h.h.h_next2.addr[0],
	   blk->h.h.h_next2.addr[1],
	   blk->h.h.h_next2.addr[2]);
    printf("Alt. Prev block (hex): %02X%02X%02X\n",
	   blk->h.h.h_prev2.addr[0],
	   blk->h.h.h_prev2.addr[1],
	   blk->h.h.h_prev2.addr[2]);

    if (!firstonly) {
	printf("Tail Next block (hex): %02X%02X%02X\n",
	   blk->t.t.t_next.addr[0],
	   blk->t.t.t_next.addr[1],
	   blk->t.t.t_next.addr[2]);
	printf("Tail checksum (hex): %02X%02X%02X%02X\n",
	   blk->t.t.t_chksum[0],
	   blk->t.t.t_chksum[1],
	   blk->t.t.t_chksum[2],
	   blk->t.t.t_chksum[3]);

	memset(c.b,0,4);
	cksum((u8 *) blk,((u8 *) &(blk->t.t.t_chksum[0])) - ((u8 *) blk),c.b);
	printf("Computed checksum:   %02X%02X%02X%02X\n",
	       c.b[0],c.b[1],c.b[2],c.b[3]);

	memset(c.b,0,4);
	cksum((u8 *) blk,((u8 *) &(blk->t.t.t_chksum[0])) - ((u8 *) blk) - 4,c.b);
	printf("Computed checksum:   %02X%02X%02X%02X\n",
	       c.b[0],c.b[1],c.b[2],c.b[3]);

	}

    printf("User data: %s\n",blk->h.h.userdata);
}



unsigned int pjb_readblkhdr()
{
    char *x;
    unsigned char buf[PJB_ClickSize];
    int err;
    FS_ALLOCBLOCK *blk;
    int low,high,cnt;
    time_t starttime,endtime;

    if ((x = getarg(0,"block number: ")) == NULL) return CMDERR;

    if (sscanf(x,"%d-%d",&low,&high) != 2) {
	low = high = atoi(x);
	}

    time(&starttime);
    cnt = 0;

    while (low <= high) {

	if (kbhit()) break;

	err = PJB_ReadAllocBlockHead(pjb,buf,low);

	if (err == 0) printf("Header read successfully\n");
	else {
	    printf("PJB_ReadAllocBlockHead: %s\n",error_text(pcli_errors,err));
	    if (err != PJB_ERR_CHECKSUM) break;
	    }

	blk = (FS_ALLOCBLOCK *) buf;
	showblock(blk,1);
	low++;
	cnt++;
	}

    time(&endtime);

    if (endtime-starttime) {
	printf("Speed: %6.4f blocks/second\n",((double)cnt)/(double)(endtime-starttime));
	}

    return 0;    
}



unsigned int pjb_reboot()
{
    int from = 1;
    int res;

    if (switchset(sDEBUG)) from = 2;
    else if (switchset(sMAIN)) from = BOOTMETHOD_MAIN;
    else if (switchset(sUPDATE)) from = BOOTMETHOD_RECOVERY;

    switch (from) {
	case BOOTMETHOD_RECOVERY:
	    printf("Rebooting to flash update program\n");
	    break;
	case 2:
	    printf("Rebooting to debug instruction\n");
	    break;
	case BOOTMETHOD_MAIN:
	    printf("Rebooting to main program\n");
	    break;
	}

    res = PJB_Reboot(pjb,from);
    if (res != 0) printf("reboot: %s\n",error_text(pcli_errors,res));
    return 0;
}

unsigned int pjb_writetoc()
{
    int fh;
    char *fname;
    unsigned char buffer[1024];
    PJB_Checksum checksum;
    int res,pjbres;
    u16 click;

    fname = getarg(0,"file name: ");
    if (fname == NULL) return CMDERR;

    fh = open(fname,O_RDONLY | O_BINARY);
    if (fh < 0) {
	perror(fname);
	return CMDERR;
	}

    memset(checksum.b,0,sizeof(checksum.b));
    click = 0;

    for (;;) {	
	memset(buffer,0,sizeof(buffer));
	res = read(fh,buffer,1024);
	if (res <= 0) break;
       
	cksum(buffer,1024,checksum.b);
	pjbres = PJB_WriteTOCClick(pjb,(u16)click,buffer);
	if (pjbres < 0) break;

	click++;
	}

    close(fh);

    if (res < 0) {
	printf("Could not read table of contents file: %s\n",fname);
	return CMDERR;
	}
    if (pjbres < 0) {
	printf("Could not write table of contents: %d (%s)\n",
	       pjbres,error_text(pcli_errors,pjbres));
	return CMDERR;
	}

    pjbres = PJB_CommitTOC(pjb,click,&checksum);
    if (pjbres < 0) {
	printf("Could not commit table of contents: %d (%s)\n",
	       pjbres,error_text(pcli_errors,pjbres));
	return CMDERR;
	}

    printf("Successfully wrote and commited a %u click TOC\n",click);
    printf("TOC's checksum was: %02X %02X %02X %02X\n",
	   checksum.b[0],
	   checksum.b[1],
	   checksum.b[2],
	   checksum.b[3]);

    return 0;  
    
}

unsigned int pjb_readtoc()
{
    int fh;
    char *fname;
    unsigned char buffer[1024];
    PJB_DiskInfo info;
    int res,pjbres;
    u16 click;
    int idx;

    fname = getarg(0,NULL);
    if (fname == NULL) {
	fh = fileno(stdout);
	}
    else {
	fh = open(fname,O_RDWR | O_CREAT | O_TRUNC | O_BINARY,
		  S_IREAD | S_IWRITE);
	if (fh < 0) {
	    perror(fname);
	    return CMDERR;
	    }
	}

    pjbres = PJB_GetDiskInfo(pjb,&info);
    if (pjbres < 0) {
	printf("Could not read filesystem info: %s\n",
	       error_text(pcli_errors,res));
	return CMDERR;
	}

    printf("Number of clicks in this TOC: %d\n",
	   info.curclicksintoc);

    for (click = 0; click < info.curclicksintoc; click++) {
	pjbres = PJB_ReadTOCClick(pjb,click,buffer);
	for (idx = 0; idx < sizeof(buffer); idx++) {
	    if (buffer[idx] == 0) break;
	    }
	write(fh,buffer,idx);
	}

    if (fname) close(fh);

    if (pjbres < 0) {
	printf("Could not read filesystem info: %s\n",
	       error_text(pcli_errors,res));
	return CMDERR;
	}

    return 0;
}


unsigned int pjb_info()
{
    PJB_INFO info;
    int res;

    res = PJB_GetInfo(pjb,&info);

    if (res < 0) {
	printf("PJB_GetDiskInfo: %s\n",error_text(pcli_errors,res));
	return CMDERR;
	}

    printf("Protocol:       %s\n","USB");
    printf("Address:        %s\n",addr_text(info.id.b));
    printf("Name:           %s\n",info.name[0] ? info.name : "(name not set)");
    printf("HwRev:          %u\n",info.hwver);
    printf("SwRev:          %u.%u.%u\n",
	   (info.swver >> 16),
	   (info.swver >> 8) & 0xFF,
	   (info.swver & 0xFF));
    printf("Features:       %06X\n",info.features);
    printf("Serial #:       %s\n",addr_text(info.ssn));

    return 0;

}

unsigned int pjb_diskinfo()
{
    PJB_DiskInfo info;
    int res;

    res = PJB_GetDiskInfo(pjb,&info);
    if (res < 0) {
	printf("PJB_GetDiskInfo: %s\n",error_text(pcli_errors,res));
	return CMDERR;
	}

    printf("Version:             %d\n",info.version);
    printf("Max clicks in toc    %d\n",info.maxclicksintoc);
    printf("num alloc blocks     %d\n",info.numallocblocks);
    printf("error count          %d\n",info.errorcount);
    printf("valid toc copies     %d\n",info.validtoccopies);
    printf("clicks in cur toc    %d\n",info.curclicksintoc);
    printf("is this an old toc   %d\n",info.isoldtoc);

    return 0;

}


unsigned int pjb_run()
{
    setswitch(sEXECUTE,NULL);
    return pjb_flash();
}


unsigned int pjb_flash()
{
    int fh;
    char *name;
    PJBFLASHIMAGE hdr;
    unsigned char buffer[1024];
    PJB_Checksum checksum;
    int mode = FLASHREC_MAINCODE;
    int res;
    int total;
    int blknum;

    if (switchset(sREBOOT)) {
	printf("Rebooting...");
	PJB_Reboot(pjb,BOOTMETHOD_RECOVERY);
	printf("Waiting for flash update mode...");
	pjbsleep(5);
	for (res = 0; res < 4; res++) {
	    if (PJB_Ping(pjb,64,0) >= 0) break;
	    }
	if (res == 4) {
	    printf("PJB did not go into flash mode\n");
	    return CMDERR;
	    }
	printf("done\n");
	}

    name = getarg(0,"file name: ");
    if (!name) return CMDERR;
    fh = open(name,O_BINARY | O_RDONLY);
    if (fh < 0) {
	perror(name);
	return CMDERR;
	}

    res = read(fh,&hdr,sizeof(hdr));
    if (res != sizeof(hdr)) {
	printf("Could not read image header\n");
	goto done;
	}

    if (memcmp(hdr.seal,"PJB\x1A",4) != 0) {
	printf("Invalid PJB image file\n");
	goto done;
	}

    printf("Valid PJB image, version %u.%u.%u, CRC %02X%02X%02X%02X, size %u bytes\n",
	   hdr.version[0],
	   hdr.version[1],
	   hdr.version[2],
	   hdr.crc[0],
	   hdr.crc[1],
	   hdr.crc[2],
	   hdr.crc[3],
	   hdr.size);

    res = PJB_OpenSoft(pjb);

    if (res < 0) {
	printf("Could not open flash software, %s\n",
	       error_text(pcli_errors,res));
	return CMDERR;
	}

    printf("Remote flash is open, transmitting data\n");

    total = 0;
    blknum = 0;
    for (;;) {
	res = read(fh,buffer,sizeof(buffer));
	if (res < 0) {
	    perror("Could not read image file");
	    goto done;
	    }
	if (res == 0) break;
	total += res;
	res = PJB_WriteSoft(pjb,(u16)blknum,buffer,res);
	if (res < 0) {
	    printf("Could not write block %u to flash: %s\n",
		   blknum,error_text(pcli_errors,res));
	    goto done;
	    }
	blknum++;
	}

    if (switchset(sEXECUTE)) mode = FLASHREC_EXECUTE;

    memcpy(checksum.b,hdr.crc,sizeof(checksum.b));
    res = PJB_CommitSoft(pjb,&checksum,mode);

    if (mode == FLASHREC_EXECUTE) printf("Executing remote image\n");
    else printf("Image transferred, writing to flash\n");
    if (res < 0) {
	if (mode == FLASHREC_EXECUTE) {
	    printf("Could not execute remote image: %s\n",
		   error_text(pcli_errors,res));
	    }
	else {
	    printf("Could not commit image to flash: %s\n",
		   error_text(pcli_errors,res));
	    }
	}

done:
    close(fh);

    return 0;
}

unsigned int do_solicit(int autoflg)
{
    PJB_INFO ids[10];
    char cmd[100];
    int maxids = 10;
    int res,idx;
    int err;

    if (pjb) PJB_Close(pjb);
    pjb = NULL;

    printf("Searching for personal jukeboxes... ");
    res = PJB_Enumerate(ids,maxids);

    if (res == PJB_ERR_TIMEOUT) res = 0;

    if (res < 0) {
	printf("Could not enumerate PJBs: %s\n",
		    error_text(pcli_errors,res));
	return CMDERR;
	}

    if (res == 0) {
	printf("No PJBs responded.\n");
	return 0;
	}

    printf("%u PJB%s responded:\n",res,(res == 1) ? "" : "s");

    for (idx = 0; idx < res; idx++) {
	printf("#%d:  SN:%s  Name:\"%s\"\n",idx,
	       addr_text(ids[idx].id.b),
	       ids[idx].name[0] ? ids[idx].name : "(name not set)");
        printf("                        HwRev:%u SwRev:%u.%u.%u Features:%06X\n",
	       ids[idx].hwver,
	       (ids[idx].swver >> 16),
	       (ids[idx].swver >> 8) & 0xFF,
	       (ids[idx].swver & 0xFF),
	       ids[idx].features);
	printf("\n");
	}	


    if ((autoflg && (res > 0)) || (res == 1)) idx = 0;
    else {

	printf("Enter number of jukebox to select: "); 
	fgets(cmd,sizeof(cmd),stdin);

	if (cmd[0] == 0) return CMDERR;
	idx = atoi(cmd);
	if (idx >= res) {
	    printf("invalid selection\n"); return CMDERR;
	    }
	}


    pjb = NULL;
    err = PJB_Open(&(ids[idx].id),&pjb);

    if (pjb == NULL) {
	printf("Could not open PJB, error %d\n",err);
	return CMDERR;
	}

    printf("PJB address has been set to: %s\n",addr_text(ids[idx].id.b));

    printf("\n");
    
    return 0;

}

unsigned int pjb_solicit()
{
    return do_solicit(FALSE);
}


unsigned int pjb_wait()
{
    char *x;
    int sec;
    
    if (x = getarg(0,NULL)) sec = atoi(x);
    else sec = 1;
    
    pjbsleep(sec);

    return 0;
}


/*  *********************************************************************
    *  Main(argc,argv)                                                  *
    *                                                                   *
    *  Yep, it's the main thing.                                        *
    *                                                                   *
    *  Input Parameters:                                                *
    *      argc,argv - guess.                                           *
    *                                                                   *
    *  Return Value:                                                    *
    *      nothing                                                      *
    ********************************************************************* */
int main(int argc,char *argv[])
{
    char cmd[100];
    int res;
    char *firstcmd = NULL;
    u8 *x;
    int initok;
    char *tmp;

    argc = scan_switch(argc,argv);    

    printf("PJBTEST - Program to test PJB network api - version 4.0 [Linux]\n");
    printf("Copyright (C) 1998,1999,2000 Compaq Computer Corporation\n");
    printf("\n");

    initok = PJB_InitLib();

    PJB_SetDebugCallback(debugcallback,NULL);

    if (initok) printf("PJBAPI interface initialized.\n");
    else printf("PJBAPI interface did not initialize\n");

    printf("\n");

    ablk = (FS_ALLOCBLOCK *) malloc(sizeof(FS_ALLOCBLOCK));

    do_solicit(TRUE);

    if (argc > 1) firstcmd = argv[1];

    while (running) {
	printf("PJBTEST> "); 
	if (firstcmd) strcpy(cmd,firstcmd);
	else fgets(cmd,sizeof(cmd),stdin);
	if (tmp = strchr(cmd,'\n')) *tmp = '\0';
	firstcmd = NULL;

	docmd(cmd);
	}

    exit(0);
    return 0;
}




