/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Command line interface to the PJB	File: PJB.C		**
   **									**
   **  This is actually more of a test program for the PJBTOC library	**
   **  but it can be used to put files onto your PJB.			**
   **									**
   **  Authors:  Compaq Corporate Research                              **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <fcntl.h>

#ifdef __WIN32__
#include <io.h>
#endif

#include "pjbtoc.h"

#ifdef __WIN32__
#define strcasecmp strcmpi
#endif

extern int switch_set(char *);
extern int scan_switch(int argc,char *argv[]);
extern int switch_value(char *,char **);

typedef struct PjbCmd {
    int (*func)(PJBTOC *t,int argc,char *argv[]);
    int needopen;
    char *cmdname;
    char *descr;
} PJBCMD;

int showtoc(PJBTOC *t,int argc,char *argv[]);
int addmp3(PJBTOC *t,int argc,char *argv[]);
int showinfo(PJBTOC *t,int argc,char *argv[]);
int flash(PJBTOC *t,int argc,char *argv[]);
int reboot(PJBTOC *t,int argc,char *argv[]);


PJBCMD cmdtab[] = {
    {showtoc,TRUE,"ls","list the contents of the PJB [-sets] [-discs] [-tracks] [-all]"},
    {addmp3,TRUE,"add","add MP3s to the PJB [-set=set_name] [-disc=disc_name] [-notags]"},
    {showinfo,TRUE,"info","display information about your PJB"},
    {reboot,FALSE,"reboot","reboot PJB.  Add [-main] for main program"},
    {flash,FALSE,"flash","update PJB flash, or [-execute] program"},
    {NULL,FALSE,NULL,NULL}
};


int reboot(PJBTOC *t,int argc,char *argv[])
{
    int res;

    res = PJBTOC_Reboot(t,switch_set("main"));
    if (res != 0) printf("Reboot: error %d\n",res);

    return res;
}

int flash(PJBTOC *t,int argc,char *argv[])
{
    int res;

    if (argc < 2) {
	printf("Must specify a file name\n");
	return -1;
	}

    res = PJBTOC_UpdateFlash(t,argv[1],switch_set("main"));

    if (res != 0) printf("Flash: error %d\n",res);

    return res;
}

/************************************************************************/

int showinfo(PJBTOC *t,int argc, char *argv[])
{
    printf("\n");
    printf("PJB's name:       %s\n",t->info.name);
    printf("Firmware Version: %u.%u.%u\n",
	   (t->info.swver >> 16) & 0xFF,
	   (t->info.swver >> 8) & 0xFF,
	   (t->info.swver >> 0) & 0xFF);
    printf("Hardware version: %u\n",t->info.hwver);
    printf("PJB's serial #:   %02X%02X%02X%02X%02X%02X\n",
	   t->info.ssn[0],
	   t->info.ssn[1],
	   t->info.ssn[2],
	   t->info.ssn[3],
	   t->info.ssn[4],
	   t->info.ssn[5]);
    printf("PJB's disk size:  %u MB\n",
	   (t->diskinfo.numallocblocks * 128) / 1024);
    printf("Size of the TOC:  %uKB\n",
	   t->diskinfo.curclicksintoc);
    printf("Available Space:  %u allocation units\n",
	   t->allocmap->nFreeBlocks);
    return 0;

}

/************************************************************************/

#define PRINT_SETS 1
#define PRINT_DISCS 2
#define PRINT_TRACKS 4
#define PRINT_DETAILS 8

int printfunc(PJBTOC *t,JNode n,void *arg)
{
    int what = *((int *) arg);
    static char *enctypes[4] = {"Unknown","MP3","PCM","AAC"};

    switch (n->type) {
	case JNODE_TYPE_SET:
	    if (what & PRINT_SETS) printf("Set: %s\n",n->text);
	    break;
	case JNODE_TYPE_DISK:
	    if (what & PRINT_DISCS) printf("    Disc: %s\n",n->text);
	    break;
	case JNODE_TYPE_TRACK:
	    if (what & PRINT_TRACKS) {
		printf("        Track: %s\n",n->text);
		if (what & PRINT_DETAILS) {
		    printf("               Start=%u.%u End=%u.%u BitRate=%u Time=%u\n",
			   n->trackAddr.pjb.start_block,
			   n->trackAddr.pjb.start_block_offset,
			   n->trackAddr.pjb.end_block,
			   n->trackAddr.pjb.end_block_bytes,
			   n->bit_rate,
			   n->time/75);
		    printf("               CDID=%d OrigTrack=%d EncType=%s\n",
			   n->iCdUID,n->iTrack,
			   ((n->enc_type >= 1) && (n->enc_type <= 3)) ? 
			   enctypes[n->enc_type] : "Unknown");
		    printf("\n");
		    }
		}
	    break;
    }

    return 0;
}


int readcallback(PJBTOC *t,u8 *buffer,long length,long *numread,void *arg)
{
    MP3READER *reader = (MP3READER *) arg;
    int res;

    printf(".");

    res = MP3Read(reader,buffer,length);

    if (res < 0) *numread = 0;
    else *numread = res;

    if (res < 0) return -1;
    else return 0;

}

int addonemp3(PJBTOC *t,char *filename,char *setname,char *discname,int usetags)
{
    JNode newnode;
    char trackname[2048];
    char discnamebuffer[2048];
    char *x;
    int res;
    MP3READER *reader;
    JInsertPt ins;
    JNode setnode;
    JNode discnode;

    reader = MP3Create();

    //
    // Open the MP3 file
    //


    res = MP3Open(reader,filename,FALSE);
    if (res != 0) {
	printf("Could not open %s: error %d\n",filename,res);	
	MP3Destroy(reader);
	return 0;		// keep going 
	}

    //
    // Construct the default track name from the file name
    // Strip off the extension and the directory name.

    x = NULL;
#ifdef __linux__
    x = strrchr(filename,'/');
#endif
#ifdef __WIN32__
    x = strrchr(filename,'\\');
#endif
    if (x) x++; else x = filename;

    strcpy(trackname,filename);
    if (x = strrchr(trackname,'.')) *x = '\0';

    printf("-----------------------\n");
    printf("File:     %s\n",filename);

    //
    // Try real hard to make sense of the tags
    //

    if (MP3IsTagged(reader)) {
	if (x = MP3GetGenre(reader)) setname = x;
	discnamebuffer[0] = '\0';
	if (x = MP3GetArtist(reader)) strcat(discnamebuffer,x);
	if (MP3GetAlbum(reader) && (x != NULL)) {
	    strcat(discnamebuffer," / ");
	    }
	if (x = MP3GetAlbum(reader)) strcat(discnamebuffer,x);

	if (discnamebuffer[0]) discname = discnamebuffer;

	if (x = MP3GetTitle(reader)) strcpy(trackname,x);
	}

    printf("Set:      %s\n",setname);
    printf("Disc:     %s\n",discname);
    printf("Track:    %s\n",trackname);
    printf("BitRate:  %d\n",MP3GetBitRate(reader));
    printf("Time:     %d sec\n",MP3GetPlayTime(reader));
    printf("Loading...");

    //
    // Try to locate the JNode for the set, create the set if
    // necessary.
    //

    setnode = PJBTOC_FindSet(t,setname,TRUE);

    // 
    // Given the set, try to find the JNode for the disc,
    // and create the disc if necessary.
    //

    discnode = PJBTOC_FindDisc(t,setnode,discname,TRUE);

    //
    // Now create the track node
    //

    newnode = MakeNode(JNODE_TYPE_TRACK,trackname);

    newnode->bit_rate = MP3GetBitRate(reader);
    newnode->time = MP3GetPlayTime75(reader);
    newnode->enc_type = TC_ENC_MP3;

    ins.parent = discnode;
    ins.pred = JNODE_LAST;

    res = PJBTOC_WriteTrack(t,newnode,&ins,readcallback,reader,
			    MP3GetID3Tag(reader));

    if (res == 0) printf("OK");
    else printf("Error %d",res);

    printf("\n");

    MP3Close(reader);

    MP3Destroy(reader);

    return 0;

}

int addmp3(PJBTOC *t,int argc,char *argv[])
{
    char *setname;
    char *discname;
    int res = 0;
    int idx;
    int usetags = FALSE;

    //
    // Default the set and disc to "untagged" if we cannot find a tag.
    //

    if (!switch_set("notags")) {
	usetags = TRUE;
	setname = "Untagged";
	discname = "Untagged";
	}

    if (!switch_value("set",&setname) || !switch_value("disc",&discname)) {
	if (!usetags) {
	    printf("You must specify both a set and disc name\n");
	    return 1;
	    }
	}

    //
    // Finally, go through the args and add each track.
    //

    for (idx = 1; idx < argc; idx++) {
	if (res = addonemp3(t,argv[idx],setname,discname,usetags)) break;

	if (t->modified && ((idx % 10) == 0)) {
	    printf("Writing TOC..."); fflush(stdout);
	    res = PJBTOC_FlushJukebox(t);
	    if (res) printf("Could not flush TOC to PJB: %d\n",res);
	    else printf("Done\n");
	    }
	}

    return res;
}




int showtoc(PJBTOC *t,int argc,char *argv[])
{
    int what = 0;

    if (switch_set("sets")) what |= PRINT_SETS;
    if (switch_set("discs")) what |= PRINT_SETS | PRINT_DISCS;
    if (switch_set("tracks")) what |= PRINT_SETS | PRINT_DISCS | PRINT_TRACKS;
    if (switch_set("all")) what |= PRINT_SETS | PRINT_DISCS | PRINT_TRACKS | PRINT_DETAILS;

    if (what == 0) what = PRINT_SETS | PRINT_DISCS;

    PJBTOC_WalkTree(t,printfunc,&what);

    return 0;
}



void usage(void)
{
    PJBCMD *c;

    printf("PJB - Command-line interface to the Compaq Personal Jukebox"
	   " - version 3.0\n");
    printf("Copyright (C) 2000 Compaq Computer Corporation\n");
    printf("THIS SOFTWARE HAS NO WARRANTY; USE AT YOUR OWN RISK!\n");
    printf("\n");
    printf("Usage: PJB command args...\n");
    printf("\n");
    printf("Available commands:\n\n");

    c = cmdtab;
    while (c->func) {
	printf("PJB %-10.10s %s\n",c->cmdname,c->descr);
	c++;
	}

    printf("\n");

    exit(1);
}


int main(int argc,char *argv[])
{
    PJBTOC *t;
    int err;
    PJBCMD *c;

    argc = scan_switch(argc,argv);

    if (argc < 2) usage();

    //
    // Try to find the command
    //

    c = cmdtab;
    while (c->func) {
	if (strcasecmp(c->cmdname,argv[1]) == 0) break;
	c++;
	}

    if (!c->func) {
	printf("Invalid command: %s\n",argv[1]);
	usage();
	exit(1);
	}

    //
    // Initialize library 
    //

    PJB_InitLib();

    //
    // Try to open the PJB
    //

    printf("Opening PJB..."); fflush(stdout);

    // under normal circumstances, always say "TRUE" for
    // the 'readtoc' parameter.  Most of the lib won't
    // work without it!
    // flash update is special, since we can't read the toc
    // during a flash upgrade.

    err = PJBTOC_OpenPjb(0,&t,c->needopen);
    if (err) {
	printf("Could not open PJB: %d\n",err);
	exit(1);
	}

    printf("Done!\n");

    if (c->needopen) {
	printf("Free blocks: %d\n",t->allocmap->nFreeBlocks);
	}


    //
    // Execute the command
    //

    err = (*(c->func))(t,argc-1,argv+1);

    //
    // Flush TOC if it has been modified
    //

    if (c->needopen && t->modified) {
	printf("Writing TOC..."); fflush(stdout);
	err = PJBTOC_FlushJukebox(t);
	if (err) printf("Could not flush TOC to PJB: %d\n",err);
	else printf("Done\n");
	}

    //
    // Close the PJB
    //

    PJBTOC_Close(t);

    exit(err);
    return err;
    
}

