/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  USB "Control" module                    File: USBCTL.C           **
   **									**
   **  This module provides a "multiplex" interface for the USB         **
   **  protocol used in the Personal Jukebox.  At present, we only      **
   **  one type of packet through this interface, but should we add     **
   **  more types in the future (perhaps for different services) then   **
   **  this would be the place to do it.                                **
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

/*
 * Note to future hackers:
 *
 * There's a real problem with the way this is layered on
 * the underlying device interface.  In particular, the 
 * USB "port" structure is not really associated with a given
 * connection to a physical PJB (i.e., the handle to the
 * device driver).  This dates back from the prototype days
 * when the PJB was an Ethernet device (yep, Ethernet!),
 * and so therefore many of these routines look too empty,
 * and I was just too lazy to fix it.  Perhaps someday when
 * we all have multiple PJBs running multiple protocols it 
 * will be worth fixing it.  Until then, caveat programmer.
 *
 */


#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pjbtypes.h"
#include "queue.h"
#include "ebuf.h"
#include "pjbapi.h"
#include "pjbdev.h"
#include "usbctl.h"

/*  *********************************************************************
    *  Constants							*
    ********************************************************************* */


/*
 * USB packets are prefixed with the following header:
 *
 * 0xFF 'P' 'J' 'B'		Seal (4 bytes)
 * <length_lo> <length_hi>	Packet length (2 bytes)
 * <cksum_lo> <cksum_hi>	1's comp checksum of data portion
 * <protocol>			Protocol code (0=pjb, 1=debug)
 * <mbz>			Reserved byte, must be zero
 */


/*  *********************************************************************
    *  Structures							*
    ********************************************************************* */

typedef struct UsbPort {
    int up_protocol;
    int up_endpoint;
    int (*up_func)(EBUF *);		/* event processing function */
} USBPORT;


/*  *********************************************************************
    *  Globals								*
    ********************************************************************* */

QBLOCK usb_freelist = {&usb_freelist,&usb_freelist};

static int usb_bound = FALSE;

USBPORT usb_ports[USB_MAX_PORTS] = {0};

/*  *********************************************************************
    *  USB_INIT(pool,poolsize)						*
    *  									*
    *  Initialize the USB interface.					*
    *  									*
    *  Input parameters: 						*
    *  	   pool - address of buffer pool				*
    *  	   poolsize - size of buffer pool, in bytes (not packed)	*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   else error code						*
    ********************************************************************* */

int usb_init(u8 *pool,int poolmemsize)
{
    u8 *bufptr;
    EBUF *ebuf;
    int idx;

    memset(&usb_ports,0,sizeof(usb_ports));

    /*
    * Initialize the buffer pool
    */

    bufptr = pool;
    poolmemsize /= sizeof(EBUF);
    q_init(&usb_freelist);

    for (idx = 0; idx < poolmemsize; idx++) {
	ebuf = (EBUF *) bufptr;
	ebuf->eb_length = 0;
	ebuf->eb_status = 0;
	q_enqueue(&(usb_freelist),(QBLOCK *) ebuf);
	bufptr += sizeof(EBUF);
	}

    usb_bound = TRUE;

    return 0;
}


/*  *********************************************************************
    *  USB_UNINIT()							*
    *  									*
    *  Undo the effect of usb_init()					*
    *  									*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */
void usb_uninit(void)
{
    if (!usb_bound) return;

    usb_bound = FALSE;
}


/*  *********************************************************************
    *  USB_OPEN(protocol,endpoint,func)					*
    *  									*
    *  Open a USB protocol						*
    *  									*
    *  Input parameters: 						*
    *  	   protocol - USB protocol ID (in packet header)		*
    *      endpoint - USB endpoint ID (0=main, 1=alternate)		*
    *	   func - function to call on tx/rx events			*
    *  	   								*
    *  Return value:							*
    *  	   port # (same as protocol #)					*
    ********************************************************************* */

int usb_open(int protocol,int endpoint,int (*func)(EBUF *))
{
    USBPORT *up;

    if (protocol >= USB_MAX_PORTS) return -1;

    up = &(usb_ports[protocol]);

    up->up_endpoint = endpoint;
    up->up_func = func;
    up->up_protocol = protocol;

    return protocol;
}

/*  *********************************************************************
    *  USB_CLOSE(endpoint)						*
    *  									*
    *  Close a port							*
    *  									*
    *  Input parameters: 						*
    *  	   endpoint - endpoin number to close			       	*
    *  	   								*
    *  Return value:							*
    *  	   0								*
    ********************************************************************* */

int usb_close(int port)
{
    USBPORT *up;

    if (port >= USB_MAX_PORTS) return -1;

    up = &(usb_ports[port]);

    up->up_protocol = 0;
    up->up_endpoint = 0;
    up->up_func = NULL;

    return 0;
}

/*  *********************************************************************
    *  USB_ALLOCBUF(port)						*
    *  									*
    *  Allocate a buffer for this port and reserve space in the		*
    *  buffer for any overhead that we plan on applying			*
    *  									*
    *  Input parameters: 						*
    *  	   port - port we want to request a buffer for			*
    *  	   								*
    *  Return value:							*
    *  	   EBUF pointer or NULL if none left.				*
    ********************************************************************* */
EBUF *usb_allocbuf(int port)
{
    EBUF *buf;

    buf = (EBUF *) q_deqnext(&usb_freelist);
    if (!buf) return NULL;

    /*
     * ptr, length set to the beginning and max size of buffer
     * Adjust packet size to account for header.
     */

    buf->eb_ptr = &(buf->eb_data[0]);
    buf->eb_length = 1518;

    buf->eb_ptr += USB_PKT_HDRLEN;
    buf->eb_length -= USB_PKT_HDRLEN;
    buf->eb_status = 0;

    /*
     * remember port we associated this buffer with
     */

    buf->eb_port = port;

    return buf;

}

/*  *********************************************************************
    *  USB_FREEBUF(buf)							*
    *  									*
    *  Free a buffer.							*
    *  									*
    *  Input parameters: 						*
    *  	   buf - buffer to free						*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

void usb_freebuf(EBUF *buf)
{
    q_enqueue(&usb_freelist,(QBLOCK *) buf);
}



/*  *********************************************************************
    *  USB_INITBUF(buf)					  	        *
    *  									*
    *  Reinitialize an ebuf so we can start adding data to it.		*
    *  									*
    *  Input parameters: 						*
    *  	   buf - ebuf to hack on					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */
void usb_initbuf(EBUF *buf)
{
    buf->eb_length = 0;
    buf->eb_ptr = &(buf->eb_data[USB_PKT_HDRLEN]);
}

/*  *********************************************************************
    *  USB_TRANSMIT(endpoint,buf)					*
    *  									*
    *  Transmit a usb packet						*
    *  									*
    *  Input parameters: 						*
    *  	   endpoint - port to transmit buffer on		       	*
    *  	   buf - buffer (EBUF) to send					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, else error						*
    ********************************************************************* */

int usb_transmit(int port,PJBCOMMHANDLE *h,EBUF *buf)
{
    unsigned int idx;
    USBPORT *p = &usb_ports[port];
    u8 *ptr;
    unsigned int chk = 0;

    /* calculate checksum */

    ptr = buf->eb_ptr;
    for (idx = 0; idx < buf->eb_length; idx++) {
	chk += *ptr++;
	}

    /* reset pointers to the beginning of the data. */

    buf->eb_ptr = &(buf->eb_data[0]);
    buf->eb_port = port;
    buf->eb_status &= EBUF_NO_TX_FREE;

    /* fill in seal */

    ptr = buf->eb_ptr;
    *ptr++ = 0xFF;
    *ptr++ = 'P';
    *ptr++ = 'J';
    *ptr++ = 'B';

    /* fill in length */

    *ptr++ = (buf->eb_length) & 0xFF;
    *ptr++ = ((buf->eb_length) >> 8);

    /* fill in checksum... well, at least it's a sum. */

    *ptr++ = chk & 0xFF;
    *ptr++ = chk >> 8;

    /* fill in protocol ID and reserved field */

    *ptr++ = p->up_protocol;
    *ptr++ = 0;

    /* EBUF's length should now include the header */

    buf->eb_length += USB_PKT_HDRLEN;

    /* transmit via port driver */

    PJBCOMM_TRANSMIT(h,buf->eb_ptr,buf->eb_length);

    if (!(buf->eb_status & EBUF_NO_TX_FREE)) usb_freebuf(buf);

    buf->eb_length -= USB_PKT_HDRLEN;
    buf->eb_ptr += USB_PKT_HDRLEN;

    return 0;
}





/*  *********************************************************************
    *  USB_RX_CHECK(buf)						*
    *  									*
    *  Do some pre-flight checks on the integrity of this		*
    *  ebuf (receive)							*
    *  									*
    *  Input parameters: 						*
    *  	   buf - buffer to checlk					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - ok							*
    *  	   FALSE - drop it.						*
    ********************************************************************* */

int usb_rx_check(EBUF *buf)
{
    u8 *ptr = buf->eb_ptr;
    unsigned int length;
    unsigned int chk;
    u8 proto;

    /*
     * verify packet length
     */

    if (buf->eb_length < USB_PKT_HDRLEN) {
	goto dropit;
	}

    /*
     * Verify seal
     */

    if (*ptr++ != 0xFF) goto dropit;
    if (*ptr++ != 'P')  goto dropit;
    if (*ptr++ != 'J')  goto dropit;
    if (*ptr++ != 'B')  goto dropit;

    /*
     * Verify length 
     */

    length = *ptr++;
    length += (*ptr++ << 8);

    if (length >= 1500) goto dropit;	/* XXX Constant here */

    /*
     * Get "checksum"
     */

    chk = *ptr++;
    chk += (*ptr++ << 8);

    /* 
     * Get protocol ID
     */

    proto = *ptr++;			

    if (proto >= USB_MAX_PORTS) goto dropit;

    /* 
     * Process MBZ field
     */

    if (*ptr++ != 0) goto dropit;	/* mbz mbz */

    /*
     * point at beginning of data portion of packet
     */

    buf->eb_length -= USB_PKT_HDRLEN;
    buf->eb_ptr = ptr;

    if (length > buf->eb_length) goto dropit;	/* header is bigger than pkt */

    buf->eb_length = length;

    /*
     * remember protocol information
     */

    buf->eb_port = proto;

    /*
     * packet passes! 
     */

    return TRUE;


dropit:
    return FALSE;


}

/*  *********************************************************************
    *  USB_RECEIVE(port,timeout)					*
    *  									*
    *  Use the blocking receive call to wait for the next packet	*
    *  to arrive on the USB.	  					*
    *  									*
    *  Input parameters: 						*
    *  	   port - portal number						*
    *  	   timeout - milliseconds to wait, 0 to wait forever.		*
    *  	   								*
    *  Return value:							*
    *  	   ebuf pointer of packet received, or NULL if timeout occured	*
    ********************************************************************* */

EBUF *usb_receive(int port,PJBCOMMHANDLE *h,int timeout)
{
    EBUF *buf;
    int err;

    buf = usb_allocbuf(port);
    if (!buf) return NULL;

    /*
     * Receive data from the USB device driver.  Put the data right
     * at the beginning of the packet.
     */

    buf->eb_ptr = &(buf->eb_data[0]);
    err = PJBCOMM_RECEIVE(h,&(buf->eb_data[0]),1500+USB_PKT_HDRLEN,timeout);

    /*
     * If we had a receive error, drop the packet.
     */

    if (err < 0) {
	usb_freebuf(buf);
	return NULL;
	}


    /*
     * The overall length of what we got goes in the length field
     */

    buf->eb_length = err;

    /*
     * test the packet for validity - this also advances
     * ebuf_ptr and decrements length accordingly.
     */

    if (usb_rx_check(buf) == FALSE) {
	usb_freebuf(buf);
	return NULL;
	}

    return buf;
}


