/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Doubly-linked queue routines		File: QUEUE.C		**
   **									**
   **  This module contains routines to manipulate doubly-linked 	**
   **  queues.                                                          **
   **									**
   **  Important note:  These queues don't provide any sort of          **
   **  multi-thread support, but you can easily provide it here.        **
   **                                                                   **
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <sys/types.h>
#include "pjbtypes.h"
#include "queue.h"

/*  *********************************************************************
    *  Globals								*
    ********************************************************************* */

static int q_initialized = FALSE;

/*  *********************************************************************
    *  Externs								*
    ********************************************************************* */



/*  *********************************************************************
    *  q_enqueue(qb,item)						*
    *  									*
    *  Add item to a queue, not interlocked				*
    *  									*
    *  Input Parameters: 						*
    *      qb - queue block						*
    *      item - item to add						*
    *      								*
    *  Return Value:							*
    *      Nothing.							*
    ********************************************************************* */

void q_enqueue(QBLOCK *qb,QBLOCK *item)
{
    qb->q_prev->q_next = item;
    item->q_next = qb;
    item->q_prev = qb->q_prev;
    qb->q_prev = item;
}


/*  *********************************************************************
    *  q_dequeue(element)						*
    *  									*
    *  Remove an element from the queue, not interlocked		*
    *  									*
    *  Input Parameters: 						*
    *      element - element to remove					*
    *      								*
    *  Return Value:							*
    *      Nothing.							*
    ********************************************************************* */

void q_dequeue(QBLOCK *element)
{
    element->q_prev->q_next = element->q_next;
    element->q_next->q_prev = element->q_prev;
}


/*  *********************************************************************
    *  q_deqnext(qb)							*
    *  									*
    *  Dequeue next element from the specified queue, not interlocked	*
    *  									*
    *  Input Parameters: 						*
    *      qb - queue block						*
    *      								*
    *  Return Value:							*
    *      next element, or NULL					*
    ********************************************************************* */

QBLOCK *q_deqnext(QBLOCK *qb)
{
    if (qb->q_next == qb) {
	return NULL;
	}

    qb = qb->q_next;

    qb->q_prev->q_next = qb->q_next;
    qb->q_next->q_prev = qb->q_prev;

    return qb;
}


/*  *********************************************************************
    *  Q_MAP(qb)							*
    *  									*
    *  "Map" a queue, calling the specified function for each		*
    *  element in the queue (not interlocked).  			*
    *  									*
    *  If the function returns nonzero, q_map will terminate.		*
    *  									*
    *  Input Parameters: 						*
    *      qb - queue block						*
    *      fn - function pointer					*
    *      a,b - parameters for the function				*
    *      								*
    *  Return Value:							*
    *      return value from function, or zero if entire queue 		*
    *      was mapped.							*
    ********************************************************************* */

unsigned int q_map(QBLOCK *qb,
		   unsigned int (*func)(QBLOCK *,
					unsigned int,
					unsigned int),
		   unsigned int a,unsigned int b)
{
    QBLOCK *qe;
    QBLOCK *nextq;
    unsigned int res;

    qe = qb;

    // Note:  The slightly convoluted algorithm below is to
    // take care of the case where the queue element is removed
    // while we are following the chain.  We need to remember
    // which is logically the _next_ element just in case
    // the current element is reused and recycled, so
    // we remember qe->q_next

    qe = qb->q_next;		// get first element.

    while (qe != qb) {

	nextq = qe->q_next;	// remember address of next element

	if (res = (*func)(qe,a,b)) return res;

	qe = nextq;		// now this is the next one.

	}

    return 0;
}



/*  *********************************************************************
    *  Q_COUNT(qb)							*
    *  									*
    *  Counts the elements on a queue (not interlocked)			*
    *  									*
    *  Input Parameters: 						*
    *      qb - queue block						*
    *      								*
    *  Return Value:							*
    *      number of elements						*
    ********************************************************************* */
int q_count(QBLOCK *qb)
{
    QBLOCK *qe;
    int res = 0;

    qe = qb;

    while (qe->q_next != qb) {
	qe = qe->q_next;
	res++;
	}

    return res;
}




/*  *********************************************************************
    *  q_find(qb,qe)							*
    *  									*
    *  Determines if a particular element is on a queue.                *
    *  									*
    *  Input Parameters: 						*
    *      qb - queue block						*
    *      qe - queue element                                           *
    *      								*
    *  Return Value:							*
    *      0 - not on queue                                             *
    *      >0 - position on queue                                       *
    ********************************************************************* */
int q_find(QBLOCK *qb,QBLOCK *qe)
{
    QBLOCK *q;
    int res = 1;

    q = qb->q_next;

    while (q != qe) {
	if (q == qb) return 0;          // not found on queue
	q = q->q_next;
	res++;
	}

    return res;                     // found, return element #
}


