/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB Protocol packet build routines	File: PJBPROTO.H	**
   **									**
   **  This module builds and parses PJB packets and item lists.	**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************

The fixed portion of the PJB protocol:

	+----+-----+---+------+------+----+------+------+---->
	|Vers|Class|Cmd|PktFmt|Status|XID |Block#|Click#|Data
	+----+-----+---+------+------+----+------+------+---->
          1     1    1    1      1      4     4     2

	Vers:	Protocol version number.  This must match or
		the packet will be dropped. 
	Class:  Packet "class" - Identifies major software 
		subsystem packet is destined for.
	Cmd:	Packet "command" - Identifies packet type.  
		Values depend on the "class" byte
		(the two bytes combined are the packet's type).
		High order bit of the Cmd byte is set to
		'1' for response packets, '0' for request packets.
	PktFmt: Indicates the format of the data that follows
		the fixed portion of the packet.
			0 - no data follows the fixed portion
			1 - bulk (block) data follows
			2 - Item-List (TLV-format) data follows
	Block#: Allocation unit number.  Blocks are 128K bytes.
	Click#: "Click" (sub-block) number. Clicks are 1K bytes.
        XID:    Transaction ID - filled in for request packets,
                and returned to sender on response packets
	Status: Error status (response packets only).  MBZ
		for command packets.


For most non "bulk-data" packets, an "item list" follows
the fixed header.  Item lists can be identified by the
value in the PktFmt field of the fixed header.  Item lists are
formatted as:

	+--------+--------+-------<>+--------+--------+------->
        |ItemCode|ItemLen |Value    |ItemCode|ItemLen |Value
        +--------+--------+-------<>+--------+--------+------->
            2        2      n           2       2        n

	ItemCode:  An identifier for this item. 
	ItemLen:   The length of the data that follows the item.
	Value:     Value data.


*********************************************************************/


/*  *********************************************************************
    *  Protocol type / multicast definitions				*
    ********************************************************************* */

/* Protocol version number */

#define PJB_PROTOCOL_VERSION	1

/* other constants */
#define DROP_PACKET	1
#define KEEP_PACKET	0

#define PJBPROTO_COMMAND_TIMEOUT	1000	/* msecs for command timeout */
#define PJBPROTO_POLL_INTERVAL		10	/* msecs between polls */
#define PJBPROTO_RETRY_COUNT		4	/* number of retries */
#define PJBPROTO_SOL_RETRY_COUNT	2	/* solicit retries */


/*  *********************************************************************
    *  Packet class definitions						*
    ********************************************************************* */

#define PJB_FMT_NODATA		0	/* nothing follows */
#define PJB_FMT_BULKDATA	1	/* raw data follows */
#define PJB_FMT_ITEMLIST	2	/* item list follows */
#define PJB_FMT_CONTINUE	3	/* I'm still working, keep waiting */
#define PJB_FMT_CONTINUE_ITEM	4	/* Still working, with item list*/

#define PJB_PCLASS_CONFIG	0	/* config & status */
#define PJB_PCLASS_TOC		1	/* table of contents mgmt */
#define PJB_PCLASS_STORAGE	2	/* storage mgmt */
#define PJB_PCLASS_UPGRADE	3	/* upgrade of software */
#define PJB_PCLASS_MAX		4

/*  *********************************************************************
    *  Packet types							*
    ********************************************************************* */

#define MTC_CMDFLAG		0x00	/* lack of bit indicates cmds */
#define MTR_RESPFLAG		0x01	/* this bit indicates responses */

#define DISPMAX(x)	((x+1)/2)

/* CONFIG class packets */

#define MTC_ECHO		0x00	/* echo request */
#define MTR_ECHO		0x01	/* echo reply */
#define MTC_SOLICIT		0x02	/* solicit request */
#define MTR_SOLICIT		0x03	/* solicit response */
#define MTC_GETINFO		0x04	/* get info */
#define MTR_GETINFO		0x05	/* get info response */
#define PJB_CONFIG_MAX		0x06	/* highest code */

/* TOC class packets */

#define MTC_READTOC		0x00	/* read TOC */
#define MTR_READTOC		0x01	/* read TOC response */
#define MTC_WRITETOC		0x02	/* write TOC */
#define MTR_WRITETOC		0x03	/* write TOC response */
#define MTC_COMMITTOC		0x04	/* commit TOC */
#define MTR_COMMITTOC		0x05 	/* commit TOC response */
#define PJB_TOC_MAX		0x06

/* STORAGE class packets */

#define MTC_READBLOCKHDR	0x00	/* read block header */
#define MTR_READBLOCKHDR	0x01	/* read block header response */
#define MTC_WRITEBLOCK		0x02	/* write block */
#define MTR_WRITEBLOCK		0x03	/* write block response */
#define PJB_STORAGE_MAX		0x04

/* UPGRADE class packets */

#define MTC_OPENSOFT		0x00	/* open software upgrade */
#define MTR_OPENSOFT		0x01	/* open software upgrade response */
#define MTC_WRITESOFT		0x02	/* write software */
#define MTR_WRITESOFT		0x03	/* write software response */
#define MTC_COMMITSOFT		0x04	/* commit software */
#define MTR_COMMITSOFT		0x05	/* commit software response */
#define MTC_REBOOT		0x06	/* reboot PJB */
#define MTR_REBOOT		0x07	/* reboot response */
#define PJB_UPGRADE_MAX		0x08

/* Boot methods */

#define BOOTMETHOD_MAIN		0
#define BOOTMETHOD_RECOVERY	1



/*  *********************************************************************
    *  Item data types							*
    ********************************************************************* */


#define IT_UINT		0x1000		/* int (32 bits on wire, machine
					   word size on CPU) */
#define IT_STRING	0x2000		/* ASCII string, null terminated */
#define IT_U32		0x3000		/* always 32 bits */
#define IT_U16		0x4000		/* always 16 bits */
#define IT_U8		0x5000		/* always 8 bits */
#define IT_4BREC	0x6000		/* 4 bytes */
#define IT_REC		0x7000		/* record (bytes) */

/*  *********************************************************************
    *  Item codes							*
    ********************************************************************* */

#define IT_END		(0)		/* last item */
#define IT_TYPE(item)   ((item)&0xF000) /* get item type from item */
#define IT_DBGSTR	(IT_STRING | 1)	/* debug message */
#define IT_FSVERSION	(IT_UINT  | 2)  /* filesystem version */
#define IT_CLICKSINTOC  (IT_UINT  | 3)  /* clicks in the TOC */
#define IT_TOTALBLOCKS  (IT_UINT  | 4)  /* total allocation blocks in disk */
#define IT_ERRORCOUNT   (IT_UINT  | 5)  /* cumulative errors on disk */
#define IT_VALIDTOCS    (IT_UINT  | 6)  /* number of valid TOCs */
#define IT_CURTOCCLICKS (IT_UINT  | 7)  /* # of clicks in the TOC */
#define IT_ISOLDTOC     (IT_UINT  | 8)  /* nonzero if it's an old TOC */
#define IT_HWVER        (IT_UINT  | 9)  /* hardware version */
#define IT_SWVER        (IT_UINT  | 10) /* software version */
#define IT_FRIENDLYNAME (IT_STRING | 11) /* friendly PJB name */
#define IT_FEATURES     (IT_UINT  | 12)  /* feature bits */
#define IT_CHECKSUM     (IT_4BREC | 13)  /* checksum field */
#define IT_DEBUG1	(IT_UINT  | 14)
#define IT_DEBUG2       (IT_UINT  | 15)
#define IT_DEBUG3       (IT_UINT  | 16)
#define IT_SECTOR       (IT_UINT  | 17)
#define IT_SSN		(IT_REC   | 18)
#define	IT_DIAGPROG	(IT_UINT  | 19)	/* diagnostic to run */
#define	IT_DIAGARG	(IT_UINT  | 20)	/* argument to diagnostic */


/*  *********************************************************************
    *  Structures (internal, not sent over the network)			*
    ********************************************************************* */

/* This structure is the internal representation of a packet header.
   On an X86 this could probably be sent as-is, but the DSP is
   not going to represent this in memory the same way */

#define PJBPROTO_HEADER_SIZE	15

typedef struct PjbPktHdr {
    u8	pph_version;
    u8	pph_class;
    u8	pph_cmd;
    u8	pph_fmt;
    u8	pph_status;
    u32	pph_xid;
    u32	pph_block;
    u16	pph_click;
} PJBPKTHDR;


/*  *********************************************************************
    *  Function prototypes						*
    ********************************************************************* */

int pjbproto_build_header_from_record(EBUF *buf,PJBPKTHDR *rec);
int pjbproto_build_header(EBUF *buf,u8 class,u8 cmd,u8 fmt,
			  u8 status,u32 xid,u32 block,u16 click);
int pjbproto_parse_header(EBUF *buf,u8 *class,u8 *cmd,u8 *fmt,
			  u8 *status,u32 *xid,u32 *block,u16 *click);
int pjbproto_parse_header_to_record(EBUF *buf,PJBPKTHDR *rec);
int pjbproto_get_header(EBUF *buf,PJBPKTHDR *rec);
int pjbproto_build_itemlist(EBUF *buf,...);
int pjbproto_parse_itemlist(EBUF *buf,...);


#ifdef __cplusplus
};
#endif
