/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB Protocol packet build routines	File: PJBPROTO.C	**
   **									**
   **  This module builds and parses PJB packets and item lists.	**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/


#include <stdarg.h>
#include <memory.h>
#include <string.h>
#include <sys/types.h>
#include "pjbtypes.h"
#include "queue.h"
#include "ebuf.h"
#include "pjbproto.h"

/*  *********************************************************************
    *  Externs								*
    ********************************************************************* */

extern void usb_initbuf(EBUF *);

/*  *********************************************************************
    *  PJBPROTO_BUILD_HEADER(buf,class,cmd,fmt,status,xid,block,click)	*
    *  									*
    *  This routine builds the fixed portion of a PJB packet.  We	*
    *  use this function call instead of a structure copy to work	*
    *  around the different representations of data between the		*
    *  56309 and the X86						*
    *  									*
    *  Input parameters: 						*
    *  	   buf - an EBUF structure					*
    *  	   class,cmd,fmt,status,xid,block,click - header fields		*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes copied to the structure			*
    ********************************************************************* */

int pjbproto_build_header(EBUF *buf,u8 class,u8 cmd,u8 fmt,
			  u8 status,u32 xid,u32 block,u16 click)
{

/*	+----+-----+---+------+------+----+------+------+---->
	|Vers|Class|Cmd|PktFmt|Status|XID |Block#|Click#|Data
	+----+-----+---+------+------+----+------+------+---->
          1     1    1    1      1      4     4     2
*/

    ebuf_init(buf);				/* start at beginning */
    ebuf_add_u8(buf,PJB_PROTOCOL_VERSION);	/* version field */
    ebuf_add_u8(buf,class);			/* class field */
    ebuf_add_u8(buf,cmd);			/* command field */
    ebuf_add_u8(buf,fmt);			/* packet format field */
    ebuf_add_u8(buf,status);			/* status field */
    ebuf_add_u32(buf,xid);			/* XID field */
    ebuf_add_u32(buf,block);			/* block number */
    ebuf_add_u16(buf,click);    		/* click number */

    return buf->eb_length;
}

/*  *********************************************************************
    *  PJBPROTO_BUILD_HEADER_FROM_RECORD(buf,hdr)			*
    *  									*
    *  This routine builds the fixed portion of a PJB packet.  We	*
    *  use this function call instead of a structure copy to work	*
    *  around the different representations of data between the		*
    *  56309 and the X86						*
    *  									*
    *  Input parameters: 						*
    *  	   buf - an EBUF structure					*
    *  	   hdr - record containing header fields			*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes copied to the structure			*
    ********************************************************************* */

int pjbproto_build_header_from_record(EBUF *buf,PJBPKTHDR *hdr)
{

/*	+----+-----+---+------+------+----+------+------+---->
	|Vers|Class|Cmd|PktFmt|Status|XID |Block#|Click#|Data
	+----+-----+---+------+------+----+------+------+---->
          1     1    1    1      1      4     4     2
*/

    ebuf_init(buf);				/* start at beginning */
    ebuf_add_u8(buf,PJB_PROTOCOL_VERSION);	/* version field */
    ebuf_add_u8(buf,hdr->pph_class);		/* class field */
    ebuf_add_u8(buf,hdr->pph_cmd);		/* command field */
    ebuf_add_u8(buf,hdr->pph_fmt);		/* packet format field */
    ebuf_add_u8(buf,hdr->pph_status);		/* status field */
    ebuf_add_u32(buf,hdr->pph_xid);		/* XID field */
    ebuf_add_u32(buf,hdr->pph_block);		/* block number */
    ebuf_add_u16(buf,hdr->pph_click);  		/* click number */

    return buf->eb_length;
}

/*  *********************************************************************
    *  PJBPROTO_PARSE_HEADER(buf,class,cmd,fmt,status,xid,block,click)	*
    *  									*
    *  This routine parses the fixed portion of a PJB packet, unpacking	*
    *  the data into a record.  We use this function call instead of 	*
    *  a structure copy to work	around the different representations 	*
    *  of data between the 56309 and the X86				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - an EBUF structure					*
    *  	   class,cmd,fmt,status,xid,block,click - pointers to header	*
    *            fields.  NULL indicates don't copy this field.		*
    *  	   								*
    *  Return value:							*
    *  	   >=0 = ok, <0 means the packet has problems.			*
    ********************************************************************* */

int pjbproto_parse_header(EBUF *buf,u8 *class,u8 *cmd,u8 *fmt,
			  u8 *status,u32 *xid,u32 *block,u16 *click)
{
    u8  tmp8;
    u16 tmp16;
    u32 tmp32;

/*	+----+-----+---+------+------+----+------+------+---->
	|Vers|Class|Cmd|PktFmt|Status|XID |Block#|Click#|Data
	+----+-----+---+------+------+----+------+------+---->
          1     1    1    1      1      4     4     2
*/

    /* make sure it's big enough. */
    if (buf->eb_length < PJBPROTO_HEADER_SIZE) return -1;

    ebuf_get_u8(buf,tmp8);			/* version field */
    if (tmp8  != PJB_PROTOCOL_VERSION) return -1;

    ebuf_get_u8(buf,tmp8);			/* class */
    if (class) *class = tmp8;

    ebuf_get_u8(buf,tmp8);			/* command */
    if (cmd) *cmd = tmp8;

    ebuf_get_u8(buf,tmp8);			/* format */
    if (fmt) *fmt = tmp8;

    ebuf_get_u8(buf,tmp8);			/* status */
    if (status) *status = tmp8;

    ebuf_get_u32(buf,tmp32);			/* XID */
    if (xid) *xid = tmp8;

    ebuf_get_u32(buf,tmp32);			/* Block */
    if (block) *block = tmp32;

    ebuf_get_u16(buf,tmp16);			/* Click */
    if (click) *click = tmp16;

    return 0;
}

/*  *********************************************************************
    *  PJBPROTO_PARSE_HEADER_TO_RECORD(buf,rec)				*
    *  									*
    *  This routine parses the fixed portion of a PJB packet, unpacking	*
    *  the data into a record.  We use this function call instead of 	*
    *  a structure copy to work	around the different representations 	*
    *  of data between the 56309 and the X86				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - an EBUF structure					*
    *  	   rec - a PJBPKTHDR record					*
    *  	   								*
    *  Return value:							*
    *  	   >=0 = ok, <0 means the packet has problems.			*
    ********************************************************************* */

int pjbproto_parse_header_to_record(EBUF *buf,PJBPKTHDR *rec)
{

    u8 tmp8;

/*	+----+-----+---+------+------+----+------+------+---->
	|Vers|Class|Cmd|PktFmt|Status|XID |Block#|Click#|Data
	+----+-----+---+------+------+----+------+------+---->
          1     1    1    1      1      4     4     2
*/

    /* make sure it's big enough. */
    if (buf->eb_length < PJBPROTO_HEADER_SIZE) return -1;

    ebuf_get_u8(buf,tmp8);			/* version field */
    if (tmp8  != PJB_PROTOCOL_VERSION) return -1;

    rec->pph_version = tmp8;			/* version field */
    ebuf_get_u8(buf,rec->pph_class);		/* class */
    ebuf_get_u8(buf,rec->pph_cmd);		/* command */
    ebuf_get_u8(buf,rec->pph_fmt);		/* format */
    ebuf_get_u8(buf,rec->pph_status);		/* status */
    ebuf_get_u32(buf,rec->pph_xid);		/* XID */
    ebuf_get_u32(buf,rec->pph_block);		/* Block */
    ebuf_get_u16(buf,rec->pph_click);		/* Click */

    return 0;
}

/*  *********************************************************************
    *  PJBPROTO_GET_HEADER(buf,rec)					*
    *  									*
    *  Retrieve the header information from a packet, without		*
    *  disturbing the packet's data pointer and length.  This		*
    *  is used in the receive routines to verify headres		*
    *  before passing the data to a handler.				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - ebuf							*
    *  	   rec - PJBPKTHDR structure					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   else <0 = problems with packet				*
    ********************************************************************* */

int pjbproto_get_header(EBUF *buf,PJBPKTHDR *rec)
{
    u8 *save_ptr;	
    int save_len;
    int res;

    save_ptr = buf->eb_ptr;
    save_len = buf->eb_length;

    res = pjbproto_parse_header_to_record(buf,rec);

    buf->eb_ptr = save_ptr;
    buf->eb_length = save_len;

    return res;    
}


/*  *********************************************************************
    *  PJBPROTO_BUILD_ITEMLIST(buf,...)					*
    *  									*
    *  Builds an item list in the specified buffer.  the arg list	*
    *  is a variable-argument list, in the form of			*
    *  itemcode,value,itemcode,value,...,0				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - EBUF structure						*
    *  	   ... - items to add						*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes added					*
    ********************************************************************* */

int pjbproto_build_itemlist(EBUF *buf,...)
{
    va_list marker;
    u16 itemcode;
    unsigned int valint;
    u32 vallong;
    u8 *valstr;
    int oldlen;

    oldlen = buf->eb_length;

    va_start(marker,buf);

/*
	+--------+--------+-------<>+--------+--------+------->
        |ItemCode|ItemLen |Value    |ItemCode|ItemLen |Value
        +--------+--------+-------<>+--------+--------+------->
            2        2      n           2       2        n

	    Ints are represented as 4 bytes on the wire,
	    even though they may be only 24 bits on the CPU.

	    Strings are stored with their null terminators.
*/

    while (itemcode = va_arg(marker,u16)) {
	switch (IT_TYPE(itemcode)) {
	    case IT_UINT:
		ebuf_add_u16(buf,itemcode);
		ebuf_add_u16(buf,4);
		valint = va_arg(marker,unsigned int);
		ebuf_add_u32(buf,valint);
		break;
	    case IT_U32:
		ebuf_add_u16(buf,itemcode);
		ebuf_add_u16(buf,4);
		vallong = va_arg(marker,u32);
		ebuf_add_u32(buf,vallong);
		break;
	    case IT_STRING:
		ebuf_add_u16(buf,itemcode);
		valstr = va_arg(marker,u8 *);
		valint = strlen(valstr) + 1;
		ebuf_add_u16(buf,(u16)valint);
		while (*valstr) {
		    ebuf_add_u8(buf,*valstr);
		    valstr++;
		    }
		ebuf_add_u8(buf,0);
		break;
	    case IT_4BREC:
		ebuf_add_u16(buf,itemcode);
		valstr = va_arg(marker,u8 *);
		ebuf_add_u16(buf,4);
		ebuf_add_bytes(buf,valstr,4);
		break;
	    case IT_REC:
		ebuf_add_u16(buf,itemcode);
		valint = va_arg(marker,int);
		valstr = va_arg(marker,u8 *);
		ebuf_add_u16(buf,(u16)valint);
		ebuf_add_bytes(buf,valstr,valint);
		break;
	    }
	}

    va_end(marker);

    /* return how many bytes we added */

    return buf->eb_length - oldlen;

}


/*  *********************************************************************
    *  PJBPROTO_FIND_ITEM(buf,code)					*
    *  									*
    *  Locate an item in an itemlist.					*
    *  									*
    *  This routine is used when parsing inbound packets.  We 		*
    *  dig out interesting items and return a pointer within the	*
    *  packet.								*
    *  									*
    *  Input parameters: 						*
    *  	   buf - ebuf structure						*
    *  	   code - code to find						*
    *      len - where to put length of found item			*
    *  	   								*
    *  Return value:							*
    *  	   void pointer to item data, or NULL if not found		*
    ********************************************************************* */

static u8 *pjbproto_find_item(EBUF *buf,u16 itemcode,int *len)
{
    u8 *save_ptr;
    int save_len;
    u16 thisitem;
    u16 thislen;
    u8 *ret_ptr = NULL;

    save_ptr = buf->eb_ptr;
    save_len = buf->eb_length;

    while (buf->eb_length > 0) {
	ebuf_get_u16(buf,thisitem);
	if (thisitem == IT_END) break;
	if (thisitem == itemcode) break;
	ebuf_get_u16(buf,thislen);
	buf->eb_ptr += thislen;
	buf->eb_length -= thislen;
	}

    ret_ptr = buf->eb_ptr;

    buf->eb_ptr = save_ptr;
    buf->eb_length = save_len;

    if (thisitem == IT_END) {
	return NULL;
	}

    *len = (*ret_ptr + (*(ret_ptr+1) << 8));

    return ret_ptr + 2;		/* skip length */
}

/*  *********************************************************************
    *  PJBPROTO_PARSE_ITEMLIST(buf,...)					*
    *  									*
    *  Parse an item list.  The input parameter is a variable-length	*
    *  arglist, in the form itemcode,ptr,itemcode,ptr,...,0  where	*
    *  ptr is a pointer to where to put the item's data.  For		*
    *  strings and records, ptr is actually a pointer to the pointer	*
    *  to the data (the data is not copied).				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - ebuf							*
    *  	   ... - list of fields to select				*
    *  	   								*
    *  Return value:							*
    *  	   number of fields extracted					*
    ********************************************************************* */

int pjbproto_parse_itemlist(EBUF *buf,...)
{
    va_list marker;
    void *itemdest;
    u16 itemcode;
    u8 *itemval;
    int count = 0;
    int itemlen;

    va_start(marker,buf);

    for (;;) {
	itemcode = va_arg(marker,u16);
	if (itemcode == 0) break;
	itemdest = va_arg(marker,void *);
	itemval = pjbproto_find_item(buf,itemcode,&itemlen);
	if (itemval == NULL) continue;
	switch (IT_TYPE(itemcode)) {
	    case IT_UINT:
		/* only do 24 bits. */
		*((unsigned int *) itemdest) = 
		    ((unsigned int) itemval[0]) |
		    (((unsigned int) itemval[1]) << 8) |
		    (((unsigned int) itemval[2]) << 16);
		count++;
		break;
	    case IT_U32:
		/* do all 32 bits. */
		*((u32 *) itemdest) = 
		    ((u32) itemval[0]) |
		    (((u32) itemval[1]) << 8) |
		    (((u32) itemval[2]) << 16) |
		    (((u32) itemval[3]) << 24);
		count++;
		break;
	    case IT_STRING:
		*((u8 **) itemdest) = itemval;
		count++;
		break;
	    case IT_4BREC:
		memcpy(itemdest,itemval,4);
		count++;
		break;
	    case IT_REC:
		*((void **) itemdest) = itemval;
		count++;
		break;
	    }
	}

    va_end(marker);
    return count;
}
