/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB Low-level communications (WIN32)	File: PJBDEV_WIN32.C	**
   **									**
   **  This module communicates with the USB device driver.  This 	**
   **  particular version is for WIN32 (Windows 98, Windows 2000)	**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <setupapi.h>
#include <basetyps.h>

#include "winioctl.h"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <errno.h>
#include <time.h>

#include "cpqpjb.h"
#include "guidpjb.h"
#include "usbdi.h"

#include "pjbtypes.h"
#include "pjbapi.h"


/*  *********************************************************************
    *  Constants							*
    ********************************************************************* */

#define MAX_PJB_DEVICES	16

#define _DUMPUSB_

/*  *********************************************************************
    *  Types								*
    ********************************************************************* */

typedef struct PjbCommHandle {
    HANDLE outHandle; 
    HANDLE inHandle;
    HANDLE rx_event;
    int isopen;
    int alternate;
    char devfilename[256];
} PJBCOMMHANDLE;

#define _PJBCOMMHANDLE_
#include "pjbdev.h"

typedef struct EnumInfo {
    PJB_INFO *ids;
    int maxids;
    int idcnt;
} ENUMINFO;

typedef struct OpenInfo {
    u8 *openssn;
    PJBCOMMHANDLE *devinfo;
} OPENINFO;


/*  *********************************************************************
    *  Pointers to routines in SETUPAPI.DLL				*
    ********************************************************************* */


HDEVINFO  (WINAPI * pSetupDiGetClassDevs)(
    IN LPGUID ClassGuid,  OPTIONAL
    IN PCSTR  Enumerator, OPTIONAL
    IN HWND   hwndParent, OPTIONAL
    IN DWORD  Flags
    ) = NULL;


BOOL (WINAPI * pSetupDiEnumDeviceInterfaces)(
    IN  HDEVINFO                  DeviceInfoSet,
    IN  PSP_DEVINFO_DATA          DeviceInfoData,     
    IN  LPGUID                    InterfaceClassGuid,
    IN  DWORD                     MemberIndex,
    OUT PSP_DEVICE_INTERFACE_DATA DeviceInterfaceData
    ) = NULL;


BOOL (WINAPI * pSetupDiGetDeviceInterfaceDetail)(
    IN  HDEVINFO                           DeviceInfoSet,
    IN  PSP_DEVICE_INTERFACE_DATA          DeviceInterfaceData,
    OUT PSP_DEVICE_INTERFACE_DETAIL_DATA_A DeviceInterfaceDetailData,     
    IN  DWORD                              DeviceInterfaceDetailDataSize,
    OUT PDWORD                             RequiredSize,                  
    OUT PSP_DEVINFO_DATA                   DeviceInfoData                 
    ) = NULL;

BOOL (WINAPI * pSetupDiDestroyDeviceInfoList)(
    IN HDEVINFO DeviceInfoSet
    ) = NULL;


/*  *********************************************************************
    *  Pointers to routines in KERNEL32.DLL				*
    ********************************************************************* */

BOOL (WINAPI *pCancelIo)(HANDLE) = NULL;

HINSTANCE hKernel32 = NULL;

/*  *********************************************************************
    *  Globals								*
    ********************************************************************* */

static PJBCOMMHANDLE *pjbdev_open(PJB_ID *id);
static void pjbdev_close(PJBCOMMHANDLE *);
static int pjbdev_enum(PJB_INFO *ids,int maxids);
static int pjbdev_rx(PJBCOMMHANDLE *info,unsigned char *buf,int len,int timeout);
static int pjbdev_tx(PJBCOMMHANDLE *info,unsigned char *buf,int len);
static PJBCOMMHANDLE *pjbdev_opendevice(char *realname);
static void pjbdev_closedevice(PJBCOMMHANDLE *pch);
static long pjbdev_getticks(void);
static void pjbdev_sleep(int sec);
static int pjbdev_usbresetpipe(HANDLE h);

PJBCOMMDEV pjbdev_win32usb = {
    pjbdev_open,
    pjbdev_close,
    pjbdev_enum,
    pjbdev_rx,
    pjbdev_tx,
    pjbdev_getticks,
    pjbdev_sleep,
    0};

PJBCOMMDEV *pjbdev = &pjbdev_win32usb;

static char *pjbdev_devname = "/dev/cpqpjb";

static char inPipe[32]  = "2";    // Input pipe name for PJB
static char outPipe[32] = "3";    // Output pipe name for PJB

static char inPipeAlt[32] = "0";	// input pipe name for debug
static char outPipeAlt[32] = "1";	// output pipe name for debug

static HINSTANCE hSetupApi = NULL;

extern int PJB_GetInfo1(PJBCOMMHANDLE *pjbdev,PJB_INFO *info);


/*  *********************************************************************
    *  PjbUsbUninitSetupLibrary()					*
    *  									*
    *  Disconnect from setupapi.dll					*
    *  									*
    *  Input parameters: 						*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void PjbUsbUninitSetupLibrary(void)
{
    if (hSetupApi) {
	FreeLibrary(hSetupApi);
	hSetupApi = NULL;

	pSetupDiGetClassDevs = NULL;
	pSetupDiEnumDeviceInterfaces = NULL;
	pSetupDiGetDeviceInterfaceDetail = NULL;
	pSetupDiDestroyDeviceInfoList = NULL;
	}
}


/*  *********************************************************************
    *  PjbUsbInitSetupLibrary()						*
    *  									*
    *  Load/open the SETUPAPI.DLL library and obtain the 		*
    *  entry points we need.						*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - library loaded OK					*
    *  	   FALSE - problems, entry points not available			*
    ********************************************************************* */

static BOOL PjbUsbInitSetupLibrary(void)
{

    if (hSetupApi) return TRUE;

    hSetupApi = LoadLibrary("SETUPAPI.DLL");
    if (hSetupApi == NULL) return FALSE;

    (PVOID) pSetupDiGetClassDevs = 
	(PVOID) GetProcAddress(hSetupApi, "SetupDiGetClassDevsA");

    (PVOID) pSetupDiEnumDeviceInterfaces = 
	(PVOID) GetProcAddress(hSetupApi, "SetupDiEnumDeviceInterfaces");

    (PVOID) pSetupDiGetDeviceInterfaceDetail = 
	(PVOID) GetProcAddress(hSetupApi, "SetupDiGetDeviceInterfaceDetailA");

    (PVOID) pSetupDiDestroyDeviceInfoList = 
	(PVOID) GetProcAddress(hSetupApi, "SetupDiDestroyDeviceInfoList");



    if (!pSetupDiGetClassDevs ||
	!pSetupDiEnumDeviceInterfaces ||
	!pSetupDiGetDeviceInterfaceDetail ||
	!pSetupDiDestroyDeviceInfoList) {
	PjbUsbUninitSetupLibrary();
	return FALSE;
	}

    return TRUE;
}

/*  *********************************************************************
    *  PjbUsbInitKernelCalls()						*
    *  									*
    *  Load/open the KERNEL32.DLL library and obtain the 		*
    *  entry points we need.						*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - library loaded OK					*
    *  	   FALSE - problems, entry points not available			*
    ********************************************************************* */

static BOOL PjbUsbInitKernelCalls()
{

    if (hKernel32) return TRUE;

    hKernel32 = LoadLibrary("KERNEL32.DLL");
    if (hKernel32 == NULL) return FALSE;

    (PVOID) pCancelIo =
	(PVOID) GetProcAddress(hKernel32, "CancelIo");


    return TRUE;
}


/*  *********************************************************************
    *  pjbdev_enumfunc(devname,arg)					*
    *  									*
    *  Callback for the usb enumerator function				*
    *  									*
    *  Input parameters: 						*
    *  	   devname - a device we found					*
    *  	   arg - reference data						*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep going, 1 to stop					*
    ********************************************************************* */

static int pjbdev_enumfunc(char *devname,void *arg)
{
    ENUMINFO *enuminfo = (ENUMINFO *) arg;
    PJBCOMMHANDLE *devinfo;
    int res;

    if (enuminfo->idcnt == enuminfo->maxids) return TRUE;

    devinfo = pjbdev_opendevice(devname);
    if (devinfo == NULL) {
        if (pjbdev->pcd_debug) printf("[Could not open %s]\n",devname);
	return FALSE;		/* could not open */
	}

    /* Obtain serial number from device */

    res = PJB_GetInfo1(devinfo,enuminfo->ids + enuminfo->idcnt);

    if (res == 0) {
	memcpy((enuminfo->ids+enuminfo->idcnt)->id.b,
	       (enuminfo->ids+enuminfo->idcnt)->ssn,
	       6);
	}

    /* add one more to the count */

    pjbdev_closedevice(devinfo);

    if (res == 0) {
	enuminfo->idcnt++;
	}

    return FALSE;
}



/*  *********************************************************************
    *  PJBDEV_OPENFUNC(devname,arg)					*
    *  									*
    *  This is called by the enumerator when we are opening		*
    *  the device driver.  If we can open this driver, this		*
    *  routine checks to be sure the serial # matches.			*
    *  									*
    *  Input parameters: 						*
    *  	   devname - actual ASCII device name				*
    *  	   arg - enum instance data (an OPENINFO structure)		*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep enumerating					*
    *  	   !=0 to stop							*
    ********************************************************************* */

static int pjbdev_openfunc(char *devname,void *arg)
{
    OPENINFO *openinfo = (OPENINFO *) arg;
    PJB_INFO pjbinfo;
    int res;

    if (openinfo->devinfo) return FALSE;

    openinfo->devinfo = pjbdev_opendevice(devname);
    if (openinfo->devinfo == NULL) return FALSE;		/* could not open */

    /* Obtain serial number from device */

    res = PJB_GetInfo1(openinfo->devinfo,&pjbinfo);

    if (res != 0) {
	pjbdev_closedevice(openinfo->devinfo);
	openinfo->devinfo = NULL;
	return FALSE;
        }

    /* add one more to the count */

    if (memcmp(pjbinfo.ssn,openinfo->openssn,6) == 0) {
	return TRUE;
	}

    pjbdev_closedevice(openinfo->devinfo);
    openinfo->devinfo = NULL;
		       
    return FALSE;
}


/*  *********************************************************************
    *  PJBDEV_GETTICKS()						*
    *  									*
    *  Return system time in ticks					*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   ticks							*
    ********************************************************************* */

static long pjbdev_getticks()
{
    return (long) GetTickCount();
}

/*  *********************************************************************
    *  PJBDEV_SLEEP(sec)						*
    *  									*
    *  Sleep for 'sec' seconds						*
    *  									*
    *  Input parameters: 						*
    *  	   sec - # of seconds to sleep					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_sleep(int sec)
{
    Sleep(sec*1000);
}



/*  *********************************************************************
    *  EnumUsbDevices(guid,enunfunc,arg)				*
    *  									*
    *  Talk to the PNP manager to get the device name for the		*
    *  connected PJBs in the system					*
    *  									*
    *  Input parameters: 						*
    *  	   pGuid - pointer to a GUID registered by the driver		*
    *  	   enumfunc - callback for each device found			*
    *	   arg - arg to pass to callback				*
    *  	   								*
    *  Return value:							*
    *  	   Number of devices returned					*
    ********************************************************************* */
static int EnumUsbDevices( LPGUID  pGuid, 
			   int (*enumfunc)(char *,void *),
			   void *arg)
{
    ULONG NumberDevices;
    HDEVINFO                 hardwareDeviceInfo;
    SP_INTERFACE_DEVICE_DATA deviceInfoData;
    ULONG                    enumidx;

    //
    // Load library if we need to 
    //

    if (PjbUsbInitSetupLibrary() == FALSE) return 0;	// no USB

    //
    // Open a handle to the plug and play dev node.
    // SetupDiGetClassDevs() returns a device information
    // set that contains info on all 
    // installed devices of a specified class.
    //

    hardwareDeviceInfo = (*pSetupDiGetClassDevs) (
	pGuid,
	NULL, // Define no enumerator (global)
	NULL, // Define no
	(DIGCF_PRESENT | // Only Devices present
	 DIGCF_INTERFACEDEVICE)); // Function class devices.

    //
    // Take a wild guess at the number of devices we have;
    // Be prepared to realloc and retry if there are more than we guessed
    //

    NumberDevices = 0;

    enumidx = 0;
    for (;;) {

	// SetupDiEnumDeviceInterfaces() returns information 
	// about device interfaces exposed by one or more devices. 
	// Each call returns information about one interface;
	// the routine can be called repeatedly to get information 
	// about several interfaces exposed by one or more devices.

	deviceInfoData.cbSize = sizeof (SP_INTERFACE_DEVICE_DATA);
	if ((*pSetupDiEnumDeviceInterfaces) (hardwareDeviceInfo,
					 0, // We don't care about specific PDOs
					 pGuid,
					 enumidx,
					 &deviceInfoData)) {

	    PSP_INTERFACE_DEVICE_DETAIL_DATA     functionClassDeviceData = NULL;
	    ULONG                                predictedLength = 0;
	    ULONG                                requiredLength = 0;
	    HANDLE     				 hOut = INVALID_HANDLE_VALUE;
	    ULONG				 done;


	    //
	    // allocate a function class device data structure to receive the
	    // goods about this particular device.
	    //
	    (*pSetupDiGetDeviceInterfaceDetail) (hardwareDeviceInfo,
					     &deviceInfoData,
					     NULL, 		
					     0, 		
					     &requiredLength,
					     NULL); 		


	    predictedLength = requiredLength;

	    functionClassDeviceData = GlobalAlloc(GMEM_FIXED,predictedLength);
	    functionClassDeviceData->cbSize = sizeof (SP_INTERFACE_DEVICE_DETAIL_DATA);

	    //
	    // Retrieve the information from Plug and Play.
	    //

	    if (! (*pSetupDiGetDeviceInterfaceDetail) (hardwareDeviceInfo,
						   &deviceInfoData,
						   functionClassDeviceData,
						   predictedLength,
						   &requiredLength,
						   NULL)) {
		GlobalFree(functionClassDeviceData);
		break;
		}


	    NumberDevices++;
	    done = (*enumfunc)(functionClassDeviceData->DevicePath,arg);

	    GlobalFree(functionClassDeviceData);

	    if (done) break;

	    }
	else {
	    break;
	    }

	enumidx++;

	}


    // SetupDiDestroyDeviceInfoList() destroys a device information set 
    // and frees all associated memory.

    (*pSetupDiDestroyDeviceInfoList) (hardwareDeviceInfo);

    //
    // Don't need the DLL anymore
    //

    PjbUsbUninitSetupLibrary();

    return NumberDevices;
}


/*  *********************************************************************
    *  pjbdev_enum(ids,maxids)						*
    *  									*
    *  Find all the PJBs connected to the USB bus			*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *      number of PJBs we found					*
    ********************************************************************* */
static int pjbdev_enum(PJB_INFO *ids,int maxids)
{
    ENUMINFO enuminfo;
    int cnt;

    memset(&enuminfo,0,sizeof(enuminfo));
    enuminfo.ids = ids;
    enuminfo.idcnt = 0;
    enuminfo.maxids = maxids;

    /* Under linux, we only support one at a time. */

    if (pjbdev->pcd_debug) printf("[enumerating devices]\n");


    cnt = EnumUsbDevices((LPGUID) &GUID_CLASS_CPQPJB,
			 pjbdev_enumfunc,&enuminfo);

    return enuminfo.idcnt;
}


PJBCOMMHANDLE *pjbdev_open(PJB_ID *id)
{
    OPENINFO openinfo;

    openinfo.openssn = id->b;
    openinfo.devinfo = NULL;

    EnumUsbDevices((LPGUID) &GUID_CLASS_CPQPJB,
			 pjbdev_openfunc,&openinfo);

    return openinfo.devinfo;
}

void pjbdev_close(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice(pch);
}



/*  *********************************************************************
    *  pjbdev_opendevice1(info)						*
    *  									*
    *  This routine is the meat of opening a USB connection - 		*
    *  given the info in the handle structure, open the connection.	*
    *  									*
    *  Input parameters: 						*
    *  	   info - USB handle info					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - open ok						*
    *  	   FALSE - could not open					*
    ********************************************************************* */

int pjbdev_opendevice1(PJBCOMMHANDLE *pch)
{
    char completeDeviceName[256] = "";  //generated from the GUID 

    sprintf(completeDeviceName,"%s\\%s",pch->devfilename,
	    pch->alternate ? inPipeAlt : inPipe);

    pch->inHandle = CreateFile(completeDeviceName,
				GENERIC_WRITE | GENERIC_READ,
				FILE_SHARE_WRITE | FILE_SHARE_READ,
				NULL,
				OPEN_EXISTING,
				FILE_FLAG_OVERLAPPED,
				NULL);


    if (pch->inHandle == INVALID_HANDLE_VALUE) {
	pch->inHandle = INVALID_HANDLE_VALUE;
	pch->outHandle = INVALID_HANDLE_VALUE;
	return FALSE;
	}


    sprintf(completeDeviceName,"%s\\%s",pch->devfilename,
	    pch->alternate ? outPipeAlt : outPipe);

    pch->outHandle = CreateFile(completeDeviceName,
				GENERIC_WRITE | GENERIC_READ,
				FILE_SHARE_WRITE | FILE_SHARE_READ,
				NULL,
				OPEN_EXISTING,
				0,
				NULL);

    if (pch->outHandle == INVALID_HANDLE_VALUE) {
	CloseHandle(pch->inHandle);
	pch->inHandle = INVALID_HANDLE_VALUE;
	pch->outHandle = INVALID_HANDLE_VALUE;
	return FALSE;
	}

    pch->isopen = TRUE;

    pch->rx_event = CreateEvent(NULL,TRUE,FALSE,NULL);

    return TRUE;
}



/*  *********************************************************************
    *  pjbdev_opendevice(info)						*
    *  									*
    *  Opens both the read and write pipes of the PJB device		*
    *  									*
    *  Input parameters: 						*
    *  	   info - pointer to info struct				*
    *  	   								*
    *  Return value:							*
    *  	   TRUE if devices opened successfully, FALSE otherwise		*
    ********************************************************************* */

static PJBCOMMHANDLE *pjbdev_opendevice(char *realname)
{ 
    PJBCOMMHANDLE *pch;

    pch = calloc(1,sizeof(PJBCOMMHANDLE));

    if (!pch) return NULL;

    pch->inHandle = INVALID_HANDLE_VALUE;
    pch->outHandle = INVALID_HANDLE_VALUE;

    strcpy(pch->devfilename,realname);
    pch->alternate = 0;

    if (pjbdev_opendevice1(pch)) {
	return pch;
        }
    else {
	free(pch);
	return NULL;
        }
}

/*  *********************************************************************
    *  pjbdev_closedevice(info)						*
    *  									*
    *  Closes both handles to the PJB device				*
    *  									*
    *  Input parameters: 						*
    *  	   info - information structure					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_closedevice1(PJBCOMMHANDLE *pch)
{
    if (pch->inHandle != INVALID_HANDLE_VALUE) CloseHandle(pch->inHandle);
    if (pch->outHandle != INVALID_HANDLE_VALUE) CloseHandle(pch->outHandle);
    if (pch->rx_event != INVALID_HANDLE_VALUE) CloseHandle(pch->rx_event);

    pch->inHandle = INVALID_HANDLE_VALUE;
    pch->outHandle = INVALID_HANDLE_VALUE;
    pch->rx_event = INVALID_HANDLE_VALUE;
}


/*  *********************************************************************
    *  pjbdev_closedevice(pch)						*
    *  									*
    *  Close the device and free the comm handle object			*
    *  									*
    *  Input parameters: 						*
    *  	   pch								*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_closedevice(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice1(pch);
    free(pch);
}


/*  *********************************************************************
    *  pjbdev_reopendevice(info)					*
    *  									*
    *  Close, then reopen file handles associated with this		*
    *  device.  This is used to reestablish broken pipes		*
    *  									*
    *  Input parameters: 						*
    *  	   info - handle info structure					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - reopened successfully					*
    *  	   FALSE - could not open					*
    ********************************************************************* */
int pjbdev_reopendevice(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice1(pch);

    if (pjbdev_opendevice1(pch)) {
	pjbdev_usbresetpipe(pch->inHandle);
	pjbdev_usbresetpipe(pch->outHandle);
	return TRUE;
	}
    return FALSE;
}




#ifdef _DUMPUSB_
/*  *********************************************************************
    *  dumpdata(str,buffer,buflen)					*
    *  									*
    *  dump data to the screen in hex format.				*
    *  									*
    *  Input Parameters: 						*
    *      str - file for output					*
    *      buffer - pointer to buffer					*
    *      buflen - length of buffer					*
    *      								*
    *  Return Value:							*
    *      nothing.							*
    ********************************************************************* */

static void dumpdata(str,buffer,buflen)
FILE *str;
unsigned char *buffer;
int buflen;
{
	int i,j;

	for (i=0; i<buflen; i+=16) {
		fprintf(str,"%04X: ",i);
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str,"   ");
			else fprintf(str,"%02X ",buffer[i+j]);
			}
		fprintf(str," | ");
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str," ");
			else fprintf(str,"%c",isprint(buffer[i+j]) ? buffer[i+j] : ' ');
			}
		fprintf(str,"\n");
		}
}
#endif


/*  *********************************************************************
    *  pjbdev_usbresetpipe(h)						*
    *  									*
    *  Reset the specified USB pipe					*
    *  									*
    *  Input parameters: 						*
    *  	   h - file handle						*
    *  	   								*
    *  Return value:							*
    *  	   TRUE if pipe was reset					*
    ********************************************************************* */

static int pjbdev_usbresetpipe(HANDLE h)
{
    DWORD returned = 0;

    if (DeviceIoControl(h,
			IOCTL_CPQPJB_RESET_PIPE,
			NULL,
			0,
			NULL,
			0,
			&returned,
			NULL) == 0) {
	if (pjbdev->pcd_debug) {
	    printf("Could not reset pipe\n");
	    OutputDebugString("Could not reset pipe\n");
	    }
	return FALSE;
	}

    return TRUE;
}

/*  *********************************************************************
    *  pjbdev_tx(PJBCOMMHANDLE *pjbdev,unsigned char *buf,int len)      *
    *  									*
    *  Transmit data to a given usb "destination" 			*
    *  									*
    *  Input parameters: 						*
    *  	   buf - buffer to send						*
    *  	   len - length of buffer to send				*
    *  	   dest - destination address					*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes written					*
    *  	   0 for a broken pipe						*
    *  	   <0 for errors						*
    ********************************************************************* */

static int pjbdev_tx(PJBCOMMHANDLE *pch,unsigned char *buf,int len)
{
    int res;
    DWORD numwritten;

    //
    // It's an error if the driver is not open
    //

    if (pch->isopen == FALSE) {
	return PJBCOMM_DEVERR;
	}

    //
    // If the length is an exact multiple of 64 bytes,
    // add one byte.  It's guaranteed to be there
    // in the ebuf, so this should be ok.
    //
    // The driver in the PJB needs to see a short
    // packet to show EOF.	
    //

    if ((len % 64) == 0) len++;

    //
    // Write the data to the device.
    //

#ifdef _DUMPUSB_
    if (pjbdev->pcd_debug) {
	printf("TransmitData--------------------\n");
	dumpdata(stdout,buf,len);
	}
#endif

    res = WriteFile(pch->outHandle,
		  buf,
		  len,	
		  &numwritten,
		  NULL);

    if (res == TRUE) res = (int) numwritten; else res = -1;

    //
    // If we got an error or wrote zero bytes, the USB connection is gone.
    // Try to reestablish it.
    //

    if (res != len) {
	if (pjbdev->pcd_debug) {
	    printf("[Reestablishing USB link]\n");
   	    }

	if (pjbdev_reopendevice(pch) == FALSE) return PJBCOMM_DEVERR;

	res = WriteFile(pch->outHandle,
			buf,
			len,	
			&numwritten,
			NULL);

	if (res == TRUE) res = (int) numwritten; else res = -1;

	if (res != len) {
	    return PJBCOMM_DEVERR;
	    }

	if (pjbdev->pcd_debug) {
	    printf("[USB link reestablished]\n");
	    }
	}

    return TRUE;
}


/*  *********************************************************************
    *  pjbdev_rx(pjbdev,*buf,len,timeout)                               *
    *  									*
    *  Receive data from the USB device					*
    *  									*
    *  Input parameters: 						*
    *  	   buffer - receive data goes here				*
    *  	   len - length of receive buffer				*
    *  	   timeout - milliseconds to wait for a timeout			*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes received					*
    *  	   <0 for error (including timeouts)				*
    ********************************************************************* */

static int pjbdev_rx(PJBCOMMHANDLE *pch,unsigned char *buf,int len,int timeout)
{
    DWORD numread;
    OVERLAPPED over;
    int res;
    int winerr;
    ULONG tmpres;

    if (pch->isopen == FALSE) {
	return PJBCOMM_DEVERR;
	}

    over.Internal = 0;
    over.InternalHigh = 0;
    over.Offset = 0;
    over.OffsetHigh = 0;
    over.hEvent = pch->rx_event;

    ResetEvent(over.hEvent);

    //
    // Attempt to read the file.  Use an OVERLAPPED structure
    // to let us do this asynchronously.
    //

    numread = 0;

    res = ReadFile(pch->inHandle,
		   buf,
		   len,	
		   &numread,
		   &over);

    if (res == 0) {

	winerr = GetLastError();

	switch (winerr) {

	    case ERROR_IO_PENDING:

		res = WaitForSingleObject(over.hEvent,(DWORD) timeout);

		switch (res) {
		    case WAIT_TIMEOUT:
			if (pCancelIo) {
			    (*pCancelIo)(pch->inHandle);
			    }
			return PJBCOMM_TIMEOUT;

		    case WAIT_OBJECT_0:

			tmpres = over.InternalHigh;
			res = GetOverlappedResult(pch->inHandle,
						  &over,
						  &numread,
						  FALSE);
			if (res == FALSE) {
			    //
			    // For some reason, GetOverlappedResult() still
			    // returns "in progress" even though it
			    // should not.  Fortunately, InternalHigh
			    // is the correct length here.
			    // This smells like a Windows bug.
			    //
			    numread = tmpres;
			    }
			break;

		    default:
			break;
		    }
		break;

	    default:
		pjbdev_reopendevice(pch);
		break;
	    }
	}

#ifdef _DUMPUSB_
    if (pjbdev->pcd_debug) {
	printf("ReceiveData--------------------\n");
	dumpdata(stdout,buf,numread);
	}
#endif

    return (int) numread;
}
