/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB Low-level communications (linux)	File: PJBDEV_LINUX.C	**
   **									**
   **  This module communicates with the USB device driver.  This 	**
   **  particular version is for Linux					**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/time.h>

#include "pjbtypes.h"
#include "pjbapi.h"


/*  *********************************************************************
    *  Constants							*
    ********************************************************************* */

#define MAX_PJB_DEVICES	16

#define _DUMPUSB_

/*  *********************************************************************
    *  Types								*
    ********************************************************************* */

typedef struct PjbCommHandle {
    int handle; 
    int isopen;
    int alternate;
    char devfilename[256];
} PJBCOMMHANDLE;

#define _PJBCOMMHANDLE_
#include "pjbdev.h"

typedef struct EnumInfo {
    PJB_INFO *ids;
    int maxids;
    int idcnt;
} ENUMINFO;

typedef struct OpenInfo {
    u8 *openssn;
    PJBCOMMHANDLE *devinfo;
} OPENINFO;


/*  *********************************************************************
    *  Globals								*
    ********************************************************************* */

static PJBCOMMHANDLE *pjbdev_open(PJB_ID *id);
static void pjbdev_close(PJBCOMMHANDLE *);
static int pjbdev_enum(PJB_INFO *ids,int maxids);
static int pjbdev_rx(PJBCOMMHANDLE *info,unsigned char *buf,int len,int timeout);
static int pjbdev_tx(PJBCOMMHANDLE *info,unsigned char *buf,int len);
static PJBCOMMHANDLE *pjbdev_opendevice(char *realname);
static void pjbdev_closedevice(PJBCOMMHANDLE *pch);
static long pjbdev_getticks(void);
static void pjbdev_sleep(int sec);

PJBCOMMDEV pjbdev_linuxusb = {
    pjbdev_open,
    pjbdev_close,
    pjbdev_enum,
    pjbdev_rx,
    pjbdev_tx,
    pjbdev_getticks,
    pjbdev_sleep,
    0};

PJBCOMMDEV *pjbdev = &pjbdev_linuxusb;

static char *pjbdev_devname = "/dev/cpqpjb";

extern int PJB_GetInfo1(PJBCOMMHANDLE *pjbdev,PJB_INFO *info);


/*  *********************************************************************
    *  pjbdev_enumfunc(devname,arg)					*
    *  									*
    *  Callback for the usb enumerator function				*
    *  									*
    *  Input parameters: 						*
    *  	   devname - a device we found					*
    *  	   arg - reference data						*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep going, 1 to stop					*
    ********************************************************************* */

static int pjbdev_enumfunc(char *devname,void *arg)
{
    ENUMINFO *enuminfo = (ENUMINFO *) arg;
    PJBCOMMHANDLE *devinfo;
    int res;

    if (enuminfo->idcnt == enuminfo->maxids) return TRUE;

    devinfo = pjbdev_opendevice(devname);
    if (devinfo == NULL) {
        if (pjbdev->pcd_debug) printf("[Could not open %s]\n",devname);
	return FALSE;		/* could not open */
	}

    /* Obtain serial number from device */

    res = PJB_GetInfo1(devinfo,enuminfo->ids + enuminfo->idcnt);

    if (res == 0) {
	memcpy((enuminfo->ids+enuminfo->idcnt)->id.b,
	       (enuminfo->ids+enuminfo->idcnt)->ssn,
	       6);
	}

    /* add one more to the count */

    pjbdev_closedevice(devinfo);

    if (res == 0) {
	enuminfo->idcnt++;
	}

    return FALSE;
}



/*  *********************************************************************
    *  PJBDEV_OPENFUNC(devname,arg)					*
    *  									*
    *  This is called by the enumerator when we are opening		*
    *  the device driver.  If we can open this driver, this		*
    *  routine checks to be sure the serial # matches.			*
    *  									*
    *  Input parameters: 						*
    *  	   devname - actual ASCII device name				*
    *  	   arg - enum instance data (an OPENINFO structure)		*
    *  	   								*
    *  Return value:							*
    *  	   0 to keep enumerating					*
    *  	   !=0 to stop							*
    ********************************************************************* */
static int pjbdev_openfunc(char *devname,void *arg)
{
    OPENINFO *openinfo = (OPENINFO *) arg;
    PJB_INFO pjbinfo;
    int res;

    if (openinfo->devinfo) return FALSE;

    openinfo->devinfo = pjbdev_opendevice(devname);
    if (openinfo->devinfo == NULL) return FALSE;		/* could not open */

    /* Obtain serial number from device */

    res = PJB_GetInfo1(openinfo->devinfo,&pjbinfo);

    if (res != 0) {
	pjbdev_closedevice(openinfo->devinfo);
	openinfo->devinfo = NULL;
	return FALSE;
        }

    /* add one more to the count */

    if (memcmp(pjbinfo.ssn,openinfo->openssn,6) == 0) {
	return TRUE;
	}

    pjbdev_closedevice(openinfo->devinfo);
    openinfo->devinfo = NULL;
		       
    return FALSE;
}


/*  *********************************************************************
    *  PJBDEV_GETTICKS()						*
    *  									*
    *  Return system time in ticks					*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   ticks							*
    ********************************************************************* */
static long pjbdev_getticks()
{
    struct timeval tv;

    gettimeofday(&tv,NULL);    

    return (tv.tv_sec*1000+tv.tv_usec/1000);
}

/*  *********************************************************************
    *  PJBDEV_SLEEP(sec)						*
    *  									*
    *  Sleep for 'sec' seconds						*
    *  									*
    *  Input parameters: 						*
    *  	   sec - # of seconds to sleep					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_sleep(int sec)
{
    sleep(sec);
}



/*  *********************************************************************
    *  pjbdev_enum(ids,maxids)						*
    *  									*
    *  Find all the PJBs connected to the USB bus			*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *      number of PJBs we found					*
    ********************************************************************* */
static int pjbdev_enum(PJB_INFO *ids,int maxids)
{
    ENUMINFO enuminfo;
    int cnt;

    memset(&enuminfo,0,sizeof(enuminfo));
    enuminfo.ids = ids;
    enuminfo.idcnt = 0;
    enuminfo.maxids = maxids;

    /* Under linux, we only support one at a time. */

    if (pjbdev->pcd_debug) printf("[enumerating devices]\n");

    //
    // XXX Repeatedly call the function below for each
    // device in the system, until the table is full.
    //
    // for now, we only have one device.
    //

    pjbdev_enumfunc(pjbdev_devname,&enuminfo);

    return enuminfo.idcnt;
}


PJBCOMMHANDLE *pjbdev_open(PJB_ID *id)
{
    OPENINFO openinfo;

    openinfo.openssn = id->b;
    openinfo.devinfo = NULL;

    //
    // XXX Repeatedly call the function below for each
    // device in the system, until openinfo.devinfo is
    // filled in.
    //
    // for now, we only have one device.
    //

    pjbdev_openfunc("/dev/cpqpjb",&openinfo);

    return openinfo.devinfo;
}

void pjbdev_close(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice(pch);
}



/*  *********************************************************************
    *  pjbdev_opendevice1(info)						*
    *  									*
    *  This routine is the meat of opening a USB connection - 		*
    *  given the info in the handle structure, open the connection.	*
    *  									*
    *  Input parameters: 						*
    *  	   info - USB handle info					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - open ok						*
    *  	   FALSE - could not open					*
    ********************************************************************* */

int pjbdev_opendevice1(PJBCOMMHANDLE *pch)
{
    fd_set rfds;
    struct timeval tv;
    int res;

    pch->handle = open(pch->devfilename,O_RDWR|O_NOCTTY);

    if (pch->handle < 0) {
	pch->handle = 0;
	if (pjbdev->pcd_debug) {
	    printf("[Could not open %s: %s]\n",
		   pch->devfilename,strerror(errno));
	    }
	return FALSE;
	}

    pch->isopen = TRUE;

    return TRUE;
}



/*  *********************************************************************
    *  pjbdev_opendevice(info)						*
    *  									*
    *  Opens both the read and write pipes of the PJB device		*
    *  									*
    *  Input parameters: 						*
    *  	   info - pointer to info struct				*
    *  	   								*
    *  Return value:							*
    *  	   TRUE if devices opened successfully, FALSE otherwise		*
    ********************************************************************* */

static PJBCOMMHANDLE *pjbdev_opendevice(char *realname)
{ 
    PJBCOMMHANDLE *pch;

    pch = calloc(1,sizeof(PJBCOMMHANDLE));

    strcpy(pch->devfilename,realname);
    pch->alternate = 0;

    if (pjbdev_opendevice1(pch)) {
	return pch;
        }
    else {
	free(pch);
	return NULL;
        }
}

/*  *********************************************************************
    *  pjbdev_closedevice(info)						*
    *  									*
    *  Closes both handles to the PJB device				*
    *  									*
    *  Input parameters: 						*
    *  	   info - information structure					*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_closedevice1(PJBCOMMHANDLE *pch)
{
    if (pch->handle) close(pch->handle);
    pch->handle = 0;
}

/*  *********************************************************************
    *  pjbdev_closedevice(pch)						*
    *  									*
    *  Close the device and free the comm handle object			*
    *  									*
    *  Input parameters: 						*
    *  	   pch								*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

static void pjbdev_closedevice(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice1(pch);
    free(pch);
}


/*  *********************************************************************
    *  pjbdev_reopendevice(info)					*
    *  									*
    *  Close, then reopen file handles associated with this		*
    *  device.  This is used to reestablish broken pipes		*
    *  									*
    *  Input parameters: 						*
    *  	   info - handle info structure					*
    *  	   								*
    *  Return value:							*
    *  	   TRUE - reopened successfully					*
    *  	   FALSE - could not open					*
    ********************************************************************* */
int pjbdev_reopendevice(PJBCOMMHANDLE *pch)
{
    pjbdev_closedevice1(pch);

    if (pjbdev_opendevice1(pch)) {
	return TRUE;
	}
    return FALSE;
}




#ifdef _DUMPUSB_
/*  *********************************************************************
    *  dumpdata(str,buffer,buflen)					*
    *  									*
    *  dump data to the screen in hex format.				*
    *  									*
    *  Input Parameters: 						*
    *      str - file for output					*
    *      buffer - pointer to buffer					*
    *      buflen - length of buffer					*
    *      								*
    *  Return Value:							*
    *      nothing.							*
    ********************************************************************* */

static void dumpdata(str,buffer,buflen)
FILE *str;
unsigned char *buffer;
int buflen;
{
	int i,j;

	for (i=0; i<buflen; i+=16) {
		fprintf(str,"%04X: ",i);
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str,"   ");
			else fprintf(str,"%02X ",buffer[i+j]);
			}
		fprintf(str," | ");
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str," ");
			else fprintf(str,"%c",isprint(buffer[i+j]) ? buffer[i+j] : ' ');
			}
		fprintf(str,"\n");
		}
}
#endif



/*  *********************************************************************
    *  pjbdev_tx(PJBCOMMHANDLE *pjbdev,unsigned char *buf,int len)      *
    *  									*
    *  Transmit data to a given usb "destination" 			*
    *  									*
    *  Input parameters: 						*
    *  	   buf - buffer to send						*
    *  	   len - length of buffer to send				*
    *  	   dest - destination address					*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes written					*
    *  	   0 for a broken pipe						*
    *  	   <0 for errors						*
    ********************************************************************* */

static int pjbdev_tx(PJBCOMMHANDLE *pch,unsigned char *buf,int len)
{
    int res;

    //
    // It's an error if the driver is not open
    //

    if (pch->isopen == FALSE) {
	return PJBCOMM_DEVERR;
	}

    //
    // If the length is an exact multiple of 64 bytes,
    // add one byte.  It's guaranteed to be there
    // in the ebuf, so this should be ok.
    //
    // The driver in the PJB needs to see a short
    // packet to show EOF.	
    //

    if ((len % 64) == 0) len++;

    //
    // Write the data to the device.
    //

#ifdef _DUMPUSB_
    if (pjbdev->pcd_debug) {
	printf("TransmitData--------------------\n");
	dumpdata(stdout,buf,len);
	}
#endif

    res = write(pch->handle,buf,len);

    //
    // If we got an error or wrote zero bytes, the USB connection is gone.
    // Try to reestablish it.
    //

    if (res != len) {
	if (pjbdev->pcd_debug) {
	    printf("[Reestablishing USB link]\n");
   	    }

	if (pjbdev_reopendevice(pch) == FALSE) return PJBCOMM_DEVERR;

	res = write(pch->handle,buf,len);

	if (res != len) {
	    return PJBCOMM_DEVERR;
	    }

	if (pjbdev->pcd_debug) {
	    printf("[USB link reestablished]\n");
	    }
	}

    return TRUE;
}


/*  *********************************************************************
    *  pjbdev_rx(pjbdev,*buf,len,timeout)                               *
    *  									*
    *  Receive data from the USB device					*
    *  									*
    *  Input parameters: 						*
    *  	   buffer - receive data goes here				*
    *  	   len - length of receive buffer				*
    *  	   timeout - milliseconds to wait for a timeout			*
    *  	   								*
    *  Return value:							*
    *  	   number of bytes received					*
    *  	   <0 for error (including timeouts)				*
    ********************************************************************* */

static int pjbdev_rx(PJBCOMMHANDLE *pch,unsigned char *buf,int len,int timeout)
{
    int res;
    int winerr;

    if (pch->isopen == FALSE) {
	return PJBCOMM_DEVERR;
	}

    // Read from the device.

    res = read(pch->handle,buf,len);

    if (res <= 0) {
	return PJBCOMM_TIMEOUT;
        }

#ifdef _DUMPUSB_
    if (pjbdev->pcd_debug) {
	printf("ReceiveData--------------------\n");
	dumpdata(stdout,buf,res);
	}
#endif

    return res;
}
