/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB API Client Routines			File: PJBCLI.C		**
   **									**
   **  This is the client end of the RPC to the Personal Jukebox	**
   **  The bulk of the API calls live in this module.			**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/


#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "pjbtypes.h"
#include "queue.h"
#include "ebuf.h"
#include "pjbproto.h"
#include "pjbapi.h"
#include "pjbdev.h"
#include "usbctl.h"
#ifdef __linux__
#include <unistd.h>
#include <sys/time.h>
#endif


/*#define _DEBUG_*/

/*  ********************************************************************
    *  Globals								*
    ********************************************************************* */

int pcli_usbport = -1;
u32 pcli_curxid = 0;

int (*pcli_debugcallback)(int,void *,char *) = NULL;
void *pcli_debugarg = NULL;

/*  *********************************************************************
    *  Structures							*
    ********************************************************************* */

typedef struct PCLI_EnumInfo {
    PJB_INFO *idarray;
    int maxids;
    int curids;
} PCLI_ENUMINFO;


/*  *********************************************************************
    *  PCLI_ALLOCBUF()						        *
    *  									*
    *  Allocate a buffer appropriate for sending to 'dest'		*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address					*
    *  	   								*
    *  Return value:							*
    *  	   EBUF , NULL if none left					*
    ********************************************************************* */
EBUF *pcli_allocbuf(void)
{
    return usb_allocbuf(pcli_usbport);
}

/*  *********************************************************************
    *  PCLI_FREEBUF(buf)						*
    *  									*
    *  Free a buffer to the correct pool, depending on where		*
    *  we were trying to send it.					*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address					*
    *  	   buf - buffer							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */

void pcli_freebuf(EBUF *buf)
{
    usb_freebuf(buf);
}


/*  *********************************************************************
    *  PCLI_DOCOMMAND_BLK(buf,outhdr,xid,dest)				*
    *  									*
    *  Execute a command, sending the request to the target 		*
    *  and waiting for a response to arrive.				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - pointer to a pointer to an ebuf.  This pointer		*
    *  	         on input is the ebuf to send, and on output		*
    *  	         is the response ebuf, or NULL.  The caller must	*
    *  	         free the resulting ebuf.				*
    *      outhdr - returned packet header				*
    *  	   xid - transaction ID - must match in response messages	*
    *  	   dest - destination Ethernet address for request		*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, 							*
    *  	   <0 = error code						*
    ********************************************************************* */

static int pcli_docommand_blk(EBUF **buf,PJBPKTHDR *hdr,u32 xid,PJBCOMMHANDLE *pjbdev)
{
    int err;
    EBUF *b;
    char *str;
    int retrycnt;
    int continuing;

    retrycnt = PJBPROTO_RETRY_COUNT;

    //
    // Send the command, making sure usb_transmit does
    // not try to free the buffer.
    //

    continuing = FALSE;		/* make sure we send something */

    while (retrycnt > 0) {

	if (!continuing) {
	    (*buf)->eb_status |= EBUF_NO_TX_FREE;

	    usb_transmit(pcli_usbport,pjbdev,*buf);
	    continuing = FALSE;
	    }


	/* get an event */
	b = usb_receive(pcli_usbport,pjbdev,5000);

	/* did we get one?  If so... */

	if (b) {
	    //
	    // Parse the header.  Drop the packet if
	    // we don't like the header, or if
	    // the transaction ID does not match,
	    // or if it's not a response packet
	    // to the request we just made.
	    //

	    if (pjbproto_parse_header_to_record(b,hdr) != 0) {
		pcli_freebuf(b);
		continue;
		}

	    if ((hdr->pph_xid != xid) ||
		((hdr->pph_cmd & MTR_RESPFLAG) == 0)) {
		pcli_freebuf(b);
		continue;
		}

	    //
	    // If the format is a "continue" packet,
	    // reset the retry counter and keep waiting
	    //

	    if ((hdr->pph_fmt == PJB_FMT_CONTINUE) ||
		(hdr->pph_fmt == PJB_FMT_CONTINUE_ITEM)) {
		retrycnt = PJBPROTO_RETRY_COUNT;
		pcli_freebuf(b);
		continuing = TRUE;

		if (hdr->pph_fmt == PJB_FMT_CONTINUE_ITEM) {
		    if (pjbproto_parse_itemlist(b,
						IT_DBGSTR,&str,
						IT_END) == 1) {
			if (pcli_debugcallback) {
			    (*pcli_debugcallback)(0,pcli_debugarg,str);
			    }
			}
		    }

		continue;
		}

	    //
	    // If we made it this far, we're going to assume
	    // we've got the right response, so break out.
	    //

	    break;
	    }


	//
	// if we got a packet, stop retrying.
	//

	if (b != NULL) break;

	// otherwise, go back and do it again.

	retrycnt--;
	}

    //
    // free the original transmit buffer
    //

    pcli_freebuf(*buf);

    //
    // find out what happened.  No buffer returned means
    // we timed out.
    //

    if (!b) {
	err = PJB_ERR_TIMEOUT;
	}
    else {
	err = 0;
	}

    // 
    // return the response
    //

    *buf = b;

    // Ok, return error code

    return err;    
}



/*  *********************************************************************
    *  PCLI_DOCOMMAND(buf,outhdr,xid,dest)				*
    *  									*
    *  Execute a command, sending the request to the target 		*
    *  and waiting for a response to arrive.				*
    *  									*
    *  Input parameters: 						*
    *  	   buf - pointer to a pointer to an ebuf.  This pointer		*
    *  	         on input is the ebuf to send, and on output		*
    *  	         is the response ebuf, or NULL.  The caller must	*
    *  	         free the resulting ebuf.				*
    *      outhdr - returned packet header				*
    *  	   xid - transaction ID - must match in response messages	*
    *  	   dest - destination Ethernet address for request		*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok, 							*
    *  	   <0 = error code						*
    ********************************************************************* */
static int pcli_docommand(EBUF **buf,PJBPKTHDR *hdr,u32 xid,PJB_HANDLE pjb)
{
    PJBHANDLEINFO *info = (PJBHANDLEINFO *) pjb;

    return pcli_docommand_blk(buf,hdr,xid,info->devhandle);
}





#ifdef _DEBUG_
/*  *********************************************************************
    *  dumpdata(str,buffer,buflen)					*
    *  									*
    *  dump data to the screen in hex format.				*
    *  									*
    *  Input Parameters: 						*
    *      str - file for output					*
    *      buffer - pointer to buffer					*
    *      buflen - length of buffer					*
    *      								*
    *  Return Value:							*
    *      nothing.							*
    ********************************************************************* */

static void dumpdata(str,buffer,buflen)
FILE *str;
unsigned char *buffer;
int buflen;
{
	int i,j;

	for (i=0; i<buflen; i+=16) {
		fprintf(str,"%04X: ",i);
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str,"   ");
			else fprintf(str,"%02X ",buffer[i+j]);
			}
		fprintf(str," | ");
		for (j=0; j<=15; j++) {
			if ((i+j) > buflen) fprintf(str," ");
			else fprintf(str,"%c",isprint(buffer[i+j]) ? buffer[i+j] : ' ');
			}
		fprintf(str,"\n");
		}
}
#endif

/*  *********************************************************************
    *  PJB_PING(handle,len,verify)					*
    *  									*
    *  Ping a PJB.							*
    *  									*
    *  Input parameters: 						*
    *  	   pjb - handle of open pjb					*
    *  	   len - length of data to echo					*
    *  	   verify - true to verify data					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ping was successful, error otherwise			*
    ********************************************************************* */

PJB_DLLEXPORT int PJB_Ping(PJB_HANDLE pjb,int len,int verify)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;
    int idx;
    u8 *ptr;
    static u8 abyte = 0;
    u8 startbyte;
    u8 b;

    if (len == 0) len = 16;
    if (len > PJB_ClickSize) len = PJB_ClickSize;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_CONFIG,		// class
			  MTC_ECHO,			// cmd
			  PJB_FMT_BULKDATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click

    /* add data */

    ptr = buf->eb_ptr;
    startbyte = abyte;
    for (idx = 0; idx < len; idx++) {
	*ptr++ = abyte++;
	}
    buf->eb_ptr += len;
    buf->eb_length += len;

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    if (buf) {

	if (verify) {
	    if ((int)(buf->eb_length) != len) {
#ifdef _DEBUG_
		printf("Lengths do not match! rx=%d expected=%d\n",
		       buf->eb_length,len);
#endif
		res = PJB_ERR_DATACORRUPT;
		}
	    ptr = buf->eb_ptr;
	    b = startbyte;
	    for (idx = 0; idx < len; idx++) {
		if (*ptr++ != b++) {
#ifdef _DEBUG_
		    dumpdata(stdout,buf->eb_ptr,buf->eb_length);
#endif
		    res = PJB_ERR_DATACORRUPT;
		    break;
		    }
		}
	    }

	pcli_freebuf(buf);
	}

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;
}



/*  *********************************************************************
    *  PJB_GetInfo1(pjb,info)						*
    *  									*
    *  Query the hardware information about the specified PJB		*
    *  and return it.							*
    *  									*
    *  Input parameters: 						*
    *  	   pjbdev - open pjb comm handle				*
    *  	   INFO - PJB_INFO structure to return				*
    *  	   								*
    *  Return value:							*
    *  	   result code							*
    ********************************************************************* */

int PJB_GetInfo1(PJBCOMMHANDLE pjbdev,PJB_INFO *info)
{      
    EBUF *buf;
    int res;
    char *friendlyname = "";
    int hwver = 0;
    int swver = 0;
    int features = 0;
    u32 xid;
    PJBPKTHDR hdr;
    u8 *ssn = NULL;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_CONFIG,		// class
			  MTC_SOLICIT,			// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click


    res = pcli_docommand_blk(&buf,&hdr,xid,pjbdev);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_fmt != PJB_FMT_ITEMLIST) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (res == 0) {
	pjbproto_parse_itemlist(buf,
			    IT_HWVER,&hwver,
			    IT_SWVER,&swver,
			    IT_FRIENDLYNAME,&friendlyname,
			    IT_FEATURES,&features,
			    IT_SSN,&ssn,
			    IT_END);
	memset(info,0,sizeof(PJB_INFO));
	info->hwver = hwver;
	info->swver = swver;
	strcpy(info->name,friendlyname);
	info->features = features;
	if (ssn) memcpy(info->ssn,ssn,sizeof(info->ssn));
	}
    else {
	memset(info,0,sizeof(PJB_INFO));
	}


    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}

/*  *********************************************************************
    *  PJB_GetInfo(pjb,info)						*
    *  									*
    *  Query the hardware information about the specified PJB		*
    *  and return it.							*
    *  									*
    *  Input parameters: 						*
    *  	   pjb - open PJB handle					*
    *  	   INFO - PJB_INFO structure to return				*
    *  	   								*
    *  Return value:							*
    *  	   result code							*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_GetInfo(PJB_HANDLE pjb,PJB_INFO *info)
{
    PJBHANDLEINFO *pci = (PJBHANDLEINFO *) pjb;
    PJBCOMMHANDLE *pjbdev = (PJBCOMMHANDLE *) pci->devhandle;
    int res;

    res = PJB_GetInfo1(pjbdev,info);
    if (res == 0) memcpy(&(info->id.b),PJB_GetID(pjb)->b,6);
    return res;
}


/*  *********************************************************************
    *  PJB_Enumerate(handlearray,maxids)		       		*
    *  									*
    *  Solicit on the LAN for PJBs and return the ones that we		*
    *  fine.								*
    *  									*
    *  Input parameters: 						*
    *  	   iaddray - array of ID structures that we return		*
    *  	   maxids - how many total we can return			*
    *  	   								*
    *  Return value:							*
    *  	   0 if some were enumerated					*
    *  	   timeout error if none were enumerated			*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_Enumerate (PJB_INFO *ids, int maxids)
{
    PCLI_ENUMINFO info;
    int res;

    info.idarray = ids;
    info.maxids = maxids;
    info.curids = 0;

    if (pcli_usbport >= 0) {
	res = PJBCOMM_ENUM(ids,maxids);
	info.curids += res;
	info.maxids -= res;	
	}

    return info.curids;		/* return # that we got */
}

/*  *********************************************************************
    *  PJB_ReadTOCClick(dest,clicknum,buffer)				*
    *  									*
    *  Read a click from the TOC on the PJB device			*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address					*
    *  	   clicknum - click number within the toc			*
    *  	   buffer - where to put the click we read			*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_ReadTOCClick (PJB_HANDLE pjb, int clicknum, void *buffer)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_TOC,		// class
			  MTC_READTOC,			// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  (u16)clicknum);		// click

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if ((signed char)hdr.pph_status < 0) {
	    res = (int) ((signed char) hdr.pph_status);
	    }
	else {
	    if ((buf->eb_length != PJB_ClickSize) ||
		(hdr.pph_fmt != PJB_FMT_BULKDATA) ||
		(hdr.pph_click != clicknum)) {
		res = PJB_ERR_PROTOCOLERR;
		}
	    }
	}

    //
    // If no errors after all that, copy the data.
    //

    if (res == 0) {
	memcpy(buffer,buf->eb_ptr,PJB_ClickSize);
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}



/*  *********************************************************************
    *  PJB_ReadAllocBlockHead(dest,blknum,buffer)		       	*
    *  									*
    *  read the first click of the given allocation unit.  This		*
    *  is used mostly for scavenging the disk.				*
    *  									*
    *  Input parameters: 						*
    *  	   dest - pjb's address						*
    *  	   blknum - alloc unit number					*
    *  	   buffer - where to put the data				*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_ReadAllocBlockHead (PJB_HANDLE pjb, void *buffer, int blknum)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_STORAGE,		// class
			  MTC_READBLOCKHDR,		// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  blknum,			// block
			  0);				// click

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if ((buf->eb_length != PJB_ClickSize) ||
	    (hdr.pph_fmt != PJB_FMT_BULKDATA) ||
	    ((int)hdr.pph_block != blknum)) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    //
    // If no errors after all that, copy the data.
    //

    if (res == 0) {
	memcpy(buffer,buf->eb_ptr,PJB_ClickSize);
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}


/*  *********************************************************************
    *  PJB_WRITETOCCLICK(dest,clicknum,buffer)				*
    *  									*
    *  Write a click to the toc.					*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination (pjb) addr				*
    *  	   clicknum - click number within toc				*
    *  	   buffer - buffer address of data to write			*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 - error code						*
    ********************************************************************* */

PJB_DLLEXPORT int PJB_WriteTOCClick (PJB_HANDLE pjb, int clicknum, void *buffer)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_TOC,		// class
			  MTC_WRITETOC,			// cmd
			  PJB_FMT_BULKDATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  (u16)clicknum);		// click

    
    ebuf_add_bytes(buf,buffer,PJB_ClickSize);

    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if ((hdr.pph_fmt != PJB_FMT_NODATA) ||
	    (hdr.pph_click != clicknum)) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}

/*  *********************************************************************
    *  PJB_WriteClick(pjb,block,clicknum,buffer)			*
    *  									*
    *  Write a click of data to the remote disk				*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address (pjb addr)			*
    *  	   block,clicknum - allocblock # and click within block		*
    *  	   buffer - data to write					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_WriteClick (PJB_HANDLE pjb, void *buffer, int clicknum, int block)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_STORAGE,		// class
			  MTC_WRITEBLOCK,		// cmd
			  PJB_FMT_BULKDATA,		// fmt
			  0,				// status
			  xid,				// xid
			  block,			// block
			  (u16)clicknum);		// click

    
    ebuf_add_bytes(buf,buffer,PJB_ClickSize);

    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_status != 0) {
	    }

	if ((hdr.pph_fmt != PJB_FMT_NODATA) ||
	    ((int)hdr.pph_block != block) || 
	    (hdr.pph_click != clicknum)) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}


/*  *********************************************************************
    *  PJB_CommitTOC(handle,nclicks,chksum)				*
    *  									*
    *  Commit the TOC we've been writing.				*
    *  									*
    *  Input parameters: 						*
    *  	   dest - pjb's address						*
    *  	   nclicks - total # of clicks we wrote				*
    *  	   chksum - toc checksum					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */

PJB_DLLEXPORT int PJB_CommitTOC(PJB_HANDLE pjb, int nclicks, PJB_Checksum *cksum)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_TOC,		// class
			  MTC_COMMITTOC,		// cmd
			  PJB_FMT_ITEMLIST,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click

    pjbproto_build_itemlist(buf,
			    IT_CLICKSINTOC,nclicks,
			    IT_CHECKSUM,cksum->b,
			    IT_END);


    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_fmt != PJB_FMT_NODATA) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}

/*  *********************************************************************
    *  PJB_GetDiskInfo(pjb,info)					*
    *  									*
    *  Obtain information about the remote disk				*
    *  									*
    *  Input parameters: 						*
    *  	   pjb - open pjb handle					*
    *  	   info - structure to fill in					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */

PJB_DLLEXPORT int PJB_GetDiskInfo (PJB_HANDLE pjb, PJB_DiskInfo *info)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_CONFIG,		// class
			  MTC_GETINFO,			// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click


    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_fmt != PJB_FMT_ITEMLIST) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (res == 0) {
	pjbproto_parse_itemlist(buf,
			    IT_FSVERSION,  &(info->version),
			    IT_CLICKSINTOC,&(info->maxclicksintoc),
			    IT_TOTALBLOCKS,&(info->numallocblocks),
			    IT_ERRORCOUNT, &(info->errorcount),
			    IT_VALIDTOCS,  &(info->validtoccopies),
			    IT_CURTOCCLICKS,&(info->curclicksintoc),
			    IT_ISOLDTOC,&(info->isoldtoc),
			    IT_END);
	}
    else {
	memset(info,0,sizeof(PJB_DiskInfo));
	}


    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}


/*  *********************************************************************
    *  PCLI_INIT()						        *
    *  									*
    *  Initialize the client portal					*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */
int pcli_init(void)
{
    if (pcli_usbport == -1) {
        pcli_usbport = usb_open(USB_PJB_PROTOCOL_PORT,
				USB_PJB_PROTOCOL_ENDPOINT,
				NULL);
       }

    pcli_curxid = (int) PJBCOMM_TICKCOUNT();
    return 0;
}



/*  *********************************************************************
    *  PCLI_UNINIT()							*
    *  									*
    *  Close the client portal						*
    *  									*
    *  Input parameters: 						*
    *  	   nothing							*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */
int pcli_uninit(int interfaces)
{
    if (pcli_usbport >= 0) usb_close(pcli_usbport);
    pcli_usbport = -1;

    return 0;
}





/*  *********************************************************************
    *  PJB_Reboot(pjb,how)					 	*
    *  									*
    *  Reboot the PJB							*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address					*
    *      how = 0 for main code, 1 for recovery code			*
    *  	   								*
    *  Return value:							*
    *  	   0 if ping was successful, error otherwise			*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_Reboot (PJB_HANDLE pjb, int how)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_UPGRADE,		// class
			  MTC_REBOOT,			// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  (u16)how);			// click (how)

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    if (buf) {
	pcli_freebuf(buf);
	}

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;
}





/*  *********************************************************************
    *  PJB_WRITESOFT(dest,block,buffer)					*
    *  									*
    *  Save PCM samples to memory					*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address (pjb addr)			*
    *  	   click - block number						*
    *  	   buffer - data to write					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_WriteSoft(PJB_HANDLE pjb,int click,void *buffer,int buflen)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_UPGRADE,		// class
			  MTC_WRITESOFT,		// cmd
			  PJB_FMT_BULKDATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  (u16)click);			// click

    
    ebuf_add_bytes(buf,buffer,buflen);

    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if ((hdr.pph_fmt != PJB_FMT_NODATA) ||
	    (hdr.pph_click != click)) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}

/*  *********************************************************************
    *  PJB_OPENSOFT(dest)						*
    *  									*
    *  Prepare to upgrade software					*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address (pjb addr)			*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_OpenSoft(PJB_HANDLE pjb)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_UPGRADE,		// class
			  MTC_OPENSOFT,			// cmd
			  PJB_FMT_NODATA,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click

    
    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_fmt != PJB_FMT_NODATA) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}

/*  *********************************************************************
    *  PJB_COMMITSOFT(dest,checksum,mode)				*
    *  									*
    *  commit software to remote flash					*
    *  									*
    *  Input parameters: 						*
    *  	   dest - destination address (pjb addr)			*
    *  	   checksum - 4-byte crc					*
    *  	   								*
    *  Return value:							*
    *  	   0 if ok							*
    *  	   <0 = error code						*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_CommitSoft(PJB_HANDLE pjb,PJB_Checksum *cksum,int mode)
{
    EBUF *buf;
    u32 xid;
    PJBPKTHDR hdr;
    int res;

    buf = pcli_allocbuf();
    if (!buf) return PJB_ERR_NO_RESOURCES;

    xid = pcli_curxid++;

    pjbproto_build_header(buf,				// buffer
			  PJB_PCLASS_UPGRADE,		// class
			  MTC_COMMITSOFT,		// cmd
			  PJB_FMT_ITEMLIST,		// fmt
			  0,				// status
			  xid,				// xid
			  0,				// block
			  0);				// click

    
    pjbproto_build_itemlist(buf,
			    IT_CHECKSUM,cksum->b,
			    IT_SECTOR,mode,
			    IT_END);

    // Do the command.

    res = pcli_docommand(&buf,&hdr,xid,pjb);

    //
    // Check for some obvious problems.
    //

    if (res == 0) {
	if (hdr.pph_fmt != PJB_FMT_NODATA) {
	    res = PJB_ERR_PROTOCOLERR;
	    }
	}

    if (buf) pcli_freebuf(buf);

    if (res == 0) res = (int) ((signed char)hdr.pph_status);

    return res;

}


/*  *********************************************************************
    *  PJB_SetDebugCallback(cb)						*
    *  									*
    *  Set callback routine for processing messages we receive		*
    *  in continue packets and such.					*
    *  									*
    *  Input parameters: 						*
    *  	   cb - callback routine, called as callback(flg,string)	*
    *  	   								*
    *  Return value:							*
    *  	   nothing							*
    ********************************************************************* */
PJB_DLLEXPORT int PJB_SetDebugCallback(int (*callback)(int,void *,char *),
				       void *arg)
{
    pcli_debugcallback = callback;
    pcli_debugarg = arg;
    return 0;
}
