/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Routines to manipulate EBUFs		File: EBUF.H		**
   **									**
   **  These routines muck with EBUFs, named so for their ancient	**
   **  Ethernet heritage.  We do most of our communications with	**
   **  the PJB in an EBUF.						**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/
#ifndef _EBUF_H
#define _EBUF_H

#ifdef __cplusplus
extern "C" {
#endif

/*  *********************************************************************
    *  Ethernet constants						*
    ********************************************************************* */

#define EBUF_MAX_PACKET 1518	/* max size of an Ethernet packet + CRC */

#define EBUF_POOL_SIZE	4	/* number of packets in the Ethernet pool */

/*  *********************************************************************
    *  Ethernet buffer structure and constants				*
    ********************************************************************* */

#define EBUF_RECEIVE	0
#define EBUF_TRANSMIT   0x8000
#define EBUF_NO_TX_FREE	0x4000
#define EBUF_TX_FAILURE	0x2000
#define EBUF_TX_OK	0x1000

typedef struct EBuf {
    QBLOCK eb_qblock;		/* queue linkage */
    unsigned int  eb_length;		/* length of buffer */
    unsigned int  eb_status;		/* status of buffer */
    unsigned int  eb_port;		/* portal ID */
    u8    *eb_ptr;		/* pointer to data */
    u8     eb_data[EBUF_MAX_PACKET];
    u8     eb_usrdata[524];	/* round up to 2048 words */
} EBUF;


#ifdef __cplusplus
};
#endif

/*  *********************************************************************
    *  Macros for parsing and building packets				*
    ********************************************************************* */

/* initialize an ebuf so we can add data to it */
#define ebuf_init(buf) usb_initbuf(buf)

/* return remaining bytes (length) while parsing packets */
#define ebuf_remlen(buf) ((buf)->eb_length)

#define ebuf_moveptr(buf,len)  (buf)->eb_length += (len); \
			       (buf)->eb_ptr += (len)

#ifdef EBUF_MACROS
/* add a byte to an ebuf, incrementing length */
#define ebuf_add_u8(buf,b) *((buf)->eb_ptr++) = (b); \
                            (buf)->eb_length++

/* add a u16 to an ebuf, incrementing length */
#define ebuf_add_u16(buf,w) ebuf_add_u8((buf),((w) & 0xFF)); \
                            ebuf_add_u8((buf),(((w)>>8)&0xFF))

/* add a u32 to an ebuf, incrementing length */
#define ebuf_add_u32(buf,d) ebuf_add_u16((buf),((d) & 0xFFFF)); \
				ebuf_add_u16((buf),(((d)>>16)&0xFFFF))
#else
void ebuf_add_u8(EBUF *buf,u8 b);
void ebuf_add_u16(EBUF *buf,u16 w);
void ebuf_add_u32(EBUF *buf,u32 w);
#endif


/* add a buffer of bytes */
#define ebuf_add_bytes(buf,bytes,cnt) memcpy((buf)->eb_ptr,bytes,cnt); \
                                (buf)->eb_ptr += (cnt); \
				(buf)->eb_length += (cnt);

/* return next u8 from the ebuf, advancing pointer and reducing length */

#define ebuf_get_u8(buf,var)    (buf)->eb_length--; \
				var = (*((buf)->eb_ptr++))

/* return next u16 from the ebuf, advancing pointer and reducing length */

#define ebuf_get_u16(buf,var)   (buf)->eb_ptr+=2; \
				(buf)->eb_length-=2; \
			     var = ( ((u16)*((buf)->eb_ptr-2)) | \
				     (((u16)*((buf)->eb_ptr-1)) << 8) ) 

/* return next u32 from the ebuf, advancing pointer and reducing length */

#define ebuf_get_u32(buf,var)   (buf)->eb_ptr+=4;	 \
			        (buf)->eb_length-=4; \
			    var = ( ((u32)*((buf)->eb_ptr-4)) | \
				    (((u32)*((buf)->eb_ptr-3)) << 8) | \
				    (((u32)*((buf)->eb_ptr-2)) << 16) | \
				    (((u32)*((buf)->eb_ptr-1)) << 24) )



#endif

