/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  Table of Contents definitions		File: TOC.H		**
   **									**
   **  This file contains on-disk-structure definitions and data	**
   **  structures related to representing a PJB's table-of-contents	**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/


#ifndef _TOC_H
#define _TOC_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef LPCTSTR
#define LPCTSTR		char const *
#endif

// Encoding types of audio data on disk

#define TC_ENC_MP3		1		// MP3 encoding
#define TC_ENC_PCM		2		// raw audio
#define TC_ENC_AAC		3		// AAC encoding (NOT SUPPORTED)

#define TC_MAJ_VERSION	2		// current major version
#define TC_MIN_VERSION	0		// current minor version


// TOC entries are of the form <CHAR><TEXT>\n.
// The preceeding <CHAR> can be one of the following.

#define TC_Version		'V'		// Version number string follows
#define TC_Root			'R'		// Jukebox root name follows
#define TC_Set			'S'		// Set name follows
#define	TC_Disk			'D'		// Disk name follows
#define	TC_Track		'T'		// Track name follows
#define	TC_Block		'B'
	// Block numbers, offsets, + encoding (see BLOCK_FORMAT)
#define	TC_ID			'I'	
	// Track UID/num + time (see TRACK_ID_FORMAT)
#define	TC_EndSegment	'.'		// End of directory or alloc map
#define	TC_AllocMap		'+'		// Alloc map entry
#define	TC_UidMap		'U'		// CD UID map entry

// Track origin

#define TC_Origin_CD	(0)		// audio CD capture
#define TC_Origin_ID3	(1)     // ID3 tagged MP3 file
#define TC_Origin_Unk	(2)     // untagged MP3 file

// Standard bitrates for encodings

#define TC_BR_32		4
#define TC_BR_64		8
#define TC_BR_96		12
#define TC_BR_128       16      // bit rate <= 1024 are in 8000 bits/sec
#define TC_BR_ESCAPE	1024    // bit rates > 1024 are in bits/sec

#define MAX_TOC_STR	  1024		// Maximum TOC line
#define MAX_NODE_STR  1000		// Maximum node name, minus uniquifiers

#define DEFAULT_NEW_JUKEBOX_NAME	"New Personal Jukebox"
#define DEFAULT_NEW_SET_NAME		"New Set"
#define DEFAULT_NEW_DISK_NAME		"New Disc"

#define TC_VERSION_FORMAT	"%d.%d"
		// a.b where a and b are the major and minor version numbers
		// major version numbers are for incompatible changes

#define TC_PJB_TRACKADDR_FORMAT	"%d %ld.%ld %ld.%ld"
		// "a b.c d.e where:
		//		a is the encoding type
		//		b.c is the start alloc block and offset in that block
		//		d.e is the end alloc block and number of track bytes in
		//			 in that block

#define TC_TRACK_ID_FORMAT	"%d.%d.%ld %ld"
#define TC_OLD_TRACK_FORMAT	"%d.%d.%ld"
		// a.b.c d where a is the CD uid index, b is the
		//      track number on that CD, c is the
		//		playing time in 75'ths, and d is the encoded
        //      bit rate (see TC_BR_* above)

#define TC_AM_FORMAT  "%d.%d.%d"
	// AllocMap run format:
	// a.b.c where a is the first block in a run, b is the length
	//          of the run, and c is last block of the previously
	//          allocated run within the same rip unit, or -1 if none 

#define FRAMES_PER_SEC	 (75)
	// the default representation of time is in "frames"

#define FRAMES_TO_SECS(x)  ((x + (FRAMES_PER_SEC/2))/FRAMES_PER_SEC)

//
// In-memory representation of a TOC
//

typedef int JNodeType;

typedef union {
	struct _pjb {
		long start_block;
		int	start_block_offset;
		long end_block;
		int end_block_bytes;
	} pjb;
	struct _repo {
		long fileid;
	} repo;
} TrackAddress;

#define MAX_FORMAT_ADDR		100


#define	JNODE_TYPE_NULL		0
#define	JNODE_TYPE_ROOT		1
#define JNODE_TYPE_SET		2
#define JNODE_TYPE_DISK		3
#define JNODE_TYPE_TRACK	4

#define JNODE_FIRST		(JNode) NULL
#define JNODE_LAST		(JNode) 0xFFFFFFFF

typedef	struct _JNodeRec *JNode;

struct _JNodeRec {
	JNodeType			type;
	LPCTSTR				text;
	JNode				parent;
	JNode				child;
	JNode				sibling;
	// following elements are valid only for type == JNODE_TYPE_TRACK
	TrackAddress		trackAddr;
	int					iCdUID;				// identifies CD (-1 iff not from CD)
	int					iTrack;				// track in original CD (-1 iff not from CD)
	long				time;				// playing time in 75'ths of sec
	int					enc_type;			// encoding type
	int					bit_rate;			// encoding bit rate (bits/sec)
};


typedef struct _JInsertPt {
	JNode		parent;
	JNode		pred;
} JInsertPt;
		// pred == JNODE_FIRST inserts as first sibling
		// pred == JNODE_LAST inserts as last sibling


// TOC types

#define TOC_TYPE_PJB		1

// errors arising from TOC_Ctx::Parse below

#define TOC_ERROR_VERSION	1
#define TOC_ERROR_FORMAT	2
#define TOC_ERROR_ALLOCMAP	3

// outbuf buffer for writing TOCs

typedef struct _TOC_IOBuf {
	char *s;	  // must be non-nil on output
	int	 pos;     // length on output, position on input
	int  maxlen;  // size of buffer
} TOC_IOBuf;

typedef struct _TOC_Ctx {
	JNode			root;
	int				tocType;
	LPCTSTR			*cdUIDs;
	int				uidCur;
	int				uidMax;
} TOC_Ctx;

TOC_Ctx	*MakeTocCtx(int tocType);
void	FreeTocCtx(TOC_Ctx *ctx);

void	InitIOBuf(TOC_IOBuf *bf, LPCTSTR s);

void	OutputTocLine(TOC_IOBuf *bf, char c, LPCTSTR t);
int		InputTocLine(TOC_IOBuf *bf, char *pc, char *ss);
			// ss must be at least MAX_TOC_STR + 1 bytes

void	OutputToc(TOC_Ctx *ctx, TOC_IOBuf *bf);
int		InputToc(TOC_Ctx *ctx, TOC_IOBuf *bf);

JNode	MakeNode(JNodeType type, LPCTSTR text);
JNode	CopyNode(JNode node);
JNode	CopyChildren(JNode node, JNode parent);
void	UnlinkNode(JNode node);
void	FreeNode(JNode node);
void	SetNodeText(JNode node, LPCTSTR text);

int		IndexFromCdUID(TOC_Ctx *ctx, LPCTSTR uid);
LPCTSTR	CdUIDFromIndex(TOC_Ctx *ctx, int i);

void	FormatTrackAddr(char *str, TOC_Ctx *ctx, JNode n);
			// str must have at least MAX_FORMATTED_ADDR bytes

void InsertNode(JNode jSrc, JInsertPt *pIns,
		int bReplace, int bCopy);
int DeleteNode(JNode node);
#ifdef __cplusplus
};
#endif

#endif
