/*
   ***********************************************************************
   **  Compaq Personal Jukebox						**
   **									**
   **  PJB API definitions			File: PJBAPI.H		**
   **									**
   **  This file contains basic definitions for the high-level API	**
   **  protocol between a host PC and a PJB.				**
   **									**
   **  Authors: Compaq Corporate Research                               **
   **									**
   ***********************************************************************
   **                                                                   **
   ** Copyright (C) 2000 by Compaq Computer Corporation                 **
   **                                                                   **
   ** This program is free software; you can redistribute it and/or     **
   ** modify it under the terms of the GNU General Public License       **
   ** as published by the Free Software Foundation; either version 2    **
   ** of the License, or (at your option) any later version.            **
   **                                                                   **
   ** This program is distributed in the hope that it will be useful,   **
   ** but WITHOUT ANY WARRANTY; without even the implied warranty of    **
   ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     **
   ** GNU General Public License for more details.                      **
   **                                                                   **
   ** You should have received a copy of the GNU General Public License **
   ** along with this program; if not, write to the Free Software       **
   ** Foundation, Inc., 59 Temple Place - Suite 330,                    **
   ** Boston, MA  02111-1307, USA.                                      **
   **                                                                   **
   ***********************************************************************
*/

#ifndef _PJBAPI_H
#define _PJBAPI_H

#ifndef _PJBTYPES_H
#include "pjbtypes.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef WIN32
#define PJB_DLLIMPORT    __declspec( dllimport )
#define PJB_DLLEXPORT    __declspec( dllexport )
#else
#define PJB_DLLIMPORT
#define PJB_DLLEXPORT
#endif



/*  *********************************************************************
    *  Constants							*
    ********************************************************************* */

#define PJB_ClickSize	1024	/* size of protocol transfer unit */
#define PJB_AllocSize	128		/* clicks in an alloc block */

/*  *********************************************************************
    *  Structures							*
    ********************************************************************* */

typedef void *PJB_HANDLE;

#define PJB_SSN_SIZE	6

typedef struct { 
    u8 b[PJB_SSN_SIZE];
} PJB_ID;	/* PJB Serial # (address) */

typedef struct { u8 b[4]; } PJB_Checksum;


typedef struct {
    PJB_ID id;
    unsigned int hwver;
    unsigned int swver;
    unsigned int features;
    char name[65];
    u8 ssn[PJB_SSN_SIZE];
} PJB_INFO;


typedef struct _pjb_diskinfo {
        int version;    /* version of this protocol; 0 at present */
        int maxclicksintoc;/* max number of clicks
                                   in a table of contents */
        int numallocblocks;     /* total number of alloc blocks on disc */
        unsigned errorcount;
                        /* a saturating count of the number of disc
                           errors seen by the device */
        int validtoccopies;     /* number of copies of the TOC.
                           Normally, this will be 2.
                           It will be 1 if a bad block
                           has been found in one of the
                           copies.  It will be zero if
                           no TOC has been written to the
                           device. */
   	int curclicksintoc; /* current number of client in 
   				table of contents */
	int isoldtoc;	/* non-zero iff the most up to date
			   valid table of contents is known
			   to be old (that is, the latest 
			   one is corrupt) */
} PJB_DiskInfo;

typedef struct fs_blkaddr {
    u8 addr[3];
} FS_BLKADDR;

typedef struct fs_allocblock {
	union {
	    struct {
		FS_BLKADDR h_next;            /* Next block in chain */
		FS_BLKADDR h_prev;            /* Previous block in chain */
		u8 userdata[1020];            /* Data not used by player */
		FS_BLKADDR h_next2;           /* Copy of "next block in chain" */
		FS_BLKADDR h_prev2;           /* Copy of "previous block in chain" */
	    } h;
	    u8 head[1032];
	} h;
        u8 body[130032];                      /* Body (mp3 data) */
	union {
	    struct {
		FS_BLKADDR t_next;            /* next block in chain */
		u8 t_chksum[4];               /* checksum of block */
		u8 t_unused;
	    } t;
	    u8 tail[8];
	} t;
} FS_ALLOCBLOCK;

typedef struct PjbFlashImage {
    unsigned char seal[4];		/* "PJB^Z" */
    unsigned char version[4];		/* major, minor, eco, reserved */
    unsigned char crc[4];		/* CRC of this image */
    unsigned int size;			/* size of image in bytes */
} PJBFLASHIMAGE;

#define FLASHREC_MAINCODE	0
#define FLASHREC_EXECUTE        3

/*  *********************************************************************
    *  Error codes							*
    ********************************************************************* */

/* error codes */
#define PJB_ERR_NODISK        (-1)    /* No disk at PJB */
#define PJB_ERR_DISK          (-2)    /* Disk I/O error */
#define PJB_ERR_ALLOCBLOCK    (-3)    /* alloc block number out of range */
#define PJB_ERR_OFFSET        (-4)    /* offset out of range */
#define PJB_ERR_CHECKSUM      (-5)    /* checksum mismatch */
#define PJB_ERR_NOTOC         (-6)    /* no table of contents */

#define PJB_ERR_COMM        (-100)    /* communications error */
#define PJB_ERR_TIMEOUT	    (-101)     /* timeout */
#define PJB_ERR_NO_RESOURCES (-102)    /* out of resources */
#define PJB_ERR_NODEVICE    (-103)     /* no ethernet device */
#define PJB_ERR_DATACORRUPT (-104)	/* network data is corrupt */
#define PJB_ERR_PROTOCOLERR (-105)	/* protocol error */
#define PJB_ERR_NOBIND	    (-106)	/* could not bind to adapter */
#define PJB_ERR_BADPARM	    (-107)
#define PJB_ERR_PJBINUSE    (-108)


/*  *********************************************************************
    *  Function prototypes						*
    ********************************************************************* */


/*
 * Library initialization
 */

PJB_DLLEXPORT int PJB_InitLib(void);
PJB_DLLEXPORT void PJB_UninitLib(void);


/*
 * Open/Close.
 */

PJB_DLLEXPORT int PJB_Open(PJB_ID *id,PJB_HANDLE *h);
PJB_DLLEXPORT int PJB_Close(PJB_HANDLE h);
PJB_DLLEXPORT PJB_ID *PJB_GetID(PJB_HANDLE h);

/*
 * Broadcast/enumerate functions
 */

PJB_DLLEXPORT int PJB_Enumerate(PJB_INFO *ids, int max_ids);
PJB_DLLEXPORT int PJB_GetInfo(PJB_HANDLE h,PJB_INFO *info);

/*
 * Diagnostics
 */

PJB_DLLEXPORT int PJB_Ping(PJB_HANDLE id,int len,int verify);

/*
 * Ripper/encoder routines
 */

PJB_DLLEXPORT int PJB_GetDiskInfo (PJB_HANDLE id, PJB_DiskInfo *info);

PJB_DLLEXPORT int PJB_ReadTOCClick (PJB_HANDLE id, int offset, void *buffer);
	/* reads one Click of TOC file, starting at offset*PJB_ClickSize */
    /* 0 == success, error code otherwise */

PJB_DLLEXPORT int PJB_WriteTOCClick (PJB_HANDLE id, int offset, void *buffer);
	/* writes one Click of TOC file, starting at offset*PJB_ClickSize */
    /* 0 == success, error code otherwise */

PJB_DLLEXPORT int PJB_CommitTOC (PJB_HANDLE id, int nclicks, PJB_Checksum *chk);
    /* Commit a new Table Of Contents, which has been written
       by some number of PJBProt_WriteTOCClick() commands issued since
       the last call to PJBProt_CommitTOC.  The device is expected
       to ensure that the new TOC is readable on disc, that it has
       the stated checksum (obtained from the cksum interface) and
       perhaps that multiple copies are stored stably.  On success,
       the device must store "len", and issue an error on a
       subsequent PJBProt_ReadTOCClick() command with "offset" >= "len".
       That is, "len" is measured in "clicks".
       Return >= 0 on success, < 0  on failure. 
       Failures include I/O error, checksum mismatch. */

PJB_DLLEXPORT int PJB_WriteClick (PJB_HANDLE id, void *buf, int offset, int allocblocknum);
    /* Write the "PJB_CLICKSIZE" bytes starting at address "buf"
       into allocation block "allocblocknum" at offset "offset*PJB_CLICKSIZE"
       bytes into the allocation block.
       Return >= 0 on success, < 0  on failure.
       Failures include I/O error, "allocblocknum" out of range,
       "offset" out of range (not in 0..PJB_ALLOCSIZE-1). */

PJB_DLLEXPORT int PJB_ReadAllocBlockHead (PJB_HANDLE id, void *buf, int allocblocknum);
    /* Return the "PJB_CLICKSIZE" bytes at offset 0 of allocation
       block "allocblocknum" into the buffer "buf".  In addition,
       the entire block is checksummed and checked against
       the checksum at the end of the block.
       Return >= 0 on success, < 0  on failure.
       Failures include I/O error, checksum mismatch,
       "allocblocknum" out of range. */

/*
 * Flash update
 */

PJB_DLLEXPORT int PJB_CommitSoft(PJB_HANDLE id,PJB_Checksum *chk,int mode);
PJB_DLLEXPORT int PJB_OpenSoft(PJB_HANDLE id);
PJB_DLLEXPORT int PJB_WriteSoft(PJB_HANDLE id,int blknum,void *buf,int len);
PJB_DLLEXPORT int PJB_Reboot(PJB_HANDLE id,int how);

/*
 * Misc.
 */
PJB_DLLEXPORT int PJB_SetDebugCallback(int (*callback)(int,void *,
						       char *),void *);

PJB_DLLEXPORT int PJB_SetDebugMode(int);

#define BOOTMETHOD_MAIN 0
#define BOOTMETHOD_RECOVERY 1


#ifdef __cplusplus
};
#endif


#endif

