unit xquery;

{*******************************************************}
{                                                       }
{       TxQuery dataset implementation                  }
{                                                       }
{       Copyright (c) 1999-2000 Alfonso moreno          }
{                                                       }
{     Written by:                                       }
{       Alfonso moreno                                  }
{       Hermosillo, Sonora, Mexico.                     }
{       Internet:  gismap@hmo.megared.net.mx            }
{                  luisarvayo@yahoo.com                 }
{                  inconmap@prodigy.net.mx              }
{       http://www.sigmap.com/txquery.htm               }
{                                                       }
{*******************************************************}

{$I XQ_FLAG.INC}
interface

uses
  SysUtils,
  Windows,
  Messages,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  StdCtrls,
  prExprQ,
  Db,
  XQMiscel,
  xqbase,
  lexlibQ,
  yacclibQ
{$ifdef level3}
  , DBTables
{$endif}
  ;

type
{-------------------------------------------------------------------------------}
{                  Define forward declarations and class definitions            }
{-------------------------------------------------------------------------------}

  TResultSet    = class;
  TxqFields     = class;
  TCustomxQuery = class;
  TDataSetClass = class of TDataSet;
  TSqlAnalizer  = class;


{-------------------------------------------------------------------------------}
{                  Define TxqField                                              }
{-------------------------------------------------------------------------------}

  TxqField = class
  private
     FFields          : TxqFields;     { the list of fields that this field belongs to                                  }
     FFieldName       : String;        { the column name sample: Customer.Addr1                                         }
     FDataType        : TExprType;     { the data type (ttString, ttFloat, ttInteger, ttBoolean)                        }
     FDataSize        : Word;          { calculated datasize for ttString (used in TxQuery)                             }
     FBufferOffset    : Integer;       { offset in the list of buffers for every record                                 }
     FFieldOffset     : Integer;       { Offset in the buffer in TxQuery dataset                                        }
     FReadOnly        : Boolean;       { = true, means that comes from single field, false = expression or joined field }
     FSourceField     : TField ;       { Field that originated this column, =nil if it is an expression                 }
     FCastType        : Word;          { field must be casted to this type on creation                                  }
     FCastLen         : Word;          { field must be of this len if CastType = RW_CHAR                                }
     FUseDisplayLabel : Boolean;       { true = use labels from SrcField.DisplayLabel for column alias                   }
     FFieldNo         : Integer;       { the number of the field, in base 1                                             }
     function GetData(Buffer: Pointer): Boolean;
     procedure SetData(Buffer: Pointer);
     function GetColWidth: Integer;
  protected
     function GetAsString: String; virtual;
     procedure SetAsString(const Value: String); virtual;
     function GetAsFloat: double; virtual;
     procedure SetAsFloat(Value: double); virtual;
     function GetAsInteger: Longint; virtual;
     procedure SetAsInteger(Value: Longint); virtual;
     function GetAsBoolean: Boolean; virtual;
     procedure SetAsBoolean(Value: Boolean); virtual;
     procedure SetDataType(Value: TExprType);
  public
     constructor Create(Fields: TxqFields; FieldNo: Integer); virtual;
     procedure Clear; virtual; abstract;

     property FieldName: String read FFieldName write FFieldName;
     property FieldNo: Integer read FFieldNo;
     property DataType: TExprType read FDataType write FDataType;
     property DataSize: Word read FDataSize write FDataSize;
     property ReadOnly: Boolean read FReadOnly write FReadOnly;
     property FieldOffset: Integer read FFieldOffset write FFieldOffset;
     property SourceField: TField read FSourceField write FSourceField;
     property CastType: Word read FCastType write FCastType;
     property CastLen: Word read FCastLen write FCastLen;
     property ColWidth: Integer read GetColWidth;
     property BufferOffset: Integer read FBufferOffset write FBufferOffset;
     property UseDisplayLabel: Boolean read FUseDisplayLabel write FUseDisplayLabel;

     property AsString: String read GetAsString write SetAsString;
     property AsFloat: double read GetAsFloat write SetAsFloat;
     property AsInteger: Longint read GetAsInteger write SetAsInteger;
     property AsBoolean: Boolean read GetAsBoolean write SetAsBoolean;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqStringField                                        }
{-------------------------------------------------------------------------------}

  TxqStringField = class(TxqField)
  private
     function GetValue(var Value: string): Boolean;
  protected
     function GetAsString: String; override;
     procedure SetAsString(const Value: String); override;
     function GetAsFloat: double; override;
     procedure SetAsFloat(Value: double); override;
     function GetAsInteger: Longint; override;
     procedure SetAsInteger(Value: Longint); override;
     function GetAsBoolean: Boolean; override;
     procedure SetAsBoolean(Value: Boolean); override;
  public
     constructor Create(Fields: TxqFields; FieldNo: Integer); override;
     procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqFloatField                                         }
{-------------------------------------------------------------------------------}

  TxqFloatField = class(TxqField)
  private
  protected
    function GetAsFloat: double; override;
    function GetAsInteger: Longint; override;
    function GetAsString: string; override;
    procedure SetAsFloat(Value: double); override;
    procedure SetAsInteger(Value: Longint); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(Fields: TxqFields; FieldNo: Integer); override;
    procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqIntegerField                                       }
{-------------------------------------------------------------------------------}

  TxqIntegerField = class(TxqField)
  private
  protected
    function GetAsFloat: double; override;
    function GetAsInteger: Longint; override;
    function GetAsString: string; override;
    procedure SetAsFloat(Value: double); override;
    procedure SetAsInteger(Value: Longint); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(Fields: TxqFields; FieldNo: Integer); override;
    procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqBooleanField                                       }
{-------------------------------------------------------------------------------}

  TxqBooleanField = class(TxqField)
  private
  protected
    function GetAsBoolean: Boolean; override;
    function GetAsString: string; override;
    procedure SetAsBoolean(Value: Boolean); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(Fields: TxqFields; FieldNo: Integer); override;
    procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxqFields                                             }
{-------------------------------------------------------------------------------}

  TxqFields = class
     FResultSet: TResultSet;
     FItems: TList;
     function GetCount: Integer;
     function GetItem(Index: Integer): TxqField;
  public
     constructor Create(ResultSet: TResultSet);
     destructor Destroy; override;
     function Add(DataType : TExprType): TxqField;
     procedure Clear;
     procedure Delete(Index: Integer);
     function FindField(const FieldName: string): TxqField;

     property  Count: Integer read GetCount;
     property  Items[Index: Integer]: TxqField read GetItem; default;
     property  ResultSet: TResultSet read FResultSet;
  end;

{-------------------------------------------------------------------------------}
{                  Define abstract TResultSet                                   }
{-------------------------------------------------------------------------------}

  TResultSet = class
  private
     FFields           : TxqFields;
     FRecNo            : Integer;
     FRecordBufferSize : Integer;
     FSourceDataSet    : TDataSet;
     FIsSequenced      : Boolean;
  protected
     function GetFieldData(Field: TxqField; Buffer: Pointer): Boolean; virtual;
     procedure SetFieldData(Field: TxqField; Buffer: Pointer); virtual; abstract;
     procedure SetRecno(Value: Integer);
     function GetRecno: Integer;
     function GetRecordCount: Integer; virtual;
     function GetSourceRecno: Integer; virtual; abstract;
     procedure SetSourceRecno(Value: Integer); virtual; abstract;
     procedure SortWithList(SortList: TxqSortList); virtual; abstract;
     procedure ClearBufferList; virtual; abstract;
  public
     { methods }
     constructor Create;
     destructor Destroy; override;
     procedure AddField( const pFieldName       : String;
                               pDataType        : TExprType;
                               pDataSize        : Integer;
                               pField           : TField;
                               pReadOnly        : Boolean;
                               pCastType        : Integer;
                               pCastLen         : Integer;
                               pUseDisplayLabel : Boolean );
     procedure Insert; virtual; abstract;
     procedure Delete; virtual; abstract;
     function FindField(const FieldName: string): TxqField;
     function FieldByName(const FieldName: string): TxqField;
     procedure InternalCalcSize;
     procedure Clear; virtual;

     { properties }
     property IsSequenced: Boolean read FIsSequenced write FIsSequenced;
     property SourceDataSet: TDataSet read FSourceDataSet write FSourceDataSet;
     property SourceRecno: Integer read GetSourceRecno write SetSourceRecno;
     property Recno: Integer read GetRecno write SetRecno;
     property RecordCount: Integer read GetRecordCount;
     property Fields: TxqFields read FFields;
  end;


{-------------------------------------------------------------------------------}
{                  Define TMemResultSet                                         }
{-------------------------------------------------------------------------------}

  TMemResultSet = class(TResultSet)
  private
     FBufferList : TList;
     function ActiveBuffer: PChar;
  protected
     function GetFieldData(Field: TxqField; Buffer: Pointer): Boolean; override;
     procedure SetFieldData(Field: TxqField; Buffer: Pointer); override;
     function GetRecordCount: Integer; override;
     function GetSourceRecno: Integer; override;
     procedure SetSourceRecno(Value: Integer); override;
     procedure SortWithList(SortList: TxqSortList); override;
     procedure ClearBufferList; override;
  public
     constructor Create;
     destructor Destroy; override;
     procedure Insert; override;
     procedure Delete; override;
     procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TFileResultSet                                        }
{-------------------------------------------------------------------------------}

  TFileResultSet = class(TResultSet)
  private
     FBufferList : TList;
     FMemMapFile : TMemMapFile;
     FTmpFile   : String;
     FBuffer     : PChar;
     function ActiveBuffer: PChar;
  protected
     function GetFieldData(Field: TxqField; Buffer: Pointer): Boolean; override;
     procedure SetFieldData(Field: TxqField; Buffer: Pointer); override;
     function GetRecordCount: Integer; override;
     function GetSourceRecno: Integer; override;
     procedure SetSourceRecno(Value: Integer); override;
     procedure SortWithList(SortList: TxqSortList); override;
     procedure ClearBufferList; override;
  public
     constructor Create(MapFileSize: Longint);
     destructor Destroy; override;
     procedure Insert; override;
     procedure Delete; override;
     procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  A TResultSet for sequenced datasets                          }
{-------------------------------------------------------------------------------}

  TSeqResultSet = class(TResultSet)
  private
     FBufferList   : TList;         { contains the source recno for the queried records         }
     FResolverList : TList;         { contains the list of resolvers for every column           }
  protected
     function GetFieldData(Field: TxqField; Buffer: Pointer): Boolean; override;
     procedure SetFieldData(Field: TxqField; Buffer: Pointer); override;
     function GetRecordCount: Integer; override;
     function GetSourceRecno: Integer; override;
     procedure SetSourceRecno(Value: Integer); override;
     procedure SortWithList(SortList: TxqSortList); override;
     procedure ClearBufferList; override;
  public
     constructor Create;
     destructor Destroy; override;
     procedure Insert; override;
     procedure Delete; override;
     procedure Clear; override;
  end;

{-------------------------------------------------------------------------------}
{                  Define TSqlAnalizer                                          }
{-------------------------------------------------------------------------------}

  TSqlAnalizer = class(TObject)
  private
    FParentAnalizer     : TSqlAnalizer;   { the parent analizer if this is a subquery       }
    FResultSet          : TResultSet;     { the Result Set                                  }
    FParams             : TParams;        { Params passed from TxQuery                      }
    FStatement          : TSqlStatement;  { The statement: SELECT, UPDATE, etc.             }
    FxQuery             : TCustomxQuery;  { Linked to this TxQuery                          }
    FDefDataSet         : TDataSet;       { the default TDataSet                            }
    FColumnList         : TColumnList;    { SELECT                                          }
    FIsDistinct         : Boolean;        { syntax: SELECT DISTINCT                         }
    FTableList          : TTableList;     { FROM                                            }
    FJoinList           : TJoinOnList;    { JOIN ON                                         }
    FLJoinCandidateList : TStringList;    { candidates for converting to join               }
    FRJoinCandidateList : TStringList;    {                                                 }
    FWhereStr           : String;         { WHERE clause expression                         }
    FMainWhereResolver  : TMainExpr;      { WHERE expression resolver class                 }
    FWhereOptimizeList: TWhereOptimizeList;{ used for optimizing the result set generation  }
    FOrderByList        : torderByList;   { ORDER BY list                                   }
    FGroupByList        : torderByList;   { GROUP BY list                                   }
    FHavingCol          : Integer;        { HAVING predicate                                }
    FSubQuery           : TSqlAnalizer;   { Sub Query (one and only one is allowed)         }
    FSubQueryKind       : TSubQueryKind;  {                                                 }
    FDoSelectAll        : Boolean;        { syntax: SELECT * FROM...                        }
    FTableAllFields     : TStringList;    { syntax: SELECT customer.* FROM customer;        }
    FUpdateColumnList   : TUpdateList;    { UPDATE statement                                }
    FInsertList         : TInsertList;    { INSERT statement                                }
    FCreateTableList   : TCreateTableList;{ CREATE TABLE statement                          }
    FIndexUnique        : Boolean;        { CREATE INDEX, DROP TABLE, DROP INDEX statements }
    FIndexDescending    : Boolean;        {                                                 }
    FIndexColumnList    : TStringList;    {                                                 }
    FIndexName          : String;         {                                                 }
    FIndexTable         : String;         {                                                 }
    FPivotStr           : String;         { syntax: TRANSFORM...PIVOT                       }
    FPivotInList        : TStringList;
    FTransformColumn    : TColumnItem;
    FTransformBaseColumns: Integer;
    FMergeAnalizer      : TSqlAnalizer;   { syntax MERGE ... with statement                 }
    FWhereFilter        : String;         { used to check if a WHERE clause can be filtered }
    FJoinNestedList     : TNestedList;

    { parameters for lex / yacc follows }
    FParser             : TCustomParser;
    FLexer              : TCustomLexer;
    FInputStream        : TMemoryStream;
    FOutputStream       : TMemoryStream;
    FErrorStream        : TMemoryStream;

    procedure SafeCreateResultSet;
    procedure CreateResultSet;
    procedure InitializeResultSet;
    function  HasDistinctAggregates: Boolean;
    function ReplaceParams(const Expr: String; var Succeed: Boolean): String;
    { define the procedure for the result set creation}
    procedure CreateJoinOnResolvers;
    procedure DoJoinOn;
    procedure AddThisRecord( pSourceRecno: Integer );
    procedure DoInsertStatement;
    procedure DoUpdateRecord;
    procedure DoGroupBy;
    procedure DoOrderBy(TheOrderList: TOrderByList);
    procedure DoTransform;
    procedure DoMerge;
    function NormalizeField(const FieldName: String): String;
  protected
    function GetResultSet: TResultSet;
    procedure SetResultSet(Value: TResultSet);
  public
    constructor Create(ParentAnalizer: TSqlAnalizer; XQuery: TCustomxQuery);
    destructor Destroy; override;
    procedure ClearSQL;
    function HasAggregates: Boolean;
    function FindDataSetByName( const Name: String): TDataSet;
    function FindFieldByName(const FieldName: String): TField;
    function QualifiedField(const FieldName: String; UseAlias: Boolean): String;
    function CheckIntegrity: Boolean;
    procedure doSelect;
    procedure doExecSQL;

    property xQuery: TCustomxQuery read FxQuery write FxQuery;
    property Params: TParams read FParams;
    property Statement: TSqlStatement read FStatement write FStatement;
    property DefDataSet: TDataSet read FDefDataSet write FDefDataSet;
    property ColumnList: TColumnList read FColumnList;
    property IsDistinct: Boolean read FIsDistinct write FIsDistinct;
    property TableList: TTableList read FTableList;
    property JoinList: TJoinOnList read FJoinList;
    property LJoinCandidateList: TStringList read FLJoinCandidateList;
    property RJoinCandidateList: TStringList read FRJoinCandidateList;
    property WhereStr: String read FWhereStr write FWhereStr;
    property WhereOptimizeList: TWhereOptimizeList read FWhereOptimizeList;
    property OrderByList: torderByList read FOrderByList;
    property GroupByList: torderByList read FGroupByList;
    property HavingCol: Integer read FHavingCol write FHavingCol;
    property SubQuery: TSqlAnalizer read FSubQuery write FSubQuery;
    property SubQueryKind: TSubQueryKind read FSubQueryKind write FSubQueryKind;
    property doSelectAll: Boolean read FDoSelectAll write FDoSelectAll;
    property TableAllFields: TStringList read FTableAllFields;
    property InsertList: TInsertList read FInsertList;
    property UpdateColumnList: TUpdateList read FUpdateColumnList;
    property CreateTableList: TCreateTableList read FCreateTableList write FCreateTableList;
    property IndexUnique: Boolean read FIndexUnique write FIndexUnique;
    property IndexDescending: Boolean read FIndexDescending write FIndexDescending;
    property IndexColumnList: TStringList read FIndexColumnList write FIndexColumnList;
    property IndexName: String read FIndexName write FIndexName;
    property IndexTable: String read FIndexTable write FIndexTable;
    property ResultSet: TResultSet read GetResultSet write SetResultSet;
    property PivotStr: String read FPivotStr write FPivotStr;
    property PivotInList: TStringList read FPivotInList;
    property TransformColumn: TColumnItem read FTransformColumn;
    property MergeAnalizer : TSqlAnalizer read FMergeAnalizer write FMergeAnalizer;

    { lex / yacc information }
    property Parser: TCustomParser read FParser;
    property Lexer : TCustomLexer read FLexer;
  end;

{-------------------------------------------------------------------------------}
{                  Define TDataSetItem                                          }
{-------------------------------------------------------------------------------}

  TxDataSetItem = class(TCollectionItem)
  private
     FDataSet  : TDataSet;
     FAlias    : String;
     procedure SetDataSet(Value: TDataSet);
  protected
     function GetDisplayName : string; override;
     function GetCaption : string;
  public
     procedure Assign(Source: TPersistent); override;
  published
     property  Alias: String read FAlias write FAlias;
     property  DataSet: TDataSet read FDataSet write SetDataSet;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxDataSets                                            }
{-------------------------------------------------------------------------------}

  TxDataSets = class(TCollection)
  private
     FxQuery: TCustomxQuery;
     FDataSetClass: TDataSetClass;
     function  GetItem(Index: Integer): TxDataSetItem;
     procedure SetItem(Index: Integer; Value: TxDataSetItem);
  protected
     function  GetOwner: TPersistent; override;
  public
     constructor Create(xquery: TCustomxQuery);
     function Add: TxDataSetItem;
     property  Items[Index: Integer]: TxDataSetItem read GetItem write SetItem; default;
     property  DataSetClass: TDataSetClass read FDataSetClass write FDataSetClass;
  end;

{-------------------------------------------------------------------------------}
{                  Define TExFunctionItem                                       }
{-------------------------------------------------------------------------------}

  TExFunctionItem = class(TCollectionItem)
  private
     FName : String;
     FResultType : TExprType;
  protected
     function GetDisplayName : string; override;
     function GetCaption : string;
  public
     procedure Assign(Source: TPersistent); override;
  published
     property  Name: String read FName write FName;
     property  ResultType: TExprType read FResultType write FResultType;
  end;

{-------------------------------------------------------------------------------}
{                  Define TExFunctions                                          }
{-------------------------------------------------------------------------------}

  TExFunctions = class(TCollection)
  private
     FxQuery   : TCustomxQuery;
     function GetItem(Index: Integer): TExFunctionItem;
     procedure SetItem(Index: Integer; Value: TExFunctionItem);
  protected
     function GetOwner: TPersistent; override;
  public
     constructor Create(xquery: TCustomxQuery);
     function Add: TExFunctionItem;
     property  Items[Index: Integer]: TExFunctionItem read GetItem write SetItem; default;
  end;

{-------------------------------------------------------------------------------}
{                  Define Events in TCustomxQuery                               }
{-------------------------------------------------------------------------------}

  TFunctionCheckEvent   = procedure(Sender     : tobject;
                              const Identifier : String;
                                    Params     : TParameterList;
                                var Accept     : Boolean) of object;

  TFunctionSolveEvent   = procedure(Sender     : tobject;
                              const Identifier : String;
                                    Params     : TParameterList;
                                var Value      : variant) of object;

  TIndexNeededForEvent  = procedure(Sender        : tobject;
                                    DataSet       : TDataSet;
                              const FieldNames    : String;
                                    ActivateIndex : Boolean;
                                var Accept        : Boolean) of object;

  TSetRangeEvent        = procedure(Sender      : tobject;
                                    RelOperator : TRelationalOperator;
                                    DataSet     : TDataSet;
                              const FieldNames,
                                    StartValues,
                                    EndValues   : String) of object;

  TCancelRangeEvent     = procedure(Sender  : tobject;
                                    DataSet : TDataSet) of object;

  TSetFilterEvent       = procedure( Sender : tobject ;
                                    DataSet : TDataSet ;
                               const Filter : String ;
                                var Handled : Boolean ) of object;

  TBlobNeededEvent      = procedure(Sender  : tobject;
                                    DataSet : TDataSet;
                                var Accept  : Boolean) of object;

  TxProgressStatus      = (psXStart,
                           psXProgress,
                           psXEnd);

  TxProgressEvent       = procedure(Sender    : tobject;
                                    Status    : TXProgressStatus;
                                    Min,
                                    Max,
                                    Position  : Integer) of object;

  TCreateTableEvent     = procedure(Sender       : tobject;
                                    CreateTable  : TCreateTableItem) of object;

  TCreateIndexEvent     = procedure(Sender         : tobject;
                                    Unique,
                                    Descending     : Boolean;
                              const TableName,
                                    IndexName      : String;
                                    ColumnExprList : TStringList) of object;

  TDropTableEvent       = procedure(Sender    : tobject;
                              const TableName : String) of object;

  TDropIndexEvent       = procedure(Sender     : TObject;
                              const TableName,
                                    IndexName  : String) of object;

  TSyntaxErrorEvent     = procedure(Sender    : tobject;
                              const ErrorMsg,
                                OffendingText : String;
                                    LineNum,
                                    ColNum,
                                    TextLen   : Integer) of object;
  TCancelQueryEvent     = procedure( Sender : TObject; var Cancel: Boolean) of object;


{-------------------------------------------------------------------------------}
{                  Define Bookmark information                                  }
{-------------------------------------------------------------------------------}

  PRecInfo = ^TRecInfo;
  TRecInfo = record
     RecordNo : Integer;
     BookmarkFlag : TBookmarkFlag;
  end;

{-------------------------------------------------------------------------------}
{                  Define TCustomxQuery dataset                                 }
{-------------------------------------------------------------------------------}

  TCustomxQuery = class( TDataSet )
  private
    { Data }
    FDataSets              : TxDataSets;     {  list of queried datasets               }
    FAllSequenced          : Boolean;        {  Means all datasets accepts RecNo prop. }
    FExFunctions           : TExFunctions;   {  addiitional FUNCTIONs added            }
    FResultSet             : TResultSet;     {  the abstract result set                }
    FRecordCount           : Integer;        {  current number of record               }
    FRecordSize            : Integer;        {  the size of the actual data            }
    FRecordBufferSize      : Integer;        {  data + housekeeping(TRecInfo)          }
    FRecordInfoOffset      : Integer;        {  offset of RecInfo in record buffer     }
    FRecno                 : Integer;        {  current record(0 to FRecordCount - 1)  }
    BofCrack               : Integer;        {  before the first record(crack)         }
    EofCrack               : Integer;        {  after the last record(crack)           }
    StartCalculated        : Integer;
    FIsOpen                : Boolean;
    FReadOnly              : Boolean;
    FSQL                   : TStrings;
    FFilterBuffer          : PChar;
    FFilterExpr            : TFilterExpr;
    FPrepared              : Boolean;
    FParams                : TParams;
    FParamCheck            : Boolean;
    FDataLink              : TDataLink;
    FAutoDisableControls,
    FUseDisplayLabel,
    FInMemResultSet        : Boolean;
    FDateFormat            : String;
    FOptimize              : Boolean;
    FDisabledDataSets      : TList;
    FMapFileSize           : Longint;           { Temporary file max size in bytes     }
    FMaxRecords            : Longint;           { Max number of records to return      }

    { events }
    FOnFunctionCheck       : TFunctionCheckEvent;
    FOnFunctionSolve       : TFunctionSolveEvent;
    FOnProgress            : TXProgressEvent;
    FOnIndexNeededFor      : TIndexNeededForEvent;
    FOnSetRange            : TSetRangeEvent;
    FOnCancelRange         : TCancelRangeEvent;
    FOnBlobNeeded          : TBlobNeededEvent;
    FOnBeforeQuery         : TNotifyEvent;
    FOnAfterQuery          : TNotifyEvent;
    FOnCreateTable         : TCreateTableEvent;
    FOnCreateIndex         : TCreateIndexEvent;
    FOnDropTable           : TDropTableEvent;
    FOnDropIndex           : TDropIndexEvent;
    FOnSyntaxError         : TSyntaxErrorEvent;
    FOnSetFilter           : TSetFilterEvent;
    FOnCancelFilter        : TCancelRangeEvent;
    FOnCancelQuery         : TCancelQueryEvent;

    function GetActiveRecordBuffer: PChar;
    function FilterRecord(Buffer: PChar): Boolean;
    procedure _ReadRecord(Buffer:PChar;IntRecNum:Integer);
    procedure SetQuery(Value: TStrings);
    function GetAbout:string;
    procedure SetAbout(const Value:String);
    procedure SetFilterData(const Text: String);
    procedure QueryChanged(Sender: tobject);
    function GetPrepared: Boolean;
    procedure SetPrepare(Value: Boolean);
    function GetParamsCount: Word;
    procedure SetParamsList(Value: TParams);
    procedure SetDataSource(Value: TDataSource);
    procedure RefreshParams;
    procedure SetParamsFromDataSet;
    function GetSourceRecno: Integer;
{$ifdef level4}
    procedure ReadParamData(Reader: TReader);
    procedure writeParamData(writer: Twriter);
{$endif}
    procedure FixDummiesForQuerying(var Filter: String);
  protected
    procedure InternalRefresh; override;
    function GetDataSource: TDataSource; override;
    function AllocRecordBuffer: PChar; override;
    procedure FreeRecordBuffer(var Buffer: PChar); override;
    procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override;
    function GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override;

    function GetRecord(Buffer: PChar; GetMode: TGetMode; doCheck: Boolean): TGetResult; override;
    function GetRecordSize: Word; override;

    function BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean; {$IFNDEF LEVEL5}override;{$endif}
    function CurrToBCD(const Curr: Currency; BCD: Pointer; Precision, Decimals: Integer): Boolean; {$IFNDEF LEVEL5}override;{$endif}

    procedure InternalClose; override;
    procedure InternalFirst; override;
    procedure InternalGotoBookmark( Bookmark: Pointer); override;
    procedure InternalInitFieldDefs; override;
    procedure InternalInitRecord(Buffer: PChar); override;
    procedure InternalLast; override;
    procedure InternalOpen; override;
    procedure InternalSetToRecord(Buffer: PChar); override;
    function IsCursorOpen: Boolean; override;
    procedure SetBookmarkFlag(Buffer: PChar;  Value: TBookmarkFlag); override;
    procedure SetBookmarkData(Buffer: PChar;  Data: Pointer); override;
    function GetCanModify: Boolean; override;
    procedure ClearCalcFields(Buffer: PChar); override;
    function GetRecordCount: Integer; override;
    procedure SetRecNo(Value: Integer); override;
    function GetRecNo: Integer; override;
    procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
    procedure InternalDelete; override;
    procedure InternalHandleException; override;
    procedure InternalPost; override;
    procedure Notification(AComponent: TComponent; Operation: toperation); override;
    procedure SetFilterText(const Value: string); override;
    procedure SetFiltered(Value: Boolean); override;
    procedure FixDummiesForFilter(var Filter: String); dynamic;
{$ifdef level4}
    procedure DefineProperties(Filer: TFiler); override;
{$endif}

{$ifdef level3}
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
{$endif}
    procedure IndexNeededFor( Sender: TObject; DataSet       : TDataSet;
                        const FieldNames    : String;
                              ActivateIndex : Boolean;
                          var Accept        : Boolean); virtual;

    procedure SetRange( Sender: TObject; RelOperator : TRelationalOperator;
                        DataSet     : TDataSet;
                  const FieldNames,
                        StartValues,
                        EndValues   : String); virtual;

    procedure CancelRange(Sender: TObject; DataSet : TDataSet); virtual;

    { properties }
    property ResultSet: TResultSet read FResultSet;
    property DataSets: TxDataSets read FDataSets write FDataSets;
    property AllSequenced: Boolean read FAllSequenced write FAllSequenced;

    { new events }
    property  OnIndexNeededFor: TIndexNeededForEvent
       read  FOnIndexNeededFor
       write FOnIndexNeededFor;
    property  OnSetRange: TSetRangeEvent
       read  FOnSetRange
       write FOnSetRange;
    property  OnCancelRange: TCancelRangeEvent
       read  FOnCancelRange
       write FOnCancelRange;
    property  OnBlobNeeded: TBlobNeededEvent
       read  FOnBlobNeeded
       write FOnBlobNeeded;
    property  OnCreateTable: TCreateTableEvent
       read  FOnCreateTable
       write FOnCreateTable;
    property  OnCreateIndex: TCreateIndexEvent
       read  FOnCreateIndex
       write FOnCreateIndex;
    property  OnDropTable: TDropTableEvent
       read  FOnDropTable
       write FOnDropTable;
    property  OnDropIndex: TDropIndexEvent
       read  FOnDropIndex
       write FOnDropIndex;
    property OnSetFilter: TSetFilterEvent
       read  FOnSetFilter
       write FOnSetFilter;
    property  OnCancelFilter: TCancelRangeEvent
       read  FOnCancelFilter
       write FOnCancelFilter;

  public
    { methods }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

{$ifdef level4}
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
    procedure SetBlockReadSize(Value: Integer); override;
{$endif}
    procedure SetFieldData(Field: TField; Buffer: Pointer); override;
    function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
    function IsSequenced: Boolean; override;
    function IsDataSetDisabled(DataSet: TDataSet): Boolean;

    function Find(const Expression: String): Boolean;
    function ResultSetIsSequenced: Boolean;
    procedure ExecSQL;
    procedure Disconnect;
    procedure Prepare;
    procedure UnPrepare;
    procedure CreateParams;
    function ParamByName(const Value: string): TParam;

    { not so common procedures }
    procedure AddDataSet(DataSet: TDataSet; const Alias: String);
    procedure ClearDataSets;
    function DataSetByName( const Name: String): TDataSet;
    function SourceDataSet: TDataSet;
    procedure WriteToTextFile( const FileName        : String ;
                                     FieldDelimChar,
                                     TxtSeparator    : Char ;
                                     IsCSV           : Boolean ;
                                     FieldNames      : TStringList ) ;

    { properties }
    property ParamCount: Word read GetParamsCount;
    property Prepared: Boolean read GetPrepared write SetPrepare;
    property ReadOnly : Boolean read FReadOnly write FReadOnly default False;
    property RecNo: Integer read GetRecNo write SetRecNo;
    property SourceRecNo: Integer read GetSourceRecNo;
    property MapFileSize: Longint read FMapFileSize write FMapFileSize default 2000000;

  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property SQL: TStrings read FSQL write SetQuery;
    property Params: TParams read FParams write SetParamsList stored False;
    property ParamCheck: Boolean read FParamCheck write FParamCheck default True;
    property About:string read GetAbout write SetAbout;
    property ExFunctions: TExFunctions read FExFunctions write FExFunctions;
    property AutoDisableControls: Boolean read FAutoDisableControls write FAutoDisableControls default True;
    property UseDisplayLabel: Boolean read FUseDisplayLabel write FUseDisplayLabel default False;
    property InMemResultSet: Boolean read FInMemResultSet write FInMemResultSet default True;
    property DateFormat: String read FDateFormat write FDateFormat;
    property Optimize: Boolean read FOptimize write FOptimize default True;
    property MaxRecords: Longint read FMaxRecords write FMaxRecords;
    { inherited properties }
    property Active;
    property Filter;
    property Filtered;

    {new events}
    property  OnFunctionCheck: TFunctionCheckEvent
       read FOnFunctionCheck
       write FOnFunctionCheck;
    property  OnFunctionSolve: TFunctionSolveEvent
       read FOnFunctionSolve
       write FOnFunctionSolve;
    property  OnProgress: TXProgressEvent
       read FOnProgress
       write FOnProgress;
    property  OnBeforeQuery: TNotifyEvent
       read FOnBeforeQuery
       write FOnBeforeQuery;
    property  OnAfterQuery: TNotifyEvent
       read FOnAfterQuery
       write FOnAfterQuery;
    property  OnSyntaxError: TSyntaxErrorEvent
       read FOnSyntaxError
       write FOnSyntaxError;
    property  OnCancelQuery: TCancelQueryEvent
       read  FOnCancelQuery
       write FOnCancelQuery;

    { inherited events }
    property BeforeOpen;
    property AfterOpen;
    property BeforeClose;
    property AfterClose;
    property BeforeInsert;
    property AfterInsert;
    property BeforeEdit;
    property AfterEdit;
    property BeforePost;
    property AfterPost;
    property BeforeCancel;
    property AfterCancel;
    property BeforeDelete;
    property AfterDelete;
    property BeforeScroll;
    property AfterScroll;
    property OnCalcFields;
    property OnDeleteError;
    property OnEditError;
    property OnFilterRecord;
    property OnNewRecord;
    property OnPostError;
  end;

{-------------------------------------------------------------------------------}
{                  Define TxQuery                                               }
{-------------------------------------------------------------------------------}

  TxQuery = class(TCustomXQuery)
  published
     { properties }
     property DataSets;
     property AllSequenced;
     property MapFileSize;

     { events }
     property OnIndexNeededFor;
     property OnSetRange;
     property OnCancelRange;
     property OnBlobNeeded;
     property OnCreateTable;
     property OnCreateIndex;
     property OnDropTable;
     property OnDropIndex;
     property OnSetFilter;
     property OnCancelFilter;
  end;


  {$ifdef XQDEMO}
  procedure ShowAbout;
  {$endif}

{$ifdef level3}
const
  ftNonTextTypes = [ ftBytes, ftvarBytes, ftBlob, ftMemo, ftGraphic, ftFmtMemo,
                     ftParadoxOle, ftDBaseOle, ftTypedBinary, ftCursor ];
{$endif}


implementation

uses
   {$ifdef XQDEMO}
   DemoReg,
   {$endif}
   xqLex, xqYacc, xqConsts;

{-------------------------------------------------------------------------------}
{                  Define as a demo section                                     }
{-------------------------------------------------------------------------------}

{$ifdef xqdemo}
var
  IsFirstTime: Boolean = True;

{*******************************************************************************}
procedure ShowAbout;
begin
   with TfrmRegister.Create(Application) do
   begin
      try
         ShowModal;
      finally
         free;
      end;
   end;
end;
{$endif}


{-------------------------------------------------------------------------------}
{                  Implements TxqField                                          }
{-------------------------------------------------------------------------------}

constructor TxqField.Create(Fields: TxqFields; FieldNo: Integer);
begin
  inherited Create;
  FFields  := Fields;
  FFieldNo := FieldNo;
end;

function TxqField.GetData( Buffer: Pointer ): Boolean;
begin
  Result := FFields.ResultSet.GetFieldData( Self, Buffer );
end;

procedure TxqField.SetData( Buffer: Pointer );
begin
  FFields.ResultSet.SetFieldData( Self, Buffer );
end;

procedure TxqField.SetDataType( Value: TExprType );
begin
  FDataType := Value;
end;

function TxqField.GetAsBoolean: Boolean;
begin
  raise EXQueryError.Create( SReadBooleanField );
end;

function TxqField.GetAsFloat: double;
begin
  raise EXQueryError.Create( SReadFloatField );
end;

function TxqField.GetAsInteger: Longint;
begin
  raise EXQueryError.Create( SReadIntegerField );
end;

function TxqField.GetAsString: string;
begin
  raise EXQueryError.Create( SReadStringField );
end;

procedure TxqField.SetAsString( const Value: String );
begin
  raise EXQueryError.Create(SwriteStringField);
end;

procedure TxqField.SetAsFloat( Value: double );
begin
  raise EXQueryError.Create(SwriteFloatField);
end;

procedure TxqField.SetAsInteger(Value: Longint);
begin
  raise EXQueryError.Create(SwriteIntegerField);
end;

procedure TxqField.SetAsBoolean(Value: Boolean);
begin
  raise EXQueryError.Create(SwriteBooleanField);
end;

function TxqField.GetColWidth: Integer;
begin
   if Assigned(FSourceField) then
      Result := FSourceField.DataSize
   else
      Result := FDataSize;
end;

{-------------------------------------------------------------------------------}
{                  Implements TxqStringField                                    }
{-------------------------------------------------------------------------------}

constructor TxqStringField.Create(Fields: TxqFields; FieldNo: Integer);
begin
   inherited Create(Fields, FieldNo);
   SetDataType(ttString);
end;

procedure TxqStringField.Clear;
begin
  SetAsString('');
end;

function TxqStringField.GetValue(var Value: string): Boolean;
var
  Buffer: array[0..dsMaxStringSize] of Char;
begin
  Result := GetData(@Buffer);
  if Result then
    Value := Buffer;
end;

function TxqStringField.GetAsString: string;
begin
  if not GetValue(Result) then Result := '';
end;

function TxqStringField.GetAsFloat: double;
begin
  Result := StrToFloat(GetAsString);
end;

function TxqStringField.GetAsInteger: Longint;
begin
  Result := StrToInt(GetAsString);
end;

function TxqStringField.GetAsBoolean: Boolean;
var
  S: string;
begin
  S := GetAsString;
  Result := (Length(S) > 0) and (S[1] in ['T', 't', 'Y', 'y']);
end;

procedure TxqStringField.SetAsString(const Value: string);
var
  Buffer: array[0..dsMaxStringSize] of Char;
  L : Integer;
begin
  FillChar(Buffer, FDataSize, 0);
  L := Length(Value);
  StrLCopy(Buffer, PChar(Value), L);
  SetData(@Buffer);
end;

procedure TxqStringField.SetAsFloat(Value: double);
begin
  SetAsString(FloatToStr(Value));
end;

procedure TxqStringField.SetAsInteger(Value: Longint);
begin
  SetAsString(IntToStr(Value));
end;

procedure TxqStringField.SetAsBoolean(Value: Boolean);
begin
  SetAsString( Copy(xqbase.NBoolean[Value], 1, 1) );
end;

{-------------------------------------------------------------------------------}
{                  Implements TxqFloatField                                     }
{-------------------------------------------------------------------------------}

constructor TxqFloatField.Create(Fields: TxqFields; FieldNo: Integer);
begin
  inherited Create(Fields, FieldNo);
  SetDataType(ttFloat);
end;

procedure TxqFloatField.Clear;
begin
  SetAsFloat(0);
end;

function TxqFloatField.GetAsFloat: double;
begin
  if not GetData(@Result) then Result := 0;
end;

function TxqFloatField.GetAsInteger: Longint;
begin
  Result := Longint(Round(GetAsFloat));
end;

function TxqFloatField.GetAsString: string;
var
  F: double;
begin
  if GetData(@F) then Result := FloatToStr(F) else Result := '';
end;

procedure TxqFloatField.SetAsFloat(Value: double);
begin
  SetData(@Value);
end;

procedure TxqFloatField.SetAsInteger(Value: Longint);
begin
  SetAsFloat(Value);
end;

procedure TxqFloatField.SetAsString(const Value: string);
var
  F: Extended;
begin
  if Value = '' then Clear else
  begin
    if not TextToFloat(PChar(Value), F, fvExtended) then
      EXQueryError.CreateFmt(SIsInvalidFloatValue, [Value]);
    SetAsFloat(F);
  end;
end;

{-------------------------------------------------------------------------------}
{                  Implements TxqIntegerField                                   }
{-------------------------------------------------------------------------------}

constructor TxqIntegerField.Create(Fields: TxqFields; FieldNo: Integer);
begin
   inherited Create(Fields, FieldNo);
   SetDataType(ttInteger);
end;

procedure TxqIntegerField.Clear;
begin
  SetAsInteger(0);
end;

function TxqIntegerField.GetAsFloat: double;
begin
   Result := GetAsInteger;
end;

function TxqIntegerField.GetAsInteger: Longint;
begin
   if not GetData(@Result) then Result := 0;
end;

function TxqIntegerField.GetAsString: string;
var
  L: Longint;
begin
  if GetData(@L) then Str(L, Result) else Result := '';
end;

procedure TxqIntegerField.SetAsFloat(Value: double);
begin
  SetAsInteger(Integer(Round(Value)));
end;

procedure TxqIntegerField.SetAsInteger(Value: Longint);
begin
  SetData(@Value);
end;

procedure TxqIntegerField.SetAsString(const Value: string);
var
  E: Integer;
  L: Longint;
begin
  Val(Value, L, E);
  if E <> 0 then EXQueryError.CreateFmt(SIsInvalidIntegerValue, [Value]);
  SetAsInteger(L);
end;

{-------------------------------------------------------------------------------}
{                  Implements TxqBooleanField                                   }
{-------------------------------------------------------------------------------}

constructor TxqBooleanField.Create(Fields: TxqFields; FieldNo: Integer);
begin
   inherited Create(Fields, FieldNo);
   SetDataType(ttBoolean);
end;

procedure TxqBooleanField.Clear;
begin
  SetAsBoolean(False);
end;

function TxqBooleanField.GetAsBoolean: Boolean;
var
  B: WordBool;
begin
  if GetData(@B) then Result := B else Result := False;
end;

function TxqBooleanField.GetAsString: string;
var
  B: WordBool;
begin
  if GetData(@B) then Result := Copy(xqbase.NBoolean[B], 1, 1) else Result := '';
end;

procedure TxqBooleanField.SetAsBoolean(Value: Boolean);
var
  B: WordBool;
begin
  if Value then Word(B) := 1 else Word(B) := 0;
  SetData(@B);
end;

procedure TxqBooleanField.SetAsString(const Value: string);
var
  L: Integer;
begin
  L := Length(Value);
  if L = 0 then
  begin
    SetAsBoolean(False);
  end else
  begin
    if AnsiCompareText(Value, Copy(xqbase.NBoolean[False], 1, L)) = 0 then
      SetAsBoolean(False)
    else
      if AnsiCompareText(Value, Copy(xqbase.NBoolean[True], 1, L)) = 0 then
        SetAsBoolean(True)
      else
        EXQueryError.CreateFmt(SIsInvalidBoolValue, [Value]);
  end;
end;

{ TxqFields }
constructor TxqFields.Create(ResultSet: TResultSet);
begin
   inherited Create;
   FResultSet := ResultSet;
   FItems := TList.Create;
end;

destructor TxqFields.Destroy;
begin
   Clear;
   FItems.Free;
   inherited Destroy;
end;

function TxqFields.FindField(const FieldName: string): TxqField;
var
  I: Integer;
begin
  for I := 0 to FItems.Count - 1 do
  begin
    Result := FItems[I];
    if AnsiCompareText(Result.FFieldName, FieldName) = 0 then Exit;
  end;
  Result := nil;
end;

function TxqFields.GetCount;
begin
   Result := FItems.Count;
end;

function TxqFields.GetItem(Index: Integer): TxqField;
begin
   Result := FItems[Index];
end;

function TxqFields.Add(DataType : TExprType): TxqField;
begin
   case DataType of
      ttString  : Result := TxqStringField.Create(Self, FItems.Count + 1);
      ttFloat   : Result := TxqFloatField.Create(Self, FItems.Count + 1);
      ttInteger : Result := TxqIntegerField.Create(Self, FItems.Count + 1);
      ttBoolean : Result := TxqBooleanField.Create(Self, FItems.Count + 1);
      ttUnknown : Result := nil; { error here if returned nil !}
   end;
   FItems.Add(Result);
end;

procedure TxqFields.Clear;
var
   I: Integer;
begin
   for I := 0 to FItems.Count - 1 do
      TxqField(FItems[I]).Free;
   FItems.Clear;
end;

procedure TxqFields.Delete(Index: Integer);
begin
   TxqField(FItems[Index]).Free;
   FItems.Delete(Index);
end;

{-------------------------------------------------------------------------------}
{                  Implements TResultSet                                        }
{-------------------------------------------------------------------------------}

constructor TResultSet.Create;
begin
   inherited Create;
   FFields := TxqFields.Create(Self);
   FRecNo := -1;
   { first 4 bytes is the SourceRecNo for every table}
   FRecordBufferSize := SizeOf(Integer);
end;

{*******************************************************************************}
destructor TResultSet.Destroy;
begin
   FFields.Free;
   inherited Destroy;
end;

{*******************************************************************************}
procedure TResultSet.Clear;
begin
   FFields.Clear;
end;

{*******************************************************************************}
function TResultSet.FindField(const FieldName: string): TxqField;
begin
  Result := FFields.FindField(FieldName);
end;

{*******************************************************************************}
function TResultSet.FieldByName(const FieldName: string): TxqField;
begin
  Result := FindField(FieldName);
  if Result = nil then EXQueryError.CreateFmt(SFieldNotFound,[FieldName]);
end;

{*******************************************************************************}
function TResultSet.GetFieldData(Field: TxqField; Buffer: Pointer): Boolean;
begin
   Result := False;
end;

{*******************************************************************************}
procedure TResultSet.AddField( const pFieldName : String;
                               pDataType        : TExprType;
                               pDataSize        : Integer;
                               pField           : TField;
                               pReadOnly        : Boolean;
                               pCastType        : Integer;
                               pCastLen         : Integer;
                               pUseDisplayLabel : Boolean );
begin
   with FFields.Add(pDataType) do
   begin
      FieldName       := pFieldName;
      DataType        := pDataType;
      SourceField     := pField;
      ReadOnly        := pReadOnly;
      CastType        := pCastType;
      CastLen         := pCastLen;
      UseDisplayLabel := pUseDisplayLabel;
      case CastType of
         RW_CHAR      : DataType := ttString;
         RW_INTEGER   : DataType := ttInteger;
         RW_BOOLEAN   : DataType := ttBoolean;
         RW_DATE,
         RW_DATETIME,
         RW_TIME,
         RW_FLOAT,
         RW_MONEY     : DataType := ttFloat;
      end;
      { Calculate the position in the record buffer }
      BufferOffset := FRecordBufferSize;
      case DataType of
         ttString:
            begin
            Inc(FRecordBufferSize, pDataSize);
            DataSize := pDataSize;
            end;
         ttFloat:
            begin
            Inc(FRecordBufferSize, SizeOf(Double));
            DataSize := SizeOf(Double);
            end;
         ttInteger:
            begin
            Inc(FRecordBufferSize, SizeOf(Integer));
            DataSize := SizeOf(Integer);
            end;
         ttBoolean:
            begin
            Inc(FRecordBufferSize, SizeOf(WordBool));
            DataSize := SizeOf(WordBool);
            end;
      end;
   end;
end;

{*******************************************************************************}
function TResultSet.GetRecno: Integer;
begin
   Result := FRecNo;
end;

{*******************************************************************************}
procedure TResultSet.SetRecno(Value: Integer);
begin
   FRecNo := Value;
   if Value = fRecno then Exit;
   if (Value < 1) or (Value > GetRecordCount) then
      raise EXQueryError.Create(SRecnoInvalid);
   FRecno := Value;
end;

{*******************************************************************************}
function TResultSet.GetRecordCount: Longint;
begin
  Result := 0;
end;

{*******************************************************************************}
procedure TResultSet.InternalCalcSize;
var
   Field     : TxqField;
   I, Offset : Integer;
begin
   Offset := 0;
   for I := 0 to FFields.Count - 1 do
   begin
      Field := FFields[I];
      with Field do
      begin
         case DataType of
            ttString : DataSize := ColWidth + 1;
            ttFloat  : DataSize := SizeOf(Double);
            ttInteger: DataSize := SizeOf(Integer);
            ttBoolean: DataSize := SizeOf(WordBool);
         end;
         (* if Assigned(Field.SourceField) and
            (Field.SourceField.DataType) in ftNonTextTypes) then
            Field.DataSize := 0; *)
         FFieldOffset := Offset;
         Inc(Offset, DataSize);
      end;
   end;
end;

{-------------------------------------------------------------------------------}
{                  Implements TMemResultSet                                     }
{-------------------------------------------------------------------------------}

constructor TMemResultSet.Create;
begin
   inherited Create;
   FBufferList := TList.Create;
end;

{*******************************************************************************}
destructor TMemResultSet.Destroy;
begin
   Clear;
   FBufferList.Free;
   inherited Destroy;
end;

{*******************************************************************************}
procedure TMemResultSet.Clear;
begin
   inherited Clear;
   ClearBufferList;
end;

{*******************************************************************************}
procedure TMemResultSet.SortWithList(SortList: TxqSortList);
var
  vTempList : TList;
  I         : Integer;
begin
  vTempList := TList.Create;
  for I := 1 to SortList.Count do
  begin
     SortList.Recno := I;
     vTempList.Add(FBufferList[SortList.SourceRecno - 1]);
  end;
  FBufferList.Free;
  FBufferList := vTempList;
end;

{*******************************************************************************}
procedure TMemResultSet.ClearBufferList;
var
   I     : Integer;
   Buffer: PChar;
begin
   for I := 0 to FBufferList.Count - 1 do
   begin
      Buffer := FBufferList[I];
      FreeMem(Buffer, FRecordBufferSize);
   end;
   FBufferList.Clear;
end;

{*******************************************************************************}
function TMemResultSet.ActiveBuffer: PChar;
begin
   Result := nil;
   if (FRecNo < 1) or (FRecNo > FBufferList.Count) then Exit;
   Result := FBufferList[FRecNo - 1];
end;

{*******************************************************************************}
function TMemResultSet.GetFieldData(Field: TxqField; Buffer: Pointer): Boolean;
var
   RecBuf: PChar;
begin
   Result := False;
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( (RecBuf + Field.BufferOffset)^, Buffer^, Field.DataSize);
   Result := True;
end;

{*******************************************************************************}
procedure TMemResultSet.SetFieldData(Field: TxqField; Buffer: Pointer);
var
   RecBuf     : PChar;
begin
   RecBuf := ActiveBuffer;
   if (RecBuf = nil) or (Buffer = nil) then Exit;
   Move( Buffer^, (RecBuf + Field.BufferOffset)^, Field.DataSize);
end;

{*******************************************************************************}
function TMemResultSet.GetRecordCount;
begin
   Result := FBufferList.Count;
end;

{*******************************************************************************}
function TMemResultSet.GetSourceRecno: Integer;
var
   Buffer: PChar;
begin
   Result := 0;
   if (FRecno <1) or (FRecno > GetRecordCount) then Exit;
   Buffer := PChar(FBufferList[FRecno - 1]);
   Move((Buffer + 0)^, Result, SizeOf(Integer));
end;

{*******************************************************************************}
procedure TMemResultSet.SetSourceRecno(Value: Integer);
var
   Buffer: PChar;
begin
   if (FRecno <1) or (FRecno >GetRecordCount) then Exit;
   Buffer := PChar(FBufferList[FRecno - 1]);
   Move(Value, (Buffer + 0)^, SizeOf(Integer));
end;

{*******************************************************************************}
procedure TMemResultSet.Insert;
var
   Buffer: PChar;
begin
   GetMem(Buffer, FRecordBufferSize);
   FillChar(Buffer^,FRecordBufferSize,0);
   FBufferList.Add( Buffer );
   FRecno := FBufferList.Count;
end;

{*******************************************************************************}
procedure TMemResultSet.Delete;
var
   Buffer: PChar;
begin
   if (FRecno < 1) or (RecNo > GetRecordCount) then Exit;
   Buffer := FBufferList[FRecno - 1];
   FreeMem(Buffer, FRecordBufferSize);
   FBufferList.Delete( FRecno - 1 );
   if FRecno > GetRecordCount then FRecno := GetRecordCount;
end;

{-------------------------------------------------------------------------------}
{                  Implements TFileResultSet                                    }
{-------------------------------------------------------------------------------}

constructor TFileResultSet.Create(MapFileSize: Longint);
begin
   inherited Create;
   FBufferList := TList.Create;
   { auxiliary files }
   FTmpFile    := GetTemporaryFileName('~dt');
   FMemMapFile := TMemMapFile.Create(FTmpFile, fmCreate, MapFileSize, True);
end;

{*******************************************************************************}
destructor TFileResultSet.Destroy;
begin
   if Assigned(FBuffer) then
      FreeMem(FBuffer, FRecordBufferSize);
   Clear;
   FBufferList.Free;
   inherited Destroy;
end;

{*******************************************************************************}
procedure TFileResultSet.Clear;
begin
   inherited Clear;
   FreeObject(FMemMapFile);
   SysUtils.DeleteFile(FTmpFile);
   FBufferList.Clear;
end;

{*******************************************************************************}
procedure TFileResultSet.SortWithList(SortList: TxqSortList);
var
  vTempList : TList;
  I         : Integer;
begin
  vTempList := TList.Create;
  for I := 1 to SortList.Count do
  begin
     SortList.Recno := I;
     vTempList.Add(FBufferList[SortList.SourceRecno - 1]);
  end;
  FBufferList.Free;
  FBufferList := vTempList;
end;

{*******************************************************************************}
function TFileResultSet.ActiveBuffer: PChar;
begin
   Result := nil;
   if (FRecNo < 1) or (FRecNo > FBufferList.Count) then Exit;
   if not Assigned(FBuffer) then
      GetMem(FBuffer, FRecordBufferSize);
   FMemMapFile.Seek(Longint(FBufferList[FRecno - 1]), 0);
   FMemMapFile.Read(FBuffer^, FRecordBufferSize);
   Result := FBuffer;
end;

{*******************************************************************************}
function TFileResultSet.GetFieldData(Field: TxqField; Buffer: Pointer): Boolean;
var
   RecBuf: PChar;
begin
   Result := False;
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( (RecBuf + Field.BufferOffset)^, Buffer^, Field.DataSize);
   Result := True;
end;

{*******************************************************************************}
procedure TFileResultSet.SetFieldData(Field: TxqField; Buffer: Pointer);
var
   RecBuf: PChar;
begin
   RecBuf := ActiveBuffer;
   if RecBuf = nil then Exit;
   Move( Buffer^, (RecBuf + Field.BufferOffset)^, Field.DataSize);
   FMemMapFile.Seek(Longint(FBufferList[Recno - 1]), 0);
   FMemMapFile.Write(RecBuf^, FRecordBufferSize);
end;

{*******************************************************************************}
function TFileResultSet.GetRecordCount;
begin
   Result := FBufferList.Count;
end;

{*******************************************************************************}
function TFileResultSet.GetSourceRecno: Integer;
begin
   Result := 0;
   if (FRecno <1) or (FRecno > GetRecordCount) then Exit;
   FMemMapFile.Seek(Longint(FBufferList[FRecno - 1]), 0);
   FMemMapFile.Read(Result,SizeOf(Integer));
end;

{*******************************************************************************}
procedure TFileResultSet.SetSourceRecno(Value: Integer);
begin
   if (FRecno <1) or (FRecno >GetRecordCount) then Exit;
   FMemMapFile.Seek(Longint(FBufferList[FRecno - 1]), 0);
   FMemMapFile.Write(Value, SizeOf(Integer));
end;

{*******************************************************************************}
procedure TFileResultSet.Insert;
var
   Offset: Integer;
begin
   if not Assigned(FBuffer) then
      GetMem(FBuffer, FRecordBufferSize);
   FillChar(FBuffer^,FRecordBufferSize,0);
   Offset := FMemMapFile.VirtualSize;
   FMemMapFile.Seek(Offset,0);
   FMemMapFile.Write(FBuffer^,FRecordBufferSize);
   FBufferList.Add(Pointer(Offset)); { the address in temp file is saved }
   FRecno := FBufferList.Count;
end;

{*******************************************************************************}
procedure TFileResultSet.Delete;
begin
  if (FRecno <1) or (RecNo >GetRecordCount) then Exit;
  FBufferList.Delete( FRecno - 1 );
  if FRecno > GetRecordCount then FRecno := GetRecordCount;
end;

{*******************************************************************************}
procedure TFileResultSet.ClearBufferList;
begin
end;

{-------------------------------------------------------------------------------}
{                  Implements TSeqResultSet                                     }
{-------------------------------------------------------------------------------}

constructor TSeqResultSet.Create;
begin
  inherited Create;
  FBufferList   := TList.Create;
  FResolverList := TList.Create;
end;

{*******************************************************************************}
destructor TSeqResultSet.Destroy;
begin
  Clear;
  FBufferList.Free;
  FResolverList.Free;
  inherited Destroy;
end;

{*******************************************************************************}
procedure TSeqResultSet.Clear;
var
  I: Integer;
  Resolver: TMainExpr;
begin
  inherited Clear;
  ClearBufferList;
  for I := 0 to FResolverList.Count - 1 do
  begin
     Resolver := TMainExpr(FResolverList[I]);
     if Assigned(Resolver) then
        Resolver.Free;
  end;
  FResolverList.Clear;
end;

{*******************************************************************************}
procedure TSeqResultSet.SortWithList(SortList: TxqSortList);
var
  vTempList : TList;
  I         : Integer;
begin
  vTempList := TList.Create;
  for I := 1 to SortList.Count do
  begin
     SortList.Recno := I;
     vTempList.Add(FBufferList[SortList.SourceRecno - 1]);
  end;
  FBufferList.Free;
  FBufferList := vTempList;
end;

{*******************************************************************************}
procedure TSeqResultSet.ClearBufferList;
begin
  FBufferList.Clear;
end;

{*******************************************************************************}
function TSeqResultSet.GetFieldData(Field: TxqField; Buffer: Pointer): Boolean;
var
  Resolver    : TMainExpr;
  vTempRecNo  : Integer;
  vFloatValue : double;
  vIntValue   : Integer;
  vBoolValue  : WordBool;
begin
  vTempRecNo := Longint(FBufferList[RecNo - 1]);
  { if it is a blob field, then it is requesting the source recno only}
  if Assigned(Field.SourceField) and (Field.SourceField is TBlobField) then
  begin
     Move(vTempRecNo, Buffer^, SizeOf(Integer));
     Result := True;
     Exit;
  end;
  { Set position to the original record no }
  SetRecordNumber(SourceDataSet, vTempRecNo);

  Resolver := TMainExpr(FResolverList[Field.FieldNo - 1]);
  case Field.DataType OF
     ttString :
        StrPCopy(PChar(Buffer), Resolver.Expression.AsString);
     ttFloat  :
        begin
        vFloatValue := Resolver.Expression.AsFloat;
        Move(vFloatValue, Buffer^, SizeOf(Double))
        end;
     ttInteger:
        begin
        vIntValue := Resolver.Expression.AsInteger;
        Move(vIntValue, Buffer^, SizeOf(Integer))
        end;
     ttBoolean:
        begin
        vBoolValue := Resolver.Expression.AsBoolean;
        Move(vBoolValue, Buffer^, SizeOf(WordBool))
        end;
  end;
  Result := True;
end;

{*******************************************************************************}
procedure TSeqResultSet.SetFieldData(Field: TxqField; Buffer: Pointer);
{var
   TempBuf: PChar;
   TmpRecNo: Integer; }
begin
   { in this kind of result set, the assignations to fields will be done      }
   { for TBlob fields only because other fields will be evaluated on the fly  }
   {if (Field.DataType = ttInteger) AND Assigned(Field.SourceField) AND
      (Field.SourceField IS TBlobField) then
   begin
      if FWithBuffer[RecNo - 1] then
         RecBuf := ActiveBuffer
      else
         RecBuf := CreateNewBuffer;
      if (RecBuf = nil) or (Buffer = nil) then Exit;
      Move( Buffer^, (RecBuf + Field.BufferOffset)^, Field.DataSize);
   end; }
end;

{*******************************************************************************}
function TSeqResultSet.GetRecordCount;
begin
  Result := FBufferList.Count;
end;

{*******************************************************************************}
function TSeqResultSet.GetSourceRecno: Integer;
begin
   Result := 0;
   if (Recno < 1) OR (Recno > GetRecordCount) then Exit;
   Result := Longint(FBufferList[RecNo - 1]);
end;

{*******************************************************************************}
procedure TSeqResultSet.SetSourceRecno(Value: Integer);
begin
   if (Recno <1) or (Recno >GetRecordCount) then Exit;
   FBufferList[RecNo - 1] := Pointer(Value);
end;

{*******************************************************************************}
procedure TSeqResultSet.Insert;
begin
   FBufferList.Add( nil );
   Recno := FBufferList.Count;
end;

{*******************************************************************************}
procedure TSeqResultSet.Delete;
begin
   if (Recno < 1) or (RecNo > GetRecordCount) then Exit;
   FBufferList.Delete( Recno - 1 );
   if Recno > GetRecordCount then Recno := GetRecordCount;
end;

{-------------------------------------------------------------------------------}
{                  Implements TSqlAnalizer                                      }
{-------------------------------------------------------------------------------}

constructor TSqlAnalizer.Create(ParentAnalizer: TSqlAnalizer; XQuery: TCustomxQuery);
var
   Temps: String;
begin
   inherited Create;
   FParentAnalizer     := ParentAnalizer;
   FxQuery             := XQuery;
   FParams             := TParams.Create{$ifndef level3}(FxQuery){$endif};
   FColumnList         := TColumnList.Create;
   FTableList          := TTableList.Create;
   FOrderByList        := torderByList.Create;
   FGroupByList        := torderByList.Create;
   FJoinList           := TJoinOnList.Create;
   FLJoinCandidateList := TStringList.Create;
   FRJoinCandidateList := TStringList.Create;
   FTableAllFields     := TStringList.Create;
   FInsertList         := TInsertList.Create;
   FUpdateColumnList   := TUpdateList.Create;
   FWhereOptimizeList  := TWhereOptimizeList.Create;
   FCreateTableList    := TCreateTableList.Create;
   FIndexColumnList    := TStringList.Create;
   FTransformColumn    := TColumnItem.Create(nil);
   FPivotInList        := TStringList.Create;
   FHavingCol          := -1;
   FJoinNestedList     := TNestedList.Create;

   { create lex / yacc information }
   Temps              := FxQuery.SQL.Text;
   FInputStream       := TMemoryStream.Create;
   FInputStream.WriteBuffer(Pointer(Temps)^, Length(Temps));
   FInputStream.Seek( 0, 0	);
   FOutputStream      := TMemoryStream.Create;
   FErrorStream       := TMemoryStream.Create;
   FLexer             := TxqLexer.Create;
   FLexer.yyinput     := FInputStream;
   FLexer.yyoutput    := FOutputStream;
   FLexer.yyerrorfile := FErrorStream;
   (FLexer as TxqLexer).DateFormat := FxQuery.DateFormat;
   FParser            := TxqParser.Create(Self);  { parser and Analizer linked }
   FParser.yyLexer    := FLexer;   { link lexer and parser }

end;

{*******************************************************************************}
destructor TSqlAnalizer.Destroy;

   { recursively free subqueries }
   procedure FreeSubQueries(var SubQ: TSqlAnalizer);
   begin
     if Assigned(SubQ.FSubQuery) then
        FreeSubQueries(SubQ.FSubQuery);
     FreeObject(SubQ);
   end;

begin
   ClearSQL;
   FColumnList.Free;
   FTableList.Free;
   FOrderByList.Free;
   FGroupByList.Free;
   if Assigned(FSubQuery) then
      FreeSubQueries(FSubQuery);
   if Assigned(FResultSet) then
      FResultSet.Free;
   FTableAllFields.Free;
   FJoinList.Free;
   FLJoinCandidateList.Free;
   FRJoinCandidateList.Free;
   FUpdateColumnList.Free;
   FInsertList.Free;
   FWhereOptimizeList.Free;
   FParams.Free;
   FCreateTableList.Free;
   FIndexColumnList.Free;
   FTransformColumn.Free;
   FPivotInList.Free;
   if Assigned(FMergeAnalizer) then
      FreeObject(FMergeAnalizer);
   FJoinNestedList.Free;

   { free lex / yacc information }
   FParser.free;
   FLexer.free;
   FInputStream.free;
   FOutputStream.free;
   FErrorStream.free;

   inherited Destroy;
end;

{*******************************************************************************}
procedure TSqlAnalizer.ClearSQL;
begin
   ColumnList.Clear;
   TableList.Clear;
   OrderByList.Clear;
   GroupByList.Clear;
   JoinList.Clear;
   UpdateColumnList.Clear;
   WhereOptimizeList.Clear;
   CreateTableList.Clear;
   if Assigned(FSubQuery) then
      FSubQuery.ClearSQL;
   if Assigned(ResultSet) then
      ResultSet.Clear;
end;

{*******************************************************************************}
procedure TSqlAnalizer.CreateJoinOnResolvers;
var
   I, J : Integer;
begin
   for I := 0 to FJoinList.Count - 1 do
      with FJoinList[I] do
      begin
         for J := 0 to LeftJoinOn.Count - 1 do
         begin
            { create join expressions }
            LeftJoinOn[J].Resolver := TMainExpr.Create(Self, FDefDataSet);
            try
              LeftJoinOn[J].Resolver.ParseExpression(LeftJoinOn[J].Expression);
            except
              LeftJoinOn[J].Resolver.Free;
              LeftJoinOn[J].Resolver := nil;
              raise;
            end;
            try
                RightJoinOn[J].Resolver := TMainExpr.Create(Self, FDefDataSet);
                RightJoinOn[J].Resolver.ParseExpression(RightJoinOn[J].Expression);
            except
                LeftJoinOn[J].Resolver.Free; LeftJoinOn[J].Resolver := nil;
                RightJoinOn[J].Resolver.Free; RightJoinOn[J].Resolver := nil;
                raise;
            end;

            if Length(AndJoinOn[J].Expression) > 0 then
            begin
               AndJoinOn[J].Resolver := TMainExpr.Create(Self, FDefDataSet);
               try
                 AndJoinOn[J].Resolver.ParseExpression(AndJoinOn[J].Expression);
                 if not (AndJoinOn[J].Resolver.Expression.ExprType = ttBoolean) then
                     raise EXQueryError.Create(SExprNotBoolean);
               except
                 AndJoinOn[J].Resolver.Free; AndJoinOn[J].Resolver := nil;
                 raise;
               end;
            end;
         end;
      end;
end;

{*******************************************************************************}
{ JOIN ON clause }
procedure TSqlAnalizer.DoJoinOn;

   {---------------  Recursively joining -------------------------------------}
   procedure RecursiveJoin( Start : Integer );
   var
      TmpDataSet          : TDataSet;
      vI, vJ              : Integer;
      vNumAccepted        : Integer;
      vNumPartial         : Integer;
      vFieldNames         : String;
      vLeftAsString       : String;
      vRightAsString      : String;
      vTemps              : String;
      vJoinWasEventCalled : Boolean;
      vAccept, vPassed    : Boolean;
      vLeftAsBoolean      : Boolean;
      vRightAsBoolean     : Boolean;
      vAccepted           : Boolean;
      vLeftAsFloat        : Double;
      vRightAsFloat       : Double;
      vLeftAsInteger      : Double;
      vRightAsInteger     : Double;
   begin
      with FJoinNestedList[Start].JoinOnItem do
      begin
         TmpDataSet := FTableList[FJoinNestedList[Start].Idx2].DataSet;
         { First I must fire event FOnSetRange in order to allow to the developer to
           set a range before looking in RightDataSet giving the chance
           to speed up the result set generation }
         vPassed:= True;
         for vJ := 0 to LeftJoinOn.Count - 1 do
            if RightJoinOn[vJ].Resolver.CheckData.FieldCount = 0 then
            begin
               vPassed := False;
               Break;
            end;
         vJoinWasEventCalled := False;
         if FxQuery.Optimize {and Assigned(FxQuery.OnIndexNeededFor) and
              Assigned(FxQuery.OnSetRange)} and vPassed then
         begin
            vFieldNames := '';
            for vJ := 0 to LeftJoinOn.Count - 1 do
               for vI := 1 to RightJoinOn[vJ].Resolver.CheckData.FieldCount do
                  vFieldNames := vFieldNames + RightJoinOn[vJ].Resolver.CheckData.Fields[vI].FieldName + ';';
            System.Delete(vFieldNames, Length(vFieldNames), 1);
            vAccept := False;
            FxQuery.IndexNeededFor(FxQuery, TmpDataSet, vFieldNames, True, vAccept);
            if vAccept then
            begin
               vTemps := '';
               for vJ := 0 to RightJoinOn.Count - 1 do
                  vTemps := vTemps + LeftJoinOn[vJ].Resolver.Expression.AsString + ';';
               System.Delete(vTemps, Length(vTemps), 1);
               FxQuery.SetRange(FxQuery, RelOperator, TmpDataSet, vFieldNames, vTemps, vTemps);
               vJoinWasEventCalled := True;
            end;
         end;

         vNumAccepted := 0;
         TmpDataSet.First;
         while not TmpDataSet.EOF do
         begin
            vNumPartial := 0;
            for vJ := 0 to RightJoinOn.Count - 1 do
            begin
               case LeftJoinOn[vJ].Resolver.Expression.ExprType of
                  ttString:
                     begin
                     vLeftAsString  := LeftJoinOn[vJ].Resolver.Expression.AsString;
                     vRightAsString := RightJoinOn[vJ].Resolver.Expression.AsString;
                     case RelOperator of
                        ropBETWEEN : vPassed := (vLeftAsString =  vRightAsString);
                        ropGT      : vPassed := (vLeftAsString >  vRightAsString);
                        ropGE      : vPassed := (vLeftAsString >= vRightAsString);
                        ropLT      : vPassed := (vLeftAsString <  vRightAsString);
                        ropLE      : vPassed := (vLeftAsString <= vRightAsString);
                        ropNEQ     : vPassed := (vLeftAsString <> vRightAsString);
                     end;
                     end;
                  ttFloat:
                     begin
                     vLeftAsFloat := LeftJoinOn[vJ].Resolver.Expression.AsFloat;
                     vRightAsFloat := RightJoinOn[vJ].Resolver.Expression.AsFloat;
                     case RelOperator of
                        ropBETWEEN: vPassed := (vLeftAsFloat =  vRightAsFloat);
                        ropGT     : vPassed := (vLeftAsFloat >  vRightAsFloat);
                        ropGE     : vPassed := (vLeftAsFloat >= vRightAsFloat);
                        ropLT     : vPassed := (vLeftAsFloat <  vRightAsFloat);
                        ropLE     : vPassed := (vLeftAsFloat <= vRightAsFloat);
                        ropNEQ    : vPassed := (vLeftAsFloat <> vRightAsFloat);
                     end;
                     end;
                  ttInteger:
                     begin
                     vLeftAsInteger := LeftJoinOn[vJ].Resolver.Expression.AsInteger;
                     vRightAsInteger := RightJoinOn[vJ].Resolver.Expression.AsInteger;
                     case RelOperator of
                        ropBETWEEN: vPassed := (vLeftAsInteger =  vRightAsInteger);
                        ropGT     : vPassed := (vLeftAsInteger >  vRightAsInteger);
                        ropGE     : vPassed := (vLeftAsInteger >= vRightAsInteger);
                        ropLT     : vPassed := (vLeftAsInteger <  vRightAsInteger);
                        ropLE     : vPassed := (vLeftAsInteger <= vRightAsInteger);
                        ropNEQ    : vPassed := (vLeftAsInteger <> vRightAsInteger);
                     end;
                     end;
                  ttBoolean:
                     begin
                     vLeftAsBoolean := LeftJoinOn[vJ].Resolver.Expression.AsBoolean;
                     vRightAsBoolean := RightJoinOn[vJ].Resolver.Expression.AsBoolean;
                     case RelOperator of
                        ropBETWEEN: vPassed := (vLeftAsBoolean =  vRightAsBoolean);
                        ropGT     : vPassed := (vLeftAsBoolean >  vRightAsBoolean);
                        ropGE     : vPassed := (vLeftAsBoolean >= vRightAsBoolean);
                        ropLT     : vPassed := (vLeftAsBoolean <  vRightAsBoolean);
                        ropLE     : vPassed := (vLeftAsBoolean <= vRightAsBoolean);
                        ropNEQ    : vPassed := (vLeftAsBoolean <> vRightAsBoolean);
                     end;
                     end;
               end;
               if vPassed then
               begin
                  { don't accept for partial and expression in JOIN }
                  if Assigned(AndJoinOn[vJ].Resolver) and
                        (AndJoinOn[vJ].Resolver.Expression.AsBoolean = False) then
                     vPassed := False;
                  if vPassed then
                    Inc(vNumPartial);
               end;
            end;
            vAccepted := (vNumPartial = LeftJoinOn.Count);
            if vAccepted then
            begin
               if Start < FJoinNestedList.Count - 1 then
                  { call recursively }
                  RecursiveJoin(Start + 1)
               else
               begin
                  vAccepted := True;
                  if Assigned(FSubQuery) then
                     { subquerys not allowed in joins yet }
                     vAccepted := False
                  else if Length(FWhereStr ) > 0 then
                     { process WHERE clause with normal expression }
                     vAccepted := Self.FMainWhereResolver.Expression.AsBoolean;
                  if vAccepted then
                  begin
                     AddThisRecord( GetRecordNumber(FDefDataSet) );
                     Inc(vNumAccepted);
                  end;
               end;
            end;

            { next record }
            TmpDataSet.Next;
         end;
         {Call event to notify user that previous range set must be canceled}
         if FxQuery.Optimize and vJoinWasEventCalled {and Assigned(FxQuery.OnCancelRange)} then
            FxQuery.CancelRange(FxQuery, TmpDataSet);
         if (JoinKind = jkLeftOuterJoin) and (vNumAccepted = 0) then
         begin
            { OUTER JOINS code - create the list of temporary disabled datasets}
            Self.FxQuery.FDisabledDataSets.Clear;
            for vI := Start + 1 to FJoinNestedList.Count - 1 do
            begin
               with Self.FxQuery.FDisabledDataSets do
               begin
                  if IndexOf(TableList[FJoinNestedList[vI].Idx1].DataSet) < 0 then
                     Add( TableList[FJoinNestedList[vI].Idx1].DataSet );
                  if IndexOf(TableList[FJoinNestedList[vI].Idx2].DataSet) < 0 then
                     Add( TableList[FJoinNestedList[vI].Idx2].DataSet );
               end;
            end;
            AddThisRecord( GetRecordNumber( FDefDataSet ) );
            Self.FxQuery.FDisabledDataSets.Clear;
         end;
      end;
   end;
   {-----------end of Recursively joining -------------------------------------}

begin
   { recursively joining }
   RecursiveJoin(0);
end;

{*******************************************************************************}
function TSqlAnalizer.ReplaceParams(const Expr: String; var Succeed: Boolean): String;
var
   I: Integer;
begin
   Result := Expr;
   if (FxQuery.Params.Count > 0) and (Length(Expr) > 0) then
      for I := 0 to FxQuery.Params.Count - 1 do
         with FxQuery.Params[I] do
         begin
            case DataType of
               ftString{$ifdef level4},ftFixedChar,ftWideString{$endif} :
                  ReplaceString(Result, ':' + Name, #39 + AsString + #39);
               ftFloat,ftCurrency,ftBCD, ftAutoInc,ftSmallInt,ftInteger,ftWord
               {$ifndef LEVEL3},ftLargeInt{$endif} :
                  ReplaceString(Result, ':' + Name, AsString);
               ftDate, ftTime, ftDateTime :
                  ReplaceString(Result, ':' + Name, FloatToStr(AsFloat));
               ftBoolean:
                  ReplaceString(Result, ':' + Name, xqbase.NBoolean[AsBoolean]);
            end;
         end;
   Succeed := (AnsiPos(':', Result) = 0);
end;

{ --- Add this record to the result set --- }
procedure TSqlAnalizer.AddThisRecord( pSourceRecno: Integer );
var
   J        : Integer;
   vField   : TxqField;
   vColumn  : TColumnItem;
begin
   if (FxQuery.FMaxRecords > 0) and (ResultSet.RecordCount >= FxQuery.FMaxRecords) then Exit;
   ResultSet.Insert;

   { dataset can optionally support RecNo property  }
   ResultSet.SourceRecno := pSourceRecno;
   if ResultSet is TSeqResultSet then Exit;

   { add all fields }
   for J := 0 to FColumnList.Count - 1 do
   begin
      vColumn := ColumnList[J];

      { this column will be evaluated in DoGroupBy method }
      if vColumn.AggregateList.Count > 0 then Continue;

      vField  := ResultSet.Fields[J];
      with vColumn do
      begin
         if Assigned(vField.SourceField) and (vField.SourceField is TBlobField) then
         begin
            { Instead of actual data, this will point to the original recno
              that have the blob field of the dataset }
            if Self.FxQuery.IsDataSetDisabled(vField.SourceField.DataSet) then
               ResultSet.Fields[J].AsInteger := 0
            else
               ResultSet.Fields[J].AsInteger :=
                  GetRecordNumber(vField.SourceField.DataSet);
         end else
         begin
            { TRANSFORM...PIVOT special case }
            case Resolver.Expression.ExprType of
               ttString : ResultSet.Fields[J].AsString  := Resolver.Expression.AsString;
               ttFloat  : ResultSet.Fields[J].AsFloat   := Resolver.Expression.AsFloat;
               ttInteger: ResultSet.Fields[J].AsInteger := Resolver.Expression.AsInteger;
               ttBoolean: ResultSet.Fields[J].AsBoolean := Resolver.Expression.AsBoolean;
            end;
         end;
      end;
   end;
end;

{ INSERT statement }
procedure TSqlAnalizer.DoInsertStatement;
var
  I          : Integer;
  J          : Integer;
  L          : Integer;
  Resolver   : TMainExpr;
  InsertItem : TInsertItem;
begin
  for L := 0 to FInsertList.Count - 1 do
  begin
    InsertItem := FInsertList[L];
    if Assigned(FSubQuery) then
    begin
       for I := 1 to FSubQuery.ResultSet.RecordCount do
       begin
          FSubQuery.ResultSet.RecNo := I;
          if InsertItem.FieldNames.Count = 0 then
          begin
             { insertion on all fields }
             InsertItem.DataSet.Insert;
             try
                for J := 0 to InsertItem.DataSet.FieldCount - 1 do
                   if J <= FSubQuery.ResultSet.FFields.Count - 1 then
                   begin
                      with InsertItem.DataSet.Fields[J] do
                         if DataType in ftNonTextTypes then
                            AsString :=  FSubQuery.ResultSet.Fields[J].AsString
                         else
                            case Field2ExprType(DataType) of
                               ttString :
                                  AsString  := FSubQuery.ResultSet.Fields[J].AsString;
                               ttFloat:
                                  AsFloat   :=  FSubQuery.ResultSet.Fields[J].AsFloat;
                               ttInteger :
                                  AsInteger := FSubQuery.ResultSet.Fields[J].AsInteger;
                               ttBoolean:
                                  AsBoolean := FSubQuery.ResultSet.Fields[J].AsBoolean;
                            end;
                   end else
                      Break;
                      { InsertItem.DataSet.Fields[J].Value := Null; }
             except
                InsertItem.DataSet.Cancel;
                raise;
             end;
             InsertItem.DataSet.Post;
          end else
          begin
             // insertion on specific fields
             InsertItem.DataSet.Insert;
             try
                for J := 0 to InsertItem.FieldNames.Count - 1 do
                   with InsertItem.DataSet.FieldByName(InsertItem.FieldNames[J]) do
                      if DataType in ftNonTextTypes then
                         AsString := FSubQuery.ResultSet.FieldByName(InsertItem.FieldNames[J]).AsString
                      else
                         case Field2ExprType(DataType) of
                            ttString :
                               AsString  := FSubQuery.ResultSet.FieldByName(InsertItem.FieldNames[J]).AsString;
                            ttFloat:
                               AsFloat   := FSubQuery.ResultSet.FieldByName(InsertItem.FieldNames[J]).AsFloat;
                            ttInteger :
                               AsInteger := FSubQuery.ResultSet.FieldByName(InsertItem.FieldNames[J]).AsInteger;
                            ttBoolean:
                               AsBoolean := FSubQuery.ResultSet.FieldByName(InsertItem.FieldNames[J]).AsBoolean;
                         end;
                { for j := 0 to InsertItem.DataSet.FieldCount - 1 do
                   if InsertItem.FieldNames.IndexOf(Insert.DataSet.Fields[J].FieldName) < 0 then
                      InsertItem.DataSet.Fields[J].Value := Null; }
             except
                InsertItem.DataSet.Cancel;
                raise;
             end;
             InsertItem.DataSet.Post;
          end;
       end;
    end else
    begin

       { only one record will be inserted }
       if InsertItem.FieldNames.Count = 0 then
       begin
          { insertion on all fields }
          InsertItem.DataSet.Insert;
          try
             for I := 0 to InsertItem.DataSet.FieldCount - 1 do
                if I <= InsertItem.ResolverList.Count - 1 then
                begin
                   with InsertItem.DataSet.Fields[I] do
                   begin
                      Resolver := TMainExpr(InsertItem.ResolverList[I]);
                      if Datatype in ftNonTextTypes then
                         AsString := Resolver.Expression.AsString
                      else
                         case Field2ExprType(DataType) of
                            ttString :
                               AsString  := Resolver.Expression.AsString;
                            ttFloat:
                               AsFloat   := Resolver.Expression.AsFloat;
                            ttInteger:
                               AsInteger := Resolver.Expression.AsInteger;
                            ttBoolean:
                               AsBoolean := Resolver.Expression.AsBoolean;
                         end;
                   end;
                end else
                   Break;
                   { InsertItem.DataSet.Fields[i].Value := Null; }
          except
             InsertItem.DataSet.Cancel;
             raise;
          end;
          InsertItem.DataSet.Post;
       end else
       begin
          { insertion on specific fields }
          InsertItem.DataSet.Insert;
          try
             for I := 0 to InsertItem.FieldNames.Count - 1 do
             begin
                with InsertItem.DataSet.FieldByName(InsertItem.FieldNames[I]) do
                begin
                   Resolver := TMainExpr(InsertItem.ResolverList[I]);
                   if DataType in ftNonTextTypes then
                      AsString := Resolver.Expression.AsString
                   else
                      case Field2ExprType(DataType) of
                         ttString :
                            AsString  := Resolver.Expression.AsString;
                         ttFloat:
                            AsFloat   := Resolver.Expression.AsFloat;
                         ttInteger:
                            AsInteger := Resolver.Expression.AsInteger;
                         ttBoolean:
                            AsBoolean := Resolver.Expression.AsBoolean;
                      end;
                end;
             end;
             { set to null all other fields }
             { for i := 0 to InsertItem.DataSet.FieldCount - 1 do
                if InsertItem.FieldNames.IndexOf(InsertItem.DataSet.Fields[i].FieldName) < 0 then
                   InsertItem.DataSet.Fields[J].Value := Null; }
          except
             InsertItem.DataSet.Cancel;
             raise;
          end;
          InsertItem.DataSet.Post;
       end;
    end;
  end;
end;

{ UPDATE statement }
procedure TSqlAnalizer.DoUpdateRecord;
var
  I : Integer;
begin
  FDefDataSet.Edit;
  for I := 0 to UpdateColumnList.Count - 1 do
     with UpdateColumnList[I] do
     begin
        case Field2Exprtype(Field.DataType) of
           ttString  : Field.AsString  := Resolver.Expression.AsString;
           ttFloat   : Field.AsFloat   := Resolver.Expression.AsFloat;
           ttInteger : Field.AsInteger := Resolver.Expression.AsInteger;
           ttBoolean : Field.AsBoolean := Resolver.Expression.AsBoolean;
        end;
     end;
  FDefDataSet.Post;
end;

{ GROUP BY clause }
procedure TSqlAnalizer.DoGroupBy;
var
  I, J, K, L, n : Integer;
  Idx           : Integer;
  vIndex        : Integer;
  vPivot        : Integer;
  vS            : String;
  vColumn       : TColumnItem;
  vTempExpr     : TMainExpr;
  vValue        : Double;
  vSortList     : TxqSortList;
begin
   { GROUP BY clause }
   if (ResultSet.RecordCount > 0) and
      ( (FGroupByList.Count > 0) or HasAggregates ) then
   begin
      ResultSet.IsSequenced := False;
      if FxQuery.InMemResultSet then
         vSortList := TMemSortList.Create
      else
         vSortList := TFileSortList.Create(FxQuery.FMapFileSize);
      try
         { syntax: SELECT COUNT(DISTINCT pricelist) FROM customer; }
         if HasDistinctAggregates then
         begin
            { create all the needed fields }
            for J := 0 to FColumnList.Count - 1 do
            begin
               vColumn := FColumnList[J];
               for K := 0 to vColumn.AggregateList.Count - 1 do
               begin
                  if not( vColumn.AggregateList[K].IsDistinctAg ) then Continue;
                  L := vColumn.AggregateList[K].ColIndex;
                  vSortList.AddField( ResultSet.Fields[L].DataType,
                                      ResultSet.Fields[L].ColWidth,
                                      False );
               end;
            end;
            for I := 1 to ResultSet.RecordCount do
            begin
               ResultSet.RecNo := I;
               Idx := 0;
               for J := 0 to FColumnList.Count - 1 do
               begin
                  vColumn := FColumnList[J];
                  for K := 0 to vColumn.AggregateList.Count - 1 do
                  begin
                     if not( vColumn.AggregateList[K].IsDistinctAg ) then Continue;
                     { this is the temporary column where the aggregate was defined }
                     vSortList.Insert;
                     vSortList.SourceRecno := I;
                     L := vColumn.AggregateList[K].ColIndex;
                     case ResultSet.Fields[L].DataType of
                        ttString  :
                           vSortList.Fields[Idx].AsString  := ResultSet.Fields[L].AsString;
                        ttFloat   :
                           vSortList.Fields[Idx].AsFloat   := ResultSet.Fields[L].AsFloat;
                        ttInteger :
                           vSortList.Fields[Idx].AsInteger := ResultSet.Fields[L].AsInteger;
                        ttBoolean :
                           vSortList.Fields[Idx].AsBoolean := ResultSet.Fields[L].AsBoolean;
                     end;
                     Inc(Idx);
                  end;
               end;
            end;
            vSortList.Sort;

            { mark records that must be deleted }
            for I := vSortList.Count downto 2 do
            begin
               if vSortList.IsEqual(I, I - 1) then
               begin
                  vSortList.Recno := I;
                  ResultSet.RecNo := vSortList.SourceRecno;
                  ResultSet.SourceRecNo := -2;
               end;
            end;

            { now, delete the records }
            for I := ResultSet.RecordCount downto 1 do
            begin
               ResultSet.RecNo := I;
               if ResultSet.SourceRecNo = -2 then
               begin
                  ResultSet.Delete;
                  FColumnList.DeleteAggregate(I);
               end;
            end;
            vSortList.Clear;
         end; { end of processing DISTINCT aggregate columns }

         { now the real grouping :
          create the group-by fields }
         for J := 0 to FGroupByList.Count - 1 do
             with FGroupByList[J] do
                vSortList.AddField( ResultSet.Fields[ColIndex].DataType,
                                    ResultSet.Fields[ColIndex].ColWidth,
                                    False );
         for I := 1 to ResultSet.RecordCount do
         begin
            ResultSet.RecNo := I;
            vSortList.Insert;
            vSortList.SourceRecno := I;
            for J := 0 to FGroupByList.Count - 1 do
               with FGroupByList[J] do
                  case ResultSet.Fields[ColIndex].DataType of
                     ttString : vSortList.Fields[J].AsString  := ResultSet.Fields[ColIndex].AsString;
                     ttFloat  : vSortList.Fields[J].AsFloat   := ResultSet.Fields[ColIndex].AsFloat;
                     ttInteger: vSortList.Fields[J].AsInteger := ResultSet.Fields[ColIndex].AsInteger;
                     ttBoolean: vSortList.Fields[J].AsBoolean := ResultSet.Fields[ColIndex].AsBoolean;
                  end;
         end;
         vSortList.Sort;

         { once sorted, group the records }
         vIndex := 1;
         vPivot := 1;

         { all but pivots are marked with negative number }
         vSortList.Recno := vPivot;
         vSortList.SourceRecno := -vSortList.SourceRecno;
         while vIndex <= ResultSet.RecordCount do
         begin
            if vSortList.IsEqual(vPivot, vIndex) then
            begin
               for I := 0 to FColumnList.Count - 1 do
               begin
                  vColumn := FColumnList[I];
                  for J := 0 to vColumn.AggregateList.Count - 1 do
                  begin
                     { set position to current record and get the values in there }
                     if not (vColumn.AggregateList[J].Aggregate = akCOUNT) then
                     begin
                        { it is in [akSUM, akAVG, akMIN, akMAX] }
                        vSortList.RecNo := vIndex;
                        ResultSet.RecNo := Abs(vSortList.SourceRecno);
                        vValue := ResultSet.Fields[vColumn.AggregateList[J].ColIndex].AsFloat;
                     end;

                     { Set position to the pivot record for this group }
                     vSortList.Recno := vPivot;
                     n := Abs(vSortList.SourceRecno);
                     with vColumn.AggregateList[J] do
                     begin
                        case Aggregate of
                           akSUM, akAVG :
                              SparseList.Values[n] := SparseList.Values[n] + vValue;
                           akMIN :
                              if SparseList.Count[n] = 0 then
                                 SparseList.Values[n] := vValue
                              else
                                 SparseList.Values[n] := Min(vValue, SparseList.Values[n]);
                           akMAX :
                              if SparseList.Count[n] = 0 then
                                 SparseList.Values[n] := vValue
                              else
                                 SparseList.Values[n] := Max(vValue, SparseList.Values[n]);
                        end;
                        { always is calculated due to that it is used in AVG,MIN,MAX,COUNT }
                        SparseList.Count[n] := SparseList.Count[n] + 1;
                     end;
                  end;
               end;
            end else
            begin
               vPivot := vIndex;
               { mark pivot with a negative number }
               vSortList.Recno := vPivot;
               vSortList.SourceRecno := -vSortList.SourceRecno;
               vIndex := Pred(vPivot);
            end;
            Inc(vIndex);
         end;

         { calculate AVG aggregate and pass results to normal field area }
         for I := 1 to vSortList.Count do
         begin
            vSortList.Recno := I;
            K := vSortList.SourceRecno;
            if K < 0 then
            begin
               n := Abs( K );
               ResultSet.RecNo := n;
               for J := 0 to FColumnList.Count - 1 do
               begin
                  vColumn := FColumnList[J];
                  if (J = FHavingCol) or (vColumn.AggregateList.Count = 0) then Continue;
                  vS := vColumn.ColumnExpr;
                  for K := 0 to vColumn.AggregateList.Count - 1 do
                  begin
                     with vColumn.AggregateList[K] do
                     begin
                        if Aggregate = akAVG then
                        begin
                           if SparseList.Count[n] > 0 then
                              SparseList.Values[n] := SparseList.Values[n] / SparseList.Count[n]
                           else
                              SparseList.Values[n] := 0;
                        end;
                        { replace expressions like (Aggregate 1) with its real value}
                        if Aggregate = akCOUNT then
                           ReplaceString( vS, Format('{Aggregate %d}',[K]), IntToStr(SparseList.Count[n]))
                        else
                           ReplaceString( vS, Format('{Aggregate %d}',[K]), FloatToStr(SparseList.Values[n]));
                     end;
                  end;
                  vColumn.Resolver.ParseExpression(vS);
                  { now set the temporary aggregate calculated value in the working value }
                  with vColumn.Resolver.Expression do
                     case ExprType of
                        ttString : ResultSet.Fields[J].AsString  := AsString;
                        ttFloat  : ResultSet.Fields[J].AsFloat   := AsFloat;
                        ttInteger: ResultSet.Fields[J].AsInteger := AsInteger;
                        ttBoolean: ResultSet.Fields[J].AsBoolean := AsBoolean;
                     end;
               end;
            end;
         end;

         { first, mark all records that must be deleted }
         for I := 1 to vSortList.Count do
         begin
            vSortList.Recno := I;
            K := vSortList.SourceRecno;
            if K > 0 then
            begin
               ResultSet.RecNo := K;
               ResultSet.SourceRecno := -2;  { sourcerecno = -2 is used to delete later }
            end;
         end;

         { now delete not needed records }
         for I := ResultSet.RecordCount downto 1 do
         begin
            ResultSet.RecNo := I;
            if ResultSet.SourceRecno = -2 then
            begin
               ResultSet.Delete;
               FColumnList.DeleteAggregate(I);
            end;
         end;

         { HAVING predicate }
         if FHavingCol >= 0 then
         begin
            vColumn := FColumnList[FHavingCol];
            vTempExpr := TMainExpr.Create(Self, FDefDataSet);
            try
               for I := ResultSet.RecordCount downto 1 do
               begin
                  ResultSet.RecNo := I;
                  vs := vColumn.ColumnExpr;
                  for K := 0 to vColumn.AggregateList.Count - 1 do
                  begin
                     with vColumn.AggregateList[K] do
                     begin
                        if Aggregate = akAVG then
                        begin
                           if SparseList.Count[I] > 0 then
                              SparseList.Values[I] := SparseList.Values[I] / SparseList.Count[I]
                           else
                              SparseList.Values[I] := 0;
                        end;
                        { replace expressions like (Aggregate 1) with its real value}
                        if Aggregate = akCOUNT then
                           ReplaceString( vS, Format('{Aggregate %d}',[K]), IntToStr(SparseList.Count[I]))
                        else
                           ReplaceString( vS, Format('{Aggregate %d}',[K]), FloatToStr(SparseList.Values[I]));
                     end;
                  end;
                  vTempExpr.ParseExpression(vS);
                  if vTempExpr.Expression.ExprType <> ttBoolean then
                     raise EXQueryError.Create(SExprNotBoolean);
                  if not vTempExpr.Expression.AsBoolean then
                  begin
                     ResultSet.Delete;      { delete this record }
                     FColumnList.DeleteAggregate( I );
                  end;
               end;
            finally
               vTempExpr.Free;
            end;
         end;

         { now sort the result set }
         vSortList.Clear;
         { create the fields }
         for J := 0 to FGroupByList.Count - 1 do
             with FGroupByList[J] do
                vSortList.AddField( ResultSet.Fields[ColIndex].DataType,
                                    ResultSet.Fields[ColIndex].ColWidth,
                                    False );
         for I := 1 to ResultSet.RecordCount do
         begin
            ResultSet.RecNo := I;
            vSortList.Insert;
            vSortList.SourceRecno := I;
            for J := 0 to FGroupByList.Count - 1 do
               with FGroupByList[J] do
                  case ResultSet.Fields[ColIndex].DataType of
                     ttString : vSortList.Fields[J].AsString  := ResultSet.Fields[ColIndex].AsString;
                     ttFloat  : vSortList.Fields[J].AsFloat   := ResultSet.Fields[ColIndex].AsFloat;
                     ttInteger: vSortList.Fields[J].AsInteger := ResultSet.Fields[ColIndex].AsInteger;
                     ttBoolean: vSortList.Fields[J].AsBoolean := ResultSet.Fields[ColIndex].AsBoolean;
                  end;
         end;
         vSortList.Sort;
         ResultSet.SortWithList( vSortList );
         FColumnList.SortAggregateWithList(vSortList);
      finally
         vSortList.Free;
      end;
   end;
end;

{ ORDER BY clause }
procedure TSqlAnalizer.DoOrderBy(TheOrderList: TOrderByList);
var
   I, J      : Integer;
   vSortList : TxqSortList;
begin
   if (ResultSet.RecordCount > 0) and (TheOrderList.Count > 0) then
   begin
      if FxQuery.InMemResultSet then
         vSortList := TMemSortList.Create
      else
         vSortList := TFileSortList.Create(FxQuery.FMapFileSize);
      for I := 0 to TheOrderList.Count - 1 do
         with TheOrderList[I] do
            vSortList.AddField( ResultSet.Fields[ColIndex].DataType,
                                ResultSet.Fields[ColIndex].ColWidth,
                                Desc );
      try
         for I := 1 to ResultSet.RecordCount do
         begin
            ResultSet.RecNo := I;
            vSortList.Insert;
            vSortList.SourceRecNo := I;
            for J := 0 to TheOrderList.Count - 1 do
               with TheOrderList[J] do
                  case ResultSet.Fields[ColIndex].DataType of
                     ttString  : vSortList.Fields[J].AsString := ResultSet.Fields[ColIndex].AsString;
                     ttFloat   : vSortList.Fields[J].AsFloat  := ResultSet.Fields[ColIndex].AsFloat;
                     ttInteger : vSortList.Fields[J].AsInteger:= ResultSet.Fields[ColIndex].AsInteger;
                     ttBoolean : vSortList.Fields[J].AsBoolean:= ResultSet.Fields[ColIndex].AsBoolean;
                  end;
         end;
         vSortList.Sort;
         { now, sort the result set }
         ResultSet.SortWithList(vSortList);
         { and the aggregates }
         FColumnList.SortAggregateWithList(vSortList);
      finally
         vSortList.Free;
      end;
   end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.DoMerge;
var
   I : Integer;
   J : Integer;
   N : Integer;
begin
   N := IMin( ResultSet.Fields.Count, FMergeAnalizer.ResultSet.Fields.Count );
   for I := 1 to FMergeAnalizer.ResultSet.RecordCount do
   begin
      FMergeAnalizer.ResultSet.RecNo := I;
      ResultSet.Insert;
      ResultSet.SourceRecno := FMergeAnalizer.ResultSet.SourceRecNo;
      for J := 0 to N - 1 do
         ResultSet.Fields[J].AsString := FMergeAnalizer.ResultSet.Fields[J].AsString;
   end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.DoTransform;
var
   vTempResultSet : TResultSet;
   I, Idx         : Integer;
   vTemps         : String;
   vIndex         : Integer;
   vColumn        : TColumnItem;
   vTempType      : TExprType;
   vPivotValue    : String;
   vValue         : Double;
   vPivotIndex    : Integer;
   vTempField     : TxqField;
   vSortList      : TMemSortList;

   procedure Transf_CreateSelectValues(Index: Integer);
   var
      I  : Integer;
   begin
      vSortList.RecNo := Index;
      for I := 0 to FTransformBaseColumns - 1 do
         case ResultSet.Fields[I].DataType of
            ttString : vSortList.Fields[I].AsString := ResultSet.Fields[I].AsString;
            ttFloat  : vSortList.Fields[I].AsFloat  := ResultSet.Fields[I].AsFloat ;
            ttInteger: vSortList.Fields[I].AsInteger:= ResultSet.Fields[I].AsInteger;
            ttBoolean: vSortList.Fields[I].AsBoolean:= ResultSet.Fields[I].AsBoolean;
         end;
   end;

   procedure Trans_AddNewRecord;
   var
     I : Integer;
   begin
      { insert a new record in the new result set }
      vTempResultSet.Insert;
      { add the select columns }
      for I := 0 to FTransformBaseColumns - 1 do
         vTempResultSet.Fields[ I ].AsString := ResultSet.Fields[ I ].AsString;
   end;

begin
   { TRANSFORM...PIVOT section}
   if (ResultSet.RecordCount > 0) and (Length(FPivotStr) > 0) then
   begin
      ResultSet.IsSequenced := False;
      vSortList := TMemSortList.Create;
      { create the comparison fields
        used TMemSortList to do the comparison }
      for I := 0 to FTransformBaseColumns - 1 do
         vSortList.AddField( ResultSet.Fields[I].DataType,
                             ResultSet.Fields[I].ColWidth,
                             False );
      { Add two dummy records for comparing purposes only }
      vSortList.Insert;
      vSortList.Insert;

      { always create a memory result set}
      vTempResultSet := TMemResultSet.Create;
      { create the original SELECT columns }
      for I := 0 to FTransformBaseColumns - 1 do
      begin
         vTempField := ResultSet.Fields[I];
         vTempResultSet.AddField( vTempField.FieldName,
                                  vTempField.DataType,
                                  vTempField.DataSize,
                                  vTempField.SourceField,
                                  vTempField.ReadOnly,
                                  vTempField.CastType,
                                  vTempField.CastLen,
                                  vTempField.UseDisplayLabel);

      end;
      if FPivotInList.Count = 0 then
      begin
         { dynamics column count:
           must search all the result set to found how many more columns to add }
         ResultSet.RecNo := 1;
         vTemps := ResultSet.Fields[ FTransformBaseColumns ].AsString;
         FPivotInList.Add( vTemps );
         for vIndex := 2 to ResultSet.RecordCount do
         begin
            ResultSet.RecNo := vIndex ;
            vTemps := ResultSet.Fields[FTransformBaseColumns].AsString ;
            if FPivotInList.IndexOf(vTemps) = -1 then
               FPivotInList.Add( vTemps );
         end;
         { by default, the column is sorted }
         FPivotInList.Sort;
      end;
      { add the fixed number of columns }
      vColumn := FColumnList[FTransformBaseColumns + 1];
      if vColumn.AggregateList[0].Aggregate in [akSUM, akAVG, akMIN, akMAX] then
         vTempType := ttFloat
      else
         vTempType := ttInteger;
      for I := 0 to FPivotInList.Count - 1 do
          vTempResultSet.AddField( FPivotInList[I],
                                   vTempType,
                                   0,
                                   nil,
                                   True,
                                   vColumn.CastType,
                                   vColumn.CastLen,
                                   False
                                    );

      ResultSet.RecNo := 1;
      { create the SELECT columns value for comparing }
      Transf_CreateSelectValues( 1 );
      { get the acumulated first value }
      if vColumn.AggregateList[0].Aggregate = akCOUNT then
         vValue := 1                             { it is a COUNT(*) }
      else
      begin
         Idx := vColumn.AggregateList[0].ColIndex;
         vValue := ResultSet.Fields[Idx].AsFloat; { it is an aggregate }
      end;
      { get the value for the pivot column }
      vPivotValue := ResultSet.Fields[ FTransformBaseColumns ].AsString;
      vPivotIndex := FPivotInList.IndexOf( vPivotValue );
      { generate a new record in the new result set }
      Trans_AddNewRecord;

      { Iterated on all the record }
      for vIndex := 2 to ResultSet.RecordCount do
      begin
         ResultSet.RecNo := vIndex ;
         { it is a change in SELECT columns ? }
         Transf_CreateSelectValues( 2 );
         if not vSortList.IsEqual( 1, 2 ) then
         begin
            { save the last value }
            if vPivotIndex >= 0 then
               vTempResultSet.Fields[FTransformBaseColumns + vPivotIndex].AsFloat := vValue;
            { generate a new record }
            Trans_AddNewRecord;
            { initialize the accumulated }
            if vColumn.AggregateList[0].Aggregate = akCOUNT then
               vValue := 1                             { it is a COUNT(*) }
            else
            begin
               Idx := vColumn.AggregateList[0].ColIndex;
               vValue := ResultSet.Fields[Idx].AsFloat; { it is an aggregate }
            end;
            Transf_CreateSelectValues( 1 );
            vPivotValue := ResultSet.Fields[FTransformBaseColumns].AsString;
            vPivotIndex := FPivotInList.IndexOf( vPivotValue );
         end;
         { it is a change in the pivot column ? }
         if vPivotValue <> ResultSet.Fields[FTransformBaseColumns].AsString then
         begin
            { save the last value }
            if vPivotIndex >= 0 then
               vTempResultSet.Fields[FTransformBaseColumns + vPivotIndex].AsFloat := vValue;
            { initialize the accumulated }
            if vColumn.AggregateList[0].Aggregate = akCOUNT then
               vValue := 1                             { it is a COUNT(*) }
            else
            begin
               Idx := vColumn.AggregateList[0].ColIndex;
               vValue := ResultSet.Fields[Idx].AsFloat; { it is an aggregate }
            end;
            vPivotValue := ResultSet.Fields[FTransformBaseColumns].AsString;
            vPivotIndex := FPivotInList.IndexOf( vPivotValue );
         end else
         begin
            if vColumn.AggregateList[0].Aggregate = akCOUNT then
               vValue := vValue + 1
            else
            begin
               Idx := vColumn.AggregateList[0].ColIndex;
               vValue := vValue + ResultSet.Fields[Idx].AsFloat;
            end;
         end;
      end;
      { assign the start value to the corresponding column }
      if vPivotIndex >= 0 then
         vTempResultSet.Fields[FTransformBaseColumns + vPivotIndex].AsFloat := vValue;

      { assign the new result set }
      FResultSet.Free;
      FResultSet := vTempResultSet;
      vSortList.Free;   { free temporary sort list used for comparisons }
   end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.CreateResultSet;
var
  vColumn           : TColumnItem;
  vIndex            : Integer;
  I, J, L, n        : Integer;
  vps               : Integer;
  vPivot            : Integer;
  vIntValue         : Integer;
  vNumAccepted      : Integer;
  vNumAsc           : Integer;
  vNumDesc          : Integer;
  vProgressMin      : Integer;
  vProgressMax      : Integer;
  vProgressPosition : Integer;
  vTmpPosition      : Integer;
  vRange, vProgress : Integer;
  vPosition         : Integer;
  vStartOptimize    : Integer;
  vEndOptimize      : Integer;
  vTmpDataSet       : TDataSet;
  vEventWasCalled   : Boolean;
  vTmpDeleted       : Boolean;
  vPass             : Boolean;
  vAccepted         : Boolean;
  vTempBool         : Boolean;
  vSucceed          : Boolean;
  vFilterActive     : Boolean;
  vSubqExpr         : TMainExpr;
  vCheckExpr        : TMainExpr;
  vTempExpr         : TMainExpr;
  vValue            : Double;
  vTempS            : String;
  vLasts            : String;
  vS                : String;
  vTableName        : String;
  vTempList         : TList;
  vOldCursor        : TCursor;
  vReadOnly         : Boolean;
  vCancel           : Boolean;
  vCanceled         : Boolean;
  vF                : TField;
  vWhereOptimize    : TWhereOptimizeItem;
  vSortList         : TxqSortList;

begin

  if Assigned(FSubQuery) then
     FSubQuery.CreateResultSet;   { exception will be raised on error }

  if not CheckIntegrity then Exit;

  { first DataSet listed in SQL is the DataSet that will be joined to other
    DataSets (if join was defined in sql) }
  if not (FStatement = ssInsert) then
     FDefDataSet := FTableList[0].DataSet
  else
     FDefDataSet := FInsertList[0].DataSet;

  { [INNER] [LEFT] [OUTER] JOIN ON }
  if FJoinList.Count > 0 then
     CreateJoinOnResolvers;

  { SELECT: create all columns expressions }
  try
    n := FColumnList.Count - 1;  { don't move this }
    for I := 0 to n do
    begin
       vColumn := FColumnList[I];
       vColumn.Resolver := TMainExpr.Create(Self, FDefDataSet);
       if vColumn.AggregateList.Count = 0 then
          vColumn.Resolver.ParseExpression( vColumn.ColumnExpr )
       else
       begin
          { First create a dummy expression in order to detect the column type }
          vS := vColumn.ColumnExpr;
          for J := 0 to vColumn.AggregateList.Count - 1 do
             ReplaceString(vS, Format('{Aggregate %d}', [J]), '1.0');
          vColumn.Resolver.ParseExpression( vS );
          { For all the aggregates in this column, add a new temporary column }
          for J := 0 to vColumn.AggregateList.Count - 1 do
          begin
             with FColumnList.Add do
             begin
                ColumnExpr     := vColumn.AggregateList[J].AggregateStr;
                IsTemporaryCol := (Length(FPivotStr) = 0);  {TRANSFORM...PIVOT special case}
                Resolver       := TMainExpr.Create(Self, FDefDataSet);
                Resolver.ParseExpression( ColumnExpr )
             end;
             vColumn.AggregateList[J].ColIndex := FColumnList.Count - 1;
          end;
       end;
    end;
  except
    for I:= 0 to FColumnList.Count - 1 do
      with FColumnList[I] do
         if Assigned(Resolver) then
         begin
            Resolver.Free; Resolver := nil;
         end;
    raise;
  end;

  { UPDATE: create all columns expressions }
  try
    for I := 0 to FUpdateColumnList.Count - 1 do
    begin
      with FUpdateColumnList[I] do
      begin
          Resolver := TMainExpr.Create(Self, FDefDataSet);
          ColExpr  := ReplaceParams( ColExpr, vSucceed );
          if not vSucceed then
             raise ExQueryError.Create(SParamsError);
          Resolver.ParseExpression( ColExpr );
      end;
    end;
  except
    for I := 0 to FUpdateColumnList.Count - 1 do
      with FUpdateColumnList[I] do
         if Assigned(Resolver) then begin
            Resolver.Free; Resolver := nil;
         end;
    raise;
  end;

  { INSERT : create all values list }
  if FStatement = ssInsert then
  begin
     for L := 0 to FInsertList.Count - 1 do
     begin
        for I := 0 to FInsertList[L].ExprList.Count - 1 do
        begin
           vTempExpr := TMainExpr.Create(Self, FInsertList[L].DataSet);
           try
              vTemps := ReplaceParams( FInsertList[L].ExprList[I], vSucceed );
              if not vSucceed then
                 raise ExQueryError.Create(SParamsError);
              vTempExpr.ParseExpression( vTemps );
              FInsertList[L].ResolverList.Add( vTempExpr );
           except
              vTempExpr.Free;
              raise;
           end;
        end;
     end;
  end;

  { it is possible to set a filter on this dataset ? }
  FWhereFilter := ReplaceParams(FWhereFilter, vSucceed);
  if not vSucceed then
     raise ExQueryError.Create(SParamsError);
  vFilterActive := False;
  if (Length(FWhereFilter) > 0) and Assigned(FxQuery.OnSetFilter) then
  begin
     { you must catch the exception in the event handler in order to
       continue processing the query }
     FxQuery.OnSetFilter(FxQuery, FDefDataSet, FWhereFilter, vFilterActive);
     if vFilterActive then begin
        FWhereOptimizeList.Clear;   { assumed the filter expression is OK }
        FWhereStr := '';            { where expression canceled also }
     end;
  end;

  { Create the apropriate TResultSet descendant }
  InitializeResultSet;

  { used after result set is created }
  ResultSet.SourceDataSet := FDefDataSet;

  { SELECT: Create result set fields }
  for I := 0 to FColumnList.Count - 1 do
  begin
    vColumn := FColumnList[I];
    vS := vColumn.ColumnExpr;
    vF := nil;
    vReadOnly := True;
    if (vColumn.AggregateList.Count = 0) and (vColumn.CastType = 0) then
    begin
       for J := 0 to FTableList.Count - 1 do
       begin
          vTmpDataSet := FTableList[J].DataSet;
          vCheckExpr := TMainExpr.Create(Self, vTmpDataSet);
          try
            try
               { is field defined in expression valid for the table? }
               if vCheckExpr.CheckExpression( vS ) then
               begin
                  with vCheckExpr.CheckData do
                  begin
                     vTemps := Format('\f"%s"', [Field.FieldName]);
                     if (RefCount = 1) and not HasMoreFunctions and
                        ( (AnsiCompareText(QualifiedField(vTemps, True), vS) = 0) or
                          (AnsiCompareText(QualifiedField(vTemps, False), vS) = 0) ) then
                     begin
                        vF := vCheckExpr.CheckData.Field;
                        case vCheckExpr.Expression.ExprType of
                           ttString : ;
                           ttFloat  : ;
                           ttInteger: ;
                           ttBoolean:
                              if vF.DataType <> ftBoolean then
                                 vF := nil;
                        end;
                     end;
                  end;
                  if (vTmpDataSet = FDefDataSet) and (vColumn.AggregateList.Count = 0) then
                     vReadOnly := False;   { means that can later be modified }
                  Break;
               end;
            except
               { ignore exception (not bad because I'm only checking expression) }
            end;
          finally
            vCheckExpr.Free;
          end;
       end;
    end;
    if vColumn.Resolver.Expression.ExprType = ttString then
    begin
      vCheckExpr := TMainExpr.Create(Self, vTmpDataSet);
      try
         vCheckExpr.CalcMaxWidth := True;
         vCheckExpr.ParseExpression(vColumn.ColumnExpr);
         n := Length(vCheckExpr.Expression.AsString) + 1;
      finally
         vCheckExpr.Free;
      end;
    end else
      n := 0;
    with vColumn do
       ResultSet.AddField( AsAlias,
                           Resolver.Expression.ExprType,
                           n,
                           vF,
                           vReadOnly,
                           CastType,
                           CastLen,
                           (not IsAsExplicit) and FxQuery.UseDisplayLabel );
  end;

  { pre-create WhereExpr }
  FMainWhereResolver := TMainExpr.Create(Self, FDefDataSet);     { used in FWhereStr }

  { pre-create SubqExpr }
  vSubqExpr := TMainExpr.Create(Self, FDefDataSet);     { used in subqueries }

  vOldCursor := Screen.Cursor;
  Screen.Cursor := crSQLWait;
  try

   { Replace Parameters in WHERE expression }
   FWhereStr := ReplaceParams(FWhereStr, vSucceed);
   if not vSucceed then
      raise ExQueryError.Create(SParamsError);

   { INSERT statement }
   if FStatement = ssInsert then
   begin
      DoInsertStatement;
      Exit;
   end;

   { pre-evaluate where expression (if possible) }
   if not Assigned(FSubQuery) and (Length(FWhereStr ) > 0) then
   begin
      FMainWhereResolver.ParseExpression(FWhereStr);
      if FMainWhereResolver.Expression.ExprType <> ttBoolean then
         raise EXQueryError.Create(SExprNotBoolean);
   end;

   { pre-evaluate subquery expression (if possible) }
   if Assigned(FSubQuery) and (Length(FWhereStr ) > 0) and
      (FSubQuery.ResultSet.RecordCount > 0) then
   begin
      vS := FWhereStr;
      {Replace the code (subquery) with a dummy function in the expression
       in order to speed up processing}
      ReplaceString(vS, '(subquery)', Format('SUBQUERY(''%s'')',
            [prExprQ.NExprType[FSubQuery.ResultSet.Fields[0].DataType]]));
      vSubqExpr.ParseExpression( vS );
      if vSubqExpr.Expression.ExprType <> ttBoolean then
         raise EXQueryError.Create( SExprNotBoolean );
   end;

   ResultSet.IsSequenced := (FStatement = ssSelect);

   { Check WHERE clause sections that can be optimized by using indexes
     JOINing have precedence over WHERE statement
     WHERE statement have precedence over an ORDER BY }

   vEventWasCalled := False;
   vStartOptimize  := 0;
   vEndOptimize    := 0;
   if FxQuery.Optimize and (FStatement = ssSelect) and (FJoinList.Count = 0) and
      (FWhereOptimizeList.Count > 0) {and Assigned(FxQuery.OnIndexNeededFor) and
      Assigned(FxQuery.OnSetRange)} then
   begin
      if FWhereOptimizeList.Count <> 1 then
         { cannot optimize when there is more than one possible optimization options }
         FWhereOptimizeList.Clear;
      if FWhereOptimizeList.Count > 0 then
      begin
         with FWhereOptimizeList[0] do  { TWhereOptimize}
         begin
            if CanOptimize then
            begin
               RangeStart := ReplaceParams(RangeStart,vSucceed);
               if not vSucceed then
                  raise ExQueryError.Create(SParamsError);
               RangeEnd   := ReplaceParams(RangeEnd,vSucceed);
               if not vSucceed then
                  raise ExQueryError.Create(SParamsError);
               vTempBool  := CanOptimize;
               FxQuery.IndexNeededFor(FxQuery, DataSet, FieldNames, False, vTempBool);
               CanOptimize := vTempBool;
               if CanOptimize then
               begin
                  vEventWasCalled := True;
               end;
            end;
         end;
      end;
   end else if FxQuery.Optimize and (FStatement= ssSelect) and (FJoinList.Count = 0) and
      (FOrderByList.Count > 0) {and Assigned(FxQuery.OnIndexNeededFor)} then
   begin
      { Check if the ordering can be optimized }
      vTempS   := '';
      vNumAsc  := 0;
      vNumDesc := 0;
      for I := 0 to FOrderByList.Count - 1 do
      begin
         with FOrderByList[I] do  { TOrderByItem }
         begin
            if Desc then Inc(vNumDesc) else Inc(vNumAsc);
            vColumn := FColumnList[ColIndex];
         end;
         if I < FOrderByList.Count - 1 then
            vTempS := vTempS + vColumn.AsAlias + ';'
         else
            vTempS := vTempS + vColumn.AsAlias;
      end;
      if (vNumAsc = FOrderByList.Count) then   { only ascending sort for now }
      begin
         { ascending homogeneous sort (not supported descending for optimizing) }
         vAccepted := False;
         FxQuery.IndexNeededFor(FxQuery, FDefDataSet, vTempS, True, vAccepted);
         if vAccepted then
         begin
            WhereOptimizeList.Clear;    { cannot optimize both: where and ordering (for now) }
            OrderByList.Clear;
         end;
      end;
      vTempS := '';
   end;

   { progress status }
   vPosition := 1;
   if Assigned(FxQuery.FOnProgress) then
   begin
      vProgressMin := 1;
      vProgressPosition := 1;
      vProgressMax := FDefDataSet.RecordCount;
      FxQuery.FOnProgress(FxQuery, psXStart, 1, FDefDataSet.RecordCount, 1);
   end;

   vCanceled := False;
   { now generate all rows }
   for I := vStartOptimize to vEndOptimize do
   begin
      if FxQuery.FOptimize and vEventWasCalled then
      begin
         { Request to user to set a range to optimize }
         vWhereOptimize := FWhereOptimizeList[I];
         with vWhereOptimize do  { TWhereOptimizeItem }
         begin
            if CanOptimize then
            begin
               vTempBool := CanOptimize;
               FxQuery.IndexNeededFor(FxQuery, DataSet, FieldNames, True, vTempBool);
               CanOptimize := vTempBool;
               if CanOptimize then
                  FxQuery.SetRange(FxQuery, RelOperator, DataSet, FieldNames, RangeStart, RangeEnd);
            end;
         end;
      end;

      FDefDataSet.First;
      while not FDefDataSet.EOF do
      begin
         if FJoinList.Count > 0 then
            { join records }
            DoJoinOn
         else
         begin
            { WHERE clause without JOIN ON clause }
            vAccepted := True;
            if Assigned(FSubQuery) then      { is a WHERE with subquery ? }
            begin
               { subqueries are issued only in where clause (defined in the grammar) }
               (* We assume that:
                 - subquery already was executed prior to entering this
                 - subquery have one and only one column
                 - subquery was executed with the ANY or ALL keywords (default ANY)
                 example:
                 SELECT * FROM customer WHERE sales > ANY (SELECT price FROM sales);
               *)
               vNumAccepted := 0;
               for J := 1 to FSubQuery.ResultSet.RecordCount do
               begin
                  FSubQuery.ResultSet.RecNo := J;
                  with TSubqueryExpr(vSubqExpr.SubqueryExpr), FSubQuery.ResultSet do
                     case Fields[0].DataType of
                        ttString : Value := Fields[0].AsString;
                        ttFloat  : Value := Fields[0].AsFloat;
                        ttInteger: Value := Fields[0].AsInteger;
                        ttBoolean: Value := Fields[0].AsBoolean;
                     end;
                  if vSubqExpr.Expression.AsBoolean = True then
                  begin
                     Inc(vNumAccepted);
                     if FSubQueryKind= skAny then Break;
                  end else if FSubQueryKind= skAll then
                     Break;
               end;
               case FSubQueryKind of
                 skAny: vAccepted := (vNumAccepted > 0);
                 skAll: vAccepted := (vNumAccepted = FSubQuery.ResultSet.RecordCount);
               end;
            end else if Length(FWhereStr ) > 0 then
            begin
               { process WHERE clause with normal expression (without subquery) }
               vAccepted := FMainWhereResolver.Expression.AsBoolean;
            end;
            if vAccepted then
            begin
               case FStatement of
                            { DataSet optionally must support RecNo property  }
                  ssSelect: AddThisRecord( GetRecordNumber(DefDataSet) );
                  ssUpdate: DoUpdateRecord;
                  ssDelete: FDefDataSet.Delete;
               end;
            end;
         end;

         if not (vAccepted and (FStatement = ssDelete)) then
            FDefDataSet.Next;  { delete moves to the next record automatically }

         if Assigned(FxQuery.OnProgress) then
         begin
            Inc(vPosition);
            vRange := vProgressMax - vProgressMin;
            vTmpPosition := MulDiv(vProgressPosition, 100, vRange);
            vProgress := MulDiv(vPosition, 100, vRange);
            if vProgress > vTmpPosition + 5 then
            begin
               vProgressPosition := vPosition;
               FxQuery.OnProgress(FxQuery, psXProgress, 0, 0, vPosition);
            end;
         end;
         if Assigned(FxQuery.OnCancelQuery) then
         begin
            vCancel := False;
            FxQuery.OnCancelQuery(FxQuery, vCancel);
            if vCancel then begin
               vCanceled := True;
               Break;
            end;
         end;
      end;

      { cancel ranges (dataset is back to its normal state) }
      if FxQuery.FOptimize and vEventWasCalled and vWhereOptimize.CanOptimize then
            {and Assigned(FxQuery.OnCancelRange) then}
         FxQuery.CancelRange(FxQuery, vWhereOptimize.DataSet);

      if vCanceled then Break;

   end;

   { cancel filters }
   if vFilterActive and Assigned(FxQuery.OnCancelFilter) then
      FxQuery.OnCancelFilter( FxQuery, FDefDataSet );

   if vCanceled then Exit;

   { GROUP BY execute}
   if Length(FPivotStr) = 0 then
      DoGroupBy
   else
   begin
      { explain: in TRANSFORM, the group by is changed to an ORDER BY }
      with FGroupByList.Add do
         ColIndex := FTransformBaseColumns;
      DoOrderBy(FGroupByList);
   end;

   { ORDER BY... execute}
   DoOrderBy(Self.FOrderByList);

   { SELECT DISTINCT... syntax }
   if FIsDistinct then
   begin
      if FxQuery.InMemResultSet then
         vSortList := TMemSortList.Create
      else
         vSortList := TFileSortList.Create(FxQuery.FMapFileSize);
      try
         for J := 0 to FColumnList.Count - 1 do
            vSortList.AddField( ResultSet.Fields[J].DataType,
                                ResultSet.Fields[J].ColWidth,
                                False );
         for I := 1 to ResultSet.RecordCount do
         begin
            ResultSet.RecNo := I;
            vSortList.Insert;
            vSortList.SourceRecno := I;
            for J := 0 to FColumnList.Count - 1 do
               case ResultSet.Fields[J].DataType of
                  ttString : vSortList.Fields[J].AsString:= ResultSet.Fields[J].AsString;
                  ttFloat  : vSortList.Fields[J].AsFloat:= ResultSet.Fields[J].AsFloat;
                  ttInteger: vSortList.Fields[J].AsInteger:= ResultSet.Fields[J].AsInteger;
                  ttBoolean: vSortList.Fields[J].AsBoolean:= ResultSet.Fields[J].AsBoolean;
               end;
         end;

         vSortList.Sort;

         { mark records that must be deleted }
         for I := vSortList.Count downto 2 do
            if vSortList.IsEqual(I, I - 1) then
            begin
               vSortList.Recno := I;
               ResultSet.RecNo := vSortList.SourceRecno;
               ResultSet.SourceRecNo := -2;
            end;

         { now, delete the records }
         for I := ResultSet.RecordCount downto 1 do
         begin
            ResultSet.RecNo := I;
            if ResultSet.SourceRecNo = -2 then
            begin
               ResultSet.Delete;
               FColumnList.DeleteAggregate( I );
            end;
         end;
      finally
         vSortList.Free;
      end;
   end;

   { Now delete temporary column from the result set
     temporary columns are used temporarily having columns and aggregate columns }
   for I := FColumnList.Count - 1 downto 0 do
      if FColumnList[I].IsTemporaryCol then
      begin
         ResultSet.Fields.Delete(I);
         FColumnList.Delete(I);
      end;

   { TRANSFORM...PIVOT statement }
   DoTransform;

  finally
     FJoinList.Clear;
     FMainWhereResolver.Free;
     vSubqExpr.Free;

     if Assigned(FxQuery.FOnProgress) then
        FxQuery.FOnProgress(FxQuery, psXEnd, 0, 0, 0);

     Screen.Cursor := vOldCursor;
  end;
end;

{*******************************************************************************}
function TSqlAnalizer.FindFieldByName(const FieldName: String): TField;
var
   I : Integer;
   F : TField;
begin
   Result := nil;
   for I := 0 to FTableList.Count - 1 do
   begin
      F := FTableList[I].DataSet.FindField(FieldName);
      if Assigned(F) then
      begin
         Result := F; Break;
      end;
   end;
end;

{*******************************************************************************}
function TSqlAnalizer.QualifiedField(const FieldName: String; UseAlias: Boolean): String;
var
   vI,vP,vJ,vK,vIndex : Integer;
   vF                 : TField;
   vTableItem         : TTableItem;
   vS, vtname,vfname  : String;
   vFieldFound        : Boolean;
   vFound             : Boolean;

   function FindColumnExplicitAlias(const fldname: String): Integer;
   var
      I: Integer;
   begin
      Result := -1;
      for I := 0 to FColumnList.Count - 1 do
         with FColumnList[I] do
         begin
            if IsAsExplicit and (AnsiCompareText(Format('\f"%s"',[fldname]),ColumnExpr)<>0) and
              (AnsiCompareText(fldname,AsAlias)=0) then
            begin
               Result := I;
               Exit;
            end;
         end;
   end;

   function GetRealTableName(const tn: String): String;
   var
      I: Integer;
   begin
      Result := tn;
      for I := 0 to FTableList.Count - 1 do
         with FTableList[I] do
            if (AnsiCompareText(TableName, tn) = 0) then Exit
            else if (AnsiCompareText(Alias, tn) = 0) then
            begin
               Result := TableName;
               Exit;
            end;
   end;

begin
   Result := FieldName; if Length(Trim(FieldName))=0 then Exit;
   vP := AnsiPos('\f"', Result);
   while (vp > 0) do
   begin
      vK := vp + 3;
      vFieldFound := False;
      while vK <= Length(Result) do
      begin
         if Result[vK] = '"' then
         begin
            vFieldFound := True;
            vs := Copy(Result, vp + 3, vK - (vp + 3));
            vJ := AnsiPos('.',vs);
            if vJ = 0 then
            begin
               vFound := False;
               vIndex := FindColumnExplicitAlias(vs);
               { if one column alias exists with same fieldname then use original field expression }
               if (vIndex >= 0) then
               begin
                  ReplaceString(Result, Format('\f"%s"',[vs]), FColumnList[vIndex].ColumnExpr);
                  vFound := True;
               end else
               begin
                  for vI := 0 to FTableList.Count - 1 do
                  begin
                     vTableItem := FTableList[vI];
                     vF := vTableItem.DataSet.FindField(vs);
                     if Assigned(vF) then
                     begin
                        if UseAlias then
                           vtname := vTableItem.Alias
                        else
                           vtname := vTableItem.TableName;
                        ReplaceString(Result, Format('\f"%s"',[vs]), Format('FIELD(''%s'',''%s'')',[vtname,vs]));
                        vFound := True;
                        break;
                     end;
                  end;
               end;
               if not vFound then
                  raise EXQueryError.CreateFmt(SFieldNotFound,[vs]);
            end else
            begin
               vtname := GetRealTableName(Copy(vs,1, vJ - 1));
               vfname := Copy(vs, vJ + 1, Length(vs));
               ReplaceString(Result, Format('\f"%s"',[vs]), Format('FIELD(''%s'',''%s'')',[vtname,vfname]));
            end;
            break;
         end;
         Inc(vK);
      end;
      if not vFieldFound then Break;
      vp := AnsiPos('\f"', Result);
   end;
end;

{*******************************************************************************}
function TSqlAnalizer.NormalizeField(const FieldName: String): String;
var
   P     : Integer;
   K     : Integer;
   J     : Integer;
   S     : String;
   ReplS : String;
   Found : Boolean;
begin
   Result := FieldName; if Length(Trim(FieldName)) = 0 then Exit;
   P := AnsiPos('\f"', Result);
   while (P > 0) do
   begin
      K := P + 3;
      Found := False;
      while K <= Length(Result) do
      begin
         if Result[K] = '"' then
         begin
            Found := True;
            S := Copy(Result, P + 3, K - (P + 3));
            J := AnsiPos('.', S);
            ReplS := S;
            if J > 0 then
               ReplS := Copy(S, J + 1, Length(S));
            ReplaceString(Result, Format('\f"%s"',[S]), ReplS);
            Break;
         end;
         Inc(K);
      end;
      if not Found then Break;
      P := AnsiPos('\f"', Result);
   end;
end;

{*******************************************************************************}
function TSqlAnalizer.CheckIntegrity: Boolean;
var
   I, J, K, L, vp1     : Integer;
   vNumAccepted        : Integer;
   vColumn             : TColumnItem;
   vTmpWhereStr        : String;
   vs                  : String;
   vFound              : Boolean;
   vGroupBy            : TOrderByItem;
   vOrderBy            : TOrderByItem;
   vLCheckExpr         : TMainExpr;
   vRCheckExpr         : TMainExpr;
   vAndCheckExpr       : TMainExpr;
   vJoinOn             : TJoinOnItem;
   vF                  : TField;
   vWhereOptimize      : TWhereOptimizeItem;
   vReferencedDataSets : TReferencedDataSetList;
   vLeftDataset        : TDataSet;
   vRightDataset       : TDataSet;
   Idx1, Idx2          : Integer;
   vNestedItem         : TNestedItem;

   procedure AddAllReferences( pTableItem: TTableItem );
   var
      K: Integer;

      procedure AddColumn(const ColText, ColAlias: string);
      begin
         with Self.FColumnList.Add do
         begin
            ColumnExpr := ColText;
            AsAlias    := ColAlias;
         end;
      end;

   begin
      { add native table fields }
      with pTableItem.DataSet do
         for K := 0 to FieldCount - 1 do
            if Fields[K].Visible then
              AddColumn(Format('FIELD(''%s'',''%s'')',[pTableItem.TableName, Fields[K].FieldName]),
                        pTableItem.TableName + '.' + Fields[K].FieldName);
   end;

begin
   vAndCheckExpr := Nil;
   if not Assigned(FxQuery) then
      raise EXQueryError.Create(SXQueryNotDefined);

   { check that all tables in FROM clause exists }
   if not(FStatement = ssInsert) and (FTableList.Count = 0) then
      raise EXQueryError.Create(SWrongTableNumber);

   { reference the dataset from the table names }
   for I := 0 to FTableList.Count - 1 do
      with FTableList[I] do
      begin
         DataSet := FxQuery.DataSetByName( TableName );
         if (DataSet = nil) then
            raise EXQueryError.CreateFmt(SWrongDataSetName, [TableName]);

         { data set must be opened }
         if not DataSet.Active then
            raise EXQueryError.CreateFmt(SDataSetNotOpened, [TableName]);
      end;

   if FStatement = ssInsert then
      for I := 0 to FInsertList.Count - 1 do
         with FInsertList[I] do
         begin
            DataSet := FxQuery.DataSetByName( TableName );
            if DataSet = nil then
               raise EXQueryError.CreateFmt(SWrongDataSetName, [TableName]);
            { data set must be opened }
            if not DataSet.Active then
               raise EXQueryError.CreateFmt(SDataSetNotOpened, [TableName]);
         end;

   { set column field to qualified fields }
   for I := 0 to FColumnList.Count - 1 do
      with FColumnList[I] do
      begin
         if Length(AsAlias) = 0 then
            AsAlias := ColumnExpr;
         ColumnExpr := QualifiedField(ColumnExpr, True);
         for J:= 0 to AggregateList.Count - 1 do
            AggregateList[J].AggregateStr := QualifiedField(AggregateList[J].AggregateStr, True);
      end;

   { UPDATE: set update columns to qualified fields and check for fields existence }
   for I := 0 to FUpdateColumnList.Count - 1 do
      with FUpdateColumnList[I] do
      begin
         ColExpr := QualifiedField(ColExpr, True);
         Field   := Self.FindFieldByName(ColName);
         if Field = nil then
            raise EXQueryError.CreateFmt(SWrongFieldName,[ColName]);
      end;

   { INSERT: check that fields in FieldNames exist in every dataset }
   if FStatement = ssInsert then
      for L := 0 to FInsertList.Count - 1 do
         for I := 0 to FInsertList[L].FieldNames.Count - 1 do
         begin
            vF := FInsertList[L].DataSet.FindField(FInsertList[L].FieldNames[I]);
            if vF = nil then
               raise EXQueryError.CreateFmt(SInsertWrongFieldName,[FInsertList[L].FieldNames[I]]);
         end;

   { check for duplicate table names }
   for I := 0 to FTableList.Count - 1 do
      with FTableList[I] do
      begin
         for K := 0 to FTableList.Count - 1 do
            if I <> K then
            begin
               if DataSet = FTableList[K].DataSet then
                     raise EXQueryError.CreateFmt(SDuplicateDataSets, [TableName]);
            end;
      end;

   { syntax SELECT * FROM }
   if FDoSelectAll then
      for I := 0 to FTableList.Count - 1 do
         AddAllReferences( FTableList[I] );

   { syntax SELECT Customer.* FROM }
   for I := 0 to FTableAllFields.Count - 1 do
      for K := 0 to FTableList.Count - 1 do
         with FTableList[K] do
            if (AnsiCompareText(TableName, FTableAllFields[I]) = 0) or
               (AnsiCompareText(Alias, FTableAllFields[I]) = 0) then
            begin
               AddAllReferences( FTableList[I] );
               Break;
            end;

   { check subquery }
   if Assigned(FSubQuery) then
   begin
      with FSubQuery do
      begin
         if FTableList.Count <> 1 then
            raise EXQueryError.Create(SSubqueryWrongTables);
         if (Self.FStatement <> ssInsert) and (FColumnList.Count <> 1) then
            raise EXQueryError.Create(SSubqueryWrongCols);
      end;
   end;

   { where expression }
   if Assigned(FxQuery.OnSetFilter) then
      FWhereFilter := NormalizeField( FWhereStr )
   else
      FWhereFilter := '';
   FWhereStr := QualifiedField(FWhereStr, True);
   { fix the date }
   FxQuery.FixDummiesForQuerying(FWhereStr);
   FxQuery.FixDummiesForFilter(FWhereStr);
   FxQuery.FixDummiesForFilter(FWhereFilter);

   { check if candidates for join in where clause are valid }
   vReferencedDataSets:= TReferencedDataSetList.Create;
   if (FStatement = ssSelect) and (FJoinList.Count = 0) then
   begin
      { try to find joins in WHERE clause }
      vTmpWhereStr := FWhereStr;
      vNumAccepted := 0;
      for I := 0 to FLJoinCandidateList.Count - 1 do
      begin
         //if Succ(I) > (FTableList.Count - 1) then break;
         { change the coded field to the workable form }
         FLJoinCandidateList[i] := QualifiedField(FLJoinCandidateList[i], True);
         FRJoinCandidateList[i] := QualifiedField(FRJoinCandidateList[i], True);

         { create dummy expression for checking fields of left and right expressions }
         vLCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
         vRCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
         try
            { check left expression }
            vReferencedDataSets.Clear;
            vLCheckExpr.ReferencedDataSets:= vReferencedDataSets;
            { exists parameteres? }
            if AnsiPos(':', FLJoinCandidateList[I]) > 0 then Continue;
            vLCheckExpr.CheckExpression( FLJoinCandidateList[I] );
            if vReferencedDataSets.Count <> 1 then Continue;

            { check right expression }
            vReferencedDataSets.Clear;
            vRCheckExpr.ReferencedDataSets:= vReferencedDataSets;
            { exists parameteres? }
            if AnsiPos(':', FRJoinCandidateList[I]) > 0 then Continue;
            vRCheckExpr.CheckExpression( FRJoinCandidateList[I] );
            if vReferencedDataSets.Count <> 1 then Continue;

            with FJoinList.Add do
            begin
               JoinKind := jkInnerJoin;
               { add one join expression }
               with LeftJoinOn.Add do
                  Expression := FLJoinCandidateList[I];
               with RightJoinOn.Add do
                  Expression := FRJoinCandidateList[I];
               AndJoinOn.Add;    { need to have same count this }
            end;
            vp1 := AnsiPos(FLJoinCandidateList[I], vTmpWhereStr);
            if vp1 > 0 then
               vTmpWhereStr := Copy(vTmpWhereStr, 1, vp1 - 1) + '0' + Copy(vTmpWhereStr,
                  vp1 + Length(FLJoinCandidateList[I]), Length(vTmpWhereStr));

            vp1 := AnsiPos(FRJoinCandidateList[I], vTmpWhereStr);
            if vp1 > 0 then
               vTmpWhereStr := Copy(vTmpWhereStr, 1, vp1 - 1) + '0' + Copy(vTmpWhereStr,
                  vp1 + Length(FRJoinCandidateList[I]), Length(vTmpWhereStr));

            Inc(vNumAccepted);
            { if vNumAccepted >= FTableList.Count - 1 then
               Break; }
         finally
            vLCheckExpr.Free;
            vRCheckExpr.Free;
         end;
      end;
      if vNumAccepted > 0 then
         FWhereStr := vTmpWhereStr;
   end;

   { check JOIN clause }
   { if (FJoinList.Count > 0) and (FJoinList.Count <> (FTableList.Count-1)) then
      raise EXQueryError.Create(SJoinPredicateWrong); }
   if FJoinList.Count > 0 then
      FJoinNestedList.Clear;
   try
     for I := 0 to FJoinList.Count - 1 do
     begin
        with FJoinList[I] do
        begin
           { LeftJoinOn can have a variable number of joins due to this syntax :
             SELECT * FROM table1 t1 INNER JOIN table2 t2 ON (t1.field1 = t2.field2) AND
                (t1.field2 = t2.field2) }
           for J := 0 to LeftJoinOn.Count - 1 do
           begin
              LeftJoinOn[J].Expression  := QualifiedField(LeftJoinOn[J].Expression, True);
              RightJoinOn[J].Expression := QualifiedField(RightJoinOn[J].Expression, True);
              if Length(AndJoinOn[J].Expression) > 0 then
                 AndJoinOn[J].Expression := QualifiedField(AndJoinOn[J].Expression, True);

              { Create dummy expression for checking fields of left and right expressions }
              vLCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
              vRCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
              if Length( AndJoinOn[J].Expression ) > 0 then
                 vAndCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
              try
                 { check if }
                 vReferencedDataSets.Clear;
                 vLCheckExpr.ReferencedDataSets:= vReferencedDataSets;
                 vLCheckExpr.CheckExpression( LeftJoinOn[J].Expression );
                 { only one table included in the expression allowed }
                 if vReferencedDataSets.Count <> 1 then
                    raise EExpression.Create(SJoinExpectedDataSet);
                 vLeftDataset := vReferencedDataSets[0].DataSet;

                 vReferencedDataSets.Clear;
                 vRCheckExpr.ReferencedDataSets := vReferencedDataSets;
                 vRCheckExpr.CheckExpression( RightJoinOn[J].Expression );
                 if vReferencedDataSets.Count <> 1 then
                    raise EExpression.Create(SJoinExpectedDataSet);
                 vRightDataSet := vReferencedDataSets[0].DataSet;

                 if Length( AndJoinOn[J].Expression ) > 0 then
                 begin
                    vReferencedDataSets.Clear;
                    vAndCheckExpr.ReferencedDataSets := vReferencedDataSets;
                    vAndCheckExpr.CheckExpression(AndJoinOn[J].Expression);
                    if vReferencedDataSets.Count <> 1 then
                       raise EExpression.Create(SJoinExpectedDataSet);
                 end;

                 { now analize vLeftDataSet and vRightDataSet }
                 if J = 0 then
                 begin
                    Idx1 := FTableList.IndexOfDataSet(vLeftDataSet);
                    Idx2 := FTableList.IndexOfDataSet(vRightDataSet);
                    FJoinNestedList.Add(FJoinList[I], Idx1, Idx2)
                 end;

              finally
                 vLCheckExpr.Free;
                 vRCheckExpr.Free;
                 vAndCheckExpr.Free;
              end;
           end;
        end;
     end;
   finally
      vReferencedDataSets.Free;
   end;

   if not FxQuery.Optimize or (FJoinList.Count > 0) or (AnsiPos('(subquery)', FWhereStr) > 0) then
   begin
      WhereOptimizeList.Clear; { no possible optimizations in WHERE when JOINing }
      FWhereFilter := '';      { no filters possible }
   end;
   if (Length(FWhereFilter) > 0) and Assigned(FSubQuery) then
      FWhereFilter := '';

   { now check for the where optimization list }
   for I := 0 to FWhereOptimizeList.Count - 1 do
   begin
      vWhereOptimize := FWhereOptimizeList[I];
      with vWhereOptimize do
      begin
         CanOptimize := False;   { first try to optimize the where statement }
         vs := QualifiedField(FieldNames, True);
         vLCheckExpr := TMainExpr.Create(Self, FTableList[0].DataSet );
         try
            if (Pos(':', vs) > 0) or   // with params cannot create optimizations 
               (not vLCheckExpr.CheckExpression( vs )) then
                  Continue;
            CanOptimize := True;
            if vLCheckExpr.CheckData.FieldCount > 0 then
            begin
               DataSet    := vLCheckExpr.CheckData.Fields[1].DataSet;
               FieldNames := '';
               for J := 1 to vLCheckExpr.CheckData.FieldCount do
                  if J < vLCheckExpr.CheckData.FieldCount then
                     FieldNames := FieldNames + vLCheckExpr.CheckData.Fields[J].FieldName + ';'
                  else
                     FieldNames := FieldNames + vLCheckExpr.CheckData.Fields[J].FieldName ;
            end;
         finally
            vLCheckExpr.Free;
         end;
      end;
   end;

   { Check GROUP BY clause }
   for I := 0 to FGroupByList.Count - 1 do
   begin
      vGroupBy := FGroupByList[I];
      { In TSQLparser columns index is given with base 1, but
        SQL must have it with index base 0, so we will correct it }
      if vGroupBy.ColIndex < 0 then
      begin
         vs := vGroupBy.Alias;
         vGroupBy.Alias := QualifiedField(vs, True);
         { find column index
          find this field in the list of fields }
         vFound := False;
         for K := 0 to FColumnList.Count - 1 do
         begin
            vColumn := FColumnList[K];
            if (AnsiCompareText(vColumn.ColumnExpr, vGroupBy.Alias) = 0) or
               (AnsiCompareText(vColumn.ColumnExpr, QualifiedField(vs, False))=0)  then
            begin
               vGroupBy.ColIndex := K;
               vFound := True;
               Break;
            end;
         end;
         if not vFound then
            raise EXQueryError.Create(SGroupByIncongruent);
      end else
      begin
         if (vGroupBy.ColIndex < 0) or (vGroupBy.ColIndex > FColumnList.Count - 1) then
            raise EXQueryError.Create(SGroupByWrongNum);
      end;
   end;

   { check ORDER BY clause }
   for I := 0 to FOrderByList.Count - 1 do
   begin
      vOrderBy := FOrderByList[I];
      if vOrderBy.ColIndex < 0 then
      begin
         vs := vOrderBy.Alias;
         vOrderBy.Alias := QualifiedField(vs, True);

         { find column index. find this field in the list of fields }
         vFound := False;
         for K := 0 to FColumnList.Count - 1 do
         begin
            vColumn := FColumnList[K];
            if (AnsiCompareText(vColumn.ColumnExpr, vOrderBy.Alias) = 0) or
               (AnsiCompareText(vColumn.ColumnExpr, QualifiedField(vs, False)) = 0) then
            begin
               vOrderBy.ColIndex := K;
               vFound := True;
               Break;
            end;
         end;
         if not vFound then
            raise EXQueryError.CreateFmt(SFieldNotFound,[''{OrderBy.Alias}]);
      end else
      begin
         if (vOrderBy.ColIndex < 0) or (vOrderBy.ColIndex > FColumnList.Count - 1) then
            raise EXQueryError.Create(SWrongIndexField);
      end;
   end;

   { TRANSFORM: check that number of columns is same as columns in group by and in ORDER BY }
   if Length(FPivotStr) > 0 then
   begin

      if not (FColumnList.Count = FGroupByList.Count) then
         raise ExQueryError.Create(STransfColumnsMismatch);
      { check that column expressions in SELECT be the same as in GROUP BY}
      for I := 0 to FGroupByList.Count - 1 do
         if not (FGroupByList[I].ColIndex = I) then
            raise ExQueryError.Create(STransfWrongColumnGroup);

      if (FOrderByList.Count > 0) then
      begin
         if not (FColumnList.Count = FOrderByList.Count) then
            raise ExQueryError.Create(STransfOrderByMismatch);
         { check that column expressions in SELECT be the same as in ORDER BY}
         for I := 0 to FOrderByList.Count - 1 do
            if not (FOrderByList[I].ColIndex = I) then
               raise ExQueryError.Create(STransfWrongColumnOrder);
      end;
      FTransformBaseColumns := FColumnList.Count;
      { add the pivot expression as the next column }
      with FColumnList.Add do
      begin
         ColumnExpr   := QualifiedField(FPivotStr, False);
         CastType     := 0;
         CastLen      := 1;
         AsAlias      := ColumnExpr;
         IsAsExplicit := False;
      end;

      { now add the aggregate function defined in TRANSFORM clause as the last column}
      with FColumnList.Add do
      begin
         ColumnExpr   := QualifiedField(FTransformColumn.ColumnExpr, False);
         AsAlias      := FTransformColumn.AsAlias;
         if Length(AsAlias) = 0 then
            AsAlias   := ColumnExpr;
         IsAsExplicit := FTransformColumn.IsAsExplicit;
         AggregateList.Assign(FTransformColumn.AggregateList);
         for J := 0 to AggregateList.Count - 1 do
            AggregateList[J].AggregateStr := QualifiedField(AggregateList[J].AggregateStr, False);
         IsTemporaryCol := FTransformColumn.IsTemporaryCol;
         CastType     := FTransformColumn.CastType;
         CastLen      := FTransformColumn.CastLen;
         { other values are irrelevant }
      end;

   end;

   Result := True;
end;

{*******************************************************************************}
function TSqlAnalizer.FindDataSetByName( const Name: String): TDataSet;
var
   vI : Integer;
begin
   Result := nil;
   for vI := 0 to FTableList.Count - 1 do
      with FTableList[vI] do
         if (AnsiCompareText(TableName, Name) = 0) or
            (AnsiCompareText(Alias, Name) = 0) then
         begin
            Result := DataSet; Exit;
         end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.SafeCreateResultSet;
var
  ListOfBookmarks : TList;
  bm              : TBookmark;
  I               : Integer;
begin
  if Assigned(FxQuery.FOnBeforeQuery) then
     FxQuery.FOnBeforeQuery(Self);
  ListOfBookmarks := TList.Create;
  try
     if FxQuery.FAutoDisableControls then
        { save position of the datasets }
        for I := 0 to FxQuery.DataSets.Count - 1 do
           with FxQuery.DataSets[I] do
           begin
              if Assigned(DataSet) and (DataSet.Active) then
              begin
                 ListOfBookmarks.Add(DataSet.GetBookmark);
                 DataSet.DisableControls;
              end else
                 ListOfBookmarks.Add(nil);
           end;

     CreateResultSet;

     if FxQuery.FAutoDisableControls then
        { restore position of the datasets }
        for I := 0 to FxQuery.DataSets.Count - 1 do
           with FxQuery.DataSets[I] do
           begin
              if Assigned(DataSet) and (DataSet.Active) then
              begin
                 bm := TBookmark(ListOfBookmarks[I]);
                 if Assigned(bm) and DataSet.BookmarkValid(bm) then
                    DataSet.GotoBookmark(bm);
                 if Assigned(bm) then
                    DataSet.FreeBookmark(bm);
                 DataSet.EnableControls;
              end;
           end;
  finally
     ListOfBookmarks.Free;
     if Assigned(FxQuery.FOnAfterQuery) then
        FxQuery.FOnAfterQuery(Self);
  end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.DoSelect;
begin
  if not(FStatement=ssSelect) then
     raise EXQueryError.Create(SIsNotValidInSelect);
  SafeCreateResultSet;
end;

{*******************************************************************************}
procedure TSqlAnalizer.DoExecSQL;
var
  I : Integer;
begin

  if (FStatement = ssSelect) then
     raise EXQueryError.Create(SIsNotValidInExecSQL);

  if FStatement = ssCreateTable then      { CREATE TABLE }
  begin

     if Assigned(FxQuery.OnCreateTable) then
     begin
        for I := 0 to FCreateTableList.Count - 1 do
           FxQuery.OnCreateTable(FxQuery, FCreateTableList[I]);
     end;
     Exit;

  end else if FStatement = ssCreateIndex then  { CREATE INDEX }
  begin

     if Assigned(FxQuery.OnCreateIndex) then
        FxQuery.OnCreateIndex(FxQuery, IndexUnique, IndexDescending, IndexTable, IndexName,
           IndexColumnList);
     Exit;

  end else if FStatement = ssDropTable then     { DROP TABLE }
  begin

     if Assigned(FxQuery.OnDropTable) then
        FxQuery.OnDropTable(FxQuery, FIndexTable);
     Exit;
  end else if FStatement = ssDropIndex then     { DROP INDEX }
  begin

     if Assigned(FxQuery.OnDropIndex) then
        FxQuery.OnDropIndex(FxQuery, FIndexTable, FIndexName);
     Exit;

  end;
  SafeCreateResultSet;
end;

{*******************************************************************************}
function TSqlAnalizer.HasAggregates: Boolean;
var
   I: Integer;
begin
   Result := False;
   for I := 0 to FColumnList.Count - 1 do
      if FColumnList[I].AggregateList.Count > 0 then
      begin
         Result := True; Exit;
      end;
end;

{*******************************************************************************}
function TSqlAnalizer.HasDistinctAggregates: Boolean;
var
   I, K : Integer;
begin
   Result := False;
   for I := 0 to FColumnList.Count - 1 do
      for K := 0 to FColumnList[I].AggregateList.Count - 1 do
      begin
         if FColumnList[I].AggregateList[K].IsDistinctAg then
         begin
            Result := True; Exit;
         end;
      end;
end;

{*******************************************************************************}
procedure TSqlAnalizer.InitializeResultSet;
var
   I: Integer;
   //Accept : Boolean;
begin
   if Assigned(FResultSet) then
      FResultSet.Free;
   //Accept := False;
   if //Accept                              { next release }
      {and} (FStatement = ssSelect)         { select statement }
      and FxQuery.FAllSequenced           { all sequenced property}
      and (TableList.Count = 1)           { no JOINing }
      and not Assigned(FSubQuery)         { no subqueries }
      and (Length(FPivotStr) = 0)         { the command is not TRANSFORM...PIVOT }
      and (FParentAnalizer = nil)         { don't have parent analizer }
      and not HasAggregates               { not aggregates }
      and not HasDistinctAggregates       { not COUNT(DISTINCT country) }
      and (Length(FWhereStr) = 0)         { no where clause }
      and (FWhereOptimizeList.Count = 0)  { no optimizations }
      then
   begin
      FResultSet := TSeqResultSet.Create;
      with TSeqResultSet( FResultSet ) do
      begin
         for I := 0 to ColumnList.Count - 1 do
         begin
            FResolverList.Add( ColumnList[ I ].Resolver );
            ColumnList[ I ].AutoFree := False;
         end;
      end;
   end else
   begin
      if FxQuery.InMemResultSet then
         FResultSet := TMemResultSet.Create
      else
         FResultSet := TFileResultSet.Create(FxQuery.FMapFileSize);
   end;
end;

{*******************************************************************************}
function TSqlAnalizer.GetResultSet: TResultSet;
begin
   Result := FResultSet;
end;

{*******************************************************************************}
procedure TSqlAnalizer.SetResultSet(Value: TResultSet);
begin
   FResultSet := Value;
end;

{-------------------------------------------------------------------------------}
{                  Implements TxDataSetItem                                     }
{-------------------------------------------------------------------------------}

procedure TxDataSetItem.Assign(Source: TPersistent);
begin
   if Source is TxDataSetItem then
   begin
      FAlias   := TxDataSetItem(Source).Alias;
      FDataSet := TxDataSetItem(Source).DataSet;
   end else
      inherited Assign( Source );
end;

function TxDataSetItem.GetDisplayName : string;
begin
  result := GetCaption;
  if Result='' then
    Result := inherited GetDisplayName;
end;

function TxDataSetItem.GetCaption : string;
begin
  if FDataSet <> nil then
    result := FDataSet.Name + ' ('+FAlias+')'
  else
    result := '(not defined)';
end;

procedure TxDataSetItem.SetDataSet(Value: TDataSet);
begin
   if Value = TxDataSets(Collection).FxQuery then
      raise exception.Create(SCircularReference);

   if not(Value is TxDataSets(Collection).DataSetClass) then
      raise exception.CreateFmt(SDataSetUnexpected, [TxDataSets(Collection).DataSetClass.ClassName]);

   FDataSet := Value;

   if Collection is TxDataSets then
      Value.FreeNotification(TxDataSets(Collection).FxQuery);

   if (FAlias='') and Assigned(Value) then
      FAlias := Value.Name;
end;

{-------------------------------------------------------------------------------}
{                  Implements TxDataSets                                        }
{-------------------------------------------------------------------------------}

constructor TxDataSets.Create(xquery: TCustomxQuery);
begin
   inherited Create(TxDataSetItem);
   FxQuery := xquery;
   FDataSetClass := TDataSet;
end;

function TxDataSets.GetItem(Index: Integer): TxDataSetItem;
begin
   Result := TxDataSetItem(inherited GetItem(Index));
end;

procedure TxDataSets.SetItem(Index: Integer; Value: TxDataSetItem);
begin
   inherited SetItem(Index, Value);
end;

function TxDataSets.GetOwner: TPersistent;
begin
   Result := FxQuery;
end;

function TxDataSets.Add: TxDataSetItem;
begin
   Result := TxDataSetItem(inherited Add);
end;

{-------------------------------------------------------------------------------}
{                  Implements TExFunctionItem                                   }
{-------------------------------------------------------------------------------}

procedure TExFunctionItem.Assign(Source: TPersistent);
begin
   if Source is TExFunctionItem then
   begin
      FName := TExFunctionItem(Source).FName;
      FResultType := TExFunctionItem(Source).FResultType;
   end else
      inherited Assign( Source );
end;

function TExFunctionItem.GetDisplayName : string;
begin
  result := GetCaption;
  if Result='' then
    Result :=inherited GetDisplayName;
end;

function TExFunctionItem.GetCaption : string;
begin
  Result := FName;
end;

{-------------------------------------------------------------------------------}
{                  Implements TExFunctions                                      }
{-------------------------------------------------------------------------------}

constructor TExFunctions.Create(xQuery: TCustomxQuery);
begin
   inherited Create(TExFunctionItem);
   FxQuery := xQuery;
end;

function TExFunctions.GetItem(Index: Integer): TExFunctionItem;
begin
   Result := TExFunctionItem(inherited GetItem(Index));
end;

procedure TExFunctions.SetItem(Index: Integer; Value: TExFunctionItem);
begin
   inherited SetItem(Index, Value);
end;

function TExFunctions.GetOwner: TPersistent;
begin
   Result := FxQuery;
end;

function TExFunctions.Add: TExFunctionItem;
begin
   Result := TExFunctionItem(inherited Add);
end;

{-------------------------------------------------------------------------------}
{          Define and implements TxqBlobStream                                  }
{-------------------------------------------------------------------------------}

type

  TxqBlobStream = class(TMemoryStream)
  private
    FField   : TBlobField;
    FDataSet : TCustomxQuery;
    FIndex   : Integer;
    procedure ReadBlobData;
  public
    constructor Create(Field: TBlobField; Mode: TBlobStreamMode);
  end;

constructor TxqBlobStream.Create(Field: TBlobField; Mode: TBlobStreamMode);
begin
  inherited Create;
  FField   := Field;
  FIndex   := FField.Index;
  FDataSet := FField.DataSet as TCustomxQuery;
  ReadBlobData;
end;

procedure TxqBlobStream.ReadBlobData;
var
  vRecNo     : Integer;
  vField     : TxqField;
  vAccepted  : Boolean;
begin
  if FDataSet.RecNo < 1 then Exit;
  FDataSet.ResultSet.RecNo := FDataSet.RecNo;
  vField := FDataSet.ResultSet.Fields[FIndex];
  vRecNo := FDataSet.ResultSet.Fields[FIndex].AsInteger;
  if vRecNo < 1 then
  begin
     if Assigned(FDataSet.OnBlobNeeded) then
     begin
        vAccepted := False;
        FDataSet.OnBlobNeeded(FDataSet, vField.SourceField.DataSet, vAccepted);
        if not vAccepted then Exit;
     end else;
        Exit;
  end else
  begin
     { position in the original record }
     SetRecordNumber(vField.SourceField.DataSet, vRecNo);
  end;
  (vField.SourceField as TBlobField).SaveToStream(Self);
  Self.Position := 0;
end;

{-------------------------------------------------------------------------------}
{                  Defines and implements TxQueryDataLink                       }
{-------------------------------------------------------------------------------}

type
  TxQueryDataLink = class({$ifdef level3}TDataLink{$ELSE}TDetailDataLink{$endif})
  private
    FxQuery: TCustomxQuery;
  protected
    procedure ActiveChanged; override;
    procedure RecordChanged(Field: TField); override;
{$IFNDEF LEVEL3}
    function GetDetailDataSet: TDataSet; override;
{$endif}
    procedure CheckBrowseMode; override;
  public
    constructor Create(AxQuery: TCustomxQuery);
  end;

constructor TxQueryDataLink.Create(AxQuery: TCustomxQuery);
begin
  inherited Create;
  FxQuery := AxQuery;
end;

procedure TxQueryDataLink.ActiveChanged;
begin
  if FxQuery.Active then FxQuery.RefreshParams;
end;

{$IFNDEF LEVEL3}
function TxQueryDataLink.GetDetailDataSet: TDataSet;
begin
  Result := FxQuery;
end;
{$endif}

procedure TxQueryDataLink.RecordChanged(Field: TField);
begin
  if (Field = nil) and FxQuery.Active then FxQuery.RefreshParams;
end;

procedure TxQueryDataLink.CheckBrowseMode;
begin
  if FxQuery.Active then FxQuery.CheckBrowseMode;
end;



{-------------------------------------------------------------------------------}
{                  Implements TCustomxQuery                                     }
{-------------------------------------------------------------------------------}

constructor TCustomxQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FReadOnly                 := True;
  FDataSets                 := TxDataSets.Create(Self);
  FSQL                      := TStringList.Create;
  TStringList(SQL).OnChange := QueryChanged;
  FDataLink                 := TxQueryDataLink.Create(Self);
  FParams                   := TParams.Create{$IFNDEF LEVEL3}(Self){$endif};
  FParamCheck               := True;
  FExFunctions              := TExFunctions.Create(Self);
  FAutoDisableControls      := True;
  FInMemResultSet           := True;
  FDateFormat               := SDefaultDateFormat;
  FOptimize                 := True;
  FDisabledDataSets         := TList.Create;
  FMapFileSize              := 2000000;
end;

destructor TCustomxQuery.Destroy;
begin
  Disconnect;
  FDataSets.Free;
  FSQL.Free;
  FExFunctions.Free;
  FFilterExpr.Free;
  FParams.Free;
  FDataLink.Free;
  FDisabledDataSets.Free;
  inherited Destroy;
end;

function TCustomxQuery.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
begin
  Result := TxqBlobStream.Create(Field as TBlobField, Mode);
end;

procedure TCustomxQuery._ReadRecord(Buffer:PChar; IntRecNum:Integer);
var
  vI         : Integer;
  vField     : TxqField;
  vs         : String;
  vf         : double;
  vn         : Integer;
  vb         : WordBool;
  vTimeStamp : TTimeStamp;
  vData      : TDateTimeRec;
  vFieldType : TFieldType;
  FldDef     : TFieldDef;
begin
  FResultSet.RecNo := IntRecNum + 1;
  FillChar(Buffer^,FRecordSize,#0);
  for vI := 0 to FResultSet.Fields.Count - 1 do
  begin
     vField := FResultSet.Fields[vI];
     case vField.DataType of
        ttString :
           begin
           vs := vField.AsString;
           if Length(vs) > 0 then
           begin
              FldDef := FieldDefs[vI];
              Move(vs[1],(Buffer + vField.FFieldOffset)^, IMin(FldDef.Size, Length(vs)));
           end;
           end;
        ttFloat :
           begin
           vf := vField.AsFloat;
           vFieldType := ftUnknown;
           if Assigned(vField.SourceField) then
              vFieldType := vField.SourceField.DataType
           else if vField.CastType > 0 then
           begin
              case vField.CastType of
                 RW_DATE     : vFieldType := ftDate;
                 RW_TIME     : vFieldType := ftTime;
                 RW_DATETIME : vFieldType := ftDateTime;
              end;
           end;
           if (vFieldType in [ftDate,ftTime,ftDateTime]) then
           begin
              if vf <> 0 then
              begin
                 vTimeStamp := DateTimeToTimeStamp(vf);
                 case vFieldType of
                    ftDate: vData.Date := vTimeStamp.Date;
                    ftTime: vData.Time := vTimeStamp.Time;
                 else
                    vData.DateTime := TimeStampToMSecs(vTimeStamp);
                 end;
                 Move(vData,(Buffer + vField.FFieldOffset)^, SizeOf(TDateTimeRec))
              end else
                 Move(vf,(Buffer + vField.FFieldOffset)^, SizeOf(vf))
           end else
              Move(vf,(Buffer + vField.FFieldOffset)^, SizeOf(Double));
           end;
        ttInteger :
           begin
           vn := vField.AsInteger;
           Move(vn,(Buffer + vField.FFieldOffset)^, SizeOf(Integer));
           end;
        ttBoolean :
           begin
           vb := vField.AsBoolean;
           Move(vb,(Buffer + vField.FFieldOffset)^, SizeOf(WordBool));
           end;
     end;
  end;
end;

procedure TCustomxQuery.SetQuery(Value: TStrings);
begin
  if SQL.Text <> Value.Text then
  begin
     Disconnect;
     SQL.beginUpdate;
     try
        SQL.Assign(Value);
     finally
        SQL.EndUpdate;
     end;
  end;
end;

procedure TCustomxQuery.ClearDataSets;
begin
   FDataSets.Clear;
end;

procedure TCustomxQuery.AddDataSet(DataSet: TDataSet; const Alias: String);
var
  Item : TxDataSetItem;
  s    : String;
begin
  if not Assigned(DataSet) then
     Exit;
  if not (DataSet is TDataSetClass) then
     raise EXQueryError.CreateFmt(SDataSetUnexpected, [TDataSetClass.ClassName]);
  Item := FDataSets.Add;
  s := Alias;
  if Length(s) = 0 then s := DataSet.Name;
  Item.Alias := s;
  Item.DataSet := DataSet;
end;

function TCustomxQuery.GetPrepared: Boolean;
begin
  Result := FPrepared;
end;

procedure TCustomxQuery.SetPrepare(Value: Boolean);
begin
  if Value then Prepare
  else UnPrepare;
end;

procedure TCustomxQuery.InternalOpen;
var
  Analizer : TSqlAnalizer;
  I        : Integer;
  Field    : TxqField;
  ErrLine,
  ErrCol   : Integer;
  ErrMsg,
  Errtxt   : String;
begin

  {$IFDEF XQDEMO}
  if IsFirstTime or (not(csDesigning in ComponentState) and not IsDelphiRunning) then
  begin
     IsFirstTime := False;
     if not (csDesigning in ComponentState) then ShowAbout;
     if not IsDelphiRunning then
        raise exception.Create( SDelphiIsNotRunning );
  end;
  {$endif}

  if Length(Trim(FSQL.Text)) = 0 then
     raise EXQueryError.Create(SSQLIsEmpty);

  if Assigned(FResultSet) then
     FreeObject(FResultSet);

  Analizer := TSqlAnalizer.Create(nil, Self);
  try
     if Analizer.parser.yyparse = 1 then
     begin
        ErrLine := Analizer.lexer.yylineno;
        ErrCol  := Analizer.lexer.yycolno - Length(Analizer.lexer.yytext) - 1;
        ErrMsg  := Analizer.parser.yyerrormsg;
        ErrTxt  := Analizer.lexer.yytext;
        if Assigned(OnSyntaxError) then
        begin
           FOnSyntaxError(Self, ErrMsg, ErrTxt, ErrLine, ErrCol, Length(ErrTxt));
           Abort;
        end else
           { if not raised an error, will raise here }
           raise ExQueryError.CreateFmt(SSyntaxErrorMsg,[ErrMsg,ErrLine,ErrCol,ErrTxt]);
     end;
     if FDataLink.DataSource <> nil then SetParamsFromDataSet;
     if (Analizer.Statement = ssMerge) and Assigned(Analizer.MergeAnalizer) then
     begin
        Analizer.Statement := ssSelect;
        Analizer.DoSelect;                 { first, execute first SELECT      }
        Analizer.MergeAnalizer.Statement := ssSelect;
        Analizer.MergeAnalizer.DoSelect;   { now, the MERGE, SELECT statement }
        Analizer.DoMerge                   { MERGE both result sets           }
     end else
        Analizer.DoSelect;                 { normal SELECT statement }
     FResultSet := Analizer.ResultSet;
     Analizer.ResultSet := nil;
     FResultSet.InternalCalcSize;
  finally
     FreeObject(Analizer);
  end;

  InternalInitFieldDefs;

  if DefaultFields then
     CreateFields;

  { Assign the display labels as column names }
  for I :=0 to FResultSet.Fields.Count - 1 do
  begin
    Field := FResultSet.Fields[I];
    if Assigned(Field.SourceField) and Field.UseDisplayLabel then
       Fields[I].DisplayLabel := Field.SourceField.DisplayLabel;
  end;

  BindFields(True);

  if Assigned(FResultSet) then
     FRecordCount := FResultSet.RecordCount;

  BofCrack          := -1;
  EofCrack          := FRecordCount;
  FRecno            := -1;
  FRecordInfoOffset := FRecordSize;
  FRecordBufferSize := FRecordSize + SizeOf(TRecInfo);
  StartCalculated   := FRecordBufferSize;
  BookmarkSize      := SizeOf(Integer);

  { now create the filter expression if exists }
  FreeObject(FFilterExpr);
  if Filtered and (Length(Filter) > 0) then
  begin
     FFilterExpr := TFilterExpr.Create(Self);
     try
        FFilterExpr.ParseExpression(Filter);
     except
        FreeObject(FFilterExpr);
        raise;
     end;
  end;

  FIsOpen := Assigned(FResultSet);

end;

procedure TCustomxQuery.InternalInitFieldDefs;
var
  I, J, Numtry    : Integer;
  Field           : TxqField;
  DataType        : TFieldType;
  Size            : Word;
  FieldName, Test : String;
  Found           : Boolean;
  FldDef          : TFieldDef;
begin
  FieldDefs.Clear;

  FRecordSize := 0;
  if not Assigned(FResultSet) then Exit;
  for I :=0 to FResultSet.Fields.Count - 1 do
  begin
    Field     := FResultSet.FFields[I];
    FieldName := Field.FieldName;
    Test      := FieldName;
    J         := AnsiPos('.', Test);
    if J > 0 then
       Test   := Copy(Test, J + 1, Length(Test));
    Numtry    := 0;
    repeat
       Found  := False;
       for J := 0 to FieldDefs.Count - 1 do
          if AnsiCompareText(FieldDefs[J].Name, Test)=0 then
          begin
             Found := True; Break;
          end;
       if Found then
       begin
          Inc(Numtry);
          Test := FieldName + '_' + IntToStr(Numtry);
       end;
    until not Found;
    FieldName := Test;
    Size := 0;
    case Field.DataType of
      ttString:
         begin
         DataType := ftString; Size := Field.DataSize;
         end;
      ttFloat:
         DataType := ftFloat;
      ttInteger:
         DataType := ftInteger;
      ttBoolean:
         DataType := ftBoolean;
    else
      DataType := ftUnknown;
    end;
    if Assigned(Field.SourceField) then
    begin
       DataType := Field.SourceField.DataType;
       if DataType in [ftString{$ifdef level4},ftFixedChar,ftWideString{$endif}] then
       begin
          Size := Field.SourceField.Size;
          if Size = 0 then Size := 1;
       end;
       (* if DataType in ftNonTextTypes then
          Size := 0; *)
       if DataType in [ftAutoInc] then
          Size := 0;
    end else if Field.CastType > 0 then
    begin
       case Field.CastType of
          RW_CHAR:    begin DataType := ftString;   Size := Field.CastLen; end;
          RW_INTEGER: begin DataType := ftInteger;  Size := 0; end;
          RW_BOOLEAN: begin DataType := ftBoolean;  Size := 0; end;
          RW_DATE:    begin DataType := ftDate;     Size := 0; end;
          RW_TIME:    begin DataType := ftTime;     Size := 0; end;
          RW_DATETIME:begin DataType := ftDateTime; Size := 0; end;
          RW_FLOAT:   begin DataType := ftFloat;    Size := 0; end;
          RW_MONEY:   begin DataType := ftCurrency; Size := 0; end;
       end;
    end;
{$ifdef LEVEL3}
    FieldDefs.Add(FieldName, DataType, Size, False);
    FldDef := FieldDefs[FieldDefs.Count - 1];
{$else}
    FldDef := FieldDefs.AddFieldDef;
    FldDef.Name := FieldName;
    FldDef.DataType := DataType;
    FldDef.Size := Size;
    FldDef.Required := False;
    if DataType = ftBCD then begin
       FldDef.Size := TBCDFIeld(Field.SourceField).Size;
       FldDef.Precision := TBCDFIeld(Field.SourceField).Precision;
    end;
{$endif}

    {TmpSize := FieldDefs[FieldDefs.Count - 1].Size; fields }
    Inc(FRecordSize, Field.DataSize);
  end;
end;

procedure TCustomxQuery.InternalClose;
begin
  BindFields(False);

  if DefaultFields then DestroyFields;

  if Assigned(FResultSet) then
     FreeObject(FResultSet);

  FIsOpen := False;
end;

function TCustomxQuery.IsCursorOpen: Boolean;
begin
  Result := FIsOpen;
end;

function TCustomxQuery.GetCanModify: Boolean;
begin
   Result := not FReadOnly;
end;

procedure TCustomxQuery.InternalGotoBookmark(Bookmark: Pointer);
var
  ReqBookmark: Integer;
begin
  ReqBookmark := PInteger(Bookmark)^;
  if(ReqBookmark >= 0) and (ReqBookmark < FRecordCount) then
    FRecno := ReqBookmark
  else
    raise EXQueryError.CreateFmt(SBookmarkNotFound, [ReqBookmark]);
end;

procedure TCustomxQuery.InternalSetToRecord(Buffer: PChar);
(*var
  ReqBookmark: Integer;*)
begin
  FRecNo:=PRecInfo(Buffer+FRecordInfoOffset).RecordNo;
  (*ReqBookmark := PRecInfo(Buffer + FRecordInfoOffset).RecordNo;
  InternalGotoBookmark(@ReqBookmark);*)
end;

function TCustomxQuery.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
  Result := PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag;
end;

procedure TCustomxQuery.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
  PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag := Value;
end;

procedure TCustomxQuery.GetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  PInteger(Data)^ := PRecInfo(Buffer + FRecordInfoOffset).RecordNo;
end;

procedure TCustomxQuery.SetBookmarkData( Buffer: PChar; Data: Pointer);
begin
  PRecInfo(Buffer + FRecordInfoOffset).RecordNo := PInteger(Data)^;
end;

procedure TCustomxQuery.InternalFirst;
begin
  FRecno := -1;
end;

procedure TCustomxQuery.InternalLast;
begin
  EofCrack := FRecordCount;
  FRecno := EofCrack;
end;

function TCustomxQuery.GetRecordCount: Longint;
var
   SaveState: TDataSetState;
   SavePosition: integer;
   TempBuffer: PChar;
begin
   CheckActive;
   if not Filtered then
      Result :=FRecordCount
   else
   begin
      Result       := 0;
      SaveState    := SetTempState(dsBrowse);
      SavePosition := FRecno;
      try
         TempBuffer := AllocRecordBuffer;
         InternalFirst;
         while GetRecord(TempBuffer,gmNext,True) = grOk do Inc(Result);
      finally
         RestoreState(SaveState);
         FRecno := SavePosition;
         FreeRecordBuffer(TempBuffer);
      end;
   end;
end;

function TCustomxQuery.GetSourceRecno: Integer;
begin
  CheckActive;
  UpdateCursorPos;
  Result := -1;
  if not FResultSet.IsSequenced then Exit;
  FResultSet.RecNo := Self.RecNo;
  Result := FResultSet.SourceRecno;
end;

function TCustomxQuery.GetRecNo: Longint;
var
  SaveState    : TDataSetState;
  SavePosition : integer;
  TempBuffer   : PChar;
begin
   CheckActive;
   UpdateCursorPos;
   if not Filtered then
   begin
      if FRecno < 0 then
         Result := 1
      else
         Result := FRecno + 1;
   end else
   begin
      Result       := 0;
      SaveState    := SetTempState(dsBrowse);
      SavePosition := FRecno;
      try
         TempBuffer := AllocRecordBuffer;
         InternalFirst;
         repeat
            if GetRecord(TempBuffer, gmNext, True)=grOk then Inc(Result);
         until PRecInfo(TempBuffer + FRecordInfoOffset).RecordNo = SavePosition;
      finally
         RestoreState(SaveState);
         FRecno := SavePosition;
         FreeRecordBuffer(TempBuffer);
      end;
   end;
end;

procedure TCustomxQuery.SetRecNo(Value: Integer);
var
  SaveState   : TDataSetState;
  SavePosition: integer;
  TempBuffer  : PChar;
begin
  CheckBrowseMode;
  if not Filtered then
  begin
     if(Value > 1) and(Value <= FRecordCount) then
     begin
       FRecno := Value - 1;
       Resync([]);
       doAfterScroll;
     end;
  end else
  begin
     SaveState := SetTempState(dsBrowse);
     SavePosition := FRecno;
     try
        TempBuffer := AllocRecordBuffer;
        InternalFirst;
        repeat
           begin
              if GetRecord(TempBuffer,gmNext,True)=grOk then
                 Dec(Value)
              else
              begin
                 FRecno := SavePosition;
                 break;
              end;
           end;
        until Value = 0;
        doAfterScroll;
     finally
        RestoreState(SaveState);
        FreeRecordBuffer(TempBuffer);
     end;
  end;
end;

function TCustomxQuery.GetRecordSize: Word;
begin
  Result := FRecordSize + SizeOf(TRecInfo) + CalcFieldsSize ;
end;

procedure TCustomxQuery.InternalInitRecord(Buffer: PChar);
begin
  FillChar(Buffer^, FRecordBufferSize, #0);
end;

function TCustomxQuery.AllocRecordBuffer: PChar;
begin
  Result := StrAlloc(FRecordBufferSize);
end;

procedure TCustomxQuery.FreeRecordBuffer(var Buffer: PChar);
begin
  StrDispose(Buffer);
end;

function TCustomxQuery.GetRecord(Buffer: PChar; GetMode: TGetMode; doCheck: Boolean): TGetResult;
var
  Acceptable: Boolean;
begin
  Result := grOk;
  Acceptable :=False;
  repeat
     case GetMode of
       gmCurrent :
          begin
             if FRecno>=FRecordCount then Result :=grEOF
             else if FRecno<0 then Result :=grBOF
             else Result :=grOk;
          end;
       gmNext :
         begin
           if(FRecno < FRecordCount-1) then
           begin
             Inc(FRecno);
             Result :=grOK;
           end else
             Result := grEOF;
         end;
       gmPrior :
         begin
           if(FRecno > 0) then
           begin
             Dec(FRecno);
             Result :=grOK;
           end else
             Result := grBOF;
         end;
     end;
     { fill record data area of buffer }
     if Result = grOK then
     begin
       _ReadRecord(Buffer, FRecno );
       with PRecInfo(Buffer + FRecordInfoOffset)^ do
       begin
          BookmarkFlag := bfCurrent;
          RecordNo := FRecno;
       end;
       ClearCalcFields(Buffer);
       GetCalcFields(Buffer);
       Acceptable :=FilterRecord(Buffer);
       if (GetMode=gmCurrent) and not Acceptable then Result :=grError;
     end;
     if (Result = grError) and doCheck then
       raise EXQueryError.Create( SGetRecordInvalid );
  until (Result<>grOk) or Acceptable;
end;

function TCustomxQuery.FilterRecord(Buffer: PChar): Boolean;
var
   SaveState: TDatasetState;
begin
   Result :=True;
   if not Filtered then exit;
   if not( Assigned(FFilterExpr) or Assigned(OnFilterRecord) ) then Exit;
   SaveState :=SetTempState(dsFilter);
   FFilterBuffer := Buffer;
   if Assigned(OnFilterRecord) then
      OnFilterRecord(self, Result);
   if Assigned(FFilterExpr) and Result then
      Result := FFilterExpr.Expression.AsBoolean;
   RestoreState(SaveState);
end;

procedure TCustomxQuery.ClearCalcFields(Buffer: PChar);
begin
   FillChar(Buffer[StartCalculated],CalcFieldsSize,0);
end;

function TCustomxQuery.GetFieldData( Field: TField; Buffer: Pointer ): Boolean;
var
  vFieldOffset : Integer;
  vxqField     : TxqField;
  vRecBuf      : PChar;
  c            : Currency;
  d            : Double;
  i            : integer;
begin
  Result := False;
  vRecBuf := GetActiveRecordBuffer;
  if not FIsOpen or (vRecBuf = nil) {or not IsEmpty} or (Field.FieldNo < 1) then Exit;
  if (Field.FieldKind = fkCalculated) or (Field.FieldKind = fkLookup) then
  begin
     Inc(vRecBuf, StartCalculated + Field.Offset);
     if (vRecBuf[0]=#0) or (Buffer=nil) then Exit
     else CopyMemory(Buffer,@vRecBuf[1],Field.DataSize);
  end else
  begin
     vxqField := FResultSet.Fields[Field.FieldNo - 1];
     vFieldOffset := vxqField.FFieldOffset;
     if Assigned(Buffer) then
     begin
        if (Field.DataType = ftBCD) and (vxqField.DataType = ttInteger) then begin
            Move((vRecBuf + vFieldOffset)^, i,SizeOf(integer));
            c := i;
            Move(c, Buffer^, SizeOf(double));
        end else if (Field.DataType = ftBCD) then begin
            Move((vRecBuf + vFieldOffset)^, d,SizeOf(Double));
            c := d;
            Move(c, Buffer^, SizeOf(double));
        end else
            Move( (vRecBuf + vFieldOffset)^, Buffer^, Field.DataSize);
        { empty date problem fixed by Stephan Trapp 05.01.2000 }
        if (vxqField.SourceField <> nil) then
        begin
          if vxqField.SourceField.DataType in [ftDate, ftTime, ftDateTime] then
          begin
            if Integer(Buffer^) = 0 then { 693594 = Delphi1-Zero-Date }
            begin
              Result := False;
              Exit;
            end;
          end;
        end;
     end else
     begin
        if Field.DataType = ftBoolean then
        begin
           Move( (vRecBuf + vFieldOffset)^, Buffer, Field.DataSize);
        end;
     end;
  end;
  Result := True;
end;

procedure TCustomxQuery.SetFieldData(Field: TField; Buffer: Pointer);
var
   DestBuffer: PChar;
begin
   DestBuffer :=GetActiveRecordBuffer;
   if (Field.FieldKind=fkCalculated) or (Field.FieldKind=fkLookup) then { this is a calculated field  }
   begin
      Inc(DestBuffer,StartCalculated+Field.Offset);
      Boolean(DestBuffer[0]) :=(Buffer<>nil);
      if Boolean(DestBuffer[0]) then CopyMemory(@DestBuffer[1],Buffer,Field.DataSize);
   end
   {will add code for editable dataset here }
end;

procedure TCustomxQuery.InternalAddRecord(Buffer: Pointer; Append: Boolean);
begin
end;

procedure TCustomxQuery.InternalDelete;
begin
end;

procedure TCustomxQuery.InternalHandleexception;
begin
   Application.Handleexception(Self);
end;

procedure TCustomxQuery.InternalPost;
begin
end;

function TCustomxQuery.DataSetByName( const Name: String): TDataSet;
var
   Index : Integer;
   Item  : TxDataSetItem;
begin
   Result := nil;
   for Index := 0 to FDataSets.Count - 1 do
   begin
      Item := FDataSets[Index];
      if Assigned(Item.DataSet) and
         (AnsiCompareText(Item.Alias, Name)=0) then
      begin
         Result := Item.DataSet;
         Exit;
      end;
   end;
end;

procedure TCustomxQuery.Notification(AComponent: TComponent; Operation: toperation);
var
   I     : Integer;
   Item  : TxDataSetItem;
   Found : Boolean;
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) and Assigned(FDataSets) then
      repeat
         Found := False;
         for I := 0 to FDataSets.Count - 1 do
         begin
            Item := FDataSets[I];
            if AComponent = Item.DataSet then
            begin
               Item.Free;
               Found := True;
               Break;
            end;
         end;
      until not Found;
end;

function TCustomxQuery.GetActiveRecordBuffer:  PChar;
begin
   case State of
   dsBrowse: if IsEmpty then Result := nil else Result := ActiveBuffer;
   dsCalcFields: Result := CalcBuffer;
   dsFilter: Result :=FFilterBuffer;
   dsEdit,dsInsert: Result :=ActiveBuffer;
   dsNewValue,dsOldValue,dsCurValue: Result :=ActiveBuffer;
{$ifdef level5}
   dsBlockRead: Result := ActiveBuffer;
{$endif}
   else Result :=nil;
   end;
end;

function TCustomxQuery.SourceDataSet: TDataSet;
begin
   Result := FResultSet.SourceDataSet;
end;

function TCustomxQuery.ResultSetIsSequenced: Boolean;
begin
   Result := FResultSet.IsSequenced;
end;

procedure TCustomxQuery.ExecSQL;
var
  Analizer : TSqlAnalizer;
  ErrLine,
  ErrCol   : Integer;
  ErrMsg,
  ErrTxt   : String;
begin

  {$IFDEF XQDEMO}
  if not(csDesigning in ComponentState) and not IsDelphiRunning then
     raise exception.Create( SDelphiIsNotRunning );
  {$endif}

  if Length(Trim(FSQL.Text)) = 0 then
     raise EXQueryError.Create(SSQLIsEmpty);

  if Assigned(FResultSet) then
     FreeObject(FResultSet);

  Analizer := TSqlAnalizer.Create(nil, Self);
  try
     if Analizer.parser.yyparse = 1 then
     begin
        ErrLine := Analizer.lexer.yylineno;
        ErrCol  := Analizer.lexer.yycolno - Length(Analizer.lexer.yytext) - 1;
        ErrMsg  := Analizer.parser.yyerrormsg;
        ErrTxt  := Analizer.lexer.yytext;
        if Assigned(FOnSyntaxError) then
        begin
           FOnSyntaxError(Self, ErrMsg, ErrTxt, ErrLine, ErrCol, Length(ErrTxt));
           Abort;
        end else
           { if not raised an error, will raise here }
           raise ExQueryError.CreateFmt(SSyntaxErrorMsg,[ErrMsg,ErrLine,ErrCol,ErrTxt]);
     end;
     Analizer.DoExecSQL;
  finally
     FreeObject(Analizer);
  end;
end;

function TCustomxQuery.BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean;
begin
   Move(BCD^, Curr, SizeOf(Currency));
   Result := True;
end;

function TCustomxQuery.CurrToBCD(const Curr: Currency; BCD: Pointer; Precision, Decimals: Integer): Boolean;
begin
   Move(Curr, BCD^, SizeOf(Currency));
   Result := True;
end;

{$ifdef level4}
procedure TCustomxQuery.SetBlockReadSize(Value: Integer);
var
   doNext: Boolean;
begin
   if Value <> BlockReadSize then
   begin
      if (Value > 0) or (Value < -1) then
      begin
         inherited;
         BlockReadNext;
      end
      else
      begin
         doNext :=Value=-1;
         Value :=0;
         inherited;

         if doNext then
            Next
         else
         begin
              CursorPosChanged;
              Resync([]);
         end;
      end;
   end;
end;
{$endif}

procedure TCustomxQuery.SetFiltered(Value: Boolean);
begin
   if FIsOpen then
   begin
      CheckBrowseMode;
      if Filtered <> Value then
      begin
         inherited SetFiltered(Value);
         if Value then
            SetFilterData(Filter);
      end;
      First;
   end else
      inherited SetFiltered(Value);
end;

procedure TCustomxQuery.SetFilterData(const Text: String);
begin
   if FIsOpen and Filtered then
   begin
      CheckBrowseMode;
      if Assigned(FFilterExpr) then
         FreeObject(FFilterExpr);
      if Length(Text) > 0 then
      begin
         FFilterExpr := TFilterExpr.Create(Self);
         try
            FFilterExpr.ParseExpression(Text);
         except
            FreeObject(FFilterExpr);
            raise;
         end;
      end;
      First;
   end;
   inherited SetFilterText(Text);
end;

procedure TCustomxQuery.SetFilterText(const Value: string);
begin
   SetFilterData(Value);
end;

function TCustomxQuery.IsSequenced: Boolean;
begin
   Result := not Filtered;
end;

function TCustomxQuery.GetAbout:string;
begin
   Result := SXQUERY_ABOUT;
end;

procedure TCustomxQuery.SetAbout(const Value:String);
begin

end;

function TCustomxQuery.Find(const Expression: String): Boolean;
var
   FindExpr: TFilterExpr;
   SavedRecno: Integer;
begin
   Result := False;
   CheckActive;
   if Length(Expression)=0 then Exit;
   FindExpr := TFilterExpr.Create(Self);
   try
      FindExpr.ParseExpression(Expression);
   except
      on E:exception do
      begin
         FindExpr.Free;
         raise;
      end;
   end;
   SavedRecNo := RecNo;
   DisableControls;
   try
      First;
      while not EOF do
      begin
         if FindExpr.Expression.AsBoolean then
         begin
            Result := True;
            Break;
         end;
         Next;
      end;
   finally
      if not Result then
         RecNo := SavedRecNo;
      FindExpr.Free;
      EnableControls;
   end;
end;

procedure TCustomxQuery.CreateParams;
var
  Analizer : TSqlAnalizer;
  List     : TParams;
  ErrLine,
  ErrCol   : Integer;
  ErrMsg,
  ErrTxt   : String;
begin
  List := TParams.Create{$IFNDEF LEVEL3}(Self){$endif};
  try
    { parse the SQL text }
    if Length(Trim(FSQL.Text)) > 0 then
    begin
       Analizer := TSqlAnalizer.Create(nil, Self);
       try
          if Analizer.parser.yyparse = 1 then
          begin
           ErrLine := Analizer.lexer.yylineno;
           ErrCol  := Analizer.lexer.yycolno - Length(Analizer.lexer.yytext) - 1;
           ErrMsg  := Analizer.parser.yyerrormsg;
           ErrTxt  := Analizer.lexer.yytext;
           if Assigned(FOnSyntaxError) then
           begin
              FOnSyntaxError(Self, ErrMsg, ErrTxt, ErrLine, ErrCol, Length(ErrTxt));
              Abort;
           end else
              { if not raised an error, will raise here }
              raise ExQueryError.CreateFmt(SSyntaxErrorMsg,[ErrMsg,ErrLine,ErrCol,ErrTxt]);
          end;
          List.Assign(Analizer.Params);
       finally
          FreeObject(Analizer);
       end;
       List.AssignValues(FParams);
       FParams.Clear;
       FParams.Assign(List);
    end else
       FParams.Clear;
  finally
    List.Free;
  end;
end;

procedure TCustomxQuery.QueryChanged(Sender: tobject);
begin
  if not (csReading in ComponentState) then
  begin
     Disconnect;
     if FParamCheck or (csDesigning in ComponentState) then
        CreateParams;
  end;
end;

procedure TCustomxQuery.Disconnect;
begin
  Close;
  UnPrepare;
end;

procedure TCustomxQuery.Prepare;
begin
{ future use }
end;

procedure TCustomxQuery.UnPrepare;
begin
{ future use }
end;

{$ifdef level4}
procedure TCustomxQuery.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.Defineproperty ('ParamData', ReadParamData, writeParamData, FParams.Count > 0);
end;

procedure TCustomxQuery.ReadParamData(Reader: TReader);
begin
  Reader.ReadValue;
  Reader.ReadCollection(FParams);
end;

procedure TCustomxQuery.writeParamData(writer: Twriter);
begin
  writer.writeCollection(FParams);
end;
{$endif}

function TCustomxQuery.GetParamsCount: Word;
begin
   Result := FParams.Count;
end;

function TCustomxQuery.ParamByName(const Value: string): TParam;
begin
   Result := FParams.ParamByName(Value);
end;

procedure TCustomxQuery.SetParamsList(Value: TParams);
begin
  FParams.AssignValues(Value);
end;

procedure TCustomxQuery.SetParamsFromDataSet;
var
  I      : Integer;
  DataSet: TDataSet;
begin
  if FDataLink.DataSource <> nil then
  begin
    DataSet := FDataLink.DataSource.DataSet;
    if DataSet <> nil then
    begin
      DataSet.FieldDefs.Update;
      for I := 0 to FParams.Count - 1 do
        with FParams[I] do
          if not Bound then
          begin
            AssignField(DataSet.FieldByName(Name));
            Bound := False;
          end;
    end;
  end;
end;

procedure TCustomxQuery.RefreshParams;
var
  DataSet: TDataSet;
begin
  DisableControls;
  try
    if FDataLink.DataSource <> nil then
    begin
      DataSet := FDataLink.DataSource.DataSet;
      if DataSet <> nil then
        if DataSet.Active and (DataSet.State <> dsSetKey) then
        begin
          Close;
          Open;
        end;
    end;
  finally
    EnableControls;
  end;
end;

procedure TCustomxQuery.SetDataSource(Value: TDataSource);
begin
  if IsLinkedTo(Value) then
     raise EXQueryError.Create(SCircularDataLink);
  FDataLink.DataSource := Value;
end;

function TCustomxQuery.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

function TCustomxQuery.IsDataSetDisabled(DataSet: TDataSet): Boolean;
begin
   Result := False;
   if FDisabledDataSets.Count = 0 then Exit;
   Result := FDisabledDataSets.IndexOf(DataSet) >= 0;
end;

procedure TCustomxQuery.FixDummiesForQuerying(var Filter: String);
var
   Ps : Integer;
   I  : Integer;
begin
   if Length(Filter) = 0 then Exit;
   { this method called in the WHERE clause is a filter in
     order to fix some flags:
     - Only working flag now is the date in the format: 'DummyDate(32445.6566)'}
   Ps := AnsiPos('DummyDate(', Filter);
   while Ps > 0 do
   begin
      { by default, the date is left as it is but in descendant classes
        the date can be changed to meet the dataset filter implementation}
      for I := Ps + 10 to Length(Filter) do
         if Filter[I] = ')' then
         begin
            System.Delete(Filter, I, 1);
            System.Delete(Filter, Ps, 10);
            Break;
         end;
      Ps := AnsiPos('DummyDate(', Filter);
   end;
end;

procedure TCustomxQuery.FixDummiesForFilter(var Filter: String);
begin
   FixDummiesForQuerying(Filter); { fix the dates}
   ReplaceString(Filter, 'DummyBoolean(True)', 'True');
   ReplaceString(Filter, 'DummyBoolean(False)', 'False');
end;

procedure TCustomxQuery.InternalRefresh;
begin
   InternalClose;
   InternalOpen;
end;

procedure TCustomxQuery.IndexNeededFor( Sender        : TObject;
                                        DataSet       : TDataSet;
                                  const FieldNames    : String;
                                        ActivateIndex : Boolean;
                                    var Accept        : Boolean); 
begin
   if Assigned(FOnIndexNeededFor) then
      FOnIndexNeededFor(Sender, DataSet, FieldNames, ActivateIndex, Accept);
end;

procedure TCustomxQuery.SetRange( Sender      : TObject;
                                  RelOperator : TRelationalOperator;
                                  DataSet     : TDataSet;
                            const FieldNames,
                                  StartValues,
                                  EndValues   : String); 
begin
   if Assigned(FOnSetRange) then
      FOnSetRange(Sender, RelOperator, DataSet, FieldNames, StartValues, EndValues);
end;

procedure TCustomxQuery.CancelRange(Sender : TObject; DataSet : TDataSet);

begin
   if Assigned(FOnCancelRange) then
      FOnCancelRange(Sender, DataSet);
end;

{-------------------------------------------------------------------------------}
{ This procedure writes a TDataSet to a text file                               }
{-------------------------------------------------------------------------------}
procedure TCustomxQuery.WriteToTextFile(const FileName        : String ;
                                              FieldDelimChar,
                                              TxtSeparator    : Char ;
                                              IsCSV           : Boolean ;
                                              FieldNames      : TStringList ) ;
const
  CSVquote = '"';

var
   FTarget : TextFile;
   Idx : Integer;

   function CSVquotedStr(const Str: string): string;
   var
     Idx     : integer;
     Changed : boolean;
   begin
     Changed :=  false;
     Result  := Str;
     for Idx := Length(Result) downto 1 do
       if (Result[Idx] = CSVquote)then
       begin
         System.Insert(CSVquote, Result, Idx);
         Changed :=  true;
       end;

     if Changed then
       Result := CSVquote + Result + CSVquote;
   end;


   procedure WritefieldNames;
   var
     Idx : integer;
     Fld : string;
     F   : TField;
   begin
     for Idx := 0 to FieldNames.Count - 1 do
     begin
       F := Self.FindField(FieldNames[idx]);
       if not (F.DataType in ftNonTextTypes) then
       begin
         Fld := FieldNames[Idx];

         if (FieldDelimChar <> #0) then
           Fld :=  FieldDelimChar + Fld + FieldDelimChar
         else
         begin
           if IsCSV then
             Fld := CSVquotedStr(Fld);
         end;

         if Idx < FieldNames.Count - 1 then
            System.Write(FTarget, Fld + TxtSeparator)
         else
            System.Write(FTarget, Fld)
         end;
     end;

     System.WriteLn(FTarget);
   end;

   procedure WriteField(FieldNum : integer);
   var
     Data : string;
     F : TField;
   begin
      F := Self.FindField(FieldNames[FieldNum]);
      if not (F.DataType in ftNonTextTypes) then
      begin
        Data := Trim(F.AsString);

        if FieldDelimChar <> #0 then
          Data := FieldDelimChar + Data  + FieldDelimChar
        else
        begin
          if IsCSV then
            Data := CSVquotedStr(Data);
        end;

        if FieldNum < FieldNames.Count - 1 then
           System.Write(FTarget, Data + TxtSeparator)
        else
           System.Write(FTarget, Data);
        end;
   end;

   procedure WriteRecords;
   var
     Idx         : integer;
   begin
      Self.DisableControls;
      Self.First;
      while not Self.Eof do
      begin
          for Idx := 0 to FieldNames.Count - 1 do
            WriteField(Idx);
          System.WriteLn(FTarget);
        Self.Next;
      end;
      Self.EnableControls;
   end;

begin
  AssignFile(FTarget, FileName);
  Rewrite(FTarget);

  if FieldNames.Count = 0 then
  begin
     { fill with all fields }
     Self.FieldDefs.Update;
     for Idx := 0 to Self.FieldCount - 1 do
        if not (Self.Fields[Idx].dataType in ftNonTextTypes) then
        FieldNames.Add( Self.Fields[Idx].FieldName );
  end;
  WritefieldNames;
  Writerecords;

  CloseFile(FTarget);
end;

end.
