/* Grammar for TxQuery dataset (Delphi 3,4,5), (c) 2000 Alfonso Moreno
   NOTES :
   DON'T FORGET TO MOVE THE GENERATED CONSTANTS TO THE PLACE INDICATED
*/

%{
{**********************************************}
{   Parser for TxQuery component               }
{   Copyright (c) 1999 by Alfonso Moreno       }
{**********************************************}
unit xqYacc;

{$I XQ_FLAG.INC}
{$R xqyacc.res}  
interface

uses SysUtils,Classes,Dialogs,Windows,LexLibQ,YaccLibQ,
     xqBase, xquery, DB, xqmiscel
{$IFDEF LEVEL3}
     , DBTables
{$ENDIF}
     ;

//
// The generated constants must be placed here
// HERE !!!!
//

type
  TxqParser = class( TCustomParser )
  private
      FInPredicateList : TStringList;
      FIsNotInList     : Boolean;
      FForLength       : String;
      FEscapeChar      : String;
      FJoinKind        : TJoinKind;
      FTrimPosition    : Integer;
      FExtractField    : Integer;
      FCastType        : Integer;
      FCastLen         : Integer;
      FCurrentJoin     : Integer;
      FIsDistinctAggr  : Boolean;
      FAnalizer        : TSqlAnalizer;
      FAsAlias         : String;
      FJoinRelOperator : TRelationalOperator;
      FNumInserts      : Integer;   { used in INSERTO INTO...}
      FAggregateList   : TAggregateList;

      { for CREATE TABLE }
      FNumTables, FFieldType, FScale, FPrecision, FSize, FBlobType: Integer;
      procedure SetFieldParams(AFieldType,
                               AScale,
                               APrecision,
                               ASize,
                               ABlobType: Integer);
      procedure SetTableName(const TableName: String);
      procedure AddCreateField(const FieldName: String);
      procedure AddPrimaryKeyField(const FieldName: String);
      function CurrentCreateTable: TCreateTableItem;
      function GetCurrentAnalizer: TSqlAnalizer;
      function CurrentInsertItem : TInsertItem;
      function CreateInListExpression( const Expr : String ): String;
  public

      constructor Create(Analizer: TSqlAnalizer);
      destructor Destroy; override;

      function yyparse : integer; override;
      procedure yyerror(const msg : string); override;

      { specials }
      procedure AddColumn(const AColText : String);
      function AddAggregate( pAggregate  : TAggregateKind;
                    const pAggregateStr : String ) : String;
      procedure AddGroupBy(const ColName: String);
      procedure AddOrderBy(const ColName: String; Descending: Boolean);
      procedure AddTable(const TableName, Alias: String);
      procedure AddJoin(const LeftExpr, RightExpr, AndExpr: String);
      procedure AddJoinCandidate(const LeftExpr, RightExpr: String);
      procedure AddHavingColumn( const ColText : String );
      procedure AddUpdateColumn(const ColumnName, ColumnExpr: String);
      procedure AddWhereOptimize(const AFieldName,
                                 ARangeStart,
                                 ARangeEnd    : String;
                                 ARelOperator : TRelationalOperator);
      procedure AddParam(const ParamName: String);

      property Analizer: TSqlAnalizer read FAnalizer write FAnalizer;
      property AsAlias: String read FAsAlias write FAsAlias;
      property CurrentAnalizer: TSqlAnalizer read GetCurrentAnalizer;
  end;

implementation

uses
   xqLex, xqConsts;

(*----------------------------------------------------------------------------*)
procedure TxqParser.yyerror(const msg : string);
begin
   yyerrorMsg := msg;
   { yyerrorMsg := IntToStr(yyLexer.yylineno) +  ': ' + msg + ' at or before '+ yyLexer.yytext + '. ' +
         Format('Line %d, Column %d',[yyLexer.yylineno,yyLexer.yycolno]);
   if Analizer.xQuery.ShowErrorMessage then
      ShowMessage( yyerrorMsg ); }
end;

constructor TxqParser.Create(Analizer: TSqlAnalizer);
begin
   inherited Create;
   FAnalizer:= Analizer;
   FInPredicateList := TStringList.Create;
end;

destructor TxqParser.Destroy;
begin
   FInPredicateList.Free;
   if Assigned(FAggregateList) then
      FAggregateList.Free;
   inherited Destroy;
end;

function TxqParser.GetCurrentAnalizer: TSqlAnalizer;
var
   NumSelects   : Integer;
   MainAnalizer : TSqlAnalizer;
   TmpAnalizer  : TSqlAnalizer;

   function StandardAnalizer : TSqlAnalizer;
   begin
     Result := MainAnalizer;
     NumSelects := (yylexer as TxqLexer).NumSelects;
     while NumSelects > 1 do
     begin
        TmpAnalizer := Result;
        Result := Result.SubQuery;
        if Result = nil then
        begin
           Result := TSqlAnalizer.Create(TmpAnalizer, FAnalizer.xQuery);
           TmpAnalizer.SubQuery := Result;
        end;
        Dec(NumSelects);
     end;
   end;

begin
   with (yylexer as TxqLexer) do begin
      if IsMergeActive or IsWithActive then
      begin
         if IsMergeActive then                { merge is active. Return standard analizer }
         begin
            MainAnalizer := Self.FAnalizer;
            Result       := StandardAnalizer;
         end else                             { with is active. Return alternate analizer }
         begin
            if Self.FAnalizer.MergeAnalizer = nil then
               Self.FAnalizer.MergeAnalizer := TSqlAnalizer.Create(nil, FAnalizer.xQuery);
            MainAnalizer := Self.FAnalizer.MergeAnalizer;
            Result       := StandardAnalizer;
         end;
      end else
      begin
         MainAnalizer := Self.FAnalizer;
         Result       := StandardAnalizer;
      end;
   end;
end;

procedure TxqParser.AddColumn(const AColText : String);
var
   Column : TColumnItem;

   function StripFields(const s: String): String;
   var
      p, i, j: Integer;
      Found: Boolean;
   begin
      Result:= s;
      p:= Pos('\f"', Result);
      while p > 0 do
      begin
         j:= p + 3;
         i:= j;
         Found:= True;
         while (j <= Length(Result)) do
         begin
            if Result[j] = '"' then
            begin
               Found:= True;
               Break;
            end;
            Inc(j);
         end;
         if Not Found then Exit;    { fatal error }
         if j <= Length(Result) then
            Result:= Copy(Result, 1, p - 1) + Copy(Result, i, j - i) +
               Copy(Result, j + 1, Length(Result));

         p:= Pos('\f"', Result);
      end;
   end;

begin
   if (CurrentAnalizer.ColumnList.Count = 0) and (yylexer as TxqLexer).IsTransform then
   begin
      Column := CurrentAnalizer.TransformColumn;
      (yylexer as TxqLexer).IsTransform:= False;
   end else
      Column := CurrentAnalizer.ColumnList.Add;
   with Column do
   begin
      ColumnExpr:= AColText;
      { this mean that aggregate columns are embedded in ColumnExpr}
      if Assigned(Self.FAggregateList) then
      begin
         Column.AggregateList.Free;    { free previous aggregate list}
         Column.AggregateList := Self.FAggregateList; { assign the current list}
         Self.FAggregateList:= nil;    { define as nil the current }
      end;
      CastType  := Self.FCastType;
      CastLen   := IMax(1, Self.FCastLen);      // only used for strings
      if Length(self.FAsAlias) > 0 then
      begin
         AsAlias:= self.FAsAlias;
         IsAsExplicit:= True;
      end else
      begin
         IsAsExplicit := False;
         AsAlias := StripFields(AColText);
         if AggregateList.Count > 0  then
            case AggregateList[0].Aggregate of
               akSUM: AsAlias := SAggrSUM + StripFields(AggregateList[0].AggregateStr);
               akAVG: AsAlias := SAggrAVG + StripFields(AggregateList[0].AggregateStr);
               akMIN: AsAlias := SAggrMIN + StripFields(AggregateList[0].AggregateStr);
               akMAX: AsAlias := SAggrMAX + StripFields(AggregateList[0].AggregateStr);
               akCOUNT: AsAlias := SAggrCOUNT;
            end;
      end;
   end;

   FAsAlias  := '';
   FCastType := 0;
   FCastLen  := 0;
end;

{ This function will return an aggregate encoded with something like :
  (Aggregate 1) }
function TxqParser.AddAggregate( pAggregate  : TAggregateKind;
                         const pAggregateStr : String ) : String;
begin
  if FAggregateList = nil then
     FAggregateList := TAggregateList.Create;
  with FAggregateList.Add do
  begin
     AggregateStr := pAggregateStr;
     Aggregate    := pAggregate;
     IsDistinctAg := Self.FIsDistinctAggr;
  end;
  Result := Format('{Aggregate %d}', [FAggregateList.Count - 1]);

  Self.FIsDistinctAggr := False;
end;

procedure TxqParser.AddGroupBy(const ColName: String);
var
   GroupBy: TOrderByItem;
   Index, Code: integer;
begin
   Val(ColName, Index, Code);
   GroupBy := CurrentAnalizer.GroupByList.Add;
   if Code = 0 then
   begin
      GroupBy.Alias := '';
      GroupBy.ColIndex := Index - 1;
   end else
   begin
      GroupBy.Alias := ColName;
      GroupBy.ColIndex:= -1;   { means: not defined by number }
   end;
end;

procedure TxqParser.AddOrderBy(const ColName: String; Descending: Boolean);
var
  OrderBy     : TOrderByItem;
  Index, Code : integer;
begin
  Val(ColName, Index, Code);
  OrderBy:= CurrentAnalizer.OrderByList.Add;
  if Code = 0 then
  begin
     OrderBy.Alias := '';
     OrderBy.ColIndex := Index - 1;
  end else
  begin
     OrderBy.Alias := ColName;
     { means: not defined by number and must be solved in checkintegrity }
     OrderBy.ColIndex:= -1;
  end;
  OrderBy.Desc:= Descending;
end;

procedure TxqParser.AddTable(const TableName, Alias: String);
var
   Table: TTableItem;
begin
   Table := CurrentAnalizer.TableList.Add;
   Table.TableName := TableName;
   if Length(Alias) > 0 then
      Table.Alias := Alias
   else
      Table.Alias := TableName;
end;

procedure TxqParser.AddJoin(const LeftExpr, RightExpr, AndExpr: String);
var
   JoinOnItem          : TJoinOnItem;
   LeftJoinOnExprItem  : TJoinOnExprItem;
   RightJoinOnExprItem : TJoinOnExprItem;
   AndJoinOnExprItem   : TJoinOnExprItem;
begin

   if CurrentAnalizer.JoinList.Count <= FCurrentJoin then
   begin
      JoinOnItem          := CurrentAnalizer.JoinList.Add;
      LeftJoinOnExprItem  := JoinOnItem.LeftJoinOn.Add;
      RightJoinOnExprItem := JoinOnItem.RightJoinOn.Add;
      AndJoinOnExprItem   := JoinOnItem.AndJoinOn.Add;
   end else
   begin
      JoinOnItem          := CurrentAnalizer.JoinList[FCurrentJoin];
      LeftJoinOnExprItem  := JoinOnItem.LeftJoinOn.Add;
      RightJoinOnExprItem := JoinOnItem.RightJoinOn.Add;
      AndJoinOnExprItem   := JoinOnItem.AndJoinOn.Add;
   end;

   with JoinOnItem do
   begin
      JoinKind:= Self.FJoinKind;
      RelOperator:= FJoinRelOperator;
   end;
   LeftJoinOnExprItem.Expression   := LeftExpr;
   RightJoinOnExprItem.Expression := RightExpr;
   AndJoinOnExprItem.Expression     := AndExpr;

   Self.FJoinKind:= jkInnerJoin;
   Self.FJoinRelOperator:= ropBETWEEN;
end;

procedure TxqParser.AddJoinCandidate(const LeftExpr, RightExpr: String);
begin
  CurrentAnalizer.LJoinCandidateList.Add( LeftExpr );
  CurrentAnalizer.RJoinCandidateList.Add( RightExpr );
end;

procedure TxqParser.AddHavingColumn( const ColText : String );
var
   Column: TColumnItem;
begin
   Column := CurrentAnalizer.ColumnList.Add;
   with Column do
   begin
      ColumnExpr := ColText;
      if Assigned(Self.FAggregateList) then
      begin
         AggregateList.Free;    { free previous aggregate list}
         AggregateList := Self.FAggregateList; { assign the current list}
         Self.FAggregateList:= nil;    { define as nil the current }
      end;
      { mark also as a temporary column that will be deleted after result set is
        generated }
      IsTemporaryCol := True;
   end;
end;

procedure TxqParser.AddUpdateColumn(const ColumnName, ColumnExpr: String);
var
   UpdateItem: TUpdateItem;
begin
   UpdateItem := CurrentAnalizer.UpdateColumnList.Add;
   with UpdateItem do
   begin
      ColName := ColumnName;
      ColExpr := ColumnExpr;
   end;

end;

procedure TxqParser.AddWhereOptimize(const AFieldName,
                                     ARangeStart,
                                     ARangeEnd: String;
                                     ARelOperator: TRelationalOperator);
var
   WhereOptimize : TWhereOptimizeItem;
   N             : Integer;

   function DeleteStringDelim(const s: String): String;
   begin
      Result := s;
      if Length(Result) > 1 then
      begin
         if (Result[1] in xqbase.SQuote) and (Result[Length(Result)] in xqbase.SQuote) then
            Result:= Copy(Result, 2, Length(Result) - 2);
      end;
   end;

begin
   if Not (yyLexer as TxqLexer).IsWhereActive then Exit;
   N := CurrentAnalizer.WhereOptimizeList.Count - 1;
   if ((ARelOperator in [ropLE, ropLT]) or (ARelOperator in [ropGE, ropGT]) ) and (N > 0) then
   begin

      { I will check if the previous command was something like (CustNo >= 1000)
         and if so, and this is something like (CustNo <= 3000) then
         I will add to the previous optimize and will generate a ropBETWEEN range   }

      WhereOptimize:= CurrentAnalizer.WhereOptimizeList[N - 1];
      if ( ((ARelOperator in [ropLE, ropLT]) and (WhereOptimize.RelOperator in [ropGE, ropGT])) or
         ((ARelOperator in [ropGE, ropGT]) and (WhereOptimize.RelOperator in [ropLE, ropLT])) ) and
         (AnsiCompareText(WhereOptimize.FieldNames, AFieldName) = 0) then
      begin
         WhereOptimize.RangeEnd    := DeleteStringDelim( ARangeEnd );
         WhereOptimize.RelOperator := ropBETWEEN;
         Exit;
      end;
   end;

   WhereOptimize := CurrentAnalizer.WhereOptimizeList.Add;
   with WhereOptimize do
   begin
      FieldNames  := AFieldName;
      RangeStart  := DeleteStringDelim(ARangeStart);
      RangeEnd    := DeleteStringDelim(ARangeEnd);
      RelOperator := ARelOperator;
      CanOptimize := False;
   end;

end;

procedure TxqParser.AddParam(const ParamName: String);
begin
  CurrentAnalizer.Params.CreateParam(ftString, ParamName, ptUnknown);
end;

{ CREATE TABLE }
function TxqParser.CurrentCreateTable: TCreateTableItem;
begin
  if FNumTables > CurrentAnalizer.CreateTableList.Count - 1 then
     Result := CurrentAnalizer.CreateTableList.Add
  else
     Result:= CurrentAnalizer.CreateTableList[ FNumTables ];
end;

procedure TxqParser.SetTableName(const TableName: String);
begin
   CurrentCreateTable.TableName := TableName;
end;

procedure TxqParser.SetFieldParams(AFieldType, AScale, APrecision, ASize, ABlobType: Integer);
begin
  FFieldType := AFieldType;
  FScale     := AScale;
  FPrecision := APrecision;
  FSize      := ASize;
  FBlobType  := ABlobType;
end;

procedure TxqParser.AddCreateField(const FieldName: String);
var
  I: Integer;
begin
  { check if field exists }
  with CurrentCreateTable do
     for I := 0 to FieldCount - 1 do
        if AnsiCompareText(Fields[I].FieldName, FieldName) = 0 then
        begin
           yyerror(SDuplicateFieldName);
           yyabort;
           Exit;
        end;
  if (FFieldType = RW_BLOB) and not (FBlobType in [1..5]) then
  begin
    yyerror(SBlobFieldWrongType);
    yyabort;
  end;
  CurrentCreateTable.Fields.AddField(FieldName,
                                     FFieldType,
                                     FScale,
                                     FPrecision,
                                     FSize,
                                     FBlobType);
end;

procedure TxqParser.AddPrimaryKeyField(const FieldName: String);
var
  I : Integer;
  Found : Boolean;
begin
  { check that the field exists in the list of field names }
  Found := False;
  with CurrentCreateTable do
     for I := 0 to FieldCount - 1 do
        if AnsiCompareText(Fields[I].FieldName, FieldName) = 0 then
        begin
           Found:= True;
           Break;
        end;
  if Not Found then
  begin
     yyerror(SFieldNameNotFound);
     yyabort;
  end;
  CurrentCreateTable.PrimaryKey.Add( FieldName );
end;

function TxqParser.CurrentInsertItem: TInsertItem;
begin
   if FNumInserts >= CurrentAnalizer.InsertList.Count then
      Result := CurrentAnalizer.InsertList.Add
   else
      Result := CurrentAnalizer.InsertList[FNumInserts];
end;

function TxqParser.CreateInListExpression( const Expr : String ) : String;
var
   i : Integer;
begin
   { This subroutine helps to change the syntax:
     custno IN (1000, 2000, 3000)
     to this :
     (custno = 1000) OR (custno = 2000) OR (custno = 3000) }
   Result := '';
   for i := 0 to FInPredicateList.Count - 1 do begin
      Result := Result + Format('(%s = %s)', [Expr, FInPredicateList[i]]);
      if i < FInPredicateList.Count - 1 then
         Result := Result + ' OR ';
   end;
   if FIsNotInList then
      Result := 'NOT ( ' + Result + ' )'
   else
      Result := '( ' + Result + ' )';
   FInPredicateList.Clear;
end;

%}

%start sql

%token _IDENTIFIER
%token _UINTEGER
%token _SINTEGER
%token _NUMERIC
%token _STRING

%token _COMA
%token _LPAREN
%token _RPAREN
%token _PERIOD
%token _SEMICOLON
%token _COLON

%left	 _EQ _NEQ _GT _LT _GE _LE RW_BETWEEN RW_IN RW_LIKE
%left	 _PLUS _SUB RW_OR
%left	 _DIV _MULT RW_AND
%right _NEG       /* Negation--unary minus */
%right _EXP       /* exponentiation */
%left  RW_NOT
%token _ILLEGAL


/* other reserved words and tokens */
%token  RW_TRUE
        RW_FALSE
        RW_SELECT
        RW_DISTINCT
        RW_FROM
        RW_WHERE
        RW_ORDER
        RW_BY
        RW_ASC
        RW_DESC
        RW_AS
		  RW_INNER
		  RW_OUTER
		  RW_FULL
		  RW_JOIN
		  RW_ON
		  RW_GROUP
		  RW_HAVING
        RW_ANY
        RW_ALL
		  RW_SUM
		  RW_AVG
		  RW_COUNT
		  RW_MIN
		  RW_MAX
		  RW_LEFT
		  RW_RIGHT
        RW_LEADING
        RW_TRAILING
        RW_BOTH
        RW_TRIM
        RW_EXTRACT
        RW_YEAR
        RW_MONTH
        RW_DAY
        RW_HOUR
        RW_MINUTE
        RW_SECOND
        RW_FOR
        RW_SUBSTRING
        RW_DELETE
        RW_UPDATE
        RW_INSERT
        RW_INTO
        RW_VALUES
        RW_SET
        RW_CAST
        RW_CHAR
        RW_INTEGER
        RW_BOOLEAN
        RW_DATE
        RW_TIME
        RW_DATETIME
        RW_FLOAT
        RW_ESCAPE
        RW_NOW
        _COMMENT
        _BLANK
        _TAB
        _NEWLINE
        RW_CREATE
        RW_TABLE
        RW_SMALLINT
        RW_MONEY
        RW_AUTOINC
        RW_PRIMARY
        RW_KEY
        RW_BLOB
        RW_INDEX
        RW_UNIQUE
        RW_DROP
        RW_TRANSFORM
        RW_PIVOT
        RW_MERGE
        RW_WITH
        RW_IS
        RW_NULL


%type <string>

%%

sql : select_statement {FAnalizer.Statement := ssSelect;}
    | transform_statement {FAnalizer.Statement := ssSelect;}
    | merge_statement {FAnalizer.Statement := ssMerge;}
    | update_statement {FAnalizer.Statement := ssUpdate;}
    | delete_statement {FAnalizer.Statement := ssDelete;}
    | insert_statement {FAnalizer.Statement := ssInsert;}
    | createtable_statement {FAnalizer.Statement := ssCreateTable;}
    | createindex_statement {FAnalizer.Statement := ssCreateIndex;}
    | droptable_statement {FAnalizer.Statement := ssDropTable;}
    | dropindex_statement {FAnalizer.Statement := ssDropIndex;}
    ;

/*  SELECT statement */
select_statement : select_clause from_clause where_clause groupby_clause orderby_clause end_statement
                 ;
/* SELECT clause */
select_clause : select_case _MULT              {CurrentAnalizer.DoSelectAll := True;}
              | select_case list_expr_field
              ;

select_case : RW_SELECT
            | RW_SELECT RW_DISTINCT {CurrentAnalizer.IsDistinct:= True;}
            ;

list_expr_field : expr_field
                | list_expr_field _COMA expr_field
                ;

expr_field : select_expr as_identifier             { AddColumn( $<string>1 ); }
           | _IDENTIFIER _PERIOD _MULT             { CurrentAnalizer.TableAllFields.Add( $<string>1 ); }
           | RW_CAST _LPAREN select_expr RW_AS data_type _RPAREN as_identifier { AddColumn( $<string>3 ); }
           ;

as_identifier : /* empty */
              | RW_AS _IDENTIFIER  {FAsAlias:= $<string>2;}
              ;

select_expr  : define_field
             | _STRING
             | _SINTEGER
             | _UINTEGER
             | _NUMERIC
             | RW_TRUE
             | RW_FALSE
             | RW_NOW
             | select_func
             | trim_function
             | extract_function
             | substring_function

             | select_expr _PLUS select_expr {$<string>$ := $<string>1 + ' + '  + $<string>3; }
             | select_expr _SUB select_expr  {$<string>$ := $<string>1 + ' - '  + $<string>3; }
             | select_expr _MULT select_expr {$<string>$ := $<string>1 + ' * '  + $<string>3; }
             | select_expr _DIV select_expr  {$<string>$ := $<string>1 + ' / '  + $<string>3; }
             | select_expr _EXP select_expr  {$<string>$ := $<string>1 + ' ^ '  + $<string>3; }

             | select_expr _GT select_expr   {$<string>$ := $<string>1 + ' > '  + $<string>3; }
             | select_expr _LT select_expr   {$<string>$ := $<string>1 + ' < '  + $<string>3; }
             | select_expr _NEQ select_expr  {$<string>$ := $<string>1 + ' <> ' + $<string>3; }
             | select_expr _GE select_expr   {$<string>$ := $<string>1 + ' >= ' + $<string>3; }
             | select_expr _LE select_expr   {$<string>$ := $<string>1 + ' <= ' + $<string>3; }
             | select_expr _EQ select_expr   {$<string>$ := $<string>1 + ' = '  + $<string>3; }

             | _SUB select_expr %prec _NEG   {$<string>$ := $<string>1 + $<string>2;}
             | _PLUS select_expr %prec _NEG  {$<string>$ := $<string>1 + $<string>2;}

             | _LPAREN select_expr _RPAREN    {$<string>$ := $<string>1 + $<string>2 + $<string>3;}

             /* Aggregate functions follows */
             | RW_SUM _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akSUM, $<string>4); }
             | RW_MIN _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akMIN, $<string>4); }
             | RW_MAX _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akMAX, $<string>4); }
             | RW_AVG _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akAVG, $<string>4); }
             | RW_COUNT _LPAREN _MULT _RPAREN
               { $<string>$ := AddAggregate(akCOUNT, '0'); }
             | RW_COUNT _LPAREN distinct_aggr select_expr _RPAREN
               { $<string>$ := AddAggregate(akCOUNT, $<string>4); }
             ;

distinct_aggr : /* empty */
              | RW_DISTINCT  { Self.FIsDistinctAggr:= True; }
              ;

select_func : _IDENTIFIER _LPAREN select_list_param _RPAREN {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4;}
            ;

trim_function : RW_TRIM _LPAREN trim_position _STRING RW_FROM where_expr _RPAREN
                {$<string>$ := Format('SQLTRIM(%s, %s, %d)',[$<string>4, $<string>6, FTrimPosition]);}
              ;

trim_position : RW_LEADING   {FTrimPosition := 0;}
              | RW_TRAILING  {FTrimPosition := 1;}
              | RW_BOTH      {FTrimPosition := 2;}
              ;

extract_function : RW_EXTRACT _LPAREN extract_field RW_FROM where_expr _RPAREN
                   {$<string>$ := Format('SQLEXTRACT(%s, %d)',[$<string>5, FExtractField]);}
                 ;

extract_field : RW_YEAR   { FExtractField:= 0; }
              | RW_MONTH  { FExtractField:= 1; }
              | RW_DAY    { FExtractField:= 2; }
              | RW_HOUR   { FExtractField:= 3; }
              | RW_MINUTE { FExtractField:= 4; }
              | RW_SECOND { FExtractField:= 5; }
              ;

substring_function : RW_SUBSTRING _LPAREN where_expr RW_FROM _UINTEGER for_length _RPAREN
                     {if Length(FForLength) > 0 then
                         $<string>$ := Format('COPY(%s,%s,%s)',[$<string>3,$<string>5,FForLength])
                      else
                         $<string>$ := Format('COPY(%s,%s,LENGTH(%s))',[$<string>3,$<string>5,$<string>3]);
                      FForLength:='';}
                   ;

for_length : /* empty */
           | RW_FOR _UINTEGER {FForLength:= $<string>2;}
           ;

data_type : RW_CHAR _LPAREN _UINTEGER _RPAREN   { FCastType := RW_CHAR;
                                                  FCastLen:= StrToInt( $<string>3 ); }
          | RW_INTEGER                          { FCastType := RW_INTEGER; }
          | RW_BOOLEAN                          { FCastType := RW_BOOLEAN; }
          | RW_DATE                             { FCastType := RW_DATE; }
          | RW_TIME                             { FCastType := RW_TIME; }
          | RW_DATETIME                         { FCastType := RW_DATETIME; }
          | RW_FLOAT                            { FCastType := RW_FLOAT; }
          | RW_MONEY                            { FCastType := RW_MONEY; }
          ;

define_field : _IDENTIFIER      {$<string>$ := Format('\f"%s"',[$<string>1]); }
             | qualified_field
             ;

define_param : _COLON _IDENTIFIER  { $<string>$ := $<string>1 + $<string>2;
                                     AddParam( $<string>2 ); }
             ;

qualified_field : _IDENTIFIER _PERIOD _IDENTIFIER
                    { $<string>$ := Format('\f"%s.%s"',[$<string>1, $<string>3]); }
                ;

select_list_param : /* empty */
                  | _MULT
                  | list_param_expr
                  ;

list_param_expr : select_expr
                | list_param_expr _COMA select_expr {$<string>$:= $<string>1 + ', ' + $<string>3;}
                ;

/* FROM clause */
from_clause : RW_FROM list_tables
            | RW_FROM table_identifier list_join
            ;

list_tables : table_identifier
            | list_tables _COMA table_identifier
            ;

table_identifier : _IDENTIFIER               {AddTable($<string>1, '');}
                 | _IDENTIFIER _IDENTIFIER   {AddTable($<string>1, $<string>2);}
                 ;

/* JOIN predicate */
list_join : list_join_table
          | list_join    list_join_table
          ;

list_join_table : join_action table_identifier RW_ON list_joined_fields  {Inc(FCurrentJoin);}
                ;

join_action : RW_JOIN                     {FJoinKind:= jkInnerJoin;     }
            | RW_INNER RW_JOIN            {FJoinKind:= jkInnerJoin;     }
            | RW_LEFT RW_JOIN             {FJoinKind:= jkLeftOuterJoin; }
            | RW_LEFT RW_OUTER RW_JOIN    {FJoinKind:= jkLeftOuterJoin; }
            | RW_RIGHT RW_JOIN            {FJoinKind:= jkRightOuterJoin;}
            | RW_RIGHT RW_OUTER RW_JOIN   {FJoinKind:= jkRightOuterJoin;}
            | RW_FULL RW_JOIN             {FJoinKind:= jkFullOuterJoin; }
            | RW_FULL RW_OUTER RW_JOIN    {FJoinKind:= jkFullOuterJoin; }
            ;

list_joined_fields: field_joined
                  | list_joined_fields RW_AND field_joined
                  ;

field_joined: _LPAREN define_field join_rel_operator define_field _RPAREN { AddJoin($<string>2, $<string>4, ''); }
            | _LPAREN define_field join_rel_operator define_field RW_AND where_expr _RPAREN { AddJoin($<string>2, $<string>4, $<string>6); }
            ;      /* select_expr */

join_rel_operator : _EQ    {FJoinRelOperator := ropBETWEEN;}
                  | _GT    {FJoinRelOperator := ropGT;}
                  | _LT    {FJoinRelOperator := ropLT;}
                  | _GE    {FJoinRelOperator := ropGE;}
                  | _LE    {FJoinRelOperator := ropLE;}
                  | _NEQ   {FJoinRelOperator := ropNEQ;}
                  ;

/* WHERE clause */
where_expr : define_field
           | where_constant
           | RW_NOW           /* NOW is a reserved word */
           | trim_function
           | extract_function
           | substring_function
           | is_null

           | _IDENTIFIER _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | RW_LEFT _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | RW_RIGHT _LPAREN where_list_param _RPAREN
             {$<string>$ := $<string>1 + $<string>2 + $<string>3 + $<string>4; }
           | where_expr RW_BETWEEN where_constant RW_AND where_constant
             {$<string>$ := Format('(%s >= %s) AND (%s <= %s)',
                  [$<string>1, $<string>3, $<string>1, $<string>5]);
             AddWhereOptimize( $<string>1, $<string>3, $<string>5, ropBETWEEN ); }
           | where_expr in_operator _LPAREN in_predicate _RPAREN
               { $<string>$ :=  CreateInListExpression( $<string>1 ); }
           | where_expr in_operator anyall_subquery
                { if Pos('NOT',UpperCase($<string>2)) = 0 then
                    $<string>$ := $<string>1 + ' = (subquery)'
                  else
                    $<string>$ := $<string>1 + ' <> (subquery)'; }

           | where_expr RW_LIKE where_constant escape_character
               { if FEscapeChar = '' then FEscapeChar := #39#39; 
                  $<string>$ := Format('SQLLIKE(%s, %s, %s)',[$<string>1, $<string>3, FEscapeChar]); FEscapeChar:= '';}
           | where_expr RW_NOT RW_LIKE where_constant escape_character
               { if FEscapeChar = '' then FEscapeChar := #39#39;
                 $<string>$ := Format('SQLNOTLIKE(%s, %s, %s)',[$<string>1, $<string>4, FEscapeChar]); FEscapeChar:= '';}

           | _SUB where_expr %prec _NEG   {$<string>$ := $<string>1 + $<string>2;}
           | _PLUS where_expr %prec _NEG  {$<string>$ := $<string>1 + $<string>2;}

           | where_expr _PLUS where_expr  {$<string>$ := $<string>1 + ' + ' + $<string>3;}
           | where_expr _SUB where_expr   {$<string>$ := $<string>1 + ' - ' + $<string>3;}
           | where_expr _MULT where_expr  {$<string>$ := $<string>1 + ' * ' + $<string>3;}
           | where_expr _DIV where_expr   {$<string>$ := $<string>1 + ' / ' + $<string>3;}
           | where_expr _EXP where_expr   {$<string>$ := $<string>1 + ' ^ ' + $<string>3;}

           | where_expr _GE where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropGE); }
           | where_expr _LE where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropLE); }
           | where_expr _GT where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropGT); }
           | where_expr _LT where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropLT); }
           | where_expr _EQ where_expr  {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddJoinCandidate($<string>1, $<string>3);
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropBETWEEN); }
           | where_expr _NEQ where_expr {$<string>$ := $<string>1 + #32 + $<string>2 + #32 + $<string>3;
                                         AddWhereOptimize( $<string>1, $<string>3, $<string>3, ropNEQ); }

           | where_expr RW_AND where_expr {$<string>$ := $<string>1 + ' AND ' + $<string>3;}
           | where_expr RW_OR where_expr  {$<string>$ := $<string>1 + ' OR ' + $<string>3;}
           | RW_NOT where_expr            {$<string>$ := ' NOT ' + $<string>2;}
           | _LPAREN where_expr _RPAREN   {$<string>$ := $<string>1 + $<string>2 + $<string>3;}
           | anyall_subquery              {$<string>$ := '(subquery)';}
           ;

is_null : define_field RW_IS RW_NULL         { $<string>$ := Format('ISNULL(%s,TRUE)', [$<string>1]); }
        | define_field RW_IS RW_NOT RW_NULL  { $<string>$ := Format('ISNULL(%s,FALSE)', [$<string>1]); }
        ;

in_operator : RW_IN         { FIsNotInList := False; }
            | RW_NOT RW_IN  { $<string>$ := $<string>1 + #32 + $<string>2; FIsNotInList := True; }
            ;

in_predicate : where_constant                    { FInPredicateList.Add( $<string>1 ); }
             | in_predicate _COMA where_constant { FInPredicateList.Add( $<string>3 ); }
             ;

where_constant : _UINTEGER
               | _SINTEGER
               | _NUMERIC
               | _STRING
               | RW_TRUE       { $<string>$ := 'DummyBoolean(True)'; }
               | RW_FALSE      { $<string>$ := 'DummyBoolean(False)'; }
               | define_param
               ;

where_list_param : where_expr
                 | where_list_param _COMA where_expr {$<string>$:= $<string>1 + ', ' + $<string>3;}
                 ;

where_clause : /* empty */
             | RW_WHERE where_expr                  { CurrentAnalizer.WhereStr := $<string>2; }
             ;

anyall_subquery : subquery         {CurrentAnalizer.SubQueryKind:= skAny;}
                | RW_ANY subquery  {CurrentAnalizer.SubQueryKind:= skAny;}
                | RW_ALL subquery  {CurrentAnalizer.SubQueryKind:= skAll;}
                ;

subquery : _LPAREN select_statement _RPAREN
           { with TxqLexer(yylexer) do NumSelects := NumSelects - 1; }
         ;

escape_character : /* empty */        {FEscapeChar := '';}
                 | RW_ESCAPE _STRING  {FEscapeChar := $<string>2;}
                 ;

/* GROUP BY clause */

groupby_clause : /* empty */
               | RW_GROUP RW_BY list_fields_group having_predicate
               ;


list_fields_group : define_field_index                         {AddGroupBy( $<string>1 );}
                  | list_fields_group _COMA define_field_index {AddGroupBy( $<string>3 );}
                  ;

                   /*  \f"field" is a special code for signaling fields without table qualifier and solved later */
define_field_index : _IDENTIFIER         { $<string>$ := Format('\f"%s"', [$<string>1]); }
                   | qualified_field
                   | _UINTEGER           { $<string>$ := $<string>1; }
                   ;

having_predicate : /* empty */
                 | RW_HAVING select_expr
                   { AddHavingColumn( $<string>2 );
                     CurrentAnalizer.HavingCol := CurrentAnalizer.ColumnList.Count-1; }
                 ;

/* ORDER BY clause */

orderby_clause : /* empty */
               | RW_ORDER RW_BY list_fields_order
               ;

list_fields_order : define_fields_order
                  | list_fields_order _COMA define_fields_order
                  ;

define_fields_order : define_field_index           {AddOrderBy( $<string>1, False );}
                    | define_field_index RW_ASC    {AddOrderBy( $<string>1, False );}
                    | define_field_index RW_DESC   {AddOrderBy( $<string>1, True );}
                    ;

/* end of select */
end_statement : /* empty */
              | _SEMICOLON
              ;

/* UPDATE statement */
update_statement : RW_UPDATE table_identifier RW_SET list_update_columns where_clause end_statement
                 ;

update_column : _IDENTIFIER _EQ where_expr {AddUpdateColumn($<string>1,$<string>3);}
              ;

list_update_columns : update_column
                    | list_update_columns _COMA update_column
                    ;

/* DELETE statement */
delete_statement : RW_DELETE RW_FROM table_identifier where_clause end_statement
                 ;

/* INSERT statement */
insert_statement : insert_satement_list end_statement
                 ;

insert_satement_list : insert_one_record                        { Inc(FNumInserts); }
                     | insert_satement_list insert_one_record   { Inc(FNumInserts); }
                     ;

insert_one_record : RW_INSERT RW_INTO _IDENTIFIER insert_columns_list insert_values_list
                      { CurrentInsertItem.TableName := $<string>3 ; }
                  ;

insert_columns_list : _LPAREN columns_list _RPAREN
                    | _LPAREN _MULT _RPAREN
                    ;

/* define the fields to update */
columns_list : _IDENTIFIER                       {CurrentInsertItem.FieldNames.Add($<string>1);}
             | columns_list _COMA _IDENTIFIER    {CurrentInsertItem.FieldNames.Add($<string>3);}
             ;

insert_values_list : RW_VALUES _LPAREN values_list _RPAREN
                   | subquery
                   ;

/* define the values to store in fields */
values_list : where_expr                    {CurrentInsertItem.ExprList.Add( $<string>1 );}
            | values_list _COMA where_expr  {CurrentInsertItem.ExprList.Add( $<string>3 );}
            ;

/* CREATE TABLE STATEMENT */
createtable_statement : createtable_list end_statement
                      ;

createtable_list : createone_table                   {Inc(FNumTables);}
                 | createtable_list createone_table  {Inc(FNumTables);}
                 ;

createone_table : RW_CREATE RW_TABLE _STRING _LPAREN create_list primary_key _RPAREN
                  {SetTableName( Copy($<string>3, 2, Length($<string>3) - 2) );}
                ;

/* the type of fields that can be created */
field_type : RW_CHAR _LPAREN _UINTEGER _RPAREN  {SetFieldParams(RW_CHAR,0,0,StrToInt($<string>3),0);}
           | RW_INTEGER                         {SetFieldParams(RW_INTEGER,0,0,0,0);}
           | RW_SMALLINT                        {SetFieldParams(RW_SMALLINT,0,0,0,0);}
           | RW_BOOLEAN                         {SetFieldParams(RW_BOOLEAN,0,0,0,0);}
           | RW_DATE                            {SetFieldParams(RW_DATE,0,0,0,0);}
           | RW_TIME                            {SetFieldParams(RW_TIME,0,0,0,0);}
           | RW_DATETIME                        {SetFieldParams(RW_DATETIME,0,0,0,0);}
           | RW_MONEY                           {SetFieldParams(RW_MONEY,0,0,0,0);}
           | RW_FLOAT                           {SetFieldParams(RW_FLOAT,0,0,0,0);}
           | RW_FLOAT _LPAREN _UINTEGER _RPAREN {SetFieldParams(RW_FLOAT,StrToInt($<string>3),0,0,0);}
           | RW_FLOAT _LPAREN _UINTEGER _COMA _UINTEGER _RPAREN {SetFieldParams(RW_FLOAT,StrToInt($<string>3),StrToInt($<string>5),0,0);}
           | RW_AUTOINC                         {SetFieldParams(RW_AUTOINC,0,0,0,0);}
           | RW_BLOB _LPAREN _UINTEGER _RPAREN  {SetFieldParams(RW_BLOB,0,0,0,StrToInt($<string>3));}
           ;

create_list : _IDENTIFIER field_type                     {AddCreateField($<string>1);}
            | create_list _COMA _IDENTIFIER field_type   {AddCreateField($<string>3);}
            ;

primary_key : /* empty */
            | RW_PRIMARY RW_KEY _LPAREN create_field_list _RPAREN
            ;

create_field_list : _IDENTIFIER                           {AddPrimaryKeyField($<string>1);}
                  | create_field_list _COMA _IDENTIFIER   {AddPrimaryKeyField($<string>3);}
                  ;

/* CREATE INDEX statement */
createindex_statement : RW_CREATE index_unique index_order RW_INDEX _IDENTIFIER RW_ON _STRING _LPAREN index_field_list _RPAREN end_statement
                        {CurrentAnalizer.IndexName  := $<string>5;
                         CurrentAnalizer.IndexTable := Copy($<string>7, 2, Length($<string>7) - 2); }
                      ;

index_unique : /* empty */
              | RW_UNIQUE  {CurrentAnalizer.IndexUnique:= True;}
              ;

index_order : /* empty */
            | RW_ASC
            | RW_DESC      {CurrentAnalizer.IndexDescending:= True;}
            ;

index_field_list : _IDENTIFIER   {CurrentAnalizer.IndexColumnList.Add($<string>1);}
                 | index_field_list _COMA _IDENTIFIER  {CurrentAnalizer.IndexColumnList.Add($<string>3);}
                 ;

/* DROP TABLE statement */
droptable_statement : RW_DROP RW_TABLE _STRING end_statement {CurrentAnalizer.IndexTable:= Copy($<string>3, 2, Length($<string>3) - 2);}
                    ;

/* DROP INDEX statement */
dropindex_statement : RW_DROP RW_INDEX _STRING _IDENTIFIER end_statement
                     { CurrentAnalizer.IndexTable:= Copy($<string>3, 2, Length($<string>3) - 2);
                       CurrentAnalizer.IndexName := $<string>4; }
                    ;

/* TRANSFORM...PIVOT statement */
transform_statement : RW_TRANSFORM transf_aggregate select_statement RW_PIVOT select_expr pivot_in_predicate end_statement
                      { CurrentAnalizer.PivotStr := $<string>5; }
                    ;

/* this is needed in order to reduce this column first */
transf_aggregate : select_expr  { AddColumn( $<string>1 ); }
                 | RW_CAST _LPAREN select_expr RW_AS data_type _RPAREN { AddColumn( $<string>3 ); }
                 ;

pivot_in_predicate : /* empty */
                   | RW_IN _LPAREN pivot_in_list _RPAREN
                   ;

pivot_in_list : where_constant
                {CurrentAnalizer.PivotInList.Add( RemoveStrDelim( $<string>1));}
              | pivot_in_list _COMA where_constant
                {CurrentAnalizer.PivotInList.Add( RemoveStrDelim( $<string>3));}
              ;

/* MERGE WITH statement */
merge_statement : RW_MERGE select_statement RW_WITH select_statement

%%




