{*******************************************************}
{                                                       }
{       Miscellaneous routines used in TxQuery dataset  }
{                                                       }
{       Copyright (c) 1999-2000 Alfonso moreno          }
{                                                       }
{     Written by:                                       }
{       Alfonso moreno                                  }
{       Hermosillo, Sonora, Mexico.                     }
{       Internet:  gismap@hmo.megared.net.mx            }
{                  luisarvayo@yahoo.com                 }
{                  inconmap@prodigy.net.mx              }
{       http://www.sigmap.com/txquery.htm               }
{                                                       }
{*******************************************************}
unit XQMiscel;

{$I XQ_FLAG.INC}
interface

uses
  SysUtils,
  Windows,
  Messages,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  StdCtrls,
  IniFiles,
  ExtCtrls,
  DB,
  prExprQ
{$IFDEF WITHBDE}
  , DBTables
  , bde
{$endIF}
  ;

type

  {Buffered read/write class - used for fast sequencial reads/writes}
  PCharArray = ^TCharArray;
  TCharArray = array [0..0] of Char;

  TBufferedReadWrite = Class(TStream)
  private
     FStream        : TStream;
     FValidBytesInSector : Integer;
     FCurrentSector: Integer;
     FOffsetInSector: Integer;
     PBuffer        : PCharArray;
     FSizeOfSector  : Integer;
     FFreeStream    : Boolean;
     FMustFlush     : Boolean;
     procedure FlushBuffer;
  public
     constructor Create(F: TStream; FreeStream: Boolean; BuffSize: Integer);
     destructor Destroy; override;
     function Read(var Buffer; Count: Longint): Longint; override;
     function Seek(Offset: Longint; Origin: Word): Longint; override;
     function Write(const Buffer; Count: Longint): Longint; override;
     procedure ResetPos;
  end;

  { Miscelaneous routines }
  procedure FreeObject(var Obj);
  procedure ReplaceString (var Work : String; const Old, NNew : String);
  function TrimCRLF(const s:string):string;
  function MessageToUser( const Msg: String; Atype: TMsgDlgtype): Word;
  function Max(const A, B: Double): Double;
  function Min(const A, B: Double): Double;
  function IMax(A, B: Integer): Integer;
  function IMin(A, B: Integer): Integer;
  function GetRecordNumber(DataSet: TDataSet): Integer;
  procedure SetRecordNumber(DataSet: TDataSet; RecNum: Integer);
  {$IFDEF XQDEMO}
  function IsDelphiRunning : boolean;
  {$endIF}
  function GetTemporaryFileName(const Prefix : string) : string;
  function AddSlash(const Path : string) : string;
  function RemoveSlash(const Path : string) : string;
  function Field2Exprtype(Datatype: TFieldtype): TExprtype;
  function RemoveStrDelim( const S: String): String;


implementation

uses
  xqbase, xquery, xqconsts;

// miscelaneous
function Field2Exprtype(Datatype: TFieldtype): TExprtype;
begin
  if Datatype in ftNonTexttypes then
     Result := ttString
  else
     case Datatype of
        ftString{$IFDEF LEVEL4},ftFixedChar,ftWideString{$endIF} :
           Result := ttString;
        ftFloat,ftCurrency,ftBCD,ftDate,ftTime,ftDateTime:
           Result := ttFloat;
        ftAutoInc,ftSmallInt,ftInteger,ftWord
        {$ifndef LEVEL3},ftLargeInt{$endif} :
           Result := ttInteger;
        ftBoolean:
           Result := ttBoolean;
        else
           Result := ttUnknown;
     end;
end;

{
procedure LeftSet(var S1: String; const S2: String);
var
  N,N1,N2: Integer;
begin
  N1 := Length(S1); if N1 = 0 then Exit;
  N2 := Length(S2);
  N := N2; if N1 < N then N := N1;
  Move(S2[1], S1[1], N);
end; }

procedure FreeObject(var Obj);
begin
  TObject(Obj).Free;
  Pointer(Obj) := nil;
end;

procedure ReplaceString (var Work : String; const Old, NNew : String);
var
  OldLen, p: Integer;
begin
    if AnsiCompareText(Old,NNew)=0 then Exit;
    OldLen := Length(Old);
    p :=Pos(Old, Work);
    while p > 0 do
    begin
       Delete(Work, p, OldLen);
       Insert(NNew, Work, p);
       p :=Pos(Old, Work);
    end;
end;

function TrimCRLF(const s:string):string;
begin
  result := Trim(s);
  ReplaceString(result, #13, '');
  ReplaceString(result, #10, '');
end;

function MessageToUser( const Msg: String; Atype: TMsgDlgtype): Word;
begin
  Result := MessageDlg(Msg, Atype, [mbOk], 0);
end;

function IMax(A, B: Integer): Integer;
begin
  if A > B then Result := A else Result := B;
end;

function IMin(A, B: Integer): Integer;
begin
  if A < B then Result := A else Result := B;
end;

function Max(const A, B: Double): Double;
begin
  if A > B then Result := A else Result := B;
end;

function Min(const A, B: Double): Double;
begin
  if A < B then Result := A else Result := B;
end;

function RemoveStrDelim( const S: String): String;
begin
   if (Length(S) >= 2) and
      (S[1] in xqbase.SQuote) and (S[Length(S)] in xqbase.SQuote) then
      Result := Copy(S, 2, Length(S) - 2)
   else
      Result := S;
end;

{$IFDEF XQDEMO}
const
  A2 = 'TAlignPalette';
  A3 = 'TPropertyInspector';
  A4 = 'TAppBuilder';

function IsDelphiRunning : boolean;
var
  H2, H3, H4 : Hwnd;
begin
  H2 := FindWindow(A2, nil);
  H3 := FindWindow(A3, nil);
  H4 := FindWindow(A4, nil);
  Result := (H2 <> 0) and (H3 <> 0) and (H4 <> 0);
end;
{$endIF}

function AddSlash(const Path : string) : string;
begin
  result := Path;
  if (Length(result) > 0) and (result[length(result)] <> '\') then
    result := result + '\'
end;

function RemoveSlash(const Path : string) : string;
var
  rlen : integer;
begin
  result := Path;
  rlen := length(result);
  if (rlen > 0) and (result[rlen] = '\') then
    Delete(result, rlen, 1);
end;


function GetRecordNumber(DataSet: TDataSet): Integer;
{$IFDEF WITHBDE}
var
  CursorProps: CurProps;
  RecordProps: RECProps;
{$endIF}
begin
  Result := 0;
  {$IFDEF WITHBDE}
  if DataSet is TBDEDataSet then
  begin
     with TBDEDataSet(DataSet) do
     begin
       if (State = dsInactive) then exit;
       Check(DbiGetCursorProps(Handle, CursorProps));
       UpdateCursorPos;
       Check(DbiGetRecord(Handle, dbiNOLOCK, nil, @RecordProps));
       case CursorProps.iSeqNums of
         0: Result := RecordProps.iPhyRecNum;
         1: Result := RecordProps.iSeqNum;
       end;
     end;
  end else
  begin
  {$endIF}
     if (DataSet.State = dsInactive) then Exit;
     Result := DataSet.RecNo;     // dataset must support recno property
  {$IFDEF WITHBDE}
  end;
  {$endIF}
end;

procedure SetRecordNumber(DataSet: TDataSet; RecNum: Integer);
{$IFDEF WITHBDE}
var
  CursorProps: CurProps;
{$endIF}
begin
  {$IFDEF WITHBDE}
  if DataSet is TBDEDataSet then
  begin
     with TBDEDataSet(DataSet) do
     begin
       if (State = dsInactive) then exit;

       Check(DbiGetCursorProps(Handle, CursorProps));

       case CursorProps.iSeqNums of
         0: Check( DBISetToRecordNo( Handle, RecNum ) );
         1: Check( DBISetToSeqNo( Handle, RecNum ) );
       end;
     end;
  end else
  begin
  {$endIF}
     if (DataSet.State = dsInactive) then Exit;
     DataSet.RecNo := RecNum;
  {$IFDEF WITHBDE}
  end;
  {$endIF}
  DataSet.ReSync([]);
end;

{ TBufferedReadWrite - class implementation
   used for fast buffered readings/writing from files }
constructor TBufferedReadWrite.Create( F: TStream; FreeStream: Boolean; BuffSize: integer);
begin
  inherited Create;

  FStream          := F;
  FFreeStream      := FreeStream;
  if BuffSize < dsMaxStringSize then
     FSizeOfSector := dsMaxStringSize
  else
     FSizeOfSector := BuffSize;

  GetMem( PBuffer, FSizeOfSector );

  FCurrentSector := -1;    { any sector available }

  Seek(F.Position, 0);

end;

destructor TBufferedReadWrite.Destroy;
begin
  FlushBuffer;
  FreeMem(PBuffer, FSizeOfSector);
  if FFreeStream then FStream.Free;
  inherited Destroy;
end;

procedure TBufferedReadWrite.ResetPos;
begin
  FlushBuffer;
  FCurrentSector := -1;
end;

function TBufferedReadWrite.Seek(Offset: Longint; Origin: Word): Longint;
var
  TmpSector       : LongInt;
begin
  if Origin = soFromBeginning then
     { from start of file }
     Result := Offset
  else if Origin = soFromCurrent then
     { from current position }
     Result := (FCurrentSector * FSizeOfSector + FOffsetInSector) + Offset
  else if Origin = soFromEnd then
  begin
     { flush the buffer in order to detect the size of the file }
     FlushBuffer;
     Result := FStream.Size + Offset;
  end;
  TmpSector := Result div FSizeOfSector;
  FOffsetInSector := Result mod FSizeOfSector;
  if FCurrentSector = TmpSector then Exit;

  FlushBuffer;
  FStream.Seek( TmpSector * FSizeOfSector, soFromBeginning );
  FValidBytesInSector := FStream.Read( PBuffer^, FSizeOfSector);
  FCurrentSector := TmpSector;
end;

function TBufferedReadWrite.Read(var Buffer; Count: Longint): Longint;
var
  N, Diff: Longint;
  { I cannot read more data than dsMaxStringSize chars at a time (take care with text) }
  Temp   : Array[0..dsMaxStringSize - 1] of char Absolute Buffer;

  function ReadNextBuffer: Boolean;
  begin
    { write the buffer if not flushed to disk }
    FlushBuffer;
    { read next buffer and return false if cannot }
    FValidBytesInSector := FStream.Read(PBuffer^, FSizeOfSector);
    Inc(FCurrentSector);
    FOffsetInSector := 0;
    Result := (FValidBytesInSector > 0);
  end;

begin
  Result := 0;
  if (Count < 1) or (Count > SizeOf(Temp)) then Exit;
  if FOffsetInSector + Count <= FValidBytesInSector then
  begin
     { in the buffer is full data }
     Move( PBuffer^[FOffsetInSector], Buffer, Count );
     Inc( FOffsetInSector, Count );
     Result := Count;
  end else
  begin
     { in the current buffer is partial data }
     N := FValidBytesInSector - FOffsetInSector;
     Move( PBuffer^[FOffsetInSector], Buffer, N );
     Result := N;
     if not ReadNextBuffer then Exit;
     Diff := Count - N;
     Move( PBuffer^[FOffsetInSector], Temp[N], Diff );
     Inc(FOffsetInSector, Diff);
     Inc(Result, Diff);
  end;
end;

function TBufferedReadWrite.Write(const Buffer; Count: Longint): Longint;
var
  N, Diff: Longint;
  { I cannot read more data than dsMaxStringSize chars at a time (take care with text) }
  Temp   : Array[0..dsMaxStringSize - 1] of char Absolute Buffer;

  procedure WriteFullBuffer;
  begin
    FStream.Seek( FCurrentSector * FSizeOfSector, 0);
    FStream.Write(PBuffer^, FSizeOfSector);
    Inc(FCurrentSector);
    FMustFlush := True;  { is a flag indicating that the current buffer is not begin written yet }
    FOffsetInSector := 0;
  end;

begin
  Result := 0;
  if (Count < 1) or (Count > SizeOf(Temp)) then Exit;
  if FOffsetInSector + Count <= FValidBytesInSector then
  begin
     { in the buffer is full data }
     Move( Buffer, PBuffer^[FOffsetInSector], Count );
     Inc( FOffsetInSector, Count );
     FMustFlush := True;
     Result := Count;
  end else
  begin
     { in the current buffer will write partial data }
     N := FValidBytesInSector - FOffsetInSector;
     Move( Buffer, PBuffer^[FOffsetInSector], N );
     Result := N;
     WriteFullBuffer;
     Diff := Count - N;
     Move( Temp[N], PBuffer^[FOffsetInSector], Diff );
     Inc(FOffsetInSector, Diff);
     Inc(Result, Diff);
     //Result := Count;
  end;
end;

procedure TBufferedReadWrite.FlushBuffer;
begin
  if (FCurrentSector >= 0) and FMustFlush and (FOffsetInSector > 0) then
  begin
     FStream.Seek( FCurrentSector * FSizeOfSector, 0);
     FStream.Write(PBuffer^, FOffsetInSector);
     FMustFlush := False;
  end;
end;

{ miscellaneous procedures }

function GetTemporaryFileName(const Prefix : string) : string;
var
  TempPath : array[0..1023] of char;
  FileName : array[0..1023] of char;
begin
  GetTempPath(1023, TempPath);
  GetTempFileName(TempPath, PChar(Prefix), 0, FileName);
  result := FileName;
end;

initialization

end.
