unit SyntaxHi;

interface

{$I XQ_FLAG.INC}
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  xqlex, xqyacc, StdCtrls, Inifiles, ComCtrls, Buttons, RichEdit, xquery;

type
  TPosChangeEvent = procedure(Sender: TObject; Row, Col: Integer) of object;
  TUpdateMode = (umCharacter, umLine);

  TColorConfig = class;    // forward declaration

  TSyntaxHighlighter = class(TComponent)
  private
     FColorConfig    : TColorConfig;
     FEditor         : TRichEdit;
     FFontFamily     : String;
     FUpdateMode     : TUpdateMode;
     FInternalModified, FChanging: Boolean;
     FLastLine       : Integer;
     FxQuery         : TCustomXQuery;        // used only to hilite table names and fields

     FSaveOnChange   : TNotifyEvent;
     FSaveOnSelectionChange: TNotifyEvent;
     FSaveOnExit     : TNotifyEvent;
     FOnPosChange    : TPosChangeEvent;
     procedure SetEditor(Value: TRichEdit);
     procedure SetXQuery(Value: TCustomXQuery);
     procedure MyOnChange(Sender: TObject);
     procedure MyOnSelectionChange(Sender: TObject);
     procedure MyOnExit(Sender: TObject);
  protected
     procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     procedure Execute;
{$ifndef BCB}
     procedure EditColorSet;
{$endif}
     procedure FontChanged;

     property ColorConfig: TColorConfig read FColorConfig;
  published
     property UpdateMode: TUpdateMode read FUpdateMode write FUpdateMode;
     property Editor: TRichEdit read FEditor write SetEditor;
     property XQuery: TCustomXQuery read FxQuery write SetxQuery;

     property OnPosChange: TPosChangeEvent read FOnPosChange write FOnPosChange;
  end;

  TElementGroup = (idWhiteSpace,
                   idComment,
                   idReservedWord,
                   idIdentifier,
                   idTable,
                   idField,
                   idString,
                   idNumber,
                   idComma,
                   idParenthesis,
                   idOperator,
                   idSemicolon,
                   idPeriod);

  PColorElement = ^TColorElement;
  TColorElement = record
     Elements: TList;
     Group: TElementGroup;
     ForeColor: TColor;
     BackColor: TColor;
     FontStyle: TFontStyles;
  end;

  TColorConfig = class(TObject)
  private
     FColorSettings: TList;
     FColorTable: TList;
     function Get(Index: Integer): TColorElement;
     procedure Put(Index: Integer; const Value: TColorElement);
  public
     constructor Create;
     destructor Destroy; override;
     procedure Clear;
     procedure Assign(Value: TColorConfig);
     procedure LoadFromFile(const FileName: String);
     procedure SaveToFile(const FileName: String);
     procedure SetColorElement(Group: TElementGroup; ForeColor,BackColor:TColor;
        FontStyle: TFontStyles);
     function Count: Integer;
{$ifndef BCB}
     procedure EditColorSettings;
{$endif}
     function FindConfig(Element: Integer; var ForeColor, BackColor: TColor;
        var FontStyle: TFontStyles): Boolean;
     procedure CreateColorTable;
     function IndexOfColor(Color: TColor): Integer;
     function IndexOfGroup(Group: TElementGroup): Integer;

     property Elements[Index: Integer]: TColorElement read Get write Put;
     property ColorTable: TList read FColorTable;
     property ColorSettings: TList read FColorSettings;
  end;

implementation

uses
  DB, xqmiscel, xqconsts
{$ifndef BCB}
  , ColorSet
{$endif}
  ;

const
  _TABLE = 1000;
  _FIELD = 1010;

type
  TCustomXQueryClass = class(TCustomXQuery);

constructor TSyntaxHighlighter.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FColorConfig:= TColorConfig.Create;
   FColorConfig.LoadFromFile(ExtractFilePath(Application.Exename)+'colortbl.cfg');
   FColorConfig.CreateColorTable;

   FLastLine:= -1;
end;

destructor TSyntaxHighlighter.Destroy;
begin
   FColorConfig.free;
   inherited Destroy;
end;

procedure TSyntaxHighlighter.SetEditor(Value: TRichEdit);
begin
   if Assigned(FEditor) then
   begin
      //Restore previous
      FEditor.OnChange:= FSaveOnChange;
      FEditor.OnSelectionChange:= FSaveOnSelectionChange;
      FEditor.OnExit:= FSaveOnExit;
   end;
   FEditor:= Value;
   if Assigned(FEditor) then
   begin
      FSaveOnChange:= FEditor.OnChange;
      FSaveOnSelectionChange:= FEditor.OnSelectionChange;
      FSaveOnExit:= FEditor.OnExit;

      FEditor.OnChange:= MyOnChange;
      FEditor.OnSelectionChange:= MyOnSelectionChange;
      FEditor.OnExit:= MyOnExit;
      FontChanged;    // calculate new font family

      Value.FreeNotification(Self);
      if not (csDesigning in ComponentState) then
         Execute;
   end;
end;

procedure TSyntaxHighlighter.SetXQuery(Value: TCustomXQuery);
begin
   FXQuery:= Value;
   if Assigned(Value) then
      Value.FreeNotification(Self);
end;

procedure TSyntaxHighlighter.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited Notification(AComponent, Operation);
   if (Operation = opRemove) then
   begin
      if AComponent=FEditor then
         SetEditor(nil)
      else if AComponent=FXQuery then
         FXQuery:= nil;
   end;
end;

{$ifndef BCB}
procedure TSyntaxHighlighter.EditColorSet;
var
   FileName: String;
begin
   FileName:= ExtractFilePath(Application.Exename)+'colortbl.cfg';
   with TfrmColorSettings.Create(Application) do
   begin
      try
         if Enter(FColorConfig)=mrOk then
         begin
            FColorConfig.SaveToFile(FileName);
            FColorConfig.CreateColorTable;
            Self.Execute;
         end;
      finally
         Free;
      end;
   end;
end;
{$endif}

procedure TSyntaxHighlighter.Execute;
var
  inputStream : TMemoryStream;
  outputStream: TMemoryStream;
  errorStream : TMemoryStream;
  s: String;
  lexer: TXQLexer;
  yychar: Integer; (* current lookahead character *)
  forecolor,backcolor:TColor;
  fontstyle:TFontStyles;
  I,token: Integer;
  atext, Reslt, RtfHeader : String;
  DataSet: TDataSet;
  Field: TField;

  procedure strToRichEdit(const S: String);
  var
    aMem : TMemoryStream;
    SelStart: Integer;
  begin
    aMem:=TMemoryStream.Create;
    FChanging:= True;
    SelStart := 0; //Basri
    try
      aMem.Write(Pointer(S)^, Length(S));
      aMem.Position:=0;
      if FEditor.Focused then SelStart:= FEditor.SelStart;
      LockWindowUpdate(FEditor.Handle);
      FEditor.Lines.LoadFromStream(aMem);
      if FEditor.Focused then FEditor.SelStart:= SelStart;
      LockWindowUpdate(0);
    finally
      aMem.Free;
      FChanging:= False;
    end;
  end;

  { converts a Delphi TColor into a RTF-color table string }
  function ColorToRtf(aColor: TColor): String;
  begin
    aColor:=ColorToRGB(aColor);
    Result:='\red'+IntToStr(GetRValue(aColor))+
            '\green'+IntToStr(GetGValue(aColor))+
            '\blue'+IntToStr(GetBValue(aColor))+';';
  end;

begin
  {$IFDEF XQDEMO}
  if not IsDelphiRunning then
  begin
     ShowAbout;
     raise Exception.Create( SDelphiIsNotRunning );
  end;
  {$ENDIF}
  if not Assigned(FEditor) or (csDestroying in ComponentState) then Exit;
  s:= FEditor.Text + ' ';
  inputStream := TMemoryStream.create;
  inputStream.WriteBuffer(Pointer(s)^, Length(s));
  inputStream.Seek( 0, 0	);
  outputStream := TMemoryStream.create;
  errorStream  := TMemoryStream.create;
  lexer := TXQLexer.Create;
  lexer.yyinput     := inputStream;
  lexer.yyoutput    := outputStream;
  lexer.yyerrorfile := errorStream;
  if Assigned(FXQuery) then
     (lexer as TXQlexer).DateFormat := FXQuery.DateFormat
  else
     (lexer as TXQlexer).DateFormat := SDefaultDateFormat;

   RtfHeader :=
     '{\rtf1\ansi\deff0\deftab720'+
     '{\fonttbl'+
     //format('{\f0\\fcharset0\fprq2\f%s %s;}}',[FFontFamily,FEditor.Font.Name])+
     '{\f0\fswiss MS Sans Serif;}'+
     format('{\f1\f%s %s;}',[FFontFamily,FEditor.Font.Name])+
     //format('{\f0\\fcharset0\fprq2\fcharset186 %s;}}',[FEditor.Font.Name])+
     '}{\colortbl;';
     // the default color
     RtfHeader:= RtfHeader + ColorToRtf(FEditor.Font.Color);    // foreground
     RtfHeader:= RtfHeader + ColorToRtf(FEditor.Color);         // background
     // Create a table of colors specified
     for I:= 0 to FColorConfig.ColorTable.Count - 1 do
        RtfHeader:= RtfHeader + ColorToRtf(TColor(FColorConfig.ColorTable[I]));
     RtfHeader:= RtfHeader + '}'+
     format('\deflang1031\pard\plain\f1\fs%d',[FEditor.Font.Size * 2]);

  Reslt:= '';     // resulting rtf string

  try
     repeat
        try
            Lexer.IgnoreBadDates:= True;
            yychar := Lexer.yylex;
        except
            { ignore syntax errors }
        end;
        if yychar<0 then yychar := 0;
        if yychar= 0 then break;             // normal termination
        atext:= Lexer.yytext;
        if yychar= _ILLEGAL then       // illegal token on input
        begin
           Reslt:= Reslt + '{\cb2\cf1\b0\i0\ul0';
           ReplaceString(atext, #10, '\line ');
           ReplaceString(atext, #13, '');
           Reslt:= Reslt + #32 + atext + '}';
        end else
        begin
           // it is a table name or database field coming from FXQuery property?
           if Assigned(FXQuery) and (yychar = _IDENTIFIER) and
              (not Lexer.IsKeyword(atext,token)) then
           begin
              for I:= 0 to TCustomXQueryClass(FxQuery).DataSets.Count - 1 do
              begin
                 if AnsiCompareText(TCustomXQueryClass(FxQuery).DataSets[I].Alias,atext)=0 then
                 begin
                    yychar:= _TABLE;
                    Break;
                 end;
                 DataSet:= TCustomXQueryClass(FxQuery).DataSets[I].DataSet;
                 if Assigned(DataSet) and DataSet.Active then
                 begin
                    Field:= DataSet.FindField(Lexer.yytext);
                    if Assigned(Field) then
                    begin
                       yychar:= _FIELD;
                       Break;
                    end;
                 end;
              end;
           end;
           if FColorConfig.findconfig(yychar,forecolor,backcolor,fontstyle) then
           else
           begin
              forecolor:=clBlack;
              backcolor:=clWhite;
              fontstyle:=[];
           end;
           Reslt:= Reslt + Format('{\cb%d\cf%d',[FColorConfig.IndexOfColor(backcolor)+3,
              FColorConfig.IndexOfColor(forecolor)+3]);
           // the font style
           if fsBold in fontstyle then Reslt:= Reslt + '\b' else Reslt:= Reslt + '\b0';
           if fsItalic in fontstyle then Reslt:= Reslt + '\i' else Reslt:= Reslt + '\i0';
           if fsUnderline in fontstyle then Reslt:= Reslt + '\ul' else Reslt:= Reslt + '\ul0';
           case yychar of
              _NEWLINE:
              Reslt:= Reslt + #32 + '\line}';
              _TAB:
              Reslt:= Reslt + #32 + '\tab}';
              _BLANK:
              Reslt:= Reslt + #32 + ' }';
           else
              begin

              ReplaceString(atext, #10, '\line ');
              ReplaceString(atext, #13, '');
              Reslt:= Reslt + #32 + atext + '}';
              end;
           end;
        end;
     until false;
     Reslt := RtfHeader + Reslt + '}';//+ '\cb2\cf1\b0\i0\ul0}';
     strToRichEdit(Reslt);
     {FEditor.DefAttributes.Assign(FEditor.Font);}
     //FEditor.SelAttributes.Assign(FEditor.Font);
     FInternalModified:= False;
  finally
     lexer.free;
     inputStream.free;
     outputStream.free;
     errorStream.free;
  end;
end;

procedure TSyntaxHighlighter.MyOnChange(Sender: TObject);
begin
   if FChanging or (csDesigning in ComponentState) or not Assigned(FEditor) then Exit;
   FInternalModified:= True;
   if not FEditor.Focused or (FUpdateMode = umCharacter) then
      Execute;
   if Assigned(FSaveOnChange) then
      FSaveOnChange(FEditor);
end;

procedure TSyntaxHighlighter.MyOnSelectionChange(Sender: TObject);
var
  CharPos: TPoint;
begin
  if FChanging or not Assigned(FEditor) then Exit;
  CharPos.Y := SendMessage(FEditor.Handle, EM_EXLINEFROMCHAR, 0,
    FEditor.SelStart);
  CharPos.X := (FEditor.SelStart -
    SendMessage(FEditor.Handle, EM_LINEINDEX, CharPos.Y, 0));
  Inc(CharPos.Y);
  Inc(CharPos.X);
  if (FUpdateMode = umLine) and FInternalModified and (CharPos.Y <> FLastLine) then
  begin
     FLastLine := CharPos.Y;
     FChanging:= True;
     Execute;
     FChanging:= False;
  end;
  if Assigned(FOnPosChange) then
     FOnPosChange(Self, CharPos.Y, CharPos.X);
  if Assigned(FSaveOnSelectionChange) then
     FSaveOnSelectionChange(FEditor);
end;

procedure TSyntaxHighlighter.FontChanged;
var
  ControlCanvas: TControlCanvas;
  FontInfo: TTextMetric;            // holds the font metric information
begin
   if FChanging or not Assigned(FEditor) then Exit;
   // calculate font family
   ControlCanvas:= TControlCanvas.Create;
   try
      ControlCanvas.Control:= FEditor;
      ControlCanvas.Font.Assign(FEditor.Font);
      GetTextMetrics(ControlCanvas.Handle, FontInfo);
   finally
      ControlCanvas.Free;
   end;

   {Get the font family}
   FFontFamily:= 'swiss';
   case (FontInfo.tmPitchAndFamily and $F0) of
     FF_DECORATIVE: FFontFamily:= 'decor';
     FF_DONTCARE:   FFontFamily:= 'swiss'; // actually 'nil'
     FF_MODERN:     FFontFamily:= 'modern';
     FF_ROMAN:      FFontFamily:= 'roman';

     FF_SCRIPT:     FFontFamily:= 'script';
     FF_SWISS:      FFontFamily:= 'swiss';
   end;

   Execute;

end;

procedure TSyntaxHighlighter.MyOnExit(Sender: TObject);
begin
   if not Assigned(FEditor) then Exit;
   if (FUpdateMode = umLine) and FInternalModified then
   begin
      Execute;
   end;
   if Assigned(FSaveOnExit) then
      FSaveOnExit(FEditor);
end;

// TColorConfig - class implementation
constructor TColorConfig.Create;
var
   ColorElement: PColorElement;
   i: Integer;
begin
   inherited Create;
   FColorSettings:= TList.Create;
   // default values

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_IDENTIFIER));
      Group:= idIdentifier;
      ForeColor:=clBlue;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_TABLE));
      Group:= idTable;
      ForeColor:=clFuchsia;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_FIELD));
      Group:= idField;
      ForeColor:=clTeal;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_SINTEGER));
      Elements.Add(Pointer(_UINTEGER));
      Elements.Add(Pointer(_NUMERIC));
      Group:= idNumber;
      ForeColor:=clFuchsia;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_STRING));;
      Group:= idString;
      ForeColor:=clPurple;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_COMA));
      Group:= idComma;
      ForeColor:=clRed;
      BackColor:=clWhite;
      FontStyle:= [fsBold];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add( Pointer(_LPAREN));
      Elements.Add( Pointer(_RPAREN));
      Group:= idParenthesis;
      ForeColor:=clRed;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      with Elements do
      begin
         Add(Pointer(_GT));
         Add(Pointer(_LT));
         Add(Pointer(_EQ));
         Add(Pointer(_MULT));
         Add(Pointer(_PLUS));
         Add(Pointer(_SUB));
         Add(Pointer(_NEG));
         Add(Pointer(_DIV));
         Add(Pointer(_NEQ));
         Add(Pointer(_GE));
         Add(Pointer(_LE));
      end;
      Group:= idOperator;
      ForeColor:=clRed;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_PERIOD));
      Group:= idPeriod;
      ForeColor:=clRed;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);          
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_SEMICOLON));
      Elements.Add(Pointer(_COLON));
      Group:= idSemicolon;
      ForeColor:=clFuchsia;
      BackColor:=clWhite;
      FontStyle:= [fsBold];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_COMMENT));
      Group:= idComment;
      ForeColor:=clGray;
      BackColor:=clWhite;
      FontStyle:= [fsItalic];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      Elements.Add(Pointer(_BLANK));
      Elements.Add(Pointer(_TAB));
      Elements.Add(Pointer(_NEWLINE));
      Group:= idWhiteSpace;
      ForeColor:=clWhite;
      BackColor:=clWhite;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   New(ColorElement);
   with ColorElement^ do
   begin
      Elements:=TList.Create;
      for i:= Low(rwords) to High(rwords) do
         Elements.Add(Pointer(rwords[i].token));
      Group:= idReservedWord;
      ForeColor:=clGreen;
      BackColor:=clBlack;
      FontStyle:= [];
   end;
   FColorSettings.Add(ColorElement);

   FColorTable:= TList.Create;
   
end;

destructor TColorConfig.Destroy;
begin
   Clear;
   FColorSettings.Free;
   FColorTable.Free;
   inherited Destroy;
end;

// color table used in creating an Rtf file
procedure TColorConfig.CreateColorTable;
var
   I, Index: Integer;
begin
   FColorTable.Clear;
   for I:= 0 to FColorSettings.Count - 1 do
   begin
      with PColorElement(FColorSettings[I])^ do
      begin
         Index:= FColorTable.IndexOf(Pointer(ForeColor));
         if Index = -1 then
            FColorTable.Add(Pointer(ForeColor));
         Index:= FColorTable.IndexOf(Pointer(BackColor));
         if Index = -1 then
            FColorTable.Add(Pointer(BackColor));
      end;
   end;
end;

function TColorConfig.IndexOfColor(Color: TColor): Integer;
begin
   Result:= FColorTable.IndexOf(Pointer(Color));
end;

procedure TColorConfig.Clear;
var
   I: Integer;
   ColorElement: PColorElement;
begin
   for I:= 0 to FColorSettings.Count - 1 do
   begin
      ColorElement:= PColorElement(FColorSettings[I]);
      ColorElement^.Elements.Free;
      Dispose(ColorElement);
   end;
   FColorSettings.Clear;
end;

procedure TColorConfig.SaveToFile(const FileName: String);
var
   Stream: TStream;
   i,j,n,ne,e: Integer;
   ColorElement: PColorElement;
begin
   Stream:= TFileStream.Create(FileName, fmCreate);
   try
      n:= FColorSettings.Count;
      Stream.Write(n,sizeof(n));
      for I:= 0 to n - 1 do
      begin
         ColorElement:= PColorElement(FColorSettings[I]);
         Stream.Write(ColorElement^, SizeOf(TColorElement));
         ne:= ColorElement^.Elements.Count;
         Stream.Write(ne,sizeof(ne));
         for j:= 0 to ne - 1 do
         begin
            e:= LongInt(ColorElement^.Elements[j]);
            Stream.Write(e,SizeOf(e));
         end;
      end;
   finally
      Stream.Free;
   end;
end;

procedure TColorConfig.LoadFromFile(const FileName: String);
var
   Stream: TStream;
   i,j,n,ne,e: Integer;
   ColorElement: PColorElement;
begin
   if not FileExists(FileName) then Exit;
   Clear;
   Stream:= TFileStream.Create(FileName, fmOpenRead or fmShareDenyNone);
   try
      Stream.Read(n,sizeof(n));
      for I:= 0 to n - 1 do
      begin
         New(ColorElement);
         Stream.Read(ColorElement^, SizeOf(TColorElement));
         ColorElement^.Elements:= TList.Create;
         Stream.Read(ne,sizeof(ne));
         for j:= 0 to ne - 1 do
         begin
            Stream.Read(e,SizeOf(e));
            ColorElement^.Elements.Add(Pointer(e));
         end;
         FColorSettings.Add(ColorElement);
      end;
   finally
      Stream.Free;
   end;
end;

procedure TColorConfig.SetColorElement(Group: TElementGroup; ForeColor,BackColor:TColor;
   FontStyle: TFontStyles);
var
   i: Integer;
   ColorElement: PColorElement;
begin
   for i:= 0 to FColorSettings.Count - 1 do
   begin
      ColorElement:= PColorElement(FColorSettings[i]);
      if ColorElement^.Group = Group then
      begin
         ColorElement^.ForeColor:= ForeColor;
         ColorElement^.BackColor:= BackColor;
         ColorElement^.FontStyle:= FontStyle;
         Break;
      end;
   end;
end;

function TColorConfig.FindConfig(Element: Integer; var ForeColor, BackColor: TColor;
   var FontStyle: TFontStyles): Boolean;
var
   i,j: Integer;
   ColorElement: PColorElement;
begin
   Result:= False;
   for i:= 0 to FColorSettings.Count - 1 do
   begin
      ColorElement:= PColorElement(FColorSettings[i]);
      for j:= 0 to ColorElement^.Elements.Count - 1 do
         if Longint(ColorElement^.Elements[j]) = Element then
         begin
            ForeColor:= ColorElement^.ForeColor;
            BackColor:= ColorElement^.BackColor;
            FontStyle:= ColorElement^.FontStyle;
            Result:= True;
            Exit;
         end;
   end;
end;

{$ifndef BCB}
procedure TColorConfig.EditColorSettings;
begin
   with TfrmColorSettings.Create(Application) do
   begin
      try
         Enter(Self);
      finally
         Free;
      end;
   end;
end;
{$endif}

function TColorConfig.Get(Index: Integer): TColorElement;
begin
   if (Index < 0) or (Index > FColorSettings.Count - 1) then Exit;
   Result:= PColorElement(FColorSettings[Index])^;
end;

procedure TColorConfig.Put(Index: Integer; const Value: TColorElement);
begin
   if (Index < 0) or (Index > FColorSettings.Count - 1) then Exit;
   PColorElement(FColorSettings[Index])^:= Value;
end;

function TColorConfig.Count: Integer;
begin
   Result:= FColorSettings.Count;
end;

procedure TColorConfig.Assign(Value: TColorConfig);
var
   ColorElement: PColorElement;
   i,j: Integer;
   TmpList: TList;
begin
   Clear;
   for i:= 0 to Value.FColorSettings.Count - 1 do
   begin
      New(ColorElement);
      ColorElement^:= PColorElement(Value.FColorSettings[I])^;
      TmpList:= ColorElement^.Elements;
      ColorElement^.Elements:= TList.Create;
      for j:= 0 to TmpList.Count - 1 do
         ColorElement^.Elements.Add(TmpList[j]);
      FColorSettings.Add(ColorElement);
   end;
end;

function TColorConfig.IndexOfGroup(Group: TElementGroup): Integer;
var
   I: Integer;
begin
   Result:= -1;
   for I:= 0 to FColorSettings.Count - 1 do
      if PColorElement(FColorSettings[I])^.Group = Group then
      begin
         Result:= I; Exit;
      end;
end;

end.
