/**********************************************************************
 *<
	FILE: NOISE.CPP

	DESCRIPTION: NOISE 3D Texture map.

	CREATED BY: Dan Silva

	HISTORY:

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/

#include "mtlhdr.h"
#include "mtlres.h"
#include "stdmat.h"

extern HINSTANCE hInstance;

#define NSUBTEX 2
#define NCOLS 2

static Class_ID noiseClassID(NOISE_CLASS_ID,0);

#define NPARAMS 7
#define NOISE_VERSION 3

#define PB_SIZE	    	0
#define PB_COL1	    	1
#define PB_COL2	    	2
#define PB_PHASE    	3
#define PB_LEVELS   	4
#define PB_LOWTHRESH	5
#define PB_HITHRESH		6

#define NOISE_REGULAR	0
#define NOISE_FRACTAL	1
#define NOISE_TURB		2

class Noise;

class NoiseDlg: public ParamDlg {
	public:
		HWND hwmedit;	 	// window handle of the materials editor dialog
		IMtlParams *ip;
		Noise *theTex;	// current Noise being edited.
		HWND hPanel; 		// Rollup pane
		ISpinnerControl *sizeSpin, *phaseSpin, *levSpin;
		ISpinnerControl *iLow, *iHigh;
		IColorSwatch *cs[2];
		TimeValue curTime; 
		ParamDlg *xyzGenDlg;
		ParamDlg *texoutDlg;
		BOOL valid;
		int isActive;
		BOOL creating;
		//-----------------------------
		NoiseDlg(HWND hwMtlEdit, IMtlParams *imp, Noise *m); 
		~NoiseDlg();
		BOOL PanelProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );
		void UpdateSubTexNames();
		void LoadDialog(BOOL draw);  // stuff params into dialog
		void ReloadDialog();
		void UpdateMtlDisplay() { ip->MtlChanged(); }
		void ActivateDlg(BOOL onOff);
		void Invalidate();

		// methods inherited from ParamDlg:
		Class_ID ClassID() {return noiseClassID;  }
		void SetThing(ReferenceTarget *m);
		ReferenceTarget* GetThing() { return (ReferenceTarget *)theTex; }
		void DeleteThis() { delete this;  }	
		void SetTime(TimeValue t);
	};



//--------------------------------------------------------------
// Noise: A 3D texture map
//--------------------------------------------------------------

#define XYZGEN_REF	0
#define PBLOCK_REF	1
#define MAP1_REF	2
#define MAP2_REF	3
#define TEXOUT_REF	4

#define NUM_REFS	5

class Noise: public Tex3D { 
	friend class NoisePostLoad;
	friend class NoiseDlg;
	Color col[NCOLS];
	float size;
	XYZGen *xyzGen;		   // ref #0
	IParamBlock *pblock;   // ref #1	
	Texmap* subTex[NSUBTEX];  // More refs (2 & 3)
	TextureOutput *texout; // ref #4
	Interval ivalid;
	int noiseType;
	float phase;
	float levels;
	float low, high, smooth, sd, hminusl;
	int vers;
	int rollScroll;
	NoiseDlg *paramDlg;
	float Turb(Point3 p);
	float NoiseFunc(float x, float y, float z);
	public:
		Noise();
		ParamDlg* CreateParamDlg(HWND hwMtlEdit, IMtlParams *imp);
		void Update(TimeValue t, Interval& valid);
		void Reset();
		Interval Validity(TimeValue t) { Interval v; Update(t,v); return ivalid; }
		void ReadSXPData(TCHAR *name, void *sxpdata);

		void SetOutputLevel(TimeValue t, float v) {texout->SetOutputLevel(t,v); }
		void SetColor(int i, Color c, TimeValue t);
		void SetSize(float f, TimeValue t);
		void SetPhase(float f, TimeValue t);
		void SetLevels(float f, TimeValue t);
		void NotifyChanged();
		void SwapInputs(); 

	
		// Evaluate the color of map for the context.
		RGBA EvalColor(ShadeContext& sc);

		// optimized evaluation for monochrome use
		float EvalMono(ShadeContext& sc);

		// For Bump mapping, need a perturbation to apply to a normal.
		// Leave it up to the Texmap to determine how to do this.
		Point3 EvalNormalPerturb(ShadeContext& sc);

		float NoiseFunc(Point3 p);

		ULONG LocalRequirements(int subMtlNum) { return xyzGen->Requirements(subMtlNum); }

		// Methods to access texture maps of material
		int NumSubTexmaps() { return NSUBTEX; }
		Texmap* GetSubTexmap(int i) { return subTex[i]; }
		void SetSubTexmap(int i, Texmap *m);
		TSTR GetSubTexmapSlotName(int i);

		Class_ID ClassID() {	return noiseClassID; }
		SClass_ID SuperClassID() { return TEXMAP_CLASS_ID; }
		void GetClassName(TSTR& s) { s= GetString(IDS_RB_NOISE); }  
		void DeleteThis() { delete this; }	

		ULONG UVSpacesNeeded() { return 0; } 

		int NumSubs() { return NUM_REFS; }  
		Animatable* SubAnim(int i);
		TSTR SubAnimName(int i);
		int SubNumToRefNum(int subNum) { return subNum; }

		// From ref
 		int NumRefs() { return NUM_REFS; }
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);

		RefTargetHandle Clone(RemapDir &remap = NoRemap());
		RefResult NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, 
		   PartID& partID, RefMessage message );

		// IO
		IOResult Save(ISave *isave);
		IOResult Load(ILoad *iload);

		int FixLevel0(); 
	};

class NoiseClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 1; }
	void *			Create(BOOL loading) { 	return new Noise; }
	const TCHAR *	ClassName() { return GetString(IDS_RB_NOISE); }
	SClass_ID		SuperClassID() { return TEXMAP_CLASS_ID; }
	Class_ID 		ClassID() { return noiseClassID; }
	const TCHAR* 	Category() { return TEXMAP_CAT_3D;  }
	};

static NoiseClassDesc noiseCD;

ClassDesc* GetNoiseDesc() { return &noiseCD;  }

static BOOL CALLBACK  PanelDlgProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	NoiseDlg *theDlg;
	if (msg==WM_INITDIALOG) {
		theDlg = (NoiseDlg*)lParam;
		theDlg->hPanel = hwndDlg;
		SetWindowLong(hwndDlg, GWL_USERDATA,lParam);
		}
	else {
	    if ( (theDlg = (NoiseDlg *)GetWindowLong(hwndDlg, GWL_USERDATA) ) == NULL )
			return FALSE; 
		}
	theDlg->isActive = 1;
	int	res = theDlg->PanelProc(hwndDlg,msg,wParam,lParam);
	theDlg->isActive = 0;
	return res;
	}

NoiseDlg::NoiseDlg(HWND hwMtlEdit, IMtlParams *imp, Noise *m) { 
	hwmedit = hwMtlEdit;
	ip = imp;
	hPanel = NULL;
	theTex = m; 
	isActive = 0;
	xyzGenDlg = theTex->xyzGen->CreateParamDlg(hwmedit, imp);	
	valid = FALSE;
	creating = TRUE;
	hPanel = ip->AddRollupPage( 
		hInstance,
		MAKEINTRESOURCE(IDD_NOISE),
		PanelDlgProc, 
		GetString(IDS_DS_NOISEPARMS), 
		(LPARAM)this );
	creating = FALSE;
	texoutDlg = theTex->texout->CreateParamDlg(hwmedit, imp);
	curTime = imp->GetTime();
	}

void NoiseDlg::ReloadDialog() {
	Interval valid;
	theTex->Update(curTime, valid);
	LoadDialog(FALSE);
	}


void NoiseDlg::Invalidate()
	{
	valid = FALSE;
	if (hPanel)
		InvalidateRect(hPanel,NULL,0);
	}

void NoiseDlg::SetTime(TimeValue t) {
	Interval valid;
	if (t!=curTime) {
		xyzGenDlg->SetTime(t);
		texoutDlg->SetTime(t);
		curTime = t;
		theTex->Update(curTime, valid);
		LoadDialog(FALSE);
		InvalidateRect(hPanel,NULL,0);
		}
	}

NoiseDlg::~NoiseDlg() {
	theTex->paramDlg = NULL;
	ReleaseISpinner(sizeSpin);
	ReleaseISpinner(phaseSpin);
	ReleaseISpinner(levSpin);
	ReleaseIColorSwatch(cs[0]);
	ReleaseIColorSwatch(cs[1]);
	ReleaseISpinner(iLow);
	ReleaseISpinner(iHigh);	
	SetWindowLong(hPanel, GWL_USERDATA, NULL);
	xyzGenDlg->DeleteThis();
	texoutDlg->DeleteThis();
	hPanel =  NULL;
	}

static int colID[2] = { IDC_NOISE_COL1, IDC_NOISE_COL2 };

BOOL NoiseDlg::PanelProc(HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam ) {
	int id = LOWORD(wParam);
	int code = HIWORD(wParam);
    switch (msg)    {
		case WM_INITDIALOG:
			{
			sizeSpin = SetupFloatSpinner(hwndDlg, IDC_NOISESIZE_SPIN, IDC_NOISESIZE_EDIT,0.001f, 100.0f, 1.0f, .1f);
			phaseSpin = SetupFloatSpinner(hwndDlg, IDC_NOISEPHASE_SPIN, IDC_NOISEPHASE_EDIT,-1000.0f,1000.0f, 0.0f, 0.1f);
			levSpin = SetupFloatSpinner(hwndDlg, IDC_NOISELEV_SPIN, IDC_NOISELEV_EDIT,1.0f,10.0f, 3.0f, 0.1f);
			for (int i=0; i<NCOLS; i++) 
   				cs[i] = GetIColorSwatch(GetDlgItem(hwndDlg, colID[i]),
   					theTex->col[i],theTex->GetSubTexmapSlotName(i).data());
			iLow    = SetupFloatSpinner(hwndDlg, IDC_NOISE_LOWTHRESHSPIN, IDC_NOISE_LOWTHRESH,0.0f,1.0f,0.0f,.005f);
			iHigh   = SetupFloatSpinner(hwndDlg, IDC_NOISE_HIGHTHRESHSPIN, IDC_NOISE_HIGHTHRESH,0.0f,1.0f,0.0f,.005f);			
			return TRUE;
			}
			break;
		case WM_COMMAND:  
			if (creating)
				return FALSE; // A HACK to fix a bug -- when the rollup
					// was created, windows sent an IDC_NOISE_REGULAR command for
					// no good reason.
		    switch (id) {
				case IDC_NOISE_TEX1: PostMessage(hwmedit,WM_TEXMAP_BUTTON,0 ,(LPARAM)theTex);	break;
				case IDC_NOISE_TEX2: PostMessage(hwmedit,WM_TEXMAP_BUTTON,1 ,(LPARAM)theTex);	break;
				
				case IDC_NOISE_REGULAR:
					theTex->noiseType =  NOISE_REGULAR;
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					levSpin->Disable();
					break;
				case IDC_NOISE_FRACT:
					theTex->noiseType =  NOISE_FRACTAL;
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					levSpin->Enable();
					break;
				case IDC_NOISE_TURB:
					theTex->noiseType =  NOISE_TURB;
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					levSpin->Enable();
					break;
					
				case IDC_NOISE_SWAP:
					theTex->SwapInputs(); 
					cs[0]->SetColor(theTex->col[0]);
					cs[1]->SetColor(theTex->col[1]);
					UpdateSubTexNames();
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					break;
				}
			break;
		case CC_COLOR_BUTTONDOWN:
			theHold.Begin();
			break;
		case CC_COLOR_BUTTONUP:
			if (HIWORD(wParam)) theHold.Accept(IDS_DS_PARAMCHG);
			else theHold.Cancel();
			break;
		case CC_COLOR_CHANGE:
			{
			int id = LOWORD(wParam);
			int buttonUp = HIWORD(wParam); 
			int n = (id==IDC_NOISE_COL1)?0:1;
			if (buttonUp) theHold.Begin();
			theTex->SetColor(n,cs[n]->GetColor(),curTime);
			if (buttonUp) {
				theHold.Accept(GetString(IDS_DS_PARAMCHG));
				theTex->NotifyChanged();
				}
			}
			break;
		case WM_LBUTTONDOWN:
		case WM_LBUTTONUP:
		case WM_MOUSEMOVE:
			ip->RollupMouseMessage(hwndDlg,msg,wParam,lParam);
			return FALSE;
		case WM_PAINT: 
			if (!valid) {
				valid = TRUE;
				ReloadDialog();
				}
			return FALSE;
		case WM_CLOSE: 	break;       
		case WM_DESTROY:		break;
		case CC_SPINNER_CHANGE: 
			if (!theHold.Holding()) theHold.Begin();
			switch (id) {
				case IDC_NOISESIZE_SPIN: 
					theTex->SetSize(sizeSpin->GetFVal(),curTime); 	
					break;
				case IDC_NOISEPHASE_SPIN: 
					theTex->SetPhase(phaseSpin->GetFVal(),curTime); 	
					break;
				case IDC_NOISELEV_SPIN: 
					theTex->SetLevels(levSpin->GetFVal(),curTime); 	
					break;
				
				case IDC_NOISE_LOWTHRESHSPIN:
					theTex->pblock->SetValue(PB_LOWTHRESH,ip->GetTime(),iLow->GetFVal());
					break;
				case IDC_NOISE_HIGHTHRESHSPIN:
					theTex->pblock->SetValue(PB_HITHRESH,ip->GetTime(),iHigh->GetFVal());
					break;				
				}
			break;
		case CC_SPINNER_BUTTONDOWN:
			theHold.Begin();
			break;
		case WM_CUSTEDIT_ENTER:
		case CC_SPINNER_BUTTONUP: 
			if (HIWORD(wParam) || msg==WM_CUSTEDIT_ENTER) theHold.Accept(GetString(IDS_DS_PARAMCHG));
			else theHold.Cancel();
			theTex->NotifyChanged();
		    UpdateMtlDisplay();
			break;

    	}
	return FALSE;
	}


static int subTexId[NSUBTEX] = { IDC_NOISE_TEX1, IDC_NOISE_TEX2 };
static int typeIDs[] = {IDC_NOISE_REGULAR,IDC_NOISE_FRACT,IDC_NOISE_TURB};

void NoiseDlg::UpdateSubTexNames() {
	for (int i=0; i<NSUBTEX; i++) {
		Texmap *m = theTex->subTex[i];
		TSTR nm;
		if (m) 	nm = m->GetFullName();
		else 	nm = GetString(IDS_DS_NONE);
		SetDlgItemText(hPanel, subTexId[i], nm.data());
		}
	}

void NoiseDlg::LoadDialog(BOOL draw) {
	if (theTex) {
		Interval valid;
		theTex->Update(curTime,valid);
		sizeSpin->SetValue(theTex->size,FALSE);
		phaseSpin->SetValue(theTex->phase,FALSE);
		levSpin->SetValue(theTex->levels,FALSE);
		cs[0]->SetColor(theTex->col[0]);
		cs[1]->SetColor(theTex->col[1]);		
		CheckDlgButton(hPanel,IDC_NOISE_REGULAR,theTex->noiseType==NOISE_REGULAR);
		CheckDlgButton(hPanel,IDC_NOISE_FRACT,theTex->noiseType==NOISE_FRACTAL);
		CheckDlgButton(hPanel,IDC_NOISE_TURB,theTex->noiseType==NOISE_TURB);
		iLow->SetValue(theTex->low,FALSE);
		iHigh->SetValue(theTex->high,FALSE);		
		UpdateSubTexNames();
		if (theTex->noiseType==NOISE_REGULAR) {
			levSpin->Disable();
		} else {
			levSpin->Enable();
			}
		}
	}

void NoiseDlg::SetThing(ReferenceTarget *m) {
	assert (m->ClassID()==noiseClassID);
	assert (m->SuperClassID()==TEXMAP_CLASS_ID);
	if (theTex) theTex->paramDlg = NULL;
	theTex = (Noise *)m;
	xyzGenDlg->SetThing(theTex->xyzGen);
	texoutDlg->SetThing(theTex->texout);
	if (theTex)
		theTex->paramDlg = this;
	LoadDialog(TRUE);
	}

void NoiseDlg::ActivateDlg(BOOL onOff) {
	for (int i=0; i<NCOLS; i++)
		cs[i]->Activate(onOff);
	}

//-----------------------------------------------------------------------------
//  Noise
//-----------------------------------------------------------------------------



// Version 1 params
static ParamBlockDescID pbdesc1[] = {
	{ TYPE_FLOAT, NULL, TRUE,0 }, 	// size
	{ TYPE_POINT3, NULL, TRUE,1 },  // col1
	{ TYPE_POINT3, NULL, TRUE,2 }   // col2
	};

static ParamBlockDescID pbdesc2[] = {
	{ TYPE_FLOAT, NULL, TRUE,0 }, 	// size
	{ TYPE_POINT3, NULL, TRUE,1 },  // col1
	{ TYPE_POINT3, NULL, TRUE,2 },  // col2
	{ TYPE_FLOAT, NULL, TRUE,3 }, 	// phase
	{ TYPE_FLOAT, NULL, TRUE,4 } 	// levels
	};

static ParamBlockDescID pbdesc[] = {
	{ TYPE_FLOAT, NULL, TRUE,0 }, 	// size
	{ TYPE_RGBA, NULL, TRUE,1 },  // col1
	{ TYPE_RGBA, NULL, TRUE,2 },  // col2
	{ TYPE_FLOAT, NULL, TRUE,3 }, 	// phase
	{ TYPE_FLOAT, NULL, TRUE,4 }, 	// levels
	{ TYPE_FLOAT, NULL, TRUE,5 },	// low thresh
	{ TYPE_FLOAT, NULL, TRUE,6 },	// high thresh	
	};

// Current version params

static ParamVersionDesc oldVersions[] = {
	ParamVersionDesc(pbdesc1,3,1),	// Version 1 params
	ParamVersionDesc(pbdesc2,4,2),	// Version 2 params
	};
#define NUM_OLDVERSIONS	2

static ParamVersionDesc curVersion(pbdesc, NPARAMS, NOISE_VERSION); 


void Noise::Reset() {
	if (xyzGen) xyzGen->Reset();
	else ReplaceReference( XYZGEN_REF, GetNewDefaultXYZGen());	
	if (texout) texout->Reset();
	else ReplaceReference( TEXOUT_REF, GetNewDefaultTextureOutput());		
	ReplaceReference( 1, CreateParameterBlock( pbdesc, NPARAMS, NOISE_VERSION) );	
	ivalid.SetEmpty();
	for (int i=0; i<NSUBTEX; i++) 
		DeleteReference(i+2);
	SetColor(0, Color(0.0f,0.0f,0.0f), TimeValue(0));
	SetColor(1, Color(1.0f,1.0f,1.0f), TimeValue(0));
	noiseType = NOISE_REGULAR;
	SetSize(25.0f, TimeValue(0));
	SetPhase(.0f,TimeValue(0));
	SetLevels(3.0f,TimeValue(0));
	pblock->SetValue(PB_HITHRESH,0,1.0f);
	}

void Noise::NotifyChanged() {
	NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
	}

Noise::Noise() {
	paramDlg = NULL;
	subTex[0] = subTex[1] = NULL;
	pblock = NULL;
	xyzGen = NULL;
	texout = NULL;
	noiseType = NOISE_REGULAR;
	Reset();
	vers = 0;
	rollScroll=0;
	}


RefTargetHandle Noise::Clone(RemapDir &remap) {
	Noise *mnew = new Noise();
	*((MtlBase*)mnew) = *((MtlBase*)this);  // copy superclass stuff
	mnew->ReplaceReference(XYZGEN_REF,remap.CloneRef(xyzGen));
	mnew->ReplaceReference(TEXOUT_REF,remap.CloneRef(texout));
	mnew->ReplaceReference(PBLOCK_REF,remap.CloneRef(pblock));
	mnew->col[0] = col[0];
	mnew->col[1] = col[1];
	mnew->noiseType = noiseType;
	mnew->size = size;
	mnew->ivalid.SetEmpty();	
	for (int i = 0; i<NSUBTEX; i++) {
		mnew->subTex[i] = NULL;
		if (subTex[i])
			mnew->ReplaceReference(i+2,remap.CloneRef(subTex[i]));
		}
	return (RefTargetHandle)mnew;
	}

ParamDlg* Noise::CreateParamDlg(HWND hwMtlEdit, IMtlParams *imp) {
	NoiseDlg *dm = new NoiseDlg(hwMtlEdit, imp, this);
	dm->LoadDialog(TRUE);	
	paramDlg = dm;
	return dm;	
	}

struct Col24 {ULONG r,g,b; };
#define NOISE_VERS 0xC79A0

struct NoiseState {
	ulong version;
	float size;
	float x1,y1,z1;
	float x2,y2,z2;
	Col24 col1,col2;
	long frame1,frame2;
	};

static Color ColrFromCol24(Col24 a) {
	Color c;
	c.r = (float)a.r/255.0f;
	c.g = (float)a.g/255.0f;
	c.b = (float)a.b/255.0f;
	return c;
	}

void Noise::ReadSXPData(TCHAR *name, void *sxpdata) {
	NoiseState *state = (NoiseState*)sxpdata;
	if (state->version==NOISE_VERS) {
		SetColor(0, ColrFromCol24(state->col1),0);
		SetColor(1, ColrFromCol24(state->col2),0);
		SetSize(state->size,0);
		}
	}

void Noise::Update(TimeValue t, Interval& valid) {		
	if (!ivalid.InInterval(t)) {
		ivalid.SetInfinite();
		xyzGen->Update(t,ivalid);
		texout->Update(t,ivalid);
		pblock->GetValue( PB_COL1, t, col[0], ivalid );
		col[0].ClampMinMax();
		pblock->GetValue( PB_COL2, t, col[1], ivalid );
		col[1].ClampMinMax();
		pblock->GetValue( PB_SIZE, t,   size, ivalid );
		pblock->GetValue( PB_PHASE, t,  phase, ivalid );
		pblock->GetValue( PB_LEVELS, t,  levels, ivalid );
		for (int i=0; i<NSUBTEX; i++) {
			if (subTex[i]) 
				subTex[i]->Update(t,ivalid);
			}		
		pblock->GetValue( PB_HITHRESH, t, high, ivalid );
		pblock->GetValue( PB_LOWTHRESH, t, low, ivalid );		
		}
	valid &= ivalid;
	}

void Noise::SwapInputs() {
	Color t = col[0]; col[0] = col[1]; col[1] = t;
	Texmap *x = subTex[0];  subTex[0] = subTex[1];  subTex[1] = x;
	pblock->SwapControllers(PB_COL1,PB_COL2);
	}

void Noise::SetColor(int i, Color c, TimeValue t) {
    col[i] = c;
	pblock->SetValue( i==0?PB_COL1:PB_COL2, t, c);
	}

void Noise::SetSize(float f, TimeValue t) { 
	size = f; 
	pblock->SetValue( PB_SIZE, t, f);
	}

void Noise::SetPhase(float f, TimeValue t) { 
	phase = f; 
	pblock->SetValue( PB_PHASE, t, f);
	}

void Noise::SetLevels(float f, TimeValue t) { 
	levels = f; 
	pblock->SetValue( PB_LEVELS, t, f);
	}

RefTargetHandle Noise::GetReference(int i) {
	switch(i) {
		case XYZGEN_REF: return xyzGen;
		case PBLOCK_REF: return pblock;
		case TEXOUT_REF: return texout;
		default:return subTex[i-2];
		}
	}

void Noise::SetReference(int i, RefTargetHandle rtarg) {
	switch(i) {
		case XYZGEN_REF: xyzGen = (XYZGen *)rtarg; break;
		case PBLOCK_REF: pblock = (IParamBlock *)rtarg; break;
		case TEXOUT_REF: texout = (TextureOutput *)rtarg; break;
		default: subTex[i-2] = (Texmap *)rtarg; break;
		}
	}

void Noise::SetSubTexmap(int i, Texmap *m) {
	ReplaceReference(i+2,m);
	if (paramDlg)
		paramDlg->UpdateSubTexNames();
	}

TSTR Noise::GetSubTexmapSlotName(int i) {
	switch(i) {
		case 0:  return TSTR(GetString(IDS_DS_COLOR1)); 
		case 1:  return TSTR(GetString(IDS_DS_COLOR2)); 
		default: return TSTR(_T(""));
		}
	}
	 
Animatable* Noise::SubAnim(int i) {
	switch (i) {
		case XYZGEN_REF: return xyzGen;
		case PBLOCK_REF: return pblock;
		case TEXOUT_REF: return texout;
		default: return subTex[i-2]; 
		}
	}

TSTR Noise::SubAnimName(int i) {
	switch (i) {
		case XYZGEN_REF: return TSTR(GetString(IDS_DS_COORDINATES));		
		case PBLOCK_REF: return TSTR(GetString(IDS_DS_PARAMETERS));		
		case TEXOUT_REF: return TSTR(GetString(IDS_DS_OUTPUT));
		default: return GetSubTexmapTVName(i-2);
		}
	}

static int nameID[] = {IDS_DS_NOISESIZE, IDS_DS_COLOR1, IDS_DS_COLOR2, IDS_DS_PHASE, IDS_DS_LEVELS,
						IDS_RB_LOWTHRESH, IDS_RB_HIGHTHRESH};

RefResult Noise::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
   PartID& partID, RefMessage message ) {
	switch (message) {
		case REFMSG_CHANGE:
			ivalid.SetEmpty();
			if (hTarget!=xyzGen&&hTarget!=texout) {
				if (paramDlg&&!paramDlg->isActive)
					paramDlg->Invalidate();
				}
			break;

		case REFMSG_GET_PARAM_DIM: {
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {
				case PB_SIZE: 
				case PB_PHASE: 
				case PB_LEVELS: 
					gpd->dim = defaultDim; break;
				case PB_COL1: 
				case PB_COL2: gpd->dim = stdColor255Dim; break;
				}
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			gpn->name = TSTR(GetString(nameID[gpn->index]));
			return REF_STOP; 
			}
		}
	return(REF_SUCCEED);
	}


#define MTL_HDR_CHUNK 	0x4000
#define DO_TURB_CHUNK 	0x1000
#define NOISETYPE_CHUNK	0x1010
#define NOISEVERS1_CHUNK 0x2001

IOResult Noise::Save(ISave *isave) { 
	IOResult res;
	ULONG nb;
	// Save common stuff
	isave->BeginChunk(MTL_HDR_CHUNK);
	res = MtlBase::Save(isave);
	if (res!=IO_OK) return res;
	isave->EndChunk();
	
	isave->BeginChunk(NOISETYPE_CHUNK);
	isave->Write(&noiseType,sizeof(noiseType),&nb);
	isave->EndChunk();
	isave->BeginChunk(NOISEVERS1_CHUNK);
	isave->EndChunk();
	return IO_OK;
	}	

int Noise::FixLevel0() {
	// old files had level==0: this is to fix them
	if (pblock) {
		float l;
		Interval ivalid;
		pblock->GetValue( PB_LEVELS, 0,  l, ivalid );
		if (l<1.0f) {
			pblock->SetValue( PB_LEVELS, 0,  1.0f );
			return 1;
			}
		}
	return 0;
	}

class NoisePostLoad : public PostLoadCallback {
	public:
		Noise *n;
		NoisePostLoad(Noise *ns) {n = ns;}
		void proc(ILoad *iload) {  
			if (n->FixLevel0())
				iload->SetObsolete();
			delete this;
			} 
	};

	
IOResult Noise::Load(ILoad *iload) { 
	ULONG nb;
	IOResult res;
	vers = 0;
	iload->RegisterPostLoadCallback(new ParamBlockPLCB(oldVersions, NUM_OLDVERSIONS, &curVersion, this, 1 /*ref # */ ));
	iload->RegisterPostLoadCallback(new NoisePostLoad(this));
	while (IO_OK==(res=iload->OpenChunk())) {
		switch(iload->CurChunkID())  {
			case MTL_HDR_CHUNK:
				res = MtlBase::Load(iload);
				break;
			case DO_TURB_CHUNK:
				noiseType = NOISE_TURB;
				break;
			case NOISETYPE_CHUNK:
				res = iload->Read(&noiseType,sizeof(noiseType),&nb);
				break;
			case NOISEVERS1_CHUNK:
				vers = 1;
				break;
			}
		iload->CloseChunk();
		if (res!=IO_OK) 
			return res;
		}
	return IO_OK;
	}

#define NOISE01(p) ((1.0f+noise4(p,phase))*.5f)

float Noise::Turb(Point3 p) {
	float sum = 0.0f;
	float l,f = 1.0f;
	for (l = levels; l>=1.0f; l-=1.0f) {
		sum += (float)fabs(noise4(p*f,phase))/f;
		f *= 2.0f;
		}
	if (l>0.0f)
		sum += l*(float)fabs(noise4(p*f,phase))/f;
	return sum;
	}

inline float Noise::NoiseFunc(Point3 p) {
	float res;
	switch (noiseType) {
		case NOISE_TURB:
			res = Turb(p);
			break;
	
		case NOISE_REGULAR:
			res = NOISE01(p);
			break;

		case NOISE_FRACTAL:
			if (levels==1.0f) {
				res = NOISE01(p);
			} else {
				float sum = 0.0f;
				float l,f = 1.0f;			
				for (l = levels; l>=1.0f; l-=1.0f) {				
					sum += noise4(p*f,phase)/f;
					f *= 2.0f;
					}
				if (l>0.0f)				
					sum += l*noise4(p*f,phase)/f;				
				res = 0.5f*(sum+1.0f);
				}
			break;
		}

	if (low<high) {
		res = threshold(res,low,high);
		}
	if (res<0.0f) res = 0.0f;
	if (res>1.0f) res = 1.0f;
	return res;
	}

RGBA Noise::EvalColor(ShadeContext& sc) {
	if (gbufID) sc.SetGBufferID(gbufID);
	Point3 p,dp;
	xyzGen->GetXYZ(sc,p,dp);
	p /= size;
    float d = NoiseFunc(p);
	//assert(d>=0.0f);
	//assert(d<=1.0f);
	RGBA c0 = subTex[0] ? subTex[0]->EvalColor(sc): col[0];
	RGBA c1 = subTex[1] ? subTex[1]->EvalColor(sc): col[1];
//	return (1.0f-d)*c0 + d*c1;
	return texout->Filter((1.0f-d)*c0 + d*c1);
	}


float Noise::EvalMono(ShadeContext& sc) {
	Point3 p,dp;
	if (gbufID) sc.SetGBufferID(gbufID);
	xyzGen->GetXYZ(sc,p,dp);
	p /= size;
    float d = NoiseFunc(p);
	float c0 = subTex[0] ? subTex[0]->EvalMono(sc): Intens(col[0]);
	float c1 = subTex[1] ? subTex[1]->EvalMono(sc): Intens(col[1]);
	return texout->Filter((1.0f-d)*c0 + d*c1);
	}


Point3 Noise::EvalNormalPerturb(ShadeContext& sc) {
	Point3 p,dp;
	if (gbufID) sc.SetGBufferID(gbufID);
	xyzGen->GetXYZ(sc,p,dp);
	p /= size;
	float del,d;
    d = NoiseFunc(p);
	//del = (dp.x+dp.y+dp.z)/(size*3.0f);
	del = .1f;
	Point3 np;					  
    np.x = (NoiseFunc(Point3(p.x+del,p.y,p.z)) - d)/del;
	np.y = (NoiseFunc(Point3(p.x,p.y+del,p.z)) - d)/del;
	np.z = (NoiseFunc(Point3(p.x,p.y,p.z+del)) - d)/del;
	return texout->Filter(np);
	}
