/**********************************************************************
 *<
	FILE: gradient.cpp

	DESCRIPTION: A simple gradient texture map.

	CREATED BY: Rolf Berteig

	HISTORY:

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/

#include "mtlhdr.h"
#include "mtlres.h"
#include <bmmlib.h>
#include "stdmat.h"

extern HINSTANCE hInstance;

#define NSUBTEX 3
#define NCOLS 3

static Class_ID gradClassID(0x66246cdf, 0x40eb43dd);

#define NOISE_REGULAR	0
#define NOISE_FRACTAL	1
#define NOISE_TURB		2

#define PB_COL1			0
#define PB_COL2			1
#define PB_COL3			2
#define PB_TYPE 		3
#define PB_AMOUNT		4
#define PB_SIZE			5
#define PB_PHASE		6
#define PB_CENTER		7
#define PB_NOISETYPE	8
#define PB_LEVELS		9
#define PB_LOWTHRESH	10
#define PB_HITHRESH		11
#define PB_THRESHSMOOTH	12

#define NPARAMS 13

#define GRAD_LINEAR	0
#define GRAD_RADIAL	1

class Gradient;
class GradientDlg;

class GradientDlg: public ParamDlg {
	public:
		HWND hwmedit;	 	// window handle of the materials editor dialog
		IMtlParams *ip;
		Gradient *theTex;	 
		HWND hPanel; 		// Rollup pane		
		IColorSwatch *cs[3];		
		ISpinnerControl *iAmount, *iSize, *iPhase, *iCenter, *iLevels;
		ISpinnerControl *iLow, *iHigh, *iSmooth;
		ICustButton *iBut[NSUBTEX];
		TexDADMgr dadMgr;
		ParamDlg *uvGenDlg;
		ParamDlg *texoutDlg;
		BOOL valid;
		BOOL isActive;
				
		GradientDlg(HWND hwMtlEdit, IMtlParams *imp, Gradient *m); 
		~GradientDlg();
		BOOL PanelProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam );		
		void LoadDialog(BOOL draw);  // stuff params into dialog
		void ReloadDialog();
		void UpdateMtlDisplay();		
		void UpdateSubTexNames();
		void ActivateDlg(BOOL onOff);
		void Invalidate();
		void DragAndDrop(int ifrom, int ito);
		BOOL KeyAtCurTime(int id);

		// methods inherited from ParamDlg:
		Class_ID ClassID() {return gradClassID;  }
		void SetThing(ReferenceTarget *m);
		ReferenceTarget* GetThing() { return (ReferenceTarget *)theTex; }
		void DeleteThis() { delete this;  }	
		void SetTime(TimeValue t);
		int FindSubTexFromHWND(HWND hw);
	};


#define UVGEN_REF	0
#define PBLOCK_REF	1
#define MAP1_REF	2
#define MAP2_REF	3
#define MAP3_REF	4
#define TEXOUT_REF	5

#define NUM_REFS	6



class Gradient;

//--------------------------------------------------------------
// MySampler: gradient sample function
//--------------------------------------------------------------
class GradSampler: public MapSampler {
	Gradient *grad;
	public:
		GradSampler() { grad= NULL; }
		GradSampler(Gradient *c) { grad= c; }
		void Set(Gradient *c) { grad = c; }
		AColor Sample(ShadeContext& sc, float u,float v);
		AColor SampleFilter(ShadeContext& sc, float u,float v, float du, float dv);
//		float SampleMono(ShadeContext& sc, float u,float v);
//		float SampleMonoFilter(ShadeContext& sc, float u,float v, float du, float dv);
	} ;


class Gradient: public GradTex { 
	public:	
		Color col[NCOLS];
		int type, noiseType;
		float amount, size, phase, size1, center, levels, low, high, smooth, sd, hminusl;
		UVGen *uvGen;		   // ref #0
		IParamBlock *pblock;   // ref #1		
		Texmap* subTex[NSUBTEX];  // More refs: 2,3,4
		BOOL mapOn[NSUBTEX];
		TextureOutput *texout; // ref #5
		TexHandle *texHandle;
		Interval ivalid;
		int rollScroll;
		GradSampler mysamp;
		GradientDlg *paramDlg;
	
		Gradient();
		~Gradient() { DiscardTexHandle(); }
		ParamDlg* CreateParamDlg(HWND hwMtlEdit, IMtlParams *imp);
		void Update(TimeValue t, Interval& valid);
		void Reset();
		Interval Validity(TimeValue t) {Interval v; Update(t,v); return ivalid;}

		StdUVGen* GetUVGen() { return (StdUVGen *)uvGen; }
		TextureOutput* GetTexout() { return texout; }
		void SetOutputLevel(TimeValue t, float v) {texout->SetOutputLevel(t,v); }
		void SetColor(int i, Color c, TimeValue t);		
		void SetMidPoint(float m, TimeValue t=0) {	pblock->SetValue(PB_CENTER,t,m);	}
		void NotifyChanged();		
		Bitmap *BuildBitmap(int size);
		float NoiseFunc(Point3 p);

		// Evaluate the color of map for the context.
		AColor EvalColor(ShadeContext& sc);
		float EvalMono(ShadeContext& sc);
		AColor EvalFunction(ShadeContext& sc, float u, float v, float du, float dv);
		AColor DispEvalFunc( float u, float v);		
		float gradFunc(float u, float v);

		// For Bump mapping, need a perturbation to apply to a normal.
		// Leave it up to the Texmap to determine how to do this.
		Point3 EvalNormalPerturb(ShadeContext& sc);

		// Methods for interactive display
		void DiscardTexHandle();
		BOOL SupportTexDisplay() { return TRUE; }
		void ActivateTexDisplay(BOOL onoff);
		DWORD GetActiveTexHandle(TimeValue t, TexHandleMaker& thmaker);
		void GetUVTransform(Matrix3 &uvtrans) { uvGen->GetUVTransform(uvtrans); }
		int GetTextureTiling() { return  uvGen->GetTextureTiling(); }
		int GetUVWSource() { return uvGen->GetUVWSource(); }
		UVGen *GetTheUVGen() { return uvGen; }

		// Requirements
		ULONG LocalRequirements(int subMtlNum) {
			return uvGen->Requirements(subMtlNum); 
			}

		// Methods to access texture maps of material
		int NumSubTexmaps() {return NSUBTEX;}
		Texmap* GetSubTexmap(int i) {return subTex[i];}		
		void SetSubTexmap(int i, Texmap *m);
		TSTR GetSubTexmapSlotName(int i);
		void InitSlotType(int sType) {if (uvGen) uvGen->InitSlotType(sType);}

		Class_ID ClassID() {return gradClassID;}
		SClass_ID SuperClassID() {return TEXMAP_CLASS_ID;}
		void GetClassName(TSTR& s) {s=GetString(IDS_RB_GRADIENT);}
		void DeleteThis() {delete this;}

		int NumSubs() {return NUM_REFS;}
		Animatable* SubAnim(int i);
		TSTR SubAnimName(int i);
		int SubNumToRefNum(int subNum) {return subNum;}

		// From ref
 		int NumRefs() {return NUM_REFS;}
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);

		RefTargetHandle Clone(RemapDir &remap = NoRemap());
		RefResult NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget, 
		   PartID& partID, RefMessage message );

		// IO
		IOResult Save(ISave *isave);
		IOResult Load(ILoad *iload);

	};

class GradientClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading) {return new Gradient;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_GRADIENT);}
	SClass_ID		SuperClassID() {return TEXMAP_CLASS_ID;}
	Class_ID 		ClassID() {return gradClassID;}
	const TCHAR* 	Category() {return TEXMAP_CAT_2D;}
	};
static GradientClassDesc gradCD;
ClassDesc* GetGradientDesc() {return &gradCD;}

static BOOL CALLBACK PanelDlgProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) 
	{
	GradientDlg *theDlg = (GradientDlg*)GetWindowLong(hWnd,GWL_USERDATA);
	if (msg==WM_INITDIALOG) {
		theDlg = (GradientDlg*)lParam;
		theDlg->hPanel = hWnd;
		SetWindowLong(hWnd,GWL_USERDATA,lParam);
		}
	if (theDlg) {	
		theDlg->isActive = TRUE;
		int	res = theDlg->PanelProc(hWnd,msg,wParam,lParam);
		theDlg->isActive = FALSE;
		return res;
		}
	else
		return FALSE;
	}


int GradientDlg::FindSubTexFromHWND(HWND hw) {
	for (int i=0; i<NSUBTEX; i++) {
		if (hw == iBut[i]->GetHwnd()) return i;
		}	
	return -1;
	}

//-------------------------------------------------------------------

GradientDlg::GradientDlg(HWND hwMtlEdit, IMtlParams *imp, Gradient *m) 
	{
	dadMgr.Init(this);
	hwmedit  = hwMtlEdit;
	ip       = imp;
	hPanel   = NULL;
	theTex   = m; 	
	valid    = FALSE;
	uvGenDlg = theTex->uvGen->CreateParamDlg(hwmedit, imp);
	for (int i=0; i<NSUBTEX; i++) iBut[i] = NULL;
	hPanel   = ip->AddRollupPage( 
		hInstance,
		MAKEINTRESOURCE(IDD_GRADIENT),
		PanelDlgProc, 
		GetString(IDS_RB_GRADIENTPARAMS), 
		(LPARAM)this);	
	texoutDlg = theTex->texout->CreateParamDlg(hwmedit, imp);
	}

void GradientDlg::Invalidate()
	{
	valid = FALSE;
	isActive = FALSE;
	Rect rect;
	rect.left = rect.top = 0;
	rect.right = rect.bottom = 10;
	InvalidateRect(hPanel,&rect,FALSE);
	}

void GradientDlg::ReloadDialog() 
	{
	Interval valid;
	theTex->Update(ip->GetTime(), valid);
	LoadDialog(FALSE);
	}

void GradientDlg::SetTime(TimeValue t) 
	{
	Interval valid;	
	uvGenDlg->SetTime(t);		
	texoutDlg->SetTime(t);
	theTex->Update(ip->GetTime(), valid);
	LoadDialog(FALSE);
	InvalidateRect(hPanel,NULL,0);	
	}

GradientDlg::~GradientDlg() 
	{
	theTex->paramDlg = NULL;	
	ReleaseIColorSwatch(cs[0]);
	ReleaseIColorSwatch(cs[1]);
	ReleaseIColorSwatch(cs[2]);
	ReleaseISpinner(iAmount);
	ReleaseISpinner(iSize);
	ReleaseISpinner(iPhase);
	ReleaseISpinner(iCenter);
	ReleaseISpinner(iLevels);
	ReleaseISpinner(iLow);
	ReleaseISpinner(iHigh);
	ReleaseISpinner(iSmooth);
	for (int i=0; i<NSUBTEX; i++) {
		ReleaseICustButton(iBut[i]);
		iBut[i] = NULL; 
		}
	SetWindowLong(hPanel, GWL_USERDATA, NULL);
	ip->DeleteRollupPage(hPanel);
	uvGenDlg->DeleteThis();
	texoutDlg->DeleteThis();
	hPanel =  NULL;
	}

static int colID[3] = { IDC_GRAD_COL1, IDC_GRAD_COL2, IDC_GRAD_COL3 };
static int subTexId[NSUBTEX] = {IDC_GRAD_TEX1, IDC_GRAD_TEX2, IDC_GRAD_TEX3};

BOOL GradientDlg::PanelProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam ) 
	{
	int id = LOWORD(wParam);
	int code = HIWORD(wParam);
    switch (msg)    {
		case WM_INITDIALOG: {			
			iAmount = SetupFloatSpinner(hWnd, IDC_GRAD_AMOUNTSPIN, IDC_GRAD_AMOUNT,0.0f,1.0f,0.0f,.01f);
			iSize   = SetupFloatSpinner(hWnd, IDC_GRAD_SIZESPIN, IDC_GRAD_SIZE,0.0f,999999999.0f,0.0f,.01f);
			iPhase  = SetupFloatSpinner(hWnd, IDC_GRAD_PHASESPIN, IDC_GRAD_PHASE,0.0f,999999999.0f,0.0f,.01f);
			iCenter = SetupFloatSpinner(hWnd, IDC_GRAD_CENTERSPIN, IDC_GRAD_CENTER,0.0f,1.0f,0.0f,.01f);
			iLevels = SetupFloatSpinner(hWnd, IDC_GRAD_LEVELSSPIN, IDC_GRAD_LEVELS,1.0f,10.0f,0.0f,.05f);
			iLow    = SetupFloatSpinner(hWnd, IDC_GRAD_LOWTHRESHSPIN, IDC_GRAD_LOWTHRESH,0.0f,1.0f,0.0f,.005f);
			iHigh   = SetupFloatSpinner(hWnd, IDC_GRAD_HIGHTHRESHSPIN, IDC_GRAD_HIGHTHRESH,0.0f,1.0f,0.0f,.005f);
			iSmooth = SetupFloatSpinner(hWnd, IDC_GRAD_THRESHSMOOTHSPIN, IDC_GRAD_THRESHSMOOTH,0.0f,1.0f,0.0f,.005f);
			for (int i=0; i<NCOLS; i++) 
   				cs[i] = GetIColorSwatch(GetDlgItem(hWnd, colID[i]),
   					theTex->col[i],theTex->GetSubTexmapSlotName(i).data());
			for (i=0; i<NSUBTEX; i++) {
				iBut[i] = GetICustButton(GetDlgItem(hWnd,subTexId[i]));
				iBut[i]->SetDADMgr(&dadMgr);
				SetCheckBox(hWnd, IDC_MAPON1+i, theTex->mapOn[i]);
				}
			return TRUE;
			}
			
		case WM_PAINT:
			if (!valid) {
				valid = TRUE;
				ReloadDialog();
				}
			return FALSE;

		case WM_COMMAND:  
		    switch (id) {
				case IDC_GRAD_TEX1: 
					PostMessage(hwmedit,WM_TEXMAP_BUTTON,0 ,(LPARAM)theTex);
					break;				
				case IDC_GRAD_TEX2: 
					PostMessage(hwmedit,WM_TEXMAP_BUTTON,1 ,(LPARAM)theTex);
					break;
				case IDC_GRAD_TEX3: 
					PostMessage(hwmedit,WM_TEXMAP_BUTTON,2 ,(LPARAM)theTex);
					break;
				case IDC_MAPON1:
				case IDC_MAPON2:
				case IDC_MAPON3:
					theTex->mapOn[id-IDC_MAPON1] = GetCheckBox(hWnd,id);
					theTex->NotifyChanged();
					UpdateMtlDisplay();
					break;

				case IDC_GRAD_LINEAR:
					theTex->pblock->SetValue(PB_TYPE,0,GRAD_LINEAR);
					theTex->NotifyChanged();
					UpdateMtlDisplay();
					break;
				case IDC_GRAD_RADIAL:
					theTex->pblock->SetValue(PB_TYPE,0,GRAD_RADIAL);
					theTex->NotifyChanged();
					UpdateMtlDisplay();
					break;
				
				case IDC_GRAD_REGULAR:
					theTex->pblock->SetValue(PB_NOISETYPE,0,NOISE_REGULAR);
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					iLevels->Disable();
					break;
				case IDC_GRAD_FRACT:
					theTex->pblock->SetValue(PB_NOISETYPE,0,NOISE_FRACTAL);
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					iLevels->Enable();
					break;
				case IDC_GRAD_TURB:
					theTex->pblock->SetValue(PB_NOISETYPE,0,NOISE_TURB);
					UpdateMtlDisplay();
					theTex->NotifyChanged();
					iLevels->Enable();
					break;
				}
			break;

		case CC_COLOR_BUTTONDOWN:
			theHold.Begin();
			break;

		case CC_COLOR_BUTTONUP:
			if (HIWORD(wParam)) theHold.Accept(GetString(IDS_DS_PARAMCHG));
			else theHold.Cancel();
			break;

		case CC_COLOR_CHANGE: {
			int id = LOWORD(wParam);
			int buttonUp = HIWORD(wParam); 
			int n;
			switch (id) {
				default:
				case IDC_GRAD_COL1: n = 0; break;
				case IDC_GRAD_COL2: n = 1; break;
				case IDC_GRAD_COL3: n = 2; break;				
				}			
			if (buttonUp) theHold.Begin();
			theTex->SetColor(n,cs[n]->GetColor(),ip->GetTime());
			cs[n]->SetKeyBrackets(KeyAtCurTime(PB_COL1+n));
			if (buttonUp) {
				theHold.Accept(GetString(IDS_DS_PARAMCHG));
				UpdateMtlDisplay();
				theTex->NotifyChanged();
				}
			break;
			}
		
		case CC_SPINNER_CHANGE: 
			if (!theHold.Holding()) theHold.Begin();
			switch (id) {
				case IDC_GRAD_AMOUNTSPIN:
					theTex->pblock->SetValue(PB_AMOUNT,ip->GetTime(),iAmount->GetFVal());
					iAmount->SetKeyBrackets(KeyAtCurTime(PB_AMOUNT));
					break;
				case IDC_GRAD_SIZESPIN:
					theTex->pblock->SetValue(PB_SIZE,ip->GetTime(),iSize->GetFVal());
					iSize->SetKeyBrackets(KeyAtCurTime(PB_SIZE));
					break;
				case IDC_GRAD_PHASESPIN:
					theTex->pblock->SetValue(PB_PHASE,ip->GetTime(),iPhase->GetFVal());
					iPhase->SetKeyBrackets(KeyAtCurTime(PB_PHASE));
					break;
				case IDC_GRAD_CENTERSPIN:
					theTex->pblock->SetValue(PB_CENTER,ip->GetTime(),iCenter->GetFVal());
					iCenter->SetKeyBrackets(KeyAtCurTime(PB_CENTER));
					break;
				case IDC_GRAD_LEVELSSPIN:
					theTex->pblock->SetValue(PB_LEVELS,ip->GetTime(),iLevels->GetFVal());
					iLevels->SetKeyBrackets(KeyAtCurTime(PB_LEVELS));
					break;
				case IDC_GRAD_LOWTHRESHSPIN:
					theTex->pblock->SetValue(PB_LOWTHRESH,ip->GetTime(),iLow->GetFVal());
					iLow->SetKeyBrackets(KeyAtCurTime(PB_LOWTHRESH));
					break;
				case IDC_GRAD_HIGHTHRESHSPIN:
					theTex->pblock->SetValue(PB_HITHRESH,ip->GetTime(),iHigh->GetFVal());
					iHigh->SetKeyBrackets(KeyAtCurTime(PB_HITHRESH));
					break;
				case IDC_GRAD_THRESHSMOOTHSPIN:
					theTex->pblock->SetValue(PB_THRESHSMOOTH,ip->GetTime(),iSmooth->GetFVal());
					iSmooth->SetKeyBrackets(KeyAtCurTime(PB_THRESHSMOOTH));
					break;
				}
			theTex->NotifyChanged();
			break;
				
		
		case CC_SPINNER_BUTTONDOWN:
			theHold.Begin();
			break;

		case WM_CUSTEDIT_ENTER:
		case CC_SPINNER_BUTTONUP: 
			if (HIWORD(wParam) || msg==WM_CUSTEDIT_ENTER) theHold.Accept(GetString(IDS_DS_PARAMCHG));
			else theHold.Cancel();
			theTex->NotifyChanged();
		    UpdateMtlDisplay();
			break;

    	}
	return FALSE;
	}


void GradientDlg::UpdateSubTexNames() 
	{
	for (int i=0; i<NSUBTEX; i++) {
		Texmap *m = theTex->subTex[i];
		TSTR nm;
		if (m) 	nm = m->GetFullName();
		else 	nm = GetString(IDS_DS_NONE);
		iBut[i]->SetText(nm.data());
		}
	}

BOOL GradientDlg::KeyAtCurTime(int id) { return theTex->pblock->KeyFrameAtTime(id,ip->GetTime()); }

void GradientDlg::LoadDialog(BOOL draw) 
	{
	if (theTex) {
		Interval valid;
		theTex->Update(ip->GetTime(),valid);		
		for (int i=0; i<3; i++) {
			cs[i]->SetColor(theTex->col[i]);
			cs[i]->SetKeyBrackets(KeyAtCurTime(PB_COL1+i));
			}
		CheckDlgButton(hPanel,IDC_GRAD_LINEAR,theTex->type==GRAD_LINEAR);
		CheckDlgButton(hPanel,IDC_GRAD_RADIAL,theTex->type==GRAD_RADIAL);				
		CheckDlgButton(hPanel,IDC_GRAD_REGULAR,theTex->noiseType==NOISE_REGULAR);
		CheckDlgButton(hPanel,IDC_GRAD_FRACT,theTex->noiseType==NOISE_FRACTAL);
		CheckDlgButton(hPanel,IDC_GRAD_TURB,theTex->noiseType==NOISE_TURB);
		iAmount->SetValue(theTex->amount,FALSE);
		iSize->SetValue(theTex->size,FALSE);
		iPhase->SetValue(theTex->phase,FALSE);
		iCenter->SetValue(theTex->center,FALSE);
		iLevels->SetValue(theTex->levels,FALSE);
		iLow->SetValue(theTex->low,FALSE);
		iHigh->SetValue(theTex->high,FALSE);
		iSmooth->SetValue(theTex->smooth,FALSE);
		UpdateSubTexNames();
		if (theTex->noiseType==NOISE_REGULAR) {
			iLevels->Disable();
		} else {
			iLevels->Enable();
			}
		iAmount->SetKeyBrackets(KeyAtCurTime(PB_AMOUNT));
		iSize->SetKeyBrackets(KeyAtCurTime(PB_SIZE));
		iPhase->SetKeyBrackets(KeyAtCurTime(PB_PHASE));
		iCenter->SetKeyBrackets(KeyAtCurTime(PB_CENTER));
		iLevels->SetKeyBrackets(KeyAtCurTime(PB_LEVELS));
		iLow->SetKeyBrackets(KeyAtCurTime(PB_LOWTHRESH));
		iHigh->SetKeyBrackets(KeyAtCurTime(PB_HITHRESH));
		iSmooth->SetKeyBrackets(KeyAtCurTime(PB_THRESHSMOOTH));
		for (i=0; i<NSUBTEX; i++) 
			SetCheckBox(hPanel, IDC_MAPON1+i, theTex->mapOn[i]);
		}
	}

void GradientDlg::SetThing(ReferenceTarget *m) 
	{
	assert (m->ClassID()==gradClassID);
	assert (m->SuperClassID()==TEXMAP_CLASS_ID);
	if (theTex) theTex->paramDlg = NULL;
	theTex = (Gradient*)m;
	uvGenDlg->SetThing(theTex->uvGen);
	texoutDlg->SetThing(theTex->texout);
	if (theTex) theTex->paramDlg = this;
	LoadDialog(TRUE);
	}

void GradientDlg::UpdateMtlDisplay() { 
	theTex->DiscardTexHandle();  
	ip->MtlChanged();  
	}

void GradientDlg::ActivateDlg(BOOL onOff) {
	for (int i=0; i<NCOLS; i++)
		cs[i]->Activate(onOff);
	}

//-----------------------------------------------------------------------------
//  GradSampler
//-----------------------------------------------------------------------------
AColor GradSampler::SampleFilter(ShadeContext& sc, float u,float v, float du, float dv) {
	return grad->EvalFunction(sc, u, v, du, dv);
	}

AColor GradSampler::Sample(ShadeContext& sc, float u,float v) {
	return grad->EvalFunction(sc, u, v, 0.0f, 0.0f);
	}
//-----------------------------------------------------------------------------
//  Gradient
//-----------------------------------------------------------------------------

#define GRADIENT_VERSION 4


static ParamBlockDescID pbdesc1[] = {	
	{ TYPE_POINT3, NULL, TRUE,0 },	// col1
	{ TYPE_POINT3, NULL, TRUE,1 }, 	// col2
	{ TYPE_POINT3, NULL, TRUE,2 },	// col3
	{ TYPE_INT, NULL, FALSE,3 },	// type
	{ TYPE_FLOAT, NULL, TRUE,4 },	// amount
	{ TYPE_FLOAT, NULL, TRUE,5 },	// size
	{ TYPE_FLOAT, NULL, TRUE,6 },	// phase
	};

static ParamBlockDescID pbdesc2[] = {	
	{ TYPE_POINT3, NULL, TRUE,0 },	// col1
	{ TYPE_POINT3, NULL, TRUE,1 }, 	// col2
	{ TYPE_POINT3, NULL, TRUE,2 },	// col3
	{ TYPE_INT,    NULL, FALSE,3 },	// type
	{ TYPE_FLOAT, NULL, TRUE,4 },	// amount
	{ TYPE_FLOAT, NULL, TRUE,5 },	// size
	{ TYPE_FLOAT, NULL, TRUE,6 },	// phase
	{ TYPE_FLOAT, NULL, TRUE,7 },	// center
	};

static ParamBlockDescID pbdesc3[] = {	
	{ TYPE_POINT3, NULL, TRUE,0 },	// col1
	{ TYPE_POINT3, NULL, TRUE,1 }, 	// col2
	{ TYPE_POINT3, NULL, TRUE,2 },	// col3
	{ TYPE_INT, NULL, FALSE,3 },	// type
	{ TYPE_FLOAT, NULL, TRUE,4 },	// amount
	{ TYPE_FLOAT, NULL, TRUE,5 },	// size
	{ TYPE_FLOAT, NULL, TRUE,6 },	// phase
	{ TYPE_FLOAT, NULL, TRUE,7 },	// center
	{ TYPE_INT, NULL, FALSE,8 },	// turbulence
	{ TYPE_FLOAT, NULL, TRUE,9 },	// levels
	};

static ParamBlockDescID pbdesc4[] = {	
	{ TYPE_RGBA, NULL, TRUE,0 },	// col1
	{ TYPE_RGBA, NULL, TRUE,1 }, 	// col2
	{ TYPE_RGBA, NULL, TRUE,2 },	// col3
	{ TYPE_INT, NULL, FALSE,3 },	// type
	{ TYPE_FLOAT, NULL, TRUE,4 },	// amount
	{ TYPE_FLOAT, NULL, TRUE,5 },	// size
	{ TYPE_FLOAT, NULL, TRUE,6 },	// phase
	{ TYPE_FLOAT, NULL, TRUE,7 },	// center
	{ TYPE_INT, NULL, FALSE,8 },	// noise type
	{ TYPE_FLOAT, NULL, TRUE,9 },	// levels
	{ TYPE_FLOAT, NULL, TRUE,10 },	// low thresh
	{ TYPE_FLOAT, NULL, TRUE,11 },	// high thresh
	{ TYPE_FLOAT, NULL, TRUE,12 },	// thresh smoothing
	};

// Array of old versions
static ParamVersionDesc versions[] = {
	ParamVersionDesc(pbdesc1,7,1),	
	ParamVersionDesc(pbdesc2,8,2),
	ParamVersionDesc(pbdesc3,10,3),
	};
#define NUM_OLDVERSIONS	3

static ParamVersionDesc curVersion(pbdesc4,NPARAMS,GRADIENT_VERSION);


void Gradient::Reset() 
	{
	if (uvGen) uvGen->Reset();
	else ReplaceReference( 0, GetNewDefaultUVGen());	
	if (texout) texout->Reset();
	else ReplaceReference( TEXOUT_REF, GetNewDefaultTextureOutput());
	ReplaceReference( 1, CreateParameterBlock(pbdesc4, NPARAMS, GRADIENT_VERSION));
	ivalid.SetEmpty();
	DeleteReference(2);
	DeleteReference(3);
	DeleteReference(4);
	SetColor(0, Color(1.0f,1.0f,1.0f), TimeValue(0));
	SetColor(1, Color(0.5f,0.5f,0.5f), TimeValue(0));
	SetColor(2, Color(0.0f,0.0f,0.0f), TimeValue(0));
	pblock->SetValue(PB_TYPE,0,GRAD_LINEAR);
	pblock->SetValue(PB_AMOUNT,0,1.0f);
	pblock->SetValue(PB_SIZE,0,1.0f);
	pblock->SetValue(PB_PHASE,0,0.0f);
	pblock->SetValue(PB_CENTER,0,0.5f);
	pblock->SetValue(PB_LEVELS,0,4.0f);
	pblock->SetValue(PB_HITHRESH,0,1.0f);
	mapOn[0] = mapOn[1] = mapOn[2] = 1;
	}

void Gradient::NotifyChanged() {
	NotifyDependents(FOREVER, PART_ALL, REFMSG_CHANGE);
	}

Gradient::Gradient() 
	{
	texHandle = NULL;
	subTex[0] = subTex[1] = subTex[2] =NULL;
	pblock    = NULL;
	uvGen     = NULL;
	texout    = NULL;
	paramDlg  = NULL;
	mysamp.Set(this);
	Reset();
	rollScroll=0;
	}

void Gradient::DiscardTexHandle() 
	{
	if (texHandle) {
		texHandle->DeleteThis();
		texHandle = NULL;
		}
	}

void Gradient::ActivateTexDisplay(BOOL onoff) {
	if (!onoff) 
		DiscardTexHandle();
	}

DWORD Gradient::GetActiveTexHandle(TimeValue t, TexHandleMaker& thmaker) {
	if (texHandle) 
		return texHandle->GetHandle();
	else {
		Bitmap *bm;
		Interval v;
		Update(t,v);
		bm = BuildBitmap(thmaker.Size());
		texHandle = thmaker.CreateHandle(bm,uvGen->SymFlags());
		bm->DeleteThis();
		}
	return texHandle->GetHandle();
	}

inline UWORD FlToWord(float r) {
	return (UWORD)(65535.0f*r);
	}

Bitmap *Gradient::BuildBitmap(int size) {
	float u,v;
	BitmapInfo bi;
	bi.SetName(GetString(IDS_RB_GRADTEMP));
	bi.SetWidth(size);
	bi.SetHeight(size);
	bi.SetType(BMM_TRUE_32);
	Bitmap *bm = TheManager->Create(&bi);
	if (bm==NULL) return NULL;
	PixelBuf l64(size);
	float d = 1.0f/float(size);
	v = 1.0f - 0.5f*d;
	for (int y=0; y<size; y++) {
		BMM_Color_64 *p64=l64.Ptr();
		u = 0.0f;
		for (int x=0; x<size; x++, p64++) {
			AColor c = DispEvalFunc(u,v);
			p64->r = FlToWord(c.r); 
			p64->g = FlToWord(c.g); 
			p64->b = FlToWord(c.b);
			p64->a = 0xffff; 
			u += d;
			}
		bm->PutPixels(0,y, size, l64.Ptr()); 
		v -= d;
		}
	return bm;
	}

float Gradient::NoiseFunc(Point3 p)
	{
	float res;
	switch (noiseType) {
		case NOISE_TURB: {
			float sum = 0.0f;
			float l,f = 1.0f;			
			for (l = levels; l>=1.0f; l-=1.0f) {				
				sum += (float)fabs(noise3(p*f))/f;
				f *= 2.0f;
				}
			if (l>0.0f) {				
				sum += l*(float)fabs(noise3(p*f))/f;
				}
			res = sum;
//			res = (1.0+sum)/2.0f;
			break;
			}
			
		case NOISE_REGULAR:
			res = (float)(1.0f+noise3(p))/2.0f;
			break;

		case NOISE_FRACTAL:
			if (levels==1.0f) {
				res = noise3(p);
			} else {
				float sum = 0.0f;
				float l,f = 1.0f;				
				for (l = levels; l>=1.0f; l-=1.0f) {					
					sum += noise3(p*f)/f;
					f *= 2.0f;
					}
				if (l>0.0f) {					
					sum += l*noise3(p*f)/f;
					}
				res = (1.0f+sum)/2.0f;
				}
			break;
		}
	
	if (low<high) {
		res = 2.0f * sramp((res+1.0f)/2.0f,low,high,sd) - 1.0f;
		}
	return res;
	}

float Gradient::gradFunc(float u, float v) {
	float a;
/*
	if (type==GRAD_LINEAR) {
		a = v;	
	} else {
		u-=0.5f;
		v-=0.5f;
		a = (float)sqrt(u*u+v*v)*2.0f;
		if (a>1.0f) a = 1.0f;
		}
*/
	a = 0.0f;
	if (amount!=0.0f) {
		a += amount*NoiseFunc(Point3(u*size1+1.0f,v*size1+1.0f,phase));
		if (a<0.0f) a = 0.0f;
		if (a>1.0f) a = 1.0f;
		}
//	a = (a*a*(3-2*a));
	return a;
	}

// simplified evaluation for interactive render
AColor Gradient::DispEvalFunc(float u, float v) 
	{
	float a = gradFunc(u,v);

	if (a<center) {
		a = a/center;
		return col[2]*(1.0f-a) + col[1]*a;
	} else 
	if (a>center) {
		a = (a-center)/(1.0f-center);
		return col[1]*(1.0f-a) + col[0]*a;
	} else return col[1];
	}

AColor Gradient::EvalFunction(
		ShadeContext& sc, float u, float v, float du, float dv) 
	{	
	int n1=0, n2=0;
	float a = gradFunc(u,v);

	if (a<center) {
		a = a/center;
		n1 = 2;
		n2 = 1;		
	} else 
	if (a>center) {
		a = (a-center)/(1.0f-center);		
		n1 = 1;
		n2 = 0;
	} else {
		return (mapOn[1]&&subTex[1]) ? subTex[1]->EvalColor(sc): col[1];		
		}

	Color c1, c2;
	c1 = mapOn[n1]&&subTex[n1] ? subTex[n1]->EvalColor(sc): col[n1];	
	c2 = mapOn[n2]&&subTex[n2] ? subTex[n2]->EvalColor(sc): col[n2];
	return c1*(1.0f-a) + c2*a;
	}

static AColor black(0.0f,0.0f,0.0f,0.0f);

AColor Gradient::EvalColor(ShadeContext& sc) {
	if (!sc.doMaps) 
		return black;
	AColor c;
	if (sc.GetCache(this,c)) 
		return c; 
	if (gbufID) sc.SetGBufferID(gbufID);
	c = texout->Filter(uvGen->EvalUVMap(sc,&mysamp));
	sc.PutCache(this,c); 
	return c;
	}

float Gradient::EvalMono(ShadeContext& sc) {
	if (!sc.doMaps) 
		return 0.0f;
	float f;
	if (sc.GetCache(this,f)) 
		return f; 
	if (gbufID) sc.SetGBufferID(gbufID);
	f = texout->Filter(uvGen->EvalUVMapMono(sc,&mysamp));
	sc.PutCache(this,f); 
	return f;
	}

Point3 Gradient::EvalNormalPerturb(ShadeContext& sc) 
	{
	Point3 dPdu, dPdv;
	if (!sc.doMaps) return Point3(0,0,0);
	if (gbufID) sc.SetGBufferID(gbufID);
	Point2 dM = uvGen->EvalDeriv(sc,&mysamp);
	uvGen->GetBumpDP(sc,dPdu,dPdv);

#if 0
	// Blinn's algorithm
	Point3 N = sc.Normal();
	Point3 uVec = CrossProd(N,dPdv);
	Point3 vVec = CrossProd(N,dPdu);
	return texout->Filter(-dM.x*uVec+dM.y*vVec);
#else 
	// Lazy algorithm
	return texout->Filter(dM.x*dPdu+dM.y*dPdv);
#endif

	}

RefTargetHandle Gradient::Clone(RemapDir &remap) 
	{
	Gradient *mnew = new Gradient();
	*((MtlBase*)mnew) = *((MtlBase*)this);  // copy superclass stuff
	mnew->ReplaceReference(0,remap.CloneRef(uvGen));
	mnew->ReplaceReference(TEXOUT_REF,remap.CloneRef(texout));
	mnew->ReplaceReference(1,remap.CloneRef(pblock));
	mnew->col[0] = col[0];
	mnew->col[1] = col[1];
	mnew->col[2] = col[2];	
	mnew->ivalid.SetEmpty();	
	for (int i = 0; i<NSUBTEX; i++) {
		mnew->subTex[i] = NULL;
		if (subTex[i])
			mnew->ReplaceReference(i+2,remap.CloneRef(subTex[i]));
		mnew->mapOn[i] = mapOn[i];
		}
	return (RefTargetHandle)mnew;
	}

ParamDlg* Gradient::CreateParamDlg(HWND hwMtlEdit, IMtlParams *imp) 
	{
	GradientDlg *dm = new GradientDlg(hwMtlEdit, imp, this);
	dm->LoadDialog(TRUE);	
	paramDlg = dm;
	return dm;	
	}


void Gradient::Update(TimeValue t, Interval& valid) 
	{
	if (!ivalid.InInterval(t)) {
		ivalid.SetInfinite();
		uvGen->Update(t,ivalid);
		texout->Update(t,ivalid);
		pblock->GetValue( PB_COL1, t, col[0], ivalid );
		col[0].ClampMinMax();
		pblock->GetValue( PB_COL2, t, col[1], ivalid );
		col[1].ClampMinMax();
		pblock->GetValue( PB_COL3, t, col[2], ivalid );
		col[2].ClampMinMax();		
		pblock->GetValue( PB_TYPE, t, type, ivalid );
		pblock->GetValue( PB_NOISETYPE, t, noiseType, ivalid );
		pblock->GetValue( PB_AMOUNT, t, amount, ivalid );
		pblock->GetValue( PB_SIZE, t, size, ivalid );
		pblock->GetValue( PB_PHASE, t, phase, ivalid );
		pblock->GetValue( PB_CENTER, t, center, ivalid );
		pblock->GetValue( PB_LEVELS, t, levels, ivalid );
		pblock->GetValue( PB_HITHRESH, t, high, ivalid );
		pblock->GetValue( PB_LOWTHRESH, t, low, ivalid );
		pblock->GetValue( PB_THRESHSMOOTH, t, smooth, ivalid );		
		if (low>high) {
			float temp = low;
			low = high;
			high = temp;
			}
		hminusl = (high-low);
		sd = hminusl*0.5f*smooth;
		if (size!=0.0f) size1 = 20.0f/size;
		else size1 = 0.0f;
		for (int i=0; i<NSUBTEX; i++) {
			if (subTex[i]) 
				subTex[i]->Update(t,ivalid);
			}
		}
	valid &= ivalid;
	}


void Gradient::SetColor(int i, Color c, TimeValue t) {
    col[i] = c;	
	pblock->SetValue(PB_COL1+i, t, c);
	}

RefTargetHandle Gradient::GetReference(int i) {
	switch(i) {
		case 0: return uvGen;
		case 1:	return pblock ;
		case TEXOUT_REF: return texout;
		default:return subTex[i-2];
		}
	}

void Gradient::SetReference(int i, RefTargetHandle rtarg) {
	switch(i) {
		case 0: uvGen = (UVGen *)rtarg; break;
		case 1:	pblock = (IParamBlock *)rtarg; break;
		case TEXOUT_REF: texout = (TextureOutput *)rtarg; break;
		default: subTex[i-2] = (Texmap *)rtarg; break;
		}
	}

void Gradient::SetSubTexmap(int i, Texmap *m) {
	ReplaceReference(i+2,m);
	if (paramDlg)
		paramDlg->UpdateSubTexNames();
	}

TSTR Gradient::GetSubTexmapSlotName(int i) {
	switch(i) {
		case 0:  return GetString(IDS_RB_COLOR1); 
		case 1:  return GetString(IDS_RB_COLOR2);
		case 2:  return GetString(IDS_RB_COLOR3);
		default: return TSTR(_T(""));
		}
	}
	 
Animatable* Gradient::SubAnim(int i) {
	switch (i) {
		case 0: return uvGen;
		case 1: return pblock;
		case TEXOUT_REF: return texout;
		default: return subTex[i-2]; 
		}
	}

TSTR Gradient::SubAnimName(int i) {
	switch (i) {
		case 0: return TSTR(GetString(IDS_DS_COORDINATES));		
		case 1: return TSTR(GetString(IDS_DS_PARAMETERS));		
		case TEXOUT_REF: return TSTR(GetString(IDS_DS_OUTPUT));
		default: return GetSubTexmapTVName(i-2);
		}
	}

RefResult Gradient::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
   PartID& partID, RefMessage message ) {
	switch (message) {
		case REFMSG_CHANGE:
			ivalid.SetEmpty();
			if (hTarget!=uvGen&&hTarget!=texout) {
//				if (paramDlg&&!paramDlg->isActive) 
				if (paramDlg) 
					paramDlg->Invalidate();					
				}
			break;

		case REFMSG_UV_SYM_CHANGE:
			DiscardTexHandle();  
			break;

		case REFMSG_GET_PARAM_DIM: {
			GetParamDim *gpd = (GetParamDim*)partID;
			switch (gpd->index) {
				case PB_COL1: 
				case PB_COL3: 
				case PB_COL2: gpd->dim = stdColor255Dim; break;
				}
			return REF_STOP; 
			}

		case REFMSG_GET_PARAM_NAME: {
			GetParamName *gpn = (GetParamName*)partID;
			switch (gpn->index) {
				case PB_COL1: 			gpn->name= GetString(IDS_RB_COLOR1); break;
				case PB_COL2: 			gpn->name= GetString(IDS_RB_COLOR2); break;
				case PB_COL3: 			gpn->name= GetString(IDS_RB_COLOR3); break;
				case PB_AMOUNT: 		gpn->name= GetString(IDS_RB_AMOUNT); break;
				case PB_SIZE: 			gpn->name= GetString(IDS_RB_SIZE); break;
				case PB_PHASE: 			gpn->name= GetString(IDS_RB_PHASE); break;
				case PB_CENTER:			gpn->name= GetString(IDS_RB_CENTER2); break;
				case PB_LEVELS:			gpn->name= GetString(IDS_RB_LEVELS); break;
				case PB_HITHRESH:		gpn->name= GetString(IDS_RB_HIGHTHRESHOLD); break;
				case PB_LOWTHRESH:		gpn->name= GetString(IDS_RB_LOWTHRESHOLD); break;
				case PB_THRESHSMOOTH:	gpn->name= GetString(IDS_RB_THRESHOLDSMOOTHING); break;
				}			
			return REF_STOP; 
			}
		}
	return(REF_SUCCEED);
	}


#define MTL_HDR_CHUNK 0x4000
#define MAPOFF_CHUNK 0x1000

IOResult Gradient::Save(ISave *isave) { 
	IOResult res;
	// Save common stuff
	isave->BeginChunk(MTL_HDR_CHUNK);
	res = MtlBase::Save(isave);
	if (res!=IO_OK) return res;
	isave->EndChunk();

	for (int i=0; i<NSUBTEX; i++) {
		if (mapOn[i]==0) {
			isave->BeginChunk(MAPOFF_CHUNK+i);
			isave->EndChunk();
			}
		}
	return IO_OK;
	}	
	  


IOResult Gradient::Load(ILoad *iload) { 
	IOResult res;
	int id;
	while (IO_OK==(res=iload->OpenChunk())) {
		switch(id = iload->CurChunkID())  {
			case MTL_HDR_CHUNK:
				res = MtlBase::Load(iload);
				break;
			case MAPOFF_CHUNK+0:
			case MAPOFF_CHUNK+1:
			case MAPOFF_CHUNK+2:
				mapOn[id-MAPOFF_CHUNK] = 0; 
				break;
			}
		iload->CloseChunk();
		if (res!=IO_OK) 
			return res;
		}
	iload->RegisterPostLoadCallback(
		new ParamBlockPLCB(versions,NUM_OLDVERSIONS,&curVersion,this,1));
	return IO_OK;
	}
