/**********************************************************************
 *<
	FILE: gravity.cpp

	DESCRIPTION:  Gravity and Wind World Space Modifier

	CREATED BY: Rolf Berteig

	HISTORY: 10-30-95

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/
#include "mods.h"
#include "iparamm.h"
#include "simpmod.h"
#include "simpobj.h"
#include "texutil.h"

#define FORCE_PLANAR	0
#define FORCE_SPHERICAL	1


class ForceObject : public SimpleWSMObject {	
	public:		
		static IParamMap *pmapParam;
		static IObjParam *ip;
		static HWND hSot;
					
		ForceObject() {}
		BOOL SupportsDynamics() {return TRUE;}

		// From Animatable		
		void DeleteThis() {delete this;}		
		void BeginEditParams( IObjParam  *ip, ULONG flags,Animatable *prev );
		void EndEditParams( IObjParam *ip, ULONG flags,Animatable *next);		
				
		// from object		
		CreateMouseCallBack* GetCreateMouseCallBack();		
		
		// From SimpleWSMObject		
		void InvalidateUI();		
		void BuildMesh(TimeValue t);

		virtual int DialogID()=0;
		virtual ParamUIDesc *UIDesc()=0;
		virtual int UIDescLength()=0;
	};
IObjParam *ForceObject::ip        = NULL;
IParamMap *ForceObject::pmapParam = NULL;
HWND       ForceObject::hSot      = NULL;


class WindObject : public ForceObject {	
	public:									
		WindObject();		

		// From Animatable		
		void DeleteThis() {delete this;}		
		Class_ID ClassID() {return Class_ID(WINDOBJECT_CLASS_ID,8888);}		
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_WIND);}
						
		// From WSMObject
		Modifier *CreateWSMMod(INode *node);
		ForceField *GetForceField(INode *node);
		
		// From SimpleWSMObject				
		ParamDimension *GetParameterDim(int pbIndex);
		TSTR GetParameterName(int pbIndex);

		int DialogID() {return IDD_WINDPARAM;}
		ParamUIDesc *UIDesc();
		int UIDescLength();
	};



class WindClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) { return new WindObject;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_WIND_CLASS);}
	SClass_ID		SuperClassID() {return WSM_OBJECT_CLASS_ID; }
	Class_ID		ClassID() {return Class_ID(WINDOBJECT_CLASS_ID,8888);}
	const TCHAR* 	Category() {return TSTR(_T("Particle Extensions"));}
	};

static WindClassDesc windDesc;
ClassDesc* GetWindObjDesc() {return &windDesc;}


//--- WindMod -----------------------------------------------------

class WindMod;

class WindField : public ForceField {
	public:
		WindObject *obj;
		INode *node;
		Matrix3 tm,invtm;
		Interval tmValid;
		Point3 force;
		Interval fValid;
		int type;
		Point3 Force(TimeValue t,const Point3 &pos, const Point3 &vel,int index);
	};

class WindMod : public SimpleWSMMod {
	public:				
		WindField force;

		WindMod() {}
		WindMod(INode *node,WindObject *obj);		

		// From Animatable
		void GetClassName(TSTR& s) {s= GetString(IDS_RB_WINDMOD);}
		SClass_ID SuperClassID() {return WSM_CLASS_ID;}
		void DeleteThis() {delete this;}
		Class_ID ClassID() { return Class_ID(WINDMOD_CLASS_ID,8888);}
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_WINDBINDING);}

		void ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node);

		// From SimpleWSMMod		
		Interval GetValidity(TimeValue t);		
		Deformer& GetDeformer(TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat);
	};


class WindModClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 0; }
	void *			Create(BOOL loading = FALSE) {return new WindMod;}
	const TCHAR *	ClassName() { return GetString(IDS_RB_WINDMOD);}
	SClass_ID		SuperClassID() {return WSM_CLASS_ID;}
	Class_ID		ClassID() {return Class_ID(WINDMOD_CLASS_ID,8888);}
	const TCHAR* 	Category() {return _T("");}
	};

static WindModClassDesc windModDesc;
ClassDesc* GetWindModDesc() {return &windModDesc;}


//--- GravityObject Parameter map/block descriptors ------------------

#define PB_STRENGTH		0


//--- WindObject Parameter map/block descriptors ------------------

#define PB_TURBULENCE	1
#define PB_FREQUENCY	2
#define PB_SCALE		3
#define PB_DISPHEIGHT		4
#define PB_DISPINNER		5
#define PB_DISPOUTER		6
#define PB_DOWN				7
#define PB_TANGENT			8
#define PB_VORTEX			9
#define PB_FRICTION			10

//
//
// Parameters

static ParamUIDesc descParamWind[] = {
	// Strength
	ParamUIDesc(
		PB_STRENGTH,
		EDITTYPE_FLOAT,
		IDC_WIND_STRENGTH,IDC_WIND_STRENGTHSPIN,
		-9999999.0f, 9999999.0f,
		0.01f),


	// Turbulence
	ParamUIDesc(
		PB_TURBULENCE,
		EDITTYPE_FLOAT,
		IDC_WIND_TURB,IDC_WIND_TURBSPIN,
		0.0f, 9999999.0f,
		0.01f),

	// Frequency
	ParamUIDesc(
		PB_FREQUENCY,
		EDITTYPE_FLOAT,
		IDC_WIND_FREQ,IDC_WIND_FREQSPIN,
		0.0f, 9999999.0f,
		0.01f),

	// Scale
	ParamUIDesc(
		PB_SCALE,
		EDITTYPE_FLOAT,
		IDC_WIND_SCALE,IDC_WIND_SCALESPIN,
		0.0f, 9999999.0f,
		0.01f),

	// Display Height
	ParamUIDesc(
		PB_DISPHEIGHT,
		EDITTYPE_FLOAT,
		IDC_DISPHEIGHT,IDC_DISPHEIGHTSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),

	// Display inner
	ParamUIDesc(
		PB_DISPINNER,
		EDITTYPE_FLOAT,
		IDC_INNER,IDC_INNERSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),


	// Display outer
	ParamUIDesc(
		PB_DISPOUTER,
		EDITTYPE_FLOAT,
		IDC_OUTER,IDC_OUTERSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),

	// Down
	ParamUIDesc(
		PB_DOWN,
		EDITTYPE_FLOAT,
		IDC_DOWN,IDC_DOWNSPIN,
		-9999999.0f, 9999999.0f,
		0.01f),

	// Tangent
	ParamUIDesc(
		PB_TANGENT,
		EDITTYPE_FLOAT,
		IDC_TANGENT,IDC_TANGENTSPIN,
		-9999999.0f, 9999999.0f,
		0.01f),


	// Vortex
	ParamUIDesc(PB_VORTEX,TYPE_SINGLECHEKBOX,IDC_VORTEX),

	// FRICTION
	ParamUIDesc(
		PB_FRICTION,
		EDITTYPE_FLOAT,
		IDC_FRICTION,IDC_FRICTIONSPIN,
		-9999999.0f, 9999999.0f,
		0.1f),



	};
#define WINDPARAMDESC_LENGTH	11


ParamBlockDescID descWindVer0[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_FLOAT, NULL, TRUE, 1 },
	{ TYPE_FLOAT, NULL, TRUE, 2 },
	{ TYPE_FLOAT, NULL, TRUE, 3 },
	{ TYPE_FLOAT, NULL, TRUE, 4 },
	{ TYPE_FLOAT, NULL, TRUE, 5 },
	{ TYPE_FLOAT, NULL, TRUE, 6 },
	{ TYPE_FLOAT, NULL, TRUE, 7 },
	{ TYPE_FLOAT, NULL, TRUE, 8 },
	{ TYPE_INT, NULL, TRUE, 9 },
	{ TYPE_FLOAT, NULL, TRUE, 10 },
};

#define WINDPBLOCK_LENGTH	11

#define CURRENT_WINDVERSION	1


//--- ForceObject Methods ---------------------------------------------

void ForceObject::BeginEditParams(
		IObjParam *ip,ULONG flags,Animatable *prev)
	{
	SimpleWSMObject::BeginEditParams(ip,flags,prev);
	this->ip = ip;

	if (pmapParam) {		
		// Left over from last SinWave ceated
		pmapParam->SetParamBlock(pblock);
	} else {		
		hSot = ip->AddRollupPage( 
			hInstance, 
			MAKEINTRESOURCE(IDD_WINDGRAV_SOT),
			DefaultSOTProc,
			GetString(IDS_RB_SOT), 
			(LPARAM)ip,APPENDROLL_CLOSED);

		// Gotta make a new one.
		pmapParam = CreateCPParamMap(
			UIDesc(),UIDescLength(),
			pblock,
			ip,
			hInstance,
			MAKEINTRESOURCE(DialogID()),
			GetString(IDS_RB_PARAMETERS),
			0);
		}
	}

void ForceObject::EndEditParams(
		IObjParam *ip, ULONG flags,Animatable *next)
	{		
	SimpleWSMObject::EndEditParams(ip,flags,next);
	this->ip = NULL;

	if (flags&END_EDIT_REMOVEUI ) {		
		DestroyCPParamMap(pmapParam);
		ip->DeleteRollupPage(hSot);
		pmapParam = NULL;		
		}	
	}




void ForceObject::BuildMesh(TimeValue t)
	{
	ivalid = FOREVER;
	float height,inner,outer,length;
	
	pblock->GetValue(PB_DISPHEIGHT,t,height,ivalid);
	pblock->GetValue(PB_DISPINNER,t,inner,ivalid);
	pblock->GetValue(PB_DISPOUTER,t,outer,ivalid);


	if (inner > outer)
		{
		float temp;
		temp = inner;
		inner = outer;
		outer = temp;
		}
	length = inner;

	float u;
	#define NUM_SEGS	16

	mesh.setNumVerts(4*NUM_SEGS+1);
	mesh.setNumFaces(4*NUM_SEGS);
	int vct;
	vct = 0;

	for (int i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i, Point3((float)cos(u) * length, (float)sin(u) * length, 0.0f));
			vct++;
			}

	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+NUM_SEGS, Point3((float)cos(u) * length, (float)sin(u) * length,height));
			vct++;
			}

	length = outer;

	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+2*NUM_SEGS, Point3((float)cos(u) * length, (float)sin(u) * length, 0.0f));
			vct++;
			}		

	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+3*NUM_SEGS, Point3((float)cos(u) * length, (float)sin(u) * length, height));
			vct++;
			}		




	mesh.setVert(4*NUM_SEGS, Point3(0.0f, 0.0f, 0.0f));
		
	for (i=0; i<(4*NUM_SEGS); i++) {
			int i1 = i+1;
			if (i1%NUM_SEGS==0) i1 -= NUM_SEGS;
			mesh.faces[i].setEdgeVisFlags(1,0,0);
			mesh.faces[i].setSmGroup(1);
			mesh.faces[i].setVerts(i,i1,4*NUM_SEGS);
			}



	mesh.InvalidateGeomCache();
	}

class ForceObjCreateCallback : public CreateMouseCallBack {
	public:
		ForceObject *ob;	
		Point3 p0, p1;
//		IPoint2 sp0;
		IPoint2 sp0, sp1;
		int proc( ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat);
	};
/*
int ForceObjCreateCallback::proc(
		ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
	{
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:								
				sp0 = m;
				p0  = vpt->GetPointOnCP(m);
				mat.SetTrans(p0);
				break;
			case 1:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPLENGTH,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				if (msg==MOUSE_POINT) {
					if (Length(m-sp0)<3) return CREATE_ABORT;
					else return CREATE_STOP;
					}
				break;
			}
	} else {
		if (msg == MOUSE_ABORT)
			return CREATE_ABORT;
		}
	
	return TRUE;
	}

*/

int ForceObjCreateCallback::proc(
		ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
	{
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:								
				sp0 = m;
				p0  = vpt->GetPointOnCP(m);
				mat.SetTrans(p0);
				ob->pblock->SetValue(PB_DISPOUTER,0,0.0f);
				ob->pblock->SetValue(PB_DISPINNER,0,0.0f);
				ob->pblock->SetValue(PB_DISPHEIGHT,0,0.0f);
				break;
			case 1:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPINNER,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				break;
			case 2:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPOUTER,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				break;

			case 3:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPHEIGHT,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				if (msg==MOUSE_POINT) {
					if (Length(m-sp0)<3) return CREATE_ABORT;
					else return CREATE_STOP;
					}
				break;

			}
	} else {
		if (msg == MOUSE_ABORT)
			return CREATE_ABORT;
		}
	
	return TRUE;

}

static ForceObjCreateCallback forceCreateCB;

CreateMouseCallBack* ForceObject::GetCreateMouseCallBack()
	{
	forceCreateCB.ob = this;
	return &forceCreateCB;
	}

void ForceObject::InvalidateUI() 
	{
	if (pmapParam) pmapParam->Invalidate();
	}






//--- WindObject methods ---------------------------------------


WindObject::WindObject()
	{
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(descWindVer0, WINDPBLOCK_LENGTH, CURRENT_WINDVERSION));
	assert(pblock);	

	pblock->SetValue(PB_STRENGTH,0,1.0f);
	pblock->SetValue(PB_DOWN,0,1.0f);
	pblock->SetValue(PB_TANGENT,0,1.0f);

	pblock->SetValue(PB_SCALE,0,1.0f);
	pblock->SetValue(PB_DISPINNER,0,10.0f);
	pblock->SetValue(PB_DISPOUTER,0,30.0f);
	pblock->SetValue(PB_DISPHEIGHT,0,30.0f);
	pblock->SetValue(PB_FRICTION,0,0.0f);
	
	pblock->SetValue(PB_VORTEX,0,1);
	}

Modifier *WindObject::CreateWSMMod(INode *node)
	{
	return new WindMod(node,this);
	}

ForceField *WindObject::GetForceField(INode *node)
	{
	WindField *wf = new WindField;	
	wf->obj  = this;
	wf->node = node;
	wf->tmValid.SetEmpty();
	wf->fValid.SetEmpty();
//	wf->obj->pblock->GetValue(PB_TYPE,0,wf->type,FOREVER);
	return wf;
	}

RefTargetHandle WindObject::Clone(RemapDir& remap) 
	{
	WindObject* newob = new WindObject();
	newob->ReplaceReference(0,pblock->Clone(remap));
	return newob;
	}

ParamDimension *WindObject::GetParameterDim(int pbIndex) 
	{
	switch (pbIndex) {		
		case 0:
		default: return defaultDim;
		}
	}

TSTR WindObject::GetParameterName(int pbIndex) 
	{
	switch (pbIndex) {		
		case PB_STRENGTH: 	return GetString(IDS_RB_STRENGTH2);
//		case PB_DECAY:		return GetString(IDS_RB_DECAY);
		case PB_TURBULENCE:	return GetString(IDS_RB_TURBULENCE);
		case PB_FREQUENCY:	return GetString(IDS_RB_FREQUENCY);
		case PB_SCALE:		return GetString(IDS_RB_SCALE);
		case PB_DISPHEIGHT:	return TSTR(_T("Height"));
		case PB_DOWN:		return TSTR(_T("Down"));
		case PB_TANGENT:		return TSTR(_T("Tangent"));
		case PB_DISPINNER:	return TSTR(_T("Inner"));
		case PB_DISPOUTER:	return TSTR(_T("Outer"));
		case PB_FRICTION:	return TSTR(_T("Friction"));
		default: 			return TSTR(_T(""));
		}
	}

ParamUIDesc *WindObject::UIDesc()
	{
	return descParamWind;
	}

int WindObject::UIDescLength()
	{
	return WINDPARAMDESC_LENGTH;
	}
// This is an adjustment to forces to make the independent of time scale.
// They were previously dependent on the old 1200 ticks per second constant.
// Note that the constants are being squared because we are dealing with acceleration not velocity.
static float forceScaleFactor = float(1200*1200)/float(TIME_TICKSPERSEC*TIME_TICKSPERSEC);


//--- WindMod methods -----------------------------------------


WindMod::WindMod(INode *node,WindObject *obj)
	{	
	//MakeRefByID(FOREVER,SIMPWSMMOD_OBREF,obj);
	MakeRefByID(FOREVER,SIMPWSMMOD_NODEREF,node);	
	pblock = NULL;
	obRef = NULL;
	}

Interval WindMod::GetValidity(TimeValue t) 
	{
	if (nodeRef) {
		Interval valid = FOREVER;
		Matrix3 tm;
		float f;		
		((WindObject*)GetWSMObject(t))->pblock->GetValue(PB_STRENGTH,t,f,valid);		
		tm = nodeRef->GetObjectTM(t,&valid);
		return valid;
	} else {
		return FOREVER;
		}
	}

class WindDeformer : public Deformer {
	public:		
		Point3 Map(int i, Point3 p) {return p;}
	};
static WindDeformer wdeformer;

Deformer& WindMod::GetDeformer(
		TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat)
	{
	return wdeformer;
	}

RefTargetHandle WindMod::Clone(RemapDir& remap) 
	{
	WindMod *newob = new WindMod(nodeRef,(WindObject*)obRef);	
	newob->SimpleWSMModClone(this);
	return newob;
	}

void WindMod::ModifyObject(
		TimeValue t, ModContext &mc, ObjectState *os, INode *node)
	{
	ParticleObject *obj = GetParticleInterface(os->obj);
	if (obj) {
		force.obj  = (WindObject*)GetWSMObject(t);
		force.node = nodeRef;
		force.tmValid.SetEmpty();
		force.fValid.SetEmpty();
//		force.obj->pblock->GetValue(PB_TYPE,t,force.type,FOREVER);
		obj->ApplyForceField(&force);
		}
	}

static float RTurbulence(Point3 p,float freq)
	{
	return noise3(p*freq);
	}


Point3 WindField::Force(
		TimeValue t,const Point3 &pos, const Point3 &vel,int index)
	{	


	if (!tmValid.InInterval(t)) {
		tmValid = FOREVER;
		tm    = node->GetObjectTM(t,&tmValid);
		invtm = Inverse(tm);
		}
	
	Point3 p, v ;
	float height, inner, outer, at,downforce,hdist,hforce;

	obj->pblock->GetValue(PB_DISPHEIGHT,t,height,FOREVER);
	obj->pblock->GetValue(PB_DISPINNER,t,inner,FOREVER);
	obj->pblock->GetValue(PB_DISPOUTER,t,outer,FOREVER);

	// Transform the point and velocity into our space
	p = pos * invtm; 
	v = VectorTransform(invtm,vel);

	// Compute the point of intersection
	Point3 tp(0.0f,0.0f,0.0f);

	if ((p.z<0.0) || (p.z > height)){
		return tp;
	} else {
		downforce = 1.0f - (p.z/height);
		}

	p.z = 0;
	hdist = Length(p);

	if (hdist > outer)
		return tp;
	else if (hdist<inner)
		hforce = 1.0f;
	else hforce =  1.0f - (hdist-inner)/(outer-inner);

	downforce = 1.0f - (downforce * downforce * downforce * downforce);
	hforce = (hforce * hforce * hforce * hforce);

	float strength,  turb,down,tangent;
	obj->pblock->GetValue(PB_TURBULENCE,t,turb,FOREVER);

//	if (!fValid.InInterval(t)  ) {
	Point3 Friction(0.0f,0.0f,0.0f);
	if (1  ) {
		fValid = FOREVER;		
		if (!tmValid.InInterval(t)) {
			tmValid = FOREVER;
			tm = node->GetObjectTM(t,&tmValid);
			}
		fValid &= tmValid;
		obj->pblock->GetValue(PB_STRENGTH,t,strength,fValid);
		obj->pblock->GetValue(PB_DOWN,t,down,fValid);
		obj->pblock->GetValue(PB_TANGENT,t,tangent,fValid);
		int vortex;
		obj->pblock->GetValue(PB_VORTEX,t,vortex,fValid);
		float friction;
		obj->pblock->GetValue(PB_FRICTION,t,friction,fValid);


		float angle = DotProd(Normalize(p),Normalize(vel)) ;

//inward
		if (vortex)
			force = VectorTransform(tm,Normalize(p) * hforce * -strength * downforce);
			else force = VectorTransform(tm,Normalize(p) * hforce * -strength );

		if (angle < 0.0f)
			{
			Friction = force * -friction * (-angle) * 0.0001f * forceScaleFactor;
			}

///down
		
		force += Normalize(tm.GetRow(2)) * -down;// * - downforce;
		Point3 UpVec(0.0f,0.0f,1.0f);

		force += CrossProd(Normalize(p),UpVec) * tangent;


		force *= 0.0001f * forceScaleFactor;
		}	
	if (turb!=0.0f) {
		float freq, scale;
		Point3 tf, pt = pos-tm.GetTrans(), p;
		obj->pblock->GetValue(PB_FREQUENCY,t,freq,FOREVER);
		obj->pblock->GetValue(PB_SCALE,t,scale,FOREVER);
		freq *= 0.01f;
		turb *= 0.0001f * forceScaleFactor;

		p    = pt;
		p.x  = freq * float(t);
		tf.x = RTurbulence(p,scale);
		p    = pt;
		p.y  = freq * float(t);
		tf.y = RTurbulence(p,scale);
		p    = pt;
		p.z  = freq * float(t);
		tf.z = RTurbulence(p,scale);

		return (force ) + (turb*tf) + Friction;
	} else {
		return force +Friction;
		}
	}

