/**********************************************************************
 *<
	FILE: deflect.cpp

	DESCRIPTION: A simple deflector object for particles

	CREATED BY: Rolf Berteig

	HISTORY: 10-31-95

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/
#include "mods.h"
#include "iparamm.h"
#include "simpmod.h"
#include "simpobj.h"
#include "texutil.h"

#define PW_DEFLECTOBJECT_CLASS_ID Class_ID(0x6b71279, 0x357c66)
#define PW_DEFLECTMOD_CLASS_ID	Class_ID(0x4550734c, 0x3e913abf)


class DeflectObject : public SimpleWSMObject {	
	public:		
		static IParamMap *pmapParam;
		static IObjParam *ip;
		static HWND hSot;
					
		DeflectObject();

		// From Animatable		
		void DeleteThis() {delete this;}		
		void BeginEditParams( IObjParam  *ip, ULONG flags,Animatable *prev);
		void EndEditParams( IObjParam *ip, ULONG flags,Animatable *next);		
		Class_ID ClassID() {return PW_DEFLECTOBJECT_CLASS_ID;}		
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_DEFLECTOR);}
				
		// from object		
		CreateMouseCallBack* GetCreateMouseCallBack();		
		
		// From SimpleWSMObject		
		void InvalidateUI();		
		void BuildMesh(TimeValue t);
		ParamDimension *GetParameterDim(int pbIndex);
		TSTR GetParameterName(int pbIndex);		
		
		// From WSMObject
		Modifier *CreateWSMMod(INode *node);		
	};

//--- ClassDescriptor and class vars ---------------------------------

IObjParam *DeflectObject::ip        = NULL;
IParamMap *DeflectObject::pmapParam = NULL;
HWND       DeflectObject::hSot      = NULL;

class DeflectorClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) { return new DeflectObject;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_DEFLECTOR);}
	SClass_ID		SuperClassID() {return WSM_OBJECT_CLASS_ID; }
	Class_ID		ClassID() {return PW_DEFLECTOBJECT_CLASS_ID;}
	const TCHAR* 	Category() {return _T("Particle Extensions");}
	};

static DeflectorClassDesc deflectDesc;
ClassDesc* GetDeflectObjDesc() {return &deflectDesc;}

//--- DeflectMod -----------------------------------------------------

class DeflectorField : public CollisionObject {
	public:		
		DeflectObject *obj;
		INode *node;
		Matrix3 tm, invtm;
		Interval tmValid;		
		BOOL CheckCollision(TimeValue t,Point3 &pos, Point3 &vel, float dt,int index, float *ct=NULL, BOOL UpdatePastCollide=TRUE);

//		BOOL CheckCollision(TimeValue t,Point3 &pos, Point3 &vel, float dt, int pindex,BOOL UpdatePastCollide);
		Object *GetSWObject();

	};

class DeflectMod : public SimpleWSMMod {
	public:				
		DeflectorField deflect;

		DeflectMod() {}
		DeflectMod(INode *node,DeflectObject *obj);		

		// From Animatable
		void GetClassName(TSTR& s) {s= GetString(IDS_RB_DEFLECTMOD);}
		SClass_ID SuperClassID() {return WSM_CLASS_ID;}
		void DeleteThis() {delete this;}
		Class_ID ClassID() { return PW_DEFLECTMOD_CLASS_ID;}
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_DEFLECTORBINDING);}

		void ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node);

		// From SimpleWSMMod		
		Interval GetValidity(TimeValue t);		
		Deformer& GetDeformer(TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat);
	};

//--- ClassDescriptor and class vars ---------------------------------

class DeflectorModClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) { return new DeflectMod;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_DEFLECTMOD);}
	SClass_ID		SuperClassID() {return WSM_CLASS_ID; }
	Class_ID		ClassID() {return PW_DEFLECTMOD_CLASS_ID;}
	const TCHAR* 	Category() {return _T("Particle Extension");}
	};

static DeflectorModClassDesc deflectModDesc;
ClassDesc* GetDeflectModDesc() {return &deflectModDesc;}

//--- DeflectObject Parameter map/block descriptors ------------------

#define PB_BOUNCE	0
#define PB_WIDTH	1
#define PB_HEIGHT	2

#define PB_FRICTION	3
#define PB_SAMPLERATE 4
#define PB_ENERGY 5

static ParamUIDesc descParam[] = {
	// Bounce
	ParamUIDesc(
		PB_BOUNCE,
		EDITTYPE_FLOAT,
		IDC_DEFLECT_BOUNCE,IDC_DEFLECT_BOUNCESPIN,
		0.0f, 9999999.0f,
		0.01f),
	
	// Width
	ParamUIDesc(
		PB_WIDTH,
		EDITTYPE_FLOAT,
		IDC_DEFLECT_WIDTH,IDC_DEFLECT_WIDTHSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),
	
	// height
	ParamUIDesc(
		PB_HEIGHT,
		EDITTYPE_FLOAT,
		IDC_DEFLECT_HEIGHT,IDC_DEFLECT_HEIGHTSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),

	// Friction
	ParamUIDesc(
		PB_FRICTION,
		EDITTYPE_FLOAT,
		IDC_DEFLECT_FRICTION,IDC_DEFLECT_FRICTIONSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),

	// Sample rate
	ParamUIDesc(
		PB_SAMPLERATE,
		EDITTYPE_POS_INT,
		IDC_DEFLECT_SAMPLERATE2,IDC_DEFLECT_SAMPLERATESPIN2,
		0.0f, 500.0f,
		SPIN_AUTOSCALE),

	// Energy Transfer
	ParamUIDesc(
		PB_ENERGY,
		EDITTYPE_FLOAT,
		IDC_DEFLECT_ENERGY,IDC_DEFLECT_ENERGYSPIN,
		-9999999.0f, 9999999.0f,
		SPIN_AUTOSCALE),


	};

#define PARAMDESC_LENGTH	6

ParamBlockDescID descVer0[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_FLOAT, NULL, TRUE, 1 },
	{ TYPE_FLOAT, NULL, TRUE, 2 },
	{ TYPE_FLOAT, NULL, TRUE, 3 },
	{ TYPE_INT,   NULL, TRUE, 4 },
	{ TYPE_FLOAT, NULL, TRUE, 5 },

};
#define PBLOCK_LENGTH	6

#define CURRENT_VERSION	0


//-- GenDlgProc ------------------------------------------------

class GenDlgProc : public ParamMapUserDlgProc {
	public:
		HWND TempHWND;

		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void DeleteThis() {delete this;}
		void SetupFrictionText(HWND hWnd,IParamBlock *pblock,TimeValue t);
		void SetupBounceText(HWND hWnd,IParamBlock *pblock,TimeValue t);
		void SetupSampleRateText(HWND hWnd,IParamBlock *pblock,TimeValue t);
		void SetupEnergyText(HWND hWnd,IParamBlock *pblock,TimeValue t);
	};


void GenDlgProc::SetupFrictionText(HWND hWnd,IParamBlock *pblock,TimeValue t)
	{
	float friction;
	pblock->GetValue(PB_FRICTION,t,friction,FOREVER);
	TSTR buf;
	if (friction < 0.25)
		buf.printf(_T("Super Sticky"));
	else if (friction < 0.9)
		buf.printf(_T("Sticky"));
	else if (friction < 1.2)
		buf.printf(_T("No Friction"));
	else if (friction < 2.0)
		buf.printf(_T("Slick"));
	else 
		buf.printf(_T("Super Slick"));

	SetWindowText(GetDlgItem(hWnd,IDC_FRICTION_TEXT),buf);
	}

void GenDlgProc::SetupBounceText(HWND hWnd,IParamBlock *pblock,TimeValue t)
	{
	float bounce;
	pblock->GetValue(PB_BOUNCE,t,bounce,FOREVER);
	TSTR buf;
	if (bounce < 0.2)
		buf.printf(_T("Hard as a rock"));
	else if (bounce < 0.5)
		buf.printf(_T("A little bounce"));
	else if (bounce < 1.5)
		buf.printf(_T("Alot of bounce"));
	else 
		buf.printf(_T("Super Ball Bounce"));

	SetWindowText(GetDlgItem(hWnd,IDC_BOUNCE_TEXT),buf);
	}

void GenDlgProc::SetupSampleRateText(HWND hWnd,IParamBlock *pblock,TimeValue t)
	{
	int sample;
	pblock->GetValue(PB_SAMPLERATE,t,sample,FOREVER);
	TSTR buf;
	if (sample < 5)
		buf.printf(_T("Use for slow/no motion"));
	else if (sample < 15)
		buf.printf(_T("Average sample rate"));
	else 
		buf.printf(_T("For super fast movement"));

	SetWindowText(GetDlgItem(hWnd,IDC_SAMPLE_TEXT),buf);
	}

void GenDlgProc::SetupEnergyText(HWND hWnd,IParamBlock *pblock,TimeValue t)
	{
	float sample;
	pblock->GetValue(PB_ENERGY,t,sample,FOREVER);
	TSTR buf;
	if (sample = 0.0f)
		buf.printf(_T("No energy transfer"));
	else if (sample > 0.0f)
		buf.printf(_T("Positive energy transfer"));
	else 
		buf.printf(_T("Negative energy transfer"));

	SetWindowText(GetDlgItem(hWnd,IDC_ENERGY_TEXT),buf);
	}





BOOL GenDlgProc::DlgProc(
		TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam)
	{
	switch (msg) {
		case WM_INITDIALOG:
			SetupFrictionText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
			SetupBounceText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
			SetupSampleRateText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
			SetupEnergyText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
			break;			
			
		case CC_SPINNER_CHANGE:
			switch (LOWORD(wParam)) {
				case IDC_DEFLECT_FRICTIONSPIN:
						SetupFrictionText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
						break;
				case IDC_DEFLECT_BOUNCESPIN:
						SetupBounceText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
						break;
				case IDC_DEFLECT_SAMPLERATESPIN2:
						SetupSampleRateText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
						break;
				case IDC_DEFLECT_ENERGY:
						SetupEnergyText(hWnd,(IParamBlock*)map->GetParamBlock(),t);
						break;
				}
			break;

		}
	return FALSE;
	}



//--- Deflect object methods -----------------------------------------

DeflectObject::DeflectObject()
	{
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(descVer0, PBLOCK_LENGTH, CURRENT_VERSION));
	assert(pblock);	

	pblock->SetValue(PB_BOUNCE,0,1.0f);
	pblock->SetValue(PB_FRICTION,0,1.0f);
	pblock->SetValue(PB_SAMPLERATE,0,10);
	pblock->SetValue(PB_ENERGY,0,1.0f);
	}

Modifier *DeflectObject::CreateWSMMod(INode *node)
	{
	return new DeflectMod(node,this);
	}

RefTargetHandle DeflectObject::Clone(RemapDir& remap) 
	{
	DeflectObject* newob = new DeflectObject();	
	newob->ReplaceReference(0,pblock->Clone(remap));
	return newob;
	}

void DeflectObject::BeginEditParams(
		IObjParam *ip,ULONG flags,Animatable *prev)
	{
	SimpleWSMObject::BeginEditParams(ip,flags,prev);
	this->ip = ip;

	if (pmapParam) {		
		// Left over
		pmapParam->SetParamBlock(pblock);
	} else {		
		hSot = ip->AddRollupPage( 
			hInstance, 
			MAKEINTRESOURCE(IDD_WINDRAIN_SOT),
			DefaultSOTProc,
			GetString(IDS_RB_SOT), 
			(LPARAM)ip,APPENDROLL_CLOSED);

		// Gotta make a new one.
		pmapParam = CreateCPParamMap(
			descParam,PARAMDESC_LENGTH,
			pblock,
			ip,
			hInstance,
			MAKEINTRESOURCE(IDD_DEFLECTORPARAM),
			GetString(IDS_RB_PARAMETERS),
			0);
		}
	pmapParam->SetUserDlgProc(new GenDlgProc());

	}

void DeflectObject::EndEditParams(
		IObjParam *ip, ULONG flags,Animatable *next)
	{		
	SimpleWSMObject::EndEditParams(ip,flags,next);
	this->ip = NULL;

	if (flags&END_EDIT_REMOVEUI ) {		
		DestroyCPParamMap(pmapParam);
		ip->DeleteRollupPage(hSot);
		pmapParam = NULL;		
		}	
	}


void DeflectObject::BuildMesh(TimeValue t)
	{
	float width, height;
	ivalid = FOREVER;
	pblock->GetValue(PB_WIDTH,t,width,ivalid);
	pblock->GetValue(PB_HEIGHT,t,height,ivalid);
	width  *= 0.5f;
	height *= 0.5f;

	mesh.setNumVerts(4);
	mesh.setNumFaces(2);
	mesh.setVert(0, Point3(-width,-height, 0.0f));
	mesh.setVert(1, Point3( width,-height, 0.0f));
	mesh.setVert(2, Point3( width, height, 0.0f));
	mesh.setVert(3, Point3(-width, height, 0.0f));
	
	mesh.faces[0].setEdgeVisFlags(1,0,1);
	mesh.faces[0].setSmGroup(1);
	mesh.faces[0].setVerts(0,1,3);

	mesh.faces[1].setEdgeVisFlags(1,1,0);
	mesh.faces[1].setSmGroup(1);
	mesh.faces[1].setVerts(1,2,3);	
	mesh.InvalidateGeomCache();	
	}


class DeflectObjCreateCallback : public CreateMouseCallBack {
	public:
		DeflectObject *ob;	
		Point3 p0, p1;
		IPoint2 sp0, sp1;
		int proc( ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat);
	};

int DeflectObjCreateCallback::proc(
		ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
	{
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:
				sp0 = m;
				p0  = vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE);
				mat.SetTrans(p0);
				ob->pblock->SetValue(PB_WIDTH,0,0.01f);
				ob->pblock->SetValue(PB_HEIGHT,0,0.01f);
				ob->pmapParam->Invalidate();
				break;

			case 1: {
				mat.IdentityMatrix();
				sp1 = m;
				p1  = vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE);
				Point3 center = (p0+p1)/float(2);
				mat.SetTrans(center);
				ob->pblock->SetValue(PB_WIDTH,0,(float)fabs(p1.x-p0.x));
				ob->pblock->SetValue(PB_HEIGHT,0,(float)fabs(p1.y-p0.y));
				ob->pmapParam->Invalidate();

				if (msg==MOUSE_POINT) {
					if (Length(m-sp0)<3) {						
						return CREATE_ABORT;
					} else {
						return CREATE_STOP;
						}
					}
				break;
				}

			}
	} else {
		if (msg == MOUSE_ABORT)
			return CREATE_ABORT;
		}	
	return TRUE;
	}
static DeflectObjCreateCallback deflectCreateCB;

CreateMouseCallBack* DeflectObject::GetCreateMouseCallBack()
	{
	deflectCreateCB.ob = this;
	return &deflectCreateCB;
	}

void DeflectObject::InvalidateUI() 
	{
	if (pmapParam) pmapParam->Invalidate();
	}

ParamDimension *DeflectObject::GetParameterDim(int pbIndex) 
	{
	switch (pbIndex) {		
		case 0:
		default: return defaultDim;
		}
	}

TSTR DeflectObject::GetParameterName(int pbIndex) 
	{
	switch (pbIndex) {				
		case PB_BOUNCE: 	return GetString(IDS_RB_BOUNCE);
		case PB_WIDTH:		return GetString(IDS_RB_WIDTH);
		case PB_HEIGHT:		return GetString(IDS_RB_HEIGHT);
		case PB_FRICTION:	return GetString(IDS_RB_FRICTION);
		case PB_ENERGY:		return GetString(IDS_RB_ENERGY);
		case PB_SAMPLERATE:	return GetString(IDS_RB_SAMPLERATE);
		default: 			return TSTR(_T(""));
		}
	}


//--- DeflectMod methods -----------------------------------------------

DeflectMod::DeflectMod(INode *node,DeflectObject *obj)
	{	
	//MakeRefByID(FOREVER,SIMPWSMMOD_OBREF,obj);
	MakeRefByID(FOREVER,SIMPWSMMOD_NODEREF,node);	
	pblock = NULL;
	obRef = NULL;
	}

Interval DeflectMod::GetValidity(TimeValue t) 
	{
	if (obRef && nodeRef) {
		Interval valid = FOREVER;
		Matrix3 tm;
		float f;		
		DeflectObject *obj = (DeflectObject*)GetWSMObject(t);
		obj->pblock->GetValue(PB_BOUNCE,t,f,valid);
		obj->pblock->GetValue(PB_WIDTH,t,f,valid);
		obj->pblock->GetValue(PB_HEIGHT,t,f,valid);
		tm = nodeRef->GetObjectTM(t,&valid);
		return valid;
	} else {
		return FOREVER;
		}
	}

class DeflectDeformer : public Deformer {
	public:		
		Point3 Map(int i, Point3 p) {return p;}
	};
static DeflectDeformer ddeformer;

Deformer& DeflectMod::GetDeformer(
		TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat)
	{
	return ddeformer;
	}

RefTargetHandle DeflectMod::Clone(RemapDir& remap) 
	{
	DeflectMod *newob = new DeflectMod(nodeRef,(DeflectObject*)obRef);	
	newob->SimpleWSMModClone(this);
	return newob;
	}


void DeflectMod::ModifyObject(
		TimeValue t, ModContext &mc, ObjectState *os, INode *node)
	{
	ParticleObject *obj = GetParticleInterface(os->obj);
	if (obj) {
		deflect.obj  = (DeflectObject*)GetWSMObject(t);
		deflect.node = nodeRef;
		deflect.tmValid.SetEmpty();		
		obj->ApplyCollisionObject(&deflect);
		}
	}

Object *DeflectorField::GetSWObject()
{ return obj;
}



BOOL DeflectorField::CheckCollision(
		TimeValue t,Point3 &pos, Point3 &vel, float dt,int index, float *ct, BOOL UpdatePastCollide)

//		TimeValue t,Point3 &pos, Point3 &vel, float dt, int pindex,BOOL UpdatePastCollide)
	{
	if (!tmValid.InInterval(t)) {
		tmValid = FOREVER;
		tm    = node->GetObjectTM(t,&tmValid);
		invtm = Inverse(tm);
		}
	
	Point3 p, v, ph,tempv1,tempv2,tempv,NewVel,zero;
	float width, height, at, bounce,friction,energy;
	int NoVel = 0,samplerate;

	obj->pblock->GetValue(PB_WIDTH,t,width,FOREVER);
	obj->pblock->GetValue(PB_HEIGHT,t,height,FOREVER);
	obj->pblock->GetValue(PB_BOUNCE,t,bounce,FOREVER);
	obj->pblock->GetValue(PB_SAMPLERATE,t,samplerate,FOREVER);
	obj->pblock->GetValue(PB_FRICTION,t,friction,FOREVER);
	obj->pblock->GetValue(PB_ENERGY,t,energy,FOREVER);
	width  *= 0.5f;
	height *= 0.5f;

	// Transform the point and velocity into our space
	p = pos * invtm; 
	v = VectorTransform(invtm,vel);

	samplerate = 0;

	if (samplerate == 0)
		{
  
		// Compute the point of intersection
		if (fabs(p.z)<0.001) {
			// We're sitting on the ground plane, just take the Z component
			// out of the velocity
			v.z = 0.0f;
			at  = 0.0f;		
		} else {
			if (v.z==0.0) 
				{
				return FALSE;
				}
			at = -p.z/v.z;
			if (at<0.0f || at>dt) 
				{
				return FALSE;
				}
			}
		ph = p + at*v;

		// See if the point is within our range
		if (ph.x<-width || ph.x>width || ph.y<-height || ph.y>height) 			
				{
				return FALSE;
				}

	// Remove the part of dt we used to get to the collision point
		dt -= at;

	// Reflect the velocity about the XY plane and attenuate with the bounce factor
		v.z = -v.z;
		if (bounce!=1.0f) v.z = v.z*bounce;

		if (friction == 0.0f)
			{
			v.x = 0.0f;
			v.y = 0.0f;
			}
		else if (friction != 1.0f)
			{
			v.x *= friction;
			v.y *= friction;
			}
	// Use up the rest of the dt

		ph += v * dt;

	// Tranform back into world space.
		pos = ph * tm;


		vel = VectorTransform(tm,v);
	
		return TRUE;
		}

//use sample rate
	else
		{
		TimeValue offset_time,starttime,time;
		TimeValue direction;
		int change = 0, above = 0;
		offset_time = (int) dt;
		starttime = (int)(t-dt);

		tmValid = FOREVER;
		tm    = node->GetObjectTM(starttime,&tmValid);
		invtm = Inverse(tm);

	// Transform the point and velocity into our space
		p = pos * invtm; 
		v = VectorTransform(invtm,vel);

		if (p.z >= 0.0f)
			{
			above = 1;
			}
			else
			{
			above = 0;
			}
		direction = 1;
		time = 0;

		for (int i= 0; i<samplerate; i++)
			{
			offset_time /= 2;
			time += direction*offset_time;
			tm    = node->GetObjectTM(starttime+time,&tmValid);
			invtm = Inverse(tm);

	// Transform the point and velocity into our space
			p = pos * invtm; 
			v = VectorTransform(invtm,vel);

			p += v*(float)time;


			if (above)
				{

				if (p.z < 0.0f)
					{
					change = 1;
					direction =  -1;
					}
					else
					{
					direction = 1;
					}
				}
			else{
				if (p.z >= 0.0f)
					{
					change = 1;
					direction = -1;
					}
					else
					{
					direction = 1;
					}
				}

			}




		if (!change) 
			{
//Last check final pos
			tm    = node->GetObjectTM(t,&tmValid);
			invtm = Inverse(tm);

	// Transform the point and velocity into our space
			p = pos * invtm; 
			v = VectorTransform(invtm,vel);
			time = (int) dt;
			p += v*(float)time;


			if (above)
				{

				if (p.z < 0.0f)
					{
					change = 1;
					direction =  -1;
					}
					else
					{
					direction = 1;
					}
				}
			else{
				if (p.z >= 0.0f)
					{
					change = 1;
					direction = -1;
					}
					else
					{
					direction = 1;
					}
				}

			if (!change)
				return FALSE;
			}
		
		tm    = node->GetObjectTM(starttime+time,&tmValid);
		invtm = Inverse(tm);

// Transform the point and velocity into our space
		p = pos * invtm; 
		v = VectorTransform(invtm,vel);
//check if in box
		ph = p + v*(float)time;

		// See if the point is within our range
		if (ph.x<-width || ph.x>width || ph.y<-height || ph.y>height) 			
				{
				return FALSE;
				}

//check for non moveing particles

/*
		if (Length(v) ==0.0f)
				{
				tm    = node->GetObjectTM(t,&tmValid);
				invtm = Inverse(tm);
				if (above)
					ph.z = 0.01f;
					else ph.z = -0.01f;

				pos = ph * tm;
//need to add transfer energy
//add slide
				vel = VectorTransform(tm,v);
				return TRUE;
				}
*/
  		at = (float)time;
		ph = p + at*v;


	// Remove the part of dt we used to get to the collision point
		dt -= at;



	// Reflect the velocity about the XY plane and attenuate with the bounce factor
		v.z = -v.z;
		if (bounce!=1.0f) v.z = v.z*bounce;

	//Apply friction
		if (friction == 0.0f)
			{
			v.x = 0.0f;
			v.y = 0.0f;
			}
		else if (friction != 1.0f)
			{
			v.x *= friction;
			v.y *= friction;
			}

	// Use up the rest of the dt
		ph += v * dt;

		
//add energy transfer





// Tranform back into world space.
		pos = ph * tm;
		vel = VectorTransform(tm,v);

//store initial pos
		Point3 InitialPos;
		InitialPos = pos;


//check for pass through one last time
		tm    = node->GetObjectTM(t,&tmValid);
		invtm = Inverse(tm);

// Transform the point and velocity into our space
		p = pos * invtm; 

		if ((above) && (p.z <= 0.0f))
			{
			p.z = 0.01f;
			}
		else if ((!above)&&(p.z >= 0.0f))
			{
			p.z = -0.01f;
			}

//check against current pos
//create new vector and add to current velocity

		pos = p * tm;

//store final pos
		Point3 FinalPos,EnergyVel;
		FinalPos = pos;

		EnergyVel=(FinalPos-InitialPos)*energy/dt;

		vel += EnergyVel;

	
		return TRUE;
			
		}
	}
