/**********************************************************************
 *<
	FILE: gravity.cpp

	DESCRIPTION:  Gravity and Wind World Space Modifier

	CREATED BY: Rolf Berteig

	HISTORY: 10-30-95

 *>	Copyright (c) 1994, All Rights Reserved.
 **********************************************************************/
#include "mods.h"
#include "iparamm.h"
#include "simpmod.h"
#include "simpobj.h"
#include "texutil.h"

#define FORCE_PLANAR	0
#define FORCE_SPHERICAL	1


#define PBLK		0
#define CUSTNODE 	1


class UniPickOperand;


class ForceObject : public SimpleWSMObject {	
	public:		
		static IParamMap *pmapParam;
		static IObjParam *ip;
		static HWND hSot;
		static HWND hParams;

					
					
		ForceObject() {}
		BOOL SupportsDynamics() {return TRUE;}

		// From Animatable		
		void DeleteThis() {delete this;}		
				
		// from object		
		CreateMouseCallBack* GetCreateMouseCallBack();		
		
		// From SimpleWSMObject		
		void InvalidateUI();		
		void BuildMesh(TimeValue t);

		virtual int DialogID()=0;
		virtual ParamUIDesc *UIDesc()=0;
		virtual int UIDescLength()=0;


	};
IObjParam *ForceObject::ip        = NULL;
IParamMap *ForceObject::pmapParam = NULL;
HWND       ForceObject::hSot      = NULL;
HWND       ForceObject::hParams   = NULL;


class WindObject : public ForceObject {	
	public:									
		WindObject();		


		INode *custnode;
		TSTR custname;

//		static BOOL creating;
		static UniPickOperand pickCB;

		void ShowName();

		// From Animatable		
		void DeleteThis() {delete this;}		
		Class_ID ClassID() {return Class_ID(WINDOBJECT_CLASS_ID,4599);}		
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_WIND);}
						
		// From WSMObject
		Modifier *CreateWSMMod(INode *node);
		ForceField *GetForceField(INode *node);
		
		// From SimpleWSMObject				
		ParamDimension *GetParameterDim(int pbIndex);
		TSTR GetParameterName(int pbIndex);

		int DialogID() {return IDD_WINDPARAM;}
		ParamUIDesc *UIDesc();
		int UIDescLength();

		int NumRefs() {return 2;}
		RefTargetHandle GetReference(int i);
		void SetReference(int i, RefTargetHandle rtarg);		
		RefResult NotifyRefChanged( Interval changeInt,RefTargetHandle hTarget, 
		   PartID& partID, RefMessage message);

		void BeginEditParams( IObjParam  *ip, ULONG flags,Animatable *prev );
		void EndEditParams( IObjParam *ip, ULONG flags,Animatable *next);		
	};


class UniPickOperand : 
		public PickModeCallback,
		public PickNodeCallback {
	public:		
		WindObject *po;
		
		UniPickOperand() {po=NULL;}

		BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);
		BOOL Pick(IObjParam *ip,ViewExp *vpt);

		void EnterMode(IObjParam *ip);
		void ExitMode(IObjParam *ip);

		BOOL RightClick(IObjParam *ip, ViewExp *vpt) { return TRUE; }
		BOOL Filter(INode *node);
		
		PickNodeCallback *GetFilter() {return this;}
	};


BOOL UniPickOperand::Filter(INode *node)
	{
	if (node)
		return TRUE;
	else return FALSE;
	}

BOOL UniPickOperand::HitTest(
		IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags)
	{	
	INode *node = ip->PickNode(hWnd,m,this);
	
	if (node)
		return TRUE;
	else return FALSE;
	}


BOOL UniPickOperand::Pick(IObjParam *ip,ViewExp *vpt)
	{BOOL groupflag=0;
	INode *node = vpt->GetClosestHit();
	assert(node);
	INodeTab nodes;
/*	if (node->IsGroupMember()) 
	{ groupflag=1;
	  while (node->IsGroupMember()) node=node->GetParentNode();
	}
	int subtree=0;
	if (groupflag) MakeGroupNodeList(node,&nodes,subtree,ip->GetTime());
	else{ nodes.SetCount(1);nodes[0]=node;}
*/
	ip->FlashNodes(&nodes);
	if (po->custnode) po->ReplaceReference(CUSTNODE,node,TRUE);
	else po->MakeRefByID(FOREVER,CUSTNODE,node);	
	po->custname = TSTR(node->GetName());
	// Automatically check show result and do one update
	po->ShowName();	
	return TRUE;
	}

void UniPickOperand::EnterMode(IObjParam *ip)
	{
	ICustButton *iBut;
	iBut=GetICustButton(GetDlgItem(po->hParams,IDC_MTRACK_PICK));
	if (iBut) iBut->SetCheck(TRUE);
	ReleaseICustButton(iBut);
	}

void UniPickOperand::ExitMode(IObjParam *ip)
	{
	ICustButton *iBut;
	iBut=GetICustButton(GetDlgItem(po->hParams,IDC_MTRACK_PICK));
	if (iBut) iBut->SetCheck(FALSE);
	ReleaseICustButton(iBut);
	}


class UniObjectDlgProc : public ParamMapUserDlgProc {
	public:
		WindObject *po;

		UniObjectDlgProc(WindObject *p) {po=p;}
		BOOL DlgProc(TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam);
		void Update(TimeValue t);
		void DeleteThis() {delete this;}
	};
void UniObjectDlgProc::Update(TimeValue t)
{	po->ShowName();
}

BOOL UniObjectDlgProc::DlgProc(
		TimeValue t,IParamMap *map,HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam)
{	switch (msg) {
		case WM_INITDIALOG: {
			ICustButton *iBut = GetICustButton(GetDlgItem(hWnd,IDC_MTRACK_PICK));
			iBut->SetType(CBT_CHECK);
			iBut->SetHighlightColor(GREEN_WASH);
			ReleaseICustButton(iBut);
			po->hParams=hWnd;
			Update(t);
			return FALSE;	// stop default keyboard focus - DB 2/27  
			}
		case WM_COMMAND:
			switch (LOWORD(wParam)) 
			{    case IDC_MTRACK_PICK:
					po->pickCB.po = po;						
					po->ip->SetPickMode(&po->pickCB);
					break;
			

			}
			break;	
		}
	return TRUE;
	}



class WindClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) { return new WindObject;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_WIND_CLASS);}
	SClass_ID		SuperClassID() {return WSM_OBJECT_CLASS_ID; }
	Class_ID		ClassID() {return Class_ID(WINDOBJECT_CLASS_ID,4599);}
	const TCHAR* 	Category() {return TSTR(_T("Particle Extensions"));}
	};

static WindClassDesc windDesc;
ClassDesc* GetWindObjDesc() {return &windDesc;}

UniPickOperand WindObject::pickCB;

//--- WindMod -----------------------------------------------------

class WindMod;

class WindField : public ForceField {
	public:
		WindObject *obj;
		INode *node;
		Tab<Point3> plist;
		Matrix3 tm,invtm;
		Interval tmValid;
		Point3 force;
		Interval fValid;
		int type;
		Point3 Force(TimeValue t,const Point3 &pos, const Point3 &vel,int index);
	};

class WindMod : public SimpleWSMMod {
	public:				
		WindField force;

		WindMod() {}
		WindMod(INode *node,WindObject *obj);		

		// From Animatable
		void GetClassName(TSTR& s) {s= GetString(IDS_RB_WINDMOD);}
		SClass_ID SuperClassID() {return WSM_CLASS_ID;}
		void DeleteThis() {delete this;}
		Class_ID ClassID() { return Class_ID(WINDMOD_CLASS_ID,4599);}
		RefTargetHandle Clone(RemapDir& remap = NoRemap());
		TCHAR *GetObjectName() {return GetString(IDS_RB_WINDBINDING);}

		void ModifyObject(TimeValue t, ModContext &mc, ObjectState *os, INode *node);

		// From SimpleWSMMod		
		Interval GetValidity(TimeValue t);		
		Deformer& GetDeformer(TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat);
	};


class WindModClassDesc:public ClassDesc {
	public:
	int 			IsPublic() { return 0; }
	void *			Create(BOOL loading = FALSE) {return new WindMod;}
	const TCHAR *	ClassName() { return GetString(IDS_RB_WINDMOD);}
	SClass_ID		SuperClassID() {return WSM_CLASS_ID;}
	Class_ID		ClassID() {return Class_ID(WINDMOD_CLASS_ID,4599);}
	const TCHAR* 	Category() {return _T("");}
	};

static WindModClassDesc windModDesc;
ClassDesc* GetWindModDesc() {return &windModDesc;}


//--- GravityObject Parameter map/block descriptors ------------------

#define PB_STRENGTH		0


//--- WindObject Parameter map/block descriptors ------------------

#define PB_TURBULENCE	1
#define PB_FREQUENCY	2
#define PB_SCALE		3
#define PB_DISPINNER	4
#define PB_DISPOUTER	5
#define PB_FRICTION		6
#define PB_FALLOFF		7
#define PB_NTH			8

//
//
// Parameters

static ParamUIDesc descParamWind[] = {
	// Strength
	ParamUIDesc(
		PB_STRENGTH,
		EDITTYPE_FLOAT,
		IDC_WIND_STRENGTH,IDC_WIND_STRENGTHSPIN,
		-9999999.0f, 9999999.0f,
		0.01f),


	// Turbulence
	ParamUIDesc(
		PB_TURBULENCE,
		EDITTYPE_FLOAT,
		IDC_WIND_TURB,IDC_WIND_TURBSPIN,
		0.0f, 9999999.0f,
		0.01f),

	// Frequency
	ParamUIDesc(
		PB_FREQUENCY,
		EDITTYPE_FLOAT,
		IDC_WIND_FREQ,IDC_WIND_FREQSPIN,
		0.0f, 9999999.0f,
		0.01f),

	// Scale
	ParamUIDesc(
		PB_SCALE,
		EDITTYPE_FLOAT,
		IDC_WIND_SCALE,IDC_WIND_SCALESPIN,
		0.0f, 9999999.0f,
		0.01f),


	// Display inner
	ParamUIDesc(
		PB_DISPINNER,
		EDITTYPE_FLOAT,
		IDC_INNER,IDC_INNERSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),


	// Display outer
	ParamUIDesc(
		PB_DISPOUTER,
		EDITTYPE_FLOAT,
		IDC_OUTER,IDC_OUTERSPIN,
		0.0f, 9999999.0f,
		SPIN_AUTOSCALE),

	// Friction
	ParamUIDesc(
		PB_FRICTION,
		EDITTYPE_FLOAT,
		IDC_FRICTION,IDC_FRICTIONSPIN,
		-100.0f, 100.0f,
		0.1f),

	// Use Falloff
	ParamUIDesc(PB_FALLOFF,TYPE_SINGLECHEKBOX,IDC_FALLOFF),

	// NTH
	ParamUIDesc(
		PB_NTH,
		EDITTYPE_INT,
		IDC_NTH,IDC_NTHSPIN,
		1.0f, 10000.0f,
		1.0f),



	};
#define WINDPARAMDESC_LENGTH	9

ParamBlockDescID descWindVer0[] = {
	{ TYPE_FLOAT, NULL, TRUE, 0 },
	{ TYPE_FLOAT, NULL, TRUE, 1 },
	{ TYPE_FLOAT, NULL, TRUE, 2 },
	{ TYPE_FLOAT, NULL, TRUE, 3 },
	{ TYPE_FLOAT, NULL, TRUE, 4 },
	{ TYPE_FLOAT, NULL, TRUE, 5 },
	{ TYPE_FLOAT, NULL, TRUE, 6 },
	{ TYPE_INT, NULL, FALSE, 7 },
	{ TYPE_INT, NULL, TRUE, 8 },
};
#define WINDPBLOCK_LENGTH	9

#define CURRENT_WINDVERSION	0


//--- ForceObject Methods ---------------------------------------------

void WindObject::BeginEditParams(
		IObjParam *ip,ULONG flags,Animatable *prev)
	{
	SimpleWSMObject::BeginEditParams(ip,flags,prev);
	this->ip = ip;

	if (pmapParam) {		
		// Left over from last SinWave ceated
		pmapParam->SetParamBlock(pblock);
	} else {		
		hSot = ip->AddRollupPage( 
			hInstance, 
			MAKEINTRESOURCE(IDD_WINDGRAV_SOT),
			DefaultSOTProc,
			GetString(IDS_RB_SOT), 
			(LPARAM)ip,APPENDROLL_CLOSED);

		// Gotta make a new one.
		pmapParam = CreateCPParamMap(
			UIDesc(),UIDescLength(),
			pblock,
			ip,
			hInstance,
			MAKEINTRESOURCE(DialogID()),
			GetString(IDS_RB_PARAMETERS),
			0);
		}
		if (pmapParam)
			pmapParam->SetUserDlgProc(new UniObjectDlgProc(this));

	}

void WindObject::EndEditParams(
		IObjParam *ip, ULONG flags,Animatable *next)
	{		
	SimpleWSMObject::EndEditParams(ip,flags,next);
	this->ip = NULL;

	if (flags&END_EDIT_REMOVEUI ) {		
		DestroyCPParamMap(pmapParam);
		ip->DeleteRollupPage(hSot);
		pmapParam = NULL;		
		}	
	ip->ClearPickMode();
	}




void ForceObject::BuildMesh(TimeValue t)
	{
	ivalid = FOREVER;
	float inner,outer;
	float length;
	
	pblock->GetValue(PB_DISPINNER,t,inner,ivalid);
	pblock->GetValue(PB_DISPOUTER,t,outer,ivalid);



	if (inner > outer)
		{
		float temp;
		temp = inner;
		inner = outer;
		outer = temp;
		}


	length = inner;

	float u;
	#define NUM_SEGS	16

	mesh.setNumVerts(3*NUM_SEGS+1+3*NUM_SEGS);
	mesh.setNumFaces(3*NUM_SEGS+3*NUM_SEGS);
	int vct;
	vct = 0;

	for (int i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i, Point3((float)cos(u) * length, (float)sin(u) * length, 0.0f));
			vct++;
			}
	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+NUM_SEGS, Point3(0.0f, (float)cos(u) * length, (float)sin(u) * length));
			vct++;
			}
	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+2*NUM_SEGS, Point3((float)cos(u) * length, 0.0f, (float)sin(u) * length));
			vct++;
			}		


	length = outer;

	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+vct, Point3((float)cos(u) * length, (float)sin(u) * length, 0.0f));
			}
	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+NUM_SEGS+vct, Point3(0.0f, (float)cos(u) * length, (float)sin(u) * length));
			}
	for (i=0; i<NUM_SEGS; i++) {
			u = float(i)/float(NUM_SEGS) * TWOPI;
			mesh.setVert(i+2*NUM_SEGS+vct, Point3((float)cos(u) * length, 0.0f, (float)sin(u) * length));
			}		



	mesh.setVert(3*NUM_SEGS, Point3(0.0f, 0.0f, 0.0f));
		
	for (i=0; i<(3*NUM_SEGS) + (3*NUM_SEGS); i++) {
			int i1 = i+1;
			if (i1%NUM_SEGS==0) i1 -= NUM_SEGS;
			mesh.faces[i].setEdgeVisFlags(1,0,0);
			mesh.faces[i].setSmGroup(1);
			mesh.faces[i].setVerts(i,i1,3*NUM_SEGS+3*NUM_SEGS);
			}

	mesh.InvalidateGeomCache();
	}

class ForceObjCreateCallback : public CreateMouseCallBack {
	public:
		ForceObject *ob;	
		Point3 p0, p1;
//		IPoint2 sp0;
		IPoint2 sp0, sp1;
		int proc( ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat);
	};
/*
int ForceObjCreateCallback::proc(
		ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
	{
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:								
				sp0 = m;
				p0  = vpt->GetPointOnCP(m);
				mat.SetTrans(p0);
				break;
			case 1:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPLENGTH,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				if (msg==MOUSE_POINT) {
					if (Length(m-sp0)<3) return CREATE_ABORT;
					else return CREATE_STOP;
					}
				break;
			}
	} else {
		if (msg == MOUSE_ABORT)
			return CREATE_ABORT;
		}
	
	return TRUE;
	}

*/

int ForceObjCreateCallback::proc(
		ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
	{
	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:								
				sp0 = m;
				p0  = vpt->GetPointOnCP(m);
				mat.SetTrans(p0);
				ob->pblock->SetValue(PB_DISPOUTER,0,0.0f);
				ob->pblock->SetValue(PB_DISPOUTER,0,0.0f);
				break;
			case 1:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPINNER,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				break;
			case 2:
				if (ob->ClassID()==Class_ID(GRAVITYOBJECT_CLASS_ID,0)) {
					mat.IdentityMatrix();
					mat.RotateX(PI);
					mat.SetTrans(p0);
					}
				p1  = vpt->GetPointOnCP(m);
				ob->pblock->SetValue(PB_DISPOUTER,0,Length(p1-p0)/2.0f);
				ob->pmapParam->Invalidate();
				if (msg==MOUSE_POINT) {
					if (Length(m-sp0)<3) return CREATE_ABORT;
					else return CREATE_STOP;
					}
				break;

			}
	} else {
		if (msg == MOUSE_ABORT)
			return CREATE_ABORT;
		}
	
	return TRUE;
	}


static ForceObjCreateCallback forceCreateCB;

CreateMouseCallBack* ForceObject::GetCreateMouseCallBack()
	{
	forceCreateCB.ob = this;
	return &forceCreateCB;
	}

void ForceObject::InvalidateUI() 
	{
	if (pmapParam) pmapParam->Invalidate();
	}






//--- WindObject methods ---------------------------------------


WindObject::WindObject()
	{
	MakeRefByID(FOREVER, 0, 
		CreateParameterBlock(descWindVer0, WINDPBLOCK_LENGTH, CURRENT_WINDVERSION));
	assert(pblock);	

	pblock->SetValue(PB_STRENGTH,0,10.0f);
	pblock->SetValue(PB_SCALE,0,1.0f);
	pblock->SetValue(PB_DISPINNER,0,10.0f);
	pblock->SetValue(PB_DISPOUTER,0,30.0f);
	pblock->SetValue(PB_FRICTION,0,5.0f);
	pblock->SetValue(PB_NTH,0,1);

	custname=TSTR(_T(" "));
	custnode=NULL;

	}

Modifier *WindObject::CreateWSMMod(INode *node)
	{
	return new WindMod(node,this);
	}

ForceField *WindObject::GetForceField(INode *node)
	{
	WindField *wf = new WindField;	
	wf->obj  = this;
	wf->node = node;
	wf->tmValid.SetEmpty();
	wf->fValid.SetEmpty();
//	wf->obj->pblock->GetValue(PB_TYPE,0,wf->type,FOREVER);
	return wf;
	}

RefTargetHandle WindObject::Clone(RemapDir& remap) 
	{
	WindObject* newob = new WindObject();
	newob->ReplaceReference(0,pblock->Clone(remap));
	return newob;
	}

ParamDimension *WindObject::GetParameterDim(int pbIndex) 
	{
	switch (pbIndex) {		
		case 0:
		default: return defaultDim;
		}
	}

TSTR WindObject::GetParameterName(int pbIndex) 
	{
	switch (pbIndex) {		
		case PB_STRENGTH: 	return GetString(IDS_RB_STRENGTH2);
//		case PB_DECAY:		return GetString(IDS_RB_DECAY);
		case PB_TURBULENCE:	return GetString(IDS_RB_TURBULENCE);
		case PB_FREQUENCY:	return GetString(IDS_RB_FREQUENCY);
		case PB_SCALE:		return GetString(IDS_RB_SCALE);
		case PB_DISPINNER:	return TSTR(_T("Inner"));
		case PB_DISPOUTER:	return TSTR(_T("Outer"));
		default: 			return TSTR(_T(""));
		}
	}

ParamUIDesc *WindObject::UIDesc()
	{
	return descParamWind;
	}

int WindObject::UIDescLength()
	{
	return WINDPARAMDESC_LENGTH;
	}


void WindObject::ShowName()
{TSTR name; 
if (custnode == NULL)
 name= TSTR(GetString(IDS_AP_NONE));
else name= TSTR(custnode->GetName());


SetWindowText(GetDlgItem(hParams, IDC_PCLOUD_PCUST), name);
}


RefTargetHandle WindObject::GetReference(int i)
{	switch(i) {
		case PBLK: return(RefTargetHandle)pblock;
		case CUSTNODE: return (RefTargetHandle)custnode;
		default: return NULL;
		}
	}

void WindObject::SetReference(int i, RefTargetHandle rtarg) { 
	switch(i) {
		case PBLK: pblock=(IParamBlock*)rtarg; return;
		case CUSTNODE: custnode = (INode *)rtarg; return;
		}
	}

RefResult WindObject::NotifyRefChanged( 
		Interval changeInt,
		RefTargetHandle hTarget, 
		PartID& partID, 
		RefMessage message )
	{				
	switch (message) {		
		case REFMSG_TARGET_DELETED:	
			{ if (hTarget==custnode) custnode=NULL;
			}
			break;
		case REFMSG_NODE_NAMECHANGE:
			{ if (hTarget==custnode) 
			  { custname = TSTR(custnode->GetName());
			    ShowName();
				}
			  break;
			}
		default: SimpleWSMObject::NotifyRefChanged(changeInt,hTarget,partID,message);
		}
	return REF_SUCCEED;
	}

// This is an adjustment to forces to make the independent of time scale.
// They were previously dependent on the old 1200 ticks per second constant.
// Note that the constants are being squared because we are dealing with acceleration not velocity.
static float forceScaleFactor = float(1200*1200)/float(TIME_TICKSPERSEC*TIME_TICKSPERSEC);


//--- WindMod methods -----------------------------------------


WindMod::WindMod(INode *node,WindObject *obj)
	{	
	//MakeRefByID(FOREVER,SIMPWSMMOD_OBREF,obj);
	MakeRefByID(FOREVER,SIMPWSMMOD_NODEREF,node);	
	pblock = NULL;
	obRef = NULL;
//	custnode = obj->custnode;
	}

Interval WindMod::GetValidity(TimeValue t) 
	{
	if (nodeRef) {
		Interval valid = FOREVER;
		Matrix3 tm;
		float f;		
		((WindObject*)GetWSMObject(t))->pblock->GetValue(PB_STRENGTH,t,f,valid);		
		tm = nodeRef->GetObjectTM(t,&valid);
		if (((WindObject*)GetWSMObject(t))->custnode != NULL)
			tm = ((WindObject*)GetWSMObject(t))->custnode->GetObjectTM(t,&valid);
		return valid;
	} else {
		return FOREVER;
		}
	}

class WindDeformer : public Deformer {
	public:		
		Point3 Map(int i, Point3 p) {return p;}
	};
static WindDeformer wdeformer;

Deformer& WindMod::GetDeformer(
		TimeValue t,ModContext &mc,Matrix3& mat,Matrix3& invmat)
	{
	return wdeformer;
	}

RefTargetHandle WindMod::Clone(RemapDir& remap) 
	{
	WindMod *newob = new WindMod(nodeRef,(WindObject*)obRef);	
	newob->SimpleWSMModClone(this);
	return newob;
	}

void WindMod::ModifyObject(
		TimeValue t, ModContext &mc, ObjectState *os, INode *node)
	{
	ParticleObject *obj = GetParticleInterface(os->obj);
	if (obj) {
		force.obj  = (WindObject*)GetWSMObject(t);
		force.node = nodeRef;
		force.tmValid.SetEmpty();
		force.fValid.SetEmpty();
//		force.obj->pblock->GetValue(PB_TYPE,t,force.type,FOREVER);
//build plist list here 
		force.plist.ZeroCount();

		if (((WindObject*)GetWSMObject(t))->custnode != NULL)
			{
			Matrix3 tm = ((WindObject*)GetWSMObject(t))->custnode->GetObjectTM(t);
			ObjectState os = ((WindObject*)GetWSMObject(t))->custnode->EvalWorldState(t);
	// Convert it to a tri object
			TriObject *obj = NULL;
			BOOL needsDel = FALSE;

			if (os.obj->IsSubClassOf(triObjectClassID)) 
				{
				obj = (TriObject*)os.obj;
				needsDel = FALSE;
				} 
			else
				{
				if (os.obj->CanConvertToType(triObjectClassID)) 
					{
					Object *oldObj = os.obj;
					obj = (TriObject*)
						os.obj->ConvertToType(t,triObjectClassID);
					needsDel = (obj != oldObj);
					}
				}
//			obj->mesh.IntersectRay(ray, at, norm, fi, bary)) {
			Point3 p;
			TSTR ch = ((WindObject*)GetWSMObject(t))->custnode->GetName();
			int ct = obj->mesh.numVerts;
			for (int i = 0; i < ct;i++)
				{
				p = obj->mesh.getVert(i);
				p = p*tm;
				force.plist.Append(1,&p,1);
				}
			if (needsDel) obj->DeleteThis();

			}

		obj->ApplyForceField(&force);
		}
	}

static float RTurbulence(Point3 p,float freq)
	{
	return noise3(p*freq);
	}


Point3 WindField::Force(
		TimeValue t,const Point3 &pos, const Point3 &vel,int index)
	{	


	
	float inner, outer;
	float friction;
	int useFalloff;
	obj->pblock->GetValue(PB_DISPINNER,t,inner,FOREVER);
	obj->pblock->GetValue(PB_DISPOUTER,t,outer,FOREVER);
	obj->pblock->GetValue(PB_FRICTION,t,friction,FOREVER);
	obj->pblock->GetValue(PB_FALLOFF,t,useFalloff,FOREVER);



	float strength, turb;
	float per = 1.0f;
	Point3 tp(0.0f,0.0f,0.0f);
	Point3 Friction(0.0f,0.0f,0.0f);
	int nth;
	obj->pblock->GetValue(PB_TURBULENCE,t,turb,FOREVER);
	obj->pblock->GetValue(PB_NTH,t,nth,FOREVER);

	fValid = FOREVER;		
	tm = node->GetObjectTM(t,&tmValid);
	fValid &= tmValid;

	obj->pblock->GetValue(PB_STRENGTH,t,strength,fValid);


	strength = strength /1000.f;

	if (plist.Count() == 0)
		force = pos-tm.GetTrans();
	else 
		{
		force = pos - plist[(index*nth)%plist.Count()];
		}
	per = 1.0f;



	if (useFalloff)
		{
		float dist;
		dist  = Length(force);
	//	if (dist!=0.0f) force /= dist;
		if (dist > outer) 
			{
			strength = 0.0f;
			return(tp);
			}
		else if (dist > inner)
			{				
			strength *= (float)(outer-dist)/(outer-inner);
			}			
		}


	force = ((-strength*force)-(friction*vel)) * 0.0001f * forceScaleFactor;

		
//		}	
	if (turb!=0.0f) {
		float freq, scale;
		Point3 tf, pt = pos-tm.GetTrans(), p;
		obj->pblock->GetValue(PB_FREQUENCY,t,freq,FOREVER);
		obj->pblock->GetValue(PB_SCALE,t,scale,FOREVER);
		freq *= 0.01f;
		turb *= 0.0001f * forceScaleFactor;

		p    = pt;
		p.x  = freq * float(t);
		tf.x = RTurbulence(p,scale);
		p    = pt;
		p.y  = freq * float(t);
		tf.y = RTurbulence(p,scale);
		p    = pt;
		p.z  = freq * float(t);
		tf.z = RTurbulence(p,scale);

		return (force ) + (turb*tf);
	} else {
		return force ;
		}
	}

