####################################################################
#                                                                  #
#  MAIL_LIB.PL (slightly revised version of MIME Lite.pm)          #
#  Modified 10/07/99                                               #
#  MIME Lite originally (c) and developed by Eryq                  #
#  Email: eryq@zeegee.com                                          #
#  URL:   http://www.zeegee.com                                    #
#  Please visit the above URL for the latest information on        #
#  MIME Lite.                                                      #
#                                                                  #
#  THIS LIBRARY/MODULE IS BEING DISTRIBUTED AS PART OF             #
#  CGI CITY's MAIL-A-FILE SCRIPT IN ACCORDANCE WITH THE            #
#  GNU GENERAL PUBLIC LICENSE.                                     #
#                                                                  #
#  CGI CITY'S MAIL-A-FILE                                          #
#  Copyright (c) 1998-2099 - CGI City - All rights reserved        #
#  Author: Peter N. Go - cgicity@icthus.net                        #
#                                                                  #
####################################################################
#                                                                  #
#  DISCLAIMER:                                                     #
#  In no event will CGI City be liable to the user of this script  #
#  or any third party for any damages, including any lost profits, #
#  lost savings or other incidental, consequential or special      #
#  damages arising out of the operation of or inability to operate #
#  this script, even if user has been advised of the possibility   #
#  of such damages.                                                #
#                                                                  #
####################################################################


package MIME::MIME;
use Carp;
use FileHandle;
use strict;
use vars qw($VERSION $QUIET $PARANOID $VANILLA);
$VERSION = substr q$Revision: 1.135 $, 10;
$QUIET = undef;
$PARANOID = 0;
$VANILLA = 0;

my $Sender     = "sendmail";
my %SenderArgs = (
    "sendmail" => ["/usr/sbin/sendmail -t -oi -oem"],
    "smtp"     => [],
    "sub"      => [],
);

my $BCount = 0;

my %KnownField = map {$_=>1} 
qw(
   bcc         cc          comments      date          encrypted 
   from        keywords    message-id    mime-version  organization
   received    references  reply-to      return-path   sender        
   subject     to
   );

my @Uses;

sub fold {
    my $str = shift;
    $str =~ s/^\s*|\s*$//g;    # trim
    $str =~ s/\n/\n /g;      
    $str;
}

sub gen_boundary {
    return ("_----------=_".($VANILLA ? '' : int(time).$$).$BCount++);
}


sub known_field {
    my $field = lc(shift);
    $KnownField{$field} or ($field =~ m{^(content|resent|x)-.});
}

sub is_mime_field {
    $_[0] =~ /^(mime\-|content\-)/i;
}

if (!$PARANOID and eval "require MIME::Base64") {
    import MIME::Base64 qw(encode_base64);
    push @Uses, "B$MIME::Base64::VERSION";
}
else {
    eval q{
sub encode_base64 {
    my $res = "";
    my $eol = "\n";

    pos($_[0]) = 0;        # thanks, Andreas!
    while ($_[0] =~ /(.{1,45})/gs) {
	$res .= substr(pack('u', $1), 1);
	chop($res);
    }
    $res =~ tr|` -_|AA-Za-z0-9+/|;

    # Fix padding at the end:
    my $padding = (3 - length($_[0]) % 3) % 3;
    $res =~ s/.{$padding}$/'=' x $padding/e if $padding;

    # Break encoded string into lines of no more than 76 characters each:
    $res =~ s/(.{1,76})/$1$eol/g if (length $eol);
    return $res;
} # sub
  } # q
} #if


if (!$PARANOID and eval "require MIME::QuotedPrint") {
    import MIME::QuotedPrint qw(encode_qp);
    push @Uses, "Q$MIME::QuotedPrint::VERSION";
}
else {
    eval q{

sub encode_qp {
    my $res = shift;
    $res =~ s/([^ \t\n!-<>-~])/sprintf("=%02X", ord($1))/eg;  # rule #2,#3
    $res =~ s/([ \t]+)$/
      join('', map { sprintf("=%02X", ord($_)) }
	           split('', $1)
      )/egm;                        # rule #3 (encode whitespace at eol)

    my $brokenlines = "";
    $brokenlines .= "$1=\n" while $res =~ s/^(.{70}([^=]{2})?)//; # 70 was 74
    $brokenlines =~ s/=\n$// unless length $res; 
    "$brokenlines$res";
} # sub
  } # q
} #if

sub encode_8bit {
    my $str = shift;
    $str =~ s/^.{990}/$&\n/mg;
    $str;
}

sub encode_7bit {
    my $str = shift;
    $str =~ s/[\x80-\xFF]//eg; 
    $str =~ s/^.{990}/$&\n/mg;
    $str;
}


sub new {
    my $class = shift;

    my $self = {
	Attrs => {},
	Header => [],    # message header
	Parts => [],     # array of parts
    };    
    bless $self, $class;

    # Build, if needed:
    return (@_ ? $self->build(@_) : $self);
}

sub attach {
    my $self = shift;

    # Create new part, if necessary:
    my $part1 = ((@_ == 1) ? shift : ref($self)->new(Top=>0, @_));

    # Do the "attach-to-singlepart" hack:
    if ($self->attr('content-type') !~ m{^multipart/}i) {

	# Create part zero:
	my $part0 = ref($self)->new;

	# Cut MIME stuff from self, and paste into part zero: 
	foreach (qw(Attrs Data Path FH)) {
	    $part0->{$_} = $self->{$_}; delete($self->{$_});
	}
	$part0->top_level(0);    # clear top-level attributes

	# Make self a top-level multipart:
	$self->{Attrs} ||= {};   # reset       
	$self->attr('content-type'              => 'multipart/mixed');
	$self->attr('content-type.boundary'     => gen_boundary());
	$self->attr('content-transfer-encoding' => '7bit');
	$self->top_level(1);     # activate top-level attributes

	# Add part 0:
	push @{$self->{Parts}}, $part0;
    }

    # Add the new part:
    push @{$self->{Parts}}, $part1;
    $part1;
}

sub build {
    my $self = shift;
    my %params = @_;
    my @params = @_;
    my $key;

    (defined($params{Data})+defined($params{Path})+defined($params{FH}) <= 1)
	or croak "supply exactly zero or one of (Data|Path|FH).\n";

    ref($self) or $self = $self->new;

    my $type = ($params{Type} || 'TEXT');
    ($type eq 'TEXT')   and $type = 'text/plain';
    ($type eq 'BINARY') and $type = 'application/octet-stream';
    $type = lc($type);
    $self->attr('content-type' => $type);
   
    my $is_multipart = ($type =~ m{^(multipart)/}i);

    if ($is_multipart) {
	my $boundary = gen_boundary();
	$self->attr('content-type.boundary' => $boundary);
    }

    if (defined($params{Data})) {
	$self->data($params{Data});
    }

    elsif (defined($params{Path})) {
	$self->path($params{Path});       # also sets filename
	$self->read_now if $params{ReadNow};
    }

    elsif (defined($params{FH})) {
	$self->fh($params{FH});
	$self->read_now if $params{ReadNow};  # implement later
    }
    
    if (defined($params{Filename})) {
	$self->filename($params{Filename});
    }
  

    my $enc = $params{Encoding} || 'binary';      # explicit value wins
    $self->attr('content-transfer-encoding' => lc($enc));
	
    if ($type =~ m{^(multipart|message)/}) {
	($enc =~ m{^(7bit|8bit|binary)\Z}) or 
	    croak "illegal MIME: can't have encoding $enc with type $type!";
    }

    my $disp = ($params{Disposition} or ($is_multipart ? undef : 'inline'));
    $self->attr('content-disposition' => $disp);

    my $length;
    if (exists($params{Length})) {   # given by caller:
	$self->attr('content-length' => $params{Length});
    }
    else {                           # compute it ourselves
	$self->get_length;
    }
    
    $self->top_level(defined($params{Top}) ? $params{Top} : 1);

    my @paramz = @params;
    my $field;
    while (@paramz) {
	my ($tag, $value) = (shift(@paramz), shift(@paramz));

	if ($tag =~ /^\-/) {       # old style, backwards-compatibility
	    $field = lc($');
	}
	elsif ($tag =~ /:$/) {     # new style
	    $field = lc($`);
	}
	elsif (known_field($field = lc($tag))) {   # known field
	    # no-op
	}
	else {                     # not a field:
	    next;
	}
	
	$self->add($field, $value);
    }

    # Done!
    $self;
}

sub top_level {
    my ($self, $onoff) = @_;	
    if ($onoff) {
	$self->attr('MIME-Version' => '1.0');
	my $uses = (@Uses ? ("(" . join("; ", @Uses) . ")") : '');
	$self->replace('X-Mailer' => "MIME::Lite $VERSION $uses")
	    unless $VANILLA;
    }
    else {
	$self->attr('MIME-Version' => undef);
	$self->delete('X-Mailer');
    }
}

sub add {
    my $self = shift;
    my $tag = lc(shift);
    my $value = shift;

    carp "Explicitly setting a MIME header field ($tag) is dangerous:\n".
	 "use the attr() method instead.\n"
	if (is_mime_field($tag) && !$QUIET);

    my @vals = ref($value) ? @{$value} : ($value);
    map { s/\n/\n /g } @vals;

    foreach (@vals) {
	push @{$self->{Header}}, [$tag, $_];
    }
}

sub attr {
    my ($self, $attr, $value) = @_;
    $attr = lc($attr);

    # Break attribute name up:
    my ($tag, $subtag) = split /\./, $attr;
    defined($subtag) or $subtag = '';

    # Set or get?
    if (@_ > 2) {   # set:
	$self->{Attrs}{$tag} ||= {};            # force hash
	delete $self->{Attrs}{$tag}{$subtag};   # delete first
	if (defined($value)) {                  # set...
	    $value =~ s/[\r\n]//g;                   # make clean
	    $self->{Attrs}{$tag}{$subtag} = $value;
	}
    }
	
    # Return current value:
    $self->{Attrs}{$tag}{$subtag};
}

sub delete {
    my $self = shift;
    my $tag = lc(shift);

    # Delete from the header:
    my $hdr = [];
    my $field;
    foreach $field (@{$self->{Header}}) {
	push @$hdr, $field if ($field->[0] ne $tag);
    }
    $self->{Header} = $hdr;
    $self;
}

sub fields {
    my $self = shift;
    my @fields;
    
    my %explicit = map { $_->[0] => 1 } @{$self->{Header}};
    
    my $tag;
    foreach $tag (sort keys %{$self->{Attrs}}) {	

	next if ($explicit{$tag});         

	my @subtags = keys %{$self->{Attrs}{$tag}}; 
	@subtags or next;

	my $value;
	defined($value = $self->{Attrs}{$tag}{''}) or next; # need default tag!
	foreach (sort @subtags) {
	    next if ($_ eq '');
	    $value .= qq{; $_="$self->{Attrs}{$tag}{$_}"};
	}
	
	push @fields, [$tag, $value];
    }
    
    foreach (@{$self->{Header}}) {
	push @fields, [@{$_}];
    }

    # Done!
    return \@fields;
}

sub filename {
    my ($self, $filename) = @_;
    if (@_ > 1) {
	$self->attr('content-type.name' => $filename);
	$self->attr('content-disposition.filename' => $filename);
    }
    $self->attr('content-disposition.filename');
}

sub get {
    my ($self, $tag, $index) = @_;
    $tag = lc($tag); 
    croak "get: can't be used with MIME fields\n" if is_mime_field($tag);
    
    my @all = map { ($_->[0] eq $tag) ? $_->[1] : ()} @{$self->{Header}};
    (defined($index) ? $all[$index] : (wantarray ? @all : $all[0]));
}

sub get_length {
    my $self = shift;

    my $is_multipart = ($self->attr('content-type') =~ m{^multipart/}i);
    my $enc = lc($self->attr('content-transfer-encoding') || 'binary');
    my $length;
    if (!$is_multipart && ($enc eq "binary")){  # might figure it out cheap:
	if    (defined($self->{Data})) {               # it's in core
	    $length = length($self->{Data});
	}
	elsif (defined($self->{FH})) {                 # it's in a filehandle
	    # no-op: it's expensive, so don't bother
	}
	elsif (-e $self->{Path}) {                     # it's a simple file!
	    $length = (-s $self->{Path});
	}
    }
    $self->attr('content-length' => $length);
    return $length;
}

sub replace {
    my ($self, $tag, $value) = @_;
    $self->delete($tag);
    $self->add($tag, $value) if defined($value);
}

sub binmode {
    my $self = shift;
    $self->{Binmode} = shift if (@_);       # argument? set override
    return (defined($self->{Binmode}) 
	    ? $self->{Binmode}
	    : ($self->attr("content-type") !~ m{^(text|message)/}i));
}

sub data {
    my $self = shift;
    if (@_) {
	$self->{Data} = ((ref($_[0]) eq 'ARRAY') ? join('', @{$_[0]}) : $_[0]);
	$self->get_length;
    }
    $self->{Data};
}


sub path {
    my $self = shift;
    if (@_) {

	# Set the path, and invalidate the content length:
	$self->{Path} = shift;

	# Re-set filename, extracting it from path if possible:
	my $filename;
	if ($self->{Path} and ($self->{Path} !~ /\|$/)) {  # non-shell path:
	    ($filename = $self->{Path}) =~ s/^<//;    
	    ($filename) = ($filename =~ m{([^\/]+)\Z});
	}
	$self->filename($filename);

	# Reset the length:
	$self->get_length;
    }
    $self->{Path};
}

sub fh {
    my $self = shift;
    $self->{FH} = shift if @_;
    $self->{FH};
}


sub resetfh {
    my $self = shift;
    seek($self->{FH},0,0);
}

sub read_now {
    my $self = shift;
    local $/ = undef;
    
    if    ($self->{FH}) {       # data from a filehandle:
	my $chunk;
	$self->{Data} = '';
	CORE::binmode($self->{FH}) if $self->binmode;
	while (read($self->{FH}, $chunk, 1024)) {$self->{Data} .= $chunk}
    }
    elsif ($self->{Path}) {     # data from a path:
	open SLURP, $self->{Path} or croak "open $self->{Path}: $!";
	CORE::binmode(SLURP) if $self->binmode;
	$self->{Data} = <SLURP>;        # sssssssssssssslurp...
	close SLURP;                    # ...aaaaaaaaahhh!
    }
}

sub sign {
    my $self = shift;
    my %params = @_;

    @_ or $params{Path} = "$ENV{HOME}/.signature";

    defined($self->{Data}) or $self->read_now;

    my $sig;
    if (!defined($sig = $params{Data})) {      # not given explicitly:
	local $/ = undef;
	open SIG, $params{Path} or croak "open sig $params{Path}: $!";
	$sig = <SIG>;
	close SIG;
    }    
    $sig = join('',@$sig) if (ref($sig) and (ref($sig) eq 'ARRAY'));

    $self->{Data} .= "\n-- \n$sig";

    # Re-compute length:
    $self->get_length;
    1;
}

sub print {
    my ($self, $out) = @_;

    $out = wrap MIME::Lite::IO_Handle $out;

    $self->print_header($out);
    $out->print("\n");

    if ($self->attr('content-type') !~ m{^multipart/}i) {	
	$self->print_body($out);  # Single part
    }
    else {                        # Multipart...
	my $boundary = $self->attr('content-type.boundary');

	$out->print("This is a multi-part message in MIME format.\n");
	
	my $part;
	foreach $part (@{$self->{Parts}}) {
	    $out->print("\n--$boundary\n");
	    $part->print($out);
	}
	$out->print("\n--$boundary--\n\n");
    }
    1;
}

sub print_body {
    my ($self, $out) = @_;

    $out = wrap MIME::Lite::IO_Handle $out;

    my $encoding = uc($self->attr('content-transfer-encoding'));

    if (defined($self->{Data})) {
      DATA: 
	{ $_ = $encoding;

	  /^BINARY$/ and do {
	      $out->print($self->{Data}); 
	      last DATA;
	  };
	  /^8BIT$/ and do {
	      $out->print(encode_8bit($self->{Data})); 
	      last DATA;
	  };
	  /^7BIT$/ and do {
	      $out->print(encode_7bit($self->{Data})); 
	      last DATA;
	  };
	  /^QUOTED-PRINTABLE$/ and do {
	      while ($self->{Data}=~ m{^.*[\r\n]*}mg) {
		  $out->print(encode_qp($&));   # have to do it line by line...
	      }
	      last DATA;	 
	  };
	  /^BASE64/ and do {
	      $out->print(encode_base64($self->{Data})); 
	      last DATA;
	  };
	  croak "unsupported encoding: `$_'";
        }
    }

    elsif (defined($self->{Path}) || defined($self->{FH})) {
	no strict 'refs';          # in case FH is not an object
	my $DATA;
	
	if (defined($self->{Path})) {
	    $DATA = new FileHandle || croak "can't get new filehandle!";
	    $DATA->open("$self->{Path}") or croak "open $self->{Path}: $!";
	}
	else {
	    $DATA=$self->{FH};
	}
	CORE::binmode($DATA) if $self->binmode;
		
      PATH: 
	{   $_ = $encoding;
	    
	    /^BINARY$/ and do {
		$out->print($_)                while read($DATA, $_, 2048); 
		last PATH;
	    };      
	    /^8BIT$/ and do {
		$out->print(encode_8bit($_))   while (<$DATA>); 
		last PATH;
	    };
	    /^7BIT$/ and do {
		$out->print(encode_7bit($_))   while (<$DATA>); 
		last PATH;
	    };
	    /^QUOTED-PRINTABLE$/ and do {
		$out->print(encode_qp($_))     while (<$DATA>); 
		last PATH;
	    };
	    /^BASE64$/ and do {
		$out->print(encode_base64($_)) while (read($DATA, $_, 45));
		last PATH;
	    };
	    croak "unsupported encoding: `$_'";
	}
	
	# Close file:
	close $DATA if defined($self->{Path});
    }
    
    else {
	croak "no data in this part!";
    }
    1;
}

sub print_header {
    my ($self, $out) = @_;

    # Coerce into a printable output handle:
    $out = wrap MIME::Lite::IO_Handle $out;

    # Output the header:
    $out->print($self->header_as_string);
    1;
}

sub as_string {
    my $self = shift;
    my $str = "";
    my $io = (wrap MIME::Lite::IO_Scalar \$str);
    $self->print($io);
    $str;
}
*stringify = \&as_string;    # backwards compatibility

sub body_as_string {
    my $self = shift;
    my $str = "";
    my $io = (wrap MIME::Lite::IO_Scalar \$str);
    $self->print_body($io);
    $str;
}
*stringify_body = \&body_as_string;    # backwards compatibility

sub header_as_string {
    my $self = shift;
    my $str = '';
    foreach (@{$self->fields}) {
	my ($tag, $value) = @$_;
	$tag =~ s/\b([a-z])/uc($1)/ge;   # make pretty
	$tag =~ s/^mime-/MIME-/ig;       # even prettier
	$str .= "$tag: $value\n";
    }
    $str;
}
*stringify_header = \&header_as_string;    # backwards compatibility

sub send {
    my $self = shift;
    if (ref($self)) {              # instance method:
	my $method = "send_by_$Sender";
	my @args   = @{$SenderArgs{$Sender} || []};
	return $self->$method(@args);
    }
    else {                         # class method:
	$Sender = shift;
	$SenderArgs{$Sender} = [@_];    # remaining args
	return 1;
    }
}

sub send_by_sendmail {
    my ($self, $sendmailcmd) = @_;

    # Do it:
    my $pid;
    open SENDMAIL, "|$sendmailcmd" or croak "open |$sendmailcmd: $!";
    $self->print(\*SENDMAIL);
    close SENDMAIL;
    return (($? >> 8) ? undef : 1);
}

sub send_by_smtp {
    my ($self, @args) = @_;

    my $hdr = $self->fields();   
    my $from = $self->get('From');
    my @to   = $self->get('To');

    require Net::SMTP;
    my $smtp = MIME::Lite::SMTP->new(@args)
        or croak "Failed to connect to mail server: $!";
    $smtp->mail($from)
        or croak "SMTP MAIL command failed: $!";
    $smtp->to(@to)
        or croak "SMTP RCPT command failed: $!";
    $smtp->data()
        or croak "SMTP DATA command failed: $!";

    $self->print($smtp);
    $smtp->dataend();
    $smtp->quit;
    1;
}

sub send_by_sub {
    my ($self, $subref, @args) = @_;
    &$subref($self, @args);
}

sub sendmail {
    my $self = shift;
    $self->send('sendmail', join(' ', @_));
}

sub quiet {
    my $class = shift;
    $QUIET = shift if @_;
    $QUIET;
}

###############################################
package MIME::Lite::SMTP;

use strict;
use vars qw( @ISA );
@ISA = qw(Net::SMTP);

sub print { shift->datasend(@_) }

###############################################
package MIME::Lite::IO_Handle;

sub wrap {
    my ($class, $fh) = @_;
    no strict 'refs';

    $fh or $fh = select;        # no filehandle means selected one
    ref($fh) or $fh = \*$fh;    # scalar becomes a globref
    
    return $fh if (ref($fh) and (ref($fh) ne 'GLOB'));

    bless \$fh, $class;         # wrap it in a printable interface
}

sub print {
    my $self = shift;
    print {$$self} @_;
}

##################################
package MIME::Lite::IO_Scalar;

sub wrap {
    my ($class, $scalarref) = @_;
    defined($scalarref) or $scalarref = \"";
    bless $scalarref, $class;
}

sub print {
    my $self = shift;
    $$self .= join('', @_);
    1;
}

1;
__END__

