// Version: 0.5
// Class name: TOrganicShapeShifter = class(TComponent)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 28 Sept 1999
// Description: Allows any windowed control to have an irregular boundary
// history:
//   28 Sept 1999 : Creation date.


unit organicShapeShifter;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls, Forms,
  organicSkin;

const
  REGISTERED_VERSION = TRUE;

type
  TOrganicShapeShifter = class(TComponent)
  private
    FInvisibleColor : TColor;
    tmpBmp : TOrganicSkin;
    FTarget: TWinControl;
    procedure setTarget(const Value: TWinControl);
    { Private declarations }
  protected
    { Protected declarations }
    maskBmp : TBitmap;

    procedure shapeIt(Sender : TObject);
    procedure SetMask(Value: TBitmap); virtual;
    procedure notification(AComponent: TComponent; Operation: TOperation); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure saveSkin(folder : string); virtual;
    procedure loadSkin(folder : string); virtual;
  published
    { Published declarations }
    property Target : TWinControl read FTarget write setTarget;
    property InvisibleColor : TColor read FInvisibleColor write FInvisibleColor;
    property Mask : TBitmap read maskBmp write setMask;
  end;

procedure Register;

implementation

uses  organicShapeControl, inifiles, FileCtrl, Dialogs;

procedure Register;
begin
  RegisterComponents('Organic Shape', [TOrganicShapeShifter]);
end;

{ TOrganicShapeMask }

constructor TOrganicShapeShifter.Create(AOwner: TComponent);
begin
  maskBmp := TBitmap.Create;
  maskBmp.PixelFormat := pfDevice;
  FInvisibleColor := clWhite;
  tmpBmp := TOrganicSkin.create;
  FTarget := nil;

  inherited Create(AOwner);
  maskBmp.OnChange := shapeIt;

  if not REGISTERED_VERSION then begin
    randomize;
    if random(15) = 1 then
      showMessage('Organic Shape Shifter By Practical Soft! practicalsoft@usa.net');
  end;
end;


destructor TOrganicShapeShifter.destroy;
begin
  {if FTarget <> nil then
    if not SetWindowRgn(FTarget.handle, createRectRgn(0, 0, FTarget.width, FTarget.height), TRUE) then
      raise OrganicShapeException.create('destroy.shapeIt: SetWindowRgn failed.');
  }

  maskBmp.free;
  tmpBmp.free;

  inherited;
end;


procedure TOrganicShapeShifter.SetMask(Value: TBitmap);
begin
  maskBmp.Assign(Value);
end;


procedure TOrganicShapeShifter.setTarget(const Value: TWinControl);
begin
  //un-mask previous target
  if FTarget <> nil then
    if not SetWindowRgn(FTarget.handle, 0, TRUE) then
      raise OrganicShapeException.create('TOrganicShapeMask.shapeIt: SetWindowRgn failed.');

  FTarget := Value;
  shapeIt(self);
end;


procedure TOrganicShapeShifter.shapeIt(Sender: TObject);
begin
  if FTarget <> nil then
    if (maskBmp.width > 0) and (maskBmp.height > 0) then begin
      tmpBmp.invisibleColor := invisibleColor;
      tmpBmp.assign(maskBmp);

      FTarget.width := tmpBmp.Width;
      FTarget.height := tmpBmp.height;
      if not SetWindowRgn(FTarget.handle, tmpBmp.region, TRUE) then
        raise OrganicShapeException.create('TOrganicShapeMask.shapeIt: SetWindowRgn failed.');
    end
    else begin
      if not SetWindowRgn(FTarget.handle, 0, TRUE) then
        raise OrganicShapeException.create('TOrganicShapeMask.shapeIt: SetWindowRgn failed.');
    end;
end;


procedure TOrganicShapeShifter.loadSkin(folder: string);
var posFile : TIniFile;
begin
  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  posFile := TIniFile.create(folder + COORD_FILE);
  try
    if FTarget <> nil then begin
      FTarget.left := posFile.ReadInteger(Name, 'left', FTarget.left);
      FTarget.top := posFile.ReadInteger(Name, 'top', FTarget.top);
      FTarget.width := posFile.ReadInteger(Name, 'width', FTarget.width);
      FTarget.height := posFile.ReadInteger(Name, 'height', FTarget.height);
    end;
    maskBmp.loadFromFile(folder + '\shifter_' + Name + '.bmp');
  finally
    posFile.free;
  end;
end;


procedure TOrganicShapeShifter.saveSkin(folder: string);
var posFile : TIniFile;
begin
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  //create folder if it does not exist yet
  if not DirectoryExists(folder) then
    if not createDir(folder) then
      raise OrganicShapeException.create('Can not create folder: '+folder);

  posFile := TIniFile.create(folder + COORD_FILE);
  try
    maskBmp.SaveToFile(folder + '\shifter_' + Name + '.bmp');
    if FTarget <> nil then begin
      posFile.WriteInteger(Name, 'left', FTarget.left);
      posFile.WriteInteger(Name, 'top', FTarget.top);
      posFile.WriteInteger(Name, 'width', FTarget.width);
      posFile.WriteInteger(Name, 'height', FTarget.height);
    end;
  finally
    posFile.free;
  end;

end;


//remove reference to a removed component
procedure TOrganicShapeShifter.notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
    if AComponent = FTarget then FTarget := nil;
end;


end.
