// Version: 2.0
// Class name: TOrganicSkin = class(TBitmap)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 6 Jan 1999
// Description: TOrganicSkin is used by all the OrganicShape components.
//   Region associated with a bitmap. The region is determined by bitmap.
//   The changed procedure recalculates the shape region everytime the 
//   bitmap changes.
// History:
//   1 Mar  1999 : region is now a property.
//   9 Sept 1999 : optimised region calculation speed by using TSpeedBitmap
//                 class to access each pixel's color.

unit OrganicSkin;

interface

uses
  windows, graphics, classes;

type
  TOrganicSkin = class(TBitmap)
  private
    savedRegion : HRGN;
    FRegion : HRGN;
    updateRGN : boolean;
    function getRegion: HRGN;
  protected
    procedure changed(anything : TObject); override;
  public
    InvisibleColor : TColor;

    constructor create; override;
    destructor destroy; override;
    procedure assign(Source: TPersistent); override;
    procedure calculateRegion;

    property Region : HRGN read getRegion write FRegion;
  end;


implementation

uses sysUtils, organicShapeControl, speedBitmap;


constructor TOrganicSkin.create;
begin
  inherited;
  
  invisibleColor := clWhite;
  FRegion := 0;
  savedRegion := 0;
  updateRGN := TRUE;
end;


destructor TOrganicSkin.destroy;
begin
  {if FRegion <> 0 then
    if not DeleteObject(FRegion) then
      raise OrganicShapeException.create('TOrganicSkin.destroy: DeleteObject failed.');}
  if savedRegion <> 0 then
    if not DeleteObject(savedRegion) then
      raise OrganicShapeException.create('TOrganicSkin.destroy: DeleteObject failed.');
      //showmessage('TOrganicSkin.destroy: DeleteObject failed.');//raise OrganicShapeException.create('TOrganicSkin.destroy: DeleteObject failed.');

  inherited;
end;


procedure TOrganicSkin.assign(Source: TPersistent);
begin
  if source is TOrganicSkin then begin
    {if FRegion <> 0 then
      if not DeleteObject(FRegion) then
        raise OrganicShapeException.create('TOrganicSkin.assign: DeleteObject failed.');}
    FRegion := CreateRectRgn(0, 0, width, height);
    if CombineRgn(FRegion, TOrganicSkin(Source).region, TOrganicSkin(Source).region, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.assign: CombineRgn failed.');
    invisibleColor := TOrganicSkin(Source).invisibleColor;
    if savedRegion <> 0 then
      if not DeleteObject(savedRegion) then
        raise OrganicShapeException.create('TOrganicSkin.assign: DeleteObject failed.');
    savedRegion := CreateRectRgn(0, 0, width, height);
    if CombineRgn(savedRegion, FRegion, FRegion, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.assign: CombineRgn failed.');
    updateRGN := FALSE;
  end;

  inherited;
end;


//Do the hard work. Iterate over each bitmap pixel to determine which are
//transparent.
procedure TOrganicSkin.calculateRegion;
var rectRgn : HRGN;
    Y, X : Integer;
    leftSqr : integer;
    edgeFound : boolean;

    a : TSpeedBitmap;
    pix : TRGBTriple;
begin
  if (width = 0) or (height = 0) then begin
    FRegion := CreateRectRgn(0, 0, Width, Height);
    exit;
  end;

  a := TSpeedBitmap.create();
  a.Assign(self);

  FRegion := CreateRectRgn(0, 0, Width, Height);
  edgeFound := FALSE;
  for y := 0 to Height-1 do begin
    leftSqr := 0;
    for X := 0 to Width-1 do begin
      //if canvas.pixels[X,Y] <> invisibleColor then begin
      pix := a.bits[x,y];
      if longInt(RGB(pix.rgbtRed, pix.rgbtGreen, pix.rgbtBlue)) <> longInt(colorToRGB(invisibleColor)) then begin
        if edgeFound then begin
          rectRgn := CreateRectRgn(leftSqr, Y, X, Y+1);
          if CombineRgn(FRegion, FRegion, rectRgn, RGN_DIFF) = ERROR then
            raise OrganicShapeException.create('TOrganicSkin.calculateRegion: CombineRgn failed.');
          if not DeleteObject(rectRgn) then
            raise OrganicShapeException.create('TOrganicSkin.calculateRegion: DeleteObject failed.');
          edgeFound := FALSE;
        end;
        leftSqr := X;
      end
      else
        edgeFound := TRUE;
    end;
    rectRgn := CreateRectRgn(leftSqr, Y, Width, Y+1);
    if CombineRgn(FRegion, FRegion, rectRgn, RGN_DIFF) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.calculateRegion: CombineRgn failed.');
    if not DeleteObject(rectRgn) then
      raise OrganicShapeException.create('TOrganicSkin.calculateRegion: DeleteObject failed.');
  end;

  //clean up
  a.free;

  //backup region
  if savedRegion <> 0 then
    if not DeleteObject(savedRegion) then
      raise OrganicShapeException.create('TOrganicSkin.calculateRegion: DeleteObject failed.');
  savedRegion := CreateRectRgn(0, 0, width, height);
  if CombineRgn(savedRegion, FRegion, FRegion, RGN_COPY) = ERROR then
    raise OrganicShapeException.create('TOrganicSkin.calculateRegion: CombineRgn failed.');
end;


//Something changed. Recalculate te region again.
procedure TOrganicSkin.changed(anything: TObject);
begin
  if updateRGN then
    calculateRegion()
  else
    updateRGN := TRUE;

  inherited;
end;


//Region getter. Always backup region.
function TOrganicSkin.getRegion: HRGN;
var tmpRgn : HRGN;
begin
  if savedRegion <> 0 then begin
    tmpRgn := CreateRectRgn(0, 0, width, height);
    if CombineRgn(tmpRgn, savedRegion, savedRegion, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.getRegion: CombineRgn failed.');

    FRegion := CreateRectRgn(0, 0, width, height);
    if CombineRgn(FRegion, tmpRgn, tmpRgn, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.getRegion: CombineRgn failed.');

    if savedRegion <> 0 then
      if not DeleteObject(savedRegion) then
        raise OrganicShapeException.create('TOrganicSkin.getRegion: DeleteObject failed.');
    savedRegion := CreateRectRgn(0, 0, width, height);
    if CombineRgn(savedRegion, tmpRgn, tmpRgn, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.getRegion: CombineRgn failed.');
    if not DeleteObject(tmpRgn) then
      raise OrganicShapeException.create('TOrganicSkin.getRegion: DeleteObject failed.');
  end
  else begin
    savedRegion := CreateRectRgn(0, 0, width, height);
    if CombineRgn(savedRegion, FRegion, FRegion, RGN_COPY) = ERROR then
      raise OrganicShapeException.create('TOrganicSkin.getRegion: CombineRgn failed.');
  end;

  result := FRegion;
end;



end.
