// Version: 2.0
// Class name: organicShapeControl = class(TCustomControl)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 3 Mar 1999
// Description: Base class for Organic Shape components
// history:
//   03 Jun 1999 : Created OrganicShapeException.
//   03 Jun 1999 : added isRelativePath helper function.
//   03 Jun 1999 : Killed bug in saveSkin and loadSkin: now they accept an
//     absolut path as argument.
//   13 Jun 1999 : repaint controll after a loadskin.
//   26 Sep 1999 : mask PixelFormat is set to pfDevice.
//   30 Sep 1999 : bug killed: when you change InvisibleColor property, the mask
//     is now being recalculated



unit OrganicShapeControl;

interface

uses
  Windows, Classes, Graphics, Controls, organicSkin, sysutils;

//filename where to store component's coordinates
const COORD_FILE = '\coord.ini';

type
  OrganicShapeException = class(Exception);

  TOrganicShapeControl = class(TCustomControl)
  private
    FInvisibleColor : TColor;
    tmpBmp : TOrganicSkin;
    FWordWrap: Boolean;
    FMargin: integer;
    procedure setMargin(const Value: integer);
    procedure setWordWrap(const Value: Boolean);
    procedure SetInvisibleColor(const Value: TColor);  // Use an Organic Skin
  protected
    maskBmp : TBitmap;

    procedure shapeIt(Sender : TObject);
    procedure SetMask(Value: TBitmap); virtual;
    procedure CreateWnd; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure saveSkin(folder : string); virtual;
    procedure loadSkin(folder : string); virtual;
    property WordWrap : Boolean read FWordWrap write setWordWrap;
    property Margin : integer read FMargin write setMargin;
  published
    property InvisibleColor : TColor read FInvisibleColor write SetInvisibleColor;
    property Mask : TBitmap read maskBmp write setMask;
  end;

function isRelativePath(path : string) : boolean;

implementation

uses inifiles, FileCtrl, Forms;

constructor TOrganicShapeControl.Create(AOwner: TComponent);
begin
  maskBmp := TBitmap.Create;
  maskBmp.PixelFormat := pfDevice;
  FInvisibleColor := clWhite;
  tmpBmp := TOrganicSkin.create;

  FWordWrap := FALSE;
  FMargin := 0;

  inherited Create(AOwner);
  maskBmp.OnChange := shapeIt;
end;


destructor TOrganicShapeControl.destroy;
begin
  maskBmp.free;
  tmpBmp.free;

  inherited;
end;


// Do the shaping work...
procedure TOrganicShapeControl.shapeIt(Sender : TObject);
begin
  if (maskBmp.width > 0) and (maskBmp.height > 0) then begin
    tmpBmp.invisibleColor := invisibleColor;
    tmpBmp.assign(maskBmp);

    if not SetWindowRgn(handle, tmpBmp.region, TRUE) then
      raise OrganicShapeException.create('TOrganicShapeControl.shapeIt: SetWindowRgn failed.');
  end
  else begin
    if not SetWindowRgn(handle, 0, TRUE) then
      raise OrganicShapeException.create('TOrganicShapeControl.shapeIt: SetWindowRgn failed.');
  end;
end;


procedure TOrganicShapeControl.SetMask(Value: TBitmap);
begin
  maskBmp.Assign(Value);
end;


procedure TOrganicShapeControl.CreateWnd;
begin
  inherited;                                

  if not(csReading in ComponentState) then
    if (tmpBmp.width > 0) and (tmpBmp.height > 0) then
      if not SetWindowRgn(handle, tmpBmp.region, TRUE) then
        raise OrganicShapeException.create('TOrganicShapeControl.CreateWnd: SetWindowRgn failed.');
end;


function isRelativePath(path : string) : boolean;
var res : string;
begin
  res := copy(path, 2, 2);
  result := res <> ':\';
end;


//Save component's coordinates.
procedure TOrganicShapeControl.saveSkin(folder: string);
var posFile : TIniFile;
begin
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  //create folder if it does not exist yet
  if not DirectoryExists(folder) then
    if not createDir(folder) then
      raise OrganicShapeException.create('Can not create folder: '+folder);

  posFile := TIniFile.create(folder + COORD_FILE);
  try
    posFile.WriteInteger(Name, 'left', left);
    posFile.WriteInteger(Name, 'top', top);
    posFile.WriteInteger(Name, 'width', width);
    posFile.WriteInteger(Name, 'height', height);
  finally
    posFile.free;
  end;
end;


//Retrieve component's coordinates
procedure TOrganicShapeControl.loadSkin(folder: string);
var posFile : TIniFile;
begin
  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  posFile := TIniFile.create(folder + COORD_FILE);
  try
    left := posFile.ReadInteger(Name, 'left', left);
    top := posFile.ReadInteger(Name, 'top', top);
    width := posFile.ReadInteger(Name, 'width', width);
    height := posFile.ReadInteger(Name, 'height', height);
  finally
    posFile.free;
  end;
  invalidate;
end;

procedure TOrganicShapeControl.setMargin(const Value: integer);
begin
  FMargin := Value;
  invalidate;
end;

procedure TOrganicShapeControl.setWordWrap(const Value: Boolean);
begin
  FWordWrap := Value;
  invalidate;
end;

procedure TOrganicShapeControl.SetInvisibleColor(const Value: TColor);
begin
  if Value <> FInvisibleColor then begin
    FInvisibleColor := Value;
    shapeIt(self);
  end;
end;

end.
