// Version: 2.0
// Class name: TOrganicShapeImage = class(TOrganicShapeControl)
// Author: Practical Soft <practicalsoft@usa.net>
// Creation Date: 2 Fev 1999
// Description: TImage like component with irregular boundaries. The 
//   boundary is set by a mask bitmap.
// Additional Properties:
//    Mask : TBitmap - the bitmap mask used to shape the image.
// history:
//   22 Fev 1999 : TOrganicShapeImage now inherits from 
//     TOrganicShapeControl.
//   23 Fev 1999 : Implemented methods: saveSkin and loadSkin.
//   24 Fev 1999 : function DestRect is now virtual.
//   24 Fev 1999 : procedure PictureChanged is now protected.
//   07 Mar 1999 : Corrected transparency bug.
//   03 Jun 1999 : Killed bug in saveSkin and loadSkin: now they accept an
//     absolut path as argument.


unit OrganicShapeImage;

interface

uses
  Windows, Messages, Classes, Graphics, Controls, Forms,
  organicShapeControl;

const
  REGISTERED_VERSION = TRUE;

type
  TOrganicShapeImage = class(TOrganicShapeControl)
  private
    FPicture: TPicture;
    FOnProgress: TProgressEvent;
    FAutoSize: Boolean;
    FStretch: Boolean;
    FCenter: Boolean;
    FIncrementalDisplay: Boolean;
    FTransparent: Boolean;
    FDrawing: Boolean;
    procedure SetAutoSize(Value: Boolean);
    procedure SetCenter(Value: Boolean);
    procedure SetPicture(Value: TPicture);
    procedure SetStretch(Value: Boolean);
    procedure SetTransparent(Value: Boolean);
  protected
    procedure PictureChanged(Sender: TObject);
    procedure Paint; override;
    function DestRect: TRect; virtual;
    function DoPaletteChange: Boolean;
    function GetPalette: HPALETTE; override;
    procedure Progress(Sender: TObject; Stage: TProgressStage;
      PercentDone: Byte; RedrawNow: Boolean; const R: TRect; const Msg: string); dynamic;
    procedure WMEraseBkgnd(var m : TWMEraseBkgnd);
  public
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    procedure saveSkin(folder : string); override;
    procedure loadSkin(folder : string); override;
  published
    property Align;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default False;
    property Center: Boolean read FCenter write SetCenter default False;
    property DragCursor;
    property DragMode;
    property Enabled;
    property IncrementalDisplay: Boolean read FIncrementalDisplay write FIncrementalDisplay default False;
    property InvisibleColor;
    property ParentShowHint;
    property Picture: TPicture read FPicture write SetPicture;
    property PopupMenu;
    property ShowHint;
    property Stretch: Boolean read FStretch write SetStretch default False;
    property Transparent: Boolean read FTransparent write SetTransparent default False;
    property Visible;
    property Mask;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnProgress: TProgressEvent read FOnProgress write FOnProgress;
    property OnStartDrag;
  end;

procedure Register;

implementation


uses jpeg, SysUtils, Dialogs;

procedure Register;
begin
  RegisterComponents('Organic Shape', [TOrganicShapeImage]);
end;


constructor TOrganicShapeImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csReplicatable];
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  FPicture.OnProgress := Progress;
  Width := 40;
  Height := 40;

  if not REGISTERED_VERSION then begin
    randomize;
    if random(15) = 1 then
      showMessage('Organic Shape Image By Practical Soft! practicalsoft@usa.net');
  end;
end;


destructor TOrganicShapeImage.destroy;
begin
  FPicture.Free;

  inherited;
end;


function TOrganicShapeImage.GetPalette: HPALETTE;
begin
  Result := 0;
  if FPicture.Graphic <> nil then
    Result := FPicture.Graphic.Palette;
end;


function TOrganicShapeImage.DestRect: TRect;
begin
  if Stretch then
    Result := ClientRect
  else if Center then
    Result := Bounds((Width - Picture.Width) div 2, (Height - Picture.Height) div 2,
      Picture.Width, Picture.Height)
  else
    Result := Rect(0, 0, Picture.Width, Picture.Height);
end;


procedure TOrganicShapeImage.Paint;
var
  Save: Boolean;
begin
  if csDesigning in ComponentState then
    with Canvas do
    begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end;
  Save := FDrawing;
  FDrawing := True;
  try
    with Canvas do
      StretchDraw(DestRect, Picture.Graphic);
  finally
    FDrawing := Save;
  end;
end;


procedure TOrganicShapeImage.WMEraseBkgnd(var m : TWMEraseBkgnd);
begin
  m.Result := LRESULT(FALSE);
end;


function TOrganicShapeImage.DoPaletteChange: Boolean;
var
  ParentForm: TCustomForm;
  Tmp: TGraphic;
begin
  Result := False;
  Tmp := Picture.Graphic;
  if Visible and (not (csLoading in ComponentState)) and (Tmp <> nil) and
    (Tmp.PaletteModified) then
  begin
    if (Tmp.Palette = 0) then
      Tmp.PaletteModified := False
    else
    begin
      ParentForm := GetParentForm(Self);
      if Assigned(ParentForm) and ParentForm.Active and Parentform.HandleAllocated then
      begin
        if FDrawing then
          ParentForm.Perform(wm_QueryNewPalette, 0, 0)
        else
          PostMessage(ParentForm.Handle, wm_QueryNewPalette, 0, 0);
        Result := True;
        Tmp.PaletteModified := False;
      end;
    end;
  end;
end;


procedure TOrganicShapeImage.Progress(Sender: TObject; Stage: TProgressStage;
  PercentDone: Byte; RedrawNow: Boolean; const R: TRect; const Msg: string);
begin
  if FIncrementalDisplay and RedrawNow then
  begin
    if DoPaletteChange then Update
    else Paint;
  end;
  if Assigned(FOnProgress) then FOnProgress(Sender, Stage, PercentDone, RedrawNow, R, Msg);
end;


procedure TOrganicShapeImage.SetAutoSize(Value: Boolean);
begin
  FAutoSize := Value;
  PictureChanged(Self);
end;

procedure TOrganicShapeImage.SetCenter(Value: Boolean);
begin
  if FCenter <> Value then
  begin
    FCenter := Value;
    PictureChanged(Self);
  end;
end;

procedure TOrganicShapeImage.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;


procedure TOrganicShapeImage.SetStretch(Value: Boolean);
begin
  if Value <> FStretch then
  begin
    FStretch := Value;
    PictureChanged(Self);
  end;
end;


procedure TOrganicShapeImage.SetTransparent(Value: Boolean);
begin
  if Value <> FTransparent then
  begin
    FTransparent := Value;
    PictureChanged(Self);
  end;
end;


procedure TOrganicShapeImage.PictureChanged(Sender: TObject);
var
  G: TGraphic;
begin
  if AutoSize and (maskBmp.Width > 0) and (maskBmp.Height > 0) then
    SetBounds(Left, Top, maskBmp.Width, maskBmp.Height);
  G := Picture.Graphic;
  if G <> nil then
  begin
    if not ((G is TMetaFile) or (G is TIcon)) then
      G.Transparent := FTransparent;
    if (not G.Transparent) and (Stretch or (G.Width >= Width)
      and (G.Height >= Height)) then
      ControlStyle := ControlStyle + [csOpaque]
    else
      ControlStyle := ControlStyle - [csOpaque];
    if DoPaletteChange and FDrawing then Update;
  end
  else ControlStyle := ControlStyle - [csOpaque];
  if not FDrawing then Invalidate;
end;


procedure TOrganicShapeImage.saveSkin(folder: string);
begin
  inherited;

  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  if FPicture.Graphic is TJPEGImage then
    FPicture.SaveToFile(folder + '\picture_' + Name + '.jpg')
  else if FPicture.Graphic is TBitmap then
    FPicture.SaveToFile(folder + '\picture_' + Name + '.bmp');

  Mask.SaveToFile(folder + '\mask_' + Name + '.bmp');
end;


procedure TOrganicShapeImage.loadSkin(folder: string);
begin
  //absolutize folder
  if isRelativePath(folder) then
    folder := ExtractFilePath(application.exename) + folder;

  if fileExists(folder + '\picture_' + Name + '.jpg') then
    FPicture.loadFromFile(folder + '\picture_' + Name + '.jpg')
  else if fileExists(folder + '\picture_' + Name + '.bmp') then
    FPicture.loadFromFile(folder + '\picture_' + Name + '.bmp');
  Mask.loadFromFile(folder + '\mask_' + Name + '.bmp');

  inherited;
end;



end.
